/*
 * Copyright 2013 Mentor Graphics, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * input/output direction clock control implementation
 */

#include <linux/clk-provider.h>
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/err.h>
#include <linux/string.h>

/**
 * struct clk_dir - clock with input/output direction
 *
 * @hw:              handle between common and hardware-specific interfaces
 * @reg:             register controlling direction
 * @bit_idx:         bit controlling direction
 * @ck_bit_idx:      bit checking direction
 * @mask:            mask of control/check bits
 * @enable_cntmask:  pointer to enable counter mask
 * @flags:           hardware-specific flags
 * @lock:            register lock
 *
 * Clock which can enable/disable its input/output buffer
 * to control its direction.
 * Implements .enable & .disable & .is_enabled
 *
 */
struct clk_dir {
	struct clk_hw   hw;
	void __iomem    *reg;
	u8              bit_idx;
	u8              ck_bit_idx;
	u32             mask;
	u32             *enable_cntmask;
	u8              flags;
	spinlock_t      *lock;
};

#define to_clk_dir(_hw) container_of(_hw, struct clk_dir, hw)

static int clk_dir_enable(struct clk_hw *hw)
{
	struct clk_dir *dir = to_clk_dir(hw);
	u32 reg;
	unsigned long flags = 0;
	int ret = 0;

	if (dir->lock)
		spin_lock_irqsave(dir->lock, flags);

	reg = readl(dir->reg);

	if (reg & dir->mask << dir->ck_bit_idx) {
		pr_err("clk-dir: reg value: 0x%02x, check bit: 0x%02x is asserted, clk %s enable failed\n",
			reg, dir->mask << dir->ck_bit_idx,
			__clk_get_name(hw->clk));
		ret = -EINVAL;
		goto out;
	}

	*dir->enable_cntmask ^= dir->mask << dir->bit_idx;

	reg |= dir->mask << dir->bit_idx;
	writel(reg, dir->reg);

out:
	if (dir->lock)
		spin_unlock_irqrestore(dir->lock, flags);

	return ret;
}

static void clk_dir_disable(struct clk_hw *hw)
{
	struct clk_dir *dir = to_clk_dir(hw);
	u32 reg;
	unsigned long flags = 0;

	if (dir->lock)
		spin_lock_irqsave(dir->lock, flags);

	*dir->enable_cntmask ^= dir->mask << dir->bit_idx;

	reg = readl(dir->reg);
	reg &= ~((*dir->enable_cntmask & (dir->mask << dir->bit_idx)) ^
			(dir->mask << dir->bit_idx));
	writel(reg, dir->reg);

	if (dir->lock)
		spin_unlock_irqrestore(dir->lock, flags);
}

static int clk_dir_is_enabled(struct clk_hw *hw)
{
	u32 reg;
	struct clk_dir *dir = to_clk_dir(hw);

	reg = readl(dir->reg);

	if (((reg >> dir->bit_idx) & dir->mask) == dir->mask)
		return 1;

	return 0;
}

static struct clk_ops clk_dir_ops = {
	.enable = clk_dir_enable,
	.disable = clk_dir_disable,
	.is_enabled = clk_dir_is_enabled,
};

struct clk *clk_register_dir(struct device *dev, const char *name,
		const char *parent_name, unsigned long flags,
		void __iomem *reg, u8 bit_idx, u8 ck_bit_idx,
		u32 mask, u32 *enable_cntmask, u8 clk_dir_flags,
		spinlock_t *lock)
{
	struct clk_dir *dir;
	struct clk *clk;
	struct clk_init_data init;

	dir = kzalloc(sizeof(struct clk_dir), GFP_KERNEL);
	if (!dir)
		return ERR_PTR(-ENOMEM);

	/* struct clk_dir assignments */
	dir->reg = reg;
	dir->bit_idx = bit_idx;
	dir->ck_bit_idx = ck_bit_idx;
	dir->mask = mask;
	dir->enable_cntmask = enable_cntmask;
	dir->flags = clk_dir_flags;
	dir->lock = lock;

	init.name = name;
	init.ops = &clk_dir_ops;
	init.flags = flags;
	init.parent_names = parent_name ? &parent_name : NULL;
	init.num_parents = parent_name ? 1 : 0;

	dir->hw.init = &init;

	clk = clk_register(dev, &dir->hw);
	if (IS_ERR(clk))
		kfree(dir);

	return clk;
}
