/*
 * Copyright (C) 2013 Mentor Graphics Inc. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Module Name:    mfw_gst_drm_plane.c
 *
 * Description:    Implementation of DRM Plane FB operations
 *
 * Portability:    This code is written for Linux OS and Gstreamer
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <stdbool.h>
#include <fcntl.h>
#include <linux/videodev2.h>
#include <string.h>
#include <stdint.h>
#include <drmhelper.h>
#include "mfw_gst_drm_plane.h"
#include "semaphore.h"

#define DRM_DEVICE_NAME  "/dev/dri/card0"

typedef struct {
	uint32_t handle;
	uint32_t paddr;
} drm_imx_get_phys_t;

struct drm_imx_global_alpha {
	bool     enable;
	uint32_t plane_id;
	uint8_t  value;
};

struct drm_imx_chromakey {
	bool     enable;
	uint32_t plane_id;
	uint32_t key;
};

#define DRM_IMX_GET_PHYS           0x00
#define DRM_IMX_GET_GLOBAL_ALPHA   0x01
#define DRM_IMX_SET_GLOBAL_ALPHA   0x02
#define DRM_IMX_GET_CHROMAKEY      0x03
#define DRM_IMX_SET_CHROMAKEY      0x04

#define DRM_IOCTL_IMX_GET_PHYS						\
	DRM_IOWR(DRM_COMMAND_BASE + DRM_IMX_GET_PHYS,			\
		 drm_imx_get_phys_t)
#define DRM_IOCTL_IMX_GET_GLOBAL_ALPHA					\
	DRM_IOR( DRM_COMMAND_BASE + DRM_IMX_GET_GLOBAL_ALPHA,		\
		 struct drm_imx_global_alpha)
#define DRM_IOCTL_IMX_SET_GLOBAL_ALPHA					\
	DRM_IOW( DRM_COMMAND_BASE + DRM_IMX_SET_GLOBAL_ALPHA,		\
		 struct drm_imx_global_alpha)
#define DRM_IOCTL_IMX_GET_CHROMAKEY					\
	DRM_IOR( DRM_COMMAND_BASE + DRM_IMX_GET_CHROMAKEY,		\
		 struct drm_imx_chromakey)
#define DRM_IOCTL_IMX_SET_CHROMAKEY					\
	DRM_IOW( DRM_COMMAND_BASE + DRM_IMX_SET_CHROMAKEY,		\
		 struct drm_imx_chromakey)

static const char * connector_type[] = {
	[DRM_MODE_CONNECTOR_Unknown] =     "Unknown",
	[DRM_MODE_CONNECTOR_VGA] =         "VGA",
	[DRM_MODE_CONNECTOR_DVII] =        "DVII",
	[DRM_MODE_CONNECTOR_DVID] =        "DVID",
	[DRM_MODE_CONNECTOR_DVIA] =        "DVIA",
	[DRM_MODE_CONNECTOR_Composite] =   "Composite",
	[DRM_MODE_CONNECTOR_SVIDEO] =      "SVIDEO",
	[DRM_MODE_CONNECTOR_LVDS] =        "LVDS",
	[DRM_MODE_CONNECTOR_Component] =   "Component",
	[DRM_MODE_CONNECTOR_9PinDIN] =     "9PinDIN",
	[DRM_MODE_CONNECTOR_DisplayPort] = "DisplayPort",
	[DRM_MODE_CONNECTOR_HDMIA] =       "HDMIA",
	[DRM_MODE_CONNECTOR_HDMIB] =       "HDMIB",
	[DRM_MODE_CONNECTOR_TV] =          "TV",
	[DRM_MODE_CONNECTOR_eDP] =         "eDP",
};
#define NUM_CONNECTOR_TYPES (DRM_MODE_CONNECTOR_eDP + 1)

struct drm_plane_priv {
	struct drm_helper *drmh;
	uint32_t type;
	sem_t* drm_master_sem;
};

sem_t* drmQueryMastership()
{
    int numRetries = 3;
    int waiting_time = 40*1000;/* 40 ms*/
    sem_t* m_drm_master_sem = NULL;

     m_drm_master_sem = sem_open("/drm_master", O_CREAT, 0666, 1);

     while (m_drm_master_sem == SEM_FAILED || m_drm_master_sem == NULL)
     {
         if ((errno == EACCES) && (numRetries > 0))
         {
             GST_WARNING("could not OPEN a drm_master sem, retry cnt: %d\n", numRetries);
             usleep(waiting_time);
             numRetries--;
             m_drm_master_sem = sem_open("/drm_master", O_CREAT, 0666, 1);
         }
         else
         {
             GST_ERROR("Could not OPEN a drm_master sem giving up,"
                                 "errno: %d, retry cnt: %d\n",
                                 errno, numRetries);
             return NULL;
         }
     }
     /*change the permission, so everyone can access
      *ignore the return value:
      *if user root creates a semaphore chmod will fail here*/
     chmod("/dev/shm/sem.drm_master", 0666);

     numRetries = 3;
     while ( sem_trywait(m_drm_master_sem) == -1)
     {
         if (( errno == EAGAIN) && (numRetries > 0))
         {
             /*wait and try again*/
             GST_WARNING("Could not ACQUIERE a drm_master sem,"
                                 "retry cnt: %d\n", numRetries);
             usleep(waiting_time);
             numRetries--;
         }
         else
         {
             GST_ERROR("Could not ACQUIERE a drm_master sem giving up,"
                                 "errno: %d, retry cnt: %d\n",
                                  errno, numRetries);
             sem_close(m_drm_master_sem);
             return NULL;
         }
     }
     /*success*/
     return m_drm_master_sem;
}

/* The public library routines follow */

drm_plane_priv_t *mfw_gst_drm_plane_init(void)
{
	drm_plane_priv_t *priv;

	priv = g_malloc(sizeof(drm_plane_priv_t));
	if (!priv) {
		GST_ERROR("Can not allocate fb private data!\n");
		return NULL;
	}

	priv->drmh = NULL;
	return priv;
}

void mfw_gst_drm_plane_release(drm_plane_priv_t *priv)
{
	g_free(priv);
}

gboolean mfw_gst_drm_plane_open(drm_plane_priv_t *priv, gchar *disp_type,
				gint min_xres, gint min_yres)
{
	struct drm_helper *drmh;
	gint type, ret;

	if (priv->drmh)
		return TRUE; /* already opened successfully */

	/* setup display type from v4l->disp_type string if provided */
	if (disp_type) {
		for (type = 0; type < NUM_CONNECTOR_TYPES; type++)
			if (!strcasecmp(disp_type, connector_type[type]))
				break;
		if (type >= NUM_CONNECTOR_TYPES) {
			GST_DEBUG("Unknown connector type %s\n", disp_type);
			return FALSE;
		}
	} else {
		type = DRM_MODE_CONNECTOR_HDMIA; /* default is HDMIA */
	}

	priv->type = type;

	priv->drm_master_sem = drmQueryMastership();
	if (!priv->drm_master_sem) {
	    GST_ERROR("drmQueryMastership() failed to acquire drm semaphore\n");
	    return FALSE;
	}

	drmh = drm_helper_create(DRM_DEVICE_NAME, type);
	if (!drmh) {

	    sem_post(priv->drm_master_sem);
	    sem_close(priv->drm_master_sem);
	    priv->drm_master_sem = NULL;

	    GST_ERROR("drm_helper_create() failed\n");
		return FALSE;
	}

	if (!drmh->crtc ||
	    drmh->crtc->mode.hdisplay < min_xres ||
	    drmh->crtc->mode.vdisplay < min_yres ||
	    !drmh->overlay) {
		ret = drm_helper_set_mode(drmh, min_xres, min_yres,
					  0, 0, true);
		if (ret) {
                        GST_ERROR("Could not set a video mode\n");
                        goto fail;
                }
        }

	priv->drmh = drmh;

	return TRUE;

fail:
	mfw_gst_drm_plane_close(priv);
	return FALSE;
}

gboolean mfw_gst_drm_plane_close(drm_plane_priv_t *priv)
{
	struct drm_mode_destroy_dumb dreq;
	struct drm_config *dcfg, *tmp;
	gint ret;

	if (!priv->drmh)
		return TRUE; /* already closed */

	ret = drmDropMaster(priv->drmh->drm_fd);
	if (ret < 0) {
	    GST_ERROR("drmDropMaster failed\n");
	}

	if (priv->drm_master_sem) {
	    sem_post(priv->drm_master_sem);
	    sem_close(priv->drm_master_sem);
	    priv->drm_master_sem = NULL;
	}

	drm_helper_cleanup(priv->drmh);
	priv->drmh = NULL;
	return TRUE;
}

gboolean mfw_gst_drm_plane_set_global_alpha(drm_plane_priv_t *priv,
					    gint alphaVal, gboolean enable)
{
	struct drm_imx_global_alpha ga;
	gint ret;

	if (!priv->drmh || !priv->drmh->overlay) {
		GST_ERROR("cannot set global alpha, drm not initialized\n");
		return FALSE;
	}

	ga.plane_id = priv->drmh->overlay->plane_id;
	ga.enable = enable;
	ga.value = alphaVal;

	ret = drmIoctl(priv->drmh->drm_fd, DRM_IOCTL_IMX_SET_GLOBAL_ALPHA, &ga);
	if (ret) {
		GST_ERROR("failed drmIoctl(DRM_IOCTL_IMX_SET_GLOBAL_ALPHA)\n");
		return FALSE;
	}

	return TRUE;
}

gboolean mfw_gst_drm_plane_set_colorkey(drm_plane_priv_t *priv,
					gulong *colorSrc, gboolean enable)
{
	struct drm_imx_chromakey ck;
	gint ret;

	if (!priv->drmh || !priv->drmh->overlay) {
		GST_ERROR("cannot set colorkey, drm not initialized\n");
		return FALSE;
	}

	ck.plane_id = priv->drmh->overlay->plane_id;
	ck.enable = enable;
	ck.key = *colorSrc;

	ret = drmIoctl(priv->drmh->drm_fd, DRM_IOCTL_IMX_SET_CHROMAKEY, &ck);
	if (ret) {
		GST_ERROR("failed drmIoctl(DRM_IOCTL_IMX_SET_CHROMAKEY)\n");
		return FALSE;
	}

	return TRUE;
}


gboolean mfw_gst_drm_plane_get_bounds(drm_plane_priv_t *priv,
				      gint *fullscreen_width,
				      gint *fullscreen_height)
{
	if (!priv->drmh || !priv->drmh->crtc) {
		GST_ERROR("cannot get bounds, drm not initialized "
			  "or no mode set\n");
		return FALSE;
	}

	if (fullscreen_width)
		*fullscreen_width = priv->drmh->crtc->mode.hdisplay;
	if (fullscreen_height)
		*fullscreen_height = priv->drmh->crtc->mode.vdisplay;

	GST_INFO("full screen size:%dx%d\n",
		 *fullscreen_width, *fullscreen_height);

	return TRUE;
}

gboolean mfw_gst_drm_plane_setup(drm_plane_priv_t *priv,
				 gboolean v4l_output,
				 gint v4l_id,
				 struct v4l2_crop *crop)
{
	struct v4l2_rect *rect = &crop->c;
	struct v4l2_capability cap;
	struct v4l2_framebuffer fbuf;
	drm_imx_get_phys_t gpr;
	uint64_t has_dumb;
	unsigned cap_mask;
	gint ret;

	if (!priv->drmh || !priv->drmh->overlay) {
		GST_ERROR("cannot setup plane, drm not initialized, "
			  "or mode not overlay compatible\n");
		return FALSE;
	}

	if (ioctl(v4l_id, VIDIOC_QUERYCAP, &cap)) {
		GST_ERROR("VIDIOC_QUERYCAP failed\n");
		return FALSE;
	}

	cap_mask = v4l_output ? V4L2_CAP_VIDEO_OUTPUT_OVERLAY :
		V4L2_CAP_VIDEO_OVERLAY;
	if (!(cap.capabilities & cap_mask)) {
		GST_ERROR("V4L device is not an %soverlay\n",
			  v4l_output ? "output " : "");
		return FALSE;
	}

	/*
	 * setup the DRM overlay plane. Make the overlay's framebuffer
	 * twice the image size of the overlay, to allow for
	 * double-buffering by v4lsrc preview mode.
	 */
	ret = drm_helper_set_overlay(priv->drmh, rect->width, rect->height,
				     rect->left, rect->top,
				     rect->width, 2 * rect->height, 0);
	if (ret) {
		GST_ERROR("drm_helper_set_overlay() failed\n");
		return FALSE;
	}

	gpr.handle = priv->drmh->ov_fb.handles[0];

	ret = drmIoctl(priv->drmh->drm_fd, DRM_IOCTL_IMX_GET_PHYS, &gpr);
	if (ret) {
		GST_ERROR("failed drmIoctl(DRM_IOCTL_IMX_GET_PHYS)\n");
		return FALSE;
	}

	/* send the DRM plane info to output driver via S_FBUF */
	fbuf.flags = V4L2_FBUF_FLAG_OVERLAY;
	fbuf.fmt.width = rect->width;
	fbuf.fmt.height = rect->height;
	fbuf.fmt.pixelformat = V4L2_PIX_FMT_RGB565;
	fbuf.base = (void *)gpr.paddr;
	fbuf.fmt.bytesperline = priv->drmh->ov_fb.pitches[0];
	fbuf.fmt.sizeimage = priv->drmh->ov_fb.size / 2;

	ret = ioctl(v4l_id, VIDIOC_S_FBUF, &fbuf);
	if (ret < 0) {
		GST_ERROR("VIDIOC_S_FBUF failed\n");
		return FALSE;
	}

	ret = drmDropMaster(priv->drmh->drm_fd);
	if (ret < 0) {
		GST_ERROR("drmDropMaster failed\n");
		return FALSE;
	}

	if (priv->drm_master_sem) {
	    sem_post(priv->drm_master_sem);
	    sem_close(priv->drm_master_sem);
	    priv->drm_master_sem = NULL;
	}

	return TRUE;
}
