/*
 * Copyright 2013 Robert Bosch Car Multimedia GmbH
 * Ulrich Schulz <ulrich.schulz@de.bosch-softtec.com>

 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 *
 */

#include <common.h>
#include <linux/lz77.h>
#include <asm/types.h>

struct token {
	char next;
	uint8_t len:4;
	uint16_t pos:12;
} __packed;

/*
 * memcpy function that can copy overlapped
 * byte regions is needed, memmove doesn't work
 */
static void lz77_memcpy(unsigned char *dest,
	const unsigned char *src, size_t size)
{
	while (size) {
		*dest++ = *src++;
		--size;
	}
}

int lz77_decompress(const unsigned char *src, size_t src_len,
		    unsigned char *dst, size_t *dst_len)
{
	struct token t;
	unsigned int k = 0;
	unsigned int stored_destlen;
	unsigned int stored_srclen;
#if __BYTE_ORDER__ != __ORDER_LITTLE_ENDIAN__
	unsigned int temp;
#endif

	stored_destlen = le32_to_cpu(*(unsigned int *)(src));
	stored_srclen = le32_to_cpu(*(unsigned int *)(src+4));
	src_len -= 8;
	/*
	 * the packed data can be maximal 3 tokens longer,
	 * to get aligned data for optimized packers
	 */
	if ((unsigned int)(src_len - stored_srclen) > 3 * sizeof(struct token))
		return LZ77_E_WRONG_PACKED_LEN;
	src += 8;
	while (src_len > 0) {
#if __BYTE_ORDER__ == __ORDER_LITTLE_ENDIAN__
		memcpy(&t, src, sizeof(t));
#else
		/* suboptimal solution for
		   big endian
		*/
		memcpy(&temp, src, sizeof(t));
		temp = le32_to_cpu(temp);
		memcpy(&t, &temp, sizeof(t));
#endif
		src += sizeof(struct token);
		src_len -= sizeof(struct token);
		if (t.pos != 0) {
			lz77_memcpy(dst + k, dst + k - t.pos, t.len);
			k += t.len;
		}
		dst[k++] = t.next;
	}
	*dst_len = stored_destlen;
	return LZ77_E_OK;
}
