/*
 * Copyright (c) 2009 Daniel Mack <daniel@caiaq.de>
 * Copyright (C) 2010 Freescale Semiconductor, Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 * for more details.
 */

#include <common.h>
#include <usb.h>
#include <errno.h>
#include <linux/compiler.h>
#include <usb/ehci-fsl.h>
#include <asm/io.h>
#include <asm/arch/imx-regs.h>
#include <asm/arch/clock.h>
#include <asm/imx-common/iomux-v3.h>

#include "ehci.h"

#define USB_OTGREGS_OFFSET	0x000
#define USB_H1REGS_OFFSET	0x200
#define USB_H2REGS_OFFSET	0x400
#define USB_H3REGS_OFFSET	0x600
#define USB_OTHERREGS_OFFSET	0x800

#define USBPHY_CTRL_SFTRST			0x80000000
#define USBPHY_CTRL_CLKGATE			0x40000000
#define USBPHY_CTRL_ENUTMILEVEL3		0x00008000
#define USBPHY_CTRL_ENUTMILEVEL2		0x00004000

#define ANADIG_USB_CHRG_DETECT_EN_B		0x00100000
#define ANADIG_USB_CHRG_DETECT_CHK_CHRG_B	0x00080000

#define ANADIG_USB_PLL_480_CTRL_BYPASS		0x00010000
#define ANADIG_USB_PLL_480_CTRL_ENABLE		0x00002000
#define ANADIG_USB_PLL_480_CTRL_POWER		0x00001000
#define ANADIG_USB_PLL_480_CTRL_EN_USB_CLKS	0x00000040


#define UCTRL_OVER_CUR_POL	(1 << 8) /* OTG Polarity of Overcurrent */
#define UCTRL_OVER_CUR_DIS	(1 << 7) /* Disable OTG Overcurrent Detection */
#define UCTRL_PWR_POL	(1 << 9) /* Power Polarity */


struct usbphy {
	u32	pwd;
	u32	pwd_set;
	u32	pwd_clr;
	u32	pwd_tog;
	u32	tx;
	u32	tx_set;
	u32	tx_clr;
	u32	tx_tog;
	u32	rx;
	u32	rx_set;
	u32	rx_clr;
	u32	rx_tog;
	u32	ctrl;
	u32	ctrl_set;
	u32	ctrl_clr;
	u32	ctrl_tog;
	u32	status;
	u32	rsvd0[3];
	u32	debug;
	u32	debug_set;
	u32	debug_clr;
	u32	debug_tog;
	u32	debug0_status;
	u32	rsvd1[3];
	u32	debug1;
	u32	debug1_set;
	u32	debug1_clr;
	u32	debug1_tog;
	u32	version;
};

struct usb_nc {
	u32	otg_ctrl;
	u32	uh1_ctrl;
	u32	uh2_ctrl;
	u32	uh3_ctrl;
	u32	uh2_hsic_ctrl;
	u32	uh3_hsic_ctrl;
	u32	otg_phy_ctrl0;
	u32	uh1_phy_ctrl0;
};

static int port_inited[2] = {0, 0};
static int clk_inited;

static int usbmx6_get_port(int index)
{
#if (!(defined(CONFIG_USB_MAX_CONTROLLER_COUNT)) || \
	(CONFIG_USB_MAX_CONTROLLER_COUNT == 1)) && \
	defined(CONFIG_MXC_USB_PORT)
	return CONFIG_MXC_USB_PORT;
#else
	return index;
#endif
}

static struct usb_ehci *usbmx6_get_ehci(int port)
{
	return (struct usb_ehci *)(USBOH3_USB_BASE_ADDR + (0x200 * port));
}

static struct usbphy *usbmx6_get_usbphy(int port)
{
	return (struct usbphy *)(port ?
		USB_PHY1_BASE_ADDR : USB_PHY0_BASE_ADDR);
}

static void usbmx6_internal_phy_clock_gate(int port, int on)
{
	struct usbphy *phy = usbmx6_get_usbphy(port);

	__raw_writel(USBPHY_CTRL_CLKGATE,
		     (on ? &phy->ctrl_clr : &phy->ctrl_set));
}

static void usbmx6_power_config(int port)
{
	struct anatop_regs *anatop = (struct anatop_regs *)ANATOP_BASE_ADDR;

	/*
	 * The external charger detector needs to be disabled
	 * or the signal at DP will be poor
	 *
	 * The VBUS regulators are enabled in board specific
	 * initialization. The PLL's power and output to usb
	 * need to be enabled at initializtion.
	 */
	switch (port) {
	case 0:	/* OTG */
		__raw_writel(ANADIG_USB_CHRG_DETECT_EN_B |
			     ANADIG_USB_CHRG_DETECT_CHK_CHRG_B,
			     &anatop->usb1_chrg_detect);

		__raw_writel(ANADIG_USB_PLL_480_CTRL_BYPASS,
			     &anatop->usb1_pll_480_ctrl_clr);

		__raw_writel(ANADIG_USB_PLL_480_CTRL_ENABLE |
			     ANADIG_USB_PLL_480_CTRL_POWER |
			     ANADIG_USB_PLL_480_CTRL_EN_USB_CLKS,
			     &anatop->usb1_pll_480_ctrl_set);
		break;
	case 1: /* Host 1 */
		__raw_writel(ANADIG_USB_CHRG_DETECT_EN_B |
			     ANADIG_USB_CHRG_DETECT_CHK_CHRG_B,
			     &anatop->usb2_chrg_detect);

		__raw_writel(ANADIG_USB_PLL_480_CTRL_BYPASS,
			     &anatop->usb2_pll_480_ctrl_clr);

		__raw_writel(ANADIG_USB_PLL_480_CTRL_ENABLE |
			     ANADIG_USB_PLL_480_CTRL_POWER |
			     ANADIG_USB_PLL_480_CTRL_EN_USB_CLKS,
			     &anatop->usb2_pll_480_ctrl_set);
		break;
	default:
		return;
	}
}

static int usbmx6_phy_enable(int port)
{
	struct usbphy *phy = usbmx6_get_usbphy(port);
	struct usb_ehci *ehci = usbmx6_get_ehci(port);

	/* Stop then Reset */
	clrbits_le32(&ehci->usbcmd, EHCI_FSL_USBCMD_RS);
	while (__raw_readl(&ehci->usbcmd) & EHCI_FSL_USBCMD_RS)
		;

	setbits_le32(&ehci->usbcmd, EHCI_FSL_USBCMD_RST);
	while (__raw_readl(&ehci->usbcmd) & EHCI_FSL_USBCMD_RST)
		;

	/* Reset USBPHY module */
	__raw_writel(USBPHY_CTRL_SFTRST, &phy->ctrl_set);
	udelay(10);

	/* Remove CLKGATE and SFTRST */
	__raw_writel((USBPHY_CTRL_CLKGATE | USBPHY_CTRL_SFTRST),
		     &phy->ctrl_clr);
	udelay(10);

	/* Power up the PHY */
	__raw_writel(0, &phy->pwd);

	/* enable FS/LS device */
	__raw_writel((USBPHY_CTRL_ENUTMILEVEL2 | USBPHY_CTRL_ENUTMILEVEL3),
		     &phy->ctrl_set);

	return 0;
}

static void usbmx6_oc_config(int port)
{
	struct usb_nc *regs = (struct usb_nc *)(USBOH3_USB_BASE_ADDR +
						USB_OTHERREGS_OFFSET);
	u32 *reg_ctrl;

	switch (port) {
	case 0:	/* OTG */
		reg_ctrl = &regs->otg_ctrl;
		break;
	case 1:	/* Host 1 */
		reg_ctrl = &regs->uh1_ctrl;
		break;
	default:
		return;
	}

#ifdef CONFIG_MXC_USB_PWR_ACTIVE_HIGH
	/* Power switch has an active-high enable input */
	setbits_le32(reg_ctrl, (UCTRL_OVER_CUR_DIS | UCTRL_PWR_POL));
#else
	setbits_le32(reg_ctrl, UCTRL_OVER_CUR_DIS);
#endif
}

/*
 * The Freescale boot ROM team has confirmed that the DMA access to the
 * IRAM area used by USB and ROM is not disabled in the USB controller
 * when the boot ROM exits (jumping to the customer boot loader).
 *
 * Freescale suggests to disable the USB controller before using the whole
 * IRAM.
 */
void usbmx6_ctrl_reset(int port)
{
	struct usb_ehci *ehci = usbmx6_get_ehci(port);

	/* Stop then Reset */
	clrbits_le32(&ehci->usbcmd, EHCI_FSL_USBCMD_RS);
	while (__raw_readl(&ehci->usbcmd) & EHCI_FSL_USBCMD_RS)
		;

	setbits_le32(&ehci->usbcmd, EHCI_FSL_USBCMD_RST);
	while (__raw_readl(&ehci->usbcmd) & EHCI_FSL_USBCMD_RST)
		;
}

int __weak board_ehci_hcd_init(int port)
{
	return 0;
}

int __weak board_ehci_hcd_stop(int port)
{
	return 0;
}

int ehci_hcd_init(int index, struct ehci_hccr **hccr, struct ehci_hcor **hcor)
{
	int port = usbmx6_get_port(index);
	struct usb_ehci *ehci;

	if ((port < 0) || (port > 1))	/* Support ports 0 and 1 only */
		return -1;

	debug("Starting USB port %d (inited=%d)\n", port, port_inited[port]);

	/* Enable clk once for all controllers */
	if (!clk_inited) {
		debug("Starting USB clock\n");
		enable_usboh3_clk(1);
		mdelay(1);
		clk_inited = 1;
	}

	if (!port_inited[port]) {
		/* Do board specific initialization */
		board_ehci_hcd_init(port);

		usbmx6_power_config(port);
		usbmx6_oc_config(port);
		usbmx6_internal_phy_clock_gate(port, 1);
		usbmx6_phy_enable(port);
	}

	ehci = usbmx6_get_ehci(port);
	*hccr = (struct ehci_hccr *)((uint32_t)&ehci->caplength);
	*hcor = (struct ehci_hcor *)((uint32_t)*hccr +
			HC_LENGTH(ehci_readl(&(*hccr)->cr_capbase)));
	setbits_le32(&ehci->usbmode, CM_HOST);

	__raw_writel(CONFIG_MXC_USB_PORTSC, &ehci->portsc);
	setbits_le32(&ehci->portsc, USB_EN);

	mdelay(10);

	port_inited[port] = 1;

	return 0;
}

int ehci_hcd_stop(int index)
{
	int port = usbmx6_get_port(index);

	if ((port < 0) || (port > 1))	/* Support ports 0 and 1 only */
		return -1;

	debug("Stopping USB port %d\n", port);

	/* Do board specific de-initialization */
	board_ehci_hcd_stop(port);

	port_inited[port] = 0;

	return 0;
}
