/*
 * (C) Copyright 2015
 * Carsten Resch Bosch CarMultimedia GmbH, Carsten.Resch@de.bosch.com
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <common.h>
#include <asm/arch/clock.h>
#include "cmd_asil_mem_test.h"

DECLARE_GLOBAL_DATA_PTR;

/* #define ASIL_TEST_SIMULATE_RAM_ERROR */

#ifndef PAGE_SIZE
#define PAGE_SIZE                       (4096)
#endif
/* Align the END_ADDR to a page boundary */
#define ASIL_TEST_UBOOT_END_ADDR        ((((int) &__bss_end__) + PAGE_SIZE) & \
					  ~(PAGE_SIZE - 1))

#define RAM_TEST_MAGIC_OFFSET           (0x009083F0)

/* Leave 16MB at top it sontains HEAP and STACK */
#define DEFAULT_ASIL_STACK_AREA_START   (CONFIG_SYS_SDRAM_BASE + \
					 gd->ram_size -	\
					 0x01000000)

#define ASIL_TEST_RESULT_FLASH_START     (0x08280000)
#define ASIL_TEST_RESULT_FLASH_END       (0x0829FFFF)

/* When we test the area which contains the pagetable
   we have to use an alternate page_table
*/
#define ASIL_TEST_ALTERNATE_PAGETABLE_ADDR  (0x15000000)

/************************************************************
	Function checks if the RAM test results differ to
	the stored ones. If so, the flashblock will be erased
	and the new results will be stored.
**************************************************************/
void store_ram_test_results(void)
{
	int err;

	/* 0) Check if results have changed. Only store on change */
	if (memcmp((void *) ASIL_TEST_RESULT_FLASH_START,
		   (void *) ERROR_ARRAY_ADDR,
		   ASIL_TEST_ERR_ARR_SIZE) == 0) {
		printf("RAMTEST: Results unchanged. Do not store\n");
		return;
	}

	/* 1) Erase flash sector */
	err = flash_sect_erase(ASIL_TEST_RESULT_FLASH_START,
			       ASIL_TEST_RESULT_FLASH_END);
	if (err != 0) {
		printf("FATAL ERROR: Could not erase flashblock (err=%d)\n",
		       err);
		return;
	}

	/* 2) Copy Data to FLash */
	err = flash_write((char *) ERROR_ARRAY_ADDR,
			  ASIL_TEST_RESULT_FLASH_START,
			  ASIL_TEST_ERR_ARR_SIZE);
	if (err != 0)
		printf("FATAL ERROR: Could not store results in flash\n");

	return;
}

/****************************************************************
	copies the function 'do_test_mem_area' to internal SRAM
****************************************************************/
void copy_test_to_internal_sram(void)
{
	int *start = (int *) do_test_mem_area;
	int *dest  = (int *) ASIL_TEST_FUNC_START;

	while (*start != 0xDEADBEEF)
		*dest++ = *start++;
	test_func = (void *) ASIL_TEST_FUNC_START;

	flush_dcache_range(ASIL_TEST_FUNC_START,
			   (unsigned long) dest);
	return;
}

/**************************************************************
	While testing the area where the pagetable is located
	we have to move it to another place and activate the 'copy'
	in TTBR0.
 **************************************************************/
void move_current_pagetable(int restore)
{

	int *new_pagetable_addr = (int *) ASIL_TEST_ALTERNATE_PAGETABLE_ADDR;

	if (restore) {
		memcpy((void *) gd->tlb_addr,
		       (void *) new_pagetable_addr,
		       gd->tlb_size);
		flush_dcache_range((unsigned long) gd->tlb_addr,
				   (unsigned long) gd->tlb_addr +
				   gd->tlb_size);

		/* Flushing the TLB is not required here
		   Since the Pagetable remains unchanged */
		asm volatile("mcr p15, 0, %0, c2, c0, 0"
			     : : "r" (gd->tlb_addr) : "memory");
	} else {
		memcpy((void *) new_pagetable_addr,
		       (void *) gd->tlb_addr,
		       gd->tlb_size);
		flush_dcache_range((unsigned long) new_pagetable_addr,
				   (unsigned long) new_pagetable_addr +
				   gd->tlb_size);

		/* Flushing the TLB is not required here
		   Since the Pagetable remains unchanged */
		asm volatile("mcr p15, 0, %0, c2, c0, 0"
			     : : "r" (new_pagetable_addr) : "memory");

	}

}

/****************************************************************
	This function calls the test-function
		start_addr:	Start address
		end_addr:	End address
		do_backup:	backup/restore required ?
		move_pagetable: move of pagetable required ?
****************************************************************/
int ram_test(int start_addr, int end_addr, int do_backup, int move_pagetable)
{
	int err;
	unsigned long time;

	if (move_pagetable)
		move_current_pagetable(0);

	time = get_us();

	err = test_func(start_addr,
			end_addr,
			do_backup);
	time = get_us() - time;

	printf("Tested 0x%08X ... 0x%08X : err = %d : took: %lu ms\n",
	       (unsigned int) start_addr,
	       (unsigned int) end_addr,
	       err,
	       time / 1000);
	if (move_pagetable)
		move_current_pagetable(1);

	return err;
}

/*******************************************************************
	This function tests the complete external RAM Area
	the test itself is excuted from internal SRAM.
	The memory ares which are used by uboot are backup'ed during test.
	Therefore the test is split into 4 section:
	1) SDRAM_BASE   - TEXT_BASE    : no backup required (free memory)
	2) TEXT_BASE    - &__bss_end__ : backup required    (uboot area)
	3) &__bss_end__ - STACK_AREA   : no backup required (free memory)
	4) STACK_AREA   - SDRAM_END    : backup required    (stack & heap area)
 *******************************************************************/
static int do_asil_mem_test(cmd_tbl_t *cmdtp, int flag, int argc,
			    char *const argv[])
{
	int err = 0;
	int i;
	int start_addr[4],  end_addr[4];
	int do_backup[4] = {0, 1, 0, 1};
	/* The last part contains the Pagetable as well
	   it has to be moved during test */
	int move_pagetable[4] = {0, 0, 0, 1};
	int *res_array = (int *) ERROR_ARRAY_ADDR;
#ifdef ASIL_TEST_SIMULATE_RAM_ERROR
	int *sim_results = (int *) RAM_TEST_MAGIC_OFFSET;
	sim_results += 3;
#endif /* #ifdef ASIL_TEST_SIMULATE_RAM_ERROR */

	/* Prepare Result array in SRAM */
	/* Initialize with 0xFFFFFFFF to compare */
	/* The results with the flash content    */
	memset((void *) ERROR_ARRAY_ADDR,
	       0xFF,
	       ASIL_TEST_ERR_ARR_SIZE);

	start_addr[0] = CONFIG_SYS_SDRAM_BASE;
	start_addr[1] = end_addr[0] = CONFIG_SYS_TEXT_BASE;
	start_addr[2] = end_addr[1] = ASIL_TEST_UBOOT_END_ADDR;
	start_addr[3] = end_addr[2] = DEFAULT_ASIL_STACK_AREA_START;

	end_addr[3] = (int) (CONFIG_SYS_SDRAM_BASE +
			    gd->ram_size);

	/* Fill the arrays with test patters */
	/* The arrays are used to make use of load- and store */
	/* multiple (ldmia/stmia) */
	int *pattern_arr_1 = (int *) ASIL_TEST_PATTER1;
	int *pattern_arr_2 = (int *) ASIL_TEST_PATTER2;

	for (i = 0; i < 8; i++) {
		pattern_arr_1[i] = ASIL_TEST_PATTERN1;
		pattern_arr_2[i] = ASIL_TEST_PATTERN2;
	}

	/* copy test func to internal sram */
	copy_test_to_internal_sram();
	printf("Execute:  asil_mem_test\n");

	for (i = 0; (i < 4) && (err < ASIL_TEST_MAX_NUM_ERRORS);
	     i++)
		err += ram_test(start_addr[i],
				end_addr[i],
				do_backup[i],
				move_pagetable[i]);

	printf("Finished: do_asil_mem_test=%d\n", err);
#ifdef ASIL_TEST_SIMULATE_RAM_ERROR
	/* This hidden magic is used to force the test
	   to report RAM errors */
	if (*sim_results == 0xACDC1234) {
		err = 2;
		printf("Simulating %d RAM erros ......\n", err);
		res_array[0] = 0x12000020;
		res_array[1] = 0x55555555;
		res_array[2] = 0x55005555;
		res_array[3] = 0x17800400;
		res_array[4] = 0xAAAAAAAA;
		res_array[5] = 0xAABBAAAA;
	}
#endif /* #ifdef ASIL_TEST_SIMULATE_RAM_ERROR */

	if (err == 0)
		*res_array = 0xF1234ACE;

	store_ram_test_results();

	return 0;
}

U_BOOT_CMD(asil_mem_test, 4, 1, do_asil_mem_test,
	   "Test the complete RAM area for reading/writing", "\n");
