/*
 * Copyright (C) 2013 Robert Bosch Car Multimedia GmbH
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <common.h>
#include <asm/arch/system.h>
#include <asm/gpio.h>
#include <spi_flash.h>

#include "boot_opt.h"

#define BOOT_MAGIC_BLOCK1_OFFSET  (0x00160000)
#define BOOT_MAGIC_BLOCK2_OFFSET  (0x00180000)
#define BOOT_RESETCOUNTER_OFFSET  (0x00908015)
#define RAM_TEST_MAGIC_OFFSET     (0x009083F0)

#define RAMTEST_MAGIC1              0x12345678
#define RAMTEST_MAGIC2              0xDEADBEEF

#define DL_UPDATE_STATE_FAILURE     0x2342ABCD
#define DL_MAGIC_BLOCK_VALID        0xDEADBEEF
#define DL_USE_BOOTCHAIN_2          0x5AA501CB

#define CPU_MAX_RESET_DYN	4
#define CPU_MAX_RESET_RECOVERY	7
#define CPU_MAX_RESET		11

struct trDownloadMagic {
	unsigned int validity_magic;
	unsigned char dummy1[12];
	unsigned int counter;
	unsigned char dummy2[12];
	unsigned int recovery_magic;
	unsigned char dummy3[12];
	unsigned int alt_recovery_magic;
	unsigned char dummy4[12];
	unsigned int bootchain_magic;
};

struct ramtest_magic {
	int magic1;
	int magic2;
};

/* Read SW_OPT GPIO */
void get_sw_opt(enum sw_opt_type *sw_opt_result)
{
	struct sw_opt_gpio *sw_opt =
		(struct sw_opt_gpio *)((ocram *)IRAM_BASE_ADDR)->board->sw_opt;
	struct sw_opt_gpio gpio;

	/* read SW_OPT GPIO */
	if (sw_opt) {
		gpio.sw_opt[0] = sw_opt->sw_opt[0];
		gpio.sw_opt[1] = sw_opt->sw_opt[1];
	} else {
		/* no GPIOs configured, use the default ones */
		gpio.sw_opt[0] = IMX_GPIO_NR(7, 12);
		gpio.sw_opt[1] = IMX_GPIO_NR(1, 0);
	}

	gpio_direction_input(gpio.sw_opt[0]);
        gpio_direction_input(gpio.sw_opt[1]);
	*sw_opt_result = (((gpio_get_value(gpio.sw_opt[1]) & 0x1) << 1) |
			 ((gpio_get_value(gpio.sw_opt[0]) & 0x1)));
}

static int device_read(uint offset, int *dest, size_t size)
{
	int ret = 0;
	enum env_type env_dev =
		(enum env_type)((ocram *)IRAM_BASE_ADDR)->board->env_dev;

	switch (env_dev) {
	case ENV_GEN3_DEV_FLASH:
		memcpy(dest,
			(const void *)(CONFIG_SYS_FLASH_BASE + offset),
			size);
		break;
	case ENV_GEN3_DEV_SPI_FLASH:
	{
		struct spi_flash *env_flash = spi_flash_probe(
				CONFIG_ENV_SPI_BUS, CONFIG_ENV_SPI_CS,
				CONFIG_ENV_SPI_MAX_HZ, CONFIG_ENV_SPI_MODE);

		if (env_flash)
			ret = spi_flash_read(env_flash, offset, size, dest);
		else
			ret = 1;
	}
		break;
	default:
		ret = 1;
		break;
	}

	return ret;
}

void print_boot_config(enum enBootOpt boot_opt, int boot_chain)
{
	printf("Boot Option: ");

	switch (boot_opt) {
	case EN_NORMAL_BOOT:
		printf("EN_NORMAL_BOOT");
		break;
	case EN_RECOVERY_DOWNLOAD:
		printf("EN_RECOVERY_DOWNLOAD");
		break;
	case EN_DEVELOPER_BOOT:
		printf("EN_DEVELOPER_BOOT");
		break;
	case EN_BOOTOPT_INVALID:
	default:
		printf("EN_BOOTOPT_INVALID");
		break;
	}
	printf(", boot chain: %d\n", boot_chain);
}

/*
 * Check in which mode we should boot:
 * either one of the following values is returned:
 *      - EN_RECOVERY_DOWNLOAD
 *      - EN_DEVELOPER_BOOT
 *      - EN_NORMAL_BOOT
 *      - EN_BOOTMODE_INVALID
 */
void evaluate_boot_option(enum enBootOpt *boot_opt, int *boot_chain)
{
	struct trDownloadMagic tr_magic_block1;
	struct trDownloadMagic tr_magic_block2;
	struct trDownloadMagic *ptr_valid_magic_block = 0;

	*boot_opt = EN_NORMAL_BOOT;

	/* Now check the boot magics in NOR flash */
	if (device_read(BOOT_MAGIC_BLOCK1_OFFSET,
			(int *)&tr_magic_block1,
			sizeof(struct trDownloadMagic))) {
		printf("Error reading Download magic1\n");
		 *boot_opt = EN_BOOTOPT_INVALID;
		return;
	}

	if (device_read(BOOT_MAGIC_BLOCK2_OFFSET,
			(int *)&tr_magic_block2,
			sizeof(struct trDownloadMagic))) {
		printf("Error reading Download magic2\n");
		*boot_opt = EN_BOOTOPT_INVALID;
		return;
	}

	if ((tr_magic_block1.validity_magic == DL_MAGIC_BLOCK_VALID) &&
		 (tr_magic_block2.validity_magic == DL_MAGIC_BLOCK_VALID)) {
		/* Both magics valid */
		/* Evaluate counter */
		if (tr_magic_block1.counter > tr_magic_block2.counter)
			ptr_valid_magic_block = &tr_magic_block1;
		else
			ptr_valid_magic_block = &tr_magic_block2;
	} else if (tr_magic_block1.validity_magic == DL_MAGIC_BLOCK_VALID) {
		ptr_valid_magic_block = &tr_magic_block1;
	} else if (tr_magic_block2.validity_magic == DL_MAGIC_BLOCK_VALID) {
		ptr_valid_magic_block = &tr_magic_block2;
	} else {
		printf("Error Download magic: Both blocks are invalid\n");
		*boot_opt = EN_BOOTOPT_INVALID;
		return;
	}

	/* Evaluate bootchain: In each case it is bootchain 1 */
	/* Just if magic 'DL_USE_BOOTCHAIN_2' is set it is 2 */
	if (ptr_valid_magic_block->bootchain_magic == DL_USE_BOOTCHAIN_2)
		*boot_chain = 2;

	if (ptr_valid_magic_block->recovery_magic == DL_UPDATE_STATE_FAILURE)
		*boot_opt = EN_RECOVERY_DOWNLOAD;
}

/*
 * Read the reset counter and depending on it:
 * reset_counter == CPU_MAX_RESET_DYN (4)      -> set reset_dyn to 1
 * reset_counter >= CPU_MAX_RESET_RECOVERY (7) -> start recovery download
 * reset_counter >= CPU_MAX_RESET (11)         -> error message & halt
 */
void evaluate_reset_counter(enum enBootOpt *boot_opt, int *reset_dyn)
{
	unsigned char reset_counter;

	reset_counter = *(volatile unsigned char *)BOOT_RESETCOUNTER_OFFSET;

	if (reset_counter >= CPU_MAX_RESET) {
		printf("Error: Reset counter (%i) exceeded. Halt ...\n",
		       reset_counter);
		while(1) {};
	}

	if (reset_counter >= CPU_MAX_RESET_RECOVERY)
		*boot_opt = EN_RECOVERY_DOWNLOAD;

	if (reset_counter == CPU_MAX_RESET_DYN)
		*reset_dyn = 1;
}

/*
 * Read the START_RAM_TEST magicss:
 * magic1 == 0x12345678 && magic2 == 0xDEADBEEF: Start RAM test      : return 1
 * else                                        : don't start RAM test: return 0
 */
int start_ram_test(void)
{
	struct ramtest_magic *magic =
		(struct ramtest_magic *) RAM_TEST_MAGIC_OFFSET;
	if ((magic->magic1 == RAMTEST_MAGIC1) &&
	    (magic->magic2 == RAMTEST_MAGIC2)) {
		/* erase SRAM magic as soon as possible */
		memset((void *) magic,
		       0,
		       sizeof(struct ramtest_magic));
		/* Invalidate cache to make sure that magics are cleared */
		flush_dcache_range((unsigned long) magic,
				   (unsigned long) magic +
				   sizeof(struct ramtest_magic));
		return 1;
	}
	return 0;
}
