/*
 * Copyright (C) 2012 Robert Bosch Car Multimedia GmbH
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <config.h>
#include <asm/arch/imx-regs.h>
#include <asm/arch/system.h>
#include <asm/arch/bosch_board_id.h>

	/*
	 * Initialize the system using the DCD table
	 * void dcd_init(const dcd_config *)
	 */
	.globl dcd_init
dcd_init:
	ldmia	r0!,{r1,r2}
	cmp	r1,#END_OF_TABLE
	bxeq	lr
	str	r2,[r1]
	b	dcd_init

	/*
	 * Read the board ID from GP1/G2 register
	 * unsigned short read_boardid(void)
	 *
	 * Note: Don't touch r6, it holds the lr of the caller function
	 */
	.globl read_boardid
read_boardid:
	/*
	 * We can have two 32-bit locations for the board ID, the GP1 and
	 * GP2 register. We use 16-bit for each ID, so we can store 4
	 * IDs using these 2 x 32-bit. We first ID is stored in the lower
	 * 16-bit of GP1, the second in the upper 16-bits of GP1. The
	 * same is done with GP2 if a third of fourth ID is written.
	 * We start reading at the end, i.e. the upper 16-bit of GP2.
	 * Reading a value of 0x0000 does mean that the board ID in
	 * GPx isn't written. Start to read GP2[31-16], and if this is 0,
	 * go on with GP2[15-0] etc until an ID != 0 is found. If an ID
	 * is found, the checksum is validated. If the checksum is verfied
	 * it's assumed that a valid board ID is detected.
	 */
	ldr	r0, =OCOTP_BASE_ADDR
	ldr	r1, [r0, #GP2]
	lsr	r2, r1, #16
	mov	r5, #4
	cmp	r2, #0			/* GP2[31-16] != 0 ? */
	bne	id_validateupper
	mov	r5, #3
	cmp	r1, #0			/* GP2[15-0] != 0 ? */
	bne	id_validate
	ldr	r1, [r0, #GP1]
	lsr	r2, r1, #16
	mov	r5, #2
	cmp	r2, #0			/* GP1[31-16] != 0 ? */
	bne	id_validateupper
	mov	r5, #1
	cmp	r1, #0			/* GP1[15-0] != 0 ? */
	bne	id_validate
	mov	r5, #0
	beq	no_id

	/*
	 * Extract the checksum bits in the upper nibble [15-12],
	 * invert it and count the 1's in the ID bits [11-0]
	 */
id_validateupper:
	mov	r1, r2
id_validate:
	lsr	r2, r1, #12
	mvn	r0, r2
	mov	r2, #0xF
	and	r0, r0, r2		/* r0: 4 bit inverted checksum == # of 1's */
	add	r2, r2, #0xFF0
	and	r1, r1, r2		/* r1: 12 bit board ID */

	/* Count the 1's in r1 */
	mov	r2, #0
	mov	r3, r1
count_loop:
	lsrs	r3, r3, #1
	addcs	r2, r2, #1
	bne	count_loop

	cmp	r0, r2
	bne	id_error		/* The checksum is wrong */
	mov	r0, r1

	/* Store the ID found in the OCRAM */
	ldr	r1, =IRAM_BASE_ADDR
	strh	r0, [r1]
	strh	r5, [r1, #2]
	
	/* Check the list of known IDs and try to match the configuration */
decode_id:

#ifdef CONFIG_BOARD_NISSAN_AIVI_A
	ldr	r1, =BOARD_ID_NISSAN_AIVI_A
	cmp	r0, r1
	ldreq	r1, =board_data_nissanaivia
	beq	id_found
#endif

#ifdef CONFIG_BOARD_VWMIBC0
	ldr	r1, =BOARD_ID_VWMIB_C0
	cmp	r0, r1
	ldreq	r1, =board_data_vwmibc0
	beq	id_found
#endif

#ifdef CONFIG_BOARD_VWMIBB0
	ldr	r1, =BOARD_ID_VWMIB_B0
	cmp	r0, r1
	ldreq	r1, =board_data_vwmibb0
	beq	id_found
#endif

#ifdef CONFIG_BOARD_VWMIBB
	ldr	r1, =BOARD_ID_VWMIB_B
	cmp	r0, r1
	ldreq	r1, =board_data_vwmibb
	beq	id_found
#endif

#ifdef CONFIG_BOARD_SUZUKISLNC
	ldr	r1, =BOARD_ID_SUZUKISLN_C
	cmp	r0, r1
	ldreq	r1, =board_data_suzukislnc
	beq	id_found

	ldr	r1, =BOARD_ID_SUZUKISLN_C_MX6DL
	cmp	r0, r1
	ldreq	r1, =board_data_suzukislnc_dl
	beq	id_found
#endif

#ifdef CONFIG_BOARD_PSARCCB2
	ldr	r1, =BOARD_ID_PSARCCA1_B2
	cmp	r0, r1
	ldreq	r1, =board_data_psarcca1b2
	beq	id_found

	ldr	r1, =BOARD_ID_PSARCCA2_B2
	cmp	r0, r1
	ldreq	r1, =board_data_psarcca2b2
	beq	id_found
#endif

#ifdef CONFIG_BOARD_PSARCCB3
	ldr	r1, =BOARD_ID_PSARCCA1_B3
	cmp	r0, r1
	ldreq	r1, =board_data_psarcca1b3
	beq	id_found

	ldr	r1, =BOARD_ID_PSARCCA2_B3
	cmp	r0, r1
	ldreq	r1, =board_data_psarcca2b3
	beq	id_found
#endif

#ifdef CONFIG_BOARD_ENA_A
	ldr	r1, =BOARD_ID_ENA_A
	cmp	r0, r1
	ldreq	r1, =board_data_enaa
	beq	id_found
#endif

#ifdef CONFIG_BOARD_JACC2_B3
	ldr	r1, =BOARD_ID_JACC2B3
	cmp	r0, r1
	ldreq	r1, =board_data_jacc2b3
	beq	id_found
#endif

#ifdef CONFIG_BOARD_JACC2_C
	ldr	r1, =BOARD_ID_JACC2C
	cmp	r0, r1
	ldreq	r1, =board_data_jacc2c
	beq	id_found
#endif

#ifdef CONFIG_BOARD_JACC2_D
	ldr	r1, =BOARD_ID_JACC2D
	cmp	r0, r1
	ldreq	r1, =board_data_jacc2d
	beq	id_found
	ldr	r1, =BOARD_ID_JACC2D_MX6DL
	cmp	r0, r1
	ldreq	r1, =board_data_jacc2d_dl
	beq	id_found
#endif

#ifdef CONFIG_BOARD_JACM6_B0
	ldr	r1, =BOARD_ID_JACM6B0
	cmp	r0, r1
	ldreq	r1, =board_data_jacm6b0
	beq	id_found
#endif

#ifdef CONFIG_BOARD_FSL_SABRELITE
	ldr	r1, =BOARD_ID_FSL_SABRELITE
	cmp	r0, r1
	ldreq	r1, =board_data_sabrelite
	beq	id_found
#endif

#ifdef CONFIG_BOARD_FSL_SABREAI_MX6S
	ldr	r1, =BOARD_ID_FSL_SABREAI_MX6S
	cmp	r0, r1
	ldreq	r1, =board_data_sabreais
	beq	id_found
#endif

#ifdef CONFIG_BOARD_FSL_SABREAI_MX6DL
	ldr	r1, =BOARD_ID_FSL_SABREAI_MX6DL
	cmp	r0, r1
	ldreq	r1, =board_data_sabreaidl
	beq	id_found
#endif

#ifdef CONFIG_BOARD_FSL_SABREAI_MX6Q
	ldr	r1, =BOARD_ID_FSL_SABREAI_MX6Q
	cmp	r0, r1
	ldreq	r1, =board_data_sabreaiq
	beq	id_found
#endif

#ifdef CONFIG_BOARD_VRNS_B1_MX6Q
	ldr	r1, =BOARD_ID_VRNS_B1_MX6Q
	cmp	r0, r1
	ldreq	r1, =board_data_vrnsb1q
	beq	id_found
#endif

#ifdef CONFIG_BOARD_VRNS_B1_MX6S
	ldr	r1, =BOARD_ID_VRNS_B1_MX6S
	cmp	r0, r1
	ldreq	r1, =board_data_vrnsb1s
	beq	id_found
#endif

#ifdef CONFIG_BOARD_VRNS_B2_MX6Q
	ldr	r1, =BOARD_ID_VRNS_B2_MX6Q
	cmp	r0, r1
	ldreq	r1, =board_data_vrnsb2q
	beq	id_found
#endif

#ifdef CONFIG_BOARD_VRNS_B2_MX6S
	ldr	r1, =BOARD_ID_VRNS_B2_MX6S
	cmp	r0, r1
	ldreq	r1, =board_data_vrnsb2s
	beq	id_found
#endif

#ifdef CONFIG_BOARD_VRNS_B2_MX6Q_1GB
	ldr	r1, =BOARD_ID_VRNS_B2_MX6Q_1GB
	cmp	r0, r1
	ldreq	r1, =board_data_vrnsb2q_1gb
	beq	id_found
#endif

#ifdef CONFIG_BOARD_VRNS_B2_MX6S_1GB
	ldr	r1, =BOARD_ID_VRNS_B2_MX6S_1GB
	cmp	r0, r1
	ldreq	r1, =board_data_vrnsb2s_1gb
	beq	id_found
#endif

#ifdef CONFIG_BOARD_ICAMA1
	ldr	r1, =BOARD_ID_ICAMA1
	cmp	r0, r1
	ldreq	r1, =board_data_icama1
	beq	id_found
#endif

#ifdef CONFIG_BOARD_ICAMB1_MX6D
	ldr	r1, =BOARD_ID_ICAMB1_MX6D
	cmp	r0, r1
	ldreq	r1, =board_data_icamb1
	beq	id_found
#endif

#ifdef CONFIG_BOARD_ICAMB2_MX6D
	ldr	r1, =BOARD_ID_ICAMB2_MX6D
	cmp	r0, r1
	ldreq	r1, =board_data_icamb2
	beq	id_found
#endif

#ifdef CONFIG_BOARD_ICAMC1_MX6D
	ldr	r1, =BOARD_ID_ICAMC1_MX6D
	cmp	r0, r1
	ldreq	r1, =board_data_icamc1
	beq	id_found
#endif

#ifdef CONFIG_BOARD_ICAMC3_MX6D
	ldr	r1, =BOARD_ID_ICAMC3_MX6D
	cmp	r0, r1
	ldreq	r1, =board_data_icamc3
	beq	id_found
#endif

#ifdef CONFIG_BOARD_MY16_A_MX6DL
	ldr	r1, =BOARD_ID_MY16_A_MX6DL
	cmp	r0, r1
	ldreq	r1, =board_data_my16a_dl
	beq	id_found
#endif

#ifdef CONFIG_BOARD_MY16_B1B3_MX6DL
	ldr	r1, =BOARD_ID_MY16_B3_G01_MX6DL5845
	cmp	r0, r1
	ldreq	r1, =board_data_my16b1b3_dl
	beq	id_found

	ldr	r1, =BOARD_ID_MY16_B3_G01_MX6DL
	cmp	r0, r1
	ldreq	r1, =board_data_my16b1b3_dl
	beq	id_found
	
	ldr	r1, =BOARD_ID_MY16_B3_G05_MX6DL
	cmp	r0, r1
	ldreq	r1, =board_data_my16b1b3_dl
	beq	id_found

	ldr	r1, =BOARD_ID_MY16_B1_G02_MX6DL
	cmp	r0, r1
	ldreq	r1, =board_data_my16b1b3_dl
	beq	id_found

	ldr	r1, =BOARD_ID_MY16_B1_G03_G04_MX6DL
	cmp	r0, r1
	ldreq	r1, =board_data_my16b1b3_dl
	beq	id_found
#endif

#ifdef CONFIG_BOARD_E1
	ldr	r1, =BOARD_ID_E1
	cmp	r0, r1
	ldreq	r1, =board_data_e1
	beq	id_found
#endif

	b	unknown_id

id_found:	
	/* Store the configuration pointer */
	ldr	r0, =IRAM_BASE_ADDR
	str	r1, [r0, #4]		
	bx	lr

	/* FIXME: Add proper error handling */
no_id:
	mov	r0, #1
	b	error

id_error:
	mov	r0, #2
	b	error
	
unknown_id:
	mov	r0, #3
	b	error

error:
	/*
	 * Error codes in r0:
	 * 1 => No ID found
	 * 2 => ID found, but checksum error
	 * 3 => ID found, checksum correct, but unknown ID (not known to this code) 
	 */
	b	error
