/*
 * Copyright (C) 2012 Mentor Graphics Corp.
 */

/*
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#include <linux/module.h>
#include <linux/of.h>
#include <linux/of_platform.h>
#include <linux/of_i2c.h>
#include <linux/clk.h>
#include <sound/soc.h>

#include "fsl_esai.h"

#define DAI_NAME_SIZE	32

enum imx_cs42888_type {
	IMX53_CS42888,
	IMX6Q_CS42888,
	IMX6DL_CS42888,
};

struct imx_cs42888_data {
	struct snd_soc_dai_link dai;
	struct snd_soc_card card;
	char codec_dai_name[DAI_NAME_SIZE];
	char platform_name[DAI_NAME_SIZE];
	/* struct clk *codec_clk; */
	unsigned int clk_frequency;
	enum imx_cs42888_type devtype;
};

static inline int is_imx53_cs42888(struct imx_cs42888_data *data)
{
	return data->devtype == IMX53_CS42888;
}

static inline int is_imx6q_cs42888(struct imx_cs42888_data *data)
{
	return data->devtype == IMX6Q_CS42888;
}

static inline int is_imx6dl_cs42888(struct imx_cs42888_data *data)
{
	return data->devtype == IMX6DL_CS42888;
}

static unsigned int network;
static unsigned int slave = 1;
static unsigned int mclk_freq[] = { 48000 * 512, 48000 * 512 }; /* HCK */

static struct platform_device_id imx_cs42888_devtype[] = {
	{
		.name = "imx53-audio-cs42888",
		.driver_data = IMX53_CS42888,
	}, {
		.name = "imx6q-audio-cs42888",
		.driver_data = IMX6Q_CS42888,
	}, {
		.name = "imx6dl-audio-cs42888",
		.driver_data = IMX6DL_CS42888,
	}, {
		/* sentinel */
	}
};
MODULE_DEVICE_TABLE(platform, imx_cs42888_devtype);

static const struct of_device_id imx_cs42888_dt_ids[] = {
	{
		.compatible = "fsl,imx53-audio-cs42888",
		.data = &imx_cs42888_devtype[IMX53_CS42888],
	}, {
		.compatible = "fsl,imx6q-audio-cs42888",
		.data = &imx_cs42888_devtype[IMX6Q_CS42888],
	}, {
		.compatible = "fsl,imx6dl-audio-cs42888",
		.data = &imx_cs42888_devtype[IMX6DL_CS42888],
	}, {
		/* sentinel */
	}
};

MODULE_DEVICE_TABLE(of, imx_cs42888_dt_ids);

static int imx_cs42888_init(struct snd_soc_pcm_runtime *rtd)
{
	struct snd_soc_dai *codec_dai = rtd->codec_dai;
	struct snd_soc_dai *cpu_dai = rtd->cpu_dai;
	struct imx_cs42888_data *data = container_of(rtd->card,
					struct imx_cs42888_data, card);

	/* set the ESAI system clock as output */
	if (is_imx53_cs42888(data) /* || machine_is_mx6q_sabreauto() */) {
		snd_soc_dai_set_sysclk(cpu_dai, ESAI_CLK_EXTAL,
				       mclk_freq[network], SND_SOC_CLOCK_OUT);
	} else if (is_imx6q_cs42888(data) /* || is_imx6dl_cs42888(data) */) {
		snd_soc_dai_set_sysclk(cpu_dai, ESAI_CLK_EXTAL_DIV,
				       mclk_freq[network] * 2,
				       slave ? SND_SOC_CLOCK_IN :
				       SND_SOC_CLOCK_OUT);
	}
	/* set the ratio */
	snd_soc_dai_set_clkdiv(cpu_dai, ESAI_TX_DIV_PSR, 1);
	if (is_imx53_cs42888(data) /* || machine_is_mx6q_sabreauto() */)
		snd_soc_dai_set_clkdiv(cpu_dai, ESAI_TX_DIV_PM, 0);
	else if (is_imx6q_cs42888(data) /* || is_imx6dl_cs42888(data) */)
		snd_soc_dai_set_clkdiv(cpu_dai, ESAI_TX_DIV_PM, 0);
	snd_soc_dai_set_clkdiv(cpu_dai, ESAI_RX_DIV_PSR, 1);
	if (is_imx53_cs42888(data) /* || machine_is_mx6q_sabreauto() */)
		snd_soc_dai_set_clkdiv(cpu_dai, ESAI_RX_DIV_PM, 0);
	else if (is_imx6q_cs42888(data) /* || is_imx6dl_cs42888(data) */)
		snd_soc_dai_set_clkdiv(cpu_dai, ESAI_RX_DIV_PM, 0);
	/* set codec Master clock */
	snd_soc_dai_set_sysclk(codec_dai, 0, mclk_freq[0],
			       slave ? SND_SOC_CLOCK_OUT : SND_SOC_CLOCK_IN);
	if (network)
		snd_soc_dai_set_tdm_slot(codec_dai, 0, 0, 8, 0);
	else
		snd_soc_dai_set_tdm_slot(codec_dai, 0, 0, 2, 0);

	return 0;
}

static int imx_cs42888_hw_params(struct snd_pcm_substream *substream,
				 struct snd_pcm_hw_params *params)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_dai *cpu_dai = rtd->cpu_dai;
	struct snd_soc_dai *codec_dai = rtd->codec_dai;
	struct imx_cs42888_data *data = container_of(rtd->card,
					struct imx_cs42888_data, card);
	unsigned int rate = params_rate(params);
	unsigned int lrclk_ratio = 0;
	unsigned int channels = params_channels(params);
	unsigned int tx_mask, rx_mask;

	if (is_imx53_cs42888(data) /* || machine_is_mx6q_sabreauto() */) {
		switch (rate) {
		case 32000:
			lrclk_ratio = 3;
			break;
		case 48000:
			lrclk_ratio = 3;
			break;
		case 64000:
			lrclk_ratio = 1;
			break;
		case 96000:
			lrclk_ratio = 1;
			break;
		case 128000:
			lrclk_ratio = 1;
			break;
		case 44100:
			lrclk_ratio = 3;
			break;
		case 88200:
			lrclk_ratio = 1;
			break;
		case 176400:
			lrclk_ratio = 0;
			break;
		case 192000:
			lrclk_ratio = 0;
			break;
		default:
			pr_info("Rate not support.\n");
			return -EINVAL;
		}
	} else if (is_imx6q_cs42888(data) /* || is_imx6dl_cs42888(data) */) {
		switch (rate) {
		case 32000:
			lrclk_ratio = network ? 3 : 5;
			break;
		case 48000:
			lrclk_ratio = network ? 2 : 5;
			break;
		case 64000:
			lrclk_ratio = network ? 1 : 2;
			break;
		case 96000:
			lrclk_ratio = 2;
			break;
		case 128000:
			lrclk_ratio = 2;
			break;
		case 44100:
			lrclk_ratio = 5;
			break;
		case 88200:
			lrclk_ratio = 2;
			break;
		case 176400:
			lrclk_ratio = 0;
			break;
		case 192000:
			lrclk_ratio = 0;
			break;
		default:
			pr_info("Rate not support.\n");
			return -EINVAL;
		}
	}

	snd_soc_dai_set_clkdiv(cpu_dai, ESAI_TX_DIV_FP, lrclk_ratio);

	snd_soc_dai_set_clkdiv(cpu_dai, ESAI_RX_DIV_FP, lrclk_ratio);

	if (network) {
		tx_mask = rx_mask = 0xffffffff >> (32 - channels);
		snd_soc_dai_set_tdm_slot(cpu_dai, tx_mask, rx_mask, 8, 32);
		snd_soc_dai_set_tdm_slot(codec_dai, tx_mask, rx_mask, 8, 32);
	} else {
		tx_mask = rx_mask = 0xffffffff;
		snd_soc_dai_set_tdm_slot(cpu_dai, tx_mask, rx_mask, 2, 32);
		snd_soc_dai_set_tdm_slot(codec_dai, tx_mask, rx_mask, 2, 32);
	}

	return 0;
}

static struct snd_soc_ops imx_cs42888_ops = {
	.hw_params = imx_cs42888_hw_params,
};

static const struct snd_soc_dapm_widget imx_cs42888_dapm_widgets[] = {
	SND_SOC_DAPM_LINE("Line Out Jack", NULL),
	SND_SOC_DAPM_LINE("Line In Jack", NULL),
};

static int imx_cs42888_probe(struct platform_device *pdev)
{
	struct device_node *esai_np, *codec_np;
	struct platform_device *esai_pdev;
	struct i2c_client *codec_dev;
	struct imx_cs42888_data *data;
	const struct of_device_id *of_id;
	int ret;

	of_id = of_match_device(imx_cs42888_dt_ids, &pdev->dev);
	if (of_id)
		pdev->id_entry = of_id->data;

	esai_np = of_parse_phandle(pdev->dev.of_node, "esai-controller", 0);
	codec_np = of_parse_phandle(pdev->dev.of_node, "audio-codec", 0);
	if (!esai_np || !codec_np) {
		dev_err(&pdev->dev, "phandle missing or invalid\n");
		ret = -EINVAL;
		goto fail;
	}

	esai_pdev = of_find_device_by_node(esai_np);
	if (!esai_pdev) {
		dev_err(&pdev->dev, "failed to find ESAI platform device\n");
		ret = -EPROBE_DEFER;
		goto fail;
	}
	codec_dev = of_find_i2c_device_by_node(codec_np);
	if (!codec_dev) {
		dev_err(&pdev->dev, "failed to find codec platform device\n");
		ret = -EPROBE_DEFER;
		goto fail;
	}

	data = devm_kzalloc(&pdev->dev, sizeof(*data), GFP_KERNEL);
	if (!data) {
		ret = -ENOMEM;
		goto fail;
	}
#if 0
	/* FIXME: should clk depends on I2C clock ? */
	data->codec_clk = clk_get(&codec_dev->dev, NULL);
	if (IS_ERR(data->codec_clk)) {
		/* assuming clock enabled by default */
		data->codec_clk = NULL;
		ret = of_property_read_u32(codec_np, "clock-frequency",
					   &data->clk_frequency);
		if (ret) {
			dev_err(&codec_dev->dev,
				"clock-frequency missing or invalid\n");
			goto fail;
		}
	} else {
		data->clk_frequency = clk_get_rate(data->codec_clk);
		clk_prepare_enable(data->codec_clk);
	}
#endif
	if (of_find_property(esai_np, "fsl,esai-net", NULL))
		network = 1;
	if (of_find_property(esai_np, "fsl,esai-master", NULL))
		slave = 0;

	data->dai.name = "HiFi";
	data->dai.stream_name = "HiFi";
	data->dai.codec_dai_name = "cs42888";
	data->dai.codec_of_node = codec_np;
	data->dai.cpu_dai_name = dev_name(&esai_pdev->dev);
	data->dai.platform_name = "imx-pcm-audio";
	data->dai.ops = &imx_cs42888_ops;
	data->dai.dai_fmt = SND_SOC_DAIFMT_LEFT_J | SND_SOC_DAIFMT_NB_NF |
	    (slave ? SND_SOC_DAIFMT_CBM_CFM : SND_SOC_DAIFMT_CBS_CFS);
	data->dai.init = imx_cs42888_init;

	data->card.dev = &pdev->dev;
	data->devtype = pdev->id_entry->driver_data;

	ret = snd_soc_of_parse_card_name(&data->card, "model");
	if (ret)
		goto clk_fail;

	ret = snd_soc_of_parse_audio_routing(&data->card, "audio-routing");
	if (ret)
		goto clk_fail;
	data->card.num_links = 1;
	data->card.dai_link = &data->dai;
	data->card.dapm_widgets = imx_cs42888_dapm_widgets;
	data->card.num_dapm_widgets = ARRAY_SIZE(imx_cs42888_dapm_widgets);

	ret = snd_soc_register_card(&data->card);
	if (ret) {
		dev_err(&pdev->dev, "snd_soc_register_card failed (%d)\n", ret);
		goto clk_fail;
	}

	platform_set_drvdata(pdev, data);
clk_fail:
	/*
	if (data->codec_clk)
		clk_put(data->codec_clk);
	*/
fail:
	if (esai_np)
		of_node_put(esai_np);
	if (codec_np)
		of_node_put(codec_np);

	return ret;
}

static int imx_cs42888_remove(struct platform_device *pdev)
{
	struct imx_cs42888_data *data = platform_get_drvdata(pdev);

	/*
	if (data->codec_clk) {
		clk_disable_unprepare(data->codec_clk);
		clk_put(data->codec_clk);
	}
	*/
	snd_soc_unregister_card(&data->card);
	return 0;
}

static struct platform_driver imx_cs42888_driver = {
	.driver = {
		   .name = "imx-cs42888",
		   .owner = THIS_MODULE,
		   .of_match_table = imx_cs42888_dt_ids,
		   },
	.probe = imx_cs42888_probe,
	.remove = imx_cs42888_remove,
};

module_platform_driver(imx_cs42888_driver);

MODULE_AUTHOR("valentin_sitdikov@mentor.com");
MODULE_DESCRIPTION("ALSA SoC cs42888 Machine Layer Driver");
MODULE_LICENSE("GPL v2");
MODULE_ALIAS("platform:imx-cs42888");
