/*
 * Freescale SSI ALSA SoC Digital Audio Interface (DAI) driver
 *
 * Author: Timur Tabi <timur@freescale.com>
 *
 * Copyright 2007-2010 Freescale Semiconductor, Inc.
 *
 * This file is licensed under the terms of the GNU General Public License
 * version 2.  This program is licensed "as is" without any warranty of any
 * kind, whether express or implied.
 */

#include <linux/init.h>
#include <linux/io.h>
#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/clk.h>
#include <linux/device.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/of_address.h>
#include <linux/of_irq.h>
#include <linux/of_platform.h>
#include <linux/hrtimer.h>
#include <linux/ktime.h>
#include <linux/spinlock.h>

#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/initval.h>
#include <sound/soc.h>

#include "fsl_ssi.h"
#include "imx-pcm.h"
#include <linux/mxc_asrc_core_internal.h>
#include <dt-bindings/clock/imx6dq-sysclk-ssi.h>

/* defaults settings */
#define SSI_DEFAULT_SLOTS 2
#define SSI_MAX_SLOT_WIDTH 32
#define SSI_DEFAULT_MASK 0xffffffff
#define SSI_DEFAULT_SLOT_WIDTH 16

#define SSI_DEFAULT_STARTUP_TIMER_TIMEOUT_US 200
#define SSI_DEFAULT_MAX_STARTUP_RETRIES   25
#define SSI_DEFAULT_WATERMARK_LEVEL	  8
#define SSI_MAX_TIMER_TIMEOUT		  65535
#define SSI_MAX_TIMER_RETRIES		  65535
#define SSI_MAX_START_FIFO_LEVEL	  15
#define SSI_MIN_WATERMARK_LEVEL		  1
#define SSI_DEFAULT_FIFO_DEPTH		  15
#define SSI_MIN_SDMA_BURST_SIZE		  1

#define SSI_TRANSMIT_ENABLED			0
#define SSI_TRANSMIT_ERROR_STREAM_NULL	       -1
#define SSI_TRANSMIT_ERROR_TRIGGER_TX_DISABLED -2

#define SSI_CLK_DEFAULT_RATE	0
#define MAX_STREAM_NAME_SIZE 64
#define MAX_DAI_NAME_SIZE 64
#define SSI_MAX_CHANNELS 32
#define SSI_MIN_CHANNELS 1
#define SSI_SUPPORT_DAI_NUM	2
#define SSI_PFDRV_NUM_SYNC 1
#define SSI_PFDRV_NUM_ASYNC 2
#ifdef PPC
#define read_ssi(addr)			 in_be32(addr)
#define write_ssi(val, addr)		 out_be32(addr, val)
#define write_ssi_mask(addr, clear, set) clrsetbits_be32(addr, clear, set)
#elif defined ARM
#define read_ssi(addr)			 readl(addr)
#define write_ssi(val, addr)		 writel(val, addr)
/*
 * FIXME: Proper locking should be added at write_ssi_mask caller level
 * to ensure this register read/modify/write sequence is race free.
 */
static inline void write_ssi_mask(u32 __iomem *addr, u32 clear, u32 set)
{
	u32 val = readl(addr);
	val = (val & ~clear) | set;
	writel(val, addr);
}
#endif

/**
 * FSLSSI_I2S_RATES: sample rates supported by the I2S
 */
#define FSLSSI_I2S_RATES (SNDRV_PCM_RATE_5512 | SNDRV_PCM_RATE_8000_192000 | \
			  SNDRV_PCM_RATE_CONTINUOUS)

/**
 * FSLSSI_I2S_FORMATS: audio formats supported by the SSI
 *
 *
 * The SSI has a limitation in that the samples must be in the same byte
 * order as the host CPU.  This is because when multiple bytes are written
 * to the STX register, the bytes and bits must be written in the same
 * order.  The STX is a shift register, so all the bits need to be aligned
 * (bit-endianness must match byte-endianness).  Processors typically write
 * the bits within a byte in the same order that the bytes of a word are
 * written in.  So if the host CPU is big-endian, then only big-endian
 * samples will be written to STX properly.
 */
/**
 * Additionally FSLSSI_I2S_FORMATS and ssi_supported_formats[] should correspond
 * to each other otherwise the formats will be constrained incorrectly.
 */
#ifdef __BIG_ENDIAN
#define FSLSSI_I2S_FORMATS (SNDRV_PCM_FMTBIT_S8 | SNDRV_PCM_FMTBIT_S16_BE | \
	 SNDRV_PCM_FMTBIT_S18_3BE | SNDRV_PCM_FMTBIT_S20_3BE | \
	 SNDRV_PCM_FMTBIT_S24_3BE | SNDRV_PCM_FMTBIT_S24_BE)
static const unsigned int ssi_supported_formats[] = {
	SNDRV_PCM_FORMAT_S8, SNDRV_PCM_FORMAT_S16_BE,
	SNDRV_PCM_FORMAT_S18_3BE, SNDRV_PCM_FORMAT_S20_3BE,
	SNDRV_PCM_FORMAT_S24_3BE, SNDRV_PCM_FORMAT_S24_BE,
};
#else
#define FSLSSI_I2S_FORMATS (SNDRV_PCM_FMTBIT_S8 | SNDRV_PCM_FMTBIT_S16_LE | \
	 SNDRV_PCM_FMTBIT_S18_3LE | SNDRV_PCM_FMTBIT_S20_3LE | \
	 SNDRV_PCM_FMTBIT_S24_3LE | SNDRV_PCM_FMTBIT_S24_LE)
static const unsigned int ssi_supported_formats[] = {
	SNDRV_PCM_FORMAT_S8, SNDRV_PCM_FORMAT_S16_LE,
	SNDRV_PCM_FORMAT_S18_3LE, SNDRV_PCM_FORMAT_S20_3LE,
	SNDRV_PCM_FORMAT_S24_3LE, SNDRV_PCM_FORMAT_S24_LE,
};
#endif

static int fsl_ssi_supported_slot_widths[] = {
	8, 10, 12, 16, 18, 20, 22, 24, 32
};

/* SIER bitflag of interrupts to enable */
#define SIER_FLAGS (CCSR_SSI_SIER_TFRC_EN | CCSR_SSI_SIER_TIE | \
		    CCSR_SSI_SIER_TUE0_EN | CCSR_SSI_SIER_TUE1_EN | \
		    CCSR_SSI_SIER_RFRC_EN | CCSR_SSI_SIER_RIE | \
		    CCSR_SSI_SIER_ROE0_EN | CCSR_SSI_SIER_ROE1_EN)

/**
 * fsl_ssi_private: per-SSI private data
 *
 * @ssi: pointer to the SSI's registers
 * @ssi_phys: physical address of the SSI registers
 * @irq: IRQ of this SSI
 * @first_stream: pointer to the stream that was opened first
 * @second_stream: pointer to second stream
 * @playback: the number of playback streams opened
 * @capture: the number of capture streams opened
 * @cpu_dai: the CPU DAI for this device
 * @dev_attr: the sysfs device attribute structure
 * @stats: SSI statistics
 * @name: name for this device
 */
struct fsl_ssi_private {
	struct ccsr_ssi __iomem *ssi;
	dma_addr_t ssi_phys;
	unsigned int irq;
	struct snd_pcm_substream *first_stream;
	struct snd_pcm_substream *second_stream;
	struct snd_pcm_substream *playback;
	struct snd_pcm_substream *capture;
	unsigned int fifo_depth;
	struct snd_soc_dai_driver cpu_dai_drv[SSI_SUPPORT_DAI_NUM];
	struct device_attribute dev_attr;
	struct platform_device *pdev;

	bool dai_in_xrun[SSI_SUPPORT_DAI_NUM];
	bool new_binding;
	bool ssi_on_imx;
	bool sysclk[SSI_SUPPORT_DAI_NUM];
	struct clk *clk;	/* ssi_ipg */
	struct clk *mclk;	/* ssi_podf */
	struct platform_device *imx_pcm_pdev[SSI_SUPPORT_DAI_NUM];
	unsigned int fmt[SSI_SUPPORT_DAI_NUM];
	int i2s_mode;
	int mode;
	int slots[SSI_SUPPORT_DAI_NUM];
	int slot_width[SSI_SUPPORT_DAI_NUM];
	unsigned int tx_mask;
	unsigned int rx_mask;
	char tx_watermark;
	char rx_watermark;
	unsigned int sync_fixed_slot_width;
	/* Boolean flag to signify if sync open checking should occur */
	bool sync_no_open_busy;
	/* indicate if dividers were set after open */
	bool div[SSI_SUPPORT_DAI_NUM];
#if defined(CONFIG_SND_SOC_IMX_ASRC_DAI_SUPPORT)
	struct fsl_asrc_core_private *asrc_core_private;
	bool ideal_clk_set[2];
#endif
	struct imx_pcm_dma_params dma_params_tx;
	struct imx_pcm_dma_params dma_params_rx;

	struct {
		unsigned int rfrc;
		unsigned int tfrc;
		unsigned int cmdau;
		unsigned int cmddu;
		unsigned int rxt;
		unsigned int rdr1;
		unsigned int rdr0;
		unsigned int tde1;
		unsigned int tde0;
		unsigned int roe1;
		unsigned int roe0;
		unsigned int tue1;
		unsigned int tue0;
		unsigned int tfs;
		unsigned int rfs;
		unsigned int tls;
		unsigned int rls;
		unsigned int rff1;
		unsigned int rff0;
		unsigned int tfe1;
		unsigned int tfe0;
	} stats;

	char ssi_playback_stream_name[MAX_STREAM_NAME_SIZE];
	char ssi_capture_stream_name[MAX_STREAM_NAME_SIZE];
	/* dai_name used in async mode */
	char ssi_playback_dai_name[MAX_DAI_NAME_SIZE];
	char ssi_capture_dai_name[MAX_DAI_NAME_SIZE];

	spinlock_t lock;

	struct hrtimer tx_startup_timer;
	long tx_start_timeout;
	int  tx_start_retries_cnt;
	int  tx_start_max_retries;
	char tx_start_fifo_level;
	bool tx_trigger_enable;
	struct device_attribute dev_attr_timeout;
	struct device_attribute dev_attr_retries;
	struct device_attribute dev_attr_retry_count;
	struct device_attribute dev_attr_fifo_level;
	/*
	*name must be last element for this structure to avoid overwriting,
	*driver allocating extra memory in probe() for dai name.
	*ssi_private = kzalloc(sizeof(struct fsl_ssi_private) + strlen(p),
	*                      GFP_KERNEL);
	*/
	char name[1];
};

static inline int fsl_ssi_active_slots(int slots, unsigned int mask)
{
	return bitmap_weight((const unsigned long *)&mask, slots);
}

static inline int fsl_ssi_mask_calc(struct device *dev,
				struct fsl_ssi_private *ssi,
				int channels, unsigned int *mask, int dai_id)
{
	int active_slots;
	unsigned int used_mask = 0;

	active_slots = fsl_ssi_active_slots(ssi->slots[dai_id], *mask);
	if (channels <= active_slots) {
		int remaining_channels = channels;
		unsigned int mask_bit = 0x1;
		while (remaining_channels > 0) {
			if (*mask & mask_bit) {
				used_mask |= mask_bit;
				remaining_channels -= 1;
			}
			mask_bit = mask_bit << 1;
		}

	} else {
		dev_err(dev, "Channel count %d is not supported\n",
			channels);
		return -EINVAL;
	}

	*mask = used_mask;
	return 0;
}

/**
 * SSI ASRC integration support functions
 */
#if defined(CONFIG_SND_SOC_IMX_ASRC_DAI_SUPPORT)

static int check_for_asrc(struct fsl_ssi_private *ssi_private,
		struct platform_device *pdev) {

	struct platform_device *asrc_core_pdev;
	struct device_node *asrc_core_np = NULL;

	/* get the parent plaform device private data handle */
	asrc_core_np = of_parse_phandle(pdev->dev.of_node,
			"asrc-core", 0);
	if (asrc_core_np) {
		/* retrieve asrc_core pdev handle */
		asrc_core_pdev = of_find_device_by_node(asrc_core_np);
		if (!asrc_core_pdev) {
			dev_err(&pdev->dev,
			"failed to find ASRC core platform device\n");
			return -EINVAL;
		}

		/* initialize data */
		ssi_private->asrc_core_private =
				platform_get_drvdata(asrc_core_pdev);
		dev_info(&pdev->dev, "mxc asrc core found: %p.\n",
				ssi_private->asrc_core_private);
	}

	return 0;
}

static struct platform_device *register_named_platform(
		struct fsl_ssi_private *ssi_private,
		struct platform_device *pdev,
		const char const *pfdrv_name, int platform_id)
{
	return platform_device_register_data(&pdev->dev,
			pfdrv_name, platform_id,
			&ssi_private->asrc_core_private,
			sizeof(struct fsl_asrc_core_private *));
}

static int update_ideal_clock(struct fsl_ssi_private *ssi_private,
		struct snd_pcm_substream *substream,
		struct snd_pcm_hw_params *hw_params, int dai_id)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	char *clock_name = NULL;
	unsigned int bit_clock_rate = 0;
	int ret;

	if (ssi_private->asrc_core_private) {
		/* for ASRC P2P */
		ret = asrc_get_ideal_clkname_from_stream(
			rtd->dai_link->name, &clock_name,
			(substream->stream == SNDRV_PCM_STREAM_PLAYBACK) ?
					ASRC_PLAYBACK : ASRC_CAPTURE);

		if (ret < 0) {
			dev_err(rtd->cpu_dai->dev, "%s: '%s' clk not found",
					__func__, clock_name);
			return ret;
		}

		/* calculate bit clock*/
		bit_clock_rate = params_rate(hw_params)
				* ssi_private->slots[dai_id]
				* ssi_private->slot_width[dai_id];

		ret = asrc_set_ideal_clock_rate(ssi_private->asrc_core_private,
				clock_name, bit_clock_rate);

		kfree(clock_name);
		clock_name = NULL;

		if (ret < 0) {
			dev_err(rtd->cpu_dai->dev, "%s: clk not found or attempting to change rate of already set clock",
					__func__);
			return ret;
		}
		ssi_private->ideal_clk_set[substream->stream] = true;
	}

	return 0;
}

static int invalidate_ideal_clock(struct fsl_ssi_private *ssi_private,
		struct snd_pcm_substream *substream)
{

	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	int ret = -EINVAL;


	if (ssi_private->asrc_core_private) {
		/* for ASRC P2P */
		if (ssi_private->ideal_clk_set[substream->stream]) {
			ret = asrc_invalidate_ideal_clock(
					ssi_private->asrc_core_private,
					rtd->dai_link->name,
					(substream->stream ==
						SNDRV_PCM_STREAM_PLAYBACK) ?
					ASRC_PLAYBACK : ASRC_CAPTURE);
			if (ret == 0)
				ssi_private->ideal_clk_set[substream->stream] =
						false;
		}
	}

	return ret;
}

#else
/**
 * No ASRC integration
 */
static int check_for_asrc(struct fsl_ssi_private *ssi_private,
		struct platform_device *pdev) { return 0; }

static struct platform_device *register_named_platform(
		struct fsl_ssi_private *ssi_private,
		struct platform_device *pdev,
		const char const *pfdrv_name, int platform_id)
{
	return platform_device_register_simple(pfdrv_name,
			platform_id, NULL, 0);
}

static int update_ideal_clock(struct fsl_ssi_private *ssi_private,
		struct snd_pcm_substream *substream,
		struct snd_pcm_hw_params *hw_params, int dai_id)
		{ return 0; }

static int invalidate_ideal_clock(struct fsl_ssi_private *ssi_private,
		struct snd_pcm_substream *substream)
		{ return 0; }
#endif

/* read tx_start_timeout value via sysfs file */
static ssize_t tx_start_timeout_read(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct fsl_ssi_private *ssi_private = container_of(attr,
							 struct fsl_ssi_private,
							 dev_attr_timeout);

	return sprintf(buf, "%ld\n", (ssi_private->tx_start_timeout / 1000));
}

/* write tx_start_timeout value via sysfs file */
static ssize_t tx_start_timeout_write(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
	int ret;
	int n_timeout;
	struct fsl_ssi_private *ssi_private = container_of(attr,
							 struct fsl_ssi_private,
							 dev_attr_timeout);

	if (ssi_private->tx_trigger_enable)
		return -EBUSY;

	ret = kstrtoint(buf, 0, &n_timeout);
	if (ret)
		return ret;

	if ((n_timeout <= 0) || (n_timeout > SSI_MAX_TIMER_TIMEOUT))
		return -EINVAL;

	ssi_private->tx_start_timeout = (n_timeout * 1000);

	return count;
}

/* read tx_start_retries value via sysfs file */
static ssize_t tx_start_retries_read(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct fsl_ssi_private *ssi_private = container_of(attr,
							 struct fsl_ssi_private,
							 dev_attr_retries);

	return sprintf(buf, "%d\n", ssi_private->tx_start_max_retries);
}

/* read tx_start_retry_count value via sysfs file */
static ssize_t tx_start_retry_count_read(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct fsl_ssi_private *ssi_private = container_of(attr,
							 struct fsl_ssi_private,
							 dev_attr_retry_count);

	return sprintf(buf, "%d\n", ssi_private->tx_start_retries_cnt);
}

/* write tx_start_retries value via sysfs file */
static ssize_t tx_start_retries_write(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
	int ret;
	int n_retries;
	struct fsl_ssi_private *ssi_private = container_of(attr,
							 struct fsl_ssi_private,
							 dev_attr_retries);

	if (ssi_private->tx_trigger_enable)
		return -EBUSY;

	ret = kstrtoint(buf, 0, &n_retries);
	if (ret)
		return ret;

	if ((n_retries < 0) || (n_retries > SSI_MAX_TIMER_RETRIES))
		return -EINVAL;

	ssi_private->tx_start_max_retries = n_retries;

	return count;
}

/* read tx_start_fifo_level value via sysfs file */
static ssize_t tx_start_fifo_level_read(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct fsl_ssi_private *ssi_private = container_of(attr,
							 struct fsl_ssi_private,
							 dev_attr_fifo_level);

	return sprintf(buf, "%d\n", ssi_private->tx_start_fifo_level);
}

/* write tx_start_fifo_level value via sysfs file */
static ssize_t tx_start_fifo_level_write(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
	int ret;
	int n_fifo_level;
	struct fsl_ssi_private *ssi_private = container_of(attr,
							 struct fsl_ssi_private,
							 dev_attr_fifo_level);

	if (ssi_private->tx_trigger_enable)
		return -EBUSY;

	ret = kstrtoint(buf, 0, &n_fifo_level);
	if (ret)
		return ret;

	if ((n_fifo_level <= 0) || (n_fifo_level > SSI_MAX_START_FIFO_LEVEL))
		return -EINVAL;

	ssi_private->tx_start_fifo_level = n_fifo_level;

	return count;
}

static inline void fsl_ssi_rise_xrun(struct snd_pcm_substream *substream)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct fsl_ssi_private *ssi_private =
			snd_soc_dai_get_drvdata(rtd->cpu_dai);

	ssi_private->dai_in_xrun[substream->stream] = true;
	snd_soc_pcm_stop(substream, SNDRV_PCM_STATE_XRUN);
}

/**
 * fsl_ssi_set_divs: Sets clock dividers
 *
 * Uses the div_product value passed to calculate the clock divider values and
 * updates the SXCCR
 *
 * @cpu_dai: the cpu dai
 * @sxccr_addr: the clock control register
 * @div_product: the clock divider product value
 */
#define DIV2_BIT_POS 18
#define PSR_BIT_POS  17

#define DIVS_LINE(v1, v2) {         \
		.max  = 256*v1*v2,           \
		.div2 = ((v1 == 1) ? 0 : 1), \
		.psr  = ((v2 == 1) ? 0 : 1), \
		.step = v1*v2 }

static struct divs_elem {
		int max;  /* DIV2 value * PSR value * max PM */
		int div2; /* DIV2 */
		int psr;  /* PSR  */
		int step; /* DIV2 value * PSR value */
} table[] = {
		DIVS_LINE(1, 1),
		DIVS_LINE(2, 1),
		DIVS_LINE(1, 8),
		DIVS_LINE(2, 8),
};

static int fsl_ssi_set_divs(struct snd_soc_dai *cpu_dai,
		void __iomem *sxccr_addr, unsigned int div_product)
{
	struct fsl_ssi_private *ssi_private = snd_soc_dai_get_drvdata(cpu_dai);
	int i;
	int rem;
	int div2;     /* SSIx_SSI_SXCCR[18]: divide-by-two divider bit value */
	int psr;      /* SSIx_SSI_SXCCR[17]: prescalar range bit value */
	int pm;       /* SSIx_SSI_SXCCR[7:0]: prescalar modulus select value */
	__be32 sxccr; /* SSI clock control register value */
	unsigned long flags;

	/* check div_product is zero */
	if (!div_product) {
		dev_err(cpu_dai->dev, "%s: clock divider product value is 0",
				__func__);

		goto err_unsupported;
	}

	/* select the range */
	for (i = 0; i < sizeof(table)/sizeof(struct divs_elem); i++) {
		if (div_product <= table[i].max)
			break;
	}

	/* check if the div_product exceeds the max range provided by the max
	 * divider values */
	if (i == sizeof(table)/sizeof(struct divs_elem)) {
		dev_err(cpu_dai->dev,
				"%s: clock divider product value (%d) value "
				"exceeds MAX divider range",
				__func__, div_product);

		goto err_unsupported;
	}

	/* calculate the dividers */
	pm   = div_product/(table[i].step) - 1;
	rem  = div_product % table[i].step;

	/* case when the bit rate either exceeds MAX bit rate supported or is
	 * not perfect multiple of the step value. Such cases cannot generate
	 * perfect divider values */
	if ((pm > 255) || (rem != 0)) {
		dev_warn(cpu_dai->dev,
				"%s: clock divider product value (%d) hits "
				"hardware limitation", __func__, div_product);
	}

	/* calculate the dividers */
	pm   = (pm > 255) ? 255 : pm; /* adjust to valid range */
	div2 = table[i].div2;
	psr  = table[i].psr;

	/* check if all are zeros */
	if (!(div2 | psr | pm)) {
		dev_err(cpu_dai->dev,
				"%s: all dividers are zero. "
				"Setting least divider.", __func__);
		pm = 1;
	}

	dev_dbg(cpu_dai->dev, "%s: div2 = %d, psr = %d, pm = %d rem = %d",
			__func__, div2, psr, pm, rem);

	spin_lock_irqsave(&ssi_private->lock, flags);
	/* update dividers in the Clock Control Register */
	sxccr = read_ssi(sxccr_addr);
	sxccr &= ~CCSR_SSI_SxCCR_DIV2;
	sxccr |= (div2 << DIV2_BIT_POS);
	sxccr &= ~CCSR_SSI_SxCCR_PSR;
	sxccr |= (psr << PSR_BIT_POS);
	sxccr &= ~CCSR_SSI_SxCCR_PM_MASK;
	sxccr |= pm;
	write_ssi(sxccr, sxccr_addr);
	spin_unlock_irqrestore(&ssi_private->lock, flags);

	return 0;

err_unsupported:
	return -EINVAL;
}

/**
 * fsl_ssi_isr: SSI interrupt handler
 *
 * Although it's possible to use the interrupt handler to send and receive
 * data to/from the SSI, we use the DMA instead.  Programming is more
 * complicated, but the performance is much better.
 *
 * This interrupt handler is used only to gather statistics.
 *
 * @irq: IRQ of the SSI device
 * @dev_id: pointer to the ssi_private structure for this SSI device
 */
static irqreturn_t fsl_ssi_isr(int irq, void *dev_id)
{
	struct fsl_ssi_private *ssi_private = dev_id;
	struct ccsr_ssi __iomem *ssi = ssi_private->ssi;
	irqreturn_t ret = IRQ_NONE;
	__be32 sisr;
	__be32 sisr2 = 0;

	/* We got an interrupt, so read the status register to see what we
	   were interrupted for.  We mask it with the Interrupt Enable register
	   so that we only check for events that we're interested in.
	 */
	sisr = read_ssi(&ssi->sisr) & SIER_FLAGS;

	if (sisr & CCSR_SSI_SISR_RFRC) {
		ssi_private->stats.rfrc++;
		sisr2 |= CCSR_SSI_SISR_RFRC;
		ret = IRQ_HANDLED;
	}

	if (sisr & CCSR_SSI_SISR_TFRC) {
		ssi_private->stats.tfrc++;
		sisr2 |= CCSR_SSI_SISR_TFRC;
		ret = IRQ_HANDLED;
	}

	if (sisr & CCSR_SSI_SISR_CMDAU) {
		ssi_private->stats.cmdau++;
		ret = IRQ_HANDLED;
	}

	if (sisr & CCSR_SSI_SISR_CMDDU) {
		ssi_private->stats.cmddu++;
		ret = IRQ_HANDLED;
	}

	if (sisr & CCSR_SSI_SISR_RXT) {
		ssi_private->stats.rxt++;
		ret = IRQ_HANDLED;
	}

	if (sisr & CCSR_SSI_SISR_RDR1) {
		ssi_private->stats.rdr1++;
		ret = IRQ_HANDLED;
	}

	if (sisr & CCSR_SSI_SISR_RDR0) {
		ssi_private->stats.rdr0++;
		ret = IRQ_HANDLED;
	}

	if (sisr & CCSR_SSI_SISR_TDE1) {
		ssi_private->stats.tde1++;
		ret = IRQ_HANDLED;
	}

	if (sisr & CCSR_SSI_SISR_TDE0) {
		ssi_private->stats.tde0++;
		ret = IRQ_HANDLED;
	}

	if (sisr & CCSR_SSI_SISR_ROE1) {
		ssi_private->stats.roe1++;
		sisr2 |= CCSR_SSI_SISR_ROE1;
		fsl_ssi_rise_xrun(ssi_private->capture);
		ret = IRQ_HANDLED;
	}

	if (sisr & CCSR_SSI_SISR_ROE0) {
		ssi_private->stats.roe0++;
		sisr2 |= CCSR_SSI_SISR_ROE0;
		fsl_ssi_rise_xrun(ssi_private->capture);
		ret = IRQ_HANDLED;
	}

	if (sisr & CCSR_SSI_SISR_TUE1) {
		ssi_private->stats.tue1++;
		sisr2 |= CCSR_SSI_SISR_TUE1;
		fsl_ssi_rise_xrun(ssi_private->playback);
		ret = IRQ_HANDLED;
	}

	if (sisr & CCSR_SSI_SISR_TUE0) {
		ssi_private->stats.tue0++;
		sisr2 |= CCSR_SSI_SISR_TUE0;
		fsl_ssi_rise_xrun(ssi_private->playback);
		ret = IRQ_HANDLED;
	}

	if (sisr & CCSR_SSI_SISR_TFS) {
		ssi_private->stats.tfs++;
		ret = IRQ_HANDLED;
	}

	if (sisr & CCSR_SSI_SISR_RFS) {
		ssi_private->stats.rfs++;
		ret = IRQ_HANDLED;
	}

	if (sisr & CCSR_SSI_SISR_TLS) {
		ssi_private->stats.tls++;
		ret = IRQ_HANDLED;
	}

	if (sisr & CCSR_SSI_SISR_RLS) {
		ssi_private->stats.rls++;
		ret = IRQ_HANDLED;
	}

	if (sisr & CCSR_SSI_SISR_RFF1) {
		ssi_private->stats.rff1++;
		ret = IRQ_HANDLED;
	}

	if (sisr & CCSR_SSI_SISR_RFF0) {
		ssi_private->stats.rff0++;
		ret = IRQ_HANDLED;
	}

	if (sisr & CCSR_SSI_SISR_TFE1) {
		ssi_private->stats.tfe1++;
		ret = IRQ_HANDLED;
	}

	if (sisr & CCSR_SSI_SISR_TFE0) {
		ssi_private->stats.tfe0++;
		ret = IRQ_HANDLED;
	}

	/* Clear the bits that we set */
	if (sisr2)
		write_ssi(sisr2, &ssi->sisr);

	return ret;
}

static int fsl_ssi_start_transmit(struct fsl_ssi_private *ssi_private)
{
	struct snd_pcm_substream *substream = ssi_private->playback;
	struct ccsr_ssi __iomem *ssi = ssi_private->ssi;

	if (!substream)
		return SSI_TRANSMIT_ERROR_STREAM_NULL;

	if (!ssi_private->tx_trigger_enable)
		return SSI_TRANSMIT_ERROR_TRIGGER_TX_DISABLED;

	write_ssi_mask(&ssi->scr, 0, CCSR_SSI_SCR_TE);

	return SSI_TRANSMIT_ENABLED;
}

/*
 * Callback for our startup timer. This will check
 * for data count in FIFO before starting up the
 * transfers.
 * Do not call hrtimer_cancel for this timer with the ssi_private->lock
 * held otherwise the system will deadlock if the timer
 * is running.
 */
static enum hrtimer_restart tx_startup_timer_callback(struct hrtimer *timer)
{
	int ret = HRTIMER_RESTART;
	struct fsl_ssi_private *ssi_private = container_of(timer,
							struct fsl_ssi_private,
							tx_startup_timer);
	struct ccsr_ssi __iomem *ssi = ssi_private->ssi;
	char tx_fifo_cnt = CCSR_SSI_SFCSR_TFCNT0(read_ssi(&ssi->sfcsr));

	ssi_private->tx_start_retries_cnt++;

	if ((ssi_private->tx_start_retries_cnt >=
		ssi_private->tx_start_max_retries)
	     || (tx_fifo_cnt >= ssi_private->tx_start_fifo_level)) {
		unsigned long flags_spin;

		/*
		 * Notify via dmesg if we couldn't
		 * fix the underrun by waiting
		 */
		if (!(tx_fifo_cnt >= ssi_private->tx_start_fifo_level)) {
			dev_warn(&ssi_private->pdev->dev,
				"unable to fill TX fifo : trying to start...");
		}

		spin_lock_irqsave(&ssi_private->lock, flags_spin);
		fsl_ssi_start_transmit(ssi_private);
		spin_unlock_irqrestore(&ssi_private->lock, flags_spin);

		ret = HRTIMER_NORESTART;
	} else {
		ktime_t ktime_now =
			hrtimer_cb_get_time(&ssi_private->tx_startup_timer);
		ktime_t ktime_int =
			ktime_set(0, ssi_private->tx_start_timeout);
		hrtimer_forward(&ssi_private->tx_startup_timer,
				ktime_now, ktime_int);
	}

	return ret;
}

/*
 * Convert hw_mode set to SSI_SCR register to I2S mode enum
 *
 * @hw_mode: the hardware mode
 */
static enum i2s_mode hw_mode_to_i2s_mode(u32 hw_mode)
{
	int ret = IMX_SSI_I2S_NOT_SET;

	if (hw_mode == CCSR_SSI_SCR_I2S_MODE_NORMAL)
		ret = IMX_SSI_I2S_NORMAL;
	else if (hw_mode == CCSR_SSI_SCR_I2S_MODE_MASTER)
		ret = IMX_SSI_I2S_MASTER;
	else
		ret = IMX_SSI_I2S_SLAVE;

	return ret;
}

/*
 * fsl_ssi_resolve_hw_mode: generates the SSI hw mode
 *
 * Generates the hardware mode in SYNC/ASYNC/ASYNC-I2S mode by checking
 * the various parameters and limitations.
 *
 *    [SYNC mode]
 *        [slot-width=32]
 *        [dir=CBS_CFS]
 *            [format=I2S/LEFT_J]
 *                mode=I2S_Master
 *                [Constraint: format-data-length<=slot-width]
 *        [dir=CBM_CFM]
 *            [format=I2S/LEFT_J]
 *                mode=I2S_Slave
 *                [Constraint: format-data-length<=slot-width]
 *                [Constraint: channels==slots]
 *                [Limitation: LEFT_J playback not supported]
 *                [Limitation: Masks not supported]
 *        [other]
 *            [NOT SUPPORTED]
 *
 *        [slot-width!=32]
 *        [dir=CBS_CFS/CBS_CFM/CBM_CFS]
 *            [format=I2S/LEFT_J/DSP_A/DSP_B]
 *                mode=I2S_Normal+Network
 *                [Constraint: slot-width==format-data-length]
 *        [dir=CBM_CFM]
 *            [format=I2S/LEFT_J]
 *                mode=I2S_Slave
 *                [Constraint: format-data-length<=slot-width]
 *                [Constraint: channels==slots]
 *                [Limitation: LEFT_J playback not supported]
 *                [Limitation: Masks not supported]
 *            [format=DSP_A/DSP_B]
 *                mode=I2S_Normal+Network
 *                [Constraint: slot-width==format-data-length]
 *
 *    [ASYNC mode]
 *        [slot-width=32]
 *        [all dir]
 *            [NOT SUPPORTED]
 *
 *        [slot-width!=32]
 *        [all dir]
 *            [all format]
 *                mode=I2S_Normal+Network
 *                [Constraint: slot-width==format-data-length]
 *
 *    [ASYNC-I2S mode]
 *        [slot-width=32]
 *        [dir=CBS_CFS]
 *            [format=I2S/LEFT_J]
 *                mode=I2S_Master
 *                [Constraint: format-data-length<=slot-width]
 *        [dir=CBM_CFM]
 *            [format=I2S/LEFT_J]
 *                mode=I2S_Slave
 *                [Constraint: format-data-length<=slot-width]
 *                [Constraint: channels==slots]
 *                [Limitation: LEFT_J playback not supported]
 *                [Limitation: Masks not supported]
 *        [other]
 *            [NOT SUPPORTED]
 *
 *        [slot-width!=32]
 *        [dir=CBM_CFM]
 *            [format=I2S/LEFT_J]
 *                mode=I2S_Slave
 *                [Constraint: format-data-length<=slot-width]
 *                [Constraint: channels==slots]
 *                [Limitation: LEFT_J playback not supported]
 *                [Limitation: Masks not supported
 *        [other]
 *            [NOT SUPPORTED]
 *
 * @substream: the pcm substream
 * @hw_mode: the hardware mode
 */
static int fsl_ssi_resolve_hw_mode(struct snd_pcm_substream *substream,
				   u32 *hw_mode)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_dai *cpu_dai = rtd->cpu_dai;
	struct fsl_ssi_private *ssi_private =
			snd_soc_dai_get_drvdata(cpu_dai);
	struct device *dev = &ssi_private->pdev->dev;
	int fmt = ssi_private->fmt[cpu_dai->id];
	int slot_width = ssi_private->slot_width[cpu_dai->id];
	unsigned long flags;

	*hw_mode = CCSR_SSI_SCR_I2S_MODE_NORMAL;

	/* slot-width=32 */
	if (slot_width == SSI_MAX_SLOT_WIDTH) {
		/* only I2S and LJF supported */
		if (!(((fmt & SND_SOC_DAIFMT_FORMAT_MASK) == SND_SOC_DAIFMT_I2S)
				|| ((fmt & SND_SOC_DAIFMT_FORMAT_MASK)
						== SND_SOC_DAIFMT_LEFT_J))) {
			dev_err(dev,
				"unsupported format, only I2S/LJF format "
				"supported for slot-width=32");
			return -EINVAL;
		}

		switch (fmt & SND_SOC_DAIFMT_MASTER_MASK) {
		case SND_SOC_DAIFMT_CBS_CFS:
			*hw_mode = CCSR_SSI_SCR_I2S_MODE_MASTER;

			break;

		case SND_SOC_DAIFMT_CBM_CFM:
			/* LJF playback won't work, hw limitation */
			if (((fmt & SND_SOC_DAIFMT_FORMAT_MASK) ==
					SND_SOC_DAIFMT_LEFT_J) &&
					(substream->stream ==
					SNDRV_PCM_STREAM_PLAYBACK)) {
					dev_err(dev,
						"LJF playback is not supported "
						"in I2S-Slave HW mode\n");
				return -EINVAL;
			}

			*hw_mode = CCSR_SSI_SCR_I2S_MODE_SLAVE;

			break;

		case SND_SOC_DAIFMT_CBM_CFS:
		case SND_SOC_DAIFMT_CBS_CFM:
			/* not supported */
			dev_err(dev,
				"CBM_CFS or CBS_CFM unsupported for slot-width=32\n");
			return -EINVAL;

		default:
			dev_err(dev,
				"invalid bit-clock/frame-sync clocking direction\n");
			return -EINVAL;
		}
	} else { /* slot_width=8,10,12,16,18,20,22,24 */
		switch (fmt & SND_SOC_DAIFMT_MASTER_MASK) {
		case SND_SOC_DAIFMT_CBS_CFS:
		case SND_SOC_DAIFMT_CBM_CFS:
		case SND_SOC_DAIFMT_CBS_CFM:
			/* use network mode with I2S Normal mode */
			*hw_mode = CCSR_SSI_SCR_I2S_MODE_NORMAL;

			break;

		case SND_SOC_DAIFMT_CBM_CFM:
			switch (fmt & SND_SOC_DAIFMT_FORMAT_MASK) {
			case SND_SOC_DAIFMT_LEFT_J:
				/* LJF playback not supported, hw limitation */
				if (!(ssi_private->mode == IMX_SSI_ASYN) &&
					substream->stream ==
					SNDRV_PCM_STREAM_PLAYBACK) {
					dev_err(dev,
						"LJF playback is not supported "
						"in I2S-Slave HW mode\n");
					return -EINVAL;
				}
			case SND_SOC_DAIFMT_I2S:
				if (ssi_private->mode == IMX_SSI_ASYN)
					*hw_mode = CCSR_SSI_SCR_I2S_MODE_NORMAL;
				else
					*hw_mode = CCSR_SSI_SCR_I2S_MODE_SLAVE;

				break;

			case SND_SOC_DAIFMT_DSP_A:
			case SND_SOC_DAIFMT_DSP_B:
				/* use network mode with I2S Normal mode */
				*hw_mode = CCSR_SSI_SCR_I2S_MODE_NORMAL;

				break;
			}

			break;

		default:
			dev_err(dev, "invalid bit-clock/frame-sync clocking direction\n");
			return -EINVAL;
		}
	}

	spin_lock_irqsave(&ssi_private->lock, flags);

	if (*hw_mode == CCSR_SSI_SCR_I2S_MODE_NORMAL) {
		if (ssi_private->mode == IMX_SSI_ASYN_I2S) {
			spin_unlock_irqrestore(&ssi_private->lock, flags);
			dev_err(dev, "Incorrect format setting : SSI has been configured with ASYNC-I2S, formats requiring SCR.I2S_MODE register set to normal mode are not supported");
			return -EINVAL;
		}
	} else {
		if (ssi_private->mode == IMX_SSI_ASYN) {
			spin_unlock_irqrestore(&ssi_private->lock, flags);
			dev_err(dev, "Incorrect format setting: SSI has been configured to I2S normal mode, can't change to I2S Master/Slave mode in ASYNC mode");
			return -EINVAL;
		}

		if (ssi_private->mode == IMX_SSI_ASYN_I2S) {
			if (ssi_private->i2s_mode != IMX_SSI_I2S_NOT_SET &&
				ssi_private->i2s_mode !=
				hw_mode_to_i2s_mode(*hw_mode)) {
				spin_unlock_irqrestore(&ssi_private->lock,
							flags);
				dev_err(dev, "Incorrect format setting: %s stream is working in i2s-%s mode, can't set %s stream to work in i2s-%s mode in ASYNC-I2S mode",
					(cpu_dai->id ==
						ID_SYNC_PLAYCAP_OR_ASYNC_PLAY)
						? "capture" : "playback",
					(*hw_mode ==
						CCSR_SSI_SCR_I2S_MODE_MASTER)
						? "slave" : "master",
					(cpu_dai->id ==
						ID_SYNC_PLAYCAP_OR_ASYNC_PLAY)
						? "playback" : "capture",
					(*hw_mode ==
						CCSR_SSI_SCR_I2S_MODE_MASTER)
						? "master" : "slave");
				return -EINVAL;
			}
		}
	}

	ssi_private->i2s_mode = hw_mode_to_i2s_mode(*hw_mode);

	spin_unlock_irqrestore(&ssi_private->lock, flags);

	return 0;
}

/*
 * Set format according to format and slot_width and
 * check if settings are correct
 *
 * @substream: the pcm substream
 */
static int fsl_ssi_verify_hw_format_settings(
		struct snd_pcm_substream *substream)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct fsl_ssi_private *ssi_private =
			snd_soc_dai_get_drvdata(rtd->cpu_dai);
	struct ccsr_ssi __iomem *ssi = ssi_private->ssi;
	u32 hw_mode;
	u32 scr;
	int ret;
	unsigned long flags;

	ret = fsl_ssi_resolve_hw_mode(substream, &hw_mode);
	if (ret)
		return ret;

	spin_lock_irqsave(&ssi_private->lock, flags);
	/* set the HW mode */
	scr = read_ssi(&ssi->scr);
	scr &= ~CCSR_SSI_SCR_I2S_MODE_MASK;
	scr |= hw_mode;
	write_ssi(scr, &ssi->scr);
	spin_unlock_irqrestore(&ssi_private->lock, flags);

	return 0;
}

/*
 * SSI Network Mode or TDM slots configuration.
 * Should only be called when port is inactive (i.e. SSIEN = 0).
 */
static int fsl_ssi_set_dai_tdm_slot(struct snd_soc_dai *cpu_dai,
	unsigned int tx_mask, unsigned int rx_mask, int slots, int slot_width)
{
	struct fsl_ssi_private *ssi_private =
		snd_soc_dai_get_drvdata(cpu_dai);
	struct ccsr_ssi __iomem *ssi = ssi_private->ssi;
	u32 scr;
	unsigned long flags;

	/* remember values and set slots to hw */
	if ((slots > 32) || (slots < 0))
		return -EINVAL;
	if ((slot_width > 32) || (slot_width < 0))
		return -EINVAL;

	if (slot_width)
		ssi_private->slot_width[cpu_dai->id] = slot_width;
	if (slots)
		ssi_private->slots[cpu_dai->id] = slots;

	spin_lock_irqsave(&ssi_private->lock, flags);
	/* SSI should be enabled to change the mask */
	scr = read_ssi(&ssi->scr);
	write_ssi_mask(&ssi->scr, 0, CCSR_SSI_SCR_SSIEN);

	if (cpu_dai->id == ID_SYNC_PLAYCAP_OR_ASYNC_PLAY) {
		ssi_private->tx_mask = tx_mask ? tx_mask : ssi_private->tx_mask;
		write_ssi_mask(&ssi->stccr, CCSR_SSI_SxCCR_DC_MASK,
			CCSR_SSI_SxCCR_DC(ssi_private->slots[cpu_dai->id]));
		write_ssi(~(ssi_private->tx_mask), &ssi->stmsk);
	}

	if (cpu_dai->id == ID_ASYNC_CAP ||
	    ssi_private->mode == IMX_SSI_SYN) {
		ssi_private->rx_mask = rx_mask ? rx_mask : ssi_private->rx_mask;
		write_ssi_mask(&ssi->srccr, CCSR_SSI_SxCCR_DC_MASK,
			CCSR_SSI_SxCCR_DC(ssi_private->slots[cpu_dai->id]));
		write_ssi(~(ssi_private->rx_mask), &ssi->srmsk);
	}

	write_ssi(scr, &ssi->scr);
	spin_unlock_irqrestore(&ssi_private->lock, flags);

	return 0;
}

/*
 * SSI DAI format configuration.
 * Should only be called when port is inactive (i.e. SSIEN = 0).
 */
/*
 * SSI DAI format configuration.
 *
 * Configures the SSI for the clock and frame sync polarity
 *
 * @cpu_dai: the cpu dai
 * @fmt:     the format specifications
 * @_sxcr:   pointer to the SSI Configuration register (SRCR/STCR)
 *
 * Returns error on invalid/unsupported format and 0 on success
 *
 * Note: the register bits used are for TX side as the bit positions are same
 * for both TX and RX
 */
static int fsl_ssi_set_dai_format(struct snd_soc_dai *cpu_dai,
		unsigned int fmt, u32 *_sxcr)
{
	u32 sxcr;

	sxcr = *_sxcr;

	/* DAI mode */
	switch (fmt & SND_SOC_DAIFMT_FORMAT_MASK) {
	case SND_SOC_DAIFMT_I2S:
		/* negative frame sync,
		 * falling edge to clock data & rising edge to latch */
		sxcr |= CCSR_SSI_STCR_TFSI | CCSR_SSI_STCR_TSCKP;

		/* word-length frame-sync,
		 * data on rising edge of bclk, frame low 1clk before data */
		sxcr &= ~CCSR_SSI_STCR_TFSL;
		sxcr |= CCSR_SSI_STCR_TEFS;
		break;

	case SND_SOC_DAIFMT_LEFT_J:
		/* positive frame sync,
		 * falling edge to clock data & rising edge to latch */
		sxcr &= ~CCSR_SSI_STCR_TFSI;
		sxcr |= CCSR_SSI_STCR_TSCKP;

		/* word-length frame-sync,
		 * data on rising edge of bclk, frame high with data */
		sxcr &= ~(CCSR_SSI_STCR_TFSL | CCSR_SSI_STCR_TEFS);

		break;

	case SND_SOC_DAIFMT_DSP_A:
		/* positive frame sync,
		 * falling edge to clock data & rising edge to latch */
		sxcr &= ~CCSR_SSI_STCR_TFSI;
		sxcr |= CCSR_SSI_STCR_TSCKP;

		/* bit-length frame-sync,
		 * data on rising edge of bclk, frame high 1clk before data */
		sxcr |= CCSR_SSI_STCR_TFSL | CCSR_SSI_STCR_TEFS;
		break;

	case SND_SOC_DAIFMT_DSP_B:
		/* positive frame sync,
		 * falling edge to clock data & rising edge to latch */
		sxcr &= ~CCSR_SSI_STCR_TFSI;
		sxcr |= CCSR_SSI_STCR_TSCKP;

		/* bit-length frame-sync,
		 * data on rising edge of bclk, frame high with data */
		sxcr |= CCSR_SSI_STCR_TFSL;
		sxcr &= ~CCSR_SSI_STCR_TEFS;
		break;

	default: /* unsupported format */
		dev_err(cpu_dai->dev, "unsupported format (fmt=%X)\n",
				(fmt & SND_SOC_DAIFMT_FORMAT_MASK));
		return -EINVAL;
	}

	/* LSB aligned */
	sxcr |= CCSR_SSI_STCR_TXBIT0;

	/* DAI clock inversion */
	switch (fmt & SND_SOC_DAIFMT_INV_MASK) {
	case SND_SOC_DAIFMT_NB_NF:
		break;

	case SND_SOC_DAIFMT_NB_IF:
		/* inverted frame-sync */
		sxcr ^= CCSR_SSI_STCR_TFSI;
		break;

	case SND_SOC_DAIFMT_IB_NF:
		/* inverted bclk */
		sxcr ^= CCSR_SSI_STCR_TSCKP;
		break;

	case SND_SOC_DAIFMT_IB_IF:
		/* inverted bclk and frame-sync */
		sxcr ^= (CCSR_SSI_STCR_TFSI | CCSR_SSI_STCR_TSCKP);
		break;

	default: /* invalid format */
		dev_err(cpu_dai->dev, "invalid clock inv format (fmt=%X)\n",
				fmt & SND_SOC_DAIFMT_INV_MASK);
		return -EINVAL;
	}

	/* DAI clock master masks */
	switch (fmt & SND_SOC_DAIFMT_MASTER_MASK) {
	case SND_SOC_DAIFMT_CBM_CFM:
		sxcr &= ~(CCSR_SSI_STCR_TFDIR | CCSR_SSI_STCR_TXDIR);
		break;

	case SND_SOC_DAIFMT_CBS_CFM:
		sxcr &= ~CCSR_SSI_STCR_TFDIR;
		sxcr |= CCSR_SSI_STCR_TXDIR;
		break;

	case SND_SOC_DAIFMT_CBM_CFS:
		sxcr |= CCSR_SSI_STCR_TFDIR;
		sxcr &= ~CCSR_SSI_STCR_TXDIR;
		break;

	case SND_SOC_DAIFMT_CBS_CFS:
		sxcr |= CCSR_SSI_STCR_TFDIR | CCSR_SSI_STCR_TXDIR;
		break;

	default: /* invalid format */
		dev_err(cpu_dai->dev, "invalid clock master mask (fmt=%X)\n",
				fmt & SND_SOC_DAIFMT_MASTER_MASK);
		return -EINVAL;
	}

	sxcr |= CCSR_SSI_STCR_TFEN0;

	*_sxcr = sxcr;

	return 0;
}

/*
 * Configures the SSI for the clock and frame sync polarity
 *
 * @cpu_dai: the cpu dai
 * @fmt: the format specifications
 *
 * NOTE: This function needs to be called during open
 */
static int fsl_ssi_set_dai_fmt(struct snd_soc_dai *cpu_dai,
		unsigned int fmt)
{
	struct fsl_ssi_private *ssi_private = snd_soc_dai_get_drvdata(cpu_dai);
	struct ccsr_ssi __iomem *ssi = ssi_private->ssi;
	u32 scr, stcr, srcr;
	int ret;
	unsigned long flags;

	/* restrict format settings if already set */
	if (cpu_dai->active > 0) {
		if (fmt == ssi_private->fmt[cpu_dai->id])
			return 0;

		dev_err(cpu_dai->dev,
		"sync mode - cannot change format another stream active\n");
		dev_err(cpu_dai->dev,
			"requested format: %x, current format: %x\n",
			fmt, ssi_private->fmt[cpu_dai->id]);
		return -EAGAIN;
	}

	ssi_private->fmt[cpu_dai->id] = fmt;

	spin_lock_irqsave(&ssi_private->lock, flags);
	scr = read_ssi(&ssi->scr);

	/* reset */
	if (cpu_dai->id == ID_SYNC_PLAYCAP_OR_ASYNC_PLAY)
		stcr = read_ssi(&ssi->stcr);
	if (cpu_dai->id == ID_ASYNC_CAP || ssi_private->mode == IMX_SSI_SYN)
		srcr = read_ssi(&ssi->srcr);

	scr |= CCSR_SSI_SCR_NET;

	write_ssi(scr, &ssi->scr);

	/* set dai format settings */
	if (cpu_dai->id == ID_SYNC_PLAYCAP_OR_ASYNC_PLAY) {
		/* configure TX side */
		ret = fsl_ssi_set_dai_format(cpu_dai, fmt, &stcr);
		if (ret) {
			spin_unlock_irqrestore(&ssi_private->lock, flags);
			return -EINVAL;
		}

		write_ssi(stcr, &ssi->stcr);
	}
	if (cpu_dai->id == ID_ASYNC_CAP ||
	    ssi_private->mode == IMX_SSI_SYN) {
		/* configure RX side */
		/* FIXME: do we need to set RX registers as well? */
		ret = fsl_ssi_set_dai_format(cpu_dai, fmt, &srcr);
		if (ret) {
			spin_unlock_irqrestore(&ssi_private->lock, flags);
			return -EINVAL;
		}

		/* make sure gated mode is not activated
		 * NOTE: Setting RXDIR in synchronous mode activates
		 * the gated clock*/
		if (ssi_private->mode == IMX_SSI_SYN)
			srcr &= ~(CCSR_SSI_SRCR_RFDIR | CCSR_SSI_SRCR_RXDIR);
		write_ssi(srcr, &ssi->srcr);
	}

	spin_unlock_irqrestore(&ssi_private->lock, flags);
	return 0;
}

/*
 * SSI system clock configuration.
 * Should only be called when port is inactive (i.e. SSIEN = 0).
 */
static int fsl_ssi_set_dai_sysclk(struct snd_soc_dai *cpu_dai,
				  int clk_id, unsigned int freq, int dir)
{
	struct fsl_ssi_private *ssi_private =
		snd_soc_dai_get_drvdata(cpu_dai);
	struct ccsr_ssi __iomem *ssi = ssi_private->ssi;
	u32 scr;
	unsigned long flags;

	if (!(cpu_dai->rate == 0)) {
		dev_warn(cpu_dai->dev,
			"sys clock is already in use, will not be modified\n");
		return -EAGAIN;
	}

	switch (clk_id) {
	case SYSCLK_SSI_FSYS:
		if (ssi_private->mclk) {
			if (freq) {
				int ret;
				ret = clk_set_rate(ssi_private->mclk, freq);
				if (ret) {
					dev_err(cpu_dai->dev,
							"unable to set mclk frequency");
					return ret;
				}
				if (ssi_private->mode != IMX_SSI_SYN)
					dev_warn(cpu_dai->dev, "Common clock FSYS is updated in ASYNC mode");
			}
		}

		spin_lock_irqsave(&ssi_private->lock, flags);
		scr = read_ssi(&ssi->scr);
		if (dir == SYSCLK_SSI_DIR_IN)
			scr &= ~CCSR_SSI_SCR_SYS_CLK_EN;
		else
			if (ssi_private->mode == IMX_SSI_SYN)
				scr |= CCSR_SSI_SCR_SYS_CLK_EN;
			else {
				scr &= ~CCSR_SSI_SCR_SYS_CLK_EN;
				write_ssi(scr, &ssi->scr);
				spin_unlock_irqrestore(&ssi_private->lock,
							flags);
				dev_err(cpu_dai->dev, "Network clock output is not available in asynchronous mode\n");
				return -EINVAL;
			}
		write_ssi(scr, &ssi->scr);
		spin_unlock_irqrestore(&ssi_private->lock, flags);
		break;
	default:
		dev_err(cpu_dai->dev,
			"%d is not a valid clock! expecting SYSCLK_SSI_FSYS (clk_id = 0)\n",
			clk_id);
		return -EINVAL;
	}

	ssi_private->sysclk[cpu_dai->id] = true;

	return 0;
}

/*
 * SSI Clock dividers
 * Should only be called when port is inactive (i.e. SSIEN = 0).
 */
static int fsl_ssi_set_dai_clkdiv(struct snd_soc_dai *cpu_dai,
				  int div_id, int div)
{
	struct fsl_ssi_private *ssi_private =
		snd_soc_dai_get_drvdata(cpu_dai);
	struct ccsr_ssi __iomem *ssi = ssi_private->ssi;
	u32 sxccr;
	u32 __iomem *addr_sxccr = cpu_dai->id == ID_SYNC_PLAYCAP_OR_ASYNC_PLAY
						? &ssi->stccr : &ssi->srccr;
	unsigned long flags;

	spin_lock_irqsave(&ssi_private->lock, flags);
	sxccr = read_ssi(addr_sxccr);

	switch (div_id) {
	case SYSCLK_SSI_DIV_DIV2:
		switch (div) {
		case SYSCLK_SSI_DIV_DIV2_DIVBY_1:
			sxccr &= ~CCSR_SSI_SxCCR_DIV2;
			break;
		case SYSCLK_SSI_DIV_DIV2_DIVBY_2:
			sxccr |= CCSR_SSI_SxCCR_DIV2;
			break;
		default:
			spin_unlock_irqrestore(&ssi_private->lock, flags);
			dev_err(cpu_dai->dev, "DIV2 only supports value 0 or 1");
			return -EINVAL;
		}
		break;
	case SYSCLK_SSI_DIV_PSR:
		switch (div) {
		case SYSCLK_SSI_DIV_PSR_DIVBY_1:
			sxccr &= ~CCSR_SSI_SxCCR_PSR;
			break;
		case SYSCLK_SSI_DIV_PSR_DIVBY_8:
			sxccr |= CCSR_SSI_SxCCR_PSR;
			break;
		default:
			spin_unlock_irqrestore(&ssi_private->lock, flags);
			dev_err(cpu_dai->dev, "PSR only supports value 0 or 1");
			return -EINVAL;
		}
		break;
	case SYSCLK_SSI_DIV_PM:
		if ((div << CCSR_SSI_SxCCR_PM_SHIFT)
				& (~CCSR_SSI_SxCCR_PM_MASK)) {
			spin_unlock_irqrestore(&ssi_private->lock, flags);
			dev_err(cpu_dai->dev, "Too large value for PM.");
			return -EINVAL;
		}
		sxccr &= ~CCSR_SSI_SxCCR_PM_MASK;
		sxccr |= CCSR_SSI_SxCCR_PM(div);
		break;
	default:
		spin_unlock_irqrestore(&ssi_private->lock, flags);
		return -EINVAL;
	}


	write_ssi(sxccr, addr_sxccr);
	spin_unlock_irqrestore(&ssi_private->lock, flags);

	ssi_private->div[cpu_dai->id] = true;

	return 0;
}

static int ssi_get_max_channels(int slots, unsigned int mask)
{
	int max_ch = -EINVAL;

	if (slots < 32)
		max_ch = hweight32(mask & ((1 << slots) - 1));
	else if (slots == 32)
		max_ch = hweight32(mask);

	return max_ch;
}

/*
 * ssi_get_channels_range: gives valid channel range
 *
 * Valid channel range supported as per the slot count and the slot mask
 *
 * @substream: the pcm substream
 * @ch: channel range
 */
static int ssi_get_channels_range(struct snd_pcm_substream *substream,
				  struct snd_interval *ch)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_dai *cpu_dai = rtd->cpu_dai;
	struct fsl_ssi_private *ssi_private =
			snd_soc_dai_get_drvdata(cpu_dai);
	int ret;
	u32 hw_mode;

	ret = fsl_ssi_resolve_hw_mode(substream, &hw_mode);

	if (ret)
		return ret;

	ch->integer = 1;

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK)
		ch->max = ssi_get_max_channels(
				ssi_private->slots[cpu_dai->id],
				ssi_private->tx_mask);
	else
		ch->max = ssi_get_max_channels(
				ssi_private->slots[cpu_dai->id],
				ssi_private->rx_mask);

	/* [Constraint: channels==slots]
	 * In I2S slave mode, masks are not supported implying channels should
	 * be equal to slots */
	if (hw_mode == CCSR_SSI_SCR_I2S_MODE_SLAVE)
		ch->min = ssi_private->slots[cpu_dai->id];
	else
		ch->min = 1;

	return 0;
}

static int ssi_channels_constraint_cb(struct snd_pcm_hw_params *params,
					       struct snd_pcm_hw_rule *rule)
{
	int ret;
	struct snd_interval ch;
	struct snd_pcm_substream *substream = rule->private;
	struct snd_interval *req_ch = hw_param_interval(params,
						   SNDRV_PCM_HW_PARAM_CHANNELS);
	snd_interval_any(&ch);

	ret = ssi_get_channels_range(substream, &ch);

	if (ret)
		return ret;

	return snd_interval_refine(req_ch, &ch);
}

static int ssi_format_constraint_cb(struct snd_pcm_hw_params *params,
					struct snd_pcm_hw_rule *rule)
{
	int i, ret;
	u32 hw_mode;
	struct snd_pcm_substream *substream = rule->private;
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_dai *cpu_dai = rtd->cpu_dai;
	struct fsl_ssi_private *ssi_private =
		snd_soc_dai_get_drvdata(cpu_dai);
	struct snd_mask fmt;
	snd_mask_none(&fmt);

	ret = fsl_ssi_resolve_hw_mode(substream, &hw_mode);

	if (ret)
		return ret;

	/*
	 * For I2S master/slave mode support all formats which have
	 * sample_size <= slot_width other only support formats which have
	 * sample_size == slot_width
	 */
	if ((hw_mode == CCSR_SSI_SCR_I2S_MODE_MASTER) ||
		(hw_mode == CCSR_SSI_SCR_I2S_MODE_SLAVE)) {
		for (i = 0; i < ARRAY_SIZE(ssi_supported_formats); i++) {
			if (ssi_private->slot_width[cpu_dai->id] >=
			    snd_pcm_format_width(ssi_supported_formats[i]))
				snd_mask_set(&fmt, ssi_supported_formats[i]);
		}
	} else {
		for (i = 0; i < ARRAY_SIZE(ssi_supported_formats); i++) {
			if (ssi_private->slot_width[cpu_dai->id] ==
			    snd_pcm_format_width(ssi_supported_formats[i]))
				snd_mask_set(&fmt, ssi_supported_formats[i]);
		}
	}

	ret = snd_mask_refine(hw_param_mask(params, SNDRV_PCM_HW_PARAM_FORMAT),
				&fmt);

	if (ret < 0)
		dev_err(rtd->cpu_dai->dev,
			"invalid combination of slot-width and format-data-width\n");

	return ret;
}

/**
 * fsl_ssi_startup: create a new substream
 *
 * This is the first function called when a stream is opened.
 *
 * If this is the first stream open, then grab the IRQ and program some of
 * the SSI registers.
 */
static int fsl_ssi_startup(struct snd_pcm_substream *substream,
			   struct snd_soc_dai *dai)
{
	unsigned long flags;
	int ret;
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct fsl_ssi_private *ssi_private =
		snd_soc_dai_get_drvdata(rtd->cpu_dai);

	clk_enable(ssi_private->mclk);
	clk_enable(ssi_private->clk);

	spin_lock_irqsave(&ssi_private->lock, flags);
	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK)
		ssi_private->playback = substream;
	else
		ssi_private->capture = substream;
	spin_unlock_irqrestore(&ssi_private->lock, flags);

	/*
	 * Add constraint callback for channels, evaluated when user
	 * tries to set channels
	 */
	ret = snd_pcm_hw_rule_add(substream->runtime, 0,
				  SNDRV_PCM_HW_PARAM_CHANNELS,
				  ssi_channels_constraint_cb, substream,
				  SNDRV_PCM_HW_PARAM_CHANNELS, -1);
	if (ret)
		return ret;

	/*
	 * Add constraint callback for formats, evaluated when user
	 * tries to set format
	 */
	ret = snd_pcm_hw_rule_add(substream->runtime, 0,
				  SNDRV_PCM_HW_PARAM_FORMAT,
				  ssi_format_constraint_cb,
				  substream,
				  SNDRV_PCM_HW_PARAM_FORMAT, -1);
	if (ret)
		return ret;

	if (ssi_private->sync_fixed_slot_width) {
		/*
		 * Add constraint for sample bits, enforced when the user
		 * attempts to change slot width
		 */
		snd_pcm_hw_constraint_minmax(substream->runtime,
				SNDRV_PCM_HW_PARAM_SAMPLE_BITS,
				ssi_private->sync_fixed_slot_width,
				ssi_private->sync_fixed_slot_width);
	}


	spin_lock_irqsave(&ssi_private->lock, flags);
	/*
	 * If this is the first stream opened, then request the IRQ
	 * and initialize the SSI registers.
	 */
	if (!ssi_private->first_stream) {
		struct ccsr_ssi __iomem *ssi = ssi_private->ssi;

		ssi_private->first_stream = substream;
		ssi_private->i2s_mode = IMX_SSI_I2S_NOT_SET;

		/* Enable the interrupts and DMA requests */
		write_ssi(SIER_FLAGS, &ssi->sier);

		/*
		 * Set the watermark for transmit FIFI 0 and receive FIFO 0. We
		 * don't use FIFO 1.  We program the transmit water to signal a
		 * DMA transfer if there are only two (or fewer) elements left
		 * in the FIFO.  Two elements equals one frame (left channel,
		 * right channel).  This value, however, depends on the depth of
		 * the transmit buffer.
		 *
		 * We program the receive FIFO to notify us if at least two
		 * elements (one frame) have been written to the FIFO.  We could
		 * make this value larger (and maybe we should), but this way
		 * data will be written to memory as soon as it's available.
		 */
		write_ssi(CCSR_SSI_SFCSR_TFWM0(ssi_private->tx_watermark) |
			CCSR_SSI_SFCSR_RFWM0(ssi_private->rx_watermark),
			&ssi->sfcsr);

		/*
		 * We keep the SSI disabled because if we enable it, then the
		 * DMA controller will start.  It's not supposed to start until
		 * the SCR.TE (or SCR.RE) bit is set, but it does anyway.  The
		 * DMA controller will transfer one "BWC" of data (i.e. the
		 * amount of data that the MR.BWC bits are set to).  The reason
		 * this is bad is because at this point, the PCM driver has not
		 * finished initializing the DMA controller.
		 */
	} else {
		if (ssi_private->mode == IMX_SSI_SYN) {
			struct snd_pcm_runtime *first_runtime =
				ssi_private->first_stream->runtime;
			/*
			 * This is the second stream open, and we're in
			 * synchronous mode, so we need to impose sample
			 * sample size constraints. This is because STCCR is
			 * used for playback and capture in synchronous mode,
			 * so there's no way to specify different word
			 * lengths.
			 *
			 * Note that this can cause a race condition if the
			 * second stream is opened before the first stream is
			 * fully initialized.  We provide some protection by
			 * checking to make sure the first stream is
			 * initialized, but it's not perfect.  ALSA sometimes
			 * re-initializes the driver with a different sample
			 * rate or size.  If the second stream is opened
			 * before the first stream has received its final
			 * parameters, then the second stream may be
			 * constrained to the wrong sample rate or size.
			 */
			if (!ssi_private->sync_fixed_slot_width &&
					!ssi_private->sync_no_open_busy) {
				if (!first_runtime->sample_bits) {
					spin_unlock_irqrestore(
							&ssi_private->lock,
							flags);
					dev_err(substream->pcm->card->dev,
						"set sample size in %s stream first\n",
						substream->stream ==
						SNDRV_PCM_STREAM_PLAYBACK
						? "capture" : "playback");
					return -EAGAIN;
				}

				snd_pcm_hw_constraint_minmax(substream->runtime,
					SNDRV_PCM_HW_PARAM_SAMPLE_BITS,
					first_runtime->sample_bits,
					first_runtime->sample_bits);
			}
		}

		ssi_private->second_stream = substream;
	}
	spin_unlock_irqrestore(&ssi_private->lock, flags);

	if (ssi_private->ssi_on_imx)
		snd_soc_dai_set_dma_data(dai, substream,
			(substream->stream == SNDRV_PCM_STREAM_PLAYBACK) ?
				&ssi_private->dma_params_tx :
				&ssi_private->dma_params_rx);

	return 0;
}

/**
 * fsl_ssi_hw_params - program the sample size
 * Unfortunately, programming
 * the SxCCR.WL bits requires the SSI to be temporarily disabled.  This can
 * cause a problem with supporting simultaneous playback and capture.  If
 * the SSI is already playing a stream, then that stream may be temporarily
 * stopped when you start capture.
 *
 * Note: The SxCCR.DC and SxCCR.PM bits are only used if the SSI is the
 * clock master.
 */
static int fsl_ssi_hw_params(struct snd_pcm_substream *substream,
	struct snd_pcm_hw_params *hw_params, struct snd_soc_dai *cpu_dai)
{
	struct fsl_ssi_private *ssi_private = snd_soc_dai_get_drvdata(cpu_dai);
	struct ccsr_ssi __iomem *ssi = ssi_private->ssi;
	unsigned int sample_size;
	u32 wl;
	int enabled = read_ssi(&ssi->scr) & CCSR_SSI_SCR_SSIEN;
	int ret;
	unsigned int slots = ssi_private->slots[cpu_dai->id];
	unsigned int fmt_width = snd_pcm_format_width(params_format(hw_params));
	unsigned long flags;

	/* update ideal bit clock in case of asrc */
	ret = update_ideal_clock(ssi_private, substream, hw_params,
				cpu_dai->id);
	if (ret)
		return ret;

	sample_size = ssi_private->slot_width[cpu_dai->id];

	wl = CCSR_SSI_SxCCR_WL(fmt_width);

	if ((cpu_dai->rate != 0) &&
	    (cpu_dai->rate != params_rate(hw_params))) {
		dev_warn(cpu_dai->dev,
		"sync mode - cannot set rate as another stream is active\n");
		dev_warn(cpu_dai->dev,
			"current rate = %d requested rate = %d\n",
			cpu_dai->rate, params_rate(hw_params));
	}

	/*
	 * Skip the params if it is the second stream
	 */
	if (enabled && ssi_private->mode == IMX_SSI_SYN)
		return 0;

	/* check correctness of format and slot_with settings and set format */
	ret = fsl_ssi_verify_hw_format_settings(substream);
	if (ret)
		return ret;

	/* Configure default clk, if not done by Machine driver*/
	if (ssi_private->sysclk[cpu_dai->id] != true)
		fsl_ssi_set_dai_sysclk(cpu_dai,
				SYSCLK_SSI_FSYS,
				SSI_CLK_DEFAULT_RATE,
				SYSCLK_SSI_DIR_IN);

	/* set clock dividers using automatic bit clock calculation according to
	 * format, slots_width, sample_size, amount of slots and etc if not
	 * already set */
	if (!ssi_private->div[cpu_dai->id]) {
		unsigned int rate = params_rate(hw_params);
		unsigned int bck = rate * slots * sample_size;
		unsigned int mck = clk_get_rate(ssi_private->mclk);
		unsigned int ipg = clk_get_rate(ssi_private->clk);

		/* check if bck exceeds 1/5 ipg_clk frequency */
		if (bck > ipg/5) {
			dev_err(cpu_dai->dev,
					"bit rate value (%d) greater than ipg/5",
					bck);

			return -EINVAL;
		}

		/* In synchronous mode, the SSI uses STCCR for capture */
		if (cpu_dai->id == ID_SYNC_PLAYCAP_OR_ASYNC_PLAY)
			fsl_ssi_set_divs(cpu_dai, &ssi->stccr,
					DIV_ROUND_CLOSEST(mck, 2 * bck));
		else
			fsl_ssi_set_divs(cpu_dai, &ssi->srccr,
					DIV_ROUND_CLOSEST(mck, 2 * bck));
	}

	/*
	 * FIXME: The documentation says that SxCCR[WL] should not be
	 * modified while the SSI is enabled.  The only time this can
	 * happen is if we're trying to do simultaneous playback and
	 * capture in asynchronous mode.  Unfortunately, I have been enable
	 * to get that to work at all on the P1022DS.  Therefore, we don't
	 * bother to disable/enable the SSI when setting SxCCR[WL], because
	 * the SSI will stop anyway.  Maybe one day, this will get fixed.
	 */
	spin_lock_irqsave(&ssi_private->lock, flags);
	if (cpu_dai->id == ID_SYNC_PLAYCAP_OR_ASYNC_PLAY)
		write_ssi_mask(&ssi->stccr, CCSR_SSI_SxCCR_WL_MASK, wl);
	else
		write_ssi_mask(&ssi->srccr, CCSR_SSI_SxCCR_WL_MASK, wl);
	spin_unlock_irqrestore(&ssi_private->lock, flags);

	return 0;
}

static int fsl_ssi_prepare(struct snd_pcm_substream *substream,
			   struct snd_soc_dai *cpu_dai) {

	struct fsl_ssi_private *ssi_private = snd_soc_dai_get_drvdata(cpu_dai);
	__be32 sor;
	unsigned long flags;

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK)
		hrtimer_cancel(&ssi_private->tx_startup_timer);

	spin_lock_irqsave(&ssi_private->lock, flags);
	sor = read_ssi(&ssi_private->ssi->sor);

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		sor |= CCSR_SSI_SOR_TX_CLR;
	} else
		sor |= CCSR_SSI_SOR_RX_CLR;

	write_ssi(sor, &ssi_private->ssi->sor);
	spin_unlock_irqrestore(&ssi_private->lock, flags);

	return 0;
}

/**
 * fsl_ssi_trigger: start and stop the DMA transfer.
 *
 * This function is called by ALSA to start, stop, pause, and resume the DMA
 * transfer of data.
 *
 * The DMA channel is in external master start and pause mode, which
 * means the SSI completely controls the flow of data.
 *
 * To use the i.MX's SDMA unit it is important to create an edge on the
 * corresponding request line. Otherwise the request will be ignored!
 */
static int fsl_ssi_trigger(struct snd_pcm_substream *substream, int cmd,
			   struct snd_soc_dai *dai)
{
	unsigned long flags;
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct fsl_ssi_private *ssi_private =
				snd_soc_dai_get_drvdata(rtd->cpu_dai);
	struct ccsr_ssi __iomem *ssi = ssi_private->ssi;

	switch (cmd) {
	case SNDRV_PCM_TRIGGER_START:
	case SNDRV_PCM_TRIGGER_PAUSE_RELEASE:

		ssi_private->dai_in_xrun[substream->stream] = false;

		if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
			unsigned int mask = ssi_private->tx_mask;
			int ret;
			char tx_fifo_cnt;

			ret = fsl_ssi_mask_calc(dai->dev,
					ssi_private,
					substream->runtime->channels,
					&mask,
					dai->id);

			if (ret)
				return ret;

			spin_lock_irqsave(&ssi_private->lock, flags);
			ssi_private->tx_trigger_enable = true;

			/* SSI should be enabled to change the mask */
			write_ssi_mask(&ssi->scr, 0, CCSR_SSI_SCR_SSIEN);
			if (mask != ssi_private->tx_mask)
				write_ssi(~mask, &ssi->stmsk);

			write_ssi_mask(&ssi->sier, 0, CCSR_SSI_SIER_TDMAE);
			spin_unlock_irqrestore(&ssi_private->lock, flags);

			ssi_private->tx_start_retries_cnt = 0;
			/* Check if we need to start our startup timer */
			tx_fifo_cnt =
				CCSR_SSI_SFCSR_TFCNT0(read_ssi(&ssi->sfcsr));
			if ((ssi_private->tx_start_max_retries == 0) ||
			    (tx_fifo_cnt >= ssi_private->tx_start_fifo_level)) {
				spin_lock_irqsave(&ssi_private->lock, flags);
				fsl_ssi_start_transmit(ssi_private);
				spin_unlock_irqrestore(&ssi_private->lock,
							flags);
			} else {
				ktime_t ktime_int = ktime_set(0,
						ssi_private->tx_start_timeout);
				hrtimer_start(&ssi_private->tx_startup_timer,
						ktime_int, HRTIMER_MODE_REL);
			}
		} else {
			unsigned int mask = ssi_private->rx_mask;
			int ret;
			ret = fsl_ssi_mask_calc(dai->dev,
					ssi_private,
					substream->runtime->channels,
					&mask,
					dai->id);

			if (ret)
				return ret;
			spin_lock_irqsave(&ssi_private->lock, flags);
			/* SSI should be enabled to change the mask */
			write_ssi_mask(&ssi->scr, 0, CCSR_SSI_SCR_SSIEN);
			if (mask != ssi_private->rx_mask)
				write_ssi(~mask, &ssi->srmsk);

			write_ssi_mask(&ssi->sier, 0, CCSR_SSI_SIER_RDMAE);
			write_ssi_mask(&ssi->scr, 0, CCSR_SSI_SCR_RE);
			spin_unlock_irqrestore(&ssi_private->lock, flags);
		}
		break;

	case SNDRV_PCM_TRIGGER_STOP:
	case SNDRV_PCM_TRIGGER_PAUSE_PUSH:
		spin_lock_irqsave(&ssi_private->lock, flags);
		if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
			ssi_private->tx_trigger_enable = false;

			write_ssi_mask(&ssi->scr, CCSR_SSI_SCR_TE, 0);
			write_ssi_mask(&ssi->sier, CCSR_SSI_SIER_TDMAE, 0);
		} else {
			write_ssi_mask(&ssi->scr, CCSR_SSI_SCR_RE, 0);
			write_ssi_mask(&ssi->sier, CCSR_SSI_SIER_RDMAE, 0);
		}

		/* FIXME - This is a workaround to a hardware issue in the case
		 * where an underrun occurs and causes a channel shift in sync
		 * slave mode.
		 *
		 * Note: The limitation of this workaround is in the case where
		 * both rx/tx are in use and one of them underruns. This will
		 * cause the hardware to be disabled for both rx and tx sides.
		 * This use case should be avoided when using this workaround */
		if (ssi_private->mode == IMX_SSI_SYN &&
				ssi_private->dai_in_xrun[substream->stream])
			write_ssi_mask(&ssi->scr, CCSR_SSI_SCR_SSIEN, 0);

		spin_unlock_irqrestore(&ssi_private->lock, flags);
		break;

	default:
		return -EINVAL;
	}

	return 0;
}

/**
 * fsl_ssi_shutdown: shutdown the SSI
 *
 * Shutdown the SSI if there are no other substreams open.
 */
static void fsl_ssi_shutdown(struct snd_pcm_substream *substream,
			     struct snd_soc_dai *dai)
{
	unsigned long flags;
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct fsl_ssi_private *ssi_private =
				snd_soc_dai_get_drvdata(rtd->cpu_dai);

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK)
		hrtimer_cancel(&ssi_private->tx_startup_timer);

	spin_lock_irqsave(&ssi_private->lock, flags);
	if (ssi_private->first_stream == substream)
		ssi_private->first_stream = ssi_private->second_stream;

	ssi_private->second_stream = NULL;

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		ssi_private->playback = NULL;
		ssi_private->tx_mask = SSI_DEFAULT_MASK;
	} else {
		ssi_private->capture = NULL;
		ssi_private->rx_mask = SSI_DEFAULT_MASK;
	}

	/*
	 * If this is the last active substream, disable the SSI.
	 */
	if (!ssi_private->first_stream) {
		struct ccsr_ssi __iomem *ssi = ssi_private->ssi;

		write_ssi_mask(&ssi->scr, CCSR_SSI_SCR_SSIEN, 0);
		/*
		   ssi_private-fmt should not be cleared otherwise
		   it brakes sgtl5000 which set format only once
		   during snd_soc_instantiate_card()
		*/
	}

	if (!dai->active) {
		/* clear setting to avoid persistence */
		ssi_private->slot_width[dai->id] = SSI_DEFAULT_SLOT_WIDTH;
		ssi_private->slots[dai->id] = SSI_DEFAULT_SLOTS;
		ssi_private->div[dai->id] = false;
		ssi_private->sysclk[dai->id] = false;
	}
	spin_unlock_irqrestore(&ssi_private->lock, flags);

	clk_disable(ssi_private->mclk);
	clk_disable(ssi_private->clk);
}

static int fsl_ssi_hw_free(struct snd_pcm_substream *substream,
			     struct snd_soc_dai *dai)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct fsl_ssi_private *ssi_private =
				snd_soc_dai_get_drvdata(rtd->cpu_dai);

	/* invalidate ideal bit clock in case of asrc */
	return invalidate_ideal_clock(ssi_private, substream);
}

static const struct snd_soc_dai_ops fsl_ssi_dai_ops = {
	.startup	= fsl_ssi_startup,
	.hw_params	= fsl_ssi_hw_params,
	.shutdown	= fsl_ssi_shutdown,
	.trigger	= fsl_ssi_trigger,
	.set_fmt        = fsl_ssi_set_dai_fmt,
	.set_clkdiv     = fsl_ssi_set_dai_clkdiv,
	.set_sysclk     = fsl_ssi_set_dai_sysclk,
	.set_tdm_slot   = fsl_ssi_set_dai_tdm_slot,
	.hw_free	= fsl_ssi_hw_free,
	.prepare	= fsl_ssi_prepare,
};


/* Template for the CPU dai driver structure in sync mode*/
static struct snd_soc_dai_driver fsl_ssi_dai_template_sync = {
	.id = ID_SYNC_PLAYCAP_OR_ASYNC_PLAY,
	.playback = {
		.channels_min = SSI_MIN_CHANNELS,
		.channels_max = SSI_MAX_CHANNELS,
		.rates = FSLSSI_I2S_RATES,
		.formats = FSLSSI_I2S_FORMATS,
	},
	.capture = {
		.channels_min = SSI_MIN_CHANNELS,
		.channels_max = SSI_MAX_CHANNELS,
		.rates = FSLSSI_I2S_RATES,
		.formats = FSLSSI_I2S_FORMATS,
	},
	.ops = &fsl_ssi_dai_ops,
	.symmetric_rates = 1,
};

/* Template for the CPU dai driver structure in async mode*/
static struct snd_soc_dai_driver fsl_ssi_dai_template_async[] = {
	{
		.id = ID_SYNC_PLAYCAP_OR_ASYNC_PLAY,
		.playback = {
			.channels_min = SSI_MIN_CHANNELS,
			.channels_max = SSI_MAX_CHANNELS,
			.rates = FSLSSI_I2S_RATES,
			.formats = FSLSSI_I2S_FORMATS,
		},
		.ops = &fsl_ssi_dai_ops,
		.symmetric_rates = 0,
	},
	{
		.id = ID_ASYNC_CAP,
		.capture = {
			.channels_min = SSI_MIN_CHANNELS,
			.channels_max = SSI_MAX_CHANNELS,
			.rates = FSLSSI_I2S_RATES,
			.formats = FSLSSI_I2S_FORMATS,
		},
		.ops = &fsl_ssi_dai_ops,
		.symmetric_rates = 0,
	},
};

/* Show the statistics of a flag only if its interrupt is enabled.  The
 * compiler will optimze this code to a no-op if the interrupt is not
 * enabled.
 */
#define SIER_SHOW(flag, name) \
	do { \
		if (SIER_FLAGS & CCSR_SSI_SIER_##flag) \
			length += sprintf(buf + length, #name "=%u\n", \
				ssi_private->stats.name); \
	} while (0)

#define REG_SHOW(name) \
	do { \
		length += sprintf(buf + length, #name "=%x\n", \
			read_ssi(&ssi_private->ssi->name)); \
	} while (0)

static ssize_t fsl_show_tx_info(struct device *dev,
	struct device_attribute *attr,  char *buf)
{
	ssize_t ret = 0;
	struct fsl_ssi_private *ssi_private =
		container_of(attr, struct fsl_ssi_private, dev_attr);
	u32 data;
	data = read_ssi(&ssi_private->ssi->scr);
	ret += sprintf(buf + ret, "%s, %s, %s\n",
			data & CCSR_SSI_SCR_SYN ? "sync" : "async",
			data & CCSR_SSI_SCR_NET ? "network" : "normal",
			data & CCSR_SSI_SCR_SYS_CLK_EN ? "sysclk" : "noclk");

	ret += sprintf(buf + ret, "I2S mode %d\n",
			(data & CCSR_SSI_SCR_I2S_MODE_MASK) >> 5);

	data = read_ssi(&ssi_private->ssi->stcr);
	ret += sprintf(buf + ret, "TX:\nframe clock:%s, %s, %s, %s\n",
			data & CCSR_SSI_STCR_TFDIR ? "out" : "in",
			data & CCSR_SSI_STCR_TEFS ? "early" : "firstbit",
			data & CCSR_SSI_STCR_TFSL ? "bit-long" : "word-long",
			data & CCSR_SSI_STCR_TFSI ? "low" : "high");
	ret += sprintf(buf + ret, "bit clock:%s, %s, %s\n",
			data & CCSR_SSI_STCR_TFDIR ? "out" : "in",
			data & CCSR_SSI_STCR_TSCKP ? "falling" : "rising",
			data & CCSR_SSI_STCR_TSHFD ? "lsb" : "msb");

	data = read_ssi(&ssi_private->ssi->stccr);
	ret += sprintf(buf + ret, "slots:%d\n",
			((data & CCSR_SSI_SxCCR_DC_MASK) >>
				CCSR_SSI_SxCCR_DC_SHIFT) + 1);
	ret += sprintf(buf + ret, "bits:%d\n",
			(((data & CCSR_SSI_SxCCR_WL_MASK) >>
				CCSR_SSI_SxCCR_WL_SHIFT) << 1) + 2);

	return ret;

}

static ssize_t fsl_show_rx_info(struct device *dev,
	struct device_attribute *attr,  char *buf)
{
	ssize_t ret = 0;
	struct fsl_ssi_private *ssi_private =
		container_of(attr, struct fsl_ssi_private, dev_attr);
	u32 data;

	data = read_ssi(&ssi_private->ssi->srcr);
	ret += sprintf(buf + ret, "RX:\nframe clock:%s, %s, %s, %s\n",
			data & CCSR_SSI_SRCR_RFDIR ? "out" : "in",
			data & CCSR_SSI_SRCR_REFS ? "early" : "firstbit",
			data & CCSR_SSI_SRCR_RFSL ? "bit-long" : "word-long",
			data & CCSR_SSI_SRCR_RFSI ? "low" : "high");
	ret += sprintf(buf + ret, "bit clock:%s, %s, %s\n",
			data & CCSR_SSI_SRCR_RFDIR ? "out" : "in",
			data & CCSR_SSI_SRCR_RSCKP ? "falling" : "rising",
			data & CCSR_SSI_SRCR_RSHFD ? "lsb" : "msb");

	data = read_ssi(&ssi_private->ssi->srccr);
	ret += sprintf(buf + ret, "slots:%d\n",
			((data & CCSR_SSI_SxCCR_DC_MASK) >>
				CCSR_SSI_SxCCR_DC_SHIFT) + 1);
	ret += sprintf(buf + ret, "bits:%d\n",
			(((data & CCSR_SSI_SxCCR_WL_MASK) >>
				CCSR_SSI_SxCCR_WL_SHIFT) << 1) + 2);

	return ret;

}

/**
 * fsl_sysfs_ssi_show: display SSI statistics
 *
 * Display the statistics for the current SSI device.  To avoid confusion,
 * we only show those counts that are enabled.
 */
static ssize_t fsl_sysfs_ssi_show(struct device *dev,
	struct device_attribute *attr, char *buf)
{
	struct fsl_ssi_private *ssi_private =
		container_of(attr, struct fsl_ssi_private, dev_attr);
	ssize_t length = 0;

	SIER_SHOW(RFRC_EN, rfrc);
	SIER_SHOW(TFRC_EN, tfrc);
	SIER_SHOW(CMDAU_EN, cmdau);
	SIER_SHOW(CMDDU_EN, cmddu);
	SIER_SHOW(RXT_EN, rxt);
	SIER_SHOW(RDR1_EN, rdr1);
	SIER_SHOW(RDR0_EN, rdr0);
	SIER_SHOW(TDE1_EN, tde1);
	SIER_SHOW(TDE0_EN, tde0);
	SIER_SHOW(ROE1_EN, roe1);
	SIER_SHOW(ROE0_EN, roe0);
	SIER_SHOW(TUE1_EN, tue1);
	SIER_SHOW(TUE0_EN, tue0);
	SIER_SHOW(TFS_EN, tfs);
	SIER_SHOW(RFS_EN, rfs);
	SIER_SHOW(TLS_EN, tls);
	SIER_SHOW(RLS_EN, rls);
	SIER_SHOW(RFF1_EN, rff1);
	SIER_SHOW(RFF0_EN, rff0);
	SIER_SHOW(TFE1_EN, tfe1);
	SIER_SHOW(TFE0_EN, tfe0);
	REG_SHOW(scr);
	REG_SHOW(stcr);
	REG_SHOW(srcr);
	REG_SHOW(stccr);
	REG_SHOW(srccr);
	REG_SHOW(stmsk);
	REG_SHOW(srmsk);
	REG_SHOW(sfcsr);
	length += fsl_show_tx_info(dev, attr, buf + length);
	length += fsl_show_rx_info(dev, attr, buf + length);

	return length;
}

/**
 * Make every character in a string lower-case
 */
static void make_lowercase(char *s)
{
	char *p = s;
	char c;

	while ((c = *p)) {
		if ((c >= 'A') && (c <= 'Z'))
			*p = c + ('a' - 'A');
		p++;
	}
}

static int init_dev_attr(struct device *dev, struct device_attribute *dev_attr,
		const char *name, umode_t mode, void *show, void *store)
{
	sysfs_attr_init(&dev_attr->attr);
	dev_attr->attr.name = name;
	dev_attr->attr.mode = mode;
	dev_attr->show = show;
	dev_attr->store = store;
	return device_create_file(dev, dev_attr);
}

static int fsl_ssi_probe(struct platform_device *pdev)
{
	struct fsl_ssi_private *ssi_private;
	int ret = 0;
	struct device_node *np = pdev->dev.of_node;
	const char *p, *sprop;
	const uint32_t *iprop;
	struct resource res;
	char name[64];
	int platform_id[SSI_SUPPORT_DAI_NUM] = {-1, -1};
	const char *attr_name;
	int watermark;
	int dma_brst_size;
	int slot_width = SSI_DEFAULT_SLOT_WIDTH;
	int i = 0;
	bool has_async = false, has_async_i2s = false;
	int pfdrv_num = SSI_PFDRV_NUM_SYNC;

	/* SSIs that are not connected on the board should have a
	 *      status = "disabled"
	 * property in their device tree nodes.
	 */
	if (!of_device_is_available(np))
		return -ENODEV;

	p = of_get_property(np, "dai_name", NULL);
	/* The DAI name is the last part of the full name of the node. */
	if (!p)
		p = strrchr(np->full_name, '/') + 1;
	ssi_private = kzalloc(sizeof(struct fsl_ssi_private) + strlen(p),
			      GFP_KERNEL);
	if (!ssi_private) {
		dev_err(&pdev->dev, "could not allocate DAI object\n");
		return -ENOMEM;
	}

	strcpy(ssi_private->name, p);
	/* Assigning stream names */
	snprintf(ssi_private->ssi_playback_stream_name,
				MAX_STREAM_NAME_SIZE - 1,
				"%s_playback", p);
	snprintf(ssi_private->ssi_capture_stream_name,
				MAX_STREAM_NAME_SIZE - 1,
				"%s_capture", p);

	/* Get the addresses and IRQ */
	ret = of_address_to_resource(np, 0, &res);
	if (ret) {
		dev_err(&pdev->dev, "could not determine device resources\n");
		goto error_kmalloc;
	}
	ssi_private->ssi = of_iomap(np, 0);
	if (!ssi_private->ssi) {
		dev_err(&pdev->dev, "could not map device resources\n");
		ret = -ENOMEM;
		goto error_kmalloc;
	}
	ssi_private->ssi_phys = res.start;

	ssi_private->irq = irq_of_parse_and_map(np, 0);
	if (ssi_private->irq == NO_IRQ) {
		dev_err(&pdev->dev, "no irq for node %s\n", np->full_name);
		ret = -ENXIO;
		goto error_iomap;
	}

	/* The 'name' should not have any slashes in it. */
	ret = request_irq(ssi_private->irq, fsl_ssi_isr, 0, ssi_private->name,
			  ssi_private);
	if (ret < 0) {
		dev_err(&pdev->dev, "could not claim irq %u\n", ssi_private->irq);
		goto error_irqmap;
	}

	/* Are the RX and the TX clocks locked? */
	if (of_find_property(np, "fsl,ssi-async", NULL))
		has_async = true;
	if (of_find_property(np, "fsl,ssi-async-i2s", NULL))
		has_async_i2s = true;

	if (has_async && !has_async_i2s)
		ssi_private->mode = IMX_SSI_ASYN;
	else if (!has_async && has_async_i2s)
		ssi_private->mode = IMX_SSI_ASYN_I2S;
	else if (!has_async && !has_async_i2s)
		ssi_private->mode = IMX_SSI_SYN;
	else {
		dev_err(&pdev->dev, "can't set fsl,ssi-async and fsl,ssi-async-i2s at the same time\n");
		goto error_irq;
	}

	ret = of_property_read_u32(np, "slot-width", &slot_width);
	if (!ret) {
		ret = -EINVAL;
		for (i = 0;
			i != ARRAY_SIZE(fsl_ssi_supported_slot_widths); i++) {
			if (fsl_ssi_supported_slot_widths[i] == slot_width) {
				ret = 0;
				break;
			}
		}
		if (ret) {
			dev_err(&pdev->dev,
				"The slot-width of %d is unsupported.\n",
				slot_width);
			goto error_irq;
		}
	}

	if (of_find_property(np, "sync-fixed-slot-width", NULL))
		ssi_private->sync_fixed_slot_width = slot_width;
	if (of_find_property(np, "sync-no-open-busy", NULL))
		ssi_private->sync_no_open_busy = true;

	if (ssi_private->sync_no_open_busy &&
			ssi_private->sync_fixed_slot_width) {
		dev_err(&pdev->dev, "can't set sync-fixed-slot-width or sync-no-open-busy with ssi-async options at the same time\n");
		goto error_irq;
	} else if ((ssi_private->sync_no_open_busy ||
			ssi_private->sync_fixed_slot_width) &&
			(ssi_private->mode != IMX_SSI_SYN)) {
		dev_err(&pdev->dev, "can't set fsl,ssi-async and fsl,ssi-async-i2s at the same time\n");
		goto error_irq;
	}

	if (ssi_private->mode == IMX_SSI_SYN) {
		pfdrv_num = SSI_PFDRV_NUM_SYNC;
		/* Initialize this copy of the CPU DAI driver structure */
		memcpy(&ssi_private->cpu_dai_drv, &fsl_ssi_dai_template_sync,
			sizeof(fsl_ssi_dai_template_sync));
		ssi_private->cpu_dai_drv[ID_SYNC_PLAYCAP_OR_ASYNC_PLAY].name =
				ssi_private->name;
		ssi_private->cpu_dai_drv[ID_SYNC_PLAYCAP_OR_ASYNC_PLAY]
				.playback.stream_name =
				ssi_private->ssi_playback_stream_name;
		ssi_private->cpu_dai_drv[ID_SYNC_PLAYCAP_OR_ASYNC_PLAY]
				.capture.stream_name =
				ssi_private->ssi_capture_stream_name;
	} else {
		pfdrv_num = SSI_PFDRV_NUM_ASYNC;
		/* Initialize this copy of the CPU DAI driver structure */
		memcpy(ssi_private->cpu_dai_drv, &fsl_ssi_dai_template_async,
			sizeof(fsl_ssi_dai_template_async));

		snprintf(ssi_private->ssi_playback_dai_name,
				MAX_DAI_NAME_SIZE - 1,
				"%s-tx", ssi_private->name);
		snprintf(ssi_private->ssi_capture_dai_name,
				MAX_DAI_NAME_SIZE - 1,
				"%s-rx", ssi_private->name);

		ssi_private->cpu_dai_drv[ID_SYNC_PLAYCAP_OR_ASYNC_PLAY].name =
				ssi_private->ssi_playback_dai_name;
		ssi_private->cpu_dai_drv[ID_ASYNC_CAP].name =
				ssi_private->ssi_capture_dai_name;
		ssi_private->cpu_dai_drv[ID_SYNC_PLAYCAP_OR_ASYNC_PLAY]
				.playback.stream_name =
				ssi_private->ssi_playback_stream_name;
		ssi_private->cpu_dai_drv[ID_ASYNC_CAP]
				.capture.stream_name =
				ssi_private->ssi_capture_stream_name;
	}

	ret = of_property_read_u32_array(np, "platform-id", platform_id,
				pfdrv_num);
	if (ret) {
		if (ssi_private->mode == IMX_SSI_SYN)
			dev_warn(&pdev->dev, "platform-id missing or invalid\n");
		else {
			dev_err(&pdev->dev, "require two platform-id in async mode\n");
			goto error_irq;
		}
	}

	/* Determine the FIFO depth. */
	iprop = of_get_property(np, "fsl,fifo-depth", NULL);
	if (iprop)
		ssi_private->fifo_depth = be32_to_cpup(iprop);
	else
                /* Older 8610 DTs didn't have the fifo-depth property */
		ssi_private->fifo_depth = SSI_DEFAULT_FIFO_DEPTH;

	iprop = of_get_property(np, "tx_channels", NULL);
	if (iprop) {
		ssi_private->cpu_dai_drv[ID_SYNC_PLAYCAP_OR_ASYNC_PLAY]
					.playback.channels_max =
					be32_to_cpup(iprop);
		dev_err(&pdev->dev, "tx_channels: %d\n",
			ssi_private->cpu_dai_drv[ID_SYNC_PLAYCAP_OR_ASYNC_PLAY]
					.playback.channels_max);
	}

	iprop = of_get_property(np, "rx_channels", NULL);
	if (iprop) {
		int dai_id = ssi_private->mode == IMX_SSI_SYN ?
					ID_SYNC_PLAYCAP_OR_ASYNC_PLAY :
					ID_ASYNC_CAP;
		ssi_private->cpu_dai_drv[dai_id].capture.channels_max =
					be32_to_cpup(iprop);
		dev_err(&pdev->dev, "rx_channels: %d\n",
			ssi_private->cpu_dai_drv[dai_id].capture.channels_max);
	}

	/* Read the watermarks from DT */
	iprop = of_get_property(np, "tx-wtrmrk", NULL);
	if (iprop)
		watermark = be32_to_cpup(iprop);
	else
		watermark = SSI_DEFAULT_WATERMARK_LEVEL;
	if ((watermark < SSI_MIN_WATERMARK_LEVEL) ||
	    (watermark > ssi_private->fifo_depth)) {
		dev_err(&pdev->dev,
		"mis-configured tx-wtrmrk the valid range is %d to %d\n",
		SSI_MIN_WATERMARK_LEVEL, ssi_private->fifo_depth);
		goto error_irq;
	}
	ssi_private->tx_watermark = watermark;

	iprop = of_get_property(np, "rx-wtrmrk", NULL);
	if (iprop)
		watermark = be32_to_cpup(iprop);
	else
		watermark = SSI_DEFAULT_WATERMARK_LEVEL;
	if ((watermark < SSI_MIN_WATERMARK_LEVEL) ||
	    (watermark > ssi_private->fifo_depth)) {
		dev_err(&pdev->dev,
		"mis-configured rx-wtrmrk the valid range is %d to %d\n",
		SSI_MIN_WATERMARK_LEVEL, ssi_private->fifo_depth);
		goto error_irq;
	}
	ssi_private->rx_watermark = watermark;

	/* Read the burst sizes from DT and check sanity against watermarks */
	iprop = of_get_property(np, "tx-burst", NULL);
	if (iprop)
		dma_brst_size = be32_to_cpup(iprop);
	else
		dma_brst_size = ssi_private->tx_watermark;
	if ((dma_brst_size < SSI_MIN_SDMA_BURST_SIZE) ||
	    (dma_brst_size > ssi_private->tx_watermark)) {
		dev_err(&pdev->dev,
		"mis-configured tx-burst the valid range is %d to %d\n",
		SSI_MIN_SDMA_BURST_SIZE,
		ssi_private->tx_watermark);
		goto error_irq;
	}
	ssi_private->dma_params_tx.burstsize = dma_brst_size;

	iprop = of_get_property(np, "rx-burst", NULL);
	if (iprop)
		dma_brst_size = be32_to_cpup(iprop);
	else
		dma_brst_size = ssi_private->rx_watermark;
	if ((dma_brst_size < SSI_MIN_SDMA_BURST_SIZE) ||
	    (dma_brst_size > ssi_private->rx_watermark)) {
		dev_err(&pdev->dev,
		"mis-configured rx-burst the valid range is %d to %d\n",
		SSI_MIN_SDMA_BURST_SIZE,
		ssi_private->rx_watermark);
		goto error_irq;
	}
	ssi_private->dma_params_rx.burstsize = dma_brst_size;

	/* Assign defaults for timer parameters */
	ssi_private->tx_start_max_retries = SSI_DEFAULT_MAX_STARTUP_RETRIES;
	ssi_private->tx_start_timeout = (SSI_DEFAULT_STARTUP_TIMER_TIMEOUT_US *
					1000);
	ssi_private->tx_start_fifo_level = (ssi_private->fifo_depth -
						ssi_private->tx_watermark + 1);

	/* Init timer attributes for sysfs */
	attr_name = "tx_start_timeout";
	ret = init_dev_attr(&pdev->dev, &ssi_private->dev_attr_timeout,
				attr_name, (S_IRUGO | S_IWUSR),
				tx_start_timeout_read, tx_start_timeout_write);
	if (ret) {
		dev_err(&pdev->dev, "could not create sysfs %s file\n",
			attr_name);
		goto error_irq;
	}

	attr_name = "tx_start_retries";
	ret = init_dev_attr(&pdev->dev, &ssi_private->dev_attr_retries,
				attr_name, (S_IRUGO | S_IWUSR),
				tx_start_retries_read, tx_start_retries_write);
	if (ret) {
		dev_err(&pdev->dev, "could not create sysfs %s file\n",
			attr_name);
		goto error_dev_attr_timeout;
	}

	attr_name = "tx_start_retry_count";
	ret = init_dev_attr(&pdev->dev, &ssi_private->dev_attr_retry_count,
				attr_name, S_IRUGO,
				tx_start_retry_count_read, NULL);
	if (ret) {
		dev_err(&pdev->dev, "could not create sysfs %s file\n",
			attr_name);
		goto error_dev_attr_retries;
	}

	attr_name = "tx_start_fifo_level";
	ret = init_dev_attr(&pdev->dev, &ssi_private->dev_attr_fifo_level,
				attr_name, (S_IRUGO | S_IWUSR),
				tx_start_fifo_level_read,
				tx_start_fifo_level_write);
	if (ret) {
		dev_err(&pdev->dev, "could not create sysfs %s file\n",
			attr_name);
		goto error_dev_attr_retry_count;
	}

	if (of_device_is_compatible(pdev->dev.of_node, "fsl,imx21-ssi")) {
		struct device_node *parent;
		u32 dma_events[2];
		ssi_private->ssi_on_imx = true;

		ssi_private->mclk = devm_clk_get(&pdev->dev, "mclk");
		if (IS_ERR(ssi_private->mclk) || !ssi_private->mclk) {
			ret = PTR_ERR(ssi_private->mclk);
			dev_err(&pdev->dev, "could not get mclk clock: %d",
				ret);
			goto error_dev_attr_flevel;
		}

		ssi_private->clk = devm_clk_get(&pdev->dev, "ipg");
		if (IS_ERR(ssi_private->clk) || !ssi_private->clk) {
			ret = PTR_ERR(ssi_private->clk);
			dev_err(&pdev->dev, "could not get ipg clock: %d", ret);
			goto error_mclk;
		}

		clk_prepare(ssi_private->mclk);
		clk_prepare(ssi_private->clk);

		ssi_private->dma_params_tx.dma_addr =
			ssi_private->ssi_phys + offsetof(struct ccsr_ssi, stx0);
		ssi_private->dma_params_rx.dma_addr =
			ssi_private->ssi_phys + offsetof(struct ccsr_ssi, srx0);
		/*
		 * TODO: This is a temporary solution and should be changed
		 * to use generic DMA binding later when the helplers get in.
		 */
		ret = of_property_read_u32_array(pdev->dev.of_node,
					"fsl,ssi-dma-events", dma_events, 2);
		if (ret) {
			dev_err(&pdev->dev, "could not get dma events\n");
			goto error_clk;
		}
		ssi_private->dma_params_tx.dma = dma_events[0];
		ssi_private->dma_params_rx.dma = dma_events[1];

		parent = of_get_parent(np);
		if (of_device_is_compatible(parent,"fsl,spba-bus")) {
			ssi_private->dma_params_tx.peripheral_type =
					IMX_DMATYPE_SSI_SP;
			ssi_private->dma_params_rx.peripheral_type =
					IMX_DMATYPE_SSI_SP;
		} else {
			ssi_private->dma_params_tx.peripheral_type =
					IMX_DMATYPE_SSI;
			ssi_private->dma_params_rx.peripheral_type =
					IMX_DMATYPE_SSI;
		}
		of_node_put(parent);
	}

	/* Initialize the the device_attribute structure */
	attr_name = "statistics";
	ret = init_dev_attr(&pdev->dev, &ssi_private->dev_attr,
				attr_name, S_IRUGO,
				fsl_sysfs_ssi_show, NULL);
	if (ret) {
		dev_err(&pdev->dev, "could not create sysfs %s file\n",
			attr_name);
		goto error_clk;
	}

	/* Initialize spin lock */
	spin_lock_init(&ssi_private->lock);

	/* Initialize our hi-res timer and setup callback */
	hrtimer_init(&ssi_private->tx_startup_timer, CLOCK_MONOTONIC,
			HRTIMER_MODE_REL);
	ssi_private->tx_startup_timer.function = &tx_startup_timer_callback;

	ssi_private->tx_trigger_enable = false;

	/* Register with ASoC */
	dev_set_drvdata(&pdev->dev, ssi_private);

	/* Internal interface defaults */
	for (i = 0; i < SSI_SUPPORT_DAI_NUM; i++) {
		ssi_private->slot_width[i] = slot_width;
		ssi_private->slots[i] = SSI_DEFAULT_SLOTS;
		ssi_private->div[i] = false;
		ssi_private->sysclk[i] = false;
	}
	ssi_private->tx_mask = SSI_DEFAULT_MASK;
	ssi_private->rx_mask = SSI_DEFAULT_MASK;
#if defined(CONFIG_SND_SOC_IMX_ASRC_DAI_SUPPORT)
	ssi_private->ideal_clk_set[SNDRV_PCM_STREAM_PLAYBACK] =
	ssi_private->ideal_clk_set[SNDRV_PCM_STREAM_CAPTURE] = false;
#endif

	if (ssi_private->mode == IMX_SSI_SYN) {
		write_ssi(CCSR_SSI_SCR_SYN, &ssi_private->ssi->scr);
		ret = snd_soc_register_dai(&pdev->dev,
			&ssi_private->cpu_dai_drv[
				ID_SYNC_PLAYCAP_OR_ASYNC_PLAY]);
	} else {
		write_ssi_mask(&ssi_private->ssi->scr, CCSR_SSI_SCR_SYN, 0);
		ret = snd_soc_register_dais(&pdev->dev,
					    ssi_private->cpu_dai_drv,
					    SSI_SUPPORT_DAI_NUM);
	}

	if (ret) {
		dev_err(&pdev->dev, "failed to register DAI(s): %d\n", ret);
		goto error_dev_attr_stat;
	}

	/* update asrc reference if asrc core available */
	ret = check_for_asrc(ssi_private, pdev);
	if (ret)
		goto error_dai;

	if (ssi_private->ssi_on_imx) {
		/* get platform driver name from device-tree */
		const char *pfdrv_name[SSI_SUPPORT_DAI_NUM];
		unsigned int i;
		bool use_default = true;

		if (of_find_property(np, "platform-name", NULL)) {
			if (of_property_count_strings(np, "platform-name") !=
				pfdrv_num) {
				dev_err(&pdev->dev, "%d platform-name(s) are required in %s mode\n",
				pfdrv_num, ssi_private->mode ==
				IMX_SSI_SYN ? "sync" : "async");
				goto error_dai;
			} else {
				for (i = 0; i < pfdrv_num; i++) {
					if (of_property_read_string_index(np,
						"platform-name", i,
						&pfdrv_name[i])) {
						dev_err(&pdev->dev, "failed to retrieve platform-name at index %d",
						i);
						goto error_dai;
					}
				}
				use_default = false;
			}
		} else
			dev_warn(&pdev->dev,
					"platform driver name not found,"
					"using default");

		if (use_default) {
			for (i = 0; i < pfdrv_num; i++)
				ssi_private->imx_pcm_pdev[i] =
					platform_device_register_simple(
							"imx-pcm-audio",
							platform_id[i],
							NULL, 0);
		} else {
			for (i = 0; i < pfdrv_num; i++)
				ssi_private->imx_pcm_pdev[i] =
					register_named_platform(ssi_private,
							pdev,
							pfdrv_name[i],
							platform_id[i]);
		}

		for (i = 0; i < pfdrv_num; i++)
			if (IS_ERR(ssi_private->imx_pcm_pdev[i])) {
				ret = PTR_ERR(ssi_private->imx_pcm_pdev[i]);
				goto error_platform;
			}
	}

	/*
	 * If codec-handle property is missing from SSI node, we assume
	 * that the machine driver uses new binding which does not require
	 * SSI driver to trigger machine driver's probe.
	 */
	if (!of_get_property(np, "codec-handle", NULL)) {
		ssi_private->new_binding = true;
		/*
		* We require the SSI platform device handle
		* for future reference in case we are using the
		* new binding mechanism. Beware that in the case
		* where new binding is not used, the platform device name
		* will be different.
		*/
		ssi_private->pdev = pdev;
		goto done;
	}

	/* Trigger the machine driver's probe function.  The platform driver
	 * name of the machine driver is taken from /compatible property of the
	 * device tree.  We also pass the address of the CPU DAI driver
	 * structure.
	 */
	sprop = of_get_property(of_find_node_by_path("/"), "compatible", NULL);
	/* Sometimes the compatible name has a "fsl," prefix, so we strip it. */
	p = strrchr(sprop, ',');
	if (p)
		sprop = p + 1;
	snprintf(name, sizeof(name), "snd-soc-%s", sprop);
	make_lowercase(name);

	ssi_private->pdev =
		platform_device_register_data(&pdev->dev, name, 0, NULL, 0);
	if (IS_ERR(ssi_private->pdev)) {
		ret = PTR_ERR(ssi_private->pdev);
		dev_err(&pdev->dev, "failed to register platform: %d\n", ret);
		goto error_platform;
	}

done:
	return 0;

error_platform:
	if (ssi_private->ssi_on_imx) {
		for (i = 0; i < pfdrv_num; i++) {
			if (!IS_ERR(ssi_private->imx_pcm_pdev[i]))
				platform_device_unregister(
					ssi_private->imx_pcm_pdev[i]);
		}
	}
error_dai:
	snd_soc_unregister_dais(&pdev->dev, pfdrv_num);

error_dev_attr_stat:
	dev_set_drvdata(&pdev->dev, NULL);
	device_remove_file(&pdev->dev, &ssi_private->dev_attr);

error_clk:
	if (ssi_private->ssi_on_imx) {
		clk_unprepare(ssi_private->clk);
		devm_clk_put(&pdev->dev, ssi_private->clk);
	}

error_mclk:
	if (ssi_private->ssi_on_imx) {
		clk_unprepare(ssi_private->mclk);
		devm_clk_put(&pdev->dev, ssi_private->mclk);
	}

error_dev_attr_flevel:
	device_remove_file(&pdev->dev, &ssi_private->dev_attr_fifo_level);

error_dev_attr_retry_count:
	device_remove_file(&pdev->dev, &ssi_private->dev_attr_retry_count);

error_dev_attr_retries:
	device_remove_file(&pdev->dev, &ssi_private->dev_attr_retries);

error_dev_attr_timeout:
	device_remove_file(&pdev->dev, &ssi_private->dev_attr_timeout);

error_irq:
	free_irq(ssi_private->irq, ssi_private);

error_irqmap:
	irq_dispose_mapping(ssi_private->irq);

error_iomap:
	iounmap(ssi_private->ssi);

error_kmalloc:
	kfree(ssi_private);

	return ret;
}

static int fsl_ssi_remove(struct platform_device *pdev)
{
	struct fsl_ssi_private *ssi_private = dev_get_drvdata(&pdev->dev);
	int pfdrv_num = ssi_private->mode == IMX_SSI_SYN ? SSI_PFDRV_NUM_SYNC :
							   SSI_PFDRV_NUM_ASYNC;
	int i;

	if (!ssi_private->new_binding)
		platform_device_unregister(ssi_private->pdev);
	if (ssi_private->ssi_on_imx) {
		for (i = 0; i < pfdrv_num; i++)
			platform_device_unregister(
				ssi_private->imx_pcm_pdev[i]);
		clk_unprepare(ssi_private->clk);
		clk_unprepare(ssi_private->mclk);
		devm_clk_put(&pdev->dev, ssi_private->mclk);
		devm_clk_put(&pdev->dev, ssi_private->clk);
	}
	snd_soc_unregister_dais(&pdev->dev, pfdrv_num);
	device_remove_file(&pdev->dev, &ssi_private->dev_attr);
	device_remove_file(&pdev->dev, &ssi_private->dev_attr_timeout);
	device_remove_file(&pdev->dev, &ssi_private->dev_attr_retries);
	device_remove_file(&pdev->dev, &ssi_private->dev_attr_retry_count);
	device_remove_file(&pdev->dev, &ssi_private->dev_attr_fifo_level);

	free_irq(ssi_private->irq, ssi_private);
	irq_dispose_mapping(ssi_private->irq);

	kfree(ssi_private);
	dev_set_drvdata(&pdev->dev, NULL);

	return 0;
}

static const struct of_device_id fsl_ssi_ids[] = {
	{ .compatible = "fsl,mpc8610-ssi", },
	{ .compatible = "fsl,imx21-ssi", },
	{}
};
MODULE_DEVICE_TABLE(of, fsl_ssi_ids);

static struct platform_driver fsl_ssi_driver = {
	.driver = {
		.name = "fsl-ssi-dai",
		.owner = THIS_MODULE,
		.of_match_table = fsl_ssi_ids,
	},
	.probe = fsl_ssi_probe,
	.remove = fsl_ssi_remove,
};

module_platform_driver(fsl_ssi_driver);

MODULE_AUTHOR("Timur Tabi <timur@freescale.com>");
MODULE_DESCRIPTION("Freescale Synchronous Serial Interface (SSI) ASoC Driver");
MODULE_LICENSE("GPL v2");
