/*
 * pcm1808.c - Codec driver for External MIC-ADC pcm1808
 *
 *   Copyright (c) 2013 Robert Bosch GmbH, Hildesheim
 *   Author: Suryachand Yellamraju (suryachand.yellamraju@in.bosch.com)
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; version 2 of the License.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/platform_device.h>
#include <linux/of_device.h>

#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/soc.h>
#include <linux/pwm.h>


#define PCM1808_SAMPLING_RATE_16000 16000
#define PCM1808_SAMPLING_RATE_44100 44100
#define PCM1808_SAMPLING_RATE_32000 32000

#define PWM1_TIME_PERIOD_32000 123

/* frequency = 4.125 MHz =>
Time period = 242.42 nano seconnds and approximate it to 243 */
#define PWM1_TIME_PERIOD_16000 243

/* frequency = 11.0 MHz =>
Time period = 90.9 nano seconnds and approximate it to 91 */
#define PWM1_TIME_PERIOD_44100 91

/* frequency = 8.192 MHz =>
Time period = 122.07 nano seconnds and approximate it to 123 */

static struct device pwm_dev;
static struct pwm_device *pwm1_handle;
static bool pwm_enabled;


/* Private data for the codec pcm 1808, can be extended beyond basic
   functionality. This is used from the reference of the skeleton driver */
struct codec_pcm1808_private {
	struct snd_soc_codec *codec;
	unsigned int mclk; /* Input frequency of the MCLK pin */
	unsigned int mode; /* The mode (I2S or left-justified) */
	int direction;
	int dai_format;
	struct snd_soc_dai_driver *codec_pcm1808_dai;
	unsigned int attribute_unlock:1;
	unsigned int attribute_mute:1;
};


/**
 * codec_pcm1808_set_dai_fmt - configure the codec for the selected audio format
 * @codec_dai: the codec DAI
 * @format: a SND_SOC_DAIFMT_x value indicating the data format
 *
 * This function takes a bitmask of SND_SOC_DAIFMT_x bits and programs the
 * codec accordingly.
 */
static int codec_pcm1808_set_dai_fmt(struct snd_soc_dai *codec_dai,
			      unsigned int format)
{
	struct snd_soc_codec *codec = codec_dai->codec;
	struct codec_pcm1808_private *codec_private =
			snd_soc_codec_get_drvdata(codec);

	dev_dbg(codec_dai->dev, "set dai format\n");
	codec_private->dai_format = format;

	return 0;
}

/**
 * codec_pcm1808_hw_params - program the codec-pcm1808 with the given hardware parameters.
 * @substream: the audio stream
 * @params: the hardware parameters to set
 * @dai: the SOC DAI is not used
 *
 * This function programs the hardware with the values provided.
 * Specifically, the sample rate and the data format.
 */
static int codec_pcm1808_hw_params(struct snd_pcm_substream *substream,
			    struct snd_pcm_hw_params *params,
			    struct snd_soc_dai *dai)
{

	int ret = 0;
	int pwm1_time_period = 0;
	unsigned int rate = 0;

	rate = params_rate(params);

	switch (rate) {
	case PCM1808_SAMPLING_RATE_16000:
		pwm1_time_period = PWM1_TIME_PERIOD_16000;
		break;

	case PCM1808_SAMPLING_RATE_32000:
		pwm1_time_period = PWM1_TIME_PERIOD_32000;
		break;

	case PCM1808_SAMPLING_RATE_44100:
		pwm1_time_period = PWM1_TIME_PERIOD_44100;
		break;

	default:
		return -EINVAL;

	}

	/* configure PWM1 to generate clock. Duty cycle=50%  */
	ret = pwm_config(pwm1_handle, (pwm1_time_period/2), pwm1_time_period);

	if (ret != 0) {
		pr_err("Error in Pwm_config in codec_pcm1808_hw_params\n");
		return -EINVAL;
	}

	if (!substream)
		return -EINVAL;

	return ret;
}

/**
 * @substream: the audio stream
 * @dai: the SOC DAI (ignored)
 */
static const struct snd_soc_dai_ops codec_pcm1808_dai_ops = {
	.set_fmt	= codec_pcm1808_set_dai_fmt,
	.hw_params	= codec_pcm1808_hw_params,
};

/* PCM1808 MIC-ADC only supports capture stream hence
		fill only the capture-stream attributes.*/
static const struct snd_soc_dai_driver codec_pcm1808_dai = {
	.name = "codec_pcm1808-dai",
	.capture = {
		.stream_name = "Capture",
		.channels_min = 1,
		.channels_max = 2,
		.rate_min = 8000,
		.rate_max = 192000,
		.rates = SNDRV_PCM_RATE_CONTINUOUS,
		.formats = (SNDRV_PCM_FMTBIT_S16_LE |
			SNDRV_PCM_FMTBIT_S24_LE | SNDRV_PCM_FMTBIT_S32_LE),
		.sig_bits = 32,
	},
	.ops = &codec_pcm1808_dai_ops,
};


/**
 * codec_pcm1808_skeleton_probe - ASoC probe function
 * @pdev: platform device
 *
 * This function is called when ASoC has all the pieces it needs to
 * instantiate a sound driver.
 */
static int codec_pcm1808_skeleton_probe(struct snd_soc_codec *codec)
{
	dev_dbg(codec->dev, "Probe snd soc codec-pcm1808\n");

	return 0;
}

/**
 * codec_pcm1808_skeleton_remove - ASoC remove function
 * @pdev: platform device
 *
 * This function is the counterpart to skeleton_probe().
 */
static int codec_pcm1808_skeleton_remove(struct snd_soc_codec *codec)
{
	dev_dbg(codec->dev, "removing snd soc codec-pcm1808\n");

	return 0;
};

/*
 * ASoC codec device structure
 *
 * Assign this variable to the codec_dev field of the machine driver's
 * snd_soc_device structure.
 */
static const struct snd_soc_codec_driver codec_pcm1808_skeleton = {
	.probe =	codec_pcm1808_skeleton_probe,
	.remove =	codec_pcm1808_skeleton_remove,
	.reg_cache_size = 0,
	.reg_word_size = sizeof(u8),
	.reg_cache_step = 1,
};

static const struct of_device_id codec_driver_ids[] = {
	{ .compatible = "fsl,codec-pcm1808", },
	{}
};

MODULE_DEVICE_TABLE(of, codec_driver_ids);

static int codec_parse_fmt(struct platform_device *pdev,
		struct device_node *np,
		struct snd_soc_pcm_stream *stream)
{
	unsigned int ret, val;

	if (np == NULL || stream == NULL)
			return 1;

	/* parse boundary channel count */
	ret = of_property_read_u32(np, "ch_mm", &val);
	if (!ret) {
			stream->channels_min = val;
			stream->channels_max = val;
	} else if (ret == -EINVAL) {
			ret = of_property_read_u32(np,
					"ch_min", &stream->channels_min);
			if (!(!ret || ret == -EINVAL))
				dev_err(&pdev->dev,
					"DT reading ch_min (%i)\n", ret);
			ret = of_property_read_u32(np,
				"ch_max", &stream->channels_max);
			if (!(!ret || ret == -EINVAL))
				dev_err(&pdev->dev,
					"DT reading ch_max (%i)\n", ret);
	} else {
			dev_err(&pdev->dev, "DT reading ch_mm (%i)\n", ret);
	}

	/* parse formats (INFO: codec fixup is done in soc-core.c) */
	of_property_read_u64(np, "fmts", &stream->formats);
	if (!(!ret || ret == -EINVAL))
			dev_err(&pdev->dev, "DT reading fmts (%i)\n", ret);

	/* parse supported rates */
	ret = of_property_match_string(np, "rates", "continuous");
	if (!ret) {
		stream->rates = SNDRV_PCM_RATE_CONTINUOUS;
	} else {
			ret = of_property_read_u32(np, "rates", &stream->rates);
			if (!(!ret || ret == -EINVAL))
				dev_err(&pdev->dev,
						"DT reading rates (%i)\n", ret);
	}

	/* parse boundary rates */
	ret = of_property_read_u32(np, "rate_mm", &val);
	if (!ret) {
			stream->rate_min = val;
			stream->rate_max = val;
	} else if (ret == -EINVAL) {
			ret = of_property_read_u32(np,
					"rate_min", &stream->rate_min);
			if (!(!ret || ret == -EINVAL))
				dev_err(&pdev->dev,
					"DT reading rate_min (%i)\n", ret);
			ret = of_property_read_u32(np, "rate_max",
					&stream->rate_max);
			if (!(!ret || ret == -EINVAL))
				dev_err(&pdev->dev,
					"DT reading rate_max (%i)\n", ret);
	} else {
			dev_err(&pdev->dev, "DT reading rate_mm (%i)\n", ret);
	}

	return 0;
}

static int codec_driver_pcm1808_probe(struct platform_device *pdev)
{
	struct codec_pcm1808_private *priv;
	struct device_node *np = pdev->dev.of_node;
	const char *sprop;
	int ret;

	dev_dbg(&pdev->dev, "ALSA SoC Codec Driver for PCM1808\n");

	priv = kzalloc(sizeof(struct codec_pcm1808_private), GFP_KERNEL);
	if (!priv)
		return -ENOMEM;

	priv->codec_pcm1808_dai = kzalloc(sizeof(struct snd_soc_dai_driver),
			GFP_KERNEL);

	if (!priv->codec_pcm1808_dai) {
		ret = -ENOMEM;
		goto fail_kzalloc_dai;
	}

	memcpy(priv->codec_pcm1808_dai, &codec_pcm1808_dai,
			sizeof(struct snd_soc_dai_driver));

	sprop = (char *)of_get_property(np, "dai_name", NULL);
	if (!sprop) {
		dev_err(&pdev->dev, "dai_name not provided\n");
		ret = -ENODEV;
		goto fail_dai_name;
	}

	priv->codec_pcm1808_dai->name = sprop;

	/* PCM1808 MIC-ADC only supports capture stream hence
		Fill only the capture-stream attributes.*/
	codec_parse_fmt(pdev, of_find_node_by_name(np, "cap"),
				&priv->codec_pcm1808_dai->capture);

	ret = dev_set_drvdata(&pdev->dev, priv);
	if (ret)
		goto fail_dai_name;

	ret = snd_soc_register_codec(&pdev->dev,
		&codec_pcm1808_skeleton, priv->codec_pcm1808_dai, 1);
	if (ret) {
		dev_err(&pdev->dev, "failed to register snd soc codec\n");
		goto fail_register_codec;
	}

	/* Add the pwm specific code here */
	pwm_dev = pdev->dev;

	/* get the handle of PWM1 */
	pwm1_handle = pwm_get(&(pwm_dev), NULL);

	/*Check whether the handle returned is NULL*/
	if (IS_ERR(pwm1_handle)) {
		pr_err("Error in pwm_get in codec_driver_pcm1808_probe\n");
		ret = PTR_ERR(pwm1_handle);
		goto fail_get_pwm_handle;
	}

	/* configure PWM1 to generate clock. Duty cycle=50% */
	/* default sampling rate is 16000 */
	ret = pwm_config(pwm1_handle, (PWM1_TIME_PERIOD_16000/2),
				PWM1_TIME_PERIOD_16000);

	if (ret != 0) {
		pr_err("Error in pwm_config in codec_driver_pcm1808_probe\n");
		ret = -EINVAL;
		goto fail_pwm_config;
	}

	/* Send Enable signal to PWM1 */
	ret = pwm_enable(pwm1_handle);

	if (ret != 0) {
		pr_err("Error in pwm_enable in codec_driver_pcm1808_probe\n");
		ret = -EINVAL;
		goto fail_pwm_config;
	}
	pwm_enabled = 1;


	return 0;

fail_pwm_config:
	pwm_put(pwm1_handle);
	pwm1_handle = NULL;
fail_get_pwm_handle:
	snd_soc_unregister_codec(&pdev->dev);
fail_register_codec:
	dev_set_drvdata(&pdev->dev, NULL);
fail_dai_name:
	kfree(priv->codec_pcm1808_dai);
fail_kzalloc_dai:
	kfree(priv);

	return ret;
}

static int codec_driver_pcm1808_remove(struct platform_device *pdev)
{
	struct codec_pcm1808_private *priv = dev_get_drvdata(&pdev->dev);

	if (pwm1_handle) {
		/* Send Disable signal to PWM1 */
		if ((pwm_enabled)) {
			pwm_disable(pwm1_handle);
			pwm_enabled = 0;
		}

		pwm_put(pwm1_handle);
		pwm1_handle = NULL;
	}
	snd_soc_unregister_codec(&pdev->dev);
	dev_set_drvdata(&pdev->dev, NULL);
	kfree(priv->codec_pcm1808_dai);
	kfree(priv);

	return 0;
}

static struct platform_driver codec_driver_pcm1808 = {
	.driver = {
		.name = "codec_driver-pcm1808",
		.owner = THIS_MODULE,
		.of_match_table = codec_driver_ids,
	},
	.probe = codec_driver_pcm1808_probe,
	.remove = codec_driver_pcm1808_remove,
};

module_platform_driver(codec_driver_pcm1808);


MODULE_AUTHOR("Suryachand Yellamraju <suryachand.yellamraju@in.bosch.com>");
MODULE_DESCRIPTION("ALSA SoC Codec Driver for External MIC-ADC PCM1808");
MODULE_LICENSE("GPL");
