/*
 * inc_proto_ssi.c - ssi protocol, only adding headers, handling xon/xoff
 *
 *   Copyright (c) 2012 Robert Bosch GmbH, Hildesheim
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */
#include <linux/module.h>
#include <linux/init.h>
#include <linux/uio.h>
#include <linux/net.h>
#include <linux/slab.h>
#include <linux/netdevice.h>
#include <linux/socket.h>
#include <linux/if_arp.h>
#include <linux/skbuff.h>
#include <linux/inetdevice.h>
#include <net/sock.h>
#include <net/tcp.h>

#include "af_inc.h"
#include "linux/inc.h"
#include "linux/inc_ports.h"
#include "linux/inet.h"
#include <linux/sysctl.h>
#include <linux/proc_fs.h>
#include <linux/of.h>

#include <linux/inetdevice.h>

#define INC_SSI_VERSION "1.0"
static __initdata const char banner[] =
	KERN_INFO "inc: SSI protocol (rev " INC_SSI_VERSION ")\n";

MODULE_DESCRIPTION("PF_INC SSI protocol");
MODULE_LICENSE("GPL");
MODULE_AUTHOR("Peter Waechtler");
MODULE_ALIAS("net-pf-" __stringify(PF_INC) "-type-1");

/**/

#define INC_PORT(inetport) ((inetport)&0x00ff)

/*
state		pstate		event	new_state	new_pstate
--------------------------------------------------------------------------
SS_FREE		--		create	SS_UNCONNECTED	TCP_CLOSE
SS_UNCONNECTED	TCP_CLOSE	listen	SS_UNCONNECTED	TCP_LISTEN
SS_UNCONNECTED	TCP_LISTEN	accept	SS_CONNECTED	TCP_ESTABLISHED
SS_UNCONNECTED	TCP_CLOSE	connect	SS_CONNECTED	TCP_ESTABLISHED
SS_CONNECTED	TCP_ESTABLISHED	shutdwn	SS_UNCONNECTED	TCP_CLOSE
*/
#define SSI_BIND_UNBOUND 0
#define SSI_BIND_MANUAL 1
#define SSI_BIND_AUTO  2


/*--------------------------------CONNECTIONS------------------*/
static struct ctl_table_header *inc_sysctl_reg_table;

/*
connection configuration
*/
#define PEER_TYPE_SERVER 0
#define PEER_TYPE_CLIENT 1
struct remote_peer {
	struct list_head lhead;
	char type;  /*server/client*/
	char refcnt;
	__be32 servaddr;/*address of local server*/
	__be16 servport;/*port of local server*/
	__be32 cliaddr;/*address of client issuing connect*/
	__be16 cliport;/*port of client issuing connect*/
};


struct connection_cfg {
	struct list_head conn_cfg;
};

static struct connection_cfg inc_conn;



static int inc_conn_add(__be32 servaddr, __be16 servport,
				__be32 cliaddr, __be16 cliport)
{
	struct remote_peer *new_peer;
	struct remote_peer *entry;

	PDEBUG("try adding: %pI4:%d %pI4:%d\n", &servaddr, ntohs(servport),
			&cliaddr, ntohs(cliport));

	/*check if exist*/
	list_for_each_entry(entry, &inc_conn.conn_cfg, lhead) {
		if ((entry->servaddr == servaddr) &&
			(entry->servport == servport) &&
			(entry->cliaddr == cliaddr) &&
			(entry->cliport == cliport)) {
				PDEBUG("already present\n");
				return -1;
		}
	}
	/*add*/
	new_peer = kmalloc(sizeof(struct remote_peer), GFP_KERNEL);

	new_peer->refcnt = 0;
	new_peer->servaddr = servaddr;/*address of local server*/
	new_peer->servport = servport;/*port of local server*/
	new_peer->cliaddr = cliaddr;/*address of client issuing connect*/
	new_peer->cliport = cliport;/*port of client issuing connect*/
	list_add_tail(&new_peer->lhead, &inc_conn.conn_cfg);
	PDEBUG("added\n");
	return 0;

}


/**/
static int inc_conn_static_add(char *sservaddr, char *sservport,
					char *scliaddr, char *scliport)
{
	unsigned long long port;
	__be32 servaddr = in_aton(sservaddr);
	__be32 cliaddr = in_aton(scliaddr);
	__be16 servport;
	__be16 cliport;
	if (kstrtoull(sservport, 10, &port) < 0)
		return -EINVAL;
	servport = htons(port);
	if (kstrtoull(scliport, 10, &port) < 0)
		return -EINVAL;
	cliport = htons(port);
	return inc_conn_add(servaddr, servport, cliaddr, cliport);
}

#ifdef CONFIG_OF

static int inc_conn_init(void)
{
	struct device_node *dn, *dn_child;
	const char *prop_str;
	const void *prop;
	__be32 servaddr;
	int p;
	INIT_LIST_HEAD(&inc_conn.conn_cfg);
	dn = of_find_node_by_path(INC_DT_NODE_PATH);
	if (!dn) {
		PDEBUG("can't find %s node in device tree\n", INC_DT_NODE_PATH);
		return -ENOENT;
	}

	for_each_child_of_node(dn, dn_child) {
		PDEBUG("dn_child: %s\n", dn_child->name);

		prop_str = "remote-addr";
		prop = of_get_property(dn_child, prop_str, NULL);
		if (prop) {
			servaddr = in_aton((char *) prop);
		} else {
			PERR("could not read dt property: %s\n", prop_str);
			return -1;
		}

		/* all allowed LUNs in manual bind region */
		for (p = PORT_OFFSET; p <
		PORT_OFFSET + INC_PORT(INC_AUTOBIND_PORT_MIN); p++) {
			if (inc_conn_add(servaddr, htons(p), 0, 0) < 0) {
				PERR("could not add connection\n");
				return -1;
			}
		}
	}
	return 0;
}
#else
static int inc_conn_init(void)
{

	INIT_LIST_HEAD(&inc_conn.conn_cfg);

	return 0;
}
#endif

/*look for existing configured connection
type CLIENT: lookup client on remote side, input: local server
type server: lookup matching server on remote side, input: remote SERVER
*/
static int inc_conn_get(struct inc_sock *ro, __be32 saddr, __be16 sport,
						__be32 caddr, __be16 cport)
{
	struct remote_peer *e;
	PDEBUG("search: %pI4:%d %pI4:%d\n", &saddr, ntohs(sport),
			&caddr, ntohs(cport));

	list_for_each_entry(e, &inc_conn.conn_cfg, lhead) {

		PDEBUG("compare %pI4:%d %pI4:%d r%d\n",
				&e->servaddr, ntohs(e->servport),
				&e->cliaddr, ntohs(e->cliport), e->refcnt);

		if ((e->servaddr == saddr) && (e->servport == sport) &&
			((e->cliaddr == caddr) || (!e->cliaddr) || (!caddr)) &&
			((e->cliport == cport) || (!e->cliport) || (!cport)) &&
			(e->refcnt == 0)) {
				e->refcnt++;
				ro->peer = e;
				PDEBUG("SUCCESS\n");
				return 0;
		}
	}
	PERRMEM("invalid connection: %pI4:%d %pI4:%d r%d\n",
			&saddr, ntohs(sport), &caddr, ntohs(cport), e->refcnt);
	return -EADDRNOTAVAIL;
}

static int inc_conn_put(struct inc_sock *ro)
{
	if (!ro->peer)
		return 0;
	ro->peer->refcnt--;
	ro->peer = NULL;
	return 0;
}


/*--------------------------------------------------------------*/


static int ssi_init(struct sock *sk);

static struct proto ssi_proto __read_mostly = {
	.name       = "INC_PROTO_SSI",
	.owner      = THIS_MODULE,
	.obj_size   = sizeof(struct inc_sock),
	.init       = ssi_init,
};

static inline struct inc_sock *inc_sk(const struct sock *sk)
{
	return (struct inc_sock *)sk;
}


static int ssi_notifier(struct notifier_block *nb,
			unsigned long msg, void *data)
{
	struct net_device *dev = (struct net_device *)data;
	struct inc_sock *ro = container_of(nb,
			struct inc_sock, notifier);
	struct sock *sk = &ro->sk;

	if (!net_eq(dev_net(dev), &init_net))
		return NOTIFY_DONE;

	if (sk->sk_bound_dev_if != dev->ifindex)
		return NOTIFY_DONE;

	switch (msg) {

	case NETDEV_UNREGISTER:
		lock_sock(sk);
		/* remove current filters & unregister */
		ro->bound   = 0;
		release_sock(sk);

		sk->sk_err = ENODEV;
		if (!sock_flag(sk, SOCK_DEAD))
			sk->sk_error_report(sk);
		break;

	case NETDEV_DOWN:
		sk->sk_err = ENETDOWN;
		if (!sock_flag(sk, SOCK_DEAD))
			sk->sk_error_report(sk);
		break;
	}

	return NOTIFY_DONE;
}

static inline void ssi_sockstate(struct sock *sk, socket_state sock_state,
						unsigned char prot_state)
{
	sk->sk_state = prot_state;
	sk->sk_socket->state = sock_state;
}

static int ssi_signal_flowctl(struct socket *sock, struct inc_sock *ro,
		enum inc_state fc)
{
	struct sock *sk = sock->sk;
	struct sk_buff *skb;
	struct net_device *dev = NULL;

	int err = 0;
	struct inc_hdr *ih;
	int header_len = sizeof(struct inc_hdr);
	unsigned char data[1] = {0}; /* dummy data */
	PDEBUG("\n");

	if (sk->sk_bound_dev_if)
		dev = dev_get_by_index(sock_net(sk), sk->sk_bound_dev_if);
	if (!dev)
		return -ENXIO;

	skb = alloc_skb(1 + header_len, GFP_ATOMIC);
	if (!skb)
		goto put_dev;

	skb_reserve(skb, header_len);
	skb_push(skb, header_len);
	memcpy(skb_put(skb, 1), data, 1);

	ih = (struct inc_hdr *) (skb->head);
	ih->src_node  = fc;
	ih->src_lun   = ntohs(ro->inet_sport);
	ih->dest_node = 0xff;
	ih->dest_lun  = 0xff;

	err = sock_tx_timestamp(sk, &skb_shinfo(skb)->tx_flags);
	if (err < 0)
		goto free_skb;

	skb->dev = dev;
	skb->sk  = sk;

	pr_debug("dev: %s: len: %d\n", dev->name, skb->len);
	err = inc_send(skb, 0);

	dev_put(dev);

	if (err)
		goto send_failed;

	return 0;

free_skb:
	kfree_skb(skb);
put_dev:
	dev_put(dev);
send_failed:
	return err;
}

static int ssi_init(struct sock *sk)
{
	struct inc_sock *ro = inc_sk(sk);
PDEBUG("\n");

	ssi_sockstate(sk, SS_UNCONNECTED, TCP_CLOSE);
	ro->notifier.notifier_call = ssi_notifier;
	ro->inet_saddr = 0;
	ro->inet_sport = 0;
	ro->inet_conaddr = 0;
	ro->inet_conport = 0;
	ro->nonagle = 0;
	ro->bound = 0;
	register_netdevice_notifier(&ro->notifier);
	return 0;
}

static int ssi_release(struct socket *sock)
{
	struct sock *sk = sock->sk;
	struct inc_sock *ro;

	PDEBUG("\n");

	if (!sk)
		return 0;

	ro = inc_sk(sk);

	/* signal socket disconnected to SSI32 driver */
	ssi_signal_flowctl(sock, ro, INC_CHANNEL_XOFF);

	unregister_netdevice_notifier(&ro->notifier);

	lock_sock(sk);
	inc_conn_put(ro);
	ssi_sockstate(sk, SS_UNCONNECTED, TCP_CLOSE);

	ro->bound   = 0;

	sock_orphan(sk);
	sock->sk = NULL;

	release_sock(sk);
	sock_put(sk);

	return 0;
}

static struct in_ifaddr *dev_ifa(struct net_device *dev)
{
	if (!dev)
		return NULL;
	if (!dev->ip_ptr)
		return NULL;
	return dev->ip_ptr->ifa_list;
}



/*find device by local or remote address*/
static int ssi_find_device(struct sock *sk)
{
	struct net_device *dev;
	struct inc_sock *ro = inc_sk(sk);
	struct rtable *rt;

	__be32 inet_addr = ro->inet_saddr;
	if (inet_addr) {
		dev = ip_dev_find(sock_net(sk), inet_addr);
		if (!dev) {
			PERRMEM("local addr not found: %pI4 %pI4\n",
					&ro->inet_saddr, &ro->inet_conaddr);
			return -ENETUNREACH;
		}

		if ((sk->sk_bound_dev_if) &&
				sk->sk_bound_dev_if != dev->ifindex) {
			dev_put(dev);
			PERRMEM("local host unreachable: %pI4 %pI4\n",
					&ro->inet_saddr, &ro->inet_conaddr);
			return -EHOSTUNREACH;

		}
		sk->sk_bound_dev_if = dev->ifindex;
		dev_put(dev);
		return 0;
	}


	inet_addr = ro->inet_conaddr;
	if (!inet_addr) {
		if (sk->sk_bound_dev_if)
			return 0;
		PERRMEM("remote addr not found: %pI4 %pI4\n",
				&ro->inet_saddr, &ro->inet_conaddr);
		return -EADDRNOTAVAIL;
	}

	rt = ip_route_output(sock_net(sk), inet_addr, 0, 0, 0);

	if (IS_ERR(rt))
		return PTR_ERR(rt);
	dev = rt->dst.dev;
	ip_rt_put(rt);

	if (sk->sk_bound_dev_if && sk->sk_bound_dev_if != dev->ifindex) {
		PERRMEM("remote host unreachable: %pI4 %pI4\n",
				&ro->inet_saddr, &ro->inet_conaddr);
		return -EHOSTUNREACH;
	}

	sk->sk_bound_dev_if = dev->ifindex;

	if (!ro->inet_saddr) {
		struct in_ifaddr *ifa = dev_ifa(dev);
		if (ifa)
			ro->inet_saddr = ifa->ifa_local;
	}

	return 0;
}


static int ssi_bind(struct socket *sock, struct sockaddr *uaddr, int len)
{
	struct sockaddr_in *addr = (struct sockaddr_in *)uaddr;
	struct sock *sk = sock->sk;
	struct inc_sock *ro = inc_sk(sk);
	int err = 0;

	PDEBUG("\n");

	if (len < sizeof(*addr))
		return -EINVAL;

	lock_sock(sk);

	if (addr->sin_addr.s_addr == INADDR_ANY) {
		PERRMEM("local INADDR_ANY not supported by AF_INC (port %d)\n",
				ntohs(addr->sin_port));
		err = -EINVAL;
		goto out;
	}
	if (ro->bound) {
		PERRMEM("already bound: %pI4:%d\n",
				&addr->sin_addr.s_addr, ntohs(addr->sin_port));
		err = -EINVAL;
		goto out;
	}
	if (inc_lookup_add(sk, addr->sin_addr.s_addr, addr->sin_port, 0) < 0) {
		PERR("bind to %pI4:%d failed\n",
				&addr->sin_addr.s_addr, ntohs(addr->sin_port));
		err = -EADDRNOTAVAIL;
	} else {
		ro->bound = SSI_BIND_MANUAL;
		/*try find device*/
		(void)ssi_find_device(sk);
	}
 out:
	release_sock(sk);

	return err;
}

static int ssi_getname(struct socket *sock, struct sockaddr *uaddr,
		       int *len, int peer)
{
	struct sockaddr_in *addr = (struct sockaddr_in *)uaddr;
	struct sock *sk = sock->sk;
	struct inc_sock *ro = inc_sk(sk);
	int err = 0;
	PDEBUG("\n");

	memset(addr, 0, sizeof(*addr));
	addr->sin_family  = AF_INC;

	if (peer) {
		addr->sin_addr.s_addr = ro->inet_conaddr;
		addr->sin_port = ro->inet_conport;
		if (!ro->inet_conport)
			err = -ENOTCONN;
	} else {
		addr->sin_addr.s_addr = ro->inet_saddr;
		addr->sin_port = ro->inet_sport;
	}

	*len = sizeof(*addr);

	return err;
}

static int ssi_setsockopt(struct socket *sock, int level, int optname,
			  char __user *optval, unsigned int optlen)
{
	struct sock *sk = sock->sk;
	struct inc_sock *ro = inc_sk(sk);
	int val;
	int err = 0;

	PDEBUG("\n");

	if (level != SOL_INC)
		return -EINVAL;

	if (optlen < sizeof(int))
		return -EINVAL;

	if (get_user(val, (int __user *)optval))
		return -EFAULT;

	lock_sock(sk);

	switch (optname) {
	/*these options are just accepted but have NO EFFECT !*/
	case TCP_NODELAY:
		if (val) {
			PDEBUG("SET NODELAY\n");
			/* TCP_NODELAY is weaker than TCP_CORK, so that
			 * this option on corked socket is remembered, but
			 * it is not activated until cork is cleared.
			 *
			 * However, when TCP_NODELAY is set we make
			 * an explicit push, which overrides even TCP_CORK
			 * for currently queued segments.
			 */
			ro->nonagle |= TCP_NAGLE_OFF|TCP_NAGLE_PUSH;
		} else {
			ro->nonagle &= ~TCP_NAGLE_OFF;
		}
		break;

	case TCP_CORK:
		/* When set indicates to always queue non-full frames.
		 * Later the user clears this option and we transmit
		 * any pending partial frames in the queue.  This is
		 * meant to be used alongside sendfile() to get properly
		 * filled frames when the user (for example) must write
		 * out headers with a write() call first and then use
		 * sendfile to send out the data parts.
		 *
		 * TCP_CORK can be set together with TCP_NODELAY and it is
		 * stronger than TCP_NODELAY.
		 */
		if (val) {
			ro->nonagle |= TCP_NAGLE_CORK;
		} else {
			ro->nonagle &= ~TCP_NAGLE_CORK;
			if (ro->nonagle&TCP_NAGLE_OFF)
				ro->nonagle |= TCP_NAGLE_PUSH;
/*			inc_push_pending_frames(sk);*/
		}
		break;
	default:
		err = -ENOPROTOOPT;
		break;
	}
	release_sock(sk);

	return err;
}

static int ssi_getsockopt(struct socket *sock, int level, int optname,
			  char __user *optval, int __user *optlen)
{
	struct sock *sk = sock->sk;
	struct inc_sock *ro = inc_sk(sk);
	int val, len;

	PDEBUG("\n");

	if (level != SOL_INC)
		return -EINVAL;

	if (get_user(len, optlen))
		return -EFAULT;

	len = min_t(unsigned int, len, sizeof(int));

	if (len < 0)
		return -EINVAL;

	switch (optname) {
	case TCP_NODELAY:
		val = !!(ro->nonagle&TCP_NAGLE_OFF);
		break;
	case TCP_CORK:
		val = !!(ro->nonagle&TCP_NAGLE_CORK);
		break;
	default:
		return -ENOPROTOOPT;
	}

	if (put_user(len, optlen))
		return -EFAULT;
	if (copy_to_user(optval, &val, len))
		return -EFAULT;
	return 0;

}

static int sender_wake_function(wait_queue_t *wait, unsigned mode, int sync,
				  void *key)
{
	unsigned long bits = (unsigned long)key;

	/*
	 * Avoid a wakeup if event not interesting for us
	 */
	if (bits && !(bits & (POLLIN | POLLERR)))
		return 0;
	return autoremove_wake_function(wait, mode, sync, key);
}

static int wait_for_xon(struct sock *sk, int port, int ifindex)
{
	int err;
	long timeo = sock_sndtimeo(sk, 0);
	__be16 lun;

	DEFINE_WAIT_FUNC(wait, sender_wake_function);

	/*
	 * Converting port to LUN ie 0xC700 to 0x00,
	 * to avoid mismatch between port & LUN.
	 *
	 * 0xC700 is represented as 00C7 in network byte order.

	 */
	lun = ((port & 0XFF00) >> 8);

	if (INC_CHANNEL_XOFF != inc_channel_getstate(ifindex, lun))
		return 0;

	PDEBUG("sk: %p, timeo: %ld\n", sk, timeo);
	prepare_to_wait(inc_lookup_waitq(ifindex, lun),
					&wait, TASK_INTERRUPTIBLE);

	/* Socket errors? */
	err = sock_error(sk);
	if (err)
		goto out;

	/* Socket shut down? */
	if (sk->sk_shutdown & SEND_SHUTDOWN)
		goto out_noerr;

	/* handle signals */
	if (signal_pending(current)) {
		err = sock_intr_errno(timeo);
		goto out;
	}

	timeo = schedule_timeout(timeo);
	PDEBUG("sk: %p finished timeo: %ld\n", sk, timeo);
	/* Socket errors after sleep? */
	err = sock_error(sk);
	if (err)
		goto out;
out:
	finish_wait(sk_sleep(sk), &wait);
	return err;

out_noerr:
	err = 0;
	goto out;
}

static int ssi_sendmsg(struct kiocb *iocb, struct socket *sock,
		       struct msghdr *msg, size_t size)
{
	struct sock *sk = sock->sk;
	struct inc_sock *ro = inc_sk(sk);
	struct sk_buff *skb;
	__be32 inet_dstaddr;
	__be16 inet_dstport;
	struct net_device *dev = NULL;

	int err = 0;
	struct inc_hdr *ih;
	int header_len = sizeof(struct inc_hdr);
	PDEBUG("\n");

	if (sock->state != SS_CONNECTED)
		return -EINVAL;

	err = sock_error(sk);
	if (err)
		return err;
	if (msg->msg_name) {
		struct sockaddr_in *addr =
			(struct sockaddr_in *)msg->msg_name;

		if (msg->msg_namelen < sizeof(*addr))
			return -EINVAL;

		if (addr->sin_family != AF_INC)
			return -EINVAL;
		/*ignore to address*/
		/*
		inet_dstaddr = addr->sin_addr.s_addr;
		inet_dstport = addr->sin_port;
		*/
		inet_dstaddr = ro->inet_conaddr;
		inet_dstport = ro->inet_conport;
	} else {
		inet_dstaddr = ro->inet_conaddr;
		inet_dstport = ro->inet_conport;
	}

	if (sk->sk_bound_dev_if)
		dev = dev_get_by_index(sock_net(sk), sk->sk_bound_dev_if);
	if (!dev)
		return -ENXIO;

	/* sending zero length messages shall basically be supported,
	  * but since the SSI protocol doesn't support it
	  * the message is simply discarded
	  */
	if (size == 0) {
		dev_dbg(&dev->dev, "zero length message");
		return 0;
	}

	err = wait_for_xon(sk, inet_dstport, sk->sk_bound_dev_if);
	if (err < 0)
		goto send_failed;

	skb = sock_alloc_send_skb(sk, size + header_len,
		msg->msg_flags & MSG_DONTWAIT, &err);
	if (!skb)
		goto put_dev;

	skb_reserve(skb, header_len);
	skb_push(skb, header_len);
	err = memcpy_fromiovec(skb_put(skb, size), msg->msg_iov, size);
	if (err < 0)
		goto free_skb;
	ih = (struct inc_hdr *) (skb->head);
	ih->src_node  = ntohl(ro->inet_saddr);
	ih->src_lun   = ntohs(ro->inet_sport);
	ih->dest_node = ntohl(inet_dstaddr);
	ih->dest_lun  = ntohs(inet_dstport);

	err = sock_tx_timestamp(sk, &skb_shinfo(skb)->tx_flags);
	if (err < 0)
		goto free_skb;

	skb->dev = dev;
	skb->sk  = sk;

	pr_debug("dev: %s: len: %d\n", dev->name, skb->len);
	err = inc_send(skb, 0);

	dev_put(dev);

	if (err)
		goto send_failed;

	return size;

free_skb:
	kfree_skb(skb);
put_dev:
	dev_put(dev);
send_failed:
	return err;
}



static int ssi_recvmsg(struct kiocb *iocb, struct socket *sock,
		       struct msghdr *msg, size_t size, int flags)
{
	struct sock *sk = sock->sk;
	struct sk_buff *skb;
	int err = 0;
	int noblock;
	struct inc_sock *ro = inc_sk(sk);
	PDEBUG("\n");

	if (sock->state != SS_CONNECTED)
		return -EINVAL;

	noblock =  flags & MSG_DONTWAIT;
	flags   &= ~MSG_DONTWAIT;

	pr_debug("%s on sk: %p\n",
		noblock ? "peek" : "wait", sk);

	skb = skb_recv_datagram(sk, flags, noblock, &err);
	if (!skb)
		return err;

	err = sock_error(sk);
		if (err)
			return err;

	pr_debug("len: %d\n", skb->len);
	if (size < skb->len)
		msg->msg_flags |= MSG_TRUNC;
	else
		size = skb->len;

	err = memcpy_toiovec(msg->msg_iov, skb->data, size);
	if (err < 0) {
		skb_free_datagram(sk, skb);
		return err;
	}

	sk->sk_stamp = skb->tstamp;
	sock_recv_ts_and_drops(msg, sk, skb);

	if (msg->msg_name) {
		struct sockaddr_in from;
		if (msg->msg_namelen > sizeof(struct sockaddr_in))
			msg->msg_namelen = sizeof(struct sockaddr_in);
		from.sin_family = AF_INC;
		from.sin_addr.s_addr = ro->inet_conaddr;
		from.sin_port = ro->inet_conaddr;
		memcpy(msg->msg_name, &from, msg->msg_namelen);
	}

	skb_free_datagram(sk, skb);

	return size;
}


/*
 *	Accept a pending connection. The TCP layer now gives BSD semantics.
 */

static int ssi_accept(struct socket *sock, struct socket *newsock, int flags)
{
	struct sock *sk = sock->sk;
	struct sock *sknew;
	struct inc_sock *ro = inc_sk(sk);
	struct inc_sock *ronew;
	int err = -EINVAL;

	PDEBUG("%d %d\n", sock->state, sk->sk_state);

	lock_sock(sk);

	if (sock->state != SS_UNCONNECTED)
		goto out;
	if (sk->sk_state != TCP_LISTEN)
		goto out;

	PDEBUG("state OK\n");

/*
validate and create socket.
*/
	if (!sk->sk_max_ack_backlog)
		goto out;

	PDEBUG("backlog OK\n");


	err = inc_conn_get(ro, ro->inet_saddr, ro->inet_sport, 0, 0);
	if (err)
		goto out;

	PDEBUG("conn OK\n");

	err = 0;
	sknew = inc_new(sock_net(sk), newsock, sk->sk_protocol, &ssi_proto);
	/*copy sk_bound_dev_if to check if client connection fits to this*/
	sknew->sk_bound_dev_if = sk->sk_bound_dev_if;
	ronew = inc_sk(sknew);
	/*move connection entry to new socket*/
	ronew->peer = ro->peer;
	ro->peer = NULL;

	ronew->nonagle = ro->nonagle;
	ronew->bound = ro->bound;
	ronew->inet_saddr = ro->inet_saddr;
	ronew->inet_sport = ro->inet_sport;
	ronew->inet_conaddr = ronew->peer->cliaddr;
	ronew->inet_conport = ronew->peer->cliport;
	sock_graft(sknew, newsock);

	err = ssi_find_device(sknew);
	if (err)
		goto out;

	err = inc_lookup_add(sknew, ronew->inet_saddr, ronew->inet_sport, 1);
	if (err)
		goto out;

	ssi_sockstate(sknew, SS_CONNECTED, TCP_ESTABLISHED);

out:
	release_sock(sk);
	return err;
}

/*
 *	Move a socket into listening state.
 */
static int ssi_listen(struct socket *sock, int backlog)
{
	struct sock *sk = sock->sk;
	struct inc_sock *ro = inc_sk(sk);
	int err;
	lock_sock(sk);
	err = -EINVAL;

	PDEBUG("%d %d\n", sock->state, sk->sk_state);

	if (sock->state != SS_UNCONNECTED)
		goto out;
	if (sk->sk_state != TCP_CLOSE)
		goto out;

	PDEBUG("state OK, bound:%d backlog:%d\n", ro->bound, backlog);

	/*server must be bound*/
	if (!ro->bound)
		goto out;

	if (!backlog)
		goto out;

	err = 0;

	/*we use this as overall counter !!*/
	sk->sk_max_ack_backlog = backlog;

	ssi_sockstate(sk, SS_UNCONNECTED, TCP_LISTEN);

out:
	release_sock(sk);
	return err;
}



static int ssi_autobind(struct sock *sk, struct sockaddr_in *remote_addr)
{
	int err = 0;
	struct inc_sock *ro = inc_sk(sk);
	if (!ro->bound) {
		err = inc_lookup_add(sk, ro->inet_saddr, INC_PORT_ANY, 0);
		ro->bound = SSI_BIND_AUTO;
	}
	return err;
}



/*
 *	Connect to a remote host.
 */
static int ssi_connect(struct socket *sock, struct sockaddr *uaddr,
			int addr_len, int flags)
{
	struct sock *sk = sock->sk;
	struct inc_sock *ro = inc_sk(sk);
	struct sockaddr_in *addr = (struct sockaddr_in *)uaddr;

	int err;

	PDEBUG("\n");

	if (addr_len < sizeof(uaddr->sa_family))
		return -EINVAL;

	lock_sock(sk);

	switch (sock->state) {
	case SS_CONNECTED:
		err = -EISCONN;
		break;
	case SS_UNCONNECTED:
		err = -EINVAL;
		if (sk->sk_state != TCP_CLOSE)
			break;

		if (ro->inet_saddr == INADDR_ANY) {
			PERRMEM("local INADDR_ANY not supported -\n");
			PERRMEM("- (port %d) call bind with local addr first\n",
					ntohs(addr->sin_port));
			break;
		}

		/*
		validate and enter connect state
		*/
		err = inc_conn_get(ro, addr->sin_addr.s_addr, addr->sin_port,
					ro->inet_saddr, ro->inet_sport);
		if (err)
			break;

		err = ssi_autobind(sk, addr);
		if (err)
			break;

		ro->inet_conaddr = addr->sin_addr.s_addr;
		ro->inet_conport = addr->sin_port;

		err = ssi_find_device(sk);
		if (err)
			break;

		ssi_sockstate(sk, SS_CONNECTED, TCP_ESTABLISHED);

		/* signal socket connected to SSI32 driver */
		err = ssi_signal_flowctl(sock, ro, INC_CHANNEL_XON);

		break;
	default:
		err = -EINVAL;
		break;
	}
	release_sock(sk);
	return err;
}

static int ssi_shutdown(struct socket *sock, int how)
{
	struct sock *sk = sock->sk;
	int err = 0;
	PDEBUG("\n");

	lock_sock(sk);
	ssi_sockstate(sk, SS_UNCONNECTED, TCP_CLOSE);
	release_sock(sk);
	return err;
}

static const struct proto_ops ssi_ops = {
	.family        = PF_INC,
	.release       = ssi_release,
	.bind          = ssi_bind,
	.connect       = ssi_connect,
	.socketpair    = sock_no_socketpair,
	.accept        = ssi_accept,
	.getname       = ssi_getname,
	.poll          = datagram_poll,
	.ioctl         = inc_ioctl,	/* use inc_ioctl() from af_inc.c */
	.listen        = ssi_listen,
	.shutdown      = ssi_shutdown,
	.setsockopt    = ssi_setsockopt,
	.getsockopt    = ssi_getsockopt,
	.sendmsg       = ssi_sendmsg,
	.recvmsg       = ssi_recvmsg,
	.mmap          = sock_no_mmap,
	.sendpage      = sock_no_sendpage,
};



static const struct inc_proto ssi_inc_proto = {
	.type       = SOCK_STREAM,
	.protocol   = INC_PROTO_SSI32,
	.ops        = &ssi_ops,
	.prot       = &ssi_proto,
};

static int entry_to_string(char *buf, struct remote_peer *entry)
{
	int of = 0;

	of += sprintf(buf+of, "%03d", (ntohl(entry->servaddr) >> 24) & 0xff);
	of += sprintf(buf+of, "%c", '.');
	of += sprintf(buf+of, "%03d", (ntohl(entry->servaddr) >> 16) & 0xff);
	of += sprintf(buf+of, "%c", '.');
	of += sprintf(buf+of, "%03d", (ntohl(entry->servaddr) >> 8) & 0xff);
	of += sprintf(buf+of, "%c", '.');
	of += sprintf(buf+of, "%03d", (ntohl(entry->servaddr)) & 0xff);
	of += sprintf(buf+of, "%c", ':');
	of += sprintf(buf+of, "%05d", ntohs(entry->servport));
	of += sprintf(buf+of, "%c", ' ');

	of += sprintf(buf+of, "%03d", (ntohl(entry->cliaddr) >> 24) & 0xff);
	of += sprintf(buf+of, "%c", '.');
	of += sprintf(buf+of, "%03d", (ntohl(entry->cliaddr) >> 16) & 0xff);
	of += sprintf(buf+of, "%c", '.');
	of += sprintf(buf+of, "%03d", (ntohl(entry->cliaddr) >> 8) & 0xff);
	of += sprintf(buf+of, "%c", '.');
	of += sprintf(buf+of, "%03d", (ntohl(entry->cliaddr)) & 0xff);
	of += sprintf(buf+of, "%c", ':');
	of += sprintf(buf+of, "%05d", ntohs(entry->cliport));
	of += sprintf(buf+of, "%c", ' ');
	if (!entry->refcnt)
		of += sprintf(buf + of, "%c", 'x');
	else
		of += sprintf(buf + of, "%01d", entry->refcnt);

	of += sprintf(buf + of, "%c", '\n');

	return of;

}
#define SYSCTLCONNLEN 44
static int fake_connections(ctl_table *table, int write,
				 void __user *buffer,
				 size_t *lenp, loff_t *ppos)
{
	char buf[SYSCTLCONNLEN+10];/*additional space for status on read*/
	int ret = 0;
	int len = 0, plen = 0;
	struct remote_peer *entry;
	int offs = 0;

	PDEBUG("wr%d len%d ppos%lld\n", write, *lenp, *ppos);

	if (write) {
		if (!*lenp)
			return -EINVAL;

		if (*lenp != SYSCTLCONNLEN)
			return -EINVAL;
		if (copy_from_user(buf, buffer, *lenp)) {
			PDEBUG("WRITE:copy failed\n");
			ret = -EFAULT;
		} else {
			len = *lenp;
			buf[(*lenp) - 1] = '\0';
			PDEBUG("WRITE: %s\n", buf);
			buf[15] = '\0';
			buf[21] = '\0';
			buf[37] = '\0';
			buf[43] = '\0';
			inc_conn_static_add(&buf[0], &buf[16],
						&buf[22], &buf[38]);

			*lenp -= len;
			*ppos += *lenp;
			ret = len;
		}
	} else {
		list_for_each_entry(entry, &inc_conn.conn_cfg, lhead) {
			if (*lenp <= (SYSCTLCONNLEN + 2))
				break;

			offs = entry_to_string(buf, entry);
			if (plen >= *ppos) {
				if (copy_to_user(buffer+len, buf, offs)) {
					ret = -EFAULT;
					break;
				}
				ret += offs;
				len += offs;
				*ppos += offs;
				*lenp -= offs;
			}
			plen += offs;
		}
		if (*lenp > (SYSCTLCONNLEN + 2))
			*lenp = 0;
	}
	return ret;
}



static ctl_table inc_sysctl_rds_table[] = {
	{
		.procname       = "allowed_connections",
		.maxlen         = SYSCTLCONNLEN,
		.mode           = 0644,
		.proc_handler   = fake_connections,
	},
	{ }
};

static __init int inc_ssi_module_init(void)
{
	int err;
	printk(banner);
	inc_conn_init();
	err = inc_proto_register(&ssi_inc_proto);
	if (err < 0)
		PERR("registration of ssi protocol failed\n");
	inc_sysctl_reg_table = register_net_sysctl(&init_net, "net/inc",
						inc_sysctl_rds_table);
	return err;
}

static __exit void inc_ssi_module_exit(void)
{
	inc_proto_unregister(&ssi_inc_proto);

	if (inc_sysctl_reg_table)
		unregister_net_sysctl_table(inc_sysctl_reg_table);

}

module_init(inc_ssi_module_init);
module_exit(inc_ssi_module_exit);
