/*
 * net/af_inc/dgram_service.c
 *
 * Datagram service for Inter Node Communication
 *
 *   Copyright (c) 2013 Robert Bosch GmbH, Hildesheim
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#include <linux/export.h>
#include <linux/module.h>
#include <linux/net.h>
#include <linux/in.h>
#include <linux/printk.h>
#include <linux/inc.h>

#include <linux/dgram_service.h>

#undef PDEBUG
#define PDEBUG(fmt, args...) pr_debug("dgram: %s: " fmt, __func__, ## args)

#undef PERR
#define PERR(fmt, args...) pr_err("dgram: %s: " fmt, __func__, ## args)

MODULE_LICENSE("GPL");

/* init datagram service on given socket.
 * options: optional - not yet used.
 */
struct sk_dgram *dgram_init(struct socket *sk, int dgram_max, void *options)
{
	struct sk_dgram *skd;
	struct sockaddr_in addr;
	int len = sizeof(struct sockaddr);
	if (kernel_getsockname(sk, (struct sockaddr *)&addr, &len) < 0)
		return NULL;
	PDEBUG("dgram init on socket af %d\n", addr.sin_family);

	if (dgram_max > DGRAM_MAX) {
		PERR("exceed maxsize (%d>%d)\n", dgram_max, DGRAM_MAX);
		return NULL;
	}
	switch (addr.sin_family) {
	case AF_INET:
	case AF_INC:
		break;
	default:
		PERR("not supported on AF %d\n", addr.sin_family);
		return NULL;
	}

	skd = kmalloc(sizeof(struct sk_dgram), GFP_KERNEL);
	if (!skd) {
		PERR("failed allocating memory\n");
		return NULL;
	}
	memset(skd, 0, sizeof(*skd));
	skd->proto = addr.sin_family;
	skd->hlen = sizeof(struct dgram_header_std);
	skd->len = dgram_max;
	skd->buf = kmalloc(skd->len+skd->hlen, GFP_KERNEL);

	if (!skd->buf) {
		PERR("failed allocating rcv buffer\n");
		kfree(skd);
		return NULL;
	}
	skd->sk = sk;
	/*ignore options*/

	return skd;
}
EXPORT_SYMBOL(dgram_init);

int dgram_exit(struct sk_dgram *skd)
{
	if (!skd) {
		PERR("invalid handle\n");
		return -EINVAL;
	}
	kfree(skd->buf);
	skd->buf = NULL;
	kfree(skd);
	return 0;
}
EXPORT_SYMBOL(dgram_exit);

/*send message as datagram
-just prepend header
*/
int dgram_send(struct sk_dgram *skd, void *ubuf, size_t ulen)
{
	struct msghdr msg;
	struct kvec iov[2];
	struct dgram_header_std h;
	int ret;

	memset(&msg, 0, sizeof(msg));

	if (skd->proto != AF_INET) {
		h.dglen = htons(ulen);
		iov[0].iov_base = ubuf;
		iov[0].iov_len = ulen;

		ret = kernel_sendmsg(skd->sk, &msg, iov, 1, ulen);
		if (ret < 0)
			PERR("could not send msg: %d\n", ret);

		return ret;
	}

	if (ulen > skd->len) {
		PERR("send: dgram exceeds bufsize (%d>%d)\n", ulen, skd->len);
		return -EMSGSIZE;
	}

	h.dglen = htons(ulen);
	iov[0].iov_base = &h;
	iov[0].iov_len = sizeof(h);
	iov[1].iov_base = ubuf;
	iov[1].iov_len = ulen;

	ret = kernel_sendmsg(skd->sk, &msg, iov, 2, ulen + sizeof(h));
	if (ret < 0)
		PERR("could not send msg: %d\n", ret);

	if (ret >= sizeof(h))
		ret -= sizeof(h);

	return ret;
}
EXPORT_SYMBOL(dgram_send);

/*receive datagram*/
int dgram_recv(struct sk_dgram *skd, void *ubuf, size_t ulen)
{
	struct kvec iov;
	struct msghdr msg;
	int err;

	memset(&msg, 0, sizeof(msg));

	if (skd->proto != AF_INET) {
		iov.iov_base = ubuf;
		iov.iov_len = ulen;
		err = kernel_recvmsg(skd->sk, &msg, &iov, 1, ulen, 0);
		if (err <= 0)
			PDEBUG("receive failed with err %d\n", err);

		return err;
	}

	if ((skd->received < skd->hlen) || ((skd->received >= skd->hlen) &&
			(skd->received < (skd->h.dglen + skd->hlen)))) {
		iov.iov_base = skd->buf + skd->received;
		iov.iov_len = (skd->len + skd->hlen) - skd->received;
		err = kernel_recvmsg(skd->sk, &msg, &iov, 1, iov.iov_len, 0);
		if (err <= 0) {
			PDEBUG("receive failed with err %d\n", err);
			return err;
		}
		skd->received += err;
	}

	if (skd->received < skd->hlen)
		return -EAGAIN;

	/*complete header available*/
	memcpy((char *)&skd->h, skd->buf, skd->hlen);
	skd->h.dglen = ntohs(skd->h.dglen);

	if (skd->received < (skd->h.dglen+skd->hlen))
		return -EAGAIN;

	/*full dgram available*/
	if (ulen < skd->h.dglen) {
		PERR("recv: dgram exceeds bufsize (%d>%d)\n",
				skd->h.dglen, ulen);
		return -EMSGSIZE;
	}

	memcpy(ubuf, skd->buf+skd->hlen, skd->h.dglen);
	skd->received -= (skd->h.dglen+skd->hlen);
	err = skd->h.dglen;
	if (skd->received > 0) {
		memmove(skd->buf, skd->buf + skd->hlen + skd->h.dglen,
				skd->received);
		if (skd->received >= skd->hlen) {
			memcpy((char *)&skd->h, skd->buf, skd->hlen);
			skd->h.dglen = ntohs(skd->h.dglen);
		}
	}
	PDEBUG("finished DGRAM of len %d\n", err);
	return err;
}
EXPORT_SYMBOL(dgram_recv);
