/*
 * Copyright (C) 2013 Mentor Graphics, Inc. All Rights Reserved.
 * Copyright 2008-2012 Freescale Semiconductor, Inc. All Rights Reserved.
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#ifndef __MXC_ASRC_CORE_INTERNAL_H__
#define __MXC_ASRC_CORE_INTERNAL_H__

#include <uapi/linux/mxc_asrc_core.h>
#include <linux/hrtimer.h>

#define ASRC_PLAYBACK		0
#define ASRC_CAPTURE		1

#define DEFAULT_FIFO_PERIOD	16
#define MAX_IDEAL_CLOCKS	64

#define ASRC_SIGNED		1
#define LSB_ALIGNED		0
#define MSB_ALIGNED		1
#define SECTION_UNSET		0
#define SECTION_SET		1
#define FIFO_DEPTH		64
#define ASRC_SUPPORTED_INPUT_FORMATS (\
		ASRC_FMTBIT_S8	   |\
		ASRC_FMTBIT_S16_LE |\
		ASRC_FMTBIT_S24_LE |\
		ASRC_FMTBIT_S32_LE)
#define ASRC_SUPPORTED_OUTPUT_FORMATS (\
		ASRC_FMTBIT_S16_LE |\
		ASRC_FMTBIT_S24_LE |\
		ASRC_FMTBIT_S32_LE)
#define ASRC_INVALID_RATE	0
#define ASRC_MIN_RATE		8000
#define ASRC_MAX_RATE		192000
#define ASRC_CHANNEL_MAX	10
#define MIN_PERIOD		1
#define MAX_PERIOD		(FIFO_DEPTH-1)
#define ASRC_CHANNEL_MAX	10
#define ASRC_CHANNEL_MIN	1
#define IRQ_OVERLOAD_THRSHLD	10
#define IRQ_XRUN_THRSHLD	2
#define TABLE_POSITION_CLK_NONE		2
#define TABLE_POSITION_CLK_ASRCK1	1

#define CHANNEL_MASK_MAX	0x7
#define CHANNEL_MASK(i) ((1 << i))
#define PREV_STATE(s) ((s-1 == SECTION_STATE_TOP) ? s : s-1)
#define NEXT_STATE(s) ((s+1 == SECTION_STATE_BOT) ? s : s+1)

#define ASRC_CORE_DEVICE_NAME	"mxc_asrc_core"
#define ASRC_DEVICE_PAIR_NAME	"mxc_asrc_pair"
#define ASRC_CLK_NONE		"NONE"
#define ASRC_CLK_ASRCK1		"ASRCK1"
#define BE_PREFIX		"be_"
#define FE_PREFIX		"fe_"
#define IDEAL_PLAYBACK_PREFIX	"p"
#define IDEAL_CAPTURE_PREFIX	"c"

#ifdef CONFIG_MXC_ASRC_SYSFS
#define ASRC_SYSFS_ACCESS	(S_IRUGO | S_IWUSR)
#else
#define ASRC_SYSFS_ACCESS	S_IRUGO
#endif
#define ASRC_SYSFS_RO		(S_IRUGO)

#define MXC_ASRC_FILTER_UNSUPPORTED	255
#define MXC_ASRC_PRE_FILTERS	3
#define MXC_ASRC_POST_FILTERS	3
#define MXC_ASRC_PRE_FILTER_MAX_UPSAMPLING	2
#define MXC_ASRC_POST_FILTER_MAX_UPSAMPLING	2
#define MXC_ASRC_PRE_FILTER_ZERO_INS_UP2_ID	0
#define MXC_ASRC_POST_FILTER_DIRECT_ID	1
#define MXC_ASRC_MAX_POLYFILTER_DOWNSAMPLING_RATIO	63
#define MXC_ASRC_FORMULA_SAFETY_DEFAULT	10 /* 100% + 10% */
#define MXC_ASRC_FORMULA_STRONG_ALIAS_THRESHOLD	10
#define MXC_ASRC_GET_ASRC_BW_INVALID_PARAM	-1
#define MXC_ASRC_GET_ASRC_BW_INVALID_POLYPHASE_FILTER_RATIO	-2
#define MXC_ASRC_GET_ASRC_BW_STRONG_ALIAS	-3
#define MXC_ASRC_GET_ASRC_BW_OK	0

/* The asrc sound platform driver assumes the order of this text */
enum asrc_clk {
	CLK_ESAI_RX = 0x00,
	CLK_SSI1_RX = 0x01,
	CLK_SSI2_RX = 0x02,
	CLK_SSI3_RX = 0x03,
	CLK_SPDIF_RX = 0x04,
	CLK_MLB = 0x05,
	CLK_EXT = 0x06,
	CLK_NONE = 0x07,
	CLK_ESAI_TX = 0x08,
	CLK_SSI1_TX = 0x09,
	CLK_SSI2_TX = 0x0a,
	CLK_SSI3_TX = 0x0b,
	CLK_SPDIF_TX = 0x0c,
	CLK_ASRCK1 = 0x0d,
};

enum data_width {
	DATA_WIDTH_24 = 0,
	DATA_WIDTH_16,
	DATA_WIDTH_8
};

enum asrc_pair_index {
	ASRC_PAIR_A = 0,
	ASRC_PAIR_B,
	ASRC_PAIR_C,
	ASRC_PAIR_NUM
};

enum asrc_error_status {
	ASRC_TASK_Q_OVERLOAD = 0x01,
	ASRC_OUTPUT_TASK_OVERLOAD = 0x02,
	ASRC_INPUT_TASK_OVERLOAD = 0x04,
	ASRC_OUTPUT_BUFFER_OVERFLOW = 0x08,
	ASRC_INPUT_BUFFER_UNDERRUN = 0x10,
};

struct asrc_section_hw_params {
	int dma_irq;
	unsigned int period; /* # of frames per interrupt */
	dma_addr_t fifo_addr;
};

struct asrc_ideal_clock_entry {
	enum asrc_clk clk_mux_id;
	char *ideal_clk_name;
	unsigned long bit_clock;
	int ideal_clk_cnt;
};

struct asrc_section_params {
	struct asrc_pair_params *asrc_pair_params; /* link to parent */
	enum asrc_section_direction direction;
	enum asrc_section_state state;
	struct asrc_section_hw_params hw_params;
	unsigned int rate;
	unsigned int channels;
	unsigned char setup_status;
	unsigned long callback_arg;
	void (*callback_func)(unsigned long);
	u64 format;
	struct asrc_ideal_clock_entry const *clock_ref;
	struct buffer_capabilities buf_capabilities;
	spinlock_t section_lock;
};

struct asrc_pair_params {
	struct asrc_pair *asrc_pair; /* link to parent */
	enum asrc_pair_index index;
	struct asrc_section_params section[SECTION_CNT];
	unsigned int pair_hold;
	unsigned int xrun_threshold;
	unsigned int channel_cnt;
	unsigned char pair_mask;
	enum filter_settings filter;
	void *xrun_data;
	void (*xrun_func)(void *);
};

struct asrc_pair_capabilities {
	u64 formats[SECTION_CNT];	/* ASRC_FMTBIT_* */
	unsigned int rate_min;		/* min rate */
	unsigned int rate_max;		/* max rate */
	unsigned int channels_min;	/* min channels */
	unsigned int channels_max;	/* max channels */
};

struct asrc_pair {
	struct device *dev; /* dummy for now - only for sysfs */
	struct fsl_asrc_core_private *asrc_priv; /* link to parent */
	struct asrc_pair_capabilities capabilities;
	struct asrc_pair_params params;
	spinlock_t pair_lock;
	struct mutex pair_mutex;
	struct {
		unsigned int aool;	/*Pair Output Task Overload*/
		unsigned int aiol;	/*Pair Input Task Overload*/
		unsigned int aodo;	/*Output Data Buffer has overflowed*/
		unsigned int aidu;	/*Input Data Buffer has underflowed*/
		/*Number of data in Output Data Buffer A > threshold */
		unsigned int aodf;
		/*Number of data in Input Data Buffer A < threshold*/
		unsigned int aide;
	} stats;
	struct hrtimer startup_timer;
	bool xrun_enabled; /* flag to enable xrun handling */
};

struct fsl_asrc_core_private {
	void __iomem *vaddr;
	dma_addr_t asrc_phys;
	struct asrc_pair asrc_pair[ASRC_PAIR_NUM];
	struct {
		unsigned int dslcnt;	/*DSL Counter Input to FIFO ready*/
		unsigned int atqol;	/*Task Queue FIFO overload*/
		unsigned int fpwt;	/*FP is in wait states*/
		unsigned int aole;	/*Overload Error Flag*/
	} stats;
	int irq;
	struct device *dev;
	spinlock_t core_lock;
	spinlock_t xrun_lock;
	unsigned int channel_bits;
	const struct asrc_ideal_clock_entry *ideal_clock_table;
	unsigned int ideal_table_length;
	const char const *clock_text[MAX_IDEAL_CLOCKS];
};

enum asrc_status {
	ASRC_ASRSTR_AOLE = 0x40,
	ASRC_ASRSTR_FPWT = 0x80,
	ASRC_ASRSTR_ATQOL = 0x100000,
	ASRC_ASRSTR_DSLCNT = 0x200000,
#define ASRC_ASRSTR_AIDE(p) (0x1 << p)
#define ASRC_ASRSTR_AODF(p) (0x8 << p)
#define ASRC_ASRSTR_AIDU(p) (0x100 << p)
#define ASRC_ASRSTR_AODO(p) (0x800 << p)
#define ASRC_ASRSTR_AIOL(p) (0x4000 << p)
#define ASRC_ASRSTR_AOOL(p) (0x20000 << p)
};

int asrc_req_pair(struct fsl_asrc_core_private *,
		unsigned char, enum asrc_pair_index *);
int asrc_get_section_from_pair(
		struct fsl_asrc_core_private *,
		enum asrc_pair_index,
		struct asrc_section_params **,
		enum asrc_section_direction);
int asrc_release_pair(struct asrc_section_params *);
int asrc_open_section(struct asrc_section_params *);
int asrc_init_section(struct asrc_section_params *);
int asrc_set_filter_settings(struct asrc_section_params *,
		enum filter_settings);
int asrc_set_period(struct asrc_section_params *, unsigned char);
unsigned char asrc_get_period(struct asrc_section_params *);
int asrc_get_capabilities(struct asrc_section_params *,
		struct asrc_section_capabilities *);
int asrc_get_buffer_capabilities(struct asrc_section_params *,
		struct buffer_capabilities *);
int asrc_setup_section(struct asrc_section_params *, struct section_config *);
int asrc_set_clock_reference(struct asrc_section_params *, const char *);
int asrc_set_clock_reference_by_idx(struct asrc_section_params *, int);
int asrc_prepare(struct asrc_section_params *);
enum asrc_section_state asrc_get_section_state(struct asrc_section_params *);
int asrc_get_dma_request(struct asrc_section_params *);
dma_addr_t asrc_get_fifo_addr(struct asrc_section_params *);
int asrc_start_conversion(struct asrc_section_params *);
int asrc_stop_conversion(struct asrc_section_params *);
int asrc_xrun(struct asrc_section_params *);
int asrc_setup(struct asrc_section_params *);
unsigned int asrc_get_format_data_bytewidth(struct asrc_section_params *);
unsigned int asrc_get_frame_bytewidth(struct asrc_section_params *);
int asrc_get_channels(struct asrc_section_params *);
int asrc_set_callback(struct asrc_section_params *,
		void (*func)(unsigned long), unsigned long);
int asrc_clear_xrun_callback(struct asrc_section_params *);
int asrc_set_xrun_callback(struct asrc_section_params *,
		void (*func)(void *), void *);
int asrc_get_ideal_clkname_from_stream(const char const *streamname,
		char **clk_name,
		int stream_direction);
int asrc_get_base_streamname(const char *streamname,
		char const **basename);
unsigned int asrc_get_ideal_clock_rate(struct asrc_section_params *params,
		const char *ideal_clk_name);
int asrc_set_ideal_clock_rate(struct fsl_asrc_core_private *asrc_priv,
		const char *ideal_clk_name,
		unsigned int clock_rate);
const char const **asrc_fetch_clock_list(
		struct fsl_asrc_core_private *asrc_core_private, int *clk_cnt);
int asrc_invalidate_ideal_clock(struct fsl_asrc_core_private *asrc_priv,
		const char *streamname, int stream_direction);

#endif				/* __MXC_ASRC_CORE_INTERNAL_H__ */
