/*
 * linux/drivers/usb/gadget/utbridge.h
 *
 * USB Driver for the Dual Role Unwired Technology USB Bridge/Hub
 *
 * Copyright (C) 2014 Advanced Driver Information Technology GmbH
 * Written by Martin Herzog (mherzog@de.adit-jv.com)
 *        and Andreas Pape (apape@de.adit-jv.com)
 *
 * Based on a reference driver by Sam Yeda (sam@unwiredtechnology.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.   See the
 * GNU General Public License for more details.
 */

#ifndef __LINUX_USB_GADGET_UTBRIDGE_H__
#define __LINUX_USB_GADGET_UTBRIDGE_H__

#define DRIVER_DESC    "UTBridge driver with gadget/udc emulator"
#define DRIVER_VERSION "06 Nov 2014"

static const char gadget_name[] = "utbridge_udc";

#define USB_MAX_CTRL_PAYLOAD	64
#define MAX_ENDPOINTS		(7)	/* max endpoint count incl. ep0 */
#define MAX_BRIDGEPORT		(4)

#define CTRL_MSG_TIMEOUT	500
#define VEND_WR_BMREQTYPE	(0x40)
#define VEND_RD_BMREQTYPE	(0xC0)
#define VEND_WR_BREQ		(0x02)
#define VEND_RD_BREQ		(0x01)

#define UBS_PROCESS		0x0001
#define UBS_STORE		0x0002
#define UBS_RELAY		0x0008
#define VENDOR_SPECIFIC_REQ	0x2000

struct utb_ep_tmpl {
	u8 ep_a;            /* endpoint address at A-host interface */
	u8 ep_b;            /* endpoint address at B-host interface */
	char epname_b[14];  /* name of the endpoint at B-host interface */
};

struct utb_info {
	struct utb_ep_tmpl ep_info[MAX_ENDPOINTS]; /* ep assignment a<->b */
	int ep_info_count;      /* number of ep assignments */
	int port_num_offset;    /* port number offset */
	int max_bridge_port;    /* maximum selectable bridge port */
	u8 api_version;         /* whether to use new vendor requests */
};

/* Unified endpoint data structure */
struct utb_ep {
	/* A-Host */
	struct usb_endpoint_descriptor *a_desc; /* endpoint Descriptor*/
	struct usb_anchor submitted; /* if we need to retract our submissions */
	unsigned char *buffer;       /* the buffer to receive data */
	size_t size;                 /* the size of the receive buffer */
	struct urb *urb;             /* the urb to read/write data with */
	struct usb_utbridge *br_dev; /* pointer back to bridge device */

	/* B-Host */
	struct list_head queue;
	struct usb_gadget *gadget;
	const struct usb_endpoint_descriptor *b_desc;
	struct usb_ep ep;
	spinlock_t lock;
	unsigned long inprogress;
	unsigned halted:1;
	unsigned wedged:1;
};

/* Structure to hold all of our USB device specific stuff */
struct usb_utbridge {
	struct usb_device *udev;         /* the usb device for this device */
	struct usb_interface *interface; /* the interface for this device */
	struct platform_device *pdev;    /* platform device for UDC */
	struct utb_info *info;           /* usb device specific info */
	struct utb_ep *eps;              /* pointer to endpoint data */
	unsigned int max_eps;            /* maximum endpoints including ep0 */
	struct kref kref;                /* reference counter */
	struct mutex io_mutex;           /* synchronize I/O with disconnect */
	int connectedport;               /* hub port connected to bridge */
	bool portpower[MAX_BRIDGEPORT];  /* power states of all hub ports */
};

static struct usb_endpoint_descriptor utbridge_ep0_desc = {
	.bLength          = USB_DT_ENDPOINT_SIZE,
	.bDescriptorType  = USB_DT_ENDPOINT,
	.bEndpointAddress = 0,
	.bmAttributes     = USB_ENDPOINT_XFER_CONTROL,
	.wMaxPacketSize   = USB_MAX_CTRL_PAYLOAD,
};

bool pre_config_change(struct usb_utbridge *br_dev, bool force);
void post_config_change(struct usb_utbridge *br_dev, bool success);

extern struct platform_driver utbridge_udc_driver;
void utbridge_delete(struct kref *kref);

#endif /* __LINUX_USB_GADGET_UTBRIDGE_H__ */
