/*
 * core.c - ChipIdea USB IP core family device controller
 *
 * Copyright (C) 2008 Chipidea - MIPS Technologies, Inc. All rights reserved.
 *
 * Author: David Lopo
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

/*
 * Description: ChipIdea USB IP core family device controller
 *
 * This driver is composed of several blocks:
 * - HW:     hardware interface
 * - DBG:    debug facilities (optional)
 * - UTIL:   utilities
 * - ISR:    interrupts handling
 * - ENDPT:  endpoint operations (Gadget API)
 * - GADGET: gadget operations (Gadget API)
 * - BUS:    bus glue code, bus abstraction layer
 *
 * Compile Options
 * - CONFIG_USB_GADGET_DEBUG_FILES: enable debug facilities
 * - STALL_IN:  non-empty bulk-in pipes cannot be halted
 *              if defined mass storage compliance succeeds but with warnings
 *              => case 4: Hi >  Dn
 *              => case 5: Hi >  Di
 *              => case 8: Hi <> Do
 *              if undefined usbtest 13 fails
 * - TRACE:     enable function tracing (depends on DEBUG)
 *
 * Main Features
 * - Chapter 9 & Mass Storage Compliance with Gadget File Storage
 * - Chapter 9 Compliance with Gadget Zero (STALL_IN undefined)
 * - Normal & LPM support
 *
 * USBTEST Report
 * - OK: 0-12, 13 (STALL_IN defined) & 14
 * - Not Supported: 15 & 16 (ISO)
 *
 * TODO List
 * - OTG
 * - Isochronous & Interrupt Traffic
 * - Handle requests which spawns into several TDs
 * - GET_STATUS(device) - always reports 0
 * - Gadget API (majority of optional features)
 * - Suspend & Remote Wakeup
 */
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/dmapool.h>
#include <linux/dma-mapping.h>
#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/module.h>
#include <linux/idr.h>
#include <linux/interrupt.h>
#include <linux/io.h>
#include <linux/irq.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/pm_runtime.h>
#include <linux/regulator/consumer.h>
#include <linux/usb.h>
#include <linux/usb/hcd.h>
#include <linux/usb/ch9.h>
#include <linux/usb/gadget.h>
#include <linux/usb/otg.h>
#include <linux/usb/chipidea.h>
#include <linux/usb/hcd.h>

#include "ci.h"
#include "udc.h"
#include "bits.h"
#include "host.h"
#include "debug.h"
#include "otg.h"

/* Controller register map */
static uintptr_t ci_regs_nolpm[] = {
	[CAP_CAPLENGTH]		= 0x000UL,
	[CAP_HCSPARAMS]		= 0x004UL,
	[CAP_HCCPARAMS]		= 0x008UL,
	[CAP_DCCPARAMS]		= 0x024UL,
	[CAP_TESTMODE]		= 0x038UL,
	[OP_USBCMD]		= 0x000UL,
	[OP_USBSTS]		= 0x004UL,
	[OP_USBINTR]		= 0x008UL,
	[OP_DEVICEADDR]		= 0x014UL,
	[OP_ENDPTLISTADDR]	= 0x018UL,
	[OP_PORTSC]		= 0x044UL,
	[OP_DEVLC]		= 0x084UL,
	[OP_OTGSC]		= 0x064UL,
	[OP_USBMODE]		= 0x068UL,
	[OP_ENDPTSETUPSTAT]	= 0x06CUL,
	[OP_ENDPTPRIME]		= 0x070UL,
	[OP_ENDPTFLUSH]		= 0x074UL,
	[OP_ENDPTSTAT]		= 0x078UL,
	[OP_ENDPTCOMPLETE]	= 0x07CUL,
	[OP_ENDPTCTRL]		= 0x080UL,
};

static uintptr_t ci_regs_lpm[] = {
	[CAP_CAPLENGTH]		= 0x000UL,
	[CAP_HCSPARAMS]		= 0x004UL,
	[CAP_HCCPARAMS]		= 0x008UL,
	[CAP_DCCPARAMS]		= 0x024UL,
	[CAP_TESTMODE]		= 0x0FCUL,
	[OP_USBCMD]		= 0x000UL,
	[OP_USBSTS]		= 0x004UL,
	[OP_USBINTR]		= 0x008UL,
	[OP_DEVICEADDR]		= 0x014UL,
	[OP_ENDPTLISTADDR]	= 0x018UL,
	[OP_PORTSC]		= 0x044UL,
	[OP_DEVLC]		= 0x084UL,
	[OP_OTGSC]		= 0x0C4UL,
	[OP_USBMODE]		= 0x0C8UL,
	[OP_ENDPTSETUPSTAT]	= 0x0D8UL,
	[OP_ENDPTPRIME]		= 0x0DCUL,
	[OP_ENDPTFLUSH]		= 0x0E0UL,
	[OP_ENDPTSTAT]		= 0x0E4UL,
	[OP_ENDPTCOMPLETE]	= 0x0E8UL,
	[OP_ENDPTCTRL]		= 0x0ECUL,
};

static void ci_handle_vbus_change(struct ci13xxx *ci);

static int hw_alloc_regmap(struct ci13xxx *ci, bool is_lpm)
{
	int i;

	kfree(ci->hw_bank.regmap);

	ci->hw_bank.regmap = kzalloc((OP_LAST + 1) * sizeof(void *),
				     GFP_KERNEL);
	if (!ci->hw_bank.regmap)
		return -ENOMEM;

	for (i = 0; i < OP_ENDPTCTRL; i++)
		ci->hw_bank.regmap[i] =
			(i <= CAP_LAST ? ci->hw_bank.cap : ci->hw_bank.op) +
			(is_lpm ? ci_regs_lpm[i] : ci_regs_nolpm[i]);

	for (; i <= OP_LAST; i++)
		ci->hw_bank.regmap[i] = ci->hw_bank.op +
			4 * (i - OP_ENDPTCTRL) +
			(is_lpm
			 ? ci_regs_lpm[OP_ENDPTCTRL]
			 : ci_regs_nolpm[OP_ENDPTCTRL]);

	return 0;
}

/**
 * hw_port_test_set: writes port test mode (execute without interruption)
 * @mode: new value
 *
 * This function returns an error code
 */
int hw_port_test_set(struct ci13xxx *ci, u8 mode)
{
	const u8 TEST_MODE_MAX = 7;

	if (mode > TEST_MODE_MAX)
		return -EINVAL;

	hw_write(ci, OP_PORTSC, PORTSC_PTC, mode << ffs_nr(PORTSC_PTC));
	return 0;
}

/**
 * hw_port_test_get: reads port test mode value
 *
 * This function returns port test mode value
 */
u8 hw_port_test_get(struct ci13xxx *ci)
{
	return hw_read(ci, OP_PORTSC, PORTSC_PTC) >> ffs_nr(PORTSC_PTC);
}

static int hw_device_init(struct ci13xxx *ci, void __iomem *base)
{
	u32 reg;

	/* bank is a module variable */
	ci->hw_bank.abs = base;

	ci->hw_bank.cap = ci->hw_bank.abs;
	ci->hw_bank.cap += ci->platdata->capoffset;
	ci->hw_bank.op = ci->hw_bank.cap + ioread8(ci->hw_bank.cap);

	hw_alloc_regmap(ci, false);
	reg = hw_read(ci, CAP_HCCPARAMS, HCCPARAMS_LEN) >>
		ffs_nr(HCCPARAMS_LEN);
	ci->hw_bank.lpm  = reg;
	hw_alloc_regmap(ci, !!reg);
	ci->hw_bank.size = ci->hw_bank.op - ci->hw_bank.abs;
	ci->hw_bank.size += OP_LAST;
	ci->hw_bank.size /= sizeof(u32);

	reg = hw_read(ci, CAP_DCCPARAMS, DCCPARAMS_DEN) >>
		ffs_nr(DCCPARAMS_DEN);
	ci->hw_ep_max = reg * 2;   /* cache hw ENDPT_MAX */

	if (ci->hw_ep_max > ENDPT_MAX)
		return -ENODEV;

	dev_dbg(ci->dev, "ChipIdea HDRC found, lpm: %d; cap: %p op: %p\n",
		ci->hw_bank.lpm, ci->hw_bank.cap, ci->hw_bank.op);

	/* setup lock mode ? */

	/* ENDPTSETUPSTAT is '0' by default */

	/* HCSPARAMS.bf.ppc SHOULD BE zero for device */

	return 0;
}

/**
 * hw_device_reset: resets chip (execute without interruption)
 * @ci: the controller
  *
 * This function returns an error code
 */
int hw_device_reset(struct ci13xxx *ci, u32 mode)
{
	/* should flush & stop before reset */
	hw_write(ci, OP_ENDPTFLUSH, ~0, ~0);
	hw_write(ci, OP_USBCMD, USBCMD_RS, 0);

	hw_write(ci, OP_USBCMD, USBCMD_RST, USBCMD_RST);
	while (hw_read(ci, OP_USBCMD, USBCMD_RST))
		udelay(10);		/* not RTOS friendly */


	if (ci->platdata->notify_event)
		ci->platdata->notify_event(ci,
			CI13XXX_CONTROLLER_RESET_EVENT);

	/* Stream mode can be disabled independently for Host and Gadget */
	if (((ci->connector_role == CI_ROLE_GADGET) && (ci->platdata->flags
				& CI13XXX_DISABLE_STREAMING_GADGET))
		|| ((ci->connector_role == CI_ROLE_HOST) && (ci->platdata->flags
				& CI13XXX_DISABLE_STREAMING_HOST))) {
		hw_write(ci, OP_USBMODE, USBMODE_CI_SDIS, USBMODE_CI_SDIS);
	} else {
		hw_write(ci, OP_USBMODE, USBMODE_CI_SDIS, ~USBMODE_CI_SDIS);
	}

	/* USBMODE should be configured step by step */
	hw_write(ci, OP_USBMODE, USBMODE_CM, USBMODE_CM_IDLE);
	hw_write(ci, OP_USBMODE, USBMODE_CM, mode);
	/* HW >= 2.3 */
	hw_write(ci, OP_USBMODE, USBMODE_SLOM, USBMODE_SLOM);

	if (hw_read(ci, OP_USBMODE, USBMODE_CM) != mode) {
		pr_err("cannot enter in %s mode", ci_role(ci)->name);
		pr_err("lpm = %i", ci->hw_bank.lpm);
		return -ENODEV;
	}

	return 0;
}

/**
 * ci_otg_role - pick role based on ID pin state
 * @ci: the controller
 */
static enum ci_role ci_otg_role(struct ci13xxx *ci)
{
	u32 sts = hw_read(ci, OP_OTGSC, ~0);
	enum ci_role role = sts & OTGSC_ID
		? CI_ROLE_GADGET
		: CI_ROLE_HOST;

	ci->connector_role = role;

	return role;
}

/**
 * hw_wait_reg: wait the register value
 *
 * Sometimes, it needs to wait register value before going on.
 * Eg, when switch to device mode, the vbus value should be lower
 * than OTGSC_BSV before connects to host.
 *
 * @ci: the controller
 * @reg: register index
 * @mask: mast bit
 * @value: the bit value to wait
 * @timeout: timeout to indicate an error
 *
 * This function returns an error code if timeout
 */
static int hw_wait_reg(struct ci13xxx *ci, enum ci13xxx_regs reg, u32 mask,
				u32 value, unsigned long timeout)
{
	unsigned long elapse = jiffies + msecs_to_jiffies(timeout);

	while (hw_read(ci, reg, mask) != value) {
		if (time_after(jiffies, elapse)) {
			dev_err(ci->dev, "timeout waiting for %08x in %d\n",
					mask, reg);
			return -ETIMEDOUT;
		}
		msleep(20);
	}

	return 0;
}

static int ci_switch_role(struct ci13xxx *ci, enum ci_role role)
{
	bool vbus_auto = true;
	dev_dbg(ci->dev, "switching from %s to %s\n",
		ci_role(ci)->name, ci->roles[role]->name);

	dev_dbg(ci->dev, "vbus-stable-timeout: %u ms\n",
			ci->vbus_stable_timeout);
	/* 1. Finish the current role */
	if (ci->connector_role == CI_ROLE_GADGET)
		vbus_auto = HCD_VBUS_AUTO(ci->hcd);
	ci_role_stop(ci);
	hw_device_reset(ci, USBMODE_CM_IDLE);

	/* 2. Turn on/off vbus according to primary role */
	if (ci->connector_role == CI_ROLE_GADGET) {
		/* ci_switch_role is protected by role_switch_lock */
		if (vbus_auto) {
			otg_set_vbus(&ci->otg, false);
			/* wait vbus lower than OTGSC_BSV */
			hw_wait_reg(ci, OP_OTGSC, OTGSC_BSV, 0,
					ci->vbus_stable_timeout);
		}

		if (role == CI_ROLE_GADGET)
			ci->transceiver->state = OTG_STATE_B_IDLE;
		else
			ci->transceiver->state = OTG_STATE_B_WAIT_ACON;
		ci_handle_vbus_change(ci);
	} else if (ci->connector_role == CI_ROLE_HOST) {
		otg_set_vbus(&ci->otg, true);
		/* wait vbus higher than OTGSC_AVV */
		hw_wait_reg(ci, OP_OTGSC, OTGSC_AVV, OTGSC_AVV,
				ci->vbus_stable_timeout);
		if (role == CI_ROLE_GADGET)
			ci->transceiver->state = OTG_STATE_A_PERIPHERAL;
		else
			ci->transceiver->state = OTG_STATE_A_WAIT_BCON;
	}

	/* 3. Begin the new role */
	return ci_role_start(ci, role);
}

static void ci_handle_id_switch(struct ci13xxx *ci)
{
	enum ci_role role = ci_otg_role(ci);

	if (role != ci->role) {
		mutex_lock(&ci->role_switch_lock);
		ci_switch_role(ci, role);
		mutex_unlock(&ci->role_switch_lock);
	}
}

/**
 * ci_otg_suspend - OTG device suspend notification
 * @ci: the controller
 */
void ci_otg_suspend(struct ci13xxx *ci)
{
	if (!ci->is_otg)
		return;

	if (ci->transceiver->state != OTG_STATE_A_PERIPHERAL)
		return;

	set_bit(CI_A_PERIPHERAL_SUSPEND, &ci->events);
	queue_work(ci->wq, &ci->work);
}

/**
 * ci_otg_start_hnp - Shart HNP host to gadget
 *
 * A device signalled the start of a HNP handshake.
 * Switch to gadget mode and wait a connection.
 *
 * @ci: the controller
 */
int ci_otg_start_hnp(struct usb_otg *otg)
{
	struct ci13xxx *ci = container_of(otg, struct ci13xxx, otg);

	set_bit(CI_START_HNP, &ci->events);
	queue_work(ci->wq, &ci->work);
	return 0;
}

static void ci_handle_vbus_change(struct ci13xxx *ci)
{
	u32 otgsc = hw_read(ci, OP_OTGSC, ~0);

	if (ci->is_otg && ci->role == CI_ROLE_HOST
			&& ci->connector_role == CI_ROLE_GADGET) {
		dev_dbg(ci->dev, "HNP: vbus lost\n");
		if ((otgsc & OTGSC_BSV) == 0) {
			mutex_lock(&ci->role_switch_lock);
			ci_switch_role(ci, CI_ROLE_GADGET);
			mutex_unlock(&ci->role_switch_lock);
		}
		return;
	}

	if (otgsc & OTGSC_BSV) {
		if (ci->transceiver->state == OTG_STATE_B_IDLE)
			ci->transceiver->state = OTG_STATE_B_PERIPHERAL;
		usb_gadget_vbus_connect(&ci->gadget);
	} else {
		if (ci->transceiver->state == OTG_STATE_B_PERIPHERAL)
			ci->transceiver->state = OTG_STATE_B_IDLE;
		usb_gadget_vbus_disconnect(&ci->gadget);
	}
}

#define CI_PORT_OC_CHANGE	BIT(0)
#define CI_PORT_OC		BIT(1)

static u32 ci_get_port_status(struct ci13xxx *ci, bool first_time)
{

	u32 portsc = hw_read(ci, OP_PORTSC, ~0);
	u32 ret = 0;

	if (first_time && (portsc & PORTSC_OCC)) {
		ret |= CI_PORT_OC_CHANGE;
		if (portsc & PORTSC_OC) {
			/* disable port power on over-current */
			if (ci->reg_vbus && regulator_is_enabled(ci->reg_vbus))
				regulator_disable(ci->reg_vbus);
			hw_write(ci, OP_PORTSC, PORTSC_WC_MASK | PORTSC_PP, 0);
			portsc = hw_read(ci, OP_PORTSC, ~0);
		}
	}

	if (portsc & PORTSC_OC)
		ret |= CI_PORT_OC;

	return ret;
}

/* Handle over-current event in gadget mode */
static void ci_handle_oc_change(struct ci13xxx *ci)
{
	u32 portstatus;
	char *envp_ocstart[2] = {"OVERCURRENT=1", NULL};
	char *envp_ocend[2] = {"OVERCURRENT=0", NULL};

	portstatus = ci_get_port_status(ci, 1);

	if (!(portstatus & CI_PORT_OC_CHANGE))
		return;

	dev_info(ci->dev, "over-current change on root port\n");

	/* Clear OCC bit */
	hw_write(ci, OP_PORTSC, PORTSC_WC_MASK, PORTSC_OCC);

	/* Wait 800ms before enabling port power again */
	msleep(800);

	hw_write(ci, OP_PORTSC, PORTSC_WC_MASK | PORTSC_PP, PORTSC_PP);
	if (ci->reg_vbus && !regulator_is_enabled(ci->reg_vbus))
		regulator_enable(ci->reg_vbus);

	/* Wait at least 100 msec for power to become stable */
	msleep(100);

	portstatus = ci_get_port_status(ci, 0);

	if ((portstatus & CI_PORT_OC) && !ci->overcurrent) {
		dev_info(ci->dev,
				"over current condition on root port start\n");
		kobject_uevent_env(&ci->dev->kobj, KOBJ_CHANGE,
				envp_ocstart);
		ci->overcurrent = true;
	}

	if (!(portstatus & CI_PORT_OC) && ci->overcurrent) {
		dev_info(ci->dev,
				"over-current condition on root port end\n");
		kobject_uevent_env(&ci->dev->kobj, KOBJ_CHANGE,
				envp_ocend);
		ci->overcurrent = false;
	}
}

/**
 * ci_otg_work - perform otg (vbus/id) event handle
 * @work: work struct
 */
static void ci_otg_work(struct work_struct *work)
{
	struct ci13xxx *ci = container_of(work, struct ci13xxx, work);

	if (test_bit(CI_RESET, &ci->events)) {
		if (hw_read(ci, OP_USBCMD, USBCMD_RST)) {
			/* reschedule until the device is reset */
			queue_work(ci->wq, &ci->work);
			return;
		}

		clear_bit(CI_RESET, &ci->events);

		/* see hw_device_reset */
		if (ci->platdata->notify_event)
			ci->platdata->notify_event(
				ci, CI13XXX_CONTROLLER_RESET_EVENT);
		if (((ci->connector_role == CI_ROLE_GADGET)
				&& (ci->platdata->flags
					& CI13XXX_DISABLE_STREAMING_GADGET))
			|| ((ci->connector_role == CI_ROLE_HOST)
				&& (ci->platdata->flags
					& CI13XXX_DISABLE_STREAMING_HOST))) {
			hw_write(ci,
				OP_USBMODE, USBMODE_CI_SDIS, USBMODE_CI_SDIS);
		} else {
			hw_write(ci,
				OP_USBMODE, USBMODE_CI_SDIS, ~USBMODE_CI_SDIS);
		}


		/* USBMODE should be configured step by step */
		hw_write(ci, OP_USBMODE, USBMODE_CM, USBMODE_CM_IDLE);
		hw_write(ci, OP_USBMODE, USBMODE_CM, USBMODE_CM_DC);
		/* HW >= 2.3 */
		hw_write(ci, OP_USBMODE, USBMODE_SLOM, USBMODE_SLOM);
	} else if (test_bit(CI_ID, &ci->events)) {
		clear_bit(CI_ID, &ci->events);
		pm_runtime_get_sync(ci->dev);
		ci_handle_id_switch(ci);
		pm_runtime_put_sync(ci->dev);
		enable_irq(ci->irq);
	} else if (test_bit(CI_B_SESS_VALID, &ci->events)) {
		clear_bit(CI_B_SESS_VALID, &ci->events);
		pm_runtime_get_sync(ci->dev);
		ci_handle_vbus_change(ci);
		pm_runtime_put_sync(ci->dev);
		enable_irq(ci->irq);
	} else if (test_bit(CI_A_PERIPHERAL_SUSPEND, &ci->events)) {
		clear_bit(CI_A_PERIPHERAL_SUSPEND, &ci->events);
		if (ci->connector_role == CI_ROLE_HOST) {
			mutex_lock(&ci->role_switch_lock);
			ci_switch_role(ci, CI_ROLE_HOST);
			mutex_unlock(&ci->role_switch_lock);
		} else if (ci->connector_role == CI_ROLE_GADGET
				&& ci->gadget.b_hnp_enable) {
			mutex_lock(&ci->role_switch_lock);
			ci_switch_role(ci, CI_ROLE_HOST);
			mutex_unlock(&ci->role_switch_lock);
		} else {
			dev_err(ci->dev, "unexpected suspend occurs at %s\n",
				__func__);
		}
	} else if (test_bit(CI_START_HNP, &ci->events)) {
		clear_bit(CI_START_HNP, &ci->events);
		mutex_lock(&ci->role_switch_lock);
		ci_switch_role(ci, CI_ROLE_GADGET);
		mutex_unlock(&ci->role_switch_lock);
		usb_gadget_vbus_connect(&ci->gadget);
		mod_timer(&ci->otg_timer, jiffies
			+ msecs_to_jiffies(OTG_TIME_A_BIDL_ADIS));
		dev_dbg(ci->dev, "Waiting for HNP reset\n");
	} else if (test_bit(CI_OC, &ci->events)) {
		clear_bit(CI_OC, &ci->events);
		pm_runtime_get_sync(ci->dev);
		ci_handle_oc_change(ci);
		pm_runtime_put_sync(ci->dev);
	} else
		dev_err(ci->dev, "unexpected event occurs at %s\n", __func__);
}

static void ci_delayed_work(struct work_struct *work)
{
	struct delayed_work *dwork = to_delayed_work(work);
	struct ci13xxx *ci = container_of(dwork, struct ci13xxx, dwork);

	otg_set_vbus(&ci->otg, true);

}

static ssize_t show_vbus(struct device *dev, struct device_attribute *attr,
		char *buf)
{
	struct ci13xxx *ci = dev_get_drvdata(dev);
	u32 portsc;
	int len;

	pm_runtime_get_sync(ci->dev);
	portsc = hw_read(ci, OP_PORTSC, ~0);
	len = sprintf(buf, "%d\n", (portsc & PORTSC_PP) ? 1 : 0);
	pm_runtime_put_sync(ci->dev);

	return len;
}

static ssize_t store_vbus(struct device *dev, struct device_attribute *attr,
		const char *buf, size_t count)
{
	struct ci13xxx *ci = dev_get_drvdata(dev);

	pm_runtime_get_sync(ci->dev);
	if (!strcmp(buf, "on")) {
		ci_hdrc_set_power_policy(ci, true);
		otg_set_vbus(&ci->otg, true);
	} else if (!strcmp(buf, "off")) {
		ci_hdrc_set_power_policy(ci, false);
		otg_set_vbus(&ci->otg, false);
	} else {
		pm_runtime_put_sync(ci->dev);
		return -EINVAL;
	}
	pm_runtime_put_sync(ci->dev);

	return count;
}

static ssize_t show_id(struct device *dev, struct device_attribute *attr,
		char *buf)
{
	struct ci13xxx *ci = dev_get_drvdata(dev);
	int ret;

	pm_runtime_get_sync(ci->dev);
	ret = sprintf(buf, "%s\n", hw_read(ci, OP_OTGSC, ~0) & OTGSC_IDIE ?
			"enabled" : "disabled");
	pm_runtime_put_sync(ci->dev);

	return ret;
}

static ssize_t store_id(struct device *dev, struct device_attribute *attr,
		const char *buf, size_t count)
{
	struct ci13xxx *ci = dev_get_drvdata(dev);

	pm_runtime_get_sync(ci->dev);
	if (!strcmp(buf, "enable"))
		hw_write(ci, OP_OTGSC, OTGSC_IDIE, OTGSC_IDIE);
	else if (!strcmp(buf, "disable"))
		hw_write(ci, OP_OTGSC, OTGSC_IDIE, ~OTGSC_IDIE);
	else
		count = -EINVAL;
	pm_runtime_put_sync(ci->dev);

	return count;
}

static ssize_t show_role(struct device *dev, struct device_attribute *attr,
			 char *buf)
{
	struct ci13xxx *ci = dev_get_drvdata(dev);

	return sprintf(buf, "%s\n", ci_role(ci)->name);
}

static ssize_t store_role(struct device *dev, struct device_attribute *attr,
			  const char *buf, size_t count)
{
	struct ci13xxx *ci = dev_get_drvdata(dev);
	enum ci_role role;
	int ret;

	mutex_lock(&ci->role_switch_lock);
	for (role = CI_ROLE_HOST; role < CI_ROLE_END; role++)
		if (ci->roles[role] && !strcmp(buf, ci->roles[role]->name))
			break;

	if (role == CI_ROLE_END || role == ci->role) {
		mutex_unlock(&ci->role_switch_lock);
		return -EINVAL;
	}

	/* Override the OTG ID pin.  */
	ci->connector_role = role;
	pm_runtime_get_sync(ci->dev);
	ret = ci_switch_role(ci, role);
	pm_runtime_put_sync(ci->dev);
	mutex_unlock(&ci->role_switch_lock);
	if (ret)
		return ret;

	return count;
}

static DEVICE_ATTR(role, S_IRUSR | S_IWUSR, show_role, store_role);
static DEVICE_ATTR(vbus, S_IRUSR | S_IWUSR, show_vbus, store_vbus);
static DEVICE_ATTR(id, S_IRUSR | S_IWUSR, show_id, store_id);

static irqreturn_t ci_irq(int irq, void *data)
{
	struct ci13xxx *ci = data;
	u32 otgsc = 0;
	u32 portsc = 0;
	u32 reg = 0;
	u32 status = 0;

	if (atomic_read(&ci->in_lpm)) {
		disable_irq_nosync(irq);
		ci->wakeup_int = true;
		pm_runtime_get(ci->dev);
		return IRQ_HANDLED;
	}

	otgsc = hw_read(ci, OP_OTGSC, ~0);
	portsc = hw_read(ci, OP_PORTSC, ~0);
	status = hw_read(ci, OP_USBSTS, ~0);
	reg = status & hw_read(ci, OP_USBINTR, ~0);

	/*
	 * Handle id change interrupt, it indicates device/host function
	 * switch.
	 */
	if (ci->is_otg && (otgsc & OTGSC_IDIE) && (otgsc & OTGSC_IDIS)) {
		set_bit(CI_ID, &ci->events);
		hw_write(ci, OP_OTGSC, OTGSC_IDIS, OTGSC_IDIS);
		disable_irq_nosync(ci->irq);
		queue_work(ci->wq, &ci->work);
		return IRQ_HANDLED;
	}

	/*
	 * Handle vbus change interrupt, it indicates device connection
	 * and disconnection events.
	 */
	if ((otgsc & OTGSC_BSVIE) && (otgsc & OTGSC_BSVIS)) {
		set_bit(CI_B_SESS_VALID, &ci->events);
		hw_write(ci, OP_OTGSC, OTGSC_BSVIS, OTGSC_BSVIS);
		disable_irq_nosync(ci->irq);
		queue_work(ci->wq, &ci->work);
		return IRQ_HANDLED;
	}

	if ((otgsc & OTGSC_AVVIE) && (otgsc & OTGSC_AVVIS)) {
		hw_write(ci, OP_OTGSC, OTGSC_AVVIS, OTGSC_AVVIS);
		return IRQ_HANDLED;

	}

	if ((otgsc & OTGSC_ASVIE) && (otgsc & OTGSC_ASVIS)) {
		hw_write(ci, OP_OTGSC, OTGSC_ASVIS, OTGSC_ASVIS);
		return IRQ_HANDLED;
	}

	/* Handle over-current in gadget mode here.
	   Whenever there is a change in overcurrent status i.e
	   overcurrent appears or goes away, we call the overcurrent handling
	   routine. Overcurrent handling routine ci_handle_oc_change() will
	   disable port power for sometime if overcurrent is active and enable
	   power again to see if overcurrent still exists or gone.
	   It will send respective uevents to userspace then.
	*/
	if (ci->role == CI_ROLE_GADGET)
		if ((reg & USBi_PCI) && (portsc & PORTSC_OCC)) {
			hw_write(ci, OP_USBSTS, ~0, reg);
			set_bit(CI_OC, &ci->events);
			queue_work(ci->wq, &ci->work);
			return IRQ_HANDLED;
		}

	/* Handle device/host interrupt */
	if (ci->role == CI_ROLE_END || !ci_role(ci)->irq(ci)) {
		/* Unhandled interrupt, reset device */
		set_bit(CI_RESET, &ci->events);

		/* flush & stop before reset */
		hw_write(ci, OP_ENDPTFLUSH, ~0, ~0);
		hw_write(ci, OP_USBCMD, USBCMD_RS, 0);
		hw_write(ci, OP_USBCMD, USBCMD_RST, USBCMD_RST);

		/* continue the reset outside the irq */
		queue_work(ci->wq, &ci->work);
	}

	/* The Freescale i.MX6 sometimes does not detect a disconnect and
	 * reconnect of one of the signal lines on the USB BUS, i.e the GND.
	 * This is shown by a timeout of the chips interupts which fire
	 * continuously with the port change bits being unset and
	 * the error and normal completion status bits being set.
	 * Additionally these interupts fire without changing the
	 * port and usb status values.
	 * This function will detect when the signal has been reconnected
	 * and the port status bits 10 and 11 are set the J State.
	 */
	if ((ci->last_portsc != 0) &&
			(portsc & PORT_CONNECT) &&
			!(portsc & PORT_CSC) &&
			!(portsc & PORT_PEC) &&
			!(portsc & PORT_OCC) &&
			((portsc & (3<<10)) == (1<<11)) &&
			ci->last_portsc == portsc &&
			ci->last_status == status &&
			(status & STS_ERR) &&
			(status & STS_INT)) {

		/* Reset the Controller with the bad state */
		usb_queue_reset_device(ci->usb_interface);
	}
	ci->last_portsc = portsc;
	ci->last_status = status;

	return IRQ_HANDLED;
}

static DEFINE_IDA(ci_ida);

struct platform_device *ci13xxx_add_device(struct device *dev,
			struct resource *res, int nres,
			struct ci13xxx_platform_data *platdata)
{
	struct platform_device *pdev;
	int id, ret;

	id = ida_simple_get(&ci_ida, 0, 0, GFP_KERNEL);
	if (id < 0)
		return ERR_PTR(id);

	pdev = platform_device_alloc("ci_hdrc", id);
	if (!pdev) {
		ret = -ENOMEM;
		goto put_id;
	}

	pdev->dev.parent = dev;
	pdev->dev.dma_mask = dev->dma_mask;
	pdev->dev.dma_parms = dev->dma_parms;
	dma_set_coherent_mask(&pdev->dev, dev->coherent_dma_mask);

	ret = platform_device_add_resources(pdev, res, nres);
	if (ret)
		goto err;

	ret = platform_device_add_data(pdev, platdata, sizeof(*platdata));
	if (ret)
		goto err;

	ret = platform_device_add(pdev);
	if (ret)
		goto err;

	return pdev;

err:
	platform_device_put(pdev);
put_id:
	ida_simple_remove(&ci_ida, id);
	return ERR_PTR(ret);
}
EXPORT_SYMBOL_GPL(ci13xxx_add_device);

void ci13xxx_remove_device(struct platform_device *pdev)
{
	int id = pdev->id;
	platform_device_unregister(pdev);
	ida_simple_remove(&ci_ida, id);
}
EXPORT_SYMBOL_GPL(ci13xxx_remove_device);

static int ci_hdrc_probe(struct platform_device *pdev)
{
	struct device	*dev = &pdev->dev;
	struct ci13xxx	*ci;
	struct resource	*res;
	void __iomem	*base;
	int		ret;
	u32		otgsc;

	if (!dev->platform_data) {
		dev_err(dev, "platform data missing\n");
		return -ENODEV;
	}

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res) {
		dev_err(dev, "missing resource\n");
		return -ENODEV;
	}

	base = devm_request_and_ioremap(dev, res);
	if (!res) {
		dev_err(dev, "can't request and ioremap resource\n");
		return -ENOMEM;
	}

	ci = devm_kzalloc(dev, sizeof(*ci), GFP_KERNEL);
	if (!ci) {
		dev_err(dev, "can't allocate device\n");
		return -ENOMEM;
	}

	ci->events = 0;
	ci->overcurrent = false;
	ci->dev = dev;
	ci->platdata = dev->platform_data;
	if (ci->platdata->phy)
		ci->transceiver = ci->platdata->phy;
	else
		ci->global_phy = true;

	ret = hw_device_init(ci, base);
	if (ret < 0) {
		dev_err(dev, "can't initialize hardware\n");
		return -ENODEV;
	}

	ci->hw_bank.phys = res->start;

	ci->irq = platform_get_irq(pdev, 0);
	if (ci->irq < 0) {
		dev_err(dev, "missing IRQ\n");
		return -ENODEV;
	}

	INIT_WORK(&ci->work, ci_otg_work);
	INIT_DELAYED_WORK(&ci->dwork, ci_delayed_work);
	ci->wq = create_singlethread_workqueue("ci_otg");
	if (!ci->wq) {
		dev_err(dev, "can't create workqueue\n");
		return -ENODEV;
	}

	/*
	 * Set wakeup capable before the child device is created.
	 * As at child device (roothub)'s init, it will set controller
	 * device as wakeup enabled. If we does not put it before creating
	 * roothub, the wakeup will not be enabled by default.
	 */
	device_set_wakeup_capable(&pdev->dev, true);

	/* Disable all interrupts bits */
	hw_write(ci, OP_USBINTR, 0xffffffff, 0);
	hw_write(ci, OP_OTGSC, OTGSC_INT_EN_BITS, 0);

	/* Clear all interrupts status bits*/
	hw_write(ci, OP_USBSTS, 0xffffffff, 0xffffffff);
	hw_write(ci, OP_OTGSC, OTGSC_INT_STATUS_BITS, OTGSC_INT_STATUS_BITS);

	/* initialize role(s) before the interrupt is requested */
	ret = ci_hdrc_host_init(ci);
	if (ret)
		dev_info(dev, "doesn't support host\n");

	ret = ci_hdrc_gadget_init(ci);
	if (ret)
		dev_info(dev, "doesn't support gadget, ret=%d\n", ret);

	if (!ci->roles[CI_ROLE_HOST] && !ci->roles[CI_ROLE_GADGET]) {
		dev_err(dev, "no supported roles\n");
		ret = -ENODEV;
		goto rm_wq;
	}

	if (ci->roles[CI_ROLE_HOST] && ci->roles[CI_ROLE_GADGET]) {
		dev_dbg(dev, "support otg\n");
		ci->is_otg = true;
		/* ID pin needs 1ms debouce time, we delay 2ms for safe */
		mdelay(2);
		ci->role = CI_ROLE_HOST;
		ci->connector_role = ci->role;
	} else {
		ci->role = ci->roles[CI_ROLE_HOST]
			? CI_ROLE_HOST
			: CI_ROLE_GADGET;
		ci->connector_role = ci->role;
	}

	if (ci->is_otg)
		/* if otg is supported, create struct usb_otg */
		ci_hdrc_otg_init(ci);

	otgsc = hw_read(ci, OP_OTGSC, ~0);

	if (ci->roles[CI_ROLE_GADGET]) {
		ret = ci->roles[CI_ROLE_GADGET]->init(ci);
		if (ret) {
			dev_err(dev, "can't init %s role, ret=%d\n",
					ci_role(ci)->name, ret);
			ret = -ENODEV;
			goto rm_wq;
		}
		/*
		 * if it is device mode:
		 * - Enable vbus detect
		 * - If it has already connected to host, notify udc
		 */
		if (ci->role == CI_ROLE_GADGET) {
			hw_write(ci, OP_OTGSC, OTGSC_BSVIE, OTGSC_BSVIE);
			ci_handle_vbus_change(ci);
		}
	}

	if (ci->role == CI_ROLE_HOST) {
		ret = ci->roles[CI_ROLE_HOST]->init(ci);
		if (ret) {
			dev_err(dev, "can't init %s role, ret=%d\n",
					ci_role(ci)->name, ret);
			ret = -ENODEV;
			goto rm_wq;
		}
	}

	hw_write(ci, OP_OTGSC, OTGSC_ASVIE, OTGSC_ASVIE);
	hw_write(ci, OP_OTGSC, OTGSC_AVVIE, OTGSC_AVVIE);

	platform_set_drvdata(pdev, ci);

	mutex_init(&ci->role_switch_lock);
	ret = device_create_file(dev, &dev_attr_role);
	if (ret)
		goto rm_attr;

	if (ci->is_otg) {
		ret = device_create_file(dev, &dev_attr_vbus);
		if (ret)
			goto rm_attr;

		ret = device_create_file(dev, &dev_attr_id);
		if (ret)
			goto rm_attr;
	}

	ret = devm_request_irq(dev, ci->irq, ci_irq, IRQF_SHARED,
			       ci->platdata->name, ci);
	if (ret)
		goto rm_attr;

	/* Handle the situation that usb device at the MicroB to A cable */
	if (ci->is_otg && !(otgsc & OTGSC_ID))
		queue_delayed_work(ci->wq, &ci->dwork, msecs_to_jiffies(500));

	/*
	 * Its parent controller driver will take pm responsibility
	 */
	pm_runtime_set_active(&pdev->dev);
	pm_runtime_enable(&pdev->dev);
	pm_runtime_no_callbacks(&pdev->dev);

	/* Set the root main configuration. */
	ci->usb_interface = usb_ifnum_to_if(ci->hcd->self.root_hub, 0);

	return ret;

rm_attr:
	mutex_destroy(&ci->role_switch_lock);
	device_remove_file(dev, &dev_attr_role);
	device_remove_file(dev, &dev_attr_vbus);
	device_remove_file(dev, &dev_attr_id);
	ci_role_stop(ci);
rm_wq:
	flush_workqueue(ci->wq);
	destroy_workqueue(ci->wq);

	return ret;
}

static int ci_hdrc_remove(struct platform_device *pdev)
{
	struct ci13xxx *ci = platform_get_drvdata(pdev);
	int ret;

	devm_free_irq(&pdev->dev, ci->irq, ci);
	cancel_delayed_work_sync(&ci->dwork);
	flush_workqueue(ci->wq);
	destroy_workqueue(ci->wq);
	mutex_destroy(&ci->role_switch_lock);

	ret = pm_runtime_get_sync(&pdev->dev);
	if (IS_ERR_VALUE(ret))
		return ret;

	pm_runtime_disable(&pdev->dev);
	pm_runtime_put_noidle(&pdev->dev);

	device_remove_file(ci->dev, &dev_attr_role);
	device_remove_file(ci->dev, &dev_attr_vbus);
	device_remove_file(ci->dev, &dev_attr_id);

	if (ci->roles[CI_ROLE_GADGET] && ci->role != CI_ROLE_GADGET)
		ci->roles[CI_ROLE_GADGET]->destroy(ci);
	ci_role_destroy(ci);

	return 0;
}

static struct platform_driver ci_hdrc_driver = {
	.probe	= ci_hdrc_probe,
	.remove	= ci_hdrc_remove,
	.driver	= {
		.name	= "ci_hdrc",
	},
};

static int __init ci_hdrc_driver_init(void)
{
	ci_hdrc_ehci_init();
	return platform_driver_register(&ci_hdrc_driver);
}
module_init(ci_hdrc_driver_init);

static void __exit ci_hdrc_driver_cleanup(void)
{
	platform_driver_unregister(&ci_hdrc_driver);
}
module_exit(ci_hdrc_driver_cleanup);

MODULE_ALIAS("platform:ci_hdrc");
MODULE_ALIAS("platform:ci13xxx");
MODULE_LICENSE("GPL v2");
MODULE_AUTHOR("David Lopo <dlopo@chipidea.mips.com>");
MODULE_DESCRIPTION("ChipIdea HDRC Driver");
