#include <linux/platform_device.h>
#include <linux/uio_driver.h>
#include <linux/spinlock.h>
#include <linux/bitops.h>
#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/stringify.h>
#include <linux/pm_runtime.h>
#include <linux/slab.h>
#include <linux/irq.h>
#include <linux/of.h>
#include <linux/of_platform.h>
#include <linux/of_address.h>
#include <linux/of_gpio.h>
#include <linux/pinctrl/consumer.h>

#define DRIVER_NAME "uio-gnss-spider"

#define GNSS_FLAG_IRQ_ENABLED 0

struct uio_gnss_spider_platdata {
	struct uio_info         *uioinfo;
	spinlock_t              lock;
	unsigned long           flags;
	struct platform_device  *pdev;
	unsigned long           int_ms_gpio;
};

static irqreturn_t uio_gnss_spider_handler(int irq, struct uio_info *uioinfo)
{
	struct uio_gnss_spider_platdata *pdata = uioinfo->priv;

	dev_dbg(&pdata->pdev->dev, "gnss-spider interrupt\n");

	if (!test_and_set_bit(GNSS_FLAG_IRQ_ENABLED, &pdata->flags))
		disable_irq_nosync(irq);

	return IRQ_HANDLED;
}

static int uio_gnss_spider_irqcontrol(struct uio_info *uioinfo, s32 irq_on)
{
	struct uio_gnss_spider_platdata *pdata = uioinfo->priv;
	unsigned long flags;

	spin_lock_irqsave(&pdata->lock, flags);
	if (irq_on) {
		if (test_and_clear_bit(GNSS_FLAG_IRQ_ENABLED, &pdata->flags))
			enable_irq(uioinfo->irq);
	} else {
		if (!test_and_set_bit(GNSS_FLAG_IRQ_ENABLED, &pdata->flags))
			disable_irq(uioinfo->irq);
	}
	spin_unlock_irqrestore(&pdata->lock, flags);

	return 0;
}

static int uio_gnss_spider_probe(struct platform_device *pdev)
{
	struct uio_info                 *uioinfo = pdev->dev.platform_data;
	struct uio_gnss_spider_platdata *pdata   = NULL;
	struct uio_mem                  *uiomem  = NULL;
	int                             ret     = -EINVAL;
	int                             i       = 0;
	struct device_node              *np     = pdev->dev.of_node;
	struct pinctrl			*pinctrl = NULL;


	if (!np) {
		ret = -ENOMEM;
		dev_err(&pdev->dev, "null pointer device node\n");
		goto err_null_device_node;
	}

	if (!uioinfo) {
		uioinfo = kzalloc(sizeof(*uioinfo), GFP_KERNEL);
		if (!uioinfo) {
			ret = -ENOMEM;
			dev_err(&pdev->dev, "unable to kmalloc\n");
			goto err_alloc_uioinfo;
		}
		uioinfo->name = pdev->dev.of_node->name;
		uioinfo->version = "v1.0";
	}

	if (!uioinfo || !uioinfo->name || !uioinfo->version) {
		dev_err(&pdev->dev, "missing platform_data\n");
		goto err_uioinfo;
	}

	if (uioinfo->handler || uioinfo->irqcontrol ||
	    uioinfo->irq_flags & IRQF_SHARED) {
		dev_err(&pdev->dev, "interrupt configuration error\n");
		goto err_uioinfo;
	}

	pdata = kzalloc(sizeof(*pdata), GFP_KERNEL);
	if (!pdata) {
		ret = -ENOMEM;
		dev_err(&pdev->dev, "unable to kmalloc\n");
		goto err_alloc_pdata;
	}

	pdata->uioinfo = uioinfo;
	spin_lock_init(&pdata->lock);
	pdata->flags = GNSS_FLAG_IRQ_ENABLED;
	pdata->pdev = pdev;

	/* enable pinmux setting configured in device tree */
	pinctrl = devm_pinctrl_get_select_default(&pdev->dev);
	if (IS_ERR(pinctrl)) {
		ret = PTR_ERR(pinctrl);
		goto err_pinctrl;
	}

	pdata->int_ms_gpio = of_get_named_gpio(np, "int_ms-gpios", 0);

	if (!gpio_is_valid(pdata->int_ms_gpio)) {
		ret = -EIO;
		goto err_gpio;
	}

	uioinfo->irq = gpio_to_irq(pdata->int_ms_gpio);

	uiomem = &uioinfo->mem[0];

	for (i = 0; i < pdev->num_resources; ++i) {
		struct resource *r = &pdev->resource[i];

		if (r->flags != IORESOURCE_MEM)
			continue;

		if (uiomem >= &uioinfo->mem[MAX_UIO_MAPS]) {
			dev_warn(&pdev->dev, "device has more than "
					__stringify(MAX_UIO_MAPS)
					" I/O memory resources.\n");
			break;
		}

		uiomem->memtype = UIO_MEM_PHYS;
		uiomem->addr = r->start;
		uiomem->size = resource_size(r);
		++uiomem;
	}

	while (uiomem < &uioinfo->mem[MAX_UIO_MAPS]) {
		uiomem->size = 0;
		++uiomem;
	}

	uioinfo->handler = uio_gnss_spider_handler;
	uioinfo->irqcontrol = uio_gnss_spider_irqcontrol;
	uioinfo->priv = pdata;

	ret = uio_register_device(&pdev->dev, pdata->uioinfo);
	if (ret) {
		dev_err(&pdev->dev, "unable to register uio device\n");
		goto err_uio_register_device;
	}

	platform_set_drvdata(pdev, pdata);

	ret = irq_set_irq_type(uioinfo->irq, IRQ_TYPE_EDGE_FALLING);

	if (ret) {
		dev_err(&pdev->dev, "unable to set irq type\n");
		goto err_set_irq_type;
	}

	dev_dbg(&pdev->dev, "gnss-spider: probed, int_ms_gpio: %lu, irq: %ld\n",
		pdata->int_ms_gpio, uioinfo->irq);

	return 0;

err_set_irq_type:
	uio_unregister_device(pdata->uioinfo);
err_uio_register_device:
err_gpio:
err_pinctrl:
	kfree(pdata);
err_uioinfo:
err_alloc_pdata:
	if (pdev->dev.of_node)
		kfree(uioinfo);
err_alloc_uioinfo:
err_null_device_node:
	return ret;
}

static int uio_gnss_spider_remove(struct platform_device *pdev)
{
	struct uio_gnss_spider_platdata *pdata = platform_get_drvdata(pdev);

	uio_unregister_device(pdata->uioinfo);

	pdata->uioinfo->handler = NULL;
	pdata->uioinfo->irqcontrol = NULL;

	if (pdev->dev.of_node)
		kfree(pdata->uioinfo);

	kfree(pdata);
	return 0;
}

static const struct of_device_id uio_gnss_spider_match[] = {
	{ .compatible = "adit,gnss-spider",},
	{},
};
MODULE_DEVICE_TABLE(of, uio_gnss_spider_match);

static struct platform_driver uio_gnss_spider = {
	.probe = uio_gnss_spider_probe,
	.remove = uio_gnss_spider_remove,
	.driver = {
		.name = DRIVER_NAME,
		.owner = THIS_MODULE,
		.of_match_table = uio_gnss_spider_match,
	},
};

module_platform_driver(uio_gnss_spider);

MODULE_AUTHOR("Wipro Ltd");
MODULE_DESCRIPTION("Userspace I/O Spider GNSS Driver");
MODULE_LICENSE("GPL v2");
MODULE_ALIAS("platform:" DRIVER_NAME);
