/*
 * SSI32 driver for INC
 *
 * Copyright (C) 2012 Andreas Pape, Robert Bosch Car Multimedia GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/hrtimer.h>
#include <linux/slab.h>
#include <linux/sched.h>
#include <linux/spi/spi.h>
#include <linux/netdevice.h>
#include "ssi32_prot.h"
#include "ssi32_dev.h"
#include "inc_netdev.h"

#define SSI_MAX_PORTS 8
#define SSI_MAX_DEVPERPORT 2 /*multiple device support; tx only on 1st slave*/
#define SSI32_MTU 4096	/*default MTU - can change to runtime*/
#define SSI_PEND_LIMIT 1 /*number of msg accepted per lun (if not XOFF)*/

/*interval of periodic xoff timer*/
#define SSI_XOFF_TINTERVAL_MS 10
#define SSI_XOFF_TINTERVAL_JIFFIES (HZ/(1000/SSI_XOFF_TINTERVAL_MS))

/*tout in amount of xoff timer intervals
real timeout is 0..SSI_XOFF_TINTERVAL_MS ms longer */
#define SSI_XOFF_TOUT_INTERVALS 1

/*maximum time to wait for transfer to finish when stopping master*/
#define SSI_ABORT_TOUT (1*(HZ)) /*1 second*/

/*for debug purpose:
if defined: stop communication on error
else: go on with retry etc.
*/
#undef SSI32_HOLD_ON_ERROR
#undef SSI32_TRIGGER_ON_UNEXPECTED


static struct ssi32_master *ssi32_ports[SSI_MAX_PORTS];
static DEFINE_MUTEX(ssi32_master_list_lock);


#define SSI_LUN_INVAL 0xff
#define SSI_LUN_MAX   256

#define SSI_MSG_POOL_SIZE 100
#define SSI_RETRY_BLK_MAX 30
#define SSI_RETRY_MSG_MAX 10
/*

SSI32 part of INC communication

*/
#define SSI32_BLOCKLEN(t) ((t)->slave->cfg.blocksize)
#define SSI32_HEADLEN 4
#define SSI32_PAYLLEN(t) (SSI32_BLOCKLEN(t)-SSI32_HEADLEN)
#define SSI32_ACKLEN 4
#define SSI32_MBHD(t) ((t)->slave->cfg.mbhdlen)

enum SSI_STA {
	SSI_STA_IDLE = 0,
	SSI_STA_START_MDATA,
	SSI_STA_DLY_XFER_DATA,
	SSI_STA_XFER_DATA,
	SSI_STA_END_DATA,

	SSI_STA_START_ACK,
	SSI_STA_DLY_XFER_ACK,
	SSI_STA_XFER_ACK,
	SSI_STA_END_ACK,
	SSI_STA_RECOVER,
	SSI_NUM_STATES,
};

enum SSI_EVT {
	SSI_EVT_SRQ_L = 0x0004,
	SSI_EVT_SRQ_H = 0x0008,
	SSI_EVT_TMO = 0x0010,
	SSI_EVT_SPI_COMPLETE = 0x0020,
};


enum SSI_TIMER {
	SSI_TIMER_SSR = 0,
	SSI_TIMER_CD1,
	SSI_TIMER_CD2,
	SSI_TIMER_ESR,
	SSI_TIMER_RECOVER,
	SSI_TIMER_INVAL = 255,
};


/*time spend on scheduling and processing,
this has to be substracted from the desired timervalues*/
#define SSI_TCD1_OVRHD (25*1000)
#define SSI_TCD2_OVRHD (25*1000)

static unsigned int ssi_timtout_ovrhd[] = {
	0,
	SSI_TCD1_OVRHD,
	SSI_TCD2_OVRHD,
	0,
	0,
};



static unsigned int ssi_timtout_ns[] = {
	1000*1000*10,	/*SSR timeout, waiting for SRQ_L*/
	1000*10,	/*CD1 delay before starting XFER*/
	1000*10,	/*CD2 delay before starting XFER on slave request*/
	1000*1000*10,	/*ESR timeout waiting for SRQ_H*/
	1000*1000*1,	/*RECOVER waittime on recover from error*/
};

#define SSI32_TXRX_DEFAULT	0x00
/*status on TX block*/
#define SSI32_TX_DEFAULT	(0x0000)
#define SSI32_TX_NO_MSG		(0x0001)
#define SSI32_TX_DATA_FINISHED	(0x0002)
#define SSI32_TX_ERR		(0x0004)


#define SSI32_TX_CHK_ERR	(0x01 << 3)
#define SSI32_TX_SEQ_ERR	(0x02 << 3)
#define SSI32_TX_SYN_ERR	(0x04 << 3)
#define SSI32_TX_XOFF		(0x08 << 3)
#define SSI32_TX_HANDSHAKE_ERR	(0x10 << 3)
#define SSI32_TX_HW_XOFF	(0x20 << 3)


#define SSI32_TX_MASK_RETRY_MSG (SSI32_TX_SEQ_ERR)
#define SSI32_TX_MASK_RETRY_BLK\
		(SSI32_TX_CHK_ERR|SSI32_TX_HANDSHAKE_ERR|SSI32_TX_SYN_ERR)

/*status on RX block*/
#define SSI32_RX_SHIFT 16

#define SSI32_RX_DEFAULT	(SSI32_TX_DEFAULT << SSI32_RX_SHIFT)
#define SSI32_RX_NO_MSG		(SSI32_TX_NO_MSG << SSI32_RX_SHIFT)
#define SSI32_RX_DATA_FINISHED	(SSI32_TX_DATA_FINISHED << SSI32_RX_SHIFT)
#define SSI32_RX_ERR		(SSI32_TX_ERR << SSI32_RX_SHIFT)

#define SSI32_RX_CHK_ERR	(SSI32_TX_CHK_ERR << SSI32_RX_SHIFT)
#define SSI32_RX_SEQ_ERR	(SSI32_TX_SEQ_ERR << SSI32_RX_SHIFT)
#define SSI32_RX_SYN_ERR	(SSI32_TX_SYN_ERR << SSI32_RX_SHIFT)
#define SSI32_RX_XOFF_ERR	(SSI32_TX_XOFF << SSI32_RX_SHIFT)
#define SSI32_RX_HANDSHAKE_ERR	(SSI32_TX_HANDSHAKE_ERR << SSI32_RX_SHIFT)



struct ssi32_lun_msg {
	struct list_head lhead;
	struct sk_buff *skb;
	size_t remain;
	size_t pend_len;
	unsigned char *pos; /*todo calc from SKB (tail - remain*/

	unsigned char retry_blk;
	unsigned char retry_msg;
	unsigned char lun;
	unsigned char ctrl; /*on tx: ctrl to send, on rx: last received ctrl*/
};

struct ssi32_lun {
	struct list_head tx_luns;	/*entry for slave->tx_luns*/
	unsigned char pend_cnt;
	unsigned char xoff_dly;		/*xoff delay*/
	unsigned char lun;
	struct ssi32_lun_msg *rx_msg;	/*1 rx transfer*/
	struct list_head tx_pending;	/*multiple tx transfers*/
};

#define SSI_UNSIGNED_BF_SIZE ((SSI_LUN_MAX+(BITS_PER_LONG-1))/BITS_PER_LONG)

struct ssi32_slavedev {
	struct ssi32_dcfg cfg;
	struct ssi32_device *ssidev;
	struct ssi32_master *master;
	unsigned int spurious;
	int ready;				/*slave device status*/

	struct ssi32_lun_msg *msg_pool_mem;	/*memory for msg pool*/

	unsigned char *rx_data_buf;		/*rx buf for actual transfer*/
	unsigned char *tx_data_buf;		/*tx buf for actual transfer*/

	unsigned char rx_ack_buf[SSI32_ACKLEN];	/*rx ack for actual transfer*/
	unsigned char tx_ack_buf[SSI32_ACKLEN];	/*tx ack for actual transfer*/

	unsigned long bf_xoff[SSI_UNSIGNED_BF_SIZE];	/*luns with XOFF*/
	unsigned long bf_xoff_sig[SSI_UNSIGNED_BF_SIZE];/*XOFF signalled*/
	unsigned long bf_rtr[SSI_UNSIGNED_BF_SIZE];/*expected retry*/
	unsigned long bf_rx_ready[SSI_UNSIGNED_BF_SIZE];/*lun ready to rx data*/

	int hw_xoff;				/*hardware xoff status*/
	int hw_xoff_retry;			/*nof hardware xoff retries*/

	struct timer_list xoff_timer;
	struct timer_list hw_xoff_timer;

	struct ssi32_lun luns[SSI_LUN_MAX];
	struct list_head msg_pool;
	struct list_head tx_luns;		/*list of tx lists*/
};


#define toNdev(trf) ((trf)->slave->master->ndev)
#define toDev(trf) ((trf)->slave->master->ndev->ndev->dev)

struct ssi32_transfer {
	enum SSI_STA state;
	enum SSI_TIMER running; /*actual armed timer*/
	struct ssi32_slavedev *slave;

	struct ssi32_lun_msg *rx;
	struct ssi32_lun_msg *tx;

	unsigned int err;
};

struct ssi32_master {
	int bus;
	struct inc_ndev *ndev;
	struct ssi32_slavedev *ssi32_slaves[SSI_MAX_DEVPERPORT];
	unsigned int srq_pend; /*pending slave requests*/
	struct ssi32_transfer transfer;
	struct hrtimer timer;
	spinlock_t smlock;
	unsigned long irq_flags;
	unsigned long flags;
	wait_queue_head_t wait;
	struct task_struct *task;
	struct sk_buff *skb;
	struct inc_hdr *hdr;
#ifdef SSI32_HOLD_ON_ERROR
	unsigned char stopped;
#endif
	struct ssi32_slavedev *bus_locked; /*slave that locked the bus*/
};

#define lock_sm(m, flg) spin_lock_irqsave(&(m)->smlock, flg)
#define unlock_sm(m, flg) spin_unlock_irqrestore(&(m)->smlock, flg)

static int ssi32_data_transfer(struct ssi32_transfer *trf);
static int ssi32_ack_transfer(struct ssi32_transfer *trf);
static int check_rx_data(struct ssi32_transfer *trf, unsigned char *rx_buf);
static int rx_data_is_ok(struct ssi32_transfer *trf);
static int setup_tx_data(struct ssi32_transfer *trf, unsigned char *tx_buf);
static int check_rx_ack(struct ssi32_transfer *trf, unsigned char *rx_buf);
static int ssi_finishTrf(struct ssi32_transfer *trf, enum SSI_EVT evt);
static int chkSRQL(struct ssi32_slavedev *slave);
static int ssi32_schedule_preprocessing(struct ssi32_master *m);


/*DEBUG*/

static const char *ssi_sta_string(enum SSI_STA sta)
{
	switch (sta) {
	case SSI_STA_IDLE:
		return "IDLE       ";
	case SSI_STA_START_MDATA:
		return "START_MDATA";
	case SSI_STA_XFER_DATA:
		return "XFER_DATA  ";
	case SSI_STA_DLY_XFER_DATA:
		return "DLY_XFER_DATA";
	case SSI_STA_END_DATA:
		return "END_DATA   ";
	case SSI_STA_START_ACK:
		return "START_ACK  ";
	case SSI_STA_DLY_XFER_ACK:
		return "DLY_XFER_ACK";
	case SSI_STA_XFER_ACK:
		return "XFER_ACK   ";
	case SSI_STA_END_ACK:
		return "END_ACK    ";
	case SSI_STA_RECOVER:
		return "RECOVER    ";
	default:
		return "-INVALID-  ";
	}
};

static const char *ssi_evt_string(enum SSI_EVT evt)
{
	switch (evt) {
	case SSI_EVT_SRQ_L:
		return "SRQ_L    ";
	case SSI_EVT_SRQ_H:
		return "SRQ_H    ";
	case SSI_EVT_TMO:
		return "TIMEOUT  ";
	case SSI_EVT_SPI_COMPLETE:
		return "SSI_EVT_SPI_COMPLETE ";
	default:
		return "-INVALID-";
	}
};


/*finalize:	0 trace actual status
		1 printout and clear history
		2 clear history
*/
static void ssi_history(enum SSI_EVT evt, enum SSI_STA new_sta, int finalize,
					struct ssi32_transfer *trf)
{
#define SSI_HIST_MAX 20
	static int idx;
	int i;
	static enum SSI_EVT evt_hist[SSI_HIST_MAX];
	static enum SSI_STA new_sta_hist[SSI_HIST_MAX];

	if ((idx > SSI_HIST_MAX) && (!finalize)) {
		/*can happen on many ignored srq changes in recover state*/
		pr_debug("SSI32: history overflow\n");
		return;
	}

	evt_hist[idx] = evt;
	new_sta_hist[idx] = new_sta;
	idx++;

	if (!finalize)
		return;
	if (finalize == 1) {
		pr_debug("SSI32: HISTORY - err TX 0x%04x RX 0x%04x",
				trf->err&0xffff, (trf->err&0xffff0000)>>16);
		for (i = 0; i < idx; i++) {
			pr_debug("SSI32: EVT %s -->STA %s\n",
			ssi_evt_string(evt_hist[i]),
			ssi_sta_string(new_sta_hist[i]));
		}
	}
	idx = 0;
	memset(evt_hist, 0, sizeof(evt_hist));
}




MODULE_LICENSE("GPL");

/*flag bits, accessed atomically*/
#define SSI_DO_WORK	1	/*something to do for worker*/
#define SSI_BUSY	2	/*transfer in progress*/
#define SSI_ABORT	3	/*abort of ongoing transfer in progress*/


#define ssi_slave_select(trf) ssi32_chipselect(trf->slave->ssidev, 1)
#define ssi_slave_deselect(trf) ssi32_chipselect(trf->slave->ssidev, 0)

static int ssi_stop_timer(struct ssi32_transfer *trf, enum SSI_TIMER timer)
{
	if (trf->running != SSI_TIMER_INVAL) {
		/*hrtimer_cancel can deadlock via lock_sm if callback
		is just executing. So we use trf->running for syncing*/
		hrtimer_try_to_cancel(&trf->slave->master->timer);
		trf->running = SSI_TIMER_INVAL;
	}
	return 0;
}

static unsigned int ssi_timeout_ns(struct ssi32_slavedev *slave,
		enum SSI_TIMER timer)
{
	unsigned int timeout = 0;

	switch (timer) {
	case SSI_TIMER_SSR:
		timeout = slave->cfg.tssr;
		break;
	case SSI_TIMER_CD1:
		timeout = slave->cfg.tcd1;
		break;
	case SSI_TIMER_CD2:
		timeout = slave->cfg.tcd2;
		break;
	case SSI_TIMER_ESR:
		timeout = slave->cfg.tesr;
		break;
	default:
		dev_err(&slave->master->ndev->ndev->dev,
				"invalid SSI_TIMER %d", timer);
		break;
	}

	return timeout;
}

static int ssi_start_timer(struct ssi32_transfer *trf, enum SSI_TIMER timer)
{
	ktime_t tout;
	unsigned int timeout_ns = ssi_timeout_ns(trf->slave, timer);

	ssi_stop_timer(trf, SSI_TIMER_INVAL);

	if (timeout_ns <= ssi_timtout_ovrhd[timer])
		return 1;

	tout = ktime_set(0, timeout_ns-ssi_timtout_ovrhd[timer]);
	trf->running = timer;
	hrtimer_start(&trf->slave->master->timer, tout,
						HRTIMER_MODE_REL_PINNED);
	return 0;
}

static int ssi_newState(struct ssi32_transfer *trf, enum SSI_STA state)
{
	trf->state = state;
	return 0;
}


static int ssi_unexpected(struct ssi32_transfer *trf, enum SSI_EVT evt)
{
	/*there is no safe way to signal handshake errors to slave -
	so we completely rely on checksum and ack.
	on TX: as soon as we have sent the data we have no chance to invalidate
	the xfer - even if handshake errors are detected
	on RX: we could signal handshake errors as SEQ/CHKSUM err until we have
	not sent the ack. But if CHKSUM of received packet seems OK there is
	no need for that.
	*/
#if 0
	/*reset line*/
	trf->err |= SSI32_TX_ERR|SSI32_TX_HANDSHAKE_ERR|
			SSI32_RX_HANDSHAKE_ERR|SSI32_RX_ERR;
	ssi_slave_deselect(trf);

#ifdef SSI32_TRIGGER_ON_UNEXPECTED
	ssi_slave_select(trf);
	ssi_slave_deselect(trf);
#endif
	/*TODO recover should only concern a specific device!!*/
	ssi_start_timer(trf, SSI_TIMER_RECOVER);
	ssi_newState(trf, SSI_STA_RECOVER);
#endif
	return 0;
}



/*checks srq line: 0 if high, !0 if low*/
static int chkSRQL(struct ssi32_slavedev *slave)
{
	return slave->master->srq_pend & (1<<slave->cfg.node);
}


static void on_tout_cd1_cd2(struct ssi32_transfer *trf, enum SSI_STA new_sta)
{
	ssi_newState(trf, new_sta);
	if (new_sta == SSI_STA_XFER_DATA)
		ssi32_data_transfer(trf);
	else
		ssi32_ack_transfer(trf);
}

static int ssi_hw_xoff_by_slave(struct ssi32_transfer *trf, enum SSI_EVT evt)
{
	if (trf->slave->cfg.tmri == 0 || trf->slave->cfg.mnr == 0)
		return 1;

	ssi_slave_deselect(trf);

	trf->err |= SSI32_TX_ERR|SSI32_TX_HW_XOFF|SSI32_RX_NO_MSG;

	dev_dbg(&toDev(trf), "%s: trf->err 0x%x\n", __func__, trf->err);

	ssi_stop_timer(trf, SSI_TIMER_INVAL);
	ssi_finishTrf(trf, evt);

	return 0;
}



static int ssi_IDLE(struct ssi32_transfer *trf, enum SSI_EVT evt)
{
	switch (evt) {
	default:
		ssi_unexpected(trf, evt);
		break;
	}
	return 0;
}

/*SSR running*/
static int ssi_START_MDATA(struct ssi32_transfer *trf, enum SSI_EVT evt)
{
	switch (evt) {
	case SSI_EVT_TMO:
		dev_dbg(&toDev(trf), "%s: t_SSR expired\n", __func__);
		/* abort transfer if HW XOFF is enabled, else continue */
		if (!ssi_hw_xoff_by_slave(trf, evt))
			break;
		ssi_unexpected(trf, evt);
		/*no break*/
	case SSI_EVT_SRQ_L:
		if (ssi_start_timer(trf, SSI_TIMER_CD1))
			on_tout_cd1_cd2(trf, SSI_STA_XFER_DATA);
		else
			ssi_newState(trf, SSI_STA_DLY_XFER_DATA);
		break;
	default:
		ssi_unexpected(trf, evt);
		break;
	}
	return 0;
}

/*CD1/CD2 running*/
static int ssi_dly_XFER_DATA(struct ssi32_transfer *trf, enum SSI_EVT evt)
{
	switch (evt) {
	case SSI_EVT_TMO:
		on_tout_cd1_cd2(trf, SSI_STA_XFER_DATA);
		break;
	default:
		ssi_unexpected(trf, evt);
		break;
	}
	return 0;

}
static int ssi_XFER_DATA(struct ssi32_transfer *trf, enum SSI_EVT evt)
{
	switch (evt) {
	case SSI_EVT_SPI_COMPLETE:
		if (ssi32_uses_hw_ack(trf->slave->ssidev)) {
			trf->err |= check_rx_data(trf, trf->slave->rx_data_buf);
			ssi32_hw_ack(trf->slave->ssidev, rx_data_is_ok(trf));
			ssi_slave_deselect(trf);
		} else {
			ssi_slave_deselect(trf);
			trf->err |= check_rx_data(trf, trf->slave->rx_data_buf);
		}

		if (chkSRQL(trf->slave)) {
			ssi_start_timer(trf, SSI_TIMER_ESR);
			ssi_newState(trf, SSI_STA_END_DATA);
		} else {
			if (ssi32_uses_hw_ack(trf->slave->ssidev)) {
				ssi_stop_timer(trf, SSI_TIMER_INVAL);
				ssi_finishTrf(trf, evt);
			} else {
				ssi_slave_select(trf);
				ssi_start_timer(trf, SSI_TIMER_SSR);
				ssi_newState(trf, SSI_STA_START_ACK);
			}
		}
		break;

	default:
		ssi_unexpected(trf, evt);
		break;
	}
	return 0;
}
static int ssi_END_DATA(struct ssi32_transfer *trf, enum SSI_EVT evt)
{
	switch (evt) {
	case SSI_EVT_TMO:
		dev_alert(&toDev(trf), "%s: t_ESR expired\n", __func__);
		/* abort transfer if HW XOFF is enabled, else continue */
		if (!ssi_hw_xoff_by_slave(trf, evt))
			break;
		ssi_unexpected(trf, evt);
		/*no break*/
	case SSI_EVT_SRQ_H:
		if (ssi32_uses_hw_ack(trf->slave->ssidev)) {
			ssi_stop_timer(trf, SSI_TIMER_INVAL);
			ssi_finishTrf(trf, evt);
		} else {
			ssi_slave_select(trf);
			ssi_start_timer(trf, SSI_TIMER_SSR);
			ssi_newState(trf, SSI_STA_START_ACK);
		}
		break;
	default:
		ssi_unexpected(trf, evt);
		break;
	}
	return 0;
}

static int ssi_START_ACK(struct ssi32_transfer *trf, enum SSI_EVT evt)
{
	switch (evt) {
	case SSI_EVT_TMO:
		dev_dbg(&toDev(trf), "%s: t_SSR expired\n", __func__);
		/* abort transfer if HW XOFF is enabled, else continue */
		if (!ssi_hw_xoff_by_slave(trf, evt))
			break;
		ssi_unexpected(trf, evt);
		/*no break*/
	case SSI_EVT_SRQ_L:
		if (ssi_start_timer(trf, SSI_TIMER_CD1))
			on_tout_cd1_cd2(trf, SSI_STA_XFER_ACK);
		else
			ssi_newState(trf, SSI_STA_DLY_XFER_ACK);
		break;
	default:
		ssi_unexpected(trf, evt);
		break;
	}
	return 0;
}

static int ssi_dly_XFER_ACK(struct ssi32_transfer *trf, enum SSI_EVT evt)
{
	switch (evt) {
	case SSI_EVT_TMO:
		on_tout_cd1_cd2(trf, SSI_STA_XFER_ACK);
		break;
	default:
		ssi_unexpected(trf, evt);
		break;
	}
	return 0;
}

static int ssi_XFER_ACK(struct ssi32_transfer *trf, enum SSI_EVT evt)
{
	switch (evt) {
	case SSI_EVT_SPI_COMPLETE:
		ssi_slave_deselect(trf);
		trf->err |= check_rx_ack(trf, trf->slave->rx_ack_buf);
		if (chkSRQL(trf->slave)) {
			ssi_start_timer(trf, SSI_TIMER_ESR);
			ssi_newState(trf, SSI_STA_END_ACK);

		} else {
			ssi_stop_timer(trf, SSI_TIMER_INVAL);
			ssi_finishTrf(trf, evt);
		}
		break;

	default:
		ssi_unexpected(trf, evt);
		break;
	}
	return 0;
}


static int ssi_END_ACK(struct ssi32_transfer *trf, enum SSI_EVT evt)
{
	switch (evt) {
	case SSI_EVT_TMO:
		dev_alert(&toDev(trf), "%s: t_ESR expired\n", __func__);
		/* abort transfer if HW XOFF is enabled, else continue */
		if (!ssi_hw_xoff_by_slave(trf, evt))
			break;
		ssi_unexpected(trf, evt);
		/*no break*/
	case SSI_EVT_SRQ_H:
		ssi_stop_timer(trf, SSI_TIMER_INVAL);
		ssi_finishTrf(trf, evt);
		break;

	default:
		ssi_unexpected(trf, evt);
		break;
	}
	return 0;
}

static int ssi_RECOVER(struct ssi32_transfer *trf, enum SSI_EVT evt)
{
	switch (evt) {
	case SSI_EVT_TMO:
		ssi_finishTrf(trf, evt);
		break;
	default:
		ssi_unexpected(trf, evt);
		break;
	}
	return 0;
}


static int (*smdispatch[SSI_NUM_STATES])(struct ssi32_transfer *trf,
				enum SSI_EVT evt) = {
	ssi_IDLE,
	ssi_START_MDATA,
	ssi_dly_XFER_DATA,
	ssi_XFER_DATA,
	ssi_END_DATA,
	ssi_START_ACK,
	ssi_dly_XFER_ACK,
	ssi_XFER_ACK,
	ssi_END_ACK,
	ssi_RECOVER
};



static void lun_add(struct ssi32_slavedev *slave, unsigned char lun)
{
	list_add_tail(&slave->luns[lun].tx_luns, &slave->tx_luns);
}

static void lun_remove(struct ssi32_slavedev *slave, unsigned char lun)
{
	list_del(&slave->luns[lun].tx_luns);
}


static int ssi32_signal_flowctl(struct ssi32_transfer *trf, enum inc_state fc)
{
	struct sk_buff *skb;
	struct inc_hdr hdr;
	hdr.src_node  = 0xff;
	hdr.src_lun   = SSI_LUN_INVAL;
	hdr.dest_node = fc;
	hdr.dest_lun  = trf->tx->lun;
	skb = incnetdev_rx_packet_alloc(toNdev(trf), &hdr, 0);
	if (skb)
		incnetdev_rx_packet_put(toNdev(trf), skb);
	return 0;
}



static int ssi_chk_bf_xoff_pend(struct ssi32_slavedev *s)
{
	return find_first_bit(s->bf_xoff, SSI_LUN_MAX) < SSI_LUN_MAX ? 1 : 0;
}


/*count down xoff timeout
return: 0: no XOFF timer pending >0: no of XOFF timer pending
*/
static int ssi_xoff_chk_tout(struct ssi32_slavedev *slave)
{
	int i = 0;
	int pend = 0;
	while ((i = find_next_bit(slave->bf_xoff, SSI_LUN_MAX, i))
							< SSI_LUN_MAX) {
		if (--slave->luns[i].xoff_dly == 0) {
			clear_bit(i, slave->bf_xoff);

			if (slave->luns[i].pend_cnt > 0)
				lun_add(slave, i);
			ssi32_schedule_preprocessing(slave->master);
		} else {
			pend++;
		}
		i++;
	}
	return pend;
}


static void ssi_xoff_timer_start(struct ssi32_slavedev *slave)
{
	slave->xoff_timer.expires = jiffies + SSI_XOFF_TINTERVAL_JIFFIES;
	add_timer(&slave->xoff_timer);
}

static void ssi_xoff_periodic_timer(unsigned long data)
{
	unsigned long flags;
	struct ssi32_slavedev *slave = (struct ssi32_slavedev *)data;
	lock_sm(slave->master, flags);
	/*rearm if still xoff's pending*/
	if (slave->ready && ssi_xoff_chk_tout(slave))
		ssi_xoff_timer_start(slave);
	unlock_sm(slave->master, flags);
}

/*fixed value for all luns*/
#define ssi_get_xoff_dly(lun)\
		(SSI_XOFF_TINTERVAL_JIFFIES*SSI_XOFF_TOUT_INTERVALS)


/*XOFF was detected/received*/
static void ssi_xoff_entry(struct ssi32_transfer *trf)
{
	/*always give one extra interval*/
	trf->slave->luns[trf->tx->lun].xoff_dly =
				ssi_get_xoff_dly(trf->tx->lun) + 1;

	/*start timer if no timer running*/
	if (!ssi_chk_bf_xoff_pend(trf->slave))
		ssi_xoff_timer_start(trf->slave);

	/*mark lun as XOFF*/
	set_bit(trf->tx->lun, trf->slave->bf_xoff);
	if (trf->slave->luns[trf->tx->lun].pend_cnt > 0)
		lun_remove(trf->slave, trf->tx->lun);

	/*signal to stack*/
	if (!test_and_set_bit(trf->tx->lun, trf->slave->bf_xoff_sig))
		ssi32_signal_flowctl(trf, INC_CHANNEL_XOFF);

	return;
}

static void ssi_xoff_exit(struct ssi32_transfer *trf)
{
	if (test_and_clear_bit(trf->tx->lun, trf->slave->bf_xoff_sig))
		/*signal to stack*/
		ssi32_signal_flowctl(trf, INC_CHANNEL_XON);
}

static void ssi_hw_xoff_timer_start(struct ssi32_slavedev *slave)
{
	unsigned long j = jiffies;

	slave->hw_xoff_timer.expires = j + (HZ/(1000/slave->cfg.tmri));
	if (slave->ready)
		add_timer(&slave->hw_xoff_timer);
}

static void ssi_hw_xoff_timer_stop(struct ssi32_slavedev *slave)
{
	del_timer_sync(&slave->hw_xoff_timer);
}

static int ssi_hw_xoff_entry(struct ssi32_slavedev *slave)
{
	dev_dbg(&slave->master->ndev->ndev->dev, "%s: retry %d\n", __func__,
			slave->hw_xoff_retry);

	if (slave->hw_xoff_retry > slave->cfg.mnr) {
		dev_alert(&slave->master->ndev->ndev->dev, "MNR exceeded\n");
		slave->hw_xoff_retry = 0;
		return 1;
	}

	slave->hw_xoff = 1;
	slave->hw_xoff_retry += 1;
	ssi_hw_xoff_timer_start(slave);

	return 0;
}

static void ssi_hw_xoff_exit(struct ssi32_slavedev *slave)
{
	slave->hw_xoff = 0;
}

static void ssi_hw_xoff_timer_callback(unsigned long data)
{
	struct ssi32_slavedev *slave = (struct ssi32_slavedev *)data;

	dev_dbg(&slave->master->ndev->ndev->dev,
			"%s: t_MRI expired\n", __func__);

	ssi_hw_xoff_exit(slave);
	ssi32_schedule_preprocessing(slave->master);
}


static struct ssi32_lun_msg *get_from_msg_pool(struct ssi32_slavedev *slave)
{
	struct ssi32_lun_msg *msg = list_first_entry(&slave->msg_pool,
						struct ssi32_lun_msg, lhead);
	if ((struct list_head *)msg != &slave->msg_pool) {
		list_del(&msg->lhead);
		return msg;
	}
	pr_err("no msg from pool available\n");

	return NULL;
}

static void put_to_msg_pool(struct ssi32_slavedev *slave,
				struct ssi32_lun_msg *msg, bool delete)
{
	msg->skb = NULL;
	if (delete)
		list_del(&msg->lhead);
	list_add_tail(&msg->lhead, &slave->msg_pool);
}




static void add_pending(struct ssi32_slavedev *slave,
						struct ssi32_lun_msg *entry)
{
	list_add_tail(&entry->lhead, &slave->luns[entry->lun].tx_pending);

	if (!test_bit(entry->lun, slave->bf_xoff)) {
		if (slave->luns[entry->lun].pend_cnt == 0)
			lun_add(slave, entry->lun);
	}
	slave->luns[entry->lun].pend_cnt++;
}

static void remove_pending(struct ssi32_transfer *trf)
{
	trf->slave->luns[trf->tx->lun].pend_cnt--;
	put_to_msg_pool(trf->slave, trf->tx, 1);

	if (trf->slave->luns[trf->tx->lun].pend_cnt == 0) {
		lun_remove(trf->slave, trf->tx->lun);
		ssi_xoff_exit(trf);
	}
}



#define SSI_CTRL_SOM	0x01
#define SSI_CTRL_EOM	0x02
#define SSI_CTRL_RTR	0x04
#define SSI_CTRL_SN	0x08
#define SSI_CTRL_NO_MSG	0x10
#define SSI_CTRL_FIXED1	0x20
#define SSI_CTRL_VER	0x40
#define SSI_CTRL_FIXED2	0x80

#define SSI_CTRL_DEFAULT (SSI_CTRL_VER)


#define SSI_ACK_CHK_ERR	0x01
#define SSI_ACK_SEQ_ERR 0x02
#define SSI_ACK_RESERVED1 0x04
#define SSI_ACK_XOFF 0x08
#define SSI_ACK_POS_ACK 0x10
#define SSI_ACK_PARITY 0x20
#define SSI_ACK_RESERVED2 0x40
#define SSI_ACK_RESERVED3 0x80

#define SSI_ACK_DEFAULT (SSI_ACK_RESERVED2|SSI_ACK_RESERVED3)
#define SSI_ACKBLOCK_CHKMASK (0xffffff00|\
		(SSI_ACK_RESERVED1|SSI_ACK_RESERVED2|SSI_ACK_RESERVED3))

static void ssi_deselect_device(struct ssi32_master *master)
{
	master->transfer.slave = NULL;
}


static void ssi_select_device(struct ssi32_master *master,
					struct ssi32_slavedev *slave)
{
/*on master request: select via node ID*/
/*on slave request: select via SRQ_L*/
	master->transfer.slave = slave;
}


static unsigned char ssi_chksum(unsigned char *buf, unsigned int len)
{
	u32 sum = 0;
	int i;
	for (i = 0; i < len; i++)
		sum += buf[i];
	/* return 2's complement */
	return (u8)((~sum) + 1);
}


/*return 0: ok
	!0 err
*/
static int check_rx_data(struct ssi32_transfer *trf, unsigned char *rx_buf)
{
	/*influences tx_ack*/
	unsigned char ctrl = rx_buf[0];
	unsigned char lun = rx_buf[1];
	unsigned char dlc = rx_buf[2];
	unsigned char chk = rx_buf[3];
	unsigned int mbh = 0; /*opt. header extension for multiblock msg*/
	struct ssi32_lun_msg *entry = NULL;
	struct sk_buff *skb = NULL;

	/*checksum */
	if (ssi_chksum(rx_buf, SSI32_BLOCKLEN(trf)) != 0) {
		dev_dbg(&toDev(trf), "chksum err: lun 0x%x\n", lun);
		/*set all retry bits*/
		memset(trf->slave->bf_rtr, 0xff, sizeof(trf->slave->bf_rtr));
		return SSI32_RX_CHK_ERR|SSI32_RX_ERR;
	}

	/*plausibility check, all-zero header*/
	if (ctrl == 0 && lun == 0 && dlc == 0 && chk == 0) {
		dev_dbg(&toDev(trf), "plausibility error, all-zero header\n");
		/*set all retry bits*/
		memset(trf->slave->bf_rtr, 0xff, sizeof(trf->slave->bf_rtr));
		return SSI32_RX_CHK_ERR|SSI32_RX_ERR;
	}

	/*syntax check*/
	if ((ctrl & SSI_CTRL_DEFAULT) != SSI_CTRL_DEFAULT) {
		dev_dbg(&toDev(trf), "ctrl: syntax error: ctrl= 0x%x\n", ctrl);
		set_bit(lun, trf->slave->bf_rtr);
		return SSI32_RX_SYN_ERR|SSI32_RX_ERR;
	}

	/*return, if no data available*/
	if (ctrl & SSI_CTRL_NO_MSG)
		return SSI32_RX_NO_MSG;

	/*flow control check*/
	if (!test_bit(lun, trf->slave->bf_rx_ready))
		return SSI32_RX_XOFF_ERR|SSI32_RX_ERR;

	/*ignore block on unexpected retry*/
	if (!test_and_clear_bit(lun, trf->slave->bf_rtr)) {
		if (ctrl & SSI_CTRL_RTR)
			return SSI32_RX_NO_MSG;
	}

	if (ctrl & SSI_CTRL_SOM) {
		struct inc_hdr hdr;
		if (ctrl & SSI_CTRL_SN) {
			dev_err(&toDev(trf), "SN on SOM: lun 0x%x\n", lun);
			return SSI32_RX_SEQ_ERR|SSI32_RX_ERR;
		}
		if (!(ctrl & SSI_CTRL_EOM))
			mbh = SSI32_MBHD(trf);

		if (dlc < mbh) {
			dev_err(&toDev(trf), "invalid packet: lun 0x%x\n", lun);
			return SSI32_RX_SEQ_ERR|SSI32_RX_ERR;
		}

		if (trf->slave->luns[lun].rx_msg != NULL) {
			dev_err(&toDev(trf), "reset rx packet: lun 0x%x\n",
					lun);
			incnetdev_rx_packet_reset(toNdev(trf),
					trf->slave->luns[lun].rx_msg->skb);
		} else {
			/* if something goes wrong, return a checksum error
			 * to make the peer re-send the transfer
			 */
			entry = get_from_msg_pool(trf->slave);
			if (!entry)
				return SSI32_RX_CHK_ERR|SSI32_RX_ERR;
			/*we only support destination LUN*/
			hdr.src_node = 0xff;
			hdr.dest_node = 0xff;
			hdr.src_lun = SSI_LUN_INVAL;
			hdr.dest_lun = lun;
			skb = incnetdev_rx_packet_alloc(toNdev(trf),
					&hdr, incnetdev_mtu(toNdev(trf)));
			if (!skb) {
				put_to_msg_pool(trf->slave, entry, 0);
				return SSI32_RX_CHK_ERR|SSI32_RX_ERR;
			}
			trf->slave->luns[lun].rx_msg = entry;
			trf->slave->luns[lun].rx_msg->lun = lun;
			trf->slave->luns[lun].rx_msg->skb = skb;
		}
	} else {
		if (!trf->slave->luns[lun].rx_msg) {
			dev_err(&toDev(trf), "SOM missing: lun 0x%x\n", lun);
			return SSI32_RX_SEQ_ERR|SSI32_RX_ERR;
		}

		/*MBM: check sequence*/
		if ((ctrl & SSI_CTRL_SN) ==
			(trf->slave->luns[lun].rx_msg->ctrl & SSI_CTRL_SN)) {
			dev_err(&toDev(trf), "SEQ ERR: lun 0x%x\n", lun);
			return SSI32_RX_SEQ_ERR|SSI32_RX_ERR;
		}
	}
	trf->slave->luns[lun].rx_msg->ctrl = ctrl; /*save ctrl*/
	trf->rx = trf->slave->luns[lun].rx_msg;

	/*data available:*/
	incnetdev_rx_packet_cpy(toNdev(trf), trf->rx->skb,
					rx_buf+SSI32_HEADLEN+mbh, dlc-mbh);

	if (ctrl & SSI_CTRL_EOM)
		return SSI32_RX_DATA_FINISHED;
	return SSI32_RX_DEFAULT;
}

static int rx_data_is_ok(struct ssi32_transfer *trf)
{
	if (!(trf->err & SSI32_RX_ERR))
		return 1;

	if (trf->err & SSI32_RX_CHK_ERR)
		dev_dbg(&toDev(trf), "%s: SSI32_RX_CHK_ERR\n", __func__);
	else if (trf->err & SSI32_RX_SEQ_ERR)
		dev_dbg(&toDev(trf), "%s: SSI32_RX_SEQ_ERR\n", __func__);
	else if (trf->err & SSI32_RX_SYN_ERR)
		dev_dbg(&toDev(trf), "%s: SSI32_RX_SYN_ERR\n", __func__);
	else if (trf->err & SSI32_RX_XOFF_ERR)
		dev_dbg(&toDev(trf), "%s: SSI32_RX_XOFF_ERR\n", __func__);
	else
		dev_dbg(&toDev(trf), "%s: error %d\n", __func__, trf->err);

	return 0;
}

/*return: SSI32_TRFTX_OK: OK
	SSI32_TRFTX_XXX: else
*/
static int check_rx_ack(struct ssi32_transfer *trf, unsigned char *rx_buf)
{
	int err = SSI32_TX_DEFAULT;
	/*influences tx_data (ctrl)*/
	unsigned int ack = *((unsigned int *)rx_buf);

	/*check ack block*/
	if ((ack & SSI_ACKBLOCK_CHKMASK) != SSI_ACK_DEFAULT) {
		err = SSI32_TX_ERR|SSI32_TX_SYN_ERR;
		dev_dbg(&toDev(trf), "RX syntax ERROR: ack=0x%x\n", ack);
		return err;
	}

	if (!(ack & SSI_ACK_POS_ACK)) {
		if (ack & SSI_ACK_CHK_ERR) {
			err = SSI32_TX_ERR|SSI32_TX_CHK_ERR;
			dev_dbg(&toDev(trf), "RX chksum ERROR: ack= 0x%x\n",
					ack);
		}
		if (ack & SSI_ACK_SEQ_ERR) {
			err = SSI32_TX_ERR|SSI32_TX_SEQ_ERR;
			dev_dbg(&toDev(trf), "RX seq ERROR: ack= 0x%x\n", ack);
		}
		if (ack & SSI_ACK_XOFF) {
			err = SSI32_TX_ERR|SSI32_TX_XOFF;
			dev_dbg(&toDev(trf), "RX XOFF: ack= 0x%x\n", ack);
		}

		if (err == 0) {
			err = SSI32_TX_ERR|SSI32_TX_SYN_ERR;
			dev_dbg(&toDev(trf), "unknown ack = 0x%x\n", ack);
		}
	}

	return err;
}

static int setup_tx_ack(struct ssi32_transfer *trf, unsigned char *tx_buf)
{
	/*influenced by rx_data*/
	unsigned char tx_ack;

	if (trf->err & SSI32_RX_CHK_ERR)
		tx_ack = SSI_ACK_DEFAULT | SSI_ACK_CHK_ERR;
	else if (trf->err & SSI32_RX_SEQ_ERR)
		tx_ack = SSI_ACK_DEFAULT | SSI_ACK_SEQ_ERR;
	else if (trf->err & SSI32_RX_SYN_ERR)
		tx_ack = SSI_ACK_DEFAULT | SSI_ACK_CHK_ERR;
	else if (trf->err & SSI32_RX_XOFF_ERR)
		tx_ack = SSI_ACK_DEFAULT | SSI_ACK_XOFF;
	else
		tx_ack = SSI_ACK_DEFAULT | SSI_ACK_POS_ACK;

	tx_buf[0] = tx_ack;
	tx_buf[1] = 0;
	tx_buf[2] = 0;
	tx_buf[3] = 0;
	return 0;
}


static int setup_tx_data(struct ssi32_transfer *trf, unsigned char *tx_buf)
{
	unsigned int dlc;
	unsigned int mbh = 0; /*opt. header extension for multiblock msg*/
	trf->tx->pend_len = 0;
	/*influenced by rx_ack*/
	if (!trf->tx->remain) {
		trf->tx->ctrl = SSI_CTRL_DEFAULT|SSI_CTRL_NO_MSG|
					SSI_CTRL_EOM|SSI_CTRL_SOM;
		dlc = 0;
	} else {
		if (trf->tx->remain == trf->tx->skb->len) {
			trf->tx->ctrl = SSI_CTRL_DEFAULT|SSI_CTRL_SOM;
			mbh = SSI32_MBHD(trf);
			if ((trf->tx->remain <= SSI32_PAYLLEN(trf)))
				mbh = 0;
		} else
			trf->tx->ctrl &= ~SSI_CTRL_SOM;

		if (trf->tx->remain <= SSI32_PAYLLEN(trf)) {
			trf->tx->ctrl |= SSI_CTRL_EOM;
			dlc = trf->tx->remain;
		} else {
			dlc = SSI32_PAYLLEN(trf);
		}

		if (trf->tx->retry_blk)
			trf->tx->ctrl |= SSI_CTRL_RTR;
		else
			trf->tx->ctrl &= ~SSI_CTRL_RTR;
	}


	tx_buf[0] = trf->tx->ctrl;
	tx_buf[1] = trf->tx->lun;
	tx_buf[2] = dlc;
	tx_buf[3] = 0; /*clear checksum!!*/

	if (mbh) {
		*((unsigned short *)(&tx_buf[4])) = htons(trf->tx->remain);
		tx_buf[6] = 0x00;
		tx_buf[7] = 0x00;
	}

	if (dlc) {
		trf->tx->pend_len = dlc-mbh;
		memcpy(&tx_buf[4+mbh], trf->tx->pos, dlc-mbh);
	}
	/*according to spec nonrelevant data has to be zero*/
	if (dlc < SSI32_PAYLLEN(trf))
		memset(&tx_buf[4+dlc], 0, SSI32_PAYLLEN(trf)-dlc);

	/*checksum: as nonrelevant bytes are zero,
			csum needed only over len = dlc*/
	tx_buf[3] = ssi_chksum(tx_buf, dlc + SSI32_HEADLEN);

	return trf->tx->remain ? SSI32_TX_DEFAULT : SSI32_TX_NO_MSG;
}



static int ssi_initXfer(struct ssi32_transfer *trf)
{
	if (ssi32_uses_hw_ack(trf->slave->ssidev))
		ssi32_hw_ack(trf->slave->ssidev, 0);

	ssi_slave_select(trf);

	trf->err |= setup_tx_data(trf, trf->slave->tx_data_buf);
	if (chkSRQL(trf->slave)) {
		if (ssi_start_timer(trf, SSI_TIMER_CD2))
			on_tout_cd1_cd2(trf, SSI_STA_XFER_DATA);
		else
			ssi_newState(trf, SSI_STA_DLY_XFER_DATA);
	} else {
		ssi_start_timer(trf, SSI_TIMER_SSR);
		ssi_newState(trf, SSI_STA_START_MDATA);
	}
	return 0;
}

/*resolve node to slave -->not yet defined, always use first slave*/
#define NODE2SLAVE(m, n) (m->ssi32_slaves[0])


static int ssi32_master_flowcontrol(struct ssi32_master *m,
		struct ssi32_slavedev *slave, struct inc_hdr *hdr)
{
	enum inc_state fc = hdr->src_node;
	uint8_t lun = hdr->src_lun;
	int is_fc = 0;

	switch (fc) {
	case INC_CHANNEL_XOFF:
		if (slave->cfg.flowcontrol) {
			dev_dbg(&m->ndev->ndev->dev,
					"CHANNEL XOFF: on lun 0X%02x\n", lun);
			clear_bit(lun, slave->bf_rx_ready);
		}
		is_fc = 1;
		break;
	case INC_CHANNEL_XON:
		if (slave->cfg.flowcontrol) {
			dev_dbg(&m->ndev->ndev->dev,
					"CHANNEL XON: on lun 0X%02x\n", lun);
			set_bit(lun, slave->bf_rx_ready);
		}
		is_fc = 1;
		break;
	case INC_CHANNEL_BROKEN:
		if (slave->cfg.flowcontrol) {
			dev_dbg(&m->ndev->ndev->dev,
					"CHANNEL BROKEN: on lun 0X%02x\n", lun);
		}
		is_fc = 1;
		break;
	default:
		break;
	}

	return is_fc;
}


/*select tx lun*/
static struct ssi32_slavedev *select_next_transfer(struct ssi32_master *m)
{
	struct ssi32_slavedev *slave, *other_slave;
	struct ssi32_lun_msg *entry = NULL;
	bool xoff;
	int pend;
	int i;
	unsigned long flags;

	/*first try to deliver/process new skb*/
	if (m->skb) {
		slave = NODE2SLAVE(m, m->hdr->dest_node);
		if (ssi32_master_flowcontrol(m, slave, m->hdr)) {
			for (i = 1; i < SSI_MAX_DEVPERPORT; i++) {
				/* apply flow control to all other
				 * valid slaves as well */
				other_slave = m->ssi32_slaves[i];
				if (other_slave && other_slave->ready)
					ssi32_master_flowcontrol(m, other_slave,
							m->hdr);
			}
			dev_kfree_skb_any(m->skb);
			lock_sm(m, flags);
			m->skb = NULL;
			m->hdr = NULL;
			incnetdev_next_tx_packets(m->ndev);
			unlock_sm(m, flags);
		} else {
			xoff = test_bit(m->hdr->dest_lun, slave->bf_xoff);
			pend = slave->luns[m->hdr->dest_lun].pend_cnt;

			/*in XOFF state we have to accept this skb
			to allow transfer on other LUN's*/
			if ((pend < SSI_PEND_LIMIT) || xoff)
				entry = get_from_msg_pool(slave);

			if (entry) {
				entry->retry_blk = 0;
				entry->retry_msg = 0;
				entry->lun = m->hdr->dest_lun;
				entry->pos = m->skb->data;
				entry->remain = m->skb->len;
				entry->skb = m->skb;
				add_pending(slave, entry);
				lock_sm(m, flags);
				m->skb = NULL;
				m->hdr = NULL;
				incnetdev_next_tx_packets(m->ndev);
				unlock_sm(m, flags);
			}
		}
	}

/*TODO: algorithm to select device.. priorities/FIFO/RR
-round robin: send/rxSlaveA/rxSlaveB/rxSlaveC/...
*/
	entry = NULL;
	slave = NULL;

	for (i = 0; (i < SSI_MAX_DEVPERPORT) && (!entry); i++) {
		struct ssi32_lun *lun;
		slave = m->ssi32_slaves[i];
		if (slave == NULL || !slave->ready || slave->hw_xoff)
			break;

		/*check tx*/
		lun = list_first_entry(&slave->tx_luns,
						struct ssi32_lun, tx_luns);
		if ((struct list_head *)lun != &slave->tx_luns) {
			entry = list_first_entry(&lun->tx_pending,
					struct ssi32_lun_msg, lhead);
			if ((struct list_head *)entry == &lun->tx_pending)
				entry = NULL;
		}
		/*check rx*/
		if (!entry) {
			if (chkSRQL(slave)) {
				entry = get_from_msg_pool(slave);
				if (entry) {
					entry->lun = SSI_LUN_INVAL;
					entry->remain = 0;
				}
			}
		}
	}

	if (!entry)
		return NULL;
	m->transfer.tx = entry;
	return slave;
}


static int ssi32_schedule_preprocessing(struct ssi32_master *m)
{
	set_bit(SSI_DO_WORK, &m->flags);
	wake_up(&m->wait);
	return 0;
}

static int ssi32_pre_processing(struct ssi32_master *m)
{
	struct ssi32_slavedev *slave = NULL;
	unsigned long flags;


#ifdef SSI32_HOLD_ON_ERROR
	if (m->stopped)
		return 0;
#endif
	if (test_and_set_bit(SSI_BUSY, &m->flags))
		return 0;

	if (test_bit(SSI_ABORT, &m->flags)) {
		/*clean up pending skb*/
		if (m->skb) {
			dev_kfree_skb_any(m->skb);
			m->skb = NULL;
			m->hdr = NULL;
		}
	} else {
		slave = select_next_transfer(m);
	}

	if (slave) {
		if (m->bus_locked == NULL)
			ssi32_bus_lock(slave->ssidev);
		m->bus_locked = slave; /*update to actual slave*/
		lock_sm(m, flags);
		ssi_select_device(m, slave);
		ssi_initXfer(&m->transfer);
		unlock_sm(m, flags);
	} else {
		if (m->bus_locked != NULL) {
			ssi32_bus_unlock(m->bus_locked->ssidev);
			m->bus_locked = NULL;
		}
		clear_bit(SSI_BUSY, &m->flags);
		if (test_bit(SSI_ABORT, &m->flags)) {
			/*finish abort*/
			clear_bit(SSI_ABORT, &m->flags);
			wake_up_interruptible(&m->wait);
		}
	}
	return 0;
}



static int ssi32_post_processing_rx(struct ssi32_transfer *trf)
{
	if (trf->err & SSI32_RX_NO_MSG)
		return 0;

	if (trf->err & SSI32_RX_XOFF_ERR)
		return 0;

	if (!(trf->err & SSI32_RX_ERR)) {
		if (trf->err & SSI32_RX_DATA_FINISHED) {
			incnetdev_rx_packet_put(toNdev(trf), trf->rx->skb);
			put_to_msg_pool(trf->slave, trf->rx, 0);
			trf->slave->luns[trf->rx->lun].rx_msg = NULL;
		}
		return 0;
	}

	if (trf->err & SSI32_RX_HANDSHAKE_ERR)
		incnetdev_carrier_err_ind(toNdev(trf));
	else
		incnetdev_rx_err_ind(toNdev(trf));
	return 0;
}


static int ssi32_post_processing_tx(struct ssi32_transfer *trf)
{
	struct ssi32_lun_msg *tx = trf->tx;

	if (trf->err & SSI32_TX_NO_MSG) {
		/*dummy transfer*/
		put_to_msg_pool(trf->slave, tx, 0);
		return 0;
	}

	if (!(trf->err & SSI32_TX_ERR)) {
		if (trf->tx->remain <= trf->tx->pend_len) {
			incnetdev_sent_tx_packet(toNdev(trf), tx->skb, 1);
			remove_pending(trf);
		} else {
			tx->retry_blk = 0;
			tx->retry_msg = 0;
			tx->remain -= trf->tx->pend_len;
			tx->pos += trf->tx->pend_len;
			/*block finished ->toggle sn*/
			tx->ctrl ^= SSI_CTRL_SN;
		}
		trf->slave->hw_xoff_retry = 0;
		return 0;
	}


	if (trf->err & SSI32_TX_MASK_RETRY_BLK) {
		if (trf->err & SSI32_TX_HANDSHAKE_ERR)
			incnetdev_carrier_err_ind(toNdev(trf));
		else
			incnetdev_tx_err_ind(toNdev(trf));

		/*retransmit blk*/
		tx->retry_blk++;
		if (tx->retry_blk > SSI_RETRY_BLK_MAX) {
#ifdef SSI_RETRY_MSG_ON_BLK_FAIL
			tx->retry_blk = 0;
			/*retransmit msg*/
			tx->remain = tx->skb->len;/*this will setup ctrl*/
			tx->pos = tx->skb->data;

			tx->retry_msg++;
			if (tx->retry_msg > SSI_RETRY_MSG_MAX) {
				incnetdev_sent_tx_packet(toNdev(trf),
								tx->skb, 0);
				ssi32_signal_flowctl(trf, INC_CHANNEL_BROKEN);
				remove_pending(trf);
			}
#else
			incnetdev_sent_tx_packet(toNdev(trf), tx->skb, 0);
			ssi32_signal_flowctl(trf, INC_CHANNEL_BROKEN);
			remove_pending(trf);
#endif

		}
		return 0;
	}

	if (trf->err & SSI32_TX_MASK_RETRY_MSG) {
		if (trf->err & SSI32_TX_HANDSHAKE_ERR)
			incnetdev_carrier_err_ind(toNdev(trf));
		else
			incnetdev_tx_err_ind(toNdev(trf));
		/*retransmit msg*/
		tx->remain = tx->skb->len;/*this will setup ctrl*/
		tx->retry_blk = 0;
		tx->pos = tx->skb->data;

		tx->retry_msg++;
		if (tx->retry_msg > SSI_RETRY_MSG_MAX) {
			incnetdev_sent_tx_packet(toNdev(trf), tx->skb, 0);
			ssi32_signal_flowctl(trf, INC_CHANNEL_BROKEN);
			remove_pending(trf);
		}
		return 0;
	}

	if (trf->err & SSI32_TX_XOFF) {
		ssi_xoff_entry(trf);
		return 0;
	}

	if (trf->err & SSI32_TX_HW_XOFF) {
		if (ssi_hw_xoff_entry(trf->slave)) {
			incnetdev_sent_tx_packet(toNdev(trf), tx->skb, 0);
			ssi32_signal_flowctl(trf, INC_CHANNEL_BROKEN);
			remove_pending(trf);
		}
		return 0;
	}

	/* -should never come here - drop packet*/
	dev_alert(&toDev(trf), "SSI32: unhandled errcode 0x%x\n", trf->err);
	incnetdev_sent_tx_packet(toNdev(trf), tx->skb, 0);
	ssi32_signal_flowctl(trf, INC_CHANNEL_BROKEN);
	remove_pending(trf);
	return 0;
}

/*finished BLOCK*/
static int ssi_finishTrf(struct ssi32_transfer *trf, enum SSI_EVT evt)
{
	struct ssi32_master *m = trf->slave->master;

	/*rx path*/
	ssi32_post_processing_rx(trf);
	trf->rx = NULL;

	/*tx path*/
	ssi32_post_processing_tx(trf);
	trf->tx = NULL;

#ifdef SSI32_HOLD_ON_ERROR
	if (trf->err != SSI32_TRFERR_OK)
		trf->stopped = 1;
#endif
	/*end communication with this device*/
	ssi_deselect_device(m);
	ssi_newState(trf, SSI_STA_IDLE);
	ssi_history(evt, trf->state, (trf->err & SSI32_TX_ERR) ? 1 : 2, trf);
	trf->err = SSI32_TXRX_DEFAULT;

	clear_bit(SSI_BUSY, &m->flags);

	/*try initiate new transfer*/
	ssi32_schedule_preprocessing(m);
	return 0;
}


static void ssi32_set_sched(struct ssi32_master *master)
{
	struct sched_param param = {.sched_priority = (MAX_RT_PRIO-2)};
	if (sched_setscheduler(current, SCHED_FIFO, &param))
		pr_err("SSI32: failed to setup scheduler\n");
}

/*threadfn*/
static int ssi32_do_work(void *data)
{
	struct ssi32_master *master = (struct ssi32_master *)data;

	ssi32_set_sched(master);
	while (!kthread_should_stop()) {
		wait_event_interruptible(master->wait,
				test_bit(SSI_DO_WORK, &master->flags) |
				kthread_should_stop());
		clear_bit(SSI_DO_WORK, &master->flags);

		if (kthread_should_stop())
			break;

		ssi32_pre_processing(master);
	}
	return 0;
}


static int ssi32_data_transfer(struct ssi32_transfer *trf)
{
	struct ssi32_device *ssidev = trf->slave->ssidev;
	return ssi32_transfer(ssidev, trf->slave->tx_data_buf,
				trf->slave->rx_data_buf, SSI32_BLOCKLEN(trf));
}

static int ssi32_ack_transfer(struct ssi32_transfer *trf)
{
	struct ssi32_device *ssidev = trf->slave->ssidev;
	setup_tx_ack(trf, trf->slave->tx_ack_buf);
	return ssi32_transfer(ssidev, trf->slave->tx_ack_buf,
				trf->slave->rx_ack_buf, SSI32_ACKLEN);
}


static void dispatch_locked(struct ssi32_transfer *trf, enum SSI_EVT evt)
{
	smdispatch[trf->state](trf, evt);
	ssi_history(evt, trf->state, 0, trf);
}


/*-----------------------------------trigger------------------------------*/

static enum hrtimer_restart ssi_hrtimer_callback(struct hrtimer *timer)
{
	struct ssi32_master *m;
	unsigned long flags;

	m = container_of(timer, struct ssi32_master, timer);
	lock_sm(m, flags);
	/*ignore timeout, if timer was already stopped
	during this callback invocation*/
	if (m->transfer.running != SSI_TIMER_INVAL) {
		m->transfer.running = SSI_TIMER_INVAL;
		dispatch_locked(&m->transfer, SSI_EVT_TMO);
	}
	unlock_sm(m, flags);

	return HRTIMER_NORESTART;
}


/*called by netdev*/
int ssi32m_tx(void *port_handle, struct sk_buff *skb, struct inc_hdr *hdr)
{
	struct ssi32_master *master = port_handle;
	unsigned long flags;
	lock_sm(master, flags);

	if (master->skb) {
		dev_err(&master->ndev->ndev->dev,
				"SSI32: TX busy but queue running\n");
		incnetdev_stop_tx_packets(master->ndev);
		unlock_sm(master, flags);
		return 1;
	}
	master->hdr = hdr;
	/* add memory barrier to make sure hdr is written before skb
	 * to avoid race condition with select_next_transfer() */
	wmb();
	master->skb = skb;
	ssi32_schedule_preprocessing(master);

	/*stop queue if skb was not yet processed*/
	if (master->skb)
		incnetdev_stop_tx_packets(master->ndev);

	unlock_sm(master, flags);
	return 0;
}


void ssi32m_spi_complete(struct ssi32_slavedev *slave)
{
	unsigned long flags;
	lock_sm(slave->master, flags);
	dispatch_locked(&slave->master->transfer, SSI_EVT_SPI_COMPLETE);
	unlock_sm(slave->master, flags);
	return;
}


static void process_srq(struct ssi32_slavedev *slave, int high)
{
	struct ssi32_master *master = slave->master;
	struct ssi32_transfer *trf = &master->transfer;

	if (high)
		master->srq_pend &= ~(1<<(slave->cfg.node));
	else
		master->srq_pend |= 1<<(slave->cfg.node);

	/*decide active or new slave*/
	if (trf->slave == slave) {
		dispatch_locked(trf, high ? SSI_EVT_SRQ_H : SSI_EVT_SRQ_L);
	/*other device than active: try to start new xfer*/
	} else {
		if (!high)
			ssi32_schedule_preprocessing(master);
	}
}


/*
high: the actual read SRQ value 0 or !0
*/
void ssi32m_slave_srq(struct ssi32_slavedev *slave, int high)
{
	int spurious;
	unsigned long flags;
	struct ssi32_master *m = slave->master;

	lock_sm(m, flags);
	/*check integrity of SRQ state*/
	spurious = (bool)high !=
		(bool)(m->srq_pend & (1<<(slave->cfg.node)));

	/*if SRQ already changed until this handler is invoked, mostly no
	further interrupt is generated
	-> thus we have to insert the 'missed' SRQ level.
	Only sometimes the second IRQ is issued -> this IRQ will be ignored
	*/
	if (spurious)
		if (slave->spurious) {
			slave->spurious = 0;
			dev_dbg(&m->ndev->ndev->dev, "ignore 2nd spurious\n");
			unlock_sm(m, flags);
			return;
		} else {
			slave->spurious = 1;
		}
	else
		slave->spurious = 0;

	if (spurious) /*insert missed event*/
		process_srq(slave, !high);
	process_srq(slave, high);

	unlock_sm(m, flags);
}

/*-----------------------------INIT/EXIT-----------------------------------*/



static int ssi32_transfer_exit(struct ssi32_transfer *transfer,
					struct ssi32_master *master)
{
	kthread_stop(master->task);
	ssi_stop_timer(transfer, SSI_TIMER_INVAL);
	return 0;
}


static int ssi32_transfer_init(struct ssi32_transfer *transfer,
					struct ssi32_master *master)
{
	init_waitqueue_head(&master->wait);
	hrtimer_init(&master->timer, CLOCK_MONOTONIC, HRTIMER_MODE_REL);
	master->timer.function = ssi_hrtimer_callback;
	master->flags = 0;
	transfer->state = SSI_STA_IDLE;
	transfer->running = SSI_TIMER_INVAL;
	spin_lock_init(&master->smlock);

	master->task = kthread_run(ssi32_do_work, (void *)master, "SSI32");
	if (IS_ERR(master->task)) {
		pr_err("SSI32: kthread_run() failed at transfer init\n");
		return PTR_ERR(master->task);
	}

	return 0;
}


static int ssi32_master_destroy(struct ssi32_master *master)
{
	inc_netdev_destroy(master->ndev);
	ssi32_transfer_exit(&master->transfer, master);
	pr_debug("SSI32: master terminated\n");
	kfree(master);
	return 0;
}


static struct ssi32_master *ssi32_master_create(struct ssi32_dcfg *cfg)
{
	struct ssi32_master *master = kmalloc(sizeof(struct ssi32_master),
						GFP_KERNEL);
	memset(master, 0 , sizeof(struct ssi32_master));

	ssi32_transfer_init(&master->transfer, master);

	master->ndev = inc_netdev_create(master, cfg->mac, SSI32_MTU,
			cfg->ifname);
	pr_debug("SSI32: master created\n");

	return master;
}



static int ssi32_register(struct ssi32_master *master,
		struct ssi32_slavedev *slave)
{
	int node;
	/*search free entry*/
	for (node = 0; node < SSI_MAX_DEVPERPORT; node++) {
		if (master->ssi32_slaves[node] == NULL) {
			/* assure limitations:
			 *   1st slave may not have "rx-only" property
			 *   2nd..Nth slave must have "rx-only" property
			 *   2nd..Nth slave may not have "inc-node" property
			 */
			if (node == 0 && slave->cfg.rxonly) {
				pr_err("SSI32: slave 0 may not be rx-only\n");
				return -EINVAL;
			}
			if (node > 0 && !slave->cfg.rxonly) {
				pr_err("SSI32: slave %d must be rx-only\n",
						node);
				return -EINVAL;
			}
			if (node > 0 && slave->cfg.addr) {
				pr_err("SSI32: slave %d may not be inc-node\n",
						node);
				return -EINVAL;
			}
			master->ssi32_slaves[node] = slave;
			slave->master = master;
			return 0;
		}
	}
	return -ENODEV;
}

static int ssi32_slave_init(struct ssi32_slavedev *slave)
{
	int ret, i;

	slave->rx_data_buf = kmalloc(slave->cfg.blocksize, GFP_KERNEL);
	slave->tx_data_buf = kmalloc(slave->cfg.blocksize, GFP_KERNEL);
	if ((!slave->tx_data_buf) || (!slave->rx_data_buf)) {
		pr_err("SSI32: couldn't allocate slave rx/tx buffer\n");
		ret = -ENOMEM;
		goto free_slave;
	}

	slave->msg_pool_mem = kmalloc(SSI_MSG_POOL_SIZE*
				sizeof(struct ssi32_lun_msg), GFP_KERNEL);
	if (!slave->msg_pool_mem) {
		pr_err("SSI32: couldn't allocate transfer pool\n");
		ret = -ENOMEM;
		goto free_slave;
	}

	INIT_LIST_HEAD(&slave->msg_pool);
	INIT_LIST_HEAD(&slave->tx_luns);

	for (i = 0; i < SSI_LUN_MAX ; i++) {
		INIT_LIST_HEAD(&slave->luns[i].tx_pending);
		INIT_LIST_HEAD(&slave->luns[i].tx_luns);
		slave->luns[i].lun = i;
	}
	/*all transfers to free list - this initializes the entries*/
	for (i = 0; i < SSI_MSG_POOL_SIZE; i++)
		list_add_tail(&slave->msg_pool_mem[i].lhead, &slave->msg_pool);

	init_timer(&slave->xoff_timer);
	slave->xoff_timer.data = (unsigned long)slave;
	slave->xoff_timer.function = ssi_xoff_periodic_timer;

	init_timer(&slave->hw_xoff_timer);
	slave->hw_xoff_timer.data = (unsigned long)slave;
	slave->hw_xoff_timer.function = ssi_hw_xoff_timer_callback;

	return 0;

free_slave:
	kfree(slave->msg_pool_mem);
	kfree(slave->rx_data_buf);
	kfree(slave->tx_data_buf);
	return ret;
}

static void ssi32_slave_exit(struct ssi32_slavedev *slave)
{
	struct ssi32_dcfg dcfg;
	struct ssi32_device *ssidev;
	struct ssi32_master *master;
	struct ssi32_lun *lun;
	struct ssi32_lun_msg *msg;
	unsigned long bf_rx_ready[SSI_UNSIGNED_BF_SIZE];
	int i;

	/* clean-up pending messages and luns */
	for (i = 0; i < SSI_LUN_MAX; i++) {
		lun = &slave->luns[i];
		/* rx message */
		if (lun->rx_msg) {
			if (lun->rx_msg->skb) {
				dev_info(&slave->master->ndev->ndev->dev,
					"%s: rx packet dropped: LUN 0x%02X\n",
					__func__, lun->rx_msg->lun);
				dev_kfree_skb_any(lun->rx_msg->skb);
			} else {
				dev_err(&slave->master->ndev->ndev->dev,
					"%s: no rx skb: LUN 0x%02X\n",
					__func__, lun->rx_msg->lun);
			}
		}
		/* tx messages */
		list_for_each_entry(msg, &lun->tx_pending, lhead) {
			if (msg->skb) {
				dev_info(&slave->master->ndev->ndev->dev,
					"%s: tx packet dropped: LUN 0x%02X\n",
					__func__, msg->lun);
				dev_kfree_skb_any(msg->skb);
			} else {
				dev_err(&slave->master->ndev->ndev->dev,
					"%s: no tx skb: LUN 0x%02X\n",
					__func__, msg->lun);
			}
		}
		/* lun */
		memset(lun, 0, sizeof(*lun));
	}

	/* stop timers */
	del_timer_sync(&slave->xoff_timer);

	/* free mem */
	kfree(slave->msg_pool_mem);
	kfree(slave->rx_data_buf);
	kfree(slave->tx_data_buf);

	/* reset variables to zero except master, ssidev, rx_ready and dcfg */
	master = slave->master;
	ssidev = slave->ssidev;
	memcpy(bf_rx_ready, slave->bf_rx_ready, SSI_UNSIGNED_BF_SIZE);
	memcpy(&dcfg, &slave->cfg, sizeof(dcfg));

	memset(slave, 0, sizeof(*slave));

	slave->master = master;
	slave->ssidev = ssidev;
	memcpy(slave->bf_rx_ready, bf_rx_ready, SSI_UNSIGNED_BF_SIZE);
	memcpy(&slave->cfg, &dcfg, sizeof(dcfg));
}

int ssi32m_unregister_slave_device(struct ssi32_slavedev *slave)
{
	struct ssi32_master *master;
	struct ssi32_master *tmp_master;
	int port;
	int node;
	int refcnt = 0;

	mutex_lock(&ssi32_master_list_lock);
	/*get master*/
	master = NULL;
	for (port = 0; port < SSI_MAX_PORTS && (!master); port++) {
		refcnt = 0;
		tmp_master = ssi32_ports[port];
		if (tmp_master) {
			for (node = 0; node < SSI_MAX_DEVPERPORT; node++) {
				if (tmp_master->ssi32_slaves[node])
					refcnt++;
				if (tmp_master->ssi32_slaves[node] == slave) {
					master = tmp_master;
					tmp_master->ssi32_slaves[node] = NULL;
					refcnt--;
					/*break;  no break to update refcnt*/
				}
			}
		}
	}

	if (!master) {
		pr_err("SSI32: unregister slave failed\n");
		mutex_unlock(&ssi32_master_list_lock);
		return -ENODEV;
	}

	/*clean up slave*/
	kfree(slave);

	/*unregister master, if unused*/
	if (refcnt == 0)
		ssi32_master_destroy(master);
	mutex_unlock(&ssi32_master_list_lock);
	return 0;
}

int ssi32m_register_slave_device(struct ssi32_device *ssidev,
		struct ssi32_dcfg *dcfg,
		struct ssi32_slavedev **slaveout)
{
	struct ssi32_master *master;
	struct ssi32_slavedev *slave;
	int ret, i;
	int refcnt = 0;

	slave = kmalloc(sizeof(*slave), GFP_KERNEL);
	if (!slave) {
		pr_err("SSI32: couldn't allocate slave\n");
		ret = -ENOMEM;
		goto out;
	}
	memset(slave, 0, sizeof(*slave));

	slave->ssidev = ssidev;
	memcpy(&slave->cfg, dcfg, sizeof(*dcfg));

	for (i = 0; i < SSI_UNSIGNED_BF_SIZE ; i++)
		slave->bf_rx_ready[i] = slave->cfg.flowcontrol ? 0 : ULONG_MAX;

	if (slave->cfg.tssr == 0)
		slave->cfg.tssr = ssi_timtout_ns[SSI_TIMER_SSR];
	if (slave->cfg.tcd1 == 0)
		slave->cfg.tcd1 = ssi_timtout_ns[SSI_TIMER_CD1];
	if (slave->cfg.tcd2 == 0)
		slave->cfg.tcd2 = ssi_timtout_ns[SSI_TIMER_CD2];
	if (slave->cfg.tesr == 0)
		slave->cfg.tesr = ssi_timtout_ns[SSI_TIMER_ESR];

	mutex_lock(&ssi32_master_list_lock);

	/*search matching port or create new one*/
	master = NULL;
	for (i = 0; i < SSI_MAX_PORTS && !master; i++) {
		if (ssi32_ports[i] && ssi32_ports[i]->bus == dcfg->bus) {
			master = ssi32_ports[i];
			refcnt++;
			break;
		}
	}
	if (!master) {
		/*new entry*/
		for (i = 0; i < SSI_MAX_PORTS; i++) {
			if (!ssi32_ports[i]) {
				master = ssi32_master_create(&slave->cfg);
				if (master) {
					ssi32_ports[i] = master;
					master->bus = dcfg->bus;
					break;
				}
			}
		}
	}

	if (!master) {
		pr_err("SSI32: couldn't get master\n");
		ret = -ENOMEM;
		goto put_slave;
	}

	ret = ssi32_register(master, slave);
	if (ret) {
		pr_err("SSI32: couldn't register slave\n");
		ssi32_ports[i] = NULL;
		/*destroy master if just created*/
		goto free_master;
	}
	mutex_unlock(&ssi32_master_list_lock);
	*slaveout = slave;
	return 0;

free_master:
	if (!refcnt)
		ssi32_master_destroy(master);
put_slave:
	mutex_unlock(&ssi32_master_list_lock);
	kfree(slave);
out:
	return ret;
}

int ssi32_slave_start(struct ssi32_slavedev *slave)
{
	int ret = 0;

	if (slave->ready)
		return 0;

	ret = ssi32_slave_init(slave);
	if (ret)
		goto error;

	slave->ready = 1;

	ret = ssi32_dev_start(slave->ssidev);
	if (ret)
		goto error;

error:
	return ret;
}

void ssi32_slave_stop(struct ssi32_slavedev *slave)
{
	if (!slave->ready)
		return;

	ssi32_master_stop(slave->master);
}

void ssi32_master_start(void *port_handle)
{
	struct ssi32_master *master = port_handle;
	struct ssi32_slavedev *slave;
	int node;

	for (node = 0; node < SSI_MAX_DEVPERPORT; node++) {
		slave = master->ssi32_slaves[node];
		if (slave != NULL)
			ssi32_slave_start(slave);
	}
}

static void ssi32_master_abort(struct ssi32_master *master)
{
	int ret;

	if (test_and_set_bit(SSI_ABORT, &master->flags)) {
		dev_err(&master->ndev->ndev->dev,
				"abort of transfer already in progress\n");
		return;
	}

	ssi32_schedule_preprocessing(master);

	ret = wait_event_interruptible_timeout(master->wait,
			!test_bit(SSI_ABORT, &master->flags), SSI_ABORT_TOUT);
	if (ret <= 0)
		dev_err(&master->ndev->ndev->dev,
				"wait for transfer abort failed: %d\n", ret);

	if (master->bus_locked != NULL)
		dev_alert(&master->ndev->ndev->dev,
				"unexpected bus_locked: %s\n",
				master->bus_locked->cfg.ifname);
}

void ssi32_master_stop(void *port_handle)
{
	struct ssi32_master *master = port_handle;
	struct ssi32_slavedev *slave;
	int node;
	unsigned long flags;

	/* stop all slaves */
	for (node = 0; node < SSI_MAX_DEVPERPORT; node++) {
		slave = master->ssi32_slaves[node];
		if (slave != NULL && slave->ready) {
			ssi32_dev_stop(slave->ssidev);
			/* now the SRQ IRQ is disabled and all executing
			 * interrupts have completed -> clear pending low SRQ
			 */
			if (chkSRQL(slave)) {
				pr_debug("SSI32: dev down clear pending SRQ\n");
				ssi32m_slave_srq(slave, 1);
			}
			lock_sm(master, flags);
			slave->ready = 0;
			ssi_hw_xoff_timer_stop(slave);
			unlock_sm(master, flags);
		}
	}

	ssi32_master_abort(master);

	/* de-initialize all slaves */
	for (node = 0; node < SSI_MAX_DEVPERPORT; node++) {
		slave = master->ssi32_slaves[node];
		if (slave != NULL)
			ssi32_slave_exit(slave);
	}
}


void ssi32m_configure(struct ssi32_slavedev *slave, __be32 addr)
{
	incnetdev_configure(slave->master->ndev->ndev, addr);
}

#define MAX_DUMP_PREFIX_SIZE 16

void ssi32m_dump(struct ssi32_slavedev *slave, int chip_select,
		char *title, char *buf, int len)
{
	char prefix[MAX_DUMP_PREFIX_SIZE];
	const char *net_name = dev_name(&slave->master->ndev->ndev->dev);

	printk(KERN_DEBUG "%s.%d: %s\n", net_name, chip_select, title);

	snprintf(prefix, MAX_DUMP_PREFIX_SIZE, "%s.%d: ",
			net_name, chip_select);
	print_hex_dump(KERN_DEBUG, prefix, DUMP_PREFIX_OFFSET, 16, 1,
			buf, len, false);
}
