/*
 * SSI32 device
 *
 * Copyright (C) 2012 Andreas Pape, Robert Bosch Car Multimedia GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/spi/spi.h>
#include <linux/interrupt.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/of_gpio.h>
#include <linux/spi/spi_bitbang.h>

#include "ssi32_dev.h"
#include "ssi32_prot.h"


#define DRIVER_NAME "SSI32"
#define SSI_MAX_PORTS 8



static int dump_spi;
static int speed_hz;
static int direct_imx;

struct ssi32_device {
	struct ssi32_slavedev *slave;
	struct spi_device *spi;
	struct spi_transfer xfer;
	struct spi_message  msg;
	int scs;
	int srq;
	int ack;
	int shared_bus;
};

struct ssi32_device_cfg {
	int blocksize;
	char mac;
	int mbhdlen;
};

static struct ssi32_device_cfg cfg_adr3 = {
	.blocksize = 68,
	.mac = 0x01,
	.mbhdlen = 0,
};

static struct ssi32_device_cfg cfg_adr3spi0 = {
	.blocksize = 256,
	.mac = 0x01,
	.mbhdlen = 0,
};

static struct ssi32_device_cfg cfg_inc_v850 = {
	.blocksize = 64,
	.mac = 0x02,
	.mbhdlen = 4,
};


static const struct of_device_id ssi32_dt_ids[] = {
	{ .compatible = "RBCM,V850_SSI32", .data = &cfg_inc_v850, },
	{ .compatible = "RBCM,ADR3_SSI32", .data = &cfg_adr3, },
	{ .compatible = "RBCM,ADR3SPI0_SSI32", .data = &cfg_adr3spi0, },
	{ /* sentinel */ }
};

static irqreturn_t ssi32_srq(int irq, void *data)
{
	struct ssi32_device *ssidev = data;
	/*route to worker:*/
	ssi32m_slave_srq(ssidev->slave, gpio_get_value(ssidev->srq));
	return IRQ_HANDLED;
}


void ssi32_chipselect(struct ssi32_device *ssidev, int is_active)
{
	gpio_set_value(ssidev->scs,
			is_active ^ !(ssidev->spi->mode & SPI_CS_HIGH));
}

void ssi32_hw_ack(struct ssi32_device *ssidev, int is_ok)
{
	gpio_set_value(ssidev->ack, is_ok);
}

int ssi32_uses_hw_ack(struct ssi32_device *ssidev)
{
	return ssidev->ack >= 0;
}

#define SSI_MAX_WORDSIZE 4
static int calc_wordsize(int len)
{
	int wrdsize = SSI_MAX_WORDSIZE;
	while (len % wrdsize)
		wrdsize >>= 1;
	return wrdsize;
}


static int ssi_htonx(char *buf, int len, int bpw)
{
	u32 *src32;
	u16 *src16;
	int i;

	switch (bpw) {
	case 1:
		break;
	case 2:
		len >>= 1;
		src16 = (u16 *)buf;
		for (i = 0; i < len; i++) {
			*src16 = htons(*src16);
			src16++;
	}
		break;
	case 4:
		len >>= 2;
		src32 = (u32 *)buf;
		for (i = 0; i < len; i++) {
			*src32 = htonl(*src32);
			src32++;
		}
		break;
	default:
		break;
	}
	return 0;
}

static int ssi_ntohx(char *buf, int len, int bpw)
{
	u32 *src32;
	u16 *src16;
	int i;

	switch (bpw) {
	case 1:
		break;
	case 2:
		len >>= 1;
		src16 = (u16 *)buf;
		for (i = 0; i < len; i++) {
			*src16 = ntohs(*src16);
			src16++;
		}
		break;
	case 4:
		len >>= 2;
		src32 = (u32 *)buf;
		for (i = 0; i < len; i++) {
			*src32 = ntohl(*src32);
			src32++;
		}
		break;
	default:
		break;
	}
	return 0;
}


static void ssi32_transfer_complete(void *context)
{
	struct ssi32_device *ssidev = context;
	ssi_ntohx(ssidev->xfer.rx_buf, ssidev->xfer.len,
					ssidev->xfer.bits_per_word/8);
	if (dump_spi)
		ssi32m_dump(ssidev->slave, ssidev->spi->chip_select, "RX",
				ssidev->xfer.rx_buf, ssidev->xfer.len);

	ssi32m_spi_complete(ssidev->slave);
	return;
}

int ssi32_transfer(struct ssi32_device *ssidev,
		char *tx_buf, char *rx_buf, int len)
{
	u32 status;
	int bpword;
	spi_message_init(&ssidev->msg);
	ssidev->msg.complete = ssi32_transfer_complete;
	ssidev->msg.context = ssidev;

	bpword = calc_wordsize(len);
	ssidev->xfer.len = len;
	ssidev->xfer.tx_buf = tx_buf;
	ssidev->xfer.rx_buf = rx_buf;
	ssidev->xfer.bits_per_word = bpword*8;
	ssidev->xfer.cs_change = 0; /*no cs and waitstates between transfers*/
	if (speed_hz)
		ssidev->xfer.speed_hz = speed_hz;
	else
		ssidev->xfer.speed_hz = ssidev->spi->max_speed_hz;
	ssidev->xfer.delay_usecs = 0;

	if (dump_spi)
		ssi32m_dump(ssidev->slave, ssidev->spi->chip_select, "TX",
				tx_buf, len);

	ssi_htonx(tx_buf, len, bpword);
	spi_message_add_tail(&ssidev->xfer, &ssidev->msg);
	status = spi_async_locked(ssidev->spi, &ssidev->msg);
	return status;
}


int ssi32_bus_lock(struct ssi32_device *ssidev)
{
	u32 status;
	char tx_buf[1];
	char rx_buf[1];

	if (!ssidev->shared_bus)
		return 0;

	/*lock bus prevents other users from adding new messages */
	spi_bus_lock(ssidev->spi->master);

	/*add dummy message and wait for completion
	to make sure no more SPI messages are pending*/
	spi_message_init(&ssidev->msg);
	ssidev->xfer.len = 0;/*len 0 will result in skipping real xfer*/
	ssidev->xfer.tx_buf = tx_buf;
	ssidev->xfer.rx_buf = rx_buf;
	ssidev->xfer.bits_per_word = 8;
	ssidev->xfer.cs_change = 0; /*no cs and waitstates between transfers*/
	ssidev->xfer.speed_hz = ssidev->spi->max_speed_hz;
	ssidev->xfer.delay_usecs = 0;

	if (dump_spi)
		ssi32m_dump(ssidev->slave, ssidev->spi->chip_select, "DUMMY",
				tx_buf, 0);

	spi_message_add_tail(&ssidev->xfer, &ssidev->msg);
	status = spi_sync_locked(ssidev->spi, &ssidev->msg);

	/*BUS LOCKED, NO OTHER SPI XFER IS PENDING*/

	return status;
}


int ssi32_bus_unlock(struct ssi32_device *ssidev)
{
	if (!ssidev->shared_bus)
		return 0;
	return spi_bus_unlock(ssidev->spi->master);
}

/*check whether shared with NON SSI32 devices*/
static int ssi32_check_shared_bus(struct spi_master *master)
{
	struct device_node *nc;
	int non_ssi32 = 0;

	if (!master->dev.of_node)
		return 1;/*unknown->treat like shared*/

	for_each_child_of_node(master->dev.of_node, nc) {

		/*search for non-SSI32 SPI devices*/
		if (!of_find_property(nc, "scs-ssi32", NULL))
			non_ssi32++;
	}
	return non_ssi32;
}

static unsigned int inet_addr(char *str)
{
	int a, b, c, d;
	char arr[4];
	sscanf(str, "%d.%d.%d.%d", &a, &b, &c, &d);
	arr[0] = a; arr[1] = b; arr[2] = c; arr[3] = d;
	return *(unsigned int *) arr;
}

int ssi32_dev_start(struct ssi32_device *ssidev)
{
	struct spi_device *spidev = ssidev->spi;
	int ret;

	/*TODO: cleanup, if GPIO request fails !*/
	ret = gpio_request(ssidev->scs, DRIVER_NAME);
	if (ret)
		dev_err(&spidev->dev, "SSI32: failed to get SCS GPIO %d\n",
							ssidev->scs);
	else
		gpio_direction_output(ssidev->scs,
				spidev->mode & SPI_CS_HIGH ? 0 : 1);

	ret = gpio_request(ssidev->srq, DRIVER_NAME);
	if (ret)
		dev_err(&spidev->dev, "SSI32: failed to get SRQ GPIO %d\n",
							ssidev->srq);
	else
		gpio_direction_input(ssidev->srq);

	ret = request_irq(gpio_to_irq(ssidev->srq), ssi32_srq,
			(IRQF_TRIGGER_RISING|IRQF_TRIGGER_FALLING),
			"ssi32_srq", ssidev);

	if (ssi32_uses_hw_ack(ssidev)) {
		ret = gpio_request(ssidev->ack, DRIVER_NAME);
		if (ret)
			dev_err(&spidev->dev,
					"SSI32: failed to get ACK GPIO %d\n",
					ssidev->ack);
		else
			gpio_direction_output(ssidev->ack, 1);
	}

	/*only notify SRQ_L - otherwise it is treated as spurious irq as SRQ
	is already registered as high on startup*/
	if (!gpio_get_value(ssidev->srq))
		ssi32m_slave_srq(ssidev->slave,	0);

	return ret;
}

void ssi32_dev_stop(struct ssi32_device *ssidev)
{
	free_irq(gpio_to_irq(ssidev->srq), ssidev);

	gpio_free(ssidev->scs);
	gpio_free(ssidev->srq);
	if (ssi32_uses_hw_ack(ssidev))
		gpio_free(ssidev->ack);

	return;
}

static int ssi32_dev_create(const struct ssi32_device_cfg *devcfg,
				struct spi_device *spidev)
{
	int ret, len;
	struct ssi32_device *ssidev =
			kzalloc(sizeof(struct ssi32_device), GFP_KERNEL);
	struct ssi32_slavedev *slave;
	struct device_node *np;
	struct ssi32_dcfg dcfg;
	struct device_node *inc_node = NULL;
	char *inc_node_path, *local_addr = NULL, *ifname = NULL;
	struct property *flowctl, *rxonly;
	const __be32 *prop;

	np = spidev->dev.of_node;
	ssidev->scs = of_get_named_gpio(np, "scs-ssi32", 0);
	ssidev->srq = of_get_named_gpio(np, "srq-ssi32", 0);
	ssidev->ack = of_get_named_gpio(np, "ack-ssi32", 0);

	inc_node_path = (char *) of_get_property(np, "inc-node", NULL);
	if (inc_node_path)
		inc_node = of_find_node_by_path(inc_node_path);
	if (inc_node) {
		local_addr = (char *) of_get_property(inc_node,
				"local-addr", NULL);
		ifname = (char *) of_get_property(inc_node,
				"interface-name", NULL);
	}

	flowctl = of_find_property(np, "flowcontrol", NULL);
	rxonly = of_find_property(np, "rx-only", NULL);

	prop = of_get_property(np, "ssr-timeout-us", &len);
	if (!prop || len < sizeof(*prop))
		dcfg.tssr = 0;
	else
		dcfg.tssr = 1000*be32_to_cpup(prop);
	prop = of_get_property(np, "cd1-timeout-us", &len);
	if (!prop || len < sizeof(*prop))
		dcfg.tcd1 = 0;
	else
		dcfg.tcd1 = 1000*be32_to_cpup(prop);
	prop = of_get_property(np, "cd2-timeout-us", &len);
	if (!prop || len < sizeof(*prop))
		dcfg.tcd2 = 0;
	else
		dcfg.tcd2 = 1000*be32_to_cpup(prop);
	prop = of_get_property(np, "esr-timeout-us", &len);
	if (!prop || len < sizeof(*prop))
		dcfg.tesr = 0;
	else
		dcfg.tesr = 1000*be32_to_cpup(prop);
	prop = of_get_property(np, "mri-timeout-us", &len);
	if (!prop || len < sizeof(*prop))
		dcfg.tmri = 0;
	else
		dcfg.tmri = be32_to_cpup(prop)/1000;
	prop = of_get_property(np, "mnr-retries", &len);
	if (!prop || len < sizeof(*prop))
		dcfg.mnr = 0;
	else
		dcfg.mnr = be32_to_cpup(prop);

	ssidev->spi = spidev;
	dcfg.bus = spidev->master->bus_num;
	dcfg.node = spidev->chip_select;
	dcfg.mac = devcfg->mac;
	dcfg.blocksize = devcfg->blocksize;
	dcfg.mbhdlen = devcfg->mbhdlen;
	dcfg.flowcontrol = flowctl == NULL ? 0 : 1;
	dcfg.addr = local_addr ? inet_addr(local_addr) : 0;
	dcfg.ifname = ifname;
	dcfg.rxonly = rxonly == NULL ? 0 : 1;
	ret = ssi32m_register_slave_device(ssidev, &dcfg, &slave);
	if (ret)
		goto error;

	ssidev->slave = slave;

	spi_set_drvdata(spidev, ssidev);

	ssidev->shared_bus = ssi32_check_shared_bus(spidev->master);

	if (local_addr)
		ssi32m_configure(slave, inet_addr(local_addr));
	else
		ret = ssi32_slave_start(slave);
	if (ret)
		goto error;

	dev_dbg(&spidev->dev, "%sSSI32: dev created: " \
			"scs=%d srq=%d ack=%d cs=%d bus=%d mode=%d\n",
			ssidev->shared_bus ? "SHARED " : "",
			ssidev->scs, ssidev->srq, ssidev->ack,
			spidev->chip_select, spidev->master->bus_num,
			spidev->mode);
	return 0;

error:
	kfree(ssidev);
	return ret;
}


static int ssi32_probe(struct spi_device *spidev)
{
	const struct of_device_id *of_id;
	if (!spidev) {
		dev_err(&spidev->dev, "SSI32 probe: No spi device\n");
		return -ENODEV;
	}
	of_id = of_match_device(ssi32_dt_ids, &spidev->dev);
	if (!of_id) {
		dev_err(&spidev->dev, "SSI32 probe: No OFID\n");
		return -ENODEV;
	}
	dev_dbg(&spidev->dev, "SSI32 probe: %s\n", of_id->compatible);
	return ssi32_dev_create(of_id->data, spidev);
}

static int ssi32_remove(struct spi_device *spidev)
{
	struct ssi32_device *ssidev;

	ssidev = spi_get_drvdata(spidev);

	ssi32_slave_stop(ssidev->slave);
	ssi32m_unregister_slave_device(ssidev->slave);
	kfree(ssidev);
	return 0;
}


static struct spi_driver ssi32_driver = {
		.driver = {
				.name = "ssi32",
				.owner = THIS_MODULE,
				.of_match_table = ssi32_dt_ids,
		},
		/* This will get called when we do spi_register_driver */
		.probe = ssi32_probe,
		.remove = ssi32_remove,
};


static void __exit ssi32dev_exit_module(void)
{
	spi_unregister_driver(&ssi32_driver);
	return;
}


static int __init ssi32dev_init_module(void)
{
	int ret;
	ret = spi_register_driver(&ssi32_driver);
	if (ret != 0)
		pr_err("SSI32: Error %d registering driver\n", ret);
	return ret;
}

module_param(dump_spi, int, S_IRUGO|S_IWUSR);
MODULE_PARM_DESC(dump_spi, "Hexdump of SPI transfer (0/1 == off/on)");

module_param(speed_hz, int, S_IRUGO|S_IWUSR);
MODULE_PARM_DESC(speed_hz, "EXPERIMENTAL: force spi speed hz");

module_param(direct_imx, int, S_IRUGO|S_IWUSR);
MODULE_PARM_DESC(direct_imx, "EXPERIMENTAL: directly call imx spi");

module_init(ssi32dev_init_module);
module_exit(ssi32dev_exit_module);



MODULE_LICENSE("GPL");
