/*
 * drivers/pwm/pwm-inc.c
 *
 * PWM port extender over INC
 *
 *	Copyright 2014 Robert Bosch Engineering and Business solutions Ltd.,
 *			Inc. All Rights Reserved.
 *
 *	  This program is free software; you can redistribute it and/or modify
 *	  it under the terms of the GNU General Public License as published by
 *	  the Free Software Foundation; either version 2 of the License, or
 *	  (at your option) any later version.
 *
 *	  This program is distributed in the hope that it will be useful,
 *	  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the
 *	  GNU General Public License for more details.
 *
 *	  You should have received a copy of the GNU General Public License
 *	  along with this program; if not, write to the Free Software
 *	  Foundation, Inc.,59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/platform_device.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/net.h>
#include <linux/in.h>
#include <linux/inc.h>
#include <linux/inc_ports.h>
#include <linux/dgram_service.h>
#include <linux/kthread.h>
#include <linux/kernel.h>
#include <linux/err.h>
#include <linux/io.h>
#include <linux/pwm.h>
#include <linux/pinctrl/consumer.h>
#include <linux/delay.h>

/*
 * for testing purposes the remote port may be set differently
 * from the local port
 */
static u16 remote_port;
module_param(remote_port, ushort, 0);
MODULE_PARM_DESC(remote_port, "INC remote port");

#define SCC_PORT_EXTENDER_PWM_C_COMPONENT_STATUS	0x20
#define SCC_PORT_EXTENDER_PWM_R_COMPONENT_STATUS	0x21
#define SCC_PORT_EXTENDER_PWM_R_REJECT			0x0B
#define SCC_PORT_EXTENDER_PWM_R_REJECT_NR		0x00
#define SCC_PORT_EXTENDER_PWM_R_REJECT_UM		0x01
#define SCC_PORT_EXTENDER_PWM_R_REJECT_IP		0x02
#define SCC_PORT_EXTENDER_PWM_R_REJECT_TU		0x03
#define SCC_PORT_EXTENDER_PWM_R_REJECT_VM		0x04

#define SCC_PORT_EXTENDER_PWM_C_SET_PWM_VAL		0x36
#define SCC_PORT_EXTENDER_PWM_R_SET_PWM_VAL		0x37

#define SCC_PORT_EXTENDER_PWM_C_GET_PWM			0x30
#define SCC_PORT_EXTENDER_PWM_R_GET_PWM			0x31

#define SCC_PORT_EXTENDER_PWM_C_GET_CONFIG		0x34
#define SCC_PORT_EXTENDER_PWM_R_GET_CONFIG		0x35

#define PWM_INC_DRV_NAME		"pwm-inc"
#define PWM_INC_STATUS_ACTIVE		0x01
#define PWM_INC_STATUS_INACTIVE		0x02
#define PWM_INC_RX_BUFSIZE		64
#define PWM_INC_MAX_NPWM		256
#define PWM_INC_DGRAM_MAX		PWM_INC_RX_BUFSIZE
#define PWM_INC_PEER_RESPONSE_TIMEOUT	HZ

struct pwm_inc {
	struct pwm_chip		chip;
	struct device		*dev;
	struct task_struct	*rx_task;
	wait_queue_head_t	waitq;
	/* lock used for rx thread task */
	struct mutex		lock;
	struct socket		*sock;
	struct sk_dgram		*dgram;
	struct sockaddr_in	local;
	struct sockaddr_in	remote;
	unsigned int		is_active:1; /* PWM chip active flag */
	unsigned int		err_stat:1; /* Err status from handle msg */
	unsigned int		rsp_status_active:1; /* COMP status flag */
	unsigned int		get_config:1;
	int			rx_flag[PWM_INC_MAX_NPWM];
	u8			check_bytes[PWM_INC_RX_BUFSIZE];
	int			chn_active[PWM_INC_MAX_NPWM];
};

static int inc_pwm_request(struct pwm_chip *chip, struct pwm_device *pwm);
static void inc_pwm_free(struct pwm_chip *chip, struct pwm_device *pwm);
static int pwm_inc_sendmsg(struct pwm_inc *pwm_inc, u8 *data, int len);
static int pwm_inc_get_dt_properties(struct pwm_inc *pwm_inc);
static int pwm_inc_probe(struct platform_device *pdev);
static int pwm_inc_remove(struct platform_device *pdev);
static int inc_pwm_config(struct pwm_chip *chip,
			  struct pwm_device *pwm, int duty_ns, int period_ns);
static int inc_pwm_enable(struct pwm_chip *chip, struct pwm_device *pwm);
static void inc_pwm_disable(struct pwm_chip *chip, struct pwm_device *pwm);
static int inc_pwm_set_polarity(struct pwm_chip *chip,
				struct pwm_device *pwm,
				enum pwm_polarity polarity);

static const struct of_device_id inc_pwmchip_match[] = {
	{ .compatible = "rbcm,pwm-inc", },
	{}
};
/*PWM File operation structure update*/
static struct pwm_ops pwm_inc_ops = {
	.enable = inc_pwm_enable,
	.disable = inc_pwm_disable,
	.config = inc_pwm_config,
	.set_polarity = inc_pwm_set_polarity,
	.request = inc_pwm_request,
	.free = inc_pwm_free,
	.owner = THIS_MODULE,
};

/*!
 * to_pwm_inc - To get the driver structure.
 *
 * @param chip: pwm_chip structure
 *
 * @return	pwm_inc pointer structure.
 *
 */
static inline struct pwm_inc *to_pwm_inc(struct pwm_chip *chip)
{
	return container_of(chip, struct pwm_inc, chip);
}

/*!
 * inet_addr - concatenate the ip address.
 *
 * @param str:  ip address string
 *
 * @return	u32 - ip concatenated values.
 *
 */
static u32 inet_addr(char *str)
{
	int a, b, c, d;
	char arr[4];

	sscanf(str, "%d.%d.%d.%d", &a, &b, &c, &d);
	arr[0] = a;
	arr[1] = b;
	arr[2] = c;
	arr[3] = d;
	return *(u32 *)arr;
}

/*!
 * pwm_inc_get_dt_properties - get the INC local and remote addr.
 *
 * @param pwm_inc:  driver data structure
 *
 * @return	0 - success or negative on failure.
 *
 */
static int pwm_inc_get_dt_properties(struct pwm_inc *pwm_inc)
{
	struct device_node *pwm_node = pwm_inc->dev->of_node;
	struct device_node *inc_node;
	const char *prop_str, *inc_node_path;
	const void *prop;

	pwm_inc->local.sin_family = AF_INET;
	pwm_inc->remote.sin_family = AF_INET;

	prop_str = "node";
	prop = of_get_property(pwm_node, prop_str, NULL);
	if (prop)
		inc_node_path = (char *)prop;
	else
		goto err;

	inc_node = of_find_node_by_path(inc_node_path);
	if (!inc_node) {
		dev_err(pwm_inc->dev, "can't find %s node in device tree\n",
			inc_node_path);
		return -ENODEV;
	}

	prop_str = "local-addr";
	prop = of_get_property(inc_node, prop_str, NULL);
	if (prop)
		pwm_inc->local.sin_addr.s_addr = inet_addr((char *)prop);
	else
		goto err;

	prop_str = "remote-addr";
	prop = of_get_property(inc_node, prop_str, NULL);
	if (prop)
		pwm_inc->remote.sin_addr.s_addr = inet_addr((char *)prop);
	else
		goto err;

	return 0;

err:
	dev_err(pwm_inc->dev, "could not read dt property: %s\n", prop_str);
	return -ENODEV;
}

/*!
 * validate_bytes - validate the bytes by comparing the source and dest to the
 *		    size specified.
 *
 * @param source:  source pointer to check
 * @param dest	:  destination pointer to check
 * @param size	:  Containing the actual size to be checked
 * @return	0 - success or negative on failure.
 *
 */
static inline int validate_bytes(u8 *source, u8 *dest, int size)
{
	int count;

	for (count = 1; count <= size; count++) {
		if (source[count] != dest[count])
			return -EIO;
	}
	return 0;
}

/*!
 * inc_pwm_config - Configures the pwm channel frequency and duty cycle.
 *
 * @param chip	:  structure containing the registered pwm chip info
 * @param pwm	:  structure containing the pwm_device info
 * @param duty_ns	:  contains duty cycle info from the user
 * @param period_ns	:  contains the time period info from the user
 * @return	0 - success or negative on failure.
 *
 */
static int inc_pwm_config(struct pwm_chip *chip,
		struct pwm_device *pwm, int duty_ns, int period_ns)
{
	u8 data[PWM_INC_RX_BUFSIZE];
	int ret;
	int freq = 1000000000/period_ns;
	struct pwm_inc *pwm_inc = to_pwm_inc(chip);

	/*check for channel active*/
	if (!pwm_inc->chn_active[pwm->hwpwm])
		return -EPERM;
	/*duty cycle must not be greater than period*/
	if (duty_ns > period_ns)
		return -EIO;
	data[0] = SCC_PORT_EXTENDER_PWM_C_SET_PWM_VAL;
	data[1] = pwm->hwpwm;
	/*freq is 4 bytes*/
	data[2] = freq >> 0;
	data[3] = freq >> 8;
	data[4] = freq >> 16;
	data[5] = freq >> 24;
	data[6] = (duty_ns * 100) / period_ns;
	pwm_inc->rx_flag[data[1]] = false;

	ret = pwm_inc_sendmsg(pwm_inc, data, 7);
	if (ret < 0)
		return ret;
	ret = wait_event_interruptible_timeout(pwm_inc->waitq,
			pwm_inc->rx_flag[data[1]],
			PWM_INC_PEER_RESPONSE_TIMEOUT);
	pwm_inc->rx_flag[data[1]] = false;

	if (ret < 0)
		return ret;
	if (ret == 0) {
		dev_alert(pwm_inc->dev, "%s: pwm%d: peer not responding\n",
			  __func__, pwm_inc->chip.base);
		return -ETIME;
	}
	ret = validate_bytes(data, pwm_inc->check_bytes, 6);
	if (ret)
		return ret;

	dev_dbg(pwm_inc->dev, "%s: freq=%dHz\n", __func__, freq);
	dev_dbg(pwm_inc->dev, "%s: duty_cycle=%d%%\n", __func__, data[6]);
	return 0;
}

/*!
 * inc_pwm_request - This function is a callback when pwm channel is  exported.
 *
 * @param chip	:  structure containing the registered pwm chip info
 * @param pwm	:  structure containing the pwm_device info
 * @return	0 - success or negative on failure.
 *
 */
static int inc_pwm_request(struct pwm_chip *chip, struct pwm_device *pwm)
{
	struct pwm_inc *pwm_inc = to_pwm_inc(chip);

	if (pwm_inc->chn_active[pwm->hwpwm])
		return -EBUSY;
	pwm_inc->chn_active[pwm->hwpwm] = true;
	return 0;
}

/*!
 * inc_pwm_free - This function is a callback when pwm channel is  unexported.
 *
 * @param chip	:  structure containing the registered pwm chip info
 * @param pwm	:  structure containing the pwm_device info
 * @return	void.
 *
 */
static void inc_pwm_free(struct pwm_chip *chip, struct pwm_device *pwm)
{
	struct pwm_inc *pwm_inc = to_pwm_inc(chip);

	if (!pwm_inc->chn_active[pwm->hwpwm])
		dev_alert(pwm_inc->dev, "%s: channel not in active", __func__);
	pwm_inc->chn_active[pwm->hwpwm] = false;
}

/*!
 * inc_pwm_enable - This function will enable PWM clock.
 *
 * @param chip	:  structure containing the registered pwm chip info
 * @param pwm	:  structure containing the pwm_device info
 * @return	0 - success or negative on failure.
 *
 */
static int inc_pwm_enable(struct pwm_chip *chip, struct pwm_device *pwm)
{
	struct pwm_inc *pwm_inc = to_pwm_inc(chip);

	if (!pwm_inc->chn_active[pwm->hwpwm])
		return -EPERM;
	/*No support from SCC*/
	return 0;
}

/*!
 * inc_pwm_disable - This function will disable PWM clock.
 *
 * @param chip	:  structure containing the registered pwm chip info
 * @param pwm	:  structure containing the pwm_device info
 * @return	void
 *
 */
static void inc_pwm_disable(struct pwm_chip *chip, struct pwm_device *pwm)
{
	struct pwm_inc *pwm_inc = to_pwm_inc(chip);

	if (!pwm_inc->chn_active[pwm->hwpwm])
		dev_alert(pwm_inc->dev,	"%s: channel not in active", __func__);
	/*No support from SCC*/
}

/*!
 * inc_pwm_set_polarity - This function will change the polarity of the pwm chn.
 *
 * @param chip	:  structure containing the registered pwm chip info
 * @param pwm	:  structure containing the pwm_device info
 * @param polarity	:  contains the polarity info from the user
 * @return	0 - success or negative on failure.
 *
 */
static int inc_pwm_set_polarity(struct pwm_chip *chip,
				struct pwm_device *pwm,
				enum pwm_polarity polarity)
{
	struct pwm_inc *pwm_inc = to_pwm_inc(chip);

	if (!pwm_inc->chn_active[pwm->hwpwm]) {
		dev_alert(pwm_inc->dev,	"%s: channel not in active", __func__);
		return -EPERM;
	}
	/*No support from SCC*/
	return 0;
}

/*!
 * pwm_inc_sendmsg - This function will be used to send msg to scc.
 *
 * @param pwm_inc	:  contains driver data structure
 * @param data	:  contains command to be send to scc
 * @param len	:  contains the length of the data.
 * @return	int written data - success or negative on failure.
 *
 */
static int pwm_inc_sendmsg(struct pwm_inc *pwm_inc, u8 *data, int len)
{
	int ret;
	int count;

	dev_dbg(pwm_inc->dev, "%s: dgram_send to %pI4:%d\n", __func__,
		&pwm_inc->remote.sin_addr.s_addr,
		ntohs(pwm_inc->remote.sin_port));

	ret = dgram_send(pwm_inc->dgram, data, len);
	if (ret < 0)
		dev_alert(pwm_inc->dev, "%s: could not send msg: %d\n",
			__func__, ret);
	dev_dbg(pwm_inc->dev, "%s: ret=%d\n", __func__,	ret);
	for (count = 0; count < len; count++)
		dev_dbg(pwm_inc->dev, "%s: data[%d]=0x%02X\n", __func__,
			count, ((u8 *)data)[count]);
	return ret;
}

/*!
 * pwm_inc_recvmsg - This function will be receives msg from scc.
 *
 * @param pwm_inc	:  contains driver data structure
 * @param data	:  contains response from scc
 * @param len	:  contains the length of the data.
 * @return	int 0 - success or negative on failure.
 *
 */
static int pwm_inc_recvmsg(struct pwm_inc *pwm_inc, void *data, int len)
{
	int ret;
	int count;

	do {
		ret = dgram_recv(pwm_inc->dgram, data, len);
		if ((ret < 0) && (ret != -EAGAIN))
			dev_alert(pwm_inc->dev, "%s: dgram_recv failed: %d\n",
				__func__, ret);
		dev_dbg(pwm_inc->dev, "%s: dgram_recv: ret=%d\n",
			__func__, ret);
		for (count = 0; count < ret; count++)
			dev_dbg(pwm_inc->dev, "%s: data[%d]=0x%02X\n",
				__func__, count, ((u8 *)data)[count]);
	} while (ret == -EAGAIN);

	return ret;
}
/*!
 * pwm_inc_sock_open - This function will be establish connection to scc.
 *
 * @param pwm_inc	:  structure containing the driver data structure.
 * @return	int 0 - success or negative on failure.
 *
 */
static int pwm_inc_sock_open(struct pwm_inc *pwm_inc)
{
	struct socket *sock = NULL;
	struct sk_dgram *dgram;
	int ret;

	dev_dbg(pwm_inc->dev, "%s: local %pI4:%d remote %pI4:%d\n", __func__,
		&pwm_inc->local.sin_addr.s_addr,
		ntohs(pwm_inc->local.sin_port),
		&pwm_inc->remote.sin_addr.s_addr,
		ntohs(pwm_inc->remote.sin_port));

	ret = sock_create_kern(AF_INC, SOCK_STREAM, 0, &sock);
	if (ret < 0) {
		dev_err(pwm_inc->dev, "%s: sock_create_kern failed: %d\n",
			__func__, ret);
		goto out;
	}

	dgram = dgram_init(sock, PWM_INC_DGRAM_MAX, NULL);
	if (dgram == NULL) {
		dev_err(pwm_inc->dev, "%s: dgram_init failed\n", __func__);
		goto out_release;
	}

	ret = kernel_bind(sock, (struct sockaddr *)&pwm_inc->local,
			  sizeof(pwm_inc->local));
	if (ret < 0) {
		dev_err(pwm_inc->dev, "%s: kernel_bind failed: %d\n",
			__func__, ret);
		goto out_release;
	}

	ret = kernel_connect(sock, (struct sockaddr *)&pwm_inc->remote,
			     sizeof(pwm_inc->remote), 0);
	if (ret < 0) {
		dev_err(pwm_inc->dev, "%s: kernel_connect failed: %d\n",
			__func__, ret);
		goto out_release;
	}

	pwm_inc->sock = sock;
	pwm_inc->dgram = dgram;

	return 0;

out_release:
	sock_release(sock);
out:
	return ret;
}

/*!
 * pwm_inc_sock_close - This function will be closes connection to scc.
 *
 * @param pwm_inc	:  structure containing the driver data structure.
 * @return	void.
 *
 */
static void pwm_inc_sock_close(struct pwm_inc *pwm_inc)
{
	dev_dbg(pwm_inc->dev, "%s: local %pI4:%d remote %pI4:%d\n", __func__,
		&pwm_inc->local.sin_addr.s_addr,
		ntohs(pwm_inc->local.sin_port),
		&pwm_inc->remote.sin_addr.s_addr,
		ntohs(pwm_inc->remote.sin_port));

	if (pwm_inc->dgram) {
		dev_dbg(pwm_inc->dev, "%s: dgram_exit\n", __func__);
		dgram_exit(pwm_inc->dgram);
	}

	if (pwm_inc->sock) {
		dev_dbg(pwm_inc->dev, "%s: sock_shutdown\n", __func__);
		kernel_sock_shutdown(pwm_inc->sock, SHUT_RDWR);
	}

	if (pwm_inc->sock) {
		dev_dbg(pwm_inc->dev, "%s: sock_release\n", __func__);
		sock_release(pwm_inc->sock);
	}

	return;
}

/*!
 * pwm_inc_activate - This function registers SCC extender pwm chn to kernel.
 *
 * @param pwm_inc	:  contains driver data structure
 * @return	int 0 - success or negative on failure.
 *
 */
static int pwm_inc_activate(struct pwm_inc *pwm_inc)
{
	struct pwm_chip *chip;
	int ret;

	chip = &pwm_inc->chip;
	chip->dev = pwm_inc->dev;
	chip->ops = &pwm_inc_ops;
	chip->base = -1;
	/*Register PWM chip*/
	ret = pwmchip_add(chip);
	if (ret < 0) {
		dev_err(pwm_inc->dev, "%s: could not register pwmchip: %d\n",
			__func__, ret);
		return ret;
	}
	/*Activate the chip flag*/
	pwm_inc->is_active = 1;
	dev_info(pwm_inc->dev, "%s: %d PWMs\n", __func__, chip->npwm);

	return 0;
}

/*!
 * pwm_inc_handle_msg - This function will be called from the receive task to
 *			handle the response bytes and to validate them.
 *
 * @param pwm_inc	:  contains driver data structure
 * @param data	:  contains response from scc
 * @param size	:  contains the length of the data.
 * @return	int 0 - success or negative on failure.
 *
 */
static int pwm_inc_handle_msg(struct pwm_inc *pwm_inc, u8 *data, int size)
{
	int ret = 0;
	int count;
	struct device *dev = pwm_inc->dev;

	if (size <= 0) {
		dev_alert(dev, "%s: invalid msg size: %d\n",
			__func__, size);
		return -EPROTO;
	}

	/* before activation discard all non-activation messages */
	if (data[0] != SCC_PORT_EXTENDER_PWM_R_COMPONENT_STATUS &&
	    !pwm_inc->rsp_status_active) {
		dev_dbg(dev, "%s: discarding msgid: 0x%02X\n",
			__func__, data[0]);
		return 0;
	}

	switch (data[0]) {
	case SCC_PORT_EXTENDER_PWM_R_COMPONENT_STATUS:
		if (size != 2) {
			dev_alert(dev, "%s: invalid msg size: 0x%02X %d\n",
				__func__, data[0], size);
			return -EPROTO;
		}
		if (data[1] != PWM_INC_STATUS_ACTIVE) {
			dev_alert(dev, "%s: invalid status: %d\n",
				__func__, data[1]);
			return -EPROTO;
		}
		pwm_inc->rsp_status_active = true;
		wake_up_interruptible(&pwm_inc->waitq);
		break;
	case SCC_PORT_EXTENDER_PWM_R_SET_PWM_VAL:
		if (!pwm_inc->is_active) {
			dev_alert(dev, "%s: inactive: 0x%02X %d\n",
				__func__, data[0], size);
			return -EPROTO;
		}
		if (size != 7) {
			dev_alert(dev, "%s: invalid msg size: 0x%02X %d\n",
				__func__, data[0], size);
			return -EPROTO;
		}
		if (data[1] >= pwm_inc->chip.npwm) {
			dev_alert(dev, "%s: invalid number of pwm: %d\n",
				__func__, data[1]);
			return -EPROTO;
		}
		for (count = 1; count < size; count++)
			pwm_inc->check_bytes[count] = data[count];
		pwm_inc->rx_flag[data[1]] = true;
		wake_up_interruptible(&pwm_inc->waitq);
		break;
	case SCC_PORT_EXTENDER_PWM_R_GET_PWM:
		if (!pwm_inc->is_active) {
			dev_alert(dev, "%s: inactive: 0x%02X %d\n",
				__func__, data[0], size);
			return -EPROTO;
		}
		if (size != 7) {
			dev_alert(dev, "%s: invalid msg size: 0x%02X %d\n",
				__func__, data[0], size);
			return -EPROTO;
		}
		if (data[1] >= pwm_inc->chip.npwm) {
			dev_alert(dev, "%s: invalid pwm id: %d\n",
				__func__, data[1]);
			return -EPROTO;
		}
		pwm_inc->rx_flag[data[1]] = true;
		wake_up_interruptible(&pwm_inc->waitq);
		break;
	case SCC_PORT_EXTENDER_PWM_R_GET_CONFIG:
		if (size != 3) {
			dev_alert(dev, "%s: invalid msg size: 0x%02X %d\n",
				__func__, data[0], size);
			return -EPROTO;
		}

		if (!pwm_inc->is_active) {
			pwm_inc->chip.npwm = data[1];
			pwm_inc->get_config = true;
			wake_up_interruptible(&pwm_inc->waitq);
		}
		break;
	case SCC_PORT_EXTENDER_PWM_R_REJECT:
		if (size != 3) {
			dev_alert(dev, "%s: invalid msg size: 0x%02X %d\n",
				__func__, data[0], size);
			return -EPROTO;
		}
		dev_alert(dev, "%s: msg rejected: id 0x%02X reason 0x%02X\n",
			__func__, data[2], data[1]);
		return -EPROTO;
	default:
		dev_alert(dev, "%s: invalid msgid: 0x%02X\n",
				__func__, data[0]);
		return -EPROTO;
	}

	return ret;
}

/*!
 * pwm_inc_rx_task - This function will be running in a loop to receive the data
 *		     from the SCC and calls handles the message.
 *
 * @param arg	:  pointer containing driver data structure
 * @return	int 0 - success or negative on failure.
 *
 */
static int pwm_inc_rx_task(void *arg)
{
	u8 data[PWM_INC_RX_BUFSIZE];
	int ret;
	struct pwm_inc *pwm_inc = arg;

	dev_dbg(pwm_inc->dev, "%s: enter\n", __func__);

	do {
		ret = pwm_inc_recvmsg(pwm_inc, data, PWM_INC_RX_BUFSIZE);
		if (ret < 0)
			break;

		if (!pwm_inc->err_stat)
			pwm_inc->err_stat =
				pwm_inc_handle_msg(pwm_inc, data, ret);

		/* in error case wake up pending system calls */
		if (pwm_inc->err_stat && pwm_inc->is_active)
			wake_up_interruptible(&pwm_inc->waitq);
		/*clear flag*/
		pwm_inc->err_stat = 0;
	} while (1);

	dev_dbg(pwm_inc->dev, "%s: exit: %d\n", __func__, ret);

	mutex_lock(&pwm_inc->lock);
	pwm_inc->rx_task = NULL;
	mutex_unlock(&pwm_inc->lock);

	return 0;
}

/*!
 * pwm_inc_cmd_status_active - This function will send the component status
 *				command to SCC for handshake.
 *
 * @param pwm_inc	:  pointer containing driver data structure
 * @return	int written data on success or negative on failure.
 *
 */
static int pwm_inc_cmd_status_active(struct pwm_inc *pwm_inc)
{
	u8 data[2] = {SCC_PORT_EXTENDER_PWM_C_COMPONENT_STATUS,
			PWM_INC_STATUS_ACTIVE};

	return pwm_inc_sendmsg(pwm_inc, data, 2);
}

/*!
 * pwm_inc_cmd_get_channel - This function will get the channel status
 *				info from SCC.
 *
 * @param pwm_inc	:  pointer containing driver data structure
 * @return	int written data on success or negative on failure.
 *
 */
static int pwm_inc_cmd_get_channel(struct pwm_inc *pwm_inc)
{
	u8 data[1] = {SCC_PORT_EXTENDER_PWM_C_GET_CONFIG};

	return pwm_inc_sendmsg(pwm_inc, data, 1);
}

/*!
 * pwm_inc_probe - This function will initialize INC and verifies the connection
 *			to SCC and registers to PWM chip. This function will be
 *			called during insmod when there is an entry in the
 *			device tree mentioning the compatible which matches.
 *
 * @param pdev	:  pointer containing device data structure for this driver
 * @return	int 0 on success or negative on failure.
 *
 */
static int pwm_inc_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct pwm_inc *pwm_inc;
	struct task_struct *task;
	int ret;
	const struct of_device_id *id =
			of_match_device(inc_pwmchip_match, dev);

	if (!id) {
		dev_err(dev, "%s: failed to find pwm controller\n", __func__);
		ret = -EFAULT;
		goto out;
	}

	dev_info(dev, "%s: compatible: %s\n", __func__, id->compatible);

	pwm_inc = devm_kzalloc(dev, sizeof(*pwm_inc), GFP_KERNEL);
	if (!pwm_inc) {
		ret = -ENOMEM;
		goto out;
	}

	pwm_inc->dev = dev;
	pwm_inc->rx_task = NULL;
	pwm_inc->sock = NULL;
	pwm_inc->dgram = NULL;
	pwm_inc->is_active = 0;
	pwm_inc->err_stat = 0;
	pwm_inc->rsp_status_active = false;
	pwm_inc->get_config = false;

	pwm_inc->local.sin_port = htons(PORT_EXTENDER_PWM_PORT);
	if (remote_port) {
		pwm_inc->remote.sin_port = htons(remote_port);
		dev_dbg(pwm_inc->dev, "%s: set remote_port %d\n",
			__func__, remote_port);
	} else {
		pwm_inc->remote.sin_port = pwm_inc->local.sin_port;
	}

	mutex_init(&pwm_inc->lock);

	init_waitqueue_head(&pwm_inc->waitq);

	platform_set_drvdata(pdev, pwm_inc);

	ret = pwm_inc_get_dt_properties(pwm_inc);
	if (ret < 0)
		goto out;

	ret = pwm_inc_sock_open(pwm_inc);
	if (ret < 0)
		goto out_socket;

	/* create and start thread */
	task = kthread_run(pwm_inc_rx_task, pwm_inc, PWM_INC_DRV_NAME);
	if (IS_ERR(task)) {
		ret = PTR_ERR(task);
		dev_err(dev, "%s: could not run thread: %d\n", __func__, ret);
		goto out_socket;
	}
	pwm_inc->rx_task = task;

	ret = pwm_inc_cmd_status_active(pwm_inc);
	if (ret < 0)
		goto out_thread;

	ret = wait_event_interruptible_timeout(pwm_inc->waitq,
			pwm_inc->rsp_status_active,
			PWM_INC_PEER_RESPONSE_TIMEOUT);

	if (ret < 0)
		goto out_thread;
	if (ret == 0) {
		dev_alert(pwm_inc->dev,
			  "%s: peer not responding to cmd_status_active",
			  __func__);
		ret = -ETIME;
		goto out_thread;
	}

	pwm_inc->get_config = false;
	ret = pwm_inc_cmd_get_channel(pwm_inc);
	if (ret < 0)
		goto out_thread;
	ret = wait_event_interruptible_timeout(pwm_inc->waitq,
			pwm_inc->get_config,
			PWM_INC_PEER_RESPONSE_TIMEOUT);
	if (ret < 0)
		goto out_thread;
	if (ret == 0) {
		dev_alert(pwm_inc->dev,
			  "%s: peer not responding to cmd_get_channel",
			  __func__);
		ret = -ETIME;
		goto out_thread;
	}
	ret = pwm_inc_activate(pwm_inc);
	if (ret < 0)
		goto out_thread;

	return 0;

out_thread:
	mutex_lock(&pwm_inc->lock);
	if (pwm_inc->rx_task)
		force_sig(SIGKILL, pwm_inc->rx_task);
	else
		dev_dbg(dev, "%s: rx_task already terminated\n",
			__func__);
	mutex_unlock(&pwm_inc->lock);

	if (pwm_inc->rx_task)
		kthread_stop(pwm_inc->rx_task);

out_socket:
	pwm_inc_sock_close(pwm_inc);

out:
	platform_set_drvdata(pdev, NULL);
	return ret;
}

/*!
 * pwm_inc_remove - This function will deinitialize INC. This function will be
 *			called during rmmod whic detach this driver.
 *
 * @param pdev	:  pointer containing device data structure for this driver
 * @return	int 0 on success or negative on failure.
 *
 */
static int pwm_inc_remove(struct platform_device *pdev)
{
	struct pwm_inc *pwm_inc = platform_get_drvdata(pdev);
	struct device *dev = &pdev->dev;
	int ret = 0;

	if (!pwm_inc)
		goto out;

	dev_dbg(dev, "%s: base %d\n", __func__, pwm_inc->chip.base);

	if (pwm_inc->is_active) {
		ret = pwmchip_remove(&pwm_inc->chip);
		if (ret < 0) {
			dev_err(dev, "%s: could not remove pwmchip: %d\n",
				__func__, ret);
			goto out;
		}
	}
	mutex_lock(&pwm_inc->lock);
	if (pwm_inc->rx_task)
		force_sig(SIGKILL, pwm_inc->rx_task);
	else
		dev_dbg(dev, "%s: rx_task already terminated\n",
			__func__);
	mutex_unlock(&pwm_inc->lock);

	if (pwm_inc->rx_task)
		kthread_stop(pwm_inc->rx_task);

	pwm_inc_sock_close(pwm_inc);

	platform_set_drvdata(pdev, NULL);

out:
	return ret;
}

static struct platform_driver pwm_inc_driver = {
	.driver = {
		.name	= PWM_INC_DRV_NAME,
		.owner	= THIS_MODULE,
		.of_match_table = inc_pwmchip_match,
	},
	.probe		= pwm_inc_probe,
	.remove		= pwm_inc_remove,
};

static int __init pwm_inc_init(void)
{
	return platform_driver_register(&pwm_inc_driver);
}
late_initcall(pwm_inc_init);

static void __exit pwm_inc_exit(void)
{
	platform_driver_unregister(&pwm_inc_driver);
}
module_exit(pwm_inc_exit);

MODULE_DESCRIPTION("PWM port extender for INC");
MODULE_LICENSE("GPL");
MODULE_AUTHOR("Suresh Dhandapani <suresh.dhandapani@in.bosch.com>");
