/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#ifndef CSR_WIFI_PRIVATE_COMMON_H__
#define CSR_WIFI_PRIVATE_COMMON_H__

#include "csr_synergy.h"

#include "csr_types.h"
#include "csr_wifi_common.h"

#ifdef __cplusplus
extern "C" {
#endif

/** Maximum number of STAs allowed to be connected */
#define CSR_WIFI_AP_MAX_ASSOC_STA                8

/* The maximum allowed length of SSID */
#define CSR_WIFI_SME_AP_SSID_MAX_LENGTH         32

/* Refer 8.4.2.27 RSN element - we support TKIP, WPA2, WAPI and PSK only, no pmkid, group cipher suite */
#define CSR_WIFI_SME_RSN_PACKED_SIZE (1 + 1 + 2 + 4 + 2 + 4 * 2 + 2 + 4 * 1 + 2 + 24)

/* Common structure for NME and SME to maintain Interface mode*/
typedef CsrUint8 CsrWifiInterfaceMode;
#define  CSR_WIFI_MODE_NONE                             ((CsrWifiInterfaceMode) 0xFF)
#define  CSR_WIFI_MODE_STA                              ((CsrWifiInterfaceMode) 0x00)
#define  CSR_WIFI_MODE_AP                               ((CsrWifiInterfaceMode) 0x01)
#define  CSR_WIFI_MODE_P2P_DEVICE                       ((CsrWifiInterfaceMode) 0x02)
#define  CSR_WIFI_MODE_P2P_CLI                          ((CsrWifiInterfaceMode) 0x03)
#define  CSR_WIFI_MODE_P2P_GO                           ((CsrWifiInterfaceMode) 0x04)
#define  CSR_WIFI_MODE_AMP                              ((CsrWifiInterfaceMode) 0x05)
#define  CSR_WIFI_MODE_WPS_ENROLLEE                     ((CsrWifiInterfaceMode) 0x06)
#define  CSR_WIFI_MODE_IBSS                             ((CsrWifiInterfaceMode) 0x07)

#define  CSR_WIFI_MODE_COUNT_MAX                        ((CsrUint8) 0x08)

/* MAC address restrictions for a particular activity encoded in the MIB */
typedef CsrUint8 CsrWifiSmeMibMacAddrCondition;
#define  CSR_WIFI_SME_MIB_MAC_ADDR_CONDITION_NONE        ((CsrWifiSmeMibMacAddrCondition) 0x00)
#define  CSR_WIFI_SME_MIB_MAC_ADDR_CONDITION_UNIVERSAL   ((CsrWifiSmeMibMacAddrCondition) 0x01)
#define  CSR_WIFI_SME_MIB_MAC_ADDR_CONDITION_LOCAL       ((CsrWifiSmeMibMacAddrCondition) 0x02)
#define  CSR_WIFI_SME_MIB_MAC_ADDR_CONDITION_ANY         ((CsrWifiSmeMibMacAddrCondition) 0x03)

/**
 * @brief
 *   Encode error values into the MIB
 *
 * @description
 *   Take an encoded set of MIB values and return a set of error encoded MIB values
 *
 * @param[in] CsrWifiDataBlock : Encoded set of MIB values
 * @param[in] CsrResult        : Error to be encoded
 *
 * @return
 *   CsrWifiDataBlock          : Error encoded MIB values
 */
CsrWifiDataBlock CsrWifiPrivateMibConvertToErrors(CsrWifiDataBlock mibSetData, CsrResult status);

/**
 * @brief
 *     Validate the MAC address type in the request for a new activity (interface mode)
 *
 * @description
 *      Validate a request for a new activity (interface mode) by comparing
 *      'the type of MAC address (Local/Universal)' that is requested against the
 *      'supported mode' information from the MIB.
 *
 * @param[in] CsrWifiInterfaceMode : The new interface mode being requested
 * @param[in] CsrBool              : Is the new mode requested on a Local/Universal MAC address
 * @param[in] CsrWifiDataBlock     : Encoded information on supported mode combinations from the MIB
 *
 * @return
 *   CsrBool                       : TRUE - New activity is supported
 *                                 : FALSE   - New activity isn't supported due to MAC address restrictions
 */
CsrBool CsrWifiPrivateIsMacAddrValid(CsrWifiInterfaceMode newInterfaceMode,
                                     CsrBool              isLocalMACAddr,
                                     CsrWifiDataBlock     mibSupportedModeCombinations);

/**
 * @brief
 *     Validate the request for a new activity (interface mode) based on the currently active mode combinations
 *
 * @description
 *      Validate a request for a new activity (interface mode) by comparing
 *      'the existing concurrent mode combination' against the 'supported mode'
 *      information from the MIB.
 *
 * @param[in] CsrWifiInterfaceMode : The new interface mode being requested
 * @param[in] CsrBool              : Currently active concurrent activities
 * @param[in] CsrWifiDataBlock     : Encoded information on supported mode combinations from the MIB
 *
 * @return
 *   CsrBool                       : TRUE   - New activity is supported
 *                                 : FALSE  - New activity isn't supported due to the current mode combinations
 */
CsrBool CsrWifiPrivateIsModeCombinationValid(CsrWifiInterfaceMode newInterfaceMode,
                                             CsrUint32            currentModeCombination,
                                             CsrWifiDataBlock     mibSupportedModeCombinations);

/**
 * @brief
 *     Validate the encoded value for supported mode combinations in the MIB
 *
 * @param[in] CsrWifiDataBlock     : Encoded information on supported mode combinations from the MIB
 *
 * @return
 *   CsrBool                       : TRUE   - If all the fields are valid
 *                                 : FALSE  - If there are any errors in the encoding
 */
CsrBool CsrWifiPrivateIsModeCombinationMibEntryValid(CsrWifiDataBlock mibSupportedModeCombinations);


#ifdef __cplusplus
}
#endif

#endif
