/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

/*
 * ---------------------------------------------------------------------------
 *
 * FILE : csr_wifi_hip_defs.h
 *
 * PURPOSE : Public API for the shared UniFi HIP and HAL structures.
 *
 * ---------------------------------------------------------------------------
 */
#ifndef CSR_WIFI_HIP_DEFS_H__
#define CSR_WIFI_HIP_DEFS_H__

#include "csr_synergy.h"

#ifdef __cplusplus
extern "C" {
#endif

#define CSR_WIFI_HIP_80211_MAC_HEADER_MAX_LEN 36 /* including 4 addresses, HT field and QOS field */
#define CSR_WIFI_HIP_LLC_HEADER_LEN 8 /* dest SAP (1), source SAP (1), commmand (1), OUI(3), protocol (2) */

#define CSR_WIFI_HIP_MAX_DATA_REFERENCES 2

typedef struct
{
    CsrUint8   *os_data_ptr;
    CsrUint32   data_length;
    const void *os_net_buf_ptr;
    CsrUint32   net_buf_length;
} CsrWifiHipBulkDataDesc;


typedef struct
{
    CsrWifiHipBulkDataDesc d[CSR_WIFI_HIP_MAX_DATA_REFERENCES];
} CsrWifiHipBulkDataParam;

#define CSR_WIFI_HIP_PARAM_SLOT_COUNT 2

/*
 * Association IDs can be in the range 16..31. This is a value allocated
 * by the chip. An AID of zero means the VIF is in station mode.
 */
#define CSR_WIFI_HIP_AID_START 16
#define CSR_WIFI_HIP_AID_END 31

/*
 * The chip supports 8 associated stations when in AP mode.
 */
#define CSR_WIFI_HIP_PEER_CONNECTIONS_MAX 8

/****************************** Configuration ********************************/

typedef struct
{
    CsrUint32 zero_copy;
    CsrUint32 traffic_qlen;
    CsrUint32 th_max_slots;
    CsrUint32 fh_max_slots;
    CsrUint32 led_mask;
    CsrUint32 poll_period;
    CsrUint32 tx_window_segment_size;
    CsrUint32 tx_force_pull_mode;
    CsrInt32  mparam_slot_count[CSR_WIFI_HIP_PARAM_SLOT_COUNT];
    CsrUint32 minVifDurationEapol;
    CsrUint32 minVifDurationDhcp;
    CsrUint32 minVifDurationArp;
    CsrUint16 minVifDurationPacketFilter;
} CsrWifiHipCardParams;

typedef CsrUint8 CsrWifiHipMode;
#define CSR_WIFI_HIP_WIFI_STACK_MODE    ((CsrWifiHipMode) 0x0) /* Normal Wi-Fi stack */
#define CSR_WIFI_HIP_RAW_SDIO_MODE      ((CsrWifiHipMode) 0x1) /* Ptest */

typedef CsrResult (*CsrWifiHipBulkDataAlloc)(void *osLayerContext, CsrWifiHipBulkDataDesc *bulkData, CsrUint32 size);
typedef void (*CsrWifiHipBulkDataFree)(void *osLayerContext, CsrWifiHipBulkDataDesc *bulkData);


typedef CsrUint8 CsrWifiHipTrafficQueue;
#define CSR_WIFI_HIP_TRAFFIC_QUEUE_BK    ((CsrWifiHipTrafficQueue) 0x0)
#define CSR_WIFI_HIP_TRAFFIC_QUEUE_BE    ((CsrWifiHipTrafficQueue) 0x1)
#define CSR_WIFI_HIP_TRAFFIC_QUEUE_VI    ((CsrWifiHipTrafficQueue) 0x2)
#define CSR_WIFI_HIP_TRAFFIC_QUEUE_VO    ((CsrWifiHipTrafficQueue) 0x3)

typedef void (*CsrWifiHipFlowControlResumeCb)(void *osLayerContext, CsrUint16 interfaceTag,
                                              CsrUint16 aid, CsrWifiHipTrafficQueue queue);

typedef void (*CsrWifiHipFlowControlPauseCb)(void *osLayerContext, CsrUint16 interfaceTag,
                                             CsrUint16 aid, CsrWifiHipTrafficQueue queue);


typedef struct
{
    CsrWifiHipMode                hipMode;
    CsrWifiHipBulkDataAlloc       allocFunc;
    CsrWifiHipBulkDataFree        freeFunc;
    CsrWifiHipFlowControlPauseCb  flowControlPauseCb;
    CsrWifiHipFlowControlResumeCb flowControlResumeCb;
    CsrInt16                      fw_init;
    CsrWifiHipCardParams          cardParams;
    CsrBool                       cmanrTestMode;
} CsrWifiHipInstanceConfig;


/***************************** Stats and debug *******************************/
#define CSR_WIFI_HIP_LAST_DEBUG_STRING_LENGTH 256
#define CSR_WIFI_HIP_LAST_DEBUG_WORD16_LENGTH 16

typedef struct
{
    CsrCharString last_debug_string[CSR_WIFI_HIP_LAST_DEBUG_STRING_LENGTH];
    CsrUint16     last_debug_word16[CSR_WIFI_HIP_LAST_DEBUG_WORD16_LENGTH];
} CsrWifiHipDriverInfoDebugInfo;

typedef struct
{
    CsrUint32 hip2_stats_rx_interrupts;
    CsrUint32 hip2_stats_rx_spurious_interrupts;
    CsrUint32 hip2_stats_rx_wake_only_interrupts;
    CsrUint32 hip2_stats_rx_wake_with_data_interrupts;
    CsrUint32 hip2_stats_rx_last_pkts_per_cycle;
    CsrUint32 hip2_stats_rx_last_pkts_per_cycle_clrs;
    CsrUint32 hip2_stats_tx_last_more_req_magic;
    CsrUint32 hip2_stats_rx_pkts_per_cycle[10];
    CsrUint32 hip2_stats_rx_pkts_per_cycle_clrs[10];
    CsrUint32 hip2_stats_rx_signals;
    CsrUint32 hip2_stats_rx_unknown_signals;
    CsrUint32 hip2_stats_rx_unsupported_signals;
    CsrUint32 hip2_stats_rx_ctrl_bytes;
    CsrUint32 hip2_stats_rx_total_bytes;
    CsrUint32 hip2_stats_rx_data_bytes;
    CsrUint32 hip2_stats_rx_ctrl_frames;
    CsrUint32 hip2_stats_rx_ctrl_frames_missed;
    CsrUint32 hip2_stats_rx_ctrl_frames_invalid;
    CsrUint32 hip2_stats_rx_ctrl_frames_value_error;
    CsrUint32 hip2_stats_rx_ctrl_frames_read_error;
    CsrUint32 hip2_stats_rx_last_ack_magic;
    CsrUint32 hip2_stats_rx_last_implicit_ack_magic;
    CsrUint32 hip2_stats_rx_last_nack_magic;
    CsrUint32 hip2_stats_rx_mac_discards;
    CsrUint32 hip2_stats_rx_window_efficiency;

    CsrUint32 hip2_stats_tx_ctrl_bytes;
    CsrUint32 hip2_stats_tx_data_bytes;
    CsrUint32 hip2_stats_tx_pad_bytes;
    CsrUint32 hip2_stats_rx_cycles;
    CsrUint32 hip2_stats_rx_cycles_clrs_only;
    CsrUint32 hip2_stats_rx_cycles_pkts;
    CsrUint32 hip2_stats_tx_cycles;
    CsrUint32 hip2_stats_tx_cycles_starved;
    CsrUint32 hip2_stats_tx_q_pkts[5];
    CsrUint32 hip2_stats_tx_pkts_per_cycle[10];
    CsrUint32 hip2_stats_tx_bulk_data_pkts;
    CsrUint32 hip2_stats_tx_bulk_data_clrs;
    CsrUint32 hip2_stats_tx_slot_congested;
    CsrUint32 hip2_stats_tx_window_starved;
    CsrUint32 hip2_stats_tx_window_congested;
    CsrUint32 hip2_stats_tx_q_pause[4];
    CsrUint32 hip2_stats_bh_sched[2];
    CsrUint32 hip2_stats_tx_window_req[12];
    CsrUint32 hip2_stats_tx_window_rec[12];
    CsrUint32 hip2_stats_tx_window_efficiency;
    CsrUint32 hip2_stats_tx_window_deferred;
    CsrUint32 hip2_stats_tx_segs_per_cycle[5];

    CsrUint32 hip2_stats_tx_realignment;

    CsrUint32 hip2_stats_tx_pull_bulk_data_pkts;
    CsrUint32 hip2_stats_tx_pull_bulk_data_bytes;

    CsrUint32 hip2_sdio_cmd53_errors[9];

    CsrUint32 hip2_stats_tx_realloc;

    CsrUint32 hip2_stats_tx_pkt_cfm;
    CsrUint32 hip2_stats_state_torpid;

    CsrUint16 fwin;
    CsrUint16 ack;
    CsrUint16 cread;
    CsrUint16 fh_pipe_limit;
    CsrUint16 fh_pipe_current;
    CsrInt32  cmd;
    CsrInt32  t0;
    CsrInt32  t1;
    CsrInt32  t2;
    CsrInt32  t3;
    CsrUint32 ctrl;
    CsrUint32 data;
    CsrInt32  cmd_q;
    CsrInt32  t0_q;
    CsrInt32  t1_q;
    CsrInt32  t2_q;
    CsrInt32  t3_q;
    CsrUint32 ctrl_data;
    CsrUint32 bulk_data;
    CsrUint16 free_slots;
    CsrInt32  packets_pending;
    CsrInt32  pm;

    CsrUint16 ctrl_buffer_h;
    CsrUint16 ctrl_re_read_buffer_h;
    CsrUint16 ack_buffer_h;
    CsrUint16 fh_pushed_buffer_h[6];
} CsrWifiHipHip2Stats;

typedef CsrUint32 CsrWifiHipLogUdiDirection;
#define CSR_WIFI_HIP_LOG_UDI_DIRECTION_FROM_HOST ((CsrWifiHipLogUdiDirection) 0x0)
#define CSR_WIFI_HIP_LOG_UDI_DIRECTION_TO_HOST   ((CsrWifiHipLogUdiDirection) 0x1)

#ifdef __cplusplus
}
#endif

#endif /* CSR_WIFI_HIP_DEFS_H__ */
