/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#ifndef CSR_WIFI_HIP_H__
#define CSR_WIFI_HIP_H__

#include "csr_synergy.h"

#include "csr_sched.h"
#include "csr_result.h"
#include "csr_util.h"
#include "csr_types.h"
#include "csr_wifi_common.h"
#include "csr_wifi_router_ctrl_prim.h"
#include "csr_wifi_hip_defs.h"

#ifdef __cplusplus
extern "C" {
#endif


/******************************* Result codes *******************************/
#define CSR_WIFI_HIP_CANCEL            ((CsrResult) 0x1)
#define CSR_WIFI_HIP_TIMEOUT           ((CsrResult) 0x2)
#define CSR_WIFI_HIP_TX_PAUSE          ((CsrResult) 0x3)
#define CSR_WIFI_HIP_CP_DROPPED_PACKET ((CsrResult) 0x4)


/**** Host tag for MA-PACKET.request ***/
#define CSR_WIFI_HIP_HOST_TAG_INVALID 0xFFFFFFFF


/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipActivateReq
 *
 *      Activates HIP. The function driver will be registered in the SDIO
 *      driver.
 *
 *  Arguments:
 *      None.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS if HIP was activated sucessfully. Otherwise
 *      CSR_RESULT_FAILURE.
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipActivateReq(void);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipDeactivateReq
 *
 *      Deactivates HIP, which include unregistering the function driver in
 *      the SDIO driver.
 *
 *  Arguments:
 *      None.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS indicates that the deactivation was successful and
 *      CSR_RESULT_FAILURE otherwise.
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipDeactivateReq(void);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipInsertedInd
 *
 *      Indicates that a new card instance has started. HIP is ready to
 *      receive a CsrWifiHipWifiOnReq if the the result indicates success.
 *      The hipHandle shall be used in all subsequent calls to HAL.
 *
 *  Arguments:
 *      hipHandle       Handle to HIP. This handle must be used in all
 *                      subsequent calls.
 *
 *  Returns:
 *      None.
 *
 * ---------------------------------------------------------------------------
 */
void CsrWifiHipInsertedInd(void *hipHandle);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipInsertedRes
 *
 *      Indicates that a new card instance has started. HIP is ready to
 *      receive a CsrWifiHipWifiOnReq if the the result indicates success.
 *      The hipHandle shall be used in all subsequent calls to HAL.
 *
 *  Arguments:
 *      hipHandle       Handle to HIP/HAL instance.
 *      osLayerContext  Pointer to the os layer context, which shall be stored
 *                      in the HAL instance data. In case of an error, the
 *                      osLayerContext will be NULL.
 *      result          Result of the os layer handling of the
 *                      CsrWifiHipInsertedInd.
 *      instance        The instance number is used to identify the card
 *                      instance. Platform specific code must ensure that a
 *                      unique number is provided so that each instance can
 *                      be identified.
 *
 *  Returns:
 *      None.
 *
 * ---------------------------------------------------------------------------
 */
void CsrWifiHipInsertedRes(void *hipHandle, void *osLayerContext, CsrResult result, CsrUint32 instance);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipRemovedInd
 *
 *      Indicates that a card instance has stopped/removed.
 *
 *  Arguments:
 *      osLayerContext  Pointer to os context received in the
 *                      CsrWifiHipInsertedRes.
 *      instance        identifier for this card instance
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS if HIP was activated sucessfully. Otherwise
 *      CSR_RESULT_FAILURE.
 *
 * ---------------------------------------------------------------------------
 */
void CsrWifiHipRemovedInd(void *osLayerContext, CsrUint32 instance);


/******************************* Memory alloc *******************************/

typedef CsrUint16 CsrWifiHipQosType;
#define CSR_WIFI_HIP_QOS_TYPE_NONE        ((CsrWifiHipQosType) 0x0000)
#define CSR_WIFI_HIP_QOS_TYPE_QOS_CONTROL ((CsrWifiHipQosType) 0x0001)

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipFrameBufferSizeReq
 *
 *      Calculates the required offset, which the bulkdata in the
 *      CsrWifiHipMaPacketReq must be offset.
 *
 *  Arguments:
 *      hipHandle       Pointer to the HIP context.
 *      interfaceTag    Virtual interface tag.
 *      qosType         Quality Of Service type (CsrWifiHipQosType)
 *      offset          Pointer to variable, which will hold the when the
 *                      function returns.
 *
 *  Returns:
 *      None
 *
 * ---------------------------------------------------------------------------
 */
void CsrWifiHipFrameBufferSizeReq(void             *hipHandle,
                                  CsrUint16         interfaceTag,
                                  CsrWifiHipQosType qosType,
                                  CsrUint8         *offset);

/**************************** Header definitions ****************************/

#define CSR_WIFI_HIP_OUI_LEN         3

typedef struct
{
    CsrUint8 dsap;                      /* always 0xAA */
    CsrUint8 ssap;                      /* always 0xAA */
    CsrUint8 ctrl;                      /* always 0x03 */
    CsrUint8 oui[CSR_WIFI_HIP_OUI_LEN]; /* organizational universal id */
} CsrWifiHipLlcSnapHdr;

typedef struct
{
    CsrWifiHipLlcSnapHdr llcSnap;
    CsrUint8             protocol[2];
} CsrWifiHipLlcSnapHeaderFrame;

#define CSR_WIFI_LLC_SNAP_HDR_PROT_OFFSET   6
#define CSR_WIFI_LLC_SNAP_HDR_LEN           8

/******************* Initialisation and de-initialisation *******************/
/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipWifiOffReq
 *
 *      Turn Wi-Fi off.
 *
 *  Arguments:
 *      blockingTerminate Blocking termination of HIP. No CsrWifiHipWifiOffCfm
 *                      will be emitted.
 *      hipHandle       Pointer to the HIP context.
 *
 *  Returns:
 *      None
 *
 * ---------------------------------------------------------------------------
 */
void CsrWifiHipWifiOffReq(CsrBool blockingTerminate, void *hipHandle);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipWifiOffCfm
 *
 *      Turn Wi-Fi off.
 *
 *  Arguments:
 *      osLayerContext  Pointer to the Synergy context.
 *      result          Result of the CsrWifiHipWifiOffReq.
 *                      CSR_RESULT_SUCCESS if the WifiOffReq succeeded
 *                      otherwise CSR_RESULT_FAILURE.
 *
 *  Returns:
 *      None
 *
 * ---------------------------------------------------------------------------
 */
void CsrWifiHipWifiOffCfm(void *osLayerContext, CsrResult result);


/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipWifiOnReq
 *
 *      Turn Wi-Fi on.
 *
 *  Arguments:
 *      hipHandle       Pointer to the HIP context.
 *      osLayerContext  Pointer to the osLayerContext
 *      smeAppHandle    App handle to the SME
 *      config          Configuration structure holding various HAL
 *                      configuration parameters.
 *
 *  Returns:
 *      None
 *
 * ---------------------------------------------------------------------------
 */

CsrResult CsrWifiHipWifiOnReq(void                    *hipHandle,
                              void                    *osLayerContext,
                              CsrSchedQid              smeAppHandle,
                              CsrWifiHipInstanceConfig config);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipWifiOffInd
 *
 *      Indicates that than unexpected event happened.
 *
 *  Arguments:
 *      osLayerContext  Pointer to the osLayerContext
 *      controlIndication Indicates the unexpected event
 *
 *  Returns:
 *      None
 *
 * ---------------------------------------------------------------------------
 */
typedef CsrUint8 CsrWifiHipControlIndication;
#define CSR_WIFI_HIP_CONTROL_ERROR    ((CsrWifiHipControlIndication) 0x1)
#define CSR_WIFI_HIP_CONTROL_EXIT     ((CsrWifiHipControlIndication) 0x2)

void CsrWifiHipWifiOffInd(void                       *osLayerContext,
                          CsrWifiHipControlIndication controlIndication);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipWifiOnInd
 *
 *      Indicates that firmware .
 *
 *  Arguments:
 *      osLayerContext  Pointer to the osLayerContext
 *      result          Result of the CsrWifiHipWifiOnReq
 *      chipId
 *      chipVersion
 *      firmwareBuild   The firmware build number
 *      firmwareHip
 *      routerBuild
 *      routerHip
 *      sdioBlockSize   The SDIO block size used.
 *
 *  Returns:
 *      None
 *
 * ---------------------------------------------------------------------------
 */
void CsrWifiHipWifiOnInd(void          *osLayerContext,
                         CsrResult      result,
                         CsrUint32      chipId,
                         CsrUint32      chipVersion,
                         CsrUint32      firmwareBuild,
                         CsrUint32      firmwareHip,
                         CsrCharString *routerBuild,
                         CsrUint32      routerHip,
                         CsrUint32      sdioBlockSize);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipWifiOnCfm
 *
 *      Indicates if the CsrWifiHipWifiOnReq was successful.
 *
 *  Arguments:
 *      osLayerContext  Pointer to the osLayerContext
 *      result          Result of the CsrWifiHipWifiOnCfm
 *
 *  Returns:
 *      None
 *
 * ---------------------------------------------------------------------------
 */
void CsrWifiHipWifiOnCfm(void *osLayerContext, CsrResult result);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipStartCoreDumpReq
 *
 *      Starts a coredump.
 *
 *  Arguments:
 *      hipHandle       Pointer to the HIP context
 *
 *  Returns:
 *      Result of the start core dump
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipStartCoreDumpReq(void *hipHandle);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipStopCoreDumpReq
 *
 *      Stops a coredump.
 *
 *  Arguments:
 *      hipHandle       Pointer to the HIP context
 *
 *  Returns:
 *      None
 *
 * ---------------------------------------------------------------------------
 */
void CsrWifiHipStopCoreDumpReq(void *hipHandle);


/******************************* Configuration ******************************/
/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPostInitReq
 *
 *      Perform the last part of the initialisation.
 *
 *  Arguments:
 *      hipHandle       Pointer to the HIP context
 *      scheduledInterrupt Indicates if scheduledInterrupts will be used
 *
 *  Returns:
 *      None
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPostInitReq(void *hipHandle, CsrBool scheduledInterrupt);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipSetMacReq
 *
 *      Sets the MAC addresses for the virtual interfaces.
 *
 *  Arguments:
 *      hipHandle       Pointer to the HIP context
 *      macAddresses    MAC addresses to be stored in the driver
 *      numOfInterfacAddress The number of MAC addresses to store
 *
 *  Returns:
 *      None
 *
 * ---------------------------------------------------------------------------
 */
void CsrWifiHipSetMacReq(void              *hipHandle,
                         CsrWifiMacAddress *macAddresses,
                         CsrUint16          numOfInterfaceAddress);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipVifModeSetReq
 *
 *      Sets the mode, MAC address and protection information for a
 *      virtual interface.
 *
 *  Arguments:
 *      hipHandle       Pointer to the HIP context
 *      interfaceTag    Virtual interface id, which will be updated
 *      mode            Holds the mode, which will be installed
 *      macAddress      The MAC address, which will be installed for the
 *                      interface
 *      protect         Holds the protection information for the interface
 *      intraBssDistEnabled Indicates if intra BSS is enabled
 *      vifIndex        Virtual interface index
 *
 *  Returns:
 *      None
 *
 * ---------------------------------------------------------------------------
 */
void CsrWifiHipVifModeSetReq(void                 *hipHandle,
                             CsrUint16             interfaceTag,
                             CsrWifiRouterCtrlMode mode,
                             CsrWifiMacAddress     macAddress,
                             CsrBool               protect,
                             CsrBool               intraBssDistEnabled,
                             CsrUint16             vifIndex);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipQosControlReq
 *
 *      Sets the Quality of Service settings in HAL for a virtual interface.
 *
 *  Arguments:
 *      hipHandle       Pointer to the HIP context
 *      interfaceTag    Virtual interface id, which will be updated
 *      mode            Holds the QoS mode, which will be installed
 *      queueConfig     Mask indicates which queues are enabled
 *
 *  Returns:
 *      None
 *
 * ---------------------------------------------------------------------------
 */
void CsrWifiHipQosControlReq(void                            *hipHandle,
                             CsrUint16                        interfaceTag,
                             CsrWifiRouterCtrlQoSControl      mode,
                             CsrWifiRouterCtrlQueueConfigMask queueConfig,
                             CsrWifiRouterCtrlQueueSettings   queueConfigs[]);

/****************************** Flow control ********************************/
/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipFlowControlPauseAllInd
 *
 *      Pauses all interfaces on a card.
 *
 *  Arguments:
 *      osLayerContext  Pointer to the osLayerContext
 *
 *  Returns:
 *      None
 *
 * ---------------------------------------------------------------------------
 */
void CsrWifiHipFlowControlPauseAllInd(void *osLayerContext);

/************************************ TX ************************************/

typedef CsrUint16 CsrWifiHipMaPacketPriority;
#define CSR_WIFI_HIP_MA_PACKET_PRIORITY_QOS_UP0    ((CsrWifiHipMaPacketPriority) 0x0000)
#define CSR_WIFI_HIP_MA_PACKET_PRIORITY_QOS_UP1    ((CsrWifiHipMaPacketPriority) 0x0001)
#define CSR_WIFI_HIP_MA_PACKET_PRIORITY_QOS_UP2    ((CsrWifiHipMaPacketPriority) 0x0002)
#define CSR_WIFI_HIP_MA_PACKET_PRIORITY_QOS_UP3    ((CsrWifiHipMaPacketPriority) 0x0003)
#define CSR_WIFI_HIP_MA_PACKET_PRIORITY_QOS_UP4    ((CsrWifiHipMaPacketPriority) 0x0004)
#define CSR_WIFI_HIP_MA_PACKET_PRIORITY_QOS_UP5    ((CsrWifiHipMaPacketPriority) 0x0005)
#define CSR_WIFI_HIP_MA_PACKET_PRIORITY_QOS_UP6    ((CsrWifiHipMaPacketPriority) 0x0006)
#define CSR_WIFI_HIP_MA_PACKET_PRIORITY_QOS_UP7    ((CsrWifiHipMaPacketPriority) 0x0007)
#define CSR_WIFI_HIP_MA_PACKET_PRIORITY_CONTENTION ((CsrWifiHipMaPacketPriority) 0x8000)
#define CSR_WIFI_HIP_MA_PACKET_PRIORITY_MANAGEMENT ((CsrWifiHipMaPacketPriority) 0x8010)
#define CSR_WIFI_HIP_MA_PACKET_PRIORITY_PROTECTED_MANAGEMENT ((CsrWifiHipMaPacketPriority) 0x8011)

#define CSR_WIFI_HIP_QOS_TYPE(priority) (priority & CSR_WIFI_HIP_MA_PACKET_PRIORITY_CONTENTION) ? CSR_WIFI_HIP_QOS_TYPE_NONE : CSR_WIFI_HIP_QOS_TYPE_QOS_CONTROL;

typedef CsrUint8 CsrWifiHipPortType;
#define CSR_WIFI_HIP_PORT_TYPE_UNCONTROLLED ((CsrWifiHipPortType) 0)
#define CSR_WIFI_HIP_PORT_TYPE_CONTROLLED   ((CsrWifiHipPortType) 1)

typedef CsrUint8 CsrWifiHipFrameType;
#define CSR_WIFI_HIP_FRAMETYPE_IEEE80211_MANAGEMENT ((CsrWifiHipFrameType) 0X00)
#define CSR_WIFI_HIP_FRAMETYPE_IEEE80211_CONTROL    ((CsrWifiHipFrameType) 0X01)
#define CSR_WIFI_HIP_FRAMETYPE_IEEE80211_DATA       ((CsrWifiHipFrameType) 0X02)
#define CSR_WIFI_HIP_FRAMETYPE_IEEE80211_RESERVED   ((CsrWifiHipFrameType) 0X03)


typedef CsrUint8 CsrWifiHipPacketType;
#define CSR_WIFI_HIP_UNICAST_PDU   ((CsrWifiHipPacketType) 0x0)
#define CSR_WIFI_HIP_MULTICAST_PDU ((CsrWifiHipPacketType) 0x1)
#define CSR_WIFI_HIP_BROADCAST_PDU ((CsrWifiHipPacketType) 0x2)
#define CSR_WIFI_HIP_PKT_TYPE_RESERVED ((CsrWifiHipPacketType) 0xF)
/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPackedSignalReq
 *
 *      Transmits a packed signal and bulkdata if included.
 *
 *  Arguments:
 *      hipHandle       Pointer to the hipHandle
 *      sigLength       Length of the signal
 *      sigData         Pointer to the packed signal buffer. The signal must
 *                      be allocated using CsrMemAlloc. The memory owernership
 *                      is handed over to HAL.
 *      bulkdata        Pointer to bulkdata. The memory ownership is handed
 *                      over to HAL.
 *
 *  Returns:
 *      Returns the result of the request to transmit the packed signal
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPackedSignalReq(void                    *hipHandle,
                                    CsrUint16                interfaceTag,
                                    CsrUint16                sigLength,
                                    CsrUint8                *sigData,
                                    CsrWifiHipBulkDataParam *bulkdata);

CsrResult CsrWifiHipUnpackedSignalReq(void                    *hipHandle,
                                      CsrUint16                interfaceTag,
                                      CsrUint32                associationId,
                                      CsrUint16                sigLength,
                                      CsrUint8                *sigData,
                                      CsrWifiHipBulkDataParam *bulkdata);

#define CSR_WIFI_HIP_M4_MESSAGE 0x80000000

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipMaPacketReq
 *
 *      Transmits a unpacked frame.
 *
 *  Arguments:
 *      hipHandle       Pointer to the hipHandle
 *      dMac            Pointer to the destionation MAC address. The memory
 *                      owernership is not handed over to HAL.
 *      sMac            Pointer to the source MAC address.The memory
 *                      owernership is not handed over to HAL.
 *      senderProcessId The CsrWifiHipMaPacketCfm will return the
 *                      senderProcessId provided in the CsrWifiHipMaPacketReq
 *      interfacetag    Virtual interface tag
 *      frameType       Specify the frame type being transmitted
 *      protocol        Specify the protocol for the data. If the caller is
 *                      not aware of protocol it can pass a value of 0.
 *      bulkdata        Pointer to bulkdata. The bulkdata must be allocated
 *                      using CsrWifiHipFrameBufferSizeReq function. The memory
 *                      ownership is handed over to HAL.
 *      priority        The requested priority the frame should be
 *                      transmitted at.
 *      transmitRate    The requested rate the frame at which the frame will
 *                      be transmitted at
 *      hostTag         Tag, which uniquely identifies the Ma frame. The
 *                      router API outlines the usage of the hostTag.
 *      requestCfm      Indicates if a MaPacketReq shall cause a MaPacketCfm
 *      allocedMacHdrMem The number of bytes that has been pre-allocated for
 *                      the MAC header. This number is returned in
 *                      CsrWifiHipFrameBufferSizeReq as offset.
 *      vifIndex        virtualInterfaceIndex associated with interfaceTag & used
 *                      for corresponding interface activity with firmware
 *  Returns:
 *      Returns the result of the request to transmit the unpakced frame
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipMaPacketReq(void                      *hipHandle,
                                CsrUint8                  *dMac,
                                CsrUint8                  *sMac,
                                CsrUint16                  senderProcessId,
                                CsrUint16                  interfaceTag,
                                CsrWifiHipFrameType        frameType,
                                CsrUint16                  protocol,
                                CsrWifiHipBulkDataParam   *bulkdata,
                                CsrWifiHipMaPacketPriority priority,
                                CsrUint16                  transmitRate,
                                CsrUint32                  hostTag,
                                CsrBool                    requestCfm,
                                CsrUint8                   allocedMacHdrMem,
                                CsrUint16                  vifIndex,
                                CsrUint16                  minVifDuration);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPriorityGetReq
 *
 *      Find the priority for the frame provided.
 *
 *  Arguments:
 *      hipHandle       Pointer to the hipHandle
 *      interfaceTag    Virtual interface tag
 *      protocol        The protocol type, i.e. IPv4: 0x0800
 *      frame           Pointer to the frame. If frame is NULL only the
 *                      qosType will be returned. The memory ownership is
 *                      not handed over to HAL.
 *      dMac            The destination MAC address. The memory ownership is
 *                      not handed over to HAL.
 *      priority        Pointer where the priority will be written. If the
 *                      result code is not success, the priority shall be
 *                      disregarded.
 *      aidp            Pointer for Association ID value.
 *
 *  Returns:
 *      Returns the result of the CsrWifiHipPriorityGetReq
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPriorityGetReq(void                       *hipHandle,
                                   CsrUint16                   interfaceTag,
                                   CsrUint16                   protocol,
                                   CsrUint8                   *frame,
                                   CsrUint8                   *dMac,
                                   CsrWifiHipMaPacketPriority *priority,
                                   CsrInt32                   *aidp);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipMaPacketCancelReq
 *
 *      Request to cancel a MaPacketReq. The MaPacketReq may already have
 *      been send, so there is no guarantees that the frame is cancelled.
 *
 *  Arguments:
 *      hipHandle       Pointer to the hipHandle
 *      senderProcessId The senderProcessId.
 *      interfaceTag    Virtual interface tag
 *      hostTag         Tag, which uniquely identifies the Ma frame. The
 *                      router API outlines the usage of the hostTag.
 *
 *  Returns:
 *      Returns the result of the CsrWifiHipQosTypeAndPriorityGet
 *
 * ---------------------------------------------------------------------------
 */
void CsrWifiHipMaPacketCancelReq(void     *hipHandle,
                                 CsrUint16 senderProcessId,
                                 CsrUint16 interfaceTag,
                                 CsrUint32 hostTag);

/************************************ RX ************************************/
/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPackedSignalInd
 *
 *      Indicates that a packed MaPacket frame has been received (i.e. HAL
 *      hasn't unpacked the frame).
 *
 *  Arguments:
 *      osLayerContext  Pointer to the os layer context
 *      sigLength       Length of the packed signal
 *      sigData         Pointer to the packed signal. HAL remains the owner
 *                      of the memory.
 *      bulkdata        Pointer to bulkdata. The memory ownership is
 *                      handed over to the os layer.
 *
 *  Returns:
 *      None
 *
 * ---------------------------------------------------------------------------
 */
void CsrWifiHipPackedSignalInd(void                    *osLayerContext,
                               CsrUint16                sigLength,
                               CsrUint8                *sigData,
                               CsrWifiHipBulkDataParam *bulkdata);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipMaPacketInd
 *
 *      Indicates that a unpacked MaPacket frame has been received (i.e. HAL
 *      has unpacked the frame).
 *
 *  Arguments:
 *      osLayerContext  Pointer to the os layer context
 *      interfaceTag    Virtual interface tag
 *      receiverProcessId The process id provided as the senderProcessId in
 *                      the CsrWifiHipMaPacketReq
 *      sigLength       Length of the signal
 *      sigData         Pointer to the signal data. HAL remains the owner
 *                      of the memory.
 *      dMac            Pointer to the destination MAC address. The
 *                      memory ownership is not handed over to the os layer
 *      sMac            Pointer to the source MAC address. The memory
 *                      ownership is not handed over to the os layer.
 *      frameType       The frame type
 *      macHeaderOffset The MAC header offset, which can be used to move the
 *                      the bulkdata pointer to point at the SNAP header.
 *      bulkdata        Pointer to bulkdata. The memory ownership is
 *                      handed over to the os layer.
 *      receptionResult The reception result for the frame.
 *      rssi            The RSSI for the received frame.
 *      snr             SNR for the received frame.
 *      rate            The rate at which the frame has been received.
 *
 *  Returns:
 *      None
 *
 * ---------------------------------------------------------------------------
 */
void CsrWifiHipMaPacketInd(void                    *osLayerContext,
                           CsrUint16                interfaceTag,
                           CsrUint16                receiverProcessId,
                           CsrUint16                sigLength,
                           CsrUint8                *sigData,
                           CsrUint8                *dMac,
                           CsrUint8                *sMac,
                           CsrWifiHipFrameType      frameType,
                           CsrUint8                 macHeaderOffset,
                           CsrWifiHipBulkDataParam *bulkData,
                           CsrResult                receptionResult,
                           CsrInt16                 rssi,
                           CsrInt16                 snr,
                           CsrUint16                rate);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipMaPacketCfm
 *
 *      Confirm for a CsrWifiHipMaPacketReq
 *
 *  Arguments:
 *      osLayerContext  Pointer to the os layer context
 *      interfaceTag    Virtual interface tag
 *      receiverProcessId The process id provided as the senderProcessId in
 *                      the CsrWifiHipMaPacketReq
 *      transmissionResult The result of the transmission
 *      rate            The rate the CsrWifiHipMaPacketReq has been
 *                      transmitted at.
 *      hostTag         Tag, which uniquely identifies a Ma frame. The
 *                      router API outlines the structure of the hostTag.
 *      sigLength       The length of the signal
 *      sigData         Pointer to the signal data. The memory ownership is
 *                      not handed over to the os layer.
 *      bulkdata        Pointer to bulkdata. The memory ownership is
 *                      handed over to the os layer.
 *
 *  Returns:
 *      None
 *
 * ---------------------------------------------------------------------------
 */
void CsrWifiHipMaPacketCfm(void                    *osLayerContext,
                           CsrUint16                interfaceTag,
                           CsrUint16                receiverProcessId,
                           CsrResult                transmissionResult,
                           CsrUint16                rate,
                           CsrUint32                hostTag,
                           CsrUint16                sigLength,
                           CsrUint8                *sigData,
                           CsrWifiHipBulkDataParam *bulkdata);


/******************************** Power mode ********************************/
/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipLowPowerModeReq
 *
 *      Request to set the power mode settings in HAL.
 *
 *  Arguments:
 *      hipHandle       Pointer to the HIP context
 *      lowPowerMode    Indicates if low power mode is enabled or not
 *      wakeHost        Indicates if the host shall be periodically be woken
 *                      during low power mode.
 *
 *  Returns:
 *      Returns if the low power mode setting were stored successfully.
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipLowPowerModeReq(void                         *hipHandle,
                                    CsrWifiRouterCtrlLowPowerMode lowPowerMode,
                                    CsrBool                       wakeHost);


/***************************** Traffic analysis *****************************/
/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipTaConfigureReq
 *
 *      Configure traffic analysis in HAL.
 *
 *  Arguments:
 *      hipHandle       Pointer to the HIP context
 *      trafficConfigType Defines the traffic config type
 *      config          Can be used to install custom filters in traffic
 *                      analysis.
 *
 *  Returns:
 *      Returns if the the traffic analysis settings were stored successfully.
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipTaConfigureReq(void                              *hipHandle,
                                   CsrWifiRouterCtrlTrafficConfigType trafficConfigType,
                                   CsrWifiRouterCtrlTrafficConfig     config);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipTaClassificationReq
 *
 *      Configure traffic classification in HAL.
 *
 *  Arguments:
 *      hipHandle       Pointer to the HIP context
 *      trafficType     Specifices the classification type
 *      period          Traffic period, if the type is set to
 *                      CSR_WIFI_ROUTER_CTRL_TRAFFIC_TYPE_PERIODIC
 *
 *  Returns:
 *      Returns if the the traffic classification settings were stored
 *      successfully.
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipTaClassificationReq(void                        *hipHandle,
                                        CsrWifiRouterCtrlTrafficType trafficType,
                                        CsrUint16                    period);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipTaIndicateProtocolInd
 *
 *      Indicate that a specific protocol has been seen.
 *
 *  Arguments:
 *      osLayerContext  Pointer to the os layer context
 *      interfaceTag    Virtual interface tag
 *      packetType      The packet type
 *      direction       The direction, i.e. TX or RX.
 *      srcAddr         The source MAC address
 *
 *  Returns:
 *      None
 *
 * ---------------------------------------------------------------------------
 */
void CsrWifiHipTaIndicateProtocolInd(void                              *osLayerContext,
                                     CsrUint16                          interfaceTag,
                                     CsrWifiRouterCtrlTrafficPacketType packetType,
                                     CsrWifiRouterCtrlProtocolDirection direction,
                                     CsrWifiMacAddress                  srcAddr);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipTaIndicateSamplingInd
 *
 *      Indicate a traffic analysis sample.
 *
 *  Arguments:
 *      osLayerContext  Pointer to the os layer context
 *      interfaceTag    Virtual interface tag
 *      trafficStats    The stats for the sampled period
 *
 *  Returns:
 *      None
 *
 * ---------------------------------------------------------------------------
 */
void CsrWifiHipTaIndicateSamplingInd(void                          *osLayerContext,
                                     CsrUint16                      interfaceTag,
                                     CsrWifiRouterCtrlTrafficStats *trafficStats);


/******************************* Capabilities *******************************/
/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipCapabilitiesReq
 *
 *      Request information about the soft queue sizes.
 *
 *  Arguments:
 *      hipHandle       Pointer to the HIP context
 *      appHandle       The appHandle, which the capabilities shall be send to
 *      clientData      Client data information
 *
 *  Returns:
 *      None
 *
 * ---------------------------------------------------------------------------
 */
void CsrWifiHipCapabilitiesReq(void       *hipHandle,
                               CsrSchedQid appHandle,
                               CsrUint16   clientData);

/***************************** Controlled port ******************************/
/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPortConfigureReq
 *
 *      Configure the controlled port information in HAL.
 *
 *  Arguments:
 *      hipHandle       Pointer to the HIP context
 *      interfaceTag    The virtual interface tag
 *      uncontrolledPortAction Set the state of the uncontrolled port
 *      controlledPortAction Set the state of the controlled port
 *      macAddress      Set controlled port for a specific peer
 *      setProtection   Shall the protection bit in the MAC header be set.
 *
 *  Returns:
 *      Returns the status of the configure port. CSR_RESULT_SUCCESS if it
 *      went well, CSR_RESULT_FAILURE if something failed.
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPortConfigureReq(void                       *hipHandle,
                                     CsrUint16                   interfaceTag,
                                     CsrWifiRouterCtrlPortAction uncontrolledPortAction,
                                     CsrWifiRouterCtrlPortAction controlledPortAction,
                                     CsrWifiMacAddress          *macAddress,
                                     CsrBool                     setProtection);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipCtrlPortQueueProcessReq
 *
 *      Process any frame that have been queued in the receive plane while the
 *      uncontrolled and controlled ports have been in block closed state.
 *
 *  Arguments:
 *      hipHandle       Pointer to the HIP context
 *      interfaceTag    The virtual interface tag
 *      portType        Indicate if the controlled or uncontrolled port queue
 *                      should be processed.
 *      macAddress      If the port has been processed for a specific peer,
 *                      the MAC address shall be provided.
 *      indicate        If the packets shall not be indicated to high layers,
 *                      i.e. the frames will be droped set indicate to FALSE.
 *                      Otherwise, the queued frames will be forwarded to the
 *                      high layers.
 *
 *  Returns:
 *      Returns if the queue was processed successful.
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipCtrlPortQueueProcessReq(void              *hipHandle,
                                            CsrUint16          interfaceTag,
                                            CsrWifiHipPortType portType,
                                            CsrWifiMacAddress *macAddress,
                                            CsrBool            indicate);

/********************************* RAW SDIO *********************************/

CsrResult CsrWifiHipRawSdioByteWrite(CsrUint8 function, CsrUint32 address, CsrUint8 data);
CsrResult CsrWifiHipRawSdioByteRead(CsrUint8 function, CsrUint32 address, CsrUint8 *pdata);
CsrResult CsrWifiHipRawSdioFirmwareDownload(CsrUint32 length, const CsrUint8 *pdata);
CsrResult CsrWifiHipRawSdioReset(void);
CsrResult CsrWifiHipRawSdioCoreDumpPrepare(CsrBool suspendSme);
CsrResult CsrWifiHipRawSdioByteBlockRead(CsrUint8 function, CsrUint32 address, CsrUint8 *pdata, CsrUint32 length);
CsrResult CsrWifiHipRawSdioGpRead16(CsrUint8 function, CsrUint32 address, CsrUint16 *pdata);
CsrResult CsrWifiHipRawSdioGpWrite16(CsrUint8 function, CsrUint32 address, CsrUint16 data);

/****************************** Suspend/resume ******************************/
/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipSuspendInd
 *
 *      Indicates that a suspend will take place.
 *
 *  Arguments:
 *      osLayerContext  Pointer to the os layer context
 *      clientData      client data
 *      hardSuspend     Is it a hard suspend
 *      d3Suspend       Is it a d3 suspend
 *
 *  Returns:
 *      None
 *
 * ---------------------------------------------------------------------------
 */
void CsrWifiHipSuspendInd(void     *osLayerContext,
                          CsrUint16 clientData,
                          CsrBool   hardSuspend,
                          CsrBool   d3Suspend);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipResumeInd
 *
 *      Indicates resume.
 *
 *  Arguments:
 *      osLayerContext  Pointer to the os layer context
 *      clientData      client data
 *      powerMaintained Indicates if the chip remained powered during the
 *                      suspend
 *
 *  Returns:
 *      None
 *
 * ---------------------------------------------------------------------------
 */
void CsrWifiHipResumeInd(void     *osLayerContext,
                         CsrUint16 clientData,
                         CsrBool   powerMaintained);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipSuspendRes
 *
 *      Indicates that the suspend has been processed.
 *
 *  Arguments:
 *      hipHandle       Pointer to the HIP context
 *      status          The result of the CsrWifiHipSuspendInd
 *
 *  Returns:
 *      None
 *
 * ---------------------------------------------------------------------------
 */
void CsrWifiHipSuspendRes(void *hipHandle, CsrResult status);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipResumeRes
 *
 *      Indicates that the resume has been processed.
 *
 *  Arguments:
 *      hipHandle       Pointer to the HIP context
 *      status          The result of the CsrWifiHipResumeInd
 *
 *  Returns:
 *      None
 *
 * ---------------------------------------------------------------------------
 */
void CsrWifiHipResumeRes(void *hipHandle, CsrResult status);

/************************************ Peer **********************************/
/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPeerAddReq
 *
 *      Add a peer to HAL.
 *
 *  Arguments:
 *      hipHandle       Pointer to the HIP context
 *      interfaceTag    The virtual interface tag
 *      peerMacAddress  The MAC address for the peer
 *      associationId   The association id
 *      staInfo         Pointer to struct, which holds information about the
 *                      peer. The caller remains the owner of the memory.
 *      peerHandle      Pointer where the peerHandle assigned by HAL will be
 *                      written
 *
 *  Returns:
 *      Returns if the the peer has been successfully stored in HAL.
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPeerAddReq(void                     *hipHandle,
                               CsrUint16                 interfaceTag,
                               CsrWifiMacAddress         peerMacAddress,
                               CsrUint16                 associationId,
                               CsrWifiRouterCtrlStaInfo *staInfo,
                               CsrUint32                *peerHandle);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPeerUpdateReq
 *
 *      Update a peer record in HAL.
 *
 *  Arguments:
 *      hipHandle       Pointer to the HIP context
 *      interfaceTag    The virtual interface tag
 *      peerHandle      The peerHandle, which was assigned by HAL in the
 *                      CsrWifiHipPeerAddReq
 *      result          The current state of the peer
 *
 *  Returns:
 *      Returns if the the peer has been successfully updated in HAL.
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPeerUpdateReq(void      *hipHandle,
                                  CsrUint16  interfaceTag,
                                  CsrUint32  peerHandle,
                                  CsrResult *result);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPeerDelReq
 *
 *      Delete a peer record in HAL.
 *
 *  Arguments:
 *      hipHandle       Pointer to the HIP context
 *      interfaceTag    The virtual interface tag
 *      peerHandle      The peerHandle, which was assigned by HAL in the
 *                      CsrWifiHipPeerAddReq
 *
 *  Returns:
 *      Returns if the the peer has been successfully deleted in HAL.
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPeerDelReq(void     *hipHandle,
                               CsrUint16 interfaceTag,
                               CsrUint32 peerHandle);

/********************************* Block Ack ********************************/
/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipBlockAckDisableReq
 *
 *      Delete a block ack session in HAL.
 *
 *  Arguments:
 *      hipHandle       Pointer to the HIP context
 *      interfaceTag    The virtual interface tag
 *      macAddress      The peer MAC address
 *      trafficStreamID The traffic stream id
 *      role            The block ack role - originator or recipient
 *
 *  Returns:
 *      Returns if the block ack session has been successfully deleted in HAL.
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipBlockAckDisableReq(void                            *hipHandle,
                                       CsrUint16                        interfaceTag,
                                       CsrWifiMacAddress                macAddress,
                                       CsrWifiRouterCtrlTrafficStreamId trafficStreamID,
                                       CsrWifiRouterCtrlBlockAckRole    role);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipBlockAckEnableReq
 *
 *      Add a block ack session in HAL.
 *
 *  Arguments:
 *      hipHandle       Pointer to the HIP context
 *      interfaceTag    The virtual interface tag
 *      macAddress      The peer MAC address
 *      trafficStreamID The traffic stream id
 *      role            The block ack role - originator or recipient
 *      bufferSize      The size of the buffer window
 *      timeout         The timeout period. Timeout will be 0 if timeout is
 *                      not enabled for the session
 *      ssn             The starting sequence number
 *
 *  Returns:
 *      Returns if the block ack session has been successfully added in HAL.
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipBlockAckEnableReq(void                            *hipHandle,
                                      CsrUint16                        interfaceTag,
                                      CsrWifiMacAddress                macAddress,
                                      CsrWifiRouterCtrlTrafficStreamId trafficStreamID,
                                      CsrWifiRouterCtrlBlockAckRole    role,
                                      CsrUint16                        bufferSize,
                                      CsrUint16                        timeout,
                                      CsrUint16                        ssn);


/*********************************** Stats **********************************/

typedef struct
{
    CsrUint32 rxPackets; /* total packets received */
    CsrUint32 rxBytes; /* total bytes received */
    CsrUint32 rxErrors; /* bad packets received */
    CsrUint32 rxDropped; /* no space */
    CsrUint32 rxFrameErrors; /* frame alignment error */
} CsrWifiHipReceiveStat;

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipStatGetReq
 *
 *      Sample the receive stats in HAL.
 *
 *  Arguments:
 *      hipHandle       Pointer to the HIP context
 *      interfaceTag    The virtual interface tag
 *      stats           Pointer to where the stats shall be written
 *
 *  Returns:
 *      Returns CSR_RESULT_SUCCESS if the stat was successfully written or
 *      CSR_RESULT_FAILURE if the CsrWifiHipStatGetReq failed.
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipStatGetReq(void *hipHandle, CsrUint16 interfaceTag, CsrWifiHipReceiveStat *stats);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipStatResetReq
 *
 *      Reset the stats in HAL.
 *
 *  Arguments:
 *      hipHandle       Pointer to the HIP context
 *      interfaceTag    The virtual interface tag
 *
 *  Returns:
 *      None
 *
 * ---------------------------------------------------------------------------
 */
void CsrWifiHipStatResetReq(void *hipHandle, CsrUint16 interfaceTag);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipDriverInfoGetReq
 *
 *      This function writes some basic driver information at the location of
 *      the data pointer.
 *
 *  Arguments:
 *      hipHandle       Pointer to the HIP context
 *      data            Pointer where the driver information will be written
 *      remaining       Pointer where info will be written to.
 *      debugInfo       Pointer to a CsrWifiHipDriverInfoDebugInfo struct
 *                      where the most recent debug signals will be written.
 *
 *  Returns:
 *      returns the number of written bytes to the 'remaining' pointer
 *
 * ---------------------------------------------------------------------------
 */
CsrInt32 CsrWifiHipDriverInfoGetReq(void                          *hipHandle,
                                    CsrCharString                 *data,
                                    CsrInt32                      *remaining,
                                    CsrWifiHipDriverInfoDebugInfo *debugInfo);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipLogUdiRegistrationReq
 *
 *      This function is used to register a UDI log function.
 *
 *  Arguments:
 *      hipHandle       Pointer to the HIP instance.
 *      logFunction     The log function of type CsrWifiHipLogUdiFunc.
 *
 *  Returns:
 *      The result of the registration.
 *
 * ---------------------------------------------------------------------------
 */

typedef void (*CsrWifiHipLogUdiFunc)(void *osLayerContext, CsrUint8 *sigdata,
                                     CsrUint32 signal_len, const CsrWifiHipBulkDataParam *bulkdata,
                                     CsrWifiHipLogUdiDirection dir);
CsrResult CsrWifiHipLogUdiRegistrationReq(void *hipHandle, CsrWifiHipLogUdiFunc logFunction);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipLogUdiUnregistrationReq
 *
 *      This function is used to unregister a UDI log function.
 *
 *  Arguments:
 *      hipHandle       Pointer to the HIP instance.
 *
 *  Returns:
 *      The result of the unregistration.
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipLogUdiUnregistrationReq(void *hipHandle);

CsrResult CsrWifiHipHip2StatsGet(void *hipHandle, CsrWifiHipHip2Stats *stats);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipInterfaceAddReq
 *
 *      This function triggers the netdev interface registering
 *      platform specific routines.
 *
 *  Arguments:
 *      hipHandle       Pointer to the HIP instance.
 *      interfaceTag    The interfaceTag, for which netdev registered
 *      macAddress      The address with which interface registered
 *
 *  Returns:
 *      Result: CsrResult
 *
 * ---------------------------------------------------------------------------
 */

CsrResult CsrWifiHipInterfaceAddReq(void             *hipHandle,
                                    CsrUint16         interfaceTag,
                                    CsrWifiMacAddress macAddress);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipInterfaceDelReq
 *
 *      This function triggers the netdev interface unregistering
 *      platform specific routines.
 *
 *  Arguments:
 *      hipHandle       Pointer to the HIP instance.
 *      interfaceTag    The interfaceTag, for which netdev unregistered
 *
 *  Returns:
 *      Result: CsrResult
 *
 * ---------------------------------------------------------------------------
 */

CsrResult CsrWifiHipInterfaceDelReq(void     *hipHandle,
                                    CsrUint16 interfaceTag);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipVifAddReq
 *
 *      This function associates the VifIndex with corresponding
 *      interfaceTag
 *
 *  Arguments:
 *      hipHandle       Pointer to the HIP instance.
 *      interfaceTag    The interfaceTag, for which vifIndex associated.
 *      VifIndex        The Virtual interface Index used by firmware for
 *                      corresponding interfaceTag
 *
 *  Returns:
 *      Result: CsrResult
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipVifAddReq(void *hipHandle, CsrUint16 interfaceTag, CsrUint16 vifIndex, CsrBool isPrimaryVifIndex);


/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipVifDelReq
 *
 *      This function removes association of VifIndex for the corresponding
 *      interfaceTag
 *
 *  Arguments:
 *      hipHandle       Pointer to the HIP instance.
 *      interfaceTag    The interfaceTag, for which vifIndex association removed.
 *      vifIndex        The Virtual interface Index used by firmware for
 *                      corresponding interfaceTag
 *      vifIndexPrimary This vifIndex has to be set as primary after
 *                      primary vifIndex removed
 *
 *  Returns:
 *      Result: CsrResult
 *
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipVifDelReq(void *hipHandle, CsrUint16 interfaceTag, CsrUint16 vifIndex, CsrUint16 vifIndexPrimary);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipInterfaceRegisterInd
 *
 *      This function has to be ported across platform
 *
 *  Arguments:
 *      osLayerContext  Pointer to the osLayerContext.
 *      interfaceTag    The interfaceTag, for netdev registration
 *      macAddress      The address with which interface registered
 *
 *  Returns:
 *      Result: CsrResult
 *
 * ---------------------------------------------------------------------------
 */

CsrResult CsrWifiHipInterfaceRegisterInd(void             *osLayerContext,
                                         CsrUint16         interfaceTag,
                                         CsrWifiMacAddress macAddress);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipInterfaceUnregisterInd
 *
 *      This function has to be ported across platform
 *
 *  Arguments:
 *      osLayerContext  Pointer to the osLayerContext.
 *      interfaceTag    The interfaceTag, for netdev unregistration
 *
 *  Returns:
 *      Result: CsrResult
 *
 * ---------------------------------------------------------------------------
 */

CsrResult CsrWifiHipInterfaceUnregisterInd(void     *osLayerContext,
                                           CsrUint16 interfaceTag);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipVifIndexGetReq
 *
 *      This function returns the primary vifIndex from database
 *
 *  Arguments:
 *      hipHandle       Pointer to the HIP instance.
 *      interfaceTag    The interfaceTag index the primary vifIndex database
 *
 *  Returns:
 *      Result: vifIndex
 *
 * ---------------------------------------------------------------------------
 */
CsrUint16 CsrWifiHipVifIndexGetReq(void *hipHandle, CsrUint16 interfaceTag);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipInterfaceTagGetReq
 *
 *      This function returns the interfaceTag from primary vifIndex
 *
 *  Arguments:
 *      hipHandle       Pointer to the HIP instance.
 *      vifIndex        The vifIndex indexes the interfaceTag database
 *
 *  Returns:
 *      Result: interfaceTag
 *
 * ---------------------------------------------------------------------------
 */
CsrUint16 CsrWifiHipInterfaceTagGetReq(void *hipHandle, CsrUint16 vifIndex);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipWapiMicFilterReq
 *
 *      This function is used the set/reset the filters used for special MIC
 *      error handling in WAPI
 *
 *  Arguments:
 *      hipHandle       Pointer to the HIP instance.
 *      interfaceTag    The interfaceTag, for which netdev unregistered
 *      status          TRUE/FALSE to set/reset the filter
 *      isUnicast       TRUE/FALSE to set the Unicast/Multicast filter
 *  Returns:
 *      Result: None
 *
 * ---------------------------------------------------------------------------
 */
void CsrWifiHipWapiMicFilterReq(void     *hipHandle,
                                CsrUint16 interfaceTag,
                                CsrBool   status,
                                CsrBool   isUnicast);

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipWapiMicVerifiedFrameRxReq
 *
 *      Data plane for MIC verified data frames from the host stack. Data will
 *      be parsed to the network stack if they pass all the normal data plane
 *      checks.
 *
 *  Arguments:
 *      hipHandle       Pointer to the HIP instance.
 *      interfaceTag    The interfaceTag.
 *      signalLength    Length of the signal.
 *      signal          Pointer to the signal. The memory ownership is handed
 *                      over to HAL.
 *      bulkdata        Pointer to the bulkdata struct, which holds pointers
 *                      and lengths of bulkdata. The bulkdata is handed over
 *                      to HAL.
 *  Returns:
 *      Result: None
 *
 * ---------------------------------------------------------------------------
 */
void CsrWifiHipWapiMicVerifiedFrameRxReq(void                    *hipHandle,
                                         CsrUint16                interfaceTag,
                                         CsrUint32                signalLength,
                                         CsrUint8                *signal,
                                         CsrWifiHipBulkDataParam *bulkdata);

/*
 * ---------------------------------------------------------------------------
 *  CsrSdioClaim and CsrSdioRelease
 *
 *      The MMC Linux SDIO driver may require a claim and release mechanism
 *      before issueing SDIO operations. The OS Linux specific code implements
 *      the required functionality when needed.
 *
 *  Arguments:
 *      function        The SDIO function handler.
 *  Returns:
 *      Result: None
 *
 * ---------------------------------------------------------------------------
 */
#ifdef CSR_WIFI_SUPPORT_MMC_DRIVER
#include "os_linux_sdio.h"
#define CsrSdioClaim(function)   _CsrSdioClaim(function)
#define CsrSdioRelease(function) _CsrSdioRelease(function)
#else
#define CsrSdioClaim(function)
#define CsrSdioRelease(function)
#endif

/*
 * ---------------------------------------------------------------------------
 *  CSR_WIFI_HIP_LINUX_INSTALL_IRQ and CSR_WIFI_HIP_LINUX_REMOVE_IRQ
 *
 *      The MMC Linux SDIO driver requires that the SDIO interrupt handler is
 *      registered through a system call. These macros are implemented in OS
 *      Linux specific code when the functionality is required.
 *
 *  Arguments:
 *      function        The SDIO function handler.
 *  Returns:
 *      Result: Result of the registration/unregistration of IRQ handler.
 *
 * ---------------------------------------------------------------------------
 */
#ifdef CSR_WIFI_SUPPORT_MMC_DRIVER
#include "os_linux_sdio.h"
#define CSR_WIFI_HIP_LINUX_INSTALL_IRQ(function) _CSR_WIFI_HIP_LINUX_INSTALL_IRQ(function)
#define CSR_WIFI_HIP_LINUX_REMOVE_IRQ(function)  _CSR_WIFI_HIP_LINUX_REMOVE_IRQ(function)
#else
/*lint -e774*/
#define CSR_WIFI_HIP_LINUX_INSTALL_IRQ(function)  CSR_RESULT_SUCCESS;
#define CSR_WIFI_HIP_LINUX_REMOVE_IRQ(function)   CSR_RESULT_SUCCESS;
#endif

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipConstructMaPacketReqSignal
 *
 *      This function is used to construct the MA-PACKET.req signal
 *
 *  Arguments:
 *      hipHandle           Pointer to the HIP instance.
 *      interfaceTag        The virtual interfaceTag
 *      priority            The requested priority the frame should be
 *                          transmitted at.
 *      transmitRate        The requested rate the frame should be
 *                          transmitted at.
 *      hostTag             Tag, which uniquely identifies the Ma frame. The
 *                          router API outlines the usage of the hostTag.
 *      transmissionControl The requested transmission controlbitmap to the frame.
 *      leSenderProcessId   The sender process id
 *      dMac                Pointer to the destination MAC address.
 *      vifIndex            Valid virtual Interface Index associated with the interfaceTag & used
 *                          for corresponding interface activity with firmware.
 *      sigData             Pointer to the CSR_SIGNAL.
 *      frameType           Specify the frame type being consturcted
 *      minVifDuration      Minimum virtual interface duration
 *  Returns:
 *      Result: None
 *
 * ---------------------------------------------------------------------------
 */

void CsrWifiHipConstructMaPacketReqSignal(void *hipHandle, CsrUint16 interfaceTag,
                                          CsrUint32 priority, CsrUint16 transmitRate, CsrUint32 hostTag,
                                          CsrUint16 transmissionControl, CsrUint16 leSenderProcessId,
                                          CsrUint8 *peerMacAddress, CsrUint16 vifIndex, CsrUint8 *sigData,
                                          CsrWifiHipFrameType frameType, CsrUint16 minVifDuration);

#ifdef CSR_WIFI_DRIVER_HYDRA
/* Registration as a Hydra SDIO function */
void CsrWifiHipHydraRegister(void *osLayerContext, void *sdioContext);
void CsrWifiHipHydraUnregister(void *osLayerContext, void *sdioContext);
#endif

#ifdef __cplusplus
}
#endif
#endif
