/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"

#ifdef PS_USPACE
#include "csr_wifi_ps_types.h"
#else
#include "csr_wifi_hip_unifi.h"
#include "csr_wifi_hip_conversions.h"
#include "csr_wifi_hip_card.h"
#include "csr_wifi_hip_core_trace.h"
#endif
#include "csr_wifi_ps_q.h"
#include "csr_wifi_ps_circ.h"
#include "csr_wifi_ps_qsig.h"

#include "csr_wifi_ps_if.h"
#include "csr_wifi_ps.h"

#ifdef PS_USPACE
#include "csr_wifi_ps_card.h"
#endif

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerListInit
 *
 *  Initialise list pointers.
 *
 *  Arguments:
 *      list              Pointer list structure.
 *
 *  Returns:
 *      Nothing.
 * ---------------------------------------------------------------------------
 */
void csrWifiHipPacketSchedulerListInit(CsrWifiHipPacketSchedulerList *list)
{
    CSR_WIFI_HIP_LIST_INIT(&(list->list_pointers));

    list->entries = 0;
    list->currentp = 0;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerListAddTail
 *
 *  Add entry to tail of list.
 *
 *  Arguments:
 *      list              Pointer list structure.
 *      entry             Pointer entry structure.
 *
 *  Returns:
 *      Nothing.
 * ---------------------------------------------------------------------------
 */
void csrWifiHipPacketSchedulerListAddTail(CsrWifiHipPacketSchedulerList *list, CsrWifiHipPacketSchedulerListEntry *entry)
{
    CSR_WIFI_HIP_LIST_ADD_TAIL_ENTRY((&(entry->list_pointers)), (&(list->list_pointers)));
    if (!list->currentp)
    {
        list->currentp = list->list_pointers.next;
    }
    entry->list = list;
    list->entries++;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerListRemoveEntry
 *
 *  Remove entry from list.
 *
 *  Arguments:
 *      list              Pointer list structure.
 *      entry             Pointer entry structure.
 *
 *  Returns:
 *      Nothing.
 * ---------------------------------------------------------------------------
 */
void csrWifiHipPacketSchedulerListRemoveEntry(CsrWifiHipPacketSchedulerList *list, CsrWifiHipPacketSchedulerListEntry *entry)
{
    list->currentp = entry->list_pointers.next;
    CSR_WIFI_HIP_LIST_DELETE_ENTRY(&entry->list_pointers);
    if (!list->list_pointers.next)
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                            "csrWifiHipPacketSchedulerListRemoveEntry: after delete, list_pointers.next=0, list_entries=%d\n",
                            list->entries - 1));
    }
    list->entries--;
    entry->list = 0;
    if (CSR_WIFI_HIP_LIST_IS_LIST_EMPTY(&list->list_pointers))
    {
        list->currentp = 0;
        if (list->entries > 1)
        {
            CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                                "csrWifiHipPacketSchedulerListRemoveEntry: list=0x%p is empty, but entries=%d\n",
                                list, list->entries - 1));
        }
    }
    else
    {
        if (list->currentp == &(list->list_pointers))
        {
            list->currentp = list->list_pointers.next;
        }
    }
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerListTakeFirst
 *
 *  Remove first entry from list.
 *
 *  Arguments:
 *      list              Pointer list structure.
 *      entry             Pointer entry structure.
 *
 *  Returns:
 *      Nothing.
 * ---------------------------------------------------------------------------
 */
CsrWifiHipPacketSchedulerListEntry *csrWifiHipPacketSchedulerListTakeFirst(CsrWifiHipPacketSchedulerList *list)
{
    CsrWifiHipPacketSchedulerListEntry *entry = 0;

    if (!CSR_WIFI_HIP_LIST_IS_LIST_EMPTY(&(list->list_pointers)))
    {
        entry = CSR_WIFI_HIP_LIST_LIST_ENTRY(list->list_pointers.next, CsrWifiHipPacketSchedulerListEntry, list_pointers);
    }
    if (entry)
    {
        if (entry->list != list)
        {
            CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "csrWifiHipPacketSchedulerListTakeFirst: entry=0x%p has list pointer 0x%p when taken from list 0x%p\n", entry, entry->list, list));
            return 0;
        }
        csrWifiHipPacketSchedulerListRemoveEntry(list, entry);
    }
    return entry;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerListGetFirst
 *
 *  Reset current pointer to first in list.
 *
 *  Arguments:
 *      list              Pointer list structure.
 *
 *  Returns:
 *      First in list.
 * ---------------------------------------------------------------------------
 */
CsrWifiHipPacketSchedulerListEntry *csrWifiHipPacketSchedulerListGetFirst(CsrWifiHipPacketSchedulerList *list)
{
    if (CSR_WIFI_HIP_LIST_IS_LIST_EMPTY(&(list->list_pointers)))
    {
        if (list->currentp)
        {
            CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "csrWifiHipPacketSchedulerListGetFirst: list is empty yet currentp != 0\n"));
            CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, " list=0x%p currentp=0x%p\n", list, list->currentp));
            list->currentp = 0;
        }
        return 0;
    }
    list->currentp = list->list_pointers.next;
    if (!list->currentp)
    {
        return 0;
    }
    return CSR_WIFI_HIP_LIST_LIST_ENTRY(list->currentp, CsrWifiHipPacketSchedulerListEntry, list_pointers);
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerListGetNext
 *
 *  Returns current pointer and then moves current pointer to next in list.
 *
 *  Arguments:
 *      list              Pointer list structure.
 *
 *  Returns:
 *      Current pointer.
 * ---------------------------------------------------------------------------
 */
CsrWifiHipPacketSchedulerListEntry *csrWifiHipPacketSchedulerListGetNext(CsrWifiHipPacketSchedulerList *list)
{
    CsrWifiHipPacketSchedulerListEntry *cur = 0;
    if (!list->currentp)
    {
        return 0;
    }
    cur = CSR_WIFI_HIP_LIST_LIST_ENTRY(list->currentp, CsrWifiHipPacketSchedulerListEntry, list_pointers);
    list->currentp = list->currentp->next;
    if (list->currentp == &(list->list_pointers))
    {
        list->currentp = list->list_pointers.next;
    }
    return cur;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerListPeek
 *
 *  Returns first entry in list.
 *
 *  Arguments:
 *      list              Pointer list structure.
 *
 *  Returns:
 *      First entry.
 * ---------------------------------------------------------------------------
 */
CsrWifiHipPacketSchedulerListEntry *csrWifiHipPacketSchedulerListPeek(CsrWifiHipPacketSchedulerList *list)
{
    if (CSR_WIFI_HIP_LIST_IS_LIST_EMPTY(&(list->list_pointers)))
    {
        return 0;
    }
    return CSR_WIFI_HIP_LIST_LIST_ENTRY(list->list_pointers.next, CsrWifiHipPacketSchedulerListEntry, list_pointers);
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerListNext
 *
 *  Returns first entry in list.
 *
 *  Arguments:
 *      list              Pointer list structure.
 *
 *  Returns:
 *      First entry.
 * ---------------------------------------------------------------------------
 */
CsrWifiHipPacketSchedulerListEntry *csrWifiHipPacketSchedulerListNext(CsrWifiHipPacketSchedulerListEntry *entry)
{
    CsrWifiHipPacketSchedulerListEntry *list;

    list = (CsrWifiHipPacketSchedulerListEntry *) entry->list;

    if (entry->list_pointers.next == &(list->list_pointers))
    {
        return 0;
    }
    return CSR_WIFI_HIP_LIST_LIST_ENTRY(entry->list_pointers.next, CsrWifiHipPacketSchedulerListEntry, list_pointers);
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerListTake
 *
 *  Returns first entry in list.
 *
 *  Arguments:
 *      list              Pointer list structure.
 *
 *  Returns:
 *      First entry.
 * ---------------------------------------------------------------------------
 */
void csrWifiHipPacketSchedulerListTake(CsrWifiHipPacketSchedulerList *list, CsrWifiHipPacketSchedulerListEntry *entry)
{
    csrWifiHipPacketSchedulerListRemoveEntry(list, entry);
}
