/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#ifndef CSR_WIFI_PS_IF_H__
#define CSR_WIFI_PS_IF_H__

#include "csr_synergy.h"

#ifdef __cplusplus
extern "C" {
#endif

struct card;
struct card_signal;
/*
 * CsrWifiHipPacketSchedulerQSPri defines the priority level for a Queue Set.
 * Use when a station is using WMM-PS to ensure that a Service Period's worth
 * of frames are delivered quickly on receipt of a trigger.
 */
typedef enum
{
    QS_PRI_HIGH = 1,
    QS_PRI_NORMAL = 2
} CsrWifiHipPacketSchedulerQSPri;

/*
 * CsrWifiHipPacketSchedulerACPri are the standard WMM-AC QoS priority levels.
 */
typedef enum
{
    AC_PRI_0 = 0,
    AC_PRI_1 = 1,
    AC_PRI_2 = 2,
    AC_PRI_3 = 3
} CsrWifiHipPacketSchedulerACPri;

/*
 * CsrWifiHipPacketSchedulerQState indicates the state of an AC queue. The
 * state is passed to the upper layer using the QStateCb function.
 */
typedef enum
{
    AC_QSTATE_FULL,
    AC_QSTATE_NOT_FULL
} CsrWifiHipPacketSchedulerQState;

/*
 * CsrWifiHipPacketSchedulerQSAuthQState indicates the state of an Auth
 * queue. The state is used by upper layers to configure the packet
 * scheduler to mirror 802.1x uncontrolled port status.
 */

typedef enum
{
    QS_AUTHQ_DISABLED = 0,
    QS_AUTHQ_ENABLED = 1
} CsrWifiHipPacketSchedulerQSAuthQState;

/*
 * CsrWifiHipPacketSchedulerQSACQState indicates the state of the
 * AC queues. The state is used by upper layers to configure packet
 * scheduler to mirror 802.1x controlled port status.
 */

typedef enum
{
    QS_ACQ_DISABLED = 0,
    QS_ACQ_ENABLED = 1
} CsrWifiHipPacketSchedulerQSACQState;


/*
 * CsrWifiHipPacketSchedulerMcastCompleteCb is a function registered with the
 * Packet Scheduler. It is called upon completion of a multicast burst.
 */
typedef void (*CsrWifiHipPacketSchedulerMcastCompleteCb)(void *ospriv, CsrInt32 vif_id);

/*
 * Statistics held for each queue.
 */
typedef struct
{
    CsrUint32 queued_signals;
    CsrUint32 queued_data;
    CsrUint32 dequeued_signals;
    CsrUint32 dequeued_data;
    CsrUint32 discarded_signals;
    CsrUint32 rejected_signals;
    CsrUint32 disabled;
    CsrUint32 enabled;
} CsrWifiHipPacketSchedulerStats;

typedef struct
{
    CsrUint32 missed;
    CsrUint32 scheduled;
    CsrInt32  current_tokens;
    CsrUint32 accumulated_tokens;
} CsrWifiHipPacketSchedulerDwrrStats;

/*
 * CsrWifiHipPacketSchedulerDiscardCb is a function registered with the Packet Scheduler.
 * It is called when a signal is discarded by the Packet Scheduler (if the signal is
 * marked as requiring confirmation). Signals can be discarded when a VIF or QS is
 * disassociated or when the discard algorithm associated with an AC queue (if there
 * is one) dictates that a signal is discarded.
 */
typedef void (*CsrWifiHipPacketSchedulerDiscardCb)(void *ospriv, CsrInt32 vif_id, struct card_signal *signal);

#define PS_MIN_VIF_ID 0
#define PS_MAX_VIF_ID (CSR_WIFI_MAX_INTERFACES - 1)

#define PS_MIN_AID 0
#define PS_MAX_AID 31
#define PS_AP_START_AID 16

/*
 * CsrWifiHipPacketSchedulerVIFAssociate: Create a VIF instance in the Packet Scheduler. The vif_id
 * is an instance number. It is not necessarily the same number as the VIF instance in the f/w.
 */
extern CsrResult CsrWifiHipPacketSchedulerVIFAssociate(struct card *card, CsrInt32 vif_id, CsrWifiHipPacketSchedulerDiscardCb discard_cb);
/*
 * CsrWifiHipPacketSchedulerVIFDisassociate: Destroy a VIF instance in the Packet Scheduler. All
 * Queue Sets associated with the VIF will have their queues purged and structures deallocated.
 */
extern CsrResult CsrWifiHipPacketSchedulerVIFDisassociate(struct card *card, CsrInt32 vif_id);
/*
 * CsrWifiHipPacketSchedulerQueueSetAssociate: Create a Queue Set instance in the Packet Scheduler.
 * The Queue Set is associated with a station. For a VIF in station mode, an aid value of zero
 * should be used. For Queue Sets in AP mode, the aid value should match the aid value of the
 * associated station as used in the f/w.
 */
extern CsrResult CsrWifiHipPacketSchedulerQueueSetAssociate(struct card *card, CsrInt32 vif_id, CsrInt32 aid, CsrWifiHipPacketSchedulerQSPri pri, CsrInt32 qlen, CsrWifiHipFlowControlPauseCb pauseCb, CsrWifiHipFlowControlResumeCb resumeCb, void *pauseResumeContext);
/*
 * CsrWifiHipPacketSchedulerQueueSetDisassociate: Destroy a Queue Set instance in the Packet Scheduler.
 * This function should be called when a station associated with a AP mode VIF disassociates.
 */
extern CsrResult CsrWifiHipPacketSchedulerQueueSetDisassociate(struct card *card, CsrInt32 vif_id, CsrInt32 aid);
/*
 * CsrWifiHipPacketSchedulerQueueSetActivate: Cause the SMOD for all configured Queue Sets to be recalculated.
 */
extern CsrResult CsrWifiHipPacketSchedulerQueueSetActivate(card_t *card, CsrInt32 vif_id, CsrInt32 aid);
/*
 * CsrWifiHipPacketSchedulerVIFACRelativePriorities: Place the AC queues into the lists according to the relative priority.
 */
extern CsrResult CsrWifiHipPacketSchedulerVIFACRelativePriorities(card_t *card, CsrInt32 vif_id, CsrInt32 rpri[]);
/*
 * CsrWifiHipPacketSchedulerQueueSetPriority: Set the Queue Set priority. This allows for Queue Sets to be
 * moved between normal and high priority. The high priority will mean that the Queue Set traffic is
 * processed before the normal priority Queue Sets. This feature is intended for use with stations using
 * U-APSD.
 */
extern CsrResult CsrWifiHipPacketSchedulerQueueSetPriority(struct card *card, CsrInt32 vif_id, CsrInt32 aid, CsrWifiHipPacketSchedulerQSPri priority);
/*
 * CsrWifiHipPacketSchedulerQueueSetSMod: Sets the SMOD value for a Queue Set. The SMOD value defines how much
 * outstanding data is allowed in the Queue Set.
 */
extern CsrResult CsrWifiHipPacketSchedulerQueueSetSMod(struct card *card, CsrInt32 vif_id, CsrInt32 aid, CsrInt32 smod);
/*
 * CsrWifiHipPacketSchedulerQueueACQMod: Sets the QMOD value for an AC queue within a Queue Set. The QMOD value
 * defines how much outstanding data is allowed from a queue.
 */
extern CsrResult CsrWifiHipPacketSchedulerQueueACQMod(struct card *card, CsrInt32 vif_id, CsrInt32 aid, CsrWifiHipPacketSchedulerACPri prio, CsrInt32 qmod);
/*
 * CsrWifiHipPacketSchedulerQueueSetAuthQState: Sets the auth queue state for the the queue set. It
 * defines when the auth queue is enabled for picking up by scheduler.
 */
extern CsrResult CsrWifiHipPacketSchedulerQueueSetAuthQState(struct card *card, CsrInt32 vif_id, CsrInt32 aid, CsrWifiHipPacketSchedulerQSAuthQState state);
/*
 * CsrWifiHipPacketSchedulerQueueSetACQState: Sets the AC queues state for the the queue set. It
 * defines when the AC queues are enabled for picking up by scheduler.
 */
extern CsrResult CsrWifiHipPacketSchedulerQueueSetACQState(struct card *card, CsrInt32 vif_id, CsrInt32 aid, CsrWifiHipPacketSchedulerQSACQState state);
/*
 * CsrWifiHipPacketSchedulerQueueMulticastBurstEnable: If enabled, set the burst maximum for the queue to
 * defined value and set the QMOD for the queue to 1. If disabled, set the burst maximum to zero and the
 * QMOD to the defined QMOD for a multicast queue.
 */
extern CsrResult CsrWifiHipPacketSchedulerQueueMulticastBurstEnable(card_t *card, CsrInt32 vif_id, CsrInt32 enable);
/*
 * CsrWifiHipPacketSchedulerQueueSetPurge: Removes all frames from all queues in the Queue Set and discards them.
 */
extern CsrResult CsrWifiHipPacketSchedulerQueueSetPurge(struct card *card, CsrInt32 vif_id, CsrInt32 aid);
/*
 * CsrWifiHipPacketSchedulerQueueSetAuthQPurge: Removes all frames from Auth queue in the Queue Set and discards them.
 */
extern CsrResult CsrWifiHipPacketSchedulerQueueSetAuthQPurge(struct card *card, CsrInt32 vif_id, CsrInt32 aid);
/*
 * CsrWifiHipPacketSchedulerQueueSetACQPurge: Removes all frames from AC queues in the Queue Set and discards them.
 */
extern CsrResult CsrWifiHipPacketSchedulerQueueSetACQPurge(struct card *card, CsrInt32 vif_id, CsrInt32 aid);
/*
 * CsrWifiHipPacketSchedulerQueueSetBurstMaximum: Sets the burst maximum (in frames) allowed from a Queue Set.
 * This feature can be used in conjunction with the Queue Set priority to provide a burst of frames from
 * a Queue Set when so triggered.
 */
extern CsrResult CsrWifiHipPacketSchedulerQueueSetBurstMaximum(struct card *card, CsrInt32 vif_id, CsrInt32 aid, CsrInt32 bm);
/*
 * CsrWifiHipPacketSchedulerQSigCount: Get the current number of not in use (free) QSIG structures.
 */
extern CsrInt32 CsrWifiHipPacketSchedulerQSigCount(void *hipHandle);
/*
 * CsrWifiHipPacketSchedulerCtrlStats: Get the control queue stats.
 */
extern CsrResult CsrWifiHipPacketSchedulerCtrlStats(void *hipHandle, CsrWifiHipPacketSchedulerStats *stats);
/*
 * CsrWifiHipPacketSchedulerMgmtStats: Get the management queue stats for a VIF.
 */
extern CsrResult CsrWifiHipPacketSchedulerMgmtStats(void *hipHandle, CsrInt32 vif_id, CsrWifiHipPacketSchedulerStats *stats);
/*
 * CsrWifiHipPacketSchedulerMulticastStats: Get the multicast queue stats for a VIF.
 */
extern CsrResult CsrWifiHipPacketSchedulerMulticastStats(void *hipHandle, CsrInt32 vif_id, CsrWifiHipPacketSchedulerStats *stats);
/*
 * CsrWifiHipPacketSchedulerQueueSetACStats: Get the queue stats for a Queue Set.
 */
extern CsrResult CsrWifiHipPacketSchedulerQueueSetACStats(void *hipHandle, CsrInt32 vif_id, CsrInt32 aid, CsrWifiHipPacketSchedulerACPri ac, CsrWifiHipPacketSchedulerStats *stats, CsrWifiHipPacketSchedulerQSACQState *acq_state);
/*
 * CsrWifiHipPacketSchedulerQueueSetMgmtStats: Get the management queue stats for a Queue Set.
 */
extern CsrResult CsrWifiHipPacketSchedulerQueueSetMgmtStats(void *hipHandle, CsrInt32 vif_id,  CsrInt32 aid, CsrWifiHipPacketSchedulerStats *stats);
/*
 * CsrWifiHipPacketSchedulerQueueSetAuthStats: Get the auth queue stats for a Queue Set.
 */
extern CsrResult CsrWifiHipPacketSchedulerQueueSetAuthStats(void *hipHandle, CsrInt32 vif_id,  CsrInt32 aid, CsrWifiHipPacketSchedulerStats *stats);
/*
 * CsrWifiHipPacketSchedulerDwrrStats: Get the DWRR stats per AC.
 */
extern CsrResult CsrWifiHipPacketSchedulerDwrrStatsGet(void *hipHandle, CsrWifiHipPacketSchedulerQSPri pri, CsrWifiHipPacketSchedulerACPri ac, CsrWifiHipPacketSchedulerDwrrStats *stats);
/*
 * CsrWifiHipPacketSchedulerSendCtrlData: Queue control signal.
 */
extern CsrResult CsrWifiHipPacketSchedulerSendCtrlData(struct card *card, struct card_signal *signal);
/*
 * CsrWifiHipPacketSchedulerSendMgmtFrame: Queue management signal for a VIF.
 */
extern CsrResult CsrWifiHipPacketSchedulerSendMgmtFrame(struct card *card, CsrInt32 vif_id, struct card_signal *signal);
/*
 * CsrWifiHipPacketSchedulerSendQsMgmtFrame: Queue management signal for a Queue Set.
 */
extern CsrResult CsrWifiHipPacketSchedulerSendQsMgmtFrame(struct card *card, CsrInt32 vif_id, CsrInt32 aid, struct card_signal *signal);
/*
 * CsrWifiHipPacketSchedulerSendQsAuthQueueFrame: Queue signal to the Auth queue (uncontrolled port) for a Queue Set.
 */
extern CsrResult CsrWifiHipPacketSchedulerSendQsAuthQueueFrame(struct card *card, CsrInt32 vif_id, CsrInt32 aid, struct card_signal *signal);
/*
 * CsrWifiHipPacketSchedulerSendUnicastFrame: Queue unicast signal.
 */
extern CsrResult CsrWifiHipPacketSchedulerSendUnicastFrame(struct card *card, CsrInt32 vif_id, CsrInt32 aid, CsrWifiHipPacketSchedulerACPri ac, struct card_signal *signal);
/*
 * CsrWifiHipPacketSchedulerSendMulticastFrame: Queue multicast signal.
 */
extern CsrResult CsrWifiHipPacketSchedulerSendMulticastFrame(struct card *card, CsrInt32 vif_id, struct card_signal *signal);
/*
 * CsrWifiHipPacketSchedulerInit: Called once, at boot, to initialise the Packet Scheduler.
 */
extern void CsrWifiHipPacketSchedulerInit(struct card *card);
/*
 * CsrWifiHipPacketSchedulerInit: Called once, at close-down, to free resource allocated to the Packet Scheduler.
 */
extern void CsrWifiHipPacketSchedulerDestroy(struct card *card);
/*
 * CsrWifiHipPacketSchedulerVIFSetMcastCompleteCb: Register callback to completion of a multicast burst.
 */
extern CsrResult CsrWifiHipPacketSchedulerVIFSetMcastCompleteCb(struct card *card, CsrInt32 vif_id, CsrWifiHipPacketSchedulerMcastCompleteCb mcast_complete_cb);
/*
 * csrWifiHipPacketSchedulerVIFGetAllQSState: Collate the configured QS and QS station state into bit maps for the specified VIF.
 */
extern CsrResult csrWifiHipPacketSchedulerVIFGetAllQSState(struct card *card, CsrInt32 vif_id, CsrUint16 *configured_aid, CsrUint16 *powersave_aid);
/*
 * CsrWifiHipPacketSchedulerQueueSetStationStateGet: Get the Queue Set station state and number of transitions between states (active and powersave).
 */
extern CsrResult CsrWifiHipPacketSchedulerQueueSetStationStateGet(void *hipHandle, CsrInt32 vif_id,  CsrInt32 aid, CsrInt32 *current_state, CsrUint32 *transitions);
/*
 * CsrWifiHipPacketSchedulerQueueSetSMODGet: Get the Queue Set SMOD (configured Station Maximum Outstanding Data) and SCOD (Station Current Outstanding Data).
 */
extern CsrResult CsrWifiHipPacketSchedulerQueueSetSMODGet(void *hipHandle, CsrInt32 vif_id,  CsrInt32 aid, CsrInt32 *smod, CsrUint32 *scod);
/*
 * CsrWifiHipPacketSchedulerSuspend: On system suspend this function is called to stop all transmit queues.
 */
extern void CsrWifiHipPacketSchedulerSuspend(struct card *card);
/*
 * CsrWifiHipPacketSchedulerResume: On system resume this function is called to start the set of transmit queues that should be started.
 */
extern void CsrWifiHipPacketSchedulerResume(struct card *card);
/*
 * CsrWifiHipPacketSchedulerCapabilitiesQueueLen: Report the circular buffer size to the calling entity.
 */
extern CsrInt32 CsrWifiHipPacketSchedulerCapabilitiesQueueLen(struct card *card);

#ifdef __cplusplus
}
#endif
#endif
