/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#ifndef CSR_WIFI_PS_CIRC_H__
#define CSR_WIFI_PS_CIRC_H__

#include "csr_synergy.h"

#ifdef __cplusplus
extern "C" {
#endif

#define UNIFI_QUEUE_LEN 32
#define CIRC_THRESHOLD (UNIFI_QUEUE_LEN / 2)

struct csrWifiHipPacketSchedulerQsig;
typedef struct
{
    struct csrWifiHipPacketSchedulerQsig *q_body[UNIFI_QUEUE_LEN];

    /* Num elements in queue (capacity is one less than this!) */
    CsrInt32 q_length;

    CsrInt32 q_wr_ptr;
    CsrInt32 q_rd_ptr;
} ps_circ_buffer_t;

#define CSR_WIFI_PS_Q_INIT(q, l) {   \
        q->q_length = l; \
        q->q_wr_ptr = q->q_rd_ptr = 0; \
}

/* Return queue slots used count
 *  params:
 *      const q_t *q
 *  returns:
 *      CsrUint16
 */
#define CSR_WIFI_PS_Q_SLOTS_USED(q)     \
    ((((q)->q_wr_ptr - (q)->q_rd_ptr) < 0) ? \
     (((q)->q_wr_ptr - (q)->q_rd_ptr) + (q)->q_length) : ((q)->q_wr_ptr - (q)->q_rd_ptr))

/* Return queue slots free count
 *  params:
 *      const q_t *q
 *  returns:
 *      CsrUint16
 */
#define CSR_WIFI_PS_Q_SLOTS_FREE(q)     \
    (((q)->q_length - CSR_WIFI_PS_Q_SLOTS_USED((q))) - 1)

/* Return slot signal data pointer
 *  params:
 *      const q_t *q
 *      CsrUint16 slot
 *  returns:
 *      card_signal_t *
 */
#define CSR_WIFI_PS_Q_SLOT_DATA(q, slot)    \
    ((q)->q_body[slot])

/* Return queue next read slot
 *  params:
 *      const q_t *q
 *  returns:
 *      CsrUint16 slot offset
 */
#define CSR_WIFI_PS_Q_NEXT_R_SLOT(q)    \
    ((q)->q_rd_ptr)

/* Return queue next write slot
 *  params:
 *      const q_t *q
 *  returns:
 *      CsrUint16 slot offset
 */
#define CSR_WIFI_PS_Q_NEXT_W_SLOT(q)    \
    ((q)->q_wr_ptr)

/* Return updated queue pointer wrapped around its length
 *  params:
 *      const q_t *q
 *      CsrUint16 x     amount to add to queue pointer
 *  returns:
 *      CsrUint16 wrapped queue pointer
 */
#define CSR_WIFI_PS_Q_WRAP(q, x)    \
    ((((x) >= (q)->q_length) ? ((x) % (q)->q_length) : (x)))

/* Advance queue read pointer
 *  params:
 *      const q_t *q
 */
#define CSR_WIFI_PS_Q_INC_R(q)  \
    ((q)->q_rd_ptr = CSR_WIFI_PS_Q_WRAP((q), (q)->q_rd_ptr + 1))

/* Advance queue write pointer
 *  params:
 *      const q_t *q
 */
#define CSR_WIFI_PS_Q_INC_W(q)  \
    ((q)->q_wr_ptr = CSR_WIFI_PS_Q_WRAP((q), (q)->q_wr_ptr + 1))


#define ps_circ_take(buf, signal) CSR_WIFI_PS_Q_INC_R(buf)

#define ps_circ_init(buf, len) CSR_WIFI_PS_Q_INIT(buf, len)

#define ps_circ_len(buf) CSR_WIFI_PS_Q_SLOTS_USED(buf)

#define ps_circ_free(buf) CSR_WIFI_PS_Q_SLOTS_FREE(buf)

extern CsrResult ps_circ_add_tail(ps_circ_buffer_t *buf, struct csrWifiHipPacketSchedulerQsig *qsig);
extern struct csrWifiHipPacketSchedulerQsig *ps_circ_peek(ps_circ_buffer_t *buf);

#ifdef __cplusplus
}
#endif
#endif
