/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"
/*
 * ---------------------------------------------------------------------------
 *  FILE:     csr_wifi_hip_card_udi.c
 *
 *  PURPOSE:
 *      Maintain a list of callbacks to log UniFi exchanges to one or more
 *      debug/monitoring client applications.
 *
 * NOTES:
 *      Just call the UDI driver log fn directly for now.
 *      When done properly, each open() on the UDI device will install
 *      a log function. We will call all log fns whenever a signal is written
 *      to or read form the UniFi.
 *
 * ---------------------------------------------------------------------------
 */
#include "csr_wifi_hip_unifi.h"
#include "csr_wifi_hip_card.h"

/*
 * ---------------------------------------------------------------------------
 *  unifi_print_status
 *
 *      Print status info to given character buffer.
 *
 *  Arguments:
 *      None.
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
CsrInt32 unifi_print_status(card_t *card, CsrCharString *str, CsrInt32 *remain)
{
    CsrCharString *p = str;
    CsrUint16 i, n;
    CsrInt32 remaining = *remain;
    CsrInt32 written;
#ifdef CSR_UNSAFE_SDIO
    CsrInt32 iostate;
    CsrResult r;
    static const CsrCharString *const states[] =
    {
        "AWAKE", "DROWSY", "TORPID"
    };
    #define SHARED_READ_RETRY_LIMIT 10
    CsrUint8 b;
    #endif

    if (remaining <= 0)
    {
        return 0;
    }

    i = n = 0;
    written = CsrSnprintf(p, remaining, "Chip ID %u\n",
                          (CsrUint16) card->chip_id);
    UNIFI_SNPRINTF_RET(p, remaining, written);
    written = CsrSnprintf(p, remaining, "Chip Version %04X\n",
                          card->chip_version);
    UNIFI_SNPRINTF_RET(p, remaining, written);
    written = CsrSnprintf(p, remaining, "HIP v%u.%u\n",
                          (card->config_data.version >> 8) & 0xFF,
                          card->config_data.version & 0xFF);
    UNIFI_SNPRINTF_RET(p, remaining, written);
    written = CsrSnprintf(p, remaining, "Build %lu: %s\n",
                          card->build_id, card->build_id_string);
    UNIFI_SNPRINTF_RET(p, remaining, written);

    written = CsrSnprintf(p, remaining, "\nInternal state:\n");
    UNIFI_SNPRINTF_RET(p, remaining, written);

    written = CsrSnprintf(p, remaining, "Last PHY PANIC: %04x:%04x\n",
                          card->last_phy_panic_code, card->last_phy_panic_arg);
    UNIFI_SNPRINTF_RET(p, remaining, written);
    written = CsrSnprintf(p, remaining, "Last MAC PANIC: %04x:%04x\n",
                          card->last_mac_panic_code, card->last_mac_panic_arg);
    UNIFI_SNPRINTF_RET(p, remaining, written);

    /* D-29954: This function must be integrated with the Packet Scheduler. */

    written = CsrSnprintf(p, remaining, "fh data slots free: %u\n",
                          card->from_host_data ? CardGetFreeFromHostDataSlots(card) : 0);
    UNIFI_SNPRINTF_RET(p, remaining, written);


    written = CsrSnprintf(p, remaining, "From host data slots:");
    UNIFI_SNPRINTF_RET(p, remaining, written);
    n = card->config_data.bf_entries;
    for (i = 0; i < n && card->from_host_data; i++)
    {
        written = CsrSnprintf(p, remaining, " %u",
                              (CsrUint16) card->from_host_data[i].bd.data_length);
        UNIFI_SNPRINTF_RET(p, remaining, written);
    }
    written = CsrSnprintf(p, remaining, "\n");
    UNIFI_SNPRINTF_RET(p, remaining, written);

    written = CsrSnprintf(p, remaining, "To host data slots:");
    UNIFI_SNPRINTF_RET(p, remaining, written);
    n = card->config_data.bt_entries;
    for (i = 0; i < n && card->to_host_data; i++)
    {
        written = CsrSnprintf(p, remaining, " %u",
                              (CsrUint16) card->to_host_data[i].data_length);
        UNIFI_SNPRINTF_RET(p, remaining, written);
    }

    written = CsrSnprintf(p, remaining, "\n");
    UNIFI_SNPRINTF_RET(p, remaining, written);

#ifdef CSR_UNSAFE_SDIO_ACCESS
    written = CsrSnprintf(p, remaining, "Host State: %s\n", states[card->host_state]);
    UNIFI_SNPRINTF_RET(p, remaining, written);

    r = unifi_check_io_status(card, &iostate);
    if (iostate == 1)
    {
        written = CsrSnprintf(p, remaining, "I/O Check: F1 disabled\n");
        UNIFI_SNPRINTF_RET(p, remaining, written);
    }
    else
    {
        if (iostate == 1)
        {
            written = CsrSnprintf(p, remaining, "I/O Check: pending interrupt\n");
            UNIFI_SNPRINTF_RET(p, remaining, written);
        }

        written = CsrSnprintf(p, remaining, "BH reason interrupt = %d\n",
                              card->bh_reason_unifi);
        UNIFI_SNPRINTF_RET(p, remaining, written);
        written = CsrSnprintf(p, remaining, "BH reason host      = %d\n",
                              card->bh_reason_host);
        UNIFI_SNPRINTF_RET(p, remaining, written);

        for (i = 0; i < SHARED_READ_RETRY_LIMIT; i++)
        {
            r = unifi_read_8_or_16(card, card->sdio_ctrl_addr + 2, &b);
            if ((r == CSR_RESULT_SUCCESS) && (!(b & 0x80)))
            {
                written = CsrSnprintf(p, remaining, "fhsr: %u (driver thinks is %u)\n",
                                      b, card->from_host_signals_r);
                UNIFI_SNPRINTF_RET(p, remaining, written);
                break;
            }
        }
        iostate = unifi_read_shared_count(card, card->sdio_ctrl_addr + 4);
        written = CsrSnprintf(p, remaining, "thsw: %u (driver thinks is %u)\n",
                              iostate, card->to_host_signals_w);
        UNIFI_SNPRINTF_RET(p, remaining, written);
    }
#endif

    written = CsrSnprintf(p, remaining, "\nStats:\n");
    UNIFI_SNPRINTF_RET(p, remaining, written);
    written = CsrSnprintf(p, remaining, "Total SDIO bytes: R=%lu W=%lu\n",
                          card->sdio_bytes_read, card->sdio_bytes_written);

    UNIFI_SNPRINTF_RET(p, remaining, written);
    written = CsrSnprintf(p, remaining, "Interrupts generated on card: %lu\n",
                          card->unifi_interrupt_seq);
    UNIFI_SNPRINTF_RET(p, remaining, written);

    *remain = remaining;
    return p - str;
} /* unifi_print_status() */
