/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"
/*
 * ---------------------------------------------------------------------------
 * FILE: csr_wifi_hip_card_sdio_mem.c
 *
 * PURPOSE: Implementation of the Card API for SDIO.
 *
 * ---------------------------------------------------------------------------
 */
#include "csr_wifi_hip_log_text.h"
#include "csr_wifi_hip_unifi.h"
#include "csr_wifi_hip_card.h"

#define SDIO_RETRIES    5
#define CSR_WIFI_HIP_SDIO_TRACE_DATA_LENGTH 16

#define MAXILBOX4 0xfb94


#define retryable_sdio_error(_csrResult) (((_csrResult) == CSR_SDIO_RESULT_CRC_ERROR) || ((_csrResult) == CSR_SDIO_RESULT_TIMEOUT))


/*
 * ---------------------------------------------------------------------------
 *  retrying_read8
 *  retrying_write8
 *
 *      These functions provide the first level of retry for SDIO operations.
 *      If an SDIO command fails for reason of a response timeout or CRC
 *      error, it is retried immediately. If three attempts fail we report a
 *      failure.
 *      If the command failed for any other reason, the failure is reported
 *      immediately.
 *
 *  Arguments:
 *      card            Pointer to card structure.
 *      funcnum         The SDIO function to access.
 *                      Function 0 is the Card Configuration Register space,
 *                      function 1/2 is the UniFi register space.
 *      addr            Address to access
 *      pdata           Pointer in which to return the value read.
 *      data            Value to write.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS  on success, non-zero error code on error:
 *      CSR_WIFI_HIP_RESULT_NO_DEVICE  card was ejected
 *      CSR_RESULT_FAILURE     an SDIO error occurred
 * ---------------------------------------------------------------------------
 */
static CsrResult retrying_read8(card_t *card, CsrInt16 funcnum, CsrUint32 addr, CsrUint8 *pdata)
{
    CsrSdioFunction *sdio = card->sdio_if;
    CsrResult r = CSR_RESULT_SUCCESS;
    CsrInt16 retries;
    CsrResult csrResult = CSR_RESULT_SUCCESS;

    retries = 0;
    while (retries++ < SDIO_RETRIES)
    {
        if (funcnum == 0)
        {
#if defined(CSR_WIFI_HIP_DEBUG_OFFLINE_ENABLE) && defined(CSR_WIFI_HIP_SDIO_TRACE_ENABLE)
            unifi_debug_log_to_buf(0, "r0@%02X", addr);
#endif
            csrResult = CsrSdioF0Read8(sdio, addr, pdata);
        }
        else
        {
#ifdef CSR_WIFI_TRANSPORT_CSPI
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: retrying_read_f0_8: F1 8-bit reads are not allowed.\n", card->instance));
            return CSR_RESULT_FAILURE;
#else
#if defined(CSR_WIFI_HIP_DEBUG_OFFLINE_ENABLE) && defined(CSR_WIFI_HIP_SDIO_TRACE_ENABLE)
            unifi_debug_log_to_buf(0, "r@%02X", addr);
#endif
            csrResult = CsrSdioRead8(sdio, addr, pdata);
#endif
        }
#if defined(CSR_WIFI_HIP_DEBUG_OFFLINE_ENABLE) && defined(CSR_WIFI_HIP_SDIO_TRACE_ENABLE)
        if (csrResult != CSR_RESULT_SUCCESS)
        {
            unifi_debug_log_to_buf(1, "error=%X", csrResult);
        }
        else
        {
            unifi_debug_log_to_buf(1, "=%X", *pdata);
        }
#endif
        if (csrResult == CSR_SDIO_RESULT_NO_DEVICE)
        {
            return CSR_WIFI_HIP_RESULT_NO_DEVICE;
        }
        /*
         * Try again for retryable (CRC or TIMEOUT) errors,
         * break on success or fatal error
         */
        if (!retryable_sdio_error(csrResult))
        {
#ifdef CSR_WIFI_HIP_DATA_PLANE_PROFILE_ENABLE
            card->cmd_prof.cmd52_count++;
#endif
            break;
        }
        CSR_LOG_TEXT_INFO((
                              CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG2,  "unifi%d: retryable SDIO error reading F%d 0x%lX\n",
                              card->instance,  funcnum, addr));
    }

    if ((csrResult == CSR_RESULT_SUCCESS) && (retries > 1))
    {
        CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID,
                              CSR_WIFI_HIP_LOG_DEF, "unifi%d: Read succeeded after %d attempts\n",
                              card->instance,  retries));
    }

    if (csrResult != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to read from UniFi (addr 0x%lX) after %d tries\n",
                               card->instance,
                               addr, retries - 1));
        /* Report any SDIO error as a general i/o error */
        r = CSR_RESULT_FAILURE;
    }

    return r;
} /* retrying_read8() */

static CsrResult retrying_write8(card_t *card, CsrInt16 funcnum, CsrUint32 addr, CsrUint8 data)
{
    CsrSdioFunction *sdio = card->sdio_if;
    CsrResult r = CSR_RESULT_SUCCESS;
    CsrInt16 retries;
    CsrResult csrResult = CSR_RESULT_SUCCESS;

    retries = 0;
    while (retries++ < SDIO_RETRIES)
    {
        if (funcnum == 0)
        {
#if defined(CSR_WIFI_HIP_DEBUG_OFFLINE_ENABLE) && defined(CSR_WIFI_HIP_SDIO_TRACE_ENABLE)
            unifi_debug_log_to_buf(0, "w0@%02X=%X", addr, data);
#endif
            csrResult = CsrSdioF0Write8(sdio, addr, data);
        }
        else
        {
#ifdef CSR_WIFI_TRANSPORT_CSPI
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: retrying_write_f0_8: F1 8-bit writes are not allowed.\n", card->instance));
            return CSR_RESULT_FAILURE;
#else
#if defined(CSR_WIFI_HIP_DEBUG_OFFLINE_ENABLE) && defined(CSR_WIFI_HIP_SDIO_TRACE_ENABLE)
            unifi_debug_log_to_buf(0, "w@%02X=%X", addr, data);
#endif
            csrResult = CsrSdioWrite8(sdio, addr, data);
#endif
        }
#if defined(CSR_WIFI_HIP_DEBUG_OFFLINE_ENABLE) && defined(CSR_WIFI_HIP_SDIO_TRACE_ENABLE)
        if (csrResult != CSR_RESULT_SUCCESS)
        {
            unifi_debug_log_to_buf(1, ",error=%X", csrResult);
        }
#endif
        if (csrResult == CSR_SDIO_RESULT_NO_DEVICE)
        {
            return CSR_WIFI_HIP_RESULT_NO_DEVICE;
        }
        /*
         * Try again for retryable (CRC or TIMEOUT) errors,
         * break on success or fatal error
         */
        if (!retryable_sdio_error(csrResult))
        {
#ifdef CSR_WIFI_HIP_DATA_PLANE_PROFILE_ENABLE
            card->cmd_prof.cmd52_count++;
#endif
            break;
        }
        CSR_LOG_TEXT_INFO((
                              CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG2,  "unifi%d: retryable SDIO error writing %02X to F%d 0x%lX\n",
                              card->instance,
                              data, funcnum, addr));
    }

    if ((csrResult == CSR_RESULT_SUCCESS) && (retries > 1))
    {
        CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID,
                              CSR_WIFI_HIP_LOG_DEF, "unifi%d: Write succeeded after %d attempts\n",
                              card->instance,  retries));
    }

    if (csrResult != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to write to UniFi (addr 0x%lX) after %d tries\n",
                               card->instance,
                               addr, retries - 1));
        /* Report any SDIO error as a general i/o error */
        r = CSR_RESULT_FAILURE;
    }

    return r;
} /* retrying_write8() */

#ifndef CSR_WIFI_DRIVER_HYDRA

static CsrResult retrying_read16(card_t *card, CsrInt16 funcnum,
                                 CsrUint32 addr, CsrUint16 *pdata)
{
    CsrSdioFunction *sdio = card->sdio_if;
    CsrResult r = CSR_RESULT_SUCCESS;
    CsrInt16 retries;
    CsrResult csrResult = CSR_RESULT_SUCCESS;

    retries = 0;
    while (retries++ < SDIO_RETRIES)
    {
        csrResult = CsrSdioRead16(sdio, addr, pdata);
        if (csrResult == CSR_SDIO_RESULT_NO_DEVICE)
        {
            return CSR_WIFI_HIP_RESULT_NO_DEVICE;
        }

        /*
         * Try again for retryable (CRC or TIMEOUT) errors,
         * break on success or fatal error
         */
        if (!retryable_sdio_error(csrResult))
        {
#ifdef CSR_WIFI_HIP_DATA_PLANE_PROFILE_ENABLE
            card->cmd_prof.cmd52_count++;
#endif
            break;
        }
        CSR_LOG_TEXT_INFO((
                              CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG2,  "unifi%d: retryable SDIO error reading F%d 0x%lX\n",
                              card->instance,  funcnum, addr));
    }

    if ((csrResult == CSR_RESULT_SUCCESS) && (retries > 1))
    {
        CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID,
                              CSR_WIFI_HIP_LOG_DEF, "unifi%d: Read succeeded after %d attempts\n",
                              card->instance,  retries));
    }

    if (csrResult != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to read from UniFi (addr 0x%lX) after %d tries\n",
                               card->instance,
                               addr, retries - 1));
        /* Report any SDIO error as a general i/o error */
        r = CSR_RESULT_FAILURE;
    }

    return r;
} /* retrying_read16() */

static CsrResult retrying_write16(card_t *card, CsrInt16 funcnum,
                                  CsrUint32 addr, CsrUint16 data)
{
    CsrSdioFunction *sdio = card->sdio_if;
    CsrResult r = CSR_RESULT_SUCCESS;
    CsrInt16 retries;
    CsrResult csrResult = CSR_RESULT_SUCCESS;

    retries = 0;
    while (retries++ < SDIO_RETRIES)
    {
        csrResult = CsrSdioWrite16(sdio, addr, data);
        if (csrResult == CSR_SDIO_RESULT_NO_DEVICE)
        {
            return CSR_WIFI_HIP_RESULT_NO_DEVICE;
        }

        /*
         * Try again for retryable (CRC or TIMEOUT) errors,
         * break on success or fatal error
         */
        if (!retryable_sdio_error(csrResult))
        {
#ifdef CSR_WIFI_HIP_DATA_PLANE_PROFILE_ENABLE
            card->cmd_prof.cmd52_count++;
#endif
            break;
        }
        CSR_LOG_TEXT_INFO((
                              CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG2,  "unifi%d: retryable SDIO error writing %02X to F%d 0x%lX\n",
                              card->instance,
                              data, funcnum, addr));
    }

    if ((csrResult == CSR_RESULT_SUCCESS) && (retries > 1))
    {
        CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID,
                              CSR_WIFI_HIP_LOG_DEF, "unifi%d: Write succeeded after %d attempts\n",
                              card->instance,  retries));
    }

    if (csrResult != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to write to UniFi (addr 0x%lX) after %d tries\n",
                               card->instance,
                               addr, retries - 1));
        /* Report any SDIO error as a general i/o error */
        r = CSR_RESULT_FAILURE;
    }

    return r;
} /* retrying_write16() */

#endif /* !CSR_WIFI_DRIVER_HYDRA */

/*
 * ---------------------------------------------------------------------------
 *  sdio_read_f0
 *
 *      Reads a byte value from the CCCR (func 0) area of UniFi.
 *
 *  Arguments:
 *      card    Pointer to card structure.
 *      addr    Address to read from
 *      pdata   Pointer in which to store the read value.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success, non-zero error code on error:
 *      CSR_WIFI_HIP_RESULT_NO_DEVICE  card was ejected
 *      CSR_RESULT_FAILURE     an SDIO error occurred
 * ---------------------------------------------------------------------------
 */
CsrResult sdio_read_f0(card_t *card, CsrUint32 addr, CsrUint8 *pdata)
{
#if defined(CSR_WIFI_HIP_DEBUG_OFFLINE_ENABLE) && defined(CSR_WIFI_HIP_DATA_PLANE_PROFILE_ENABLE)
    card->cmd_prof.cmd52_f0_r_count++;
#endif
    return retrying_read8(card, 0, addr, pdata);
} /* sdio_read_f0() */

/*
 * ---------------------------------------------------------------------------
 *  sdio_write_f0
 *
 *      Writes a byte value to the CCCR (func 0) area of UniFi.
 *
 *  Arguments:
 *      card    Pointer to card structure.
 *      addr    Address to read from
 *      data    Data value to write.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success, non-zero error code on error:
 *      CSR_WIFI_HIP_RESULT_NO_DEVICE  card was ejected
 *      CSR_RESULT_FAILURE     an SDIO error occurred
 * ---------------------------------------------------------------------------
 */
CsrResult sdio_write_f0(card_t *card, CsrUint32 addr, CsrUint8 data)
{
#if defined(CSR_WIFI_HIP_DEBUG_OFFLINE_ENABLE) && defined(CSR_WIFI_HIP_DATA_PLANE_PROFILE_ENABLE)
    card->cmd_prof.cmd52_f0_w_count++;
#endif
    return retrying_write8(card, 0, addr, data);
} /* sdio_write_f0() */

/*
 * ---------------------------------------------------------------------------
 * unifi_read_direct_8_or_16
 *
 *      Read a 8-bit value from the UniFi SDIO interface.
 *
 *  Arguments:
 *      card    Pointer to card structure.
 *      addr    Address to read from
 *      pdata   Pointer in which to return data.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success, non-zero error code on error:
 * ---------------------------------------------------------------------------
 */
CsrResult unifi_read_direct_8_or_16(card_t *card, CsrUint32 addr, CsrUint8 *pdata)
{
#ifdef CSR_WIFI_DRIVER_HYDRA
    if (!UF_IS_TH_SCRATCH(addr))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: Error: Not a To-Host register 0x%x\n", card->instance, addr));
        return CSR_RESULT_FAILURE;
    }
#endif

#ifdef CSR_WIFI_TRANSPORT_CSPI
    CsrUint16 w;
    CsrResult r;

    r = retrying_read16(card, card->function, addr, &w);
    *pdata = (CsrUint8) (w & 0xFF);
    return r;
#else
    return retrying_read8(card, card->function, addr, pdata);
#endif
} /* unifi_read_direct_8_or_16() */

/*
 * ---------------------------------------------------------------------------
 *  unifi_write_direct_8_or_16
 *
 *      Write a byte value to the UniFi SDIO interface.
 *
 *  Arguments:
 *      card    Pointer to card structure.
 *      addr    Address to write to
 *      data    Value to write.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success, non-zero error code on error
 *
 *  Notes:
 *      If 8-bit write is used, the even address *must* be written second.
 *      This is because writes to odd bytes are cached and not committed
 *      to memory until the preceding even address is written.
 * ---------------------------------------------------------------------------
 */
CsrResult unifi_write_direct_8_or_16(card_t *card, CsrUint32 addr, CsrUint8 data)
{
#ifdef CSR_WIFI_DRIVER_HYDRA
    if (!UF_IS_FH_SCRATCH(addr))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: Error: Not a From-Host register 0x%x\n", card->instance, addr));
        return CSR_RESULT_FAILURE;
    }

    /* On Hydra chips the scratch registers are 8 bits wide, however the hardware
     * consumes the additional 8 bits written, so a 16 bit access works
     */
#else
    if (addr & 1)
    {
        CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID,
                              CSR_WIFI_HIP_LOG_DEF, "unifi%d: Warning: Byte write to an odd address (0x%lX) is dangerous\n",
                              card->instance,
                              addr));
    }
#endif

#ifdef CSR_WIFI_TRANSPORT_CSPI
    return retrying_write16(card, card->function, addr, (CsrUint16) data);
#else
    return retrying_write8(card, card->function, addr, data);
#endif
} /* unifi_write_direct_8_or_16() */

#ifndef CSR_WIFI_DRIVER_HYDRA
/*
 * ---------------------------------------------------------------------------
 *  unifi_read_direct16
 *
 *      Read a 16-bit value from the UniFi SDIO interface.
 *
 *  Arguments:
 *      card    Pointer to card structure.
 *      addr    Address to read from
 *      pdata   Pointer in which to return data.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success, non-zero error code on error:
 *      CSR_WIFI_HIP_RESULT_NO_DEVICE  card was ejected
 *      CSR_RESULT_FAILURE     an SDIO error occurred
 *
 *  Notes:
 *      The even address *must* be read first. This is because reads from
 *      odd bytes are cached and read from memory when the preceding
 *      even address is read.
 * ---------------------------------------------------------------------------
 */
CsrResult unifi_read_direct16(card_t *card, CsrUint32 addr, CsrUint16 *pdata)
{
    return retrying_read16(card, card->function, addr, pdata);
} /* unifi_read_direct16() */

/*
 * ---------------------------------------------------------------------------
 *  unifi_write_direct16
 *
 *      Write a 16-bit value to the UniFi SDIO interface.
 *
 *  Arguments:
 *      card    Pointer to card structure.
 *      addr    Address to write to
 *      data    Value to write.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success, non-zero error code on error:
 *      CSR_WIFI_HIP_RESULT_NO_DEVICE  card was ejected
 *      CSR_RESULT_FAILURE     an SDIO error occurred
 *
 *  Notes:
 *      The even address *must* be written second. This is because writes to
 *      odd bytes are cached and not committed to memory until the preceding
 *      even address is written.
 * ---------------------------------------------------------------------------
 */
CsrResult unifi_write_direct16(card_t *card, CsrUint32 addr, CsrUint16 data)
{
    return retrying_write16(card, card->function, addr, data);
} /* unifi_write_direct16() */

/*
 * ---------------------------------------------------------------------------
 *  unifi_read_direct32
 *
 *      Read a 32-bit value from the UniFi SDIO interface.
 *
 *  Arguments:
 *      card    Pointer to card structure.
 *      addr    Address to read from
 *      pdata   Pointer in which to return data.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success, non-zero error code on error:
 *      CSR_WIFI_HIP_RESULT_NO_DEVICE  card was ejected
 *      CSR_RESULT_FAILURE     an SDIO error occurred
 * ---------------------------------------------------------------------------
 */
CsrResult unifi_read_direct32(card_t *card, CsrUint32 addr, CsrUint32 *pdata)
{
    CsrResult r;
    CsrUint16 w0, w1;

    r = retrying_read16(card, card->function, addr, &w0);
    if (r != CSR_RESULT_SUCCESS)
    {
        return r;
    }

    r = retrying_read16(card, card->function, addr + 2, &w1);
    if (r != CSR_RESULT_SUCCESS)
    {
        return r;
    }

    *pdata = ((CsrUint32) w1 << 16) | (CsrUint32) w0;

    return CSR_RESULT_SUCCESS;
} /* unifi_read_direct32() */

/*
 * ---------------------------------------------------------------------------
 *  unifi_read_directn_match
 *
 *      Read multiple 8-bit values from the UniFi SDIO interface,
 *      stopping when either we have read 'len' bytes or we have read
 *      a octet equal to 'match'.  If 'match' is not a valid octet
 *      then this function is the same as 'unifi_read_directn'.
 *
 *  Arguments:
 *      card            Pointer to card structure.
 *      addr            Start address to read from.
 *      pdata           Pointer to which to write data.
 *      len             Maximum umber of bytes to read
 *      match           The value to stop reading at.
 *      num             Pointer to buffer to write number of bytes read
 *
 *  Returns:
 *      number of octets read on success, negative error code on error:
 *      CSR_WIFI_HIP_RESULT_NO_DEVICE  card was ejected
 *      CSR_RESULT_FAILURE     an SDIO error occurred
 *
 *  Notes:
 *      The even address *must* be read first. This is because reads from
 *      odd bytes are cached and read from memory when the preceding
 *      even address is read.
 * ---------------------------------------------------------------------------
 */
static CsrResult unifi_read_directn_match(card_t *card, CsrUint32 addr, void *pdata, CsrUint16 len, CsrInt8 m, CsrUint32 *num)
{
    CsrResult r;
    CsrUint32 i;
    CsrUint8 *cptr;
    CsrUint16 w;

    *num = 0;

    cptr = (CsrUint8 *) pdata;
    for (i = 0; i < len; i += 2)
    {
        r = retrying_read16(card, card->function, addr, &w);
        if (r != CSR_RESULT_SUCCESS)
        {
            return r;
        }

        *cptr++ = ((CsrUint8) w & 0xFF);
        if ((m >= 0) && (((CsrInt8) w & 0xFF) == m))
        {
            break;
        }

        if (i + 1 == len)
        {
            /* The len is odd. Ignore the last high byte */
            break;
        }

        *cptr++ = ((CsrUint8) (w >> 8) & 0xFF);
        if ((m >= 0) && (((CsrInt8) (w >> 8) & 0xFF) == m))
        {
            break;
        }

        addr += 2;
    }

    *num = (CsrUint32) (cptr - (CsrUint8 *) pdata);
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  unifi_read_directn
 *
 *      Read multiple 8-bit values from the UniFi SDIO interface.
 *
 *  Arguments:
 *      card            Pointer to card structure.
 *      addr            Start address to read from.
 *      pdata           Pointer to which to write data.
 *      len             Number of bytes to read
 *
 *  Returns:
 *      0 on success, non-zero error code on error:
 *      CSR_WIFI_HIP_RESULT_NO_DEVICE  card was ejected
 *      CSR_RESULT_FAILURE     an SDIO error occurred
 *
 *  Notes:
 *      The even address *must* be read first. This is because reads from
 *      odd bytes are cached and read from memory when the preceding
 *      even address is read.
 * ---------------------------------------------------------------------------
 */
CsrResult unifi_read_directn(card_t *card, CsrUint32 addr, void *pdata, CsrUint16 len)
{
    CsrUint32 num;

    return unifi_read_directn_match(card, addr, pdata, len, -1, &num);
} /* unifi_read_directn() */

/*
 * ---------------------------------------------------------------------------
 *  unifi_write_directn
 *
 *      Write multiple 8-bit values to the UniFi SDIO interface.
 *
 *  Arguments:
 *      card            Pointer to card structure.
 *      addr            Start address to write to.
 *      pdata           Source data pointer.
 *      len             Number of bytes to write, must be even.
 *
 *  Returns:
 *      0 on success, non-zero error code on error:
 *      CSR_WIFI_HIP_RESULT_NO_DEVICE  card was ejected
 *      CSR_RESULT_FAILURE     an SDIO error occurred
 *
 *  Notes:
 *      The UniFi has a peculiar 16-bit bus architecture. Writes are only
 *      committed to memory when an even address is accessed. Writes to
 *      odd addresses are cached and only committed if the next write is
 *      to the preceding address.
 *      This means we must write data as pairs of bytes in reverse order.
 * ---------------------------------------------------------------------------
 */
CsrResult unifi_write_directn(card_t *card, CsrUint32 addr, void *pdata, CsrUint16 len)
{
    CsrResult r;
    CsrUint8 *cptr;
    CsrInt16 signed_len;

    cptr = (CsrUint8 *) pdata;
    signed_len = (CsrInt16) len;
    while (signed_len > 0)
    {
        /* This is UniFi-1 specific code. CSPI not supported so 8-bit write allowed */
        r = retrying_write16(card, card->function, addr, *cptr);
        if (r != CSR_RESULT_SUCCESS)
        {
            return r;
        }

        cptr += 2;
        addr += 2;
        signed_len -= 2;
    }

    return CSR_RESULT_SUCCESS;
} /* unifi_write_directn() */

/*
 * ---------------------------------------------------------------------------
 *  set_dmem_page
 *  set_pmem_page
 *
 *      Set up the page register for the shared data memory window or program
 *      memory window.
 *
 *  Arguments:
 *      card            Pointer to card structure.
 *      dmem_addr       UniFi shared-data-memory address to access.
 *      pmem_addr       UniFi program memory address to access. This includes
 *                        External FLASH memory at    0x000000
 *                        Processor program memory at 0x200000
 *                        External SRAM at memory     0x400000
 *      paddr           Location to write an SDIO address (24-bit) for
 *                       use in a unifi_read_direct or unifi_write_direct call.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success
 *      CSR_WIFI_HIP_RESULT_NO_DEVICE card was ejected
 *      CSR_RESULT_FAILURE an SDIO error occurred
 * ---------------------------------------------------------------------------
 */
static CsrResult set_dmem_page(card_t *card, CsrUint32 dmem_addr, CsrUint32 *paddr)
{
    CsrUint16 page, addr;
    CsrUint32 len;
    CsrResult r;

    *paddr = 0;

    if (!ChipHelper_DecodeWindow(card->helper,
                                 CHIP_HELPER_WINDOW_3,
                                 CHIP_HELPER_WT_SHARED,
                                 dmem_addr / 2,
                                 &page, &addr, &len))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to decode SHARED_DMEM_PAGE %08lx\n",
                               card->instance,  dmem_addr));
        return CSR_WIFI_HIP_RESULT_INVALID_VALUE;
    }

    if (page != card->dmem_page)
    {
        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  "unifi%d: setting dmem page=0x%X, addr=0x%lX\n",
                               card->instance,  page, addr));

        /* change page register */
        r = unifi_write_direct16(card, ChipHelper_HOST_WINDOW3_PAGE(card->helper) * 2, page);
        if (r != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to write SHARED_DMEM_PAGE\n", card->instance));
            return r;
        }

        card->dmem_page = page;
    }

    *paddr = ((CsrInt32) addr * 2) + (dmem_addr & 1);

    return CSR_RESULT_SUCCESS;
} /* set_dmem_page() */

static CsrResult set_pmem_page(card_t *card, CsrUint32 pmem_addr,
                               enum chip_helper_window_type mem_type, CsrUint32 *paddr)
{
    CsrUint16 page, addr;
    CsrUint32 len;
    CsrResult r;

    *paddr = 0;

    if (!ChipHelper_DecodeWindow(card->helper,
                                 CHIP_HELPER_WINDOW_2,
                                 mem_type,
                                 pmem_addr / 2,
                                 &page, &addr, &len))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to decode PROG MEM PAGE %08lx %d\n",
                               card->instance,  pmem_addr, mem_type));
        return CSR_WIFI_HIP_RESULT_INVALID_VALUE;
    }

    if (page != card->pmem_page)
    {
        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  "unifi%d: setting pmem page=0x%X, addr=0x%lX\n",
                               card->instance,  page, addr));

        /* change page register */
        r = unifi_write_direct16(card, ChipHelper_HOST_WINDOW2_PAGE(card->helper) * 2, page);
        if (r != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to write PROG MEM PAGE\n", card->instance));
            return r;
        }

        card->pmem_page = page;
    }

    *paddr = ((CsrInt32) addr * 2) + (pmem_addr & 1);

    return CSR_RESULT_SUCCESS;
} /* set_pmem_page() */

/*
 * ---------------------------------------------------------------------------
 *  set_page
 *
 *      Sets up the appropriate page register to access the given address.
 *      Returns the sdio address at which the unifi address can be accessed.
 *
 *  Arguments:
 *      card            Pointer to card structure.
 *      generic_addr    UniFi internal address to access, in Generic Pointer
 *                      format, i.e. top byte is space indicator.
 *      paddr           Location to write page address
 *                          SDIO address (24-bit) for use in a unifi_read_direct or
 *                          unifi_write_direct call
 *
 *  Returns:
 *      CSR_WIFI_HIP_RESULT_NO_DEVICE  card was ejected
 *      CSR_RESULT_FAILURE     an SDIO error occurred
 *      CSR_WIFI_HIP_RESULT_INVALID_VALUE  the address is invalid
 * ---------------------------------------------------------------------------
 */
static CsrResult set_page(card_t *card, CsrUint32 generic_addr, CsrUint32 *paddr)
{
    CsrInt32 space;
    CsrUint32 addr;
    CsrResult r = CSR_RESULT_SUCCESS;

    if (!paddr)
    {
        return CSR_WIFI_HIP_RESULT_INVALID_VALUE;
    }
    *paddr = 0;
    space = UNIFI_GP_SPACE(generic_addr);
    addr = UNIFI_GP_OFFSET(generic_addr);
    switch (space)
    {
        case UNIFI_SH_DMEM:
            /* Shared Data Memory is accessed via the Shared Data Memory window */
            r = set_dmem_page(card, addr, paddr);
            if (r != CSR_RESULT_SUCCESS)
            {
                return r;
            }
            break;

        case UNIFI_EXT_FLASH:
            if (!ChipHelper_HasFlash(card->helper))
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: Bad address space for chip in generic pointer 0x%08lX (helper=0x%x)\n",
                                       card->instance,
                                       generic_addr, card->helper));
                return CSR_WIFI_HIP_RESULT_INVALID_VALUE;
            }
            /* External FLASH is accessed via the Program Memory window */
            r = set_pmem_page(card, addr, CHIP_HELPER_WT_FLASH, paddr);
            break;

        case UNIFI_EXT_SRAM:
            if (!ChipHelper_HasExtSram(card->helper))
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: Bad address space for chip in generic pointer 0x%08l (helper=0x%x)\n",
                                       card->instance,
                                       generic_addr, card->helper));
                return CSR_WIFI_HIP_RESULT_INVALID_VALUE;
            }
            /* External SRAM is accessed via the Program Memory window */
            r = set_pmem_page(card, addr, CHIP_HELPER_WT_EXT_SRAM, paddr);
            break;

        case UNIFI_REGISTERS:
            /* Registers are accessed directly */
            *paddr = addr;
            break;

        case UNIFI_PHY_DMEM:
            r = unifi_set_proc_select(card, UNIFI_PROC_PHY);
            if (r != CSR_RESULT_SUCCESS)
            {
                return r;
            }
            *paddr = ChipHelper_DATA_MEMORY_RAM_OFFSET(card->helper) * 2 + addr;
            break;

        case UNIFI_MAC_DMEM:
            r = unifi_set_proc_select(card, UNIFI_PROC_MAC);
            if (r != CSR_RESULT_SUCCESS)
            {
                return r;
            }
            *paddr = ChipHelper_DATA_MEMORY_RAM_OFFSET(card->helper) * 2 + addr;
            break;

        case UNIFI_BT_DMEM:
            if (!ChipHelper_HasBt(card->helper))
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: Bad address space for chip in generic pointer 0x%08lX (helper=0x%x)\n",
                                       card->instance,
                                       generic_addr, card->helper));
                return CSR_WIFI_HIP_RESULT_INVALID_VALUE;
            }
            r = unifi_set_proc_select(card, UNIFI_PROC_BT);
            if (r != CSR_RESULT_SUCCESS)
            {
                return r;
            }
            *paddr = ChipHelper_DATA_MEMORY_RAM_OFFSET(card->helper) * 2 + addr;
            break;

        case UNIFI_PHY_PMEM:
            r = unifi_set_proc_select(card, UNIFI_PROC_PHY);
            if (r != CSR_RESULT_SUCCESS)
            {
                return r;
            }
            r = set_pmem_page(card, addr, CHIP_HELPER_WT_CODE_RAM, paddr);
            break;

        case UNIFI_MAC_PMEM:
            r = unifi_set_proc_select(card, UNIFI_PROC_MAC);
            if (r != CSR_RESULT_SUCCESS)
            {
                return r;
            }
            r = set_pmem_page(card, addr, CHIP_HELPER_WT_CODE_RAM, paddr);
            break;

        case UNIFI_BT_PMEM:
            if (!ChipHelper_HasBt(card->helper))
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: Bad address space for chip in generic pointer 0x%08lX (helper=0x%x)\n",
                                       card->instance,
                                       generic_addr, card->helper));
                return CSR_WIFI_HIP_RESULT_INVALID_VALUE;
            }
            r = unifi_set_proc_select(card, UNIFI_PROC_BT);
            if (r != CSR_RESULT_SUCCESS)
            {
                return r;
            }
            r = set_pmem_page(card, addr, CHIP_HELPER_WT_CODE_RAM, paddr);
            break;

        case UNIFI_PHY_ROM:
            if (!ChipHelper_HasRom(card->helper))
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: Bad address space for chip in generic pointer 0x%08lX (helper=0x%x)\n",
                                       card->instance,
                                       generic_addr, card->helper));
                return CSR_WIFI_HIP_RESULT_INVALID_VALUE;
            }
            r = unifi_set_proc_select(card, UNIFI_PROC_PHY);
            if (r != CSR_RESULT_SUCCESS)
            {
                return r;
            }
            r = set_pmem_page(card, addr, CHIP_HELPER_WT_ROM, paddr);
            break;

        case UNIFI_MAC_ROM:
            if (!ChipHelper_HasRom(card->helper))
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: Bad address space for chip in generic pointer 0x%08lX (helper=0x%x)\n",
                                       card->instance,
                                       generic_addr, card->helper));
                return CSR_WIFI_HIP_RESULT_INVALID_VALUE;
            }
            r = unifi_set_proc_select(card, UNIFI_PROC_MAC);
            if (r != CSR_RESULT_SUCCESS)
            {
                return r;
            }
            r = set_pmem_page(card, addr, CHIP_HELPER_WT_ROM, paddr);
            break;

        case UNIFI_BT_ROM:
            if (!ChipHelper_HasRom(card->helper) || !ChipHelper_HasBt(card->helper))
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: Bad address space for chip in generic pointer 0x%08lX (helper=0x%x)\n",
                                       card->instance,
                                       generic_addr, card->helper));
                return CSR_WIFI_HIP_RESULT_INVALID_VALUE;
            }
            r = unifi_set_proc_select(card, UNIFI_PROC_BT);
            if (r != CSR_RESULT_SUCCESS)
            {
                return r;
            }
            r = set_pmem_page(card, addr, CHIP_HELPER_WT_ROM, paddr);
            break;

        default:
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: Bad address space %d in generic pointer 0x%08lX (helper=0x%x)\n",
                                   card->instance,
                                   space, generic_addr, card->helper));
            return CSR_WIFI_HIP_RESULT_INVALID_VALUE;
    }

    return r;
} /* set_page() */

/*
 * ---------------------------------------------------------------------------
 *  unifi_set_proc_select
 *
 *
 *  Arguments:
 *      card            Pointer to card structure.
 *      xap_sel         Which XAP core to select
 *
 *  Returns:
 *      0 on success, non-zero error code on error:
 *      CSR_WIFI_HIP_RESULT_NO_DEVICE  card was ejected
 *      CSR_RESULT_FAILURE     an SDIO error occurred
 * ---------------------------------------------------------------------------
 */
CsrResult unifi_set_proc_select(card_t *card, enum unifi_dbg_processors_select xap_sel)
{
    CsrResult r;

    /* Verify the the select value is allowed. */
    switch (xap_sel)
    {
        case UNIFI_PROC_MAC:
        case UNIFI_PROC_PHY:
        case UNIFI_PROC_BOTH:
            break;


        default:
            return CSR_WIFI_HIP_RESULT_INVALID_VALUE;
    }

    if (card->proc_select != (CsrUint32) xap_sel)
    {
        r = unifi_write_direct16(card,
                                 ChipHelper_DBG_HOST_PROC_SELECT(card->helper) * 2,
                                 (CsrUint8) xap_sel);
        if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
        {
            return r;
        }
        if (r != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to write to Proc Select register\n", card->instance));
            return r;
        }

        card->proc_select = (CsrUint32) xap_sel;
    }

    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 * unifi_read_8_or_16
 *
 * Performs a byte read of the given address in shared data memory.
 * Set up the shared data memory page register as required.
 *
 * Arguments:
 * card Pointer to card structure.
 * unifi_addr UniFi shared-data-memory address to access.
 * pdata Pointer to a byte variable for the value read.
 *
 * Returns:
 * CSR_RESULT_SUCCESS on success, non-zero error code on error:
 * CSR_WIFI_HIP_RESULT_NO_DEVICE card was ejected
 * CSR_RESULT_FAILURE an SDIO error occurred
 * CSR_WIFI_HIP_RESULT_INVALID_VALUE a bad generic pointer was specified
 * ---------------------------------------------------------------------------
 */
CsrResult unifi_read_8_or_16(card_t *card, CsrUint32 unifi_addr, CsrUint8 *pdata)
{
    CsrUint32 sdio_addr;
    CsrResult r;
#ifdef CSR_WIFI_TRANSPORT_CSPI
    CsrUint16 w;
#endif

    r = set_page(card, unifi_addr, &sdio_addr);
    if (r != CSR_RESULT_SUCCESS)
    {
        return r;
    }

#if defined(CSR_WIFI_HIP_DEBUG_OFFLINE_ENABLE) && defined(CSR_WIFI_HIP_DATA_PLANE_PROFILE_ENABLE)
    card->cmd_prof.cmd52_r8or16_count++;
#endif
#ifdef CSR_WIFI_TRANSPORT_CSPI
    r = retrying_read16(card, card->function, sdio_addr, &w);
    *pdata = (CsrUint8) (w & 0xFF);
    return r;
#else
    return retrying_read8(card, card->function, sdio_addr, pdata);
#endif
} /* unifi_read_8_or_16() */

/*
 * ---------------------------------------------------------------------------
 * unifi_write_8_or_16
 *
 * Performs a byte write of the given address in shared data memory.
 * Set up the shared data memory page register as required.
 *
 * Arguments:
 * card Pointer to card context struct.
 * unifi_addr UniFi shared-data-memory address to access.
 * data Value to write.
 *
 * Returns:
 * CSR_RESULT_SUCCESS on success, non-zero error code on error:
 * CSR_WIFI_HIP_RESULT_NO_DEVICE card was ejected
 * CSR_RESULT_FAILURE an SDIO error occurred
 * CSR_WIFI_HIP_RESULT_INVALID_VALUE a bad generic pointer was specified
 *
 * Notes:
 * Beware using unifi_write8() because byte writes are not safe on UniFi.
 * Writes to odd bytes are cached, writes to even bytes perform a 16-bit
 * write with the previously cached odd byte.
 * ---------------------------------------------------------------------------
 */
CsrResult unifi_write_8_or_16(card_t *card, CsrUint32 unifi_addr, CsrUint8 data)
{
    CsrUint32 sdio_addr;
    CsrResult r;
#ifdef CSR_WIFI_TRANSPORT_CSPI
    CsrUint16 w;
#endif

    r = set_page(card, unifi_addr, &sdio_addr);
    if (r != CSR_RESULT_SUCCESS)
    {
        return r;
    }

    if (sdio_addr & 1)
    {
        CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID,
                              CSR_WIFI_HIP_LOG_DEF, "unifi%d: Warning: Byte write to an odd address (0x%lX) is dangerous\n",
                              card->instance,
                              sdio_addr));
    }

#if defined(CSR_WIFI_HIP_DEBUG_OFFLINE_ENABLE) && defined(CSR_WIFI_HIP_DATA_PLANE_PROFILE_ENABLE)
    card->cmd_prof.cmd52_w8or16_count++;
#endif
#ifdef CSR_WIFI_TRANSPORT_CSPI
    w = data;
    return retrying_write16(card, card->function, sdio_addr, w);
#else
    return retrying_write8(card, card->function, sdio_addr, data);
#endif
} /* unifi_write_8_or_16() */

/*
 * ---------------------------------------------------------------------------
 *  unifi_card_read16
 *
 *      Performs a 16-bit read of the given address in shared data memory.
 *      Set up the shared data memory page register as required.
 *
 *  Arguments:
 *      card            Pointer to card structure.
 *      unifi_addr      UniFi shared-data-memory address to access.
 *      pdata           Pointer to a 16-bit int variable for the value read.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success, non-zero error code on error:
 *      CSR_WIFI_HIP_RESULT_NO_DEVICE  card was ejected
 *      CSR_RESULT_FAILURE     an SDIO error occurred
 *      CSR_WIFI_HIP_RESULT_INVALID_VALUE  a bad generic pointer was specified
 * ---------------------------------------------------------------------------
 */
CsrResult unifi_card_read16(card_t *card, CsrUint32 unifi_addr, CsrUint16 *pdata)
{
    CsrUint32 sdio_addr;
    CsrResult r;

    r = set_page(card, unifi_addr, &sdio_addr);
    if (r != CSR_RESULT_SUCCESS)
    {
        return r;
    }

#if defined(CSR_WIFI_HIP_DEBUG_OFFLINE_ENABLE) && defined(CSR_WIFI_HIP_DATA_PLANE_PROFILE_ENABLE)
    card->cmd_prof.cmd52_r16_count++;
#endif
    return unifi_read_direct16(card, sdio_addr, pdata);
} /* unifi_card_read16() */

/*
 * ---------------------------------------------------------------------------
 *  unifi_card_write16
 *
 *      Performs a 16-bit write of the given address in shared data memory.
 *      Set up the shared data memory page register as required.
 *
 *  Arguments:
 *      card            Pointer to card structure.
 *      unifi_addr      UniFi shared-data-memory address to access.
 *      pdata           Pointer to a byte variable for the value write.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success, non-zero error code on error:
 *      CSR_WIFI_HIP_RESULT_NO_DEVICE  card was ejected
 *      CSR_RESULT_FAILURE     an SDIO error occurred
 *      CSR_WIFI_HIP_RESULT_INVALID_VALUE  a bad generic pointer was specified
 * ---------------------------------------------------------------------------
 */
CsrResult unifi_card_write16(card_t *card, CsrUint32 unifi_addr, CsrUint16 data)
{
    CsrUint32 sdio_addr;
    CsrResult r;

    r = set_page(card, unifi_addr, &sdio_addr);
    if (r != CSR_RESULT_SUCCESS)
    {
        return r;
    }

#if defined(CSR_WIFI_HIP_DEBUG_OFFLINE_ENABLE) && defined(CSR_WIFI_HIP_DATA_PLANE_PROFILE_ENABLE)
    card->cmd_prof.cmd52_w16_count++;
#endif
    return unifi_write_direct16(card, sdio_addr, data);
} /* unifi_card_write16() */

/*
 * ---------------------------------------------------------------------------
 *  unifi_read32
 *
 *      Performs a 32-bit read of the given address in shared data memory.
 *      Set up the shared data memory page register as required.
 *
 *  Arguments:
 *      card            Pointer to card structure.
 *      unifi_addr      UniFi shared-data-memory address to access.
 *      pdata           Pointer to a int variable for the value read.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success, non-zero error code on error:
 *      CSR_WIFI_HIP_RESULT_NO_DEVICE  card was ejected
 *      CSR_RESULT_FAILURE     an SDIO error occurred
 *      CSR_WIFI_HIP_RESULT_INVALID_VALUE  a bad generic pointer was specified
 * ---------------------------------------------------------------------------
 */
CsrResult unifi_read32(card_t *card, CsrUint32 unifi_addr, CsrUint32 *pdata)
{
    CsrUint32 sdio_addr;
    CsrResult r;

    r = set_page(card, unifi_addr, &sdio_addr);
    if (r != CSR_RESULT_SUCCESS)
    {
        return r;
    }

#if defined(CSR_WIFI_HIP_DEBUG_OFFLINE_ENABLE) && defined(CSR_WIFI_HIP_DATA_PLANE_PROFILE_ENABLE)
    card->cmd_prof.cmd52_r32_count++;
#endif
    return unifi_read_direct32(card, sdio_addr, pdata);
} /* unifi_read32() */

/*
 * ---------------------------------------------------------------------------
 *  unifi_card_readn
 *  unifi_readnz
 *
 *      Read multiple 8-bit values from the UniFi SDIO interface.
 *      This function interprets the address as a GenericPointer as
 *      defined in the UniFi Host Interface Protocol Specification.
 *      The readnz version of this function will stop when it reads a
 *      zero octet.
 *
 *  Arguments:
 *      card            Pointer to card structure.
 *      unifi_addr      UniFi shared-data-memory address to access.
 *      pdata           Pointer to which to write data.
 *      len             Number of bytes to read
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success, non-zero error code on error:
 *      CSR_WIFI_HIP_RESULT_NO_DEVICE  card was ejected
 *      CSR_RESULT_FAILURE     an SDIO error occurred
 *      CSR_WIFI_HIP_RESULT_INVALID_VALUE  a bad generic pointer was specified
 * ---------------------------------------------------------------------------
 */
CsrResult unifi_readn_match(card_t *card, CsrUint32 unifi_addr, void *pdata, CsrUint16 len, CsrInt8 match)
{
    CsrUint32 sdio_addr;
    CsrResult r;
    CsrUint32 num;

    r = set_page(card, unifi_addr, &sdio_addr);
    if (r != CSR_RESULT_SUCCESS)
    {
        return r;
    }

    r = unifi_read_directn_match(card, sdio_addr, pdata, len, match, &num);
    return r;
} /* unifi_readn_match() */

CsrResult unifi_card_readn(card_t *card, CsrUint32 unifi_addr, void *pdata, CsrUint16 len)
{
    return unifi_readn_match(card, unifi_addr, pdata, len, -1);
} /* unifi_card_readn() */

CsrResult unifi_readnz(card_t *card, CsrUint32 unifi_addr, void *pdata, CsrUint16 len)
{
    return unifi_readn_match(card, unifi_addr, pdata, len, 0);
} /* unifi_readnz() */

#endif /* !CSR_WIFI_DRIVER_HYDRA */

/*
 * ---------------------------------------------------------------------------
 *  unifi_read_shared_count
 *
 *      Read signal count locations, checking for an SDIO error.  The
 *      signal count locations only contain a valid number if the
 *      highest bit isn't set.
 *
 *  Arguments:
 *      card            Pointer to card context structure.
 *      addr            Shared-memory address to read.
 *
 *  Returns:
 *      Value read from memory (0-127) or -1 on error
 * ---------------------------------------------------------------------------
 */
CsrInt32 unifi_read_shared_count(card_t *card, CsrUint32 addr)
{
    CsrUint8 b;
    /* I've increased this count, because I have seen cases where
     * there were three reads in a row with the top bit set.  I'm not
     * sure why this might have happened, but I can't see a problem
     * with increasing this limit.  It's better to take a while to
     * recover than to fail. */
#define SHARED_READ_RETRY_LIMIT 10
    CsrInt32 i;

    /*
     * Get the to-host-signals-written count.
     * The top-bit will be set if the firmware was in the process of
     * changing the value, in which case we read again.
     */
    /* Limit the number of repeats so we don't freeze */
    for (i = 0; i < SHARED_READ_RETRY_LIMIT; i++)
    {
        CsrResult r;
#ifdef CSR_WIFI_DRIVER_HYDRA
        r = unifi_read_direct_8_or_16(card, addr, &b);
#else
        r = unifi_read_8_or_16(card, addr, &b);
#endif
        if (r != CSR_RESULT_SUCCESS)
        {
            return -1;
        }
        if (!(b & 0x80))
        {
            /* There is a chance that the MSB may have contained invalid data
             * (overflow) at the time it was read. Therefore mask off the MSB.
             * This avoids a race between driver read and firmware write of the
             * word, the value we need is in the lower 8 bits anway.
             */
            return (CsrInt32) (b & 0xff);
        }
    }

    return -1;                  /* this function has changed in WMM mods */
} /* unifi_read_shared_count() */

#ifndef CSR_WIFI_DRIVER_HYDRA
/*
 * ---------------------------------------------------------------------------
 *  unifi_writen
 *
 *      Write multiple 8-bit values to the UniFi SDIO interface using CMD52
 *      This function interprets the address as a GenericPointer as
 *      defined in the UniFi Host Interface Protocol Specification.
 *
 *  Arguments:
 *      card            Pointer to card structure.
 *      unifi_addr      UniFi shared-data-memory address to access.
 *      pdata           Pointer to which to write data.
 *      len             Number of bytes to write
 *
 *  Returns:
 *      0 on success, non-zero error code on error:
 *      CSR_WIFI_HIP_RESULT_NO_DEVICE  card was ejected
 *      CSR_RESULT_FAILURE     an SDIO error occurred
 *      CSR_WIFI_HIP_RESULT_INVALID_VALUE    an odd length or length too big.
 * ---------------------------------------------------------------------------
 */
CsrResult unifi_writen(card_t *card, CsrUint32 unifi_addr, void *pdata, CsrUint16 len)
{
    CsrUint32 sdio_addr;
    CsrResult r;

    r = set_page(card, unifi_addr, &sdio_addr);
    if (r != CSR_RESULT_SUCCESS)
    {
        return r;
    }

    return unifi_write_directn(card, sdio_addr, pdata, len);
} /* unifi_writen() */

#endif /* !CSR_WIFI_DRIVER_HYDRA */

static CsrResult csr_sdio_block_rw(card_t *card, CsrInt16 funcnum,
                                   CsrUint32 addr, CsrUint8 *pdata,
                                   CsrUint16 count, CsrInt16 dir_is_write)
{
    CsrResult csrResult;

    if (dir_is_write == UNIFI_SDIO_READ)
    {
#if defined(CSR_WIFI_HIP_DEBUG_OFFLINE_ENABLE) && defined(CSR_WIFI_HIP_SDIO_TRACE_ENABLE)
        unifi_debug_log_to_buf(0, "r@%02X#%X=", addr, count);
#endif
#if defined(CSR_WIFI_HIP_DEBUG_OFFLINE_ENABLE) && defined(CSR_WIFI_HIP_DATA_PLANE_PROFILE_ENABLE)
        unifi_debug_string_to_buf(0, "R");
#endif
        csrResult = CsrSdioRead(card->sdio_if, addr, pdata, count);
        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: block_rw: Read %d bytes from handle %d (r=%d)\n",
                               card->instance,  count, addr, csrResult));
#if defined(CSR_WIFI_HIP_DEBUG_OFFLINE_ENABLE) && defined(CSR_WIFI_HIP_DATA_PLANE_PROFILE_ENABLE)
        unifi_debug_string_to_buf(1, "<");
#endif
    }
    else
    {
#if defined(CSR_WIFI_HIP_DEBUG_OFFLINE_ENABLE) && defined(CSR_WIFI_HIP_SDIO_TRACE_ENABLE)
        unifi_debug_log_to_buf(0, "w@%02X#%X=", addr, count);
        unifi_debug_hex_to_buf(pdata, count > CSR_WIFI_HIP_SDIO_TRACE_DATA_LENGTH ? CSR_WIFI_HIP_SDIO_TRACE_DATA_LENGTH : count);
#endif
#if defined(CSR_WIFI_HIP_DEBUG_OFFLINE_ENABLE) && defined(CSR_WIFI_HIP_DATA_PLANE_PROFILE_ENABLE)
        unifi_debug_string_to_buf(0, "W");
#endif
        csrResult = CsrSdioWrite(card->sdio_if, addr, pdata, count);
        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: block_rw: Wrote %d bytes to handle %d (r=%d)\n",
                               card->instance,  count, addr, csrResult));
#if defined(CSR_WIFI_HIP_DEBUG_OFFLINE_ENABLE) && defined(CSR_WIFI_HIP_DATA_PLANE_PROFILE_ENABLE)
        unifi_debug_string_to_buf(1, ">");
#endif
    }
#ifdef CSR_WIFI_HIP_DATA_PLANE_PROFILE_ENABLE
    card->cmd_prof.cmd53_count++;
#endif
#if defined(CSR_WIFI_HIP_DEBUG_OFFLINE_ENABLE) && defined(CSR_WIFI_HIP_SDIO_TRACE_ENABLE)
    if (csrResult != CSR_RESULT_SUCCESS)
    {
        unifi_debug_log_to_buf(1, "error=%X", csrResult);
    }
    else if (dir_is_write == UNIFI_SDIO_READ)
    {
        unifi_debug_hex_to_buf(pdata, count > CSR_WIFI_HIP_SDIO_TRACE_DATA_LENGTH ? CSR_WIFI_HIP_SDIO_TRACE_DATA_LENGTH : count);
    }
#endif
    return csrResult;  /* CSR SDIO (not HIP) error code */
}

#define ENTRY_UNIFI_BULK_RW()
#define RETURN_UNIFI_BULK_RW(r) return r

/*
 * ---------------------------------------------------------------------------
 *  unifi_bulk_rw
 *
 *      Transfer bulk data to or from the UniFi SDIO interface.
 *      This function is used to read or write signals and bulk data.
 *
 *  Arguments:
 *      card            Pointer to card structure.
 *      handle          Value to put in the Register Address field of the CMD53 req.
 *      data            Pointer to data to write.
 *      direction       One of UNIFI_SDIO_READ or UNIFI_SDIO_WRITE
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success, non-zero error code on error:
 *      CSR_WIFI_HIP_RESULT_NO_DEVICE  card was ejected
 *      CSR_RESULT_FAILURE     an SDIO error occurred
 *
 *  Notes:
 *      This function uses SDIO CMD53, which is the block transfer mode.
 * ---------------------------------------------------------------------------
 */
CsrResult unifi_bulk_rw(card_t *card, CsrUint32 handle, void *pdata,
                        CsrUint32 len, CsrUint32 index)
{
#define CMD53_RETRIES 3
    /*
     * Ideally instead of sleeping, we want to busy wait.
     * Currently there is no framework API to do this. When it becomes available,
     * we can use it to busy wait using usecs
     */
#define REWIND_RETRIES          15    /* when REWIND_DELAY==1msec, or 250 when REWIND_DELAY==50usecs */
#define REWIND_POLLING_RETRIES  5
#define REWIND_DELAY            1     /* msec or 50usecs */
    CsrResult csrResult;              /* SDIO error code */
    CsrResult r = CSR_RESULT_SUCCESS; /* HIP error code */
    CsrInt16 retries = CMD53_RETRIES;
    CsrInt16 stat_retries;
    CsrUint16 stat;
    CsrInt16 dump_read;
#ifdef UNIFI_DEBUG
    CsrUint8 *pdata_lsb = ((CsrUint8 *) &pdata) + card->lsb;
#endif
    CsrInt16 direction;
    CsrUint16 expected_seq;
#ifdef CSR_WIFI_HIP_EMULATE_SDIO_ERRORS
    static CsrUint32 fake_error = 0;
    static CsrUint32 fake_tx_error = 0;
    static CsrUint32 fake_div = 20;
#endif

    ENTRY_UNIFI_BULK_RW();

    switch (index)
    {
        case UNIFI_SDIO_READ:
        case UNIFI_SDIO_READ_CONFIG:
        case UNIFI_SDIO_READ_CONTROL:
        case UNIFI_SDIO_READ_TO_HOST:
            direction = UNIFI_SDIO_READ;
            break;
        case UNIFI_SDIO_WRITE_CONFIG:
        case UNIFI_SDIO_WRITE_ACK:
        case UNIFI_SDIO_WRITE_FROM_HOST_PUSHED:
        case UNIFI_SDIO_WRITE_FROM_HOST_PULLED:
        case UNIFI_SDIO_WRITE:
            direction = UNIFI_SDIO_WRITE;
            break;
        default:
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: bulk_rw: index value (%d) out of range\n",
                                   card->instance,  index));
            RETURN_UNIFI_BULK_RW(CSR_RESULT_FAILURE);
    }

    dump_read = 0;
#ifdef UNIFI_DEBUG
    if (*pdata_lsb & 1)
    {
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,
                           CSR_WIFI_HIP_LOG_DEF, "unifi%d: CD53 request on a unaligned buffer (addr: 0x%X) dir %s-Host\n",
                           card->instance,
                           pdata, (direction == UNIFI_SDIO_READ) ? "To" : "From"));
        if (direction == UNIFI_SDIO_WRITE)
        {
            CSR_LOG_TEXT_BUFFER_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, len, pdata, "data"));
        }
        else
        {
            dump_read = 1;
        }
    }
#endif

    /* Defensive checks */
    if (!pdata)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: Null pdata for unifi_bulk_rw() len: %d\n",
                               card->instance,  len));
        RETURN_UNIFI_BULK_RW(CSR_WIFI_HIP_RESULT_INVALID_VALUE);
    }
    if ((len & 1) || (len > 0xffff))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: Impossible CMD53 length requested: %d\n",
                               card->instance,  len));
        RETURN_UNIFI_BULK_RW(CSR_WIFI_HIP_RESULT_INVALID_VALUE);
    }

    for ( ; ; )
    {
        CsrUint16 index_le;
#ifndef CSR_WIFI_DRIVER_HYDRA     /* D-30105: Fix this: Don't retry for Hydra/HIP2 until rewind is implemented */
        CsrUint16 offset, offset_le;
#endif

#ifdef CSR_WIFI_HIP_EMULATE_SDIO_ERRORS
        csrResult = CSR_RESULT_SUCCESS;
        if ((fake_error % fake_div == 0) && (direction == UNIFI_SDIO_WRITE))
        {
            if (fake_tx_error % 2 == 0)
            {
                /*
                 * There is a limit to faking the write error. To test the rewind, we
                 * can not write at all and ask the f/w to rewind the handle to where
                 * we are already.
                 */
                CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi%d: Faking a CMD53 write error for index %d len %d\n", card->instance, index, len));
                csrResult = CSR_SDIO_RESULT_CRC_ERROR;
            }
        }
        if (csrResult == CSR_RESULT_SUCCESS)
#endif
        {
            csrResult = csr_sdio_block_rw(card, card->function, handle,
                                          (CsrUint8 *) pdata, (CsrUint16) len,
                                          direction);
            if (csrResult == CSR_SDIO_RESULT_NO_DEVICE)
            {
                RETURN_UNIFI_BULK_RW(CSR_WIFI_HIP_RESULT_NO_DEVICE);
            }
        }
#ifdef CSR_WIFI_HIP_EMULATE_SDIO_ERRORS
        if ((fake_error % fake_div == 0) && (csrResult == CSR_RESULT_SUCCESS))
        {
            if (direction == UNIFI_SDIO_READ)
            {
                CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi%d: Faking a CMD53 read error for index %d\n", card->instance, index));
                csrResult = CSR_SDIO_RESULT_CRC_ERROR;
            }
            else
            {
                if (fake_tx_error % 2 != 0)
                {
                    CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi%d: Faking a CMD53 write error 2 for index %d pdata=0x%p\n", card->instance, index, pdata));
                    csrResult = CSR_SDIO_RESULT_CRC_ERROR;
                }
            }
        }
        fake_error++;
#endif
        if (csrResult == CSR_RESULT_SUCCESS)
        {
            if (dump_read)
            {
                CSR_LOG_TEXT_BUFFER_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, len, pdata, "data"));
            }
            break;
        }

        /*
         * At this point the SDIO driver should have written the I/O Abort
         * register to notify UniFi that the command has failed.
         * UniFi-1 and UniFi-2 (not UF6xxx) use the same register to store the
         * Deep Sleep State. This means we have to restore the Deep Sleep
         * State (AWAKE in any case since we can not perform a CD53 in any other
         * state) by rewriting the I/O Abort register to its previous value.
         */
        if (card->chip_id <= SDIO_CARD_ID_UNIFI_2)
        {
            (void) unifi_set_host_state(card, UNIFI_HOST_STATE_AWAKE);
        }

        /* If csr_sdio_block_rw() failed in a non-retryable way, or retries exhausted
         * then stop retrying
         */
#ifndef CSR_WIFI_DRIVER_HYDRA     /* D-30105: Fix this: Don't retry for Hydra/HIP2 until rewind is implemented */
        if (!retryable_sdio_error(csrResult))
#endif
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Fatal error in a CMD53 transfer\n", card->instance));
            break;
        }


        /*
         * These happen from time to time, try again
         */
        if (--retries == 0)
        {
            break;
        }

        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,
                               "unifi%d: Error in a CMD53 transfer, retrying index=%d handle=%d len=%d\n",
                               card->instance,
                               index, handle, len));

#ifndef CSR_WIFI_DRIVER_HYDRA     /* D-30105: Fix this: Hydra/HIP2 rewind needs to be implemented */
#ifdef CSR_WIFI_HIP_EMULATE_SDIO_ERRORS
        if (direction == UNIFI_SDIO_WRITE)
        {
            fake_tx_error++;
        }
#endif
        switch (index)
        {
            case UNIFI_SDIO_READ_TO_HOST:
                /* Pages read. */
                offset = card->th_buffer_d.block_ctr;
                break;
            case UNIFI_SDIO_WRITE_FROM_HOST_PULLED:
                /* Pages written. */
                offset = card->fh_pulled_buffer_d.winp / card->config_data.block_round_size;
                break;
            case UNIFI_SDIO_WRITE_FROM_HOST_PUSHED:
                offset = card->fh_buffer_d.fh_pipe_current;
                break;
            case UNIFI_SDIO_WRITE_ACK:
            case UNIFI_SDIO_READ_CONFIG:
            case UNIFI_SDIO_WRITE_CONFIG:
            case UNIFI_SDIO_READ_CONTROL:
            default:
                /* Zero. */
                offset = 0;
                break;
        }
        CSR_COPY_UINT16_TO_LITTLE_ENDIAN(offset, &offset_le);
        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,
                               "unifi%d: Error in a CMD53 transfer: writing 0x%04x (offset=%d) to 0x%08x\n",
                               card->instance, offset_le, offset, MAXILBOX4 * 2));
        r = unifi_write_direct16(card, MAXILBOX4 * 2, offset_le);
        if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_write_direct16(...,0x%x,%d) failed with no device\n",
                                   card->instance,
                                   MAXILBOX4 * 2, offset));
            RETURN_UNIFI_BULK_RW(r);
        }

        /* The transfer failed, rewind and try again */
        CSR_COPY_UINT16_TO_LITTLE_ENDIAN(index, &index_le);
        r = unifi_write_direct16(card, ChipHelper_MAILBOX3(card->helper) * 2, index_le);
#endif
        if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_write_direct16(...,0x%x,%d) failed with no device\n",
                                   card->instance,
                                   ChipHelper_MAILBOX3(card->helper) * 2, index));
            RETURN_UNIFI_BULK_RW(r);
        }
        if (r != CSR_RESULT_SUCCESS)
        {
            /*
             * If we can't even do CMD52 (register read/write) then
             * stop here.
             */
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_write_direct16(...,0x%x,%d) failed with general error\n",
                                   card->instance,
                                   ChipHelper_MAILBOX3(card->helper) * 2, (CsrUint8) (index & 0xff)));
            RETURN_UNIFI_BULK_RW(r);
        }

        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6, "unifi%d: index 0x%04x written to mailbox3\n",
                               card->instance,  index_le));

        /* Signal the UniFi to look for the rewind request. */
        r = CardGenInt(card);
        if (r != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: CardGenInt() failed with %d\n",
                                   card->instance,  r));
            RETURN_UNIFI_BULK_RW(r);
        }

        /* Wait for UniFi to acknowledge the rewind */
        stat_retries = REWIND_RETRIES;

        expected_seq = card->rewind_seq + 1;
        if (expected_seq > 7)
        {
            expected_seq = 1;
        }

        for ( ; ; )
        {
            CsrUint16 seq, stat_le;

            stat_le = 0xdead;
#ifndef CSR_WIFI_DRIVER_HYDRA     /* D-30105: Fix this: Hydra/HIP2 */
            r = unifi_read_direct16(card, ChipHelper_MAILBOX3(card->helper) * 2, &stat_le);
#endif
            if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_read_direct16(...,0x%x,) failed with no device\n",
                                       card->instance,
                                       ChipHelper_MAILBOX3(card->helper) * 2));
                RETURN_UNIFI_BULK_RW(r);
            }
            if (r != CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_read_direct16(...,0x%x,) failed with general error\n",
                                       card->instance,
                                       ChipHelper_MAILBOX3(card->helper) * 2));
                RETURN_UNIFI_BULK_RW(CSR_RESULT_FAILURE);
            }
            stat = CSR_GET_UINT16_FROM_LITTLE_ENDIAN((CsrUint8 *) &stat_le);

            /* sequence number is stored in bits 12-14 */
            seq = (stat >> 4) & 0x7;
            if (expected_seq == seq)
            {
                card->hip2Stats.hip2_sdio_cmd53_errors[index]++;
                card->rewind_seq = seq;
                if ((stat >> 7) & 1)
                {
                    CSR_LOG_TEXT_DEBUG((
                                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4, "unifi%d: bulk_rw: stat=0x%04x, rewind unnecessary\n",
                                           card->instance,  stat));
                    RETURN_UNIFI_BULK_RW(CSR_RESULT_SUCCESS);
                }
                /* F/w has completed rewind. */
                CSR_LOG_TEXT_DEBUG((
                                       CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4, "unifi%d: bulk_rw: rewind completed by f/w (stat=0x%04x), retrying cmd53\n"
                                       , card->instance, stat));
                break;
            }
            if (--stat_retries == 0)
            {
                CSR_LOG_TEXT_DEBUG((
                                       CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_LOG_DEF, "unifi%d: bulk_rw: retries exceeded (stat=0x%04x)\n"
                                       , card->instance, stat));
                csrResult = CSR_RESULT_FAILURE;
                goto error;
            }

            /* Poll for the ack a few times */
            if (stat_retries < REWIND_RETRIES - REWIND_POLLING_RETRIES)
            {
                CsrThreadSleep(REWIND_DELAY);
            }
        }
    }
error:
    /* The call to csr_sdio_block_rw() still failed after retrying */
    if (csrResult != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: Block %s failed for index %d after %d retries\n",
                               card->instance,
                               (direction == UNIFI_SDIO_READ) ? "read" : "write",
                               index,
                               CMD53_RETRIES - retries));
        /* Report any SDIO error as a general i/o error */
        RETURN_UNIFI_BULK_RW(CSR_RESULT_FAILURE);
    }

    /* Collect some stats */
    if (direction == UNIFI_SDIO_READ)
    {
        card->sdio_bytes_read += len;
    }
    else
    {
        card->sdio_bytes_written += len;
    }

    RETURN_UNIFI_BULK_RW(CSR_RESULT_SUCCESS);
} /* unifi_bulk_rw() */

/*
 * ---------------------------------------------------------------------------
 *  unifi_bulk_rw_noretry
 *
 *      Transfer bulk data to or from the UniFi SDIO interface.
 *      This function is used to read or write signals and bulk data.
 *
 *  Arguments:
 *      card            Pointer to card structure.
 *      handle          Value to put in the Register Address field of
 *                      the CMD53 req.
 *      data            Pointer to data to write.
 *      direction       One of UNIFI_SDIO_READ or UNIFI_SDIO_WRITE
 *
 *  Returns:
 *      0 on success, non-zero error code on error:
 *      CSR_WIFI_HIP_RESULT_NO_DEVICE  card was ejected
 *      CSR_RESULT_FAILURE     an SDIO error occurred
 *
 *  Notes:
 *      This function uses SDIO CMD53, which is the block transfer mode.
 * ---------------------------------------------------------------------------
 */
CsrResult unifi_bulk_rw_noretry(card_t *card, CsrUint32 handle, void *pdata,
                                CsrUint32 len, CsrUint32 direction)
{
    CsrResult csrResult;

    csrResult = csr_sdio_block_rw(card, card->function, handle,
                                  (CsrUint8 *) pdata, (CsrUint16) len, direction);
    if (csrResult != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: Block %s failed\n",
                               card->instance,
                               (direction == UNIFI_SDIO_READ) ? "read" : "write"));
        return csrResult;
    }

    return CSR_RESULT_SUCCESS;
} /* unifi_bulk_rw_noretry() */
