/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"
/*
 * ---------------------------------------------------------------------------
 * FILE: csr_wifi_hip_card_sdio.c
 *
 * PURPOSE: Implementation of the Card API for SDIO.
 *
 * NOTES:
 *      CardInit() is called from the SDIO probe callback when a card is
 *      inserted. This performs the basic SDIO initialisation, enabling i/o
 *      etc.
 *
 * ---------------------------------------------------------------------------
 */
#include "csr_wifi_hip_log_text.h"
#include "csr_wifi_hip_core_trace.h"
#include "csr_wifi_hip_unifi.h"
#include "csr_wifi_hip_conversions.h"
#include "csr_wifi_hip_unifiversion.h"

#include "csr_wifi_hip_card.h"
#include "csr_wifi_hip_card_sdio.h"
#include "csr_wifi_hip_chiphelper.h"

#include "csr_wifi_ps_qsig.h"

/*ALSI_DL 20140206*/   
/* Time to wait between attempts to read MAILBOX0 */
/*#define MAILBOX1_TIMEOUT                10*/  /* in millisecs */
/*#define MAILBOX1_ATTEMPTS               200*/ /* 2 seconds */
#define MAILBOX1_TIMEOUT                50  /* in millisecs */
#define MAILBOX1_ATTEMPTS               200 /* 10 seconds */

#define MAILBOX2_TIMEOUT                5   /* in millisecs */
#define MAILBOX2_ATTEMPTS               10  /* 50ms */

#define MAILBOX2_RESET_ATTEMPTS         10
#define MAILBOX2_RESET_TIMEOUT          5   /* in millisecs */

#define RESET_SETTLE_DELAY              25  /* in millisecs */

CSR_LOG_TEXT_HANDLE_DEFINE(CsrWifiHipLto);
static CsrResult card_init_slots(card_t *card);
static CsrResult card_hw_init(card_t *card);
#ifndef CSR_WIFI_DRIVER_HYDRA
static CsrResult firmware_present_in_flash(card_t *card);
static void bootstrap_chip_hw(card_t *card);
static CsrResult card_access_panic(card_t *card);
#endif
static CsrResult unifi_reset_hardware(card_t *card);
static CsrResult unifi_hip_init(card_t *card);
static CsrResult unifi_read_chip_version(card_t *card);

static void card_init_slot_list(card_t *card);

static CsrInt16 card_allocate_slot(card_t *card, CsrUint16 *slot, CsrWifiHipPacketSchedulerQsig *qsig);
static void card_free_slot(card_t *card, CsrUint16 slot);


/*
 * ---------------------------------------------------------------------------
 *  unifi_alloc_card
 *
 *      Allocate and initialise the card context structure.
 *
 *  Arguments:
 *      sdio            Pointer to SDIO context pointer to pass to low
 *                      level i/o functions.
 *      ospriv          Pointer to O/S private struct to pass when calling
 *                      callbacks to the higher level system.
 *
 *  Returns:
 *      Pointer to card struct, which represents the driver context or
 *      NULL if the allocation failed.
 * ---------------------------------------------------------------------------
 */
card_t *unifi_alloc_card(CsrSdioFunction *sdio, void *ospriv)
{
    card_t *card;
#ifdef CSR_LOG_ENABLE
    static const CsrCharString *subOrigins[] = CSR_WIFI_HIP_CORE_SUBORIGINS;
#endif

    func_enter();

#ifdef CSR_LOG_ENABLE
    CsrLogTextRegister2(&CsrWifiHipLto, "HIP", CSR_ARRAY_SIZE(subOrigins), subOrigins);
#endif
    card = (card_t *) CsrMemAlloc(sizeof(card_t));
    if (card == NULL)
    {
        return NULL;
    }
    CsrMemSet(card, 0, sizeof(card_t));

    card->sdio_if = sdio;
    card->ospriv = ospriv;
    card->instance = 0; /* for now */

    card->unifi_interrupt_seq = 1;

    /* Make these invalid. */
    card->proc_select = (CsrUint32) (-1);
    card->dmem_page = (CsrUint32) (-1);
    card->pmem_page = (CsrUint32) (-1);

    card->bh_reason_host = 0;
    card->bh_reason_unifi = 0;
    card->bh_reason_unifi_int = 0;

    card->memory_resources_allocated = 0;

    card->low_power_mode = UNIFI_LOW_POWER_DISABLED;
    card->periodic_wake_mode = UNIFI_PERIODIC_WAKE_HOST_DISABLED;

    card->host_state = UNIFI_HOST_STATE_AWAKE;
    card->intmode = CSR_WIFI_INTMODE_DEFAULT;

    /*
     * Memory resources for buffers are allocated when the chip is initialised
     * because we need configuration information from the firmware.
     */

#ifdef UNIFI_DEBUG
    /* Determine offset of LSB in pointer for later alignment sanity check.
     * Synergy integer types have specific widths, which cause compiler
     * warnings when casting pointer types, e.g. on 64-bit systems.
     */
    {
        CsrUint32 val = 0x01234567;

        if (*((CsrUint8 *) &val) == 0x01)
        {
            card->lsb = sizeof(void *) - 1;     /* BE */
        }
        else
        {
            card->lsb = 0;                      /* LE */
        }
    }
#endif
    func_exit();
    return card;
} /* unifi_alloc_card() */

/*
 * ---------------------------------------------------------------------------
 *  unifi_set_segment_size
 *
 *      Reset the hardware and perform HIP initialization
 *
 *  Arguments:
 *      card        Pointer to card struct
 *
 *  Returns:
 *      CsrResult code
 *      CSR_RESULT_SUCCESS if successful
 * ---------------------------------------------------------------------------
 */
void unifi_set_segment_size(card_t *card)
{
    if (card->config_params.tx_window_segment_size)
    {
        card->fh_buffer_d.segment_size = card->buff_maximum_size_bytes * card->config_params.tx_window_segment_size;
        if (card->fh_buffer_d.segment_size > card->fh_buffer_d.max_window)
        {
            card->fh_buffer_d.segment_size = card->fh_buffer_d.max_window;
        }
    }
    else
    {
        /* By default we split the fh-pushed window into 2 segments. On UF6xxx, one segment will be
         * the same size as an 8K h/w buffer. On Hydra, h/w buffers are 64K, but the window max be
         * limited to less than this by the push_maximum_size config param.
         */
        card->fh_buffer_d.segment_size = card->fh_buffer_d.max_window / CSR_WIFI_HIP_NUM_FH_PUSHED_SEGMENTS;
    }
    CSR_LOG_TEXT_DEBUG((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_INIT, "unifi%d: Tx segment size=%d\n",
                        card->instance,  card->fh_buffer_d.segment_size));
}

/*
 * ---------------------------------------------------------------------------
 *  unifi_init_card
 *
 *      Reset the hardware and perform HIP initialization
 *
 *  Arguments:
 *      card        Pointer to card struct
 *
 *  Returns:
 *      CsrResult code
 *      CSR_RESULT_SUCCESS if successful
 * ---------------------------------------------------------------------------
 */
CsrResult unifi_init_card(card_t *card, CsrWifiHipCardParams *config_params)
{
    CsrResult r;

    func_enter();

    if (card == NULL)
    {
        func_exit_r(CSR_WIFI_HIP_RESULT_INVALID_VALUE);
        return CSR_WIFI_HIP_RESULT_INVALID_VALUE;
    }

    card->zero_copy = config_params->zero_copy;
    card->packetFilter = config_params->minVifDurationPacketFilter | card->ta_sampling.packet_filter;
    card->config_params = *config_params;
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,
                        CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_init_card: config_params.zero_copy=%d\n",
                        card->instance,  card->config_params.zero_copy));

    r = CSR_WIFI_HIP_SPINLOCK_CREATE(&card->fh_interrupt_lock);
    if (r != CSR_RESULT_SUCCESS)
    {
        func_exit_r(r);
        return r;
    }

    r = CSR_WIFI_HIP_SPINLOCK_CREATE(&card->fh_count_lock);
    if (r != CSR_RESULT_SUCCESS)
    {
        func_exit_r(r);
        return r;
    }

    r = unifi_init(card);
    if (r != CSR_RESULT_SUCCESS)
    {
        func_exit_r(r);
        return r;
    }

    r = unifi_hip_init(card);
    if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
    {
        func_exit_r(r);
        return r;
    }
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to start host protocol.\n", card->instance));
        func_exit_r(r);
        return r;
    }

    func_exit();
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  unifi_init
 *
 *      Init the hardware.
 *
 *  Arguments:
 *      card        Pointer to card struct
 *
 *  Returns:
 *      CsrResult code
 *      CSR_RESULT_SUCCESS if successful
 * ---------------------------------------------------------------------------
 */
CsrResult unifi_init(card_t *card)
{
    CsrResult r;
    CsrResult csrResult;

    func_enter();

    if (card == NULL)
    {
        func_exit_r(CSR_WIFI_HIP_RESULT_INVALID_VALUE);
        return CSR_WIFI_HIP_RESULT_INVALID_VALUE;
    }

    /*
     * Disable the SDIO interrupts while initialising UniFi.
     * Re-enable them when f/w is running.
     */
    csrResult = CsrSdioInterruptDisable(card->sdio_if);
    if (csrResult == CSR_SDIO_RESULT_NO_DEVICE)
    {
        return CSR_WIFI_HIP_RESULT_NO_DEVICE;
    }

#ifndef CSR_WIFI_DRIVER_HYDRA
    /*
     * UniFi's PLL may start with a slow clock (~ 1 MHz) so initially
     * set the SDIO bus clock to a similar value or SDIO accesses may
     * fail.
     */
    csrResult = CsrSdioMaxBusClockFrequencySet(card->sdio_if, UNIFI_SDIO_CLOCK_SAFE_HZ);
    if (csrResult != CSR_RESULT_SUCCESS)
    {
        r = ConvertCsrSdioToCsrHipResult(card, csrResult);
        func_exit_r(r);
        return r;
    }
    card->sdio_clock_speed = UNIFI_SDIO_CLOCK_SAFE_HZ;
#else
    /* On Hydra (multifunction) chips use the system wide default */
    card->sdio_clock_speed = UNIFI_SDIO_CLOCK_MAX_HZ;
#endif

    /*
     * Reset UniFi. Note, this only resets the WLAN function part of the chip,
     * the SDIO interface is not reset.
     */
    CSR_LOG_TEXT_INFO((
                          CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: Resetting UniFi\n"
                          , card->instance));
    r = unifi_reset_hardware(card);
    if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
    {
        return r;
    }
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to reset UniFi\n", card->instance));
        func_exit_r(r);
        return r;
    }

    /* Initialise the power save mode values. Power mode is configured later with the bottom half running. */
    card->low_power_mode = UNIFI_LOW_POWER_DISABLED;
    card->periodic_wake_mode = UNIFI_PERIODIC_WAKE_HOST_DISABLED;

#ifndef CSR_WIFI_DRIVER_HYDRA
    /*
     * Set initial value of page registers.
     * The page registers will be maintained by unifi_read...() and
     * unifi_write...().
     */
    card->proc_select = (CsrUint32) (-1);
    card->dmem_page = (CsrUint32) (-1);
    card->pmem_page = (CsrUint32) (-1);
    r = unifi_write_direct16(card, ChipHelper_HOST_WINDOW3_PAGE(card->helper) * 2, 0);
    if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
    {
        return r;
    }
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to write SHARED_DMEM_PAGE\n", card->instance));
        func_exit_r(r);
        return r;
    }
    r = unifi_write_direct16(card, ChipHelper_HOST_WINDOW2_PAGE(card->helper) * 2, 0);
    if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
    {
        return r;
    }
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to write PROG_MEM2_PAGE\n", card->instance));
        func_exit_r(r);
        return r;
    }

    /*
     * Probe to see if the UniFi has ROM/flash to boot from. CSR6xxx should do.
     */
    r = firmware_present_in_flash(card);
    if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
    {
        return r;
    }
    if (r == CSR_WIFI_HIP_RESULT_NOT_FOUND)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: No firmware found\n", card->instance));
    }
    else if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Probe for Flash failed\n", card->instance));
    }
#endif

    func_exit_r(r);
    return r;
} /* unifi_init() */

/*
 * ---------------------------------------------------------------------------
 *  unifi_download
 *
 *      Load the firmware.
 *
 *  Arguments:
 *      card        Pointer to card struct
 *      led_mask    Loader LED mask
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success
 *      CsrResult error code on failure.
 * ---------------------------------------------------------------------------
 */
CsrResult unifi_download(card_t *card, CsrInt32 led_mask)
{
#ifndef CSR_WIFI_DRIVER_HYDRA
    CsrResult r;
    void *dlpriv;

    func_enter();

    if (card == NULL)
    {
        func_exit_r(CSR_WIFI_HIP_RESULT_INVALID_VALUE);
        return CSR_WIFI_HIP_RESULT_INVALID_VALUE;
    }

    /* Set the loader led mask */
    card->loader_led_mask = led_mask;

    /* Get the firmware file information */
    CSR_LOG_TEXT_INFO((
                          CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: downloading firmware...\n"
                          , card->instance));

    dlpriv = unifi_dl_fw_read_start(card, UNIFI_FW_STA);
    if (dlpriv == NULL)
    {
        func_exit_r(CSR_WIFI_HIP_RESULT_NOT_FOUND);
        return CSR_WIFI_HIP_RESULT_NOT_FOUND;
    }

    /* Download the firmware. */
    r = unifi_dl_firmware(card, dlpriv);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to download firmware\n", card->instance));
        func_exit_r(r);
        return r;
    }

    /* Free the firmware file information. */
    unifi_fw_read_stop(card->ospriv, dlpriv);

    func_exit();
#else
    /* Hydra chips have the firmware download managed by the Curator, and
     * the firmware will always be running and patched before the driver
     * enables the SDIO function.
     */
#endif
    return CSR_RESULT_SUCCESS;
} /* unifi_download() */

/*
 * ---------------------------------------------------------------------------
 *  unifi_hip_init
 *
 *      This function performs the f/w initialisation sequence as described
 *      in the Unifi Host Interface Protocol Specification.
 *      It allocates memory for host-side slot- data and signal queues.
 *
 *  Arguments:
 *      card        Pointer to card struct
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success or else a CSR error code
 *
 *  Notes:
 *      The firmware must have been downloaded.
 * ---------------------------------------------------------------------------
 */
static CsrResult unifi_hip_init(card_t *card)
{
    CsrResult r;

    func_enter();

    r = card_hw_init(card);
    if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
    {
        return r;
    }
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to establish communication with UniFi\n", card->instance));
        func_exit_r(r);
        return r;
    }

    /*
     * Allocate memory for host-side slot data and signal queues.
     * We need the config info read from the firmware to know how much
     * memory to allocate.
     */
    r = card_init_slots(card);
    if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
    {
        return r;
    }
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: Init slots failed: %d\n",
                               card->instance,  r));
        func_exit_r(r);
        return r;
    }

    CSR_LOG_TEXT_INFO((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_INIT, "unifi%d: Sending first UniFi interrupt\n"
                       , card->instance));

    r = unifi_set_host_state(card, UNIFI_HOST_STATE_AWAKE);
    if (r != CSR_RESULT_SUCCESS)
    {
        func_exit_r(r);
        return r;
    }

    /* Signal the UniFi to start handling messages */
    r = CardGenInt(card);
    if (r != CSR_RESULT_SUCCESS)
    {
        func_exit_r(r);
        return r;
    }

    func_exit();

    return CSR_RESULT_SUCCESS;
} /* unifi_hip_init() */

/*
 * ---------------------------------------------------------------------------
 *  unpack_sdio_config_data
 *
 *      Unpack the SDIO configuration information from a buffer read from
 *      UniFi into a host structure.
 *
 *
 *  Arguments:
 *      card            Pointer to card struct
 *      cfg_data        Destination structure to unpack into.
 *      cfg_data_buf    Source buffer to read from. This should be the raw
 *                      data read from UniFi.
 *
 *  Returns:
 *      CSR_RESULT_SUCEESS on success,
 *      a CSR error code on failure
 * ---------------------------------------------------------------------------
 */
static CsrResult unpack_sdio_config_data(card_t *card, sdio_config_data_t *cfg_data, const CsrUint8 *cfg_data_buf)
{
    const CsrUint8 *buf_start = cfg_data_buf;
    CsrResult r;
    CsrUint32 i;

    CSR_CONVERT_32_FROM_XAP(cfg_data->signature, cfg_data_buf);

    CSR_CONVERT_16_FROM_XAP(cfg_data->length, cfg_data_buf);

    CSR_CONVERT_16_FROM_XAP(cfg_data->version, cfg_data_buf);

    CSR_CONVERT_16_FROM_XAP(cfg_data->sub_version, cfg_data_buf);

    CSR_CONVERT_32_FROM_XAP(cfg_data->fw_version, cfg_data_buf);

    CSR_CONVERT_32_FROM_XAP(cfg_data->patch_version, cfg_data_buf);

    CSR_CONVERT_16_FROM_XAP(cfg_data->signal_chunk_size, cfg_data_buf);

    CSR_CONVERT_16_FROM_XAP(cfg_data->block_round_size, cfg_data_buf);

    CSR_CONVERT_16_FROM_XAP(cfg_data->ack_record_size, cfg_data_buf);

    CSR_CONVERT_16_FROM_XAP(cfg_data->ack_buffer_blocks, cfg_data_buf);

    CSR_CONVERT_16_FROM_XAP(cfg_data->ctrl_info_size, cfg_data_buf);

    CSR_CONVERT_16_FROM_XAP(cfg_data->ctrl_info_blocks, cfg_data_buf);

    CSR_CONVERT_16_FROM_XAP(cfg_data->buff_maximum_size_blocks, cfg_data_buf);

    CSR_CONVERT_16_FROM_XAP(cfg_data->push_maximum_size, cfg_data_buf);

    CSR_CONVERT_16_FROM_XAP(cfg_data->th_ctrl_round, cfg_data_buf);

    CSR_CONVERT_16_FROM_XAP(cfg_data->th_bulk_round, cfg_data_buf);

    CSR_CONVERT_16_FROM_XAP(cfg_data->fh_pulled_round, cfg_data_buf);

    CSR_CONVERT_16_FROM_XAP(cfg_data->fh_pushed_round, cfg_data_buf);

    CSR_CONVERT_16_FROM_XAP(cfg_data->fh_pushed_block_gap, cfg_data_buf);

    CSR_CONVERT_16_FROM_XAP(cfg_data->bt_entries, cfg_data_buf);

    CSR_CONVERT_16_FROM_XAP(cfg_data->bf_entries, cfg_data_buf);

    CSR_CONVERT_16_FROM_XAP(cfg_data->config_response_buffer_h, cfg_data_buf);

    CSR_CONVERT_16_FROM_XAP(cfg_data->ctrl_buffer_h, cfg_data_buf);

    CSR_CONVERT_16_FROM_XAP(cfg_data->ctrl_re_read_buffer_h, cfg_data_buf);

    CSR_CONVERT_16_FROM_XAP(cfg_data->ack_buffer_h, cfg_data_buf);

    for (i = 0;
         i < SDIO_FH_PUSHED_HANDLES && i < sizeof(cfg_data->fh_pushed_buffer_h) / sizeof(cfg_data->fh_pushed_buffer_h[0]);
         i++)
    {
        CSR_CONVERT_16_FROM_XAP(cfg_data->fh_pushed_buffer_h[i], cfg_data_buf);

        card->hip2Stats.fh_pushed_buffer_h[i] = cfg_data->fh_pushed_buffer_h[i];
    }

    /* The f/w can store log information in this handle. */
    CSR_CONVERT_16_FROM_XAP(cfg_data->log_buffer_h, cfg_data_buf);

    card->hip2Stats.ctrl_buffer_h = cfg_data->ctrl_buffer_h;
    card->hip2Stats.ctrl_re_read_buffer_h = cfg_data->ctrl_re_read_buffer_h;
    card->hip2Stats.ack_buffer_h = cfg_data->ack_buffer_h;

    /*
     * check sanity of structure
     *
     * fw reports length in bytes
     */
    if ((cfg_data->length) == (cfg_data_buf - buf_start))
    {
        r = CSR_RESULT_SUCCESS;
    }
    else
    {
        /*
         * The fw's view of the structure is different to ours, this is
         * probably bad
         */
        r = CSR_RESULT_FAILURE;
        CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID,
                            CSR_WIFI_HIP_LOG_DEF, "unifi%d: configuration structure size mis-match: len=%d (read %d)\n",
                            card->instance,  cfg_data->length, cfg_data_buf - buf_start));
    }

    return r;
}

/*
 * ---------------------------------------------------------------------------
 *  pack_sdio_config_data
 *
 *      pack the SDIO configuration information from a UniFi a host
 *      structure into a buffer
 *
 *  Arguments:
 *      card         Pointer to card struct
 *      cfg_data     Source buffer to read from.
 *      cfg_data_buf Destination structure to pack into. This should be the raw
 *                   data to write to UniFi.
 *
 * Returns:
 *      CSR_RESULT_SUCEESS on success,
 *      a CSR error code on failure
 * ---------------------------------------------------------------------------
 */
static void pack_sdio_config_data(card_t *card, sdio_config_data_t *cfg_data, CsrUint8 *cfg_data_buf)
{
    CsrUint32 i;

    CSR_ADD_UINT32_TO_XAP(cfg_data_buf, cfg_data->signature);

    CSR_ADD_UINT16_TO_XAP(cfg_data_buf, cfg_data->length);

    CSR_ADD_UINT16_TO_XAP(cfg_data_buf, cfg_data->version);

    CSR_ADD_UINT16_TO_XAP(cfg_data_buf, cfg_data->sub_version);

    CSR_ADD_UINT32_TO_XAP(cfg_data_buf, cfg_data->fw_version);

    CSR_ADD_UINT32_TO_XAP(cfg_data_buf, cfg_data->patch_version);

    CSR_ADD_UINT16_TO_XAP(cfg_data_buf, cfg_data->signal_chunk_size);

    CSR_ADD_UINT16_TO_XAP(cfg_data_buf, cfg_data->block_round_size);

    CSR_ADD_UINT16_TO_XAP(cfg_data_buf, cfg_data->ack_record_size);

    CSR_ADD_UINT16_TO_XAP(cfg_data_buf, cfg_data->ack_buffer_blocks);

    CSR_ADD_UINT16_TO_XAP(cfg_data_buf, cfg_data->ctrl_info_size);

    CSR_ADD_UINT16_TO_XAP(cfg_data_buf, cfg_data->ctrl_info_blocks);

    CSR_ADD_UINT16_TO_XAP(cfg_data_buf, cfg_data->buff_maximum_size_blocks);

    CSR_ADD_UINT16_TO_XAP(cfg_data_buf, cfg_data->push_maximum_size);

    CSR_ADD_UINT16_TO_XAP(cfg_data_buf, cfg_data->th_ctrl_round);

    CSR_ADD_UINT16_TO_XAP(cfg_data_buf, cfg_data->th_bulk_round);

    CSR_ADD_UINT16_TO_XAP(cfg_data_buf, cfg_data->fh_pulled_round);

    CSR_ADD_UINT16_TO_XAP(cfg_data_buf, cfg_data->fh_pushed_round);

    CSR_ADD_UINT16_TO_XAP(cfg_data_buf, cfg_data->fh_pushed_block_gap);

    CSR_ADD_UINT16_TO_XAP(cfg_data_buf, cfg_data->bt_entries);

    CSR_ADD_UINT16_TO_XAP(cfg_data_buf, cfg_data->bf_entries);

    CSR_ADD_UINT16_TO_XAP(cfg_data_buf, cfg_data->config_response_buffer_h);

    CSR_ADD_UINT16_TO_XAP(cfg_data_buf, cfg_data->ctrl_buffer_h);

    CSR_ADD_UINT16_TO_XAP(cfg_data_buf, cfg_data->ctrl_re_read_buffer_h);

    CSR_ADD_UINT16_TO_XAP(cfg_data_buf, cfg_data->ack_buffer_h);

    for (i = 0; i < SDIO_FH_PUSHED_HANDLES; i++)
    {
        CSR_ADD_UINT16_TO_XAP(cfg_data_buf, cfg_data->fh_pushed_buffer_h[i]);
    }

    CSR_ADD_UINT16_TO_XAP(cfg_data_buf, cfg_data->log_buffer_h);
}

/*
 * - Function ----------------------------------------------------------------
 * card_hw_init()
 *
 *      Perform the initialisation procedure described in the UniFi Host
 *      Interface Protocol document (section 3.3.8) and read the run-time
 *      configuration information from the UniFi. This is stuff like number
 *      of bulk data slots etc.
 *
 *      The card enumeration and SD initialisation has already been done by
 *      the SDIO library, see card_sdio_init().
 *
 *      The initialisation is done when firmware is ready, i.e. this may need
 *      to be called after a f/w download operation.
 *
 *      The initialisation procedure goes like this:
 *       - Wait for UniFi to start-up by polling SHARED_MAILBOX1
 *       - Find the symbol table and look up SLT_SDIO_SLOT_CONFIG
 *       - Read the config structure
 *       - Check the "SDIO initialised" flag, if not zero do a h/w reset and
 *         start again
 *       - Decide the number of bulk data slots to allocate, allocate them and
 *         set "SDIO initialised" flag (and generate an interrupt) to say so.
 *
 * Arguments:
 *      card        Pointer to card struct
 *
 * Returns:
 *      CSR_RESULT_SUCEESS on success,
 *      a CSR error code on failure
 *
 * Notes:
 *      All data in the f/w is stored in a little endian format, without any
 *      padding bytes. Every read from this memory has to be transformed in
 *      host (cpu specific) format, before it is stored in driver's parameters
 *      or/and structures. Athough unifi_card_read16() and unifi_read32() do perform
 *      the convertion internally, unifi_readn() does not.
 * ---------------------------------------------------------------------------
 */
static CsrResult card_hw_init(card_t *card)
{
    sdio_config_data_t *cfg_data = NULL;
    CsrResult r;
    CsrInt16 ver_major, ver_minor;
#ifndef CSR_WIFI_DRIVER_HYDRA
    CsrUint16 mbox3 = 0;
    CsrUint32 slut_address = 0;
    CsrUint16 finger_print;
    void *dlpriv;
    symbol_t slut;
    CsrInt16 search_4slut_again;
    CsrResult csrResult;
#else
    CsrUint8 mbox3 = 0;
#endif

    func_enter();

    /*
     * The device revision from the TPLMID_MANF and TPLMID_CARD fields
     * of the CIS are available as
     *   card->sdio_if->pDevice->ManfID
     *   card->sdio_if->pDevice->AppID
     */
#ifndef CSR_WIFI_DRIVER_HYDRA
    /*
     * Run in a loop so we can patch.
     */
    do
    {
        /* Reset these each time around the loop. */
        search_4slut_again = 0;
        cfg_data = NULL;

        r = card_wait_for_firmware_to_start(card, &slut_address);
        if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
        {
            return r;
        }
        if (r != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Firmware hasn't started\n", card->instance));
            func_exit_r(r);
            return r;
        }
        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: SLUT addr 0x%lX\n",
                               card->instance,  slut_address));

        /*
         * Firmware has started, but doesn't know full clock configuration yet
         * as some of the information may be in the MIB. Therefore we set an
         * initial SDIO clock speed, faster than UNIFI_SDIO_CLOCK_SAFE_HZ, for
         * the patch download and subsequent firmware initialisation, and
         * full speed UNIFI_SDIO_CLOCK_MAX_HZ will be set once the f/w tells us
         * that it is ready.
         */
        csrResult = CsrSdioMaxBusClockFrequencySet(card->sdio_if, UNIFI_SDIO_CLOCK_INIT_HZ);
        if (csrResult != CSR_RESULT_SUCCESS)
        {
            r = ConvertCsrSdioToCsrHipResult(card, csrResult);
            func_exit_r(r);
            return r;
        }
        card->sdio_clock_speed = UNIFI_SDIO_CLOCK_INIT_HZ;

        /*
         * Check the SLUT fingerprint.
         * The slut_address is a generic pointer so we must use unifi_card_read16().
         */
        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: Looking for SLUT finger print\n"
                               , card->instance));
        finger_print = 0;
        r = unifi_card_read16(card, slut_address, &finger_print);
        if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
        {
            return r;
        }
        if (r != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to read SLUT finger print\n", card->instance));
            func_exit_r(r);
            return r;
        }

        if (finger_print != SLUT_FINGERPRINT)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to find Symbol lookup table fingerprint\n", card->instance));
            func_exit_r(CSR_RESULT_FAILURE);
            return CSR_RESULT_FAILURE;
        }

        /* Symbol table starts imedately after the fingerprint */
        slut_address += 2;

        /* Search the table until either the end marker is found, or the
         * loading of patch firmware invalidates the current table.
         */
        while (!search_4slut_again)
        {
            CsrUint16 s;
            CsrUint32 l;

            r = unifi_card_read16(card, slut_address, &s);
            if (r != CSR_RESULT_SUCCESS)
            {
                func_exit_r(r);
                return r;
            }
            slut_address += 2;

            if (s == CSR_SLT_END)
            {
                CSR_LOG_TEXT_DEBUG((
                                       CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG3,  "unifi%d:   found CSR_SLT_END\n"
                                       , card->instance));
                break;
            }

            r = unifi_read32(card, slut_address, &l);
            if (r != CSR_RESULT_SUCCESS)
            {
                func_exit_r(r);
                return r;
            }
            slut_address += 4;

            slut.id = s;
            slut.obj = l;

            CSR_LOG_TEXT_DEBUG((
                                   CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG3,  "unifi%d:   found SLUT id %02d.%08lx\n",
                                   card->instance,  slut.id, slut.obj));
            switch (slut.id)
            {
                case CSR_SLT_SDIO_SLOT_CONFIG:
                    break;

                case CSR_SLT_BUILD_ID_NUMBER:
                {
                    CsrUint32 n;
                    r = unifi_read32(card, slut.obj, &n);
                    if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
                    {
                        return r;
                    }
                    if (r != CSR_RESULT_SUCCESS)
                    {
                        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to read build id\n", card->instance));
                        func_exit_r(r);
                        return r;
                    }
                    card->build_id = n;
                    break;
                }

                case CSR_SLT_BUILD_ID_STRING:
                    r = unifi_readnz(card, slut.obj, card->build_id_string,
                                     sizeof(card->build_id_string));
                    if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
                    {
                        return r;
                    }
                    if (r != CSR_RESULT_SUCCESS)
                    {
                        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to read build string\n", card->instance));
                        func_exit_r(r);
                        return r;
                    }
                    break;

                case CSR_SLT_PERSISTENT_STORE_DB:
                    break;

                case CSR_SLT_BOOT_LOADER_CONTROL:

                    /* This command copies most of the station firmware
                     * image from ROM into program RAM.  It also clears
                     * out the zerod data and sets up the initialised
                     * data. */
                    r = unifi_do_loader_op(card, slut.obj + 6, UNIFI_BOOT_LOADER_LOAD_STA);
                    if (r != CSR_RESULT_SUCCESS)
                    {
                        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to write loader load image command\n", card->instance));
                        func_exit_r(r);
                        return r;
                    }

                    dlpriv = unifi_dl_fw_read_start(card, UNIFI_FW_STA);

                    /* dlpriv might be NULL, we still need to do the do_loader_op step. */
                    if (dlpriv != NULL)
                    {
                        /* Download the firmware. */
                        r = unifi_dl_patch(card, dlpriv, slut.obj);

                        /* Free the firmware file information. */
                        unifi_fw_read_stop(card->ospriv, dlpriv);

                        if (r != CSR_RESULT_SUCCESS)
                        {
                            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to patch firmware\n", card->instance));
                            func_exit_r(r);
                            return r;
                        }
                    }

                    /* This command starts the firmware image that we want (the
                    * station by default) with any patches required applied. */
                    r = unifi_do_loader_op(card, slut.obj + 6, UNIFI_BOOT_LOADER_RESTART);
                    if (r != CSR_RESULT_SUCCESS)
                    {
                        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to write loader restart command\n", card->instance));
                        func_exit_r(r);
                        return r;
                    }

                    /* The now running patch f/w defines a new SLUT data structure -
                     * the current one is no longer valid. We must drop out of the
                     * processing loop and enumerate the new SLUT (which may appear
                     * at a different offset).
                     */
                    search_4slut_again = 1;
                    break;

                case CSR_SLT_PANIC_DATA_PHY:
                    card->panic_data_phy_addr = slut.obj;
                    break;

                case CSR_SLT_PANIC_DATA_MAC:
                    card->panic_data_mac_addr = slut.obj;
                    break;

                default:
                    /* do nothing */
                    break;
            }
        } /* while */
    } while (search_4slut_again);

    /*
     * get config buffer handle from mailbox3
     */
    r = unifi_read_direct16(card, ChipHelper_MAILBOX3(card->helper) * 2, &mbox3);
#else /* CSR_WIFI_DRIVER_HYDRA */
      /*
       * Hydra: get config buffer handle from th-scratch 0
       */
    r = unifi_read_direct_8_or_16(card, UNIFI_SDIO_CTRL_CFG_R, &mbox3);
#endif

    if ((mbox3 > 0) && (r == CSR_RESULT_SUCCESS))
    {
        CsrUint8 cfg_data_buf[SDIO_CONFIG_DATA_RW_SIZE];
        CSR_LOG_TEXT_INFO((
                              CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG2,  "unifi%d: card_hw_init: mbox3=0x%04x\n",
                              card->instance,  mbox3));
#ifdef CSR_WIFI_DRIVER_HYDRA
        /* D-30105: The (4<<8) is the Hydra WLAN subsystem ID shifted into the high byte of the handle.
         * This should be removed once the firmware is fixed. Also the auto-rewind bit should be set,
         * AND a single SDIO block-sized read used for better error recovery.
         */
        r = unifi_bulk_rw_noretry(card, (/*UNIFI_SDIO_AUTO_REWIND |*/ mbox3 | (4 << 8)),
                                  cfg_data_buf, SDIO_CONFIG_DATA_RW_SIZE, UNIFI_SDIO_READ);
#else
        r = unifi_bulk_rw(card, mbox3, cfg_data_buf, SDIO_CONFIG_DATA_RW_SIZE, UNIFI_SDIO_READ_CONFIG);
#endif
        if (CSR_RESULT_SUCCESS == r)
        {
            CsrUint32 i;

            cfg_data = &card->config_data;

            r = unpack_sdio_config_data(card, cfg_data, cfg_data_buf);
            if (CSR_RESULT_SUCCESS == r)
            {
                /* buffer_max_size is passed in blocks, convert it to bytes */
                card->buff_maximum_size_bytes = cfg_data->buff_maximum_size_blocks * cfg_data->block_round_size;
                if (cfg_data->bf_entries > card->config_params.mparam_slot_count[D4_T_2_MAX_BULK_SLOTS_FROM])
                {
                    cfg_data->bf_entries = (CsrUint16) card->config_params.mparam_slot_count[D4_T_2_MAX_BULK_SLOTS_FROM];
                }
                if (cfg_data->bt_entries > card->config_params.mparam_slot_count[D4_T_2_MAX_BULK_SLOTS_TO])
                {
                    cfg_data->bt_entries = (CsrUint16) card->config_params.mparam_slot_count[D4_T_2_MAX_BULK_SLOTS_TO];
                }

                if (cfg_data->ctrl_info_size != SIZEOF_CTRL_BUFFER(card->ctrl_buffer))
                {
                    if (cfg_data->ctrl_info_size > SIZEOF_CTRL_BUFFER(card->ctrl_buffer))
                    {
                        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                               "unifi%d: <-- F/w HIP control buffer is larger than host version.\n", card->instance));
                        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                               "unifi%d: Extra control data will be ignored.-->\n", card->instance));
                    }
                    else
                    {
                        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                               "unifi%d: HIP control buffer size advertised by the f/w is too small (%d) - host control size %d\n",
                                               card->instance, cfg_data->ctrl_info_size, sizeof(ctrl_buffer_t)));
                        return CSR_RESULT_FAILURE;
                    }
                }
                card->fh_buffer_d.num_handles = 0;
                for (i = 0; i < SDIO_FH_PUSHED_HANDLES; i++)
                {
                    if (card->config_data.fh_pushed_buffer_h[i] == 0)
                    {
                        break;
                    }
                    card->fh_buffer_d.handles[i] = card->config_data.fh_pushed_buffer_h[i];
                    card->fh_buffer_d.num_handles++;
                }
                CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_UDBG2, "unifi%d: card_hw_init: FH handle count=%d\n",
                                   card->instance,  card->fh_buffer_d.num_handles));
                card->fh_buffer_d.pulled_round_bytes = cfg_data->block_round_size * cfg_data->fh_pulled_round;
                card->fh_buffer_d.pushed_round_bytes = cfg_data->block_round_size * cfg_data->fh_pushed_round;
                card->fh_buffer_d.max_window = card->buff_maximum_size_bytes * card->fh_buffer_d.num_handles;
                if (((CsrUint32) cfg_data->push_maximum_size * (CsrUint32) cfg_data->block_round_size) < card->fh_buffer_d.max_window)
                {
                    /* push_maximum_size can reduce the window size below a fixed multiple of the handle max */
                    card->fh_buffer_d.max_window = cfg_data->push_maximum_size * cfg_data->block_round_size;
                }
                card->th_buffer_d.ctrl_round_bytes = cfg_data->block_round_size * cfg_data->th_ctrl_round;
                card->th_buffer_d.bulk_round_bytes = cfg_data->block_round_size * cfg_data->th_bulk_round;
                card->ack_outstanding = 0;
                unifi_set_segment_size(card);
            }
            else
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: HIP config read error\n", card->instance));
                return r;
            }
        }
        else
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to read config data from buffer handle %d\n",
                                   card->instance,  mbox3));
            return CSR_RESULT_FAILURE;
        }
    }
    else
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to find config buffer handle from mailbox3\n", card->instance));
        return CSR_RESULT_FAILURE;
    }

    card->Pm = 0;

    /*
     * Now check the UniFi firmware version
     */
    ver_major = (cfg_data->version >> 8) & 0xFF;
    ver_minor = cfg_data->version & 0xFF;
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,
                        CSR_WIFI_HIP_LOG_DEF, "unifi%d: UniFi f/w protocol version %d.%d (driver %d.%d)\n",
                        card->instance,
                        ver_major, ver_minor,
                        UNIFI_HIP_MAJOR_VERSION, UNIFI_HIP_MINOR_VERSION));

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,
                        CSR_WIFI_HIP_LOG_DEF, "unifi%d: Firmware build %u: %s\n",
                        card->instance,
                        card->build_id, card->build_id_string));

    if (ver_major != UNIFI_HIP_MAJOR_VERSION)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: UniFi f/w protocol major version (%d) is different from driver (v%d.%d)\n",
                               card->instance,
                               ver_major, UNIFI_HIP_MAJOR_VERSION, UNIFI_HIP_MINOR_VERSION));
#ifndef CSR_WIFI_DISABLE_HIP_VERSION_CHECK
#define CSR_WIFI_DISABLE_HIP_VERSION_CHECK 1 /* TEMP */
#endif
#ifndef CSR_WIFI_DISABLE_HIP_VERSION_CHECK
        func_exit_r(CSR_RESULT_FAILURE);
        return CSR_RESULT_FAILURE;
#endif
    }
    if (ver_minor < UNIFI_HIP_MINOR_VERSION)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: UniFi f/w protocol version (v%d.%d) is older than minimum required by driver (v%d.%d).\n",
                               card->instance,
                               ver_major, ver_minor,
                               UNIFI_HIP_MAJOR_VERSION, UNIFI_HIP_MINOR_VERSION));
#ifndef CSR_WIFI_DISABLE_HIP_VERSION_CHECK
        func_exit_r(CSR_RESULT_FAILURE);
        return CSR_RESULT_FAILURE;
#endif
    }

    /* Read panic codes from a previous firmware panic. If the firmware has
     * not panicked since power was applied (e.g. power-off hard reset)
     * the stored panic codes will not be updated.
     */
    unifi_read_panic(card);

    func_exit();
    return CSR_RESULT_SUCCESS;
} /* card_hw_init() */

#ifndef CSR_WIFI_DRIVER_HYDRA
/*
 * ---------------------------------------------------------------------------
 *  card_wait_for_unifi_to_reset
 *
 *      Waits for a reset to complete by polling the WLAN function enable
 *      bit (which is cleared on reset).
 *
 *  Arguments:
 *      card            Pointer to card struct
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success, CSR error code on failure.
 * ---------------------------------------------------------------------------
 */
static CsrResult card_wait_for_unifi_to_reset(card_t *card)
{
    CsrInt16 i;
    CsrResult r;
    CsrUint8 io_enable;
    CsrResult csrResult;

    func_enter();

    r = CSR_RESULT_SUCCESS;
    for (i = 0; i < MAILBOX2_ATTEMPTS; i++)
    {
        CSR_LOG_TEXT_INFO((
                              CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: waiting for reset to complete, attempt %d\n",
                              card->instance,  i));
        if (card->chip_id > SDIO_CARD_ID_UNIFI_2)
        {
            /* It's quite likely that this read will timeout for the
             * first few tries - especially if we have reset via
             * DBG_RESET.
             */
#if defined(CSR_WIFI_HIP_DEBUG_OFFLINE_ENABLE) && defined(CSR_WIFI_HIP_SDIO_TRACE_ENABLE)
            unifi_debug_log_to_buf(0, "m0@%02X=", SDIO_IO_READY);
#endif
            csrResult = CsrSdioF0Read8(card->sdio_if, SDIO_IO_READY, &io_enable);
#if defined(CSR_WIFI_HIP_DEBUG_OFFLINE_ENABLE) && defined(CSR_WIFI_HIP_SDIO_TRACE_ENABLE)
            if (csrResult != CSR_RESULT_SUCCESS)
            {
                unifi_debug_log_to_buf(1, "error=%X", csrResult);
            }
            else
            {
                unifi_debug_log_to_buf(1, "%X", io_enable);
            }
#endif
            if (csrResult == CSR_SDIO_RESULT_NO_DEVICE)
            {
                return CSR_WIFI_HIP_RESULT_NO_DEVICE;
            }
            r = CSR_RESULT_SUCCESS;
            if (csrResult != CSR_RESULT_SUCCESS)
            {
                r = ConvertCsrSdioToCsrHipResult(card, csrResult);
            }
        }
        else
        {
            r = sdio_read_f0(card, SDIO_IO_ENABLE, &io_enable);
        }
        if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
        {
            return r;
        }
        if (r == CSR_RESULT_SUCCESS)
        {
            CsrUint16 mbox2;
            CsrInt16 enabled = io_enable & (1 << card->function);

            if (!enabled)
            {
                CSR_LOG_TEXT_INFO((
                                      CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                                      "unifi%d: Reset complete (function %d is disabled) in ~ %u msecs\n",
                                      card->instance,
                                      card->function, i * MAILBOX2_TIMEOUT));

                /* Enable WLAN function and verify MAILBOX2 is zero'd */
                csrResult = CsrSdioFunctionEnable(card->sdio_if);
                if (csrResult != CSR_RESULT_SUCCESS)
                {
                    r = ConvertCsrSdioToCsrHipResult(card, csrResult);
                    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                           CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrSdioFunctionEnable failed %d\n",
                                           card->instance,  r));
                    break;
                }
            }

            r = unifi_read_direct16(card, ChipHelper_SDIO_HIP_HANDSHAKE(card->helper) * 2, &mbox2);
            if (r != CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: read HIP_HANDSHAKE failed %d\n",
                                       card->instance,  r));
                break;
            }
            if (mbox2 != 0)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: MAILBOX2 non-zero after reset (mbox2 = %04x)\n",
                                       card->instance,  mbox2));
                r = CSR_RESULT_FAILURE;
            }
            break;
        }
        else
        {
            if (card->chip_id > SDIO_CARD_ID_UNIFI_2)
            {
                /* We ignore read failures for the first few reads,
                 * they are probably benign. */
                if (i > MAILBOX2_ATTEMPTS / 4)
                {
                    CSR_LOG_TEXT_INFO((
                                          CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: Failed to read CCCR IO Ready register while polling for reset\n"
                                          , card->instance));
                }
            }
            else
            {
                CSR_LOG_TEXT_INFO((
                                      CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: Failed to read CCCR IO Enable register while polling for reset\n"
                                      , card->instance));
            }
        }
        CsrThreadSleep(MAILBOX2_TIMEOUT);
    }

    if ((r == CSR_RESULT_SUCCESS) && (i == MAILBOX2_ATTEMPTS))
    {
        CSR_LOG_TEXT_INFO((
                              CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: Timeout waiting for UniFi to complete reset\n"
                              , card->instance));
        r = CSR_RESULT_FAILURE;
    }

    func_exit();
    return r;
} /* card_wait_for_unifi_to_reset() */

/*
 * ---------------------------------------------------------------------------
 *  card_wait_for_unifi_to_disable
 *
 *      Waits for the function to become disabled by polling the
 *      IO_READY bit.
 *
 *  Arguments:
 *      card            Pointer to card struct
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success, CSR error code on failure.
 *
 *  Notes: This function can only be used with
 *         card->chip_id > SDIO_CARD_ID_UNIFI_2
 * ---------------------------------------------------------------------------
 */
static CsrResult card_wait_for_unifi_to_disable(card_t *card)
{
    CsrInt16 i;
    CsrResult r;
    CsrUint8 io_enable;
    CsrResult csrResult;

    func_enter();

    if (card->chip_id <= SDIO_CARD_ID_UNIFI_2)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: Function reset method not supported for chip_id=%d\n",
                               card->instance,
                               card->chip_id));
        func_exit();
        return CSR_RESULT_FAILURE;
    }

    r = CSR_RESULT_SUCCESS;
    for (i = 0; i < MAILBOX2_ATTEMPTS; i++)
    {
        CSR_LOG_TEXT_INFO((
                              CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: waiting for disable to complete, attempt %d\n",
                              card->instance,  i));

        /*
         * It's quite likely that this read will timeout for the
         * first few tries - especially if we have reset via
         * DBG_RESET.
         */
#if defined(CSR_WIFI_HIP_DEBUG_OFFLINE_ENABLE) && defined(CSR_WIFI_HIP_SDIO_TRACE_ENABLE)
        unifi_debug_log_to_buf(0, "r0@%02X=", SDIO_IO_READY);
#endif
        csrResult = CsrSdioF0Read8(card->sdio_if, SDIO_IO_READY, &io_enable);
#if defined(CSR_WIFI_HIP_DEBUG_OFFLINE_ENABLE) && defined(CSR_WIFI_HIP_SDIO_TRACE_ENABLE)
        if (csrResult != CSR_RESULT_SUCCESS)
        {
            unifi_debug_log_to_buf(1, "error=%X", csrResult);
        }
        else
        {
            unifi_debug_log_to_buf(1, "%X", io_enable);
        }
#endif
        if (csrResult == CSR_SDIO_RESULT_NO_DEVICE)
        {
            return CSR_WIFI_HIP_RESULT_NO_DEVICE;
        }
        if (csrResult == CSR_RESULT_SUCCESS)
        {
            CsrInt16 enabled = io_enable & (1 << card->function);
            r = CSR_RESULT_SUCCESS;
            if (!enabled)
            {
                CSR_LOG_TEXT_INFO((
                                      CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                                      "unifi%d: Disable complete (function %d is disabled) in ~ %u msecs\n",
                                      card->instance,
                                      card->function, i * MAILBOX2_TIMEOUT));

                break;
            }
        }
        else
        {
            /*
             * We ignore read failures for the first few reads,
             * they are probably benign.
             */
            r = ConvertCsrSdioToCsrHipResult(card, csrResult);
            if (i > (MAILBOX2_ATTEMPTS / 4))
            {
                CSR_LOG_TEXT_INFO((
                                      CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                                      "unifi%d: Failed to read CCCR IO Ready register while polling for disable\n"
                                      , card->instance));
            }
        }
        CsrThreadSleep(MAILBOX2_TIMEOUT);
    }

    if ((r == CSR_RESULT_SUCCESS) && (i == MAILBOX2_ATTEMPTS))
    {
        CSR_LOG_TEXT_INFO((
                              CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: Timeout waiting for UniFi to complete disable\n"
                              , card->instance));
        r = CSR_RESULT_FAILURE;
    }

    func_exit();
    return r;
} /* card_wait_for_unifi_to_reset() */

#endif

/*
 * ---------------------------------------------------------------------------
 *  card_wait_for_firmware_to_start
 *
 *      Polls the MAILBOX1 register for a non-zero value.
 *      Then reads MAILBOX0 and forms the two values into a 32-bit address
 *      which is returned to the caller.
 *
 *  Arguments:
 *      card            Pointer to card struct
 *      paddr           Pointer to receive the UniFi address formed
 *                      by concatenating MAILBOX1 and MAILBOX0.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success, CSR error code on failure.
 * ---------------------------------------------------------------------------
 */
#ifndef CSR_WIFI_DRIVER_HYDRA
CsrResult card_wait_for_firmware_to_start(card_t *card, CsrUint32 *paddr)
{
    CsrInt32 i;
    CsrUint16 mbox0, mbox1;
    CsrResult r;

    func_enter();

    /*
     * Wait for UniFi to initialise its data structures by polling
     * the SHARED_MAILBOX1 register.
     * Experience shows this is typically 120ms.
     */
    CsrThreadSleep(MAILBOX1_TIMEOUT);

    mbox1 = 0;
    CSR_LOG_TEXT_INFO((
                          CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: waiting for MAILBOX1 to be non-zero...\n"
                          , card->instance));
    for (i = 0; i < MAILBOX1_ATTEMPTS; i++)
    {
        r = unifi_read_direct16(card, ChipHelper_MAILBOX1(card->helper) * 2, &mbox1);
        if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
        {
            return r;
        }
        if (r != CSR_RESULT_SUCCESS)
        {
            /* These reads can fail if UniFi isn't up yet, so try again */
            CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to read UniFi Mailbox1 register\n", card->instance));
        }

        if ((r == CSR_RESULT_SUCCESS) && (mbox1 != 0))
        {
            CSR_LOG_TEXT_INFO((
                                  CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: MAILBOX1 ready (0x%04X) in %u millisecs\n",
                                  card->instance,
                                  mbox1, i * MAILBOX1_TIMEOUT));

            /* Read the MAILBOX1 again in case we caught the value as it
             * changed. */
            r = unifi_read_direct16(card, ChipHelper_MAILBOX1(card->helper) * 2, &mbox1);
            if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
            {
                return r;
            }
            if (r != CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to read UniFi Mailbox1 register for second time\n", card->instance));
                func_exit_r(r);
                return r;
            }
            CSR_LOG_TEXT_INFO((
                                  CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: MAILBOX1 value=0x%04X\n",
                                  card->instance,  mbox1));

            break;
        }

        CsrThreadSleep(MAILBOX1_TIMEOUT);
        if ((i % 100) == 99)
        {
            CSR_LOG_TEXT_INFO((
                                  CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG2,  "unifi%d: MAILBOX1 not ready (0x%X), still trying...\n",
                                  card->instance,  mbox1));
        }
    }

    if ((r == CSR_RESULT_SUCCESS) && (mbox1 == 0))
    {
        CSR_LOG_TEXT_INFO((
                              CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: Timeout waiting for firmware to start, Mailbox1 still 0 after %d ms\n",
                              card->instance,
                              MAILBOX1_ATTEMPTS * MAILBOX1_TIMEOUT));
        func_exit_r(CSR_RESULT_FAILURE);
        return CSR_RESULT_FAILURE;
    }


    /*
     * Complete the reset handshake by setting MAILBOX2 to 0xFFFF
     */
    r = unifi_write_direct16(card, ChipHelper_SDIO_HIP_HANDSHAKE(card->helper) * 2, 0xFFFF);
    if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
    {
        return r;
    }
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to write f/w startup handshake to MAILBOX2\n", card->instance));
        func_exit_r(r);
        return r;
    }


    /*
     * Read the Symbol Look Up Table (SLUT) offset.
     * Top 16 bits are in mbox1, read the lower 16 bits from mbox0.
     */
    mbox0 = 0;
    r = unifi_read_direct16(card, ChipHelper_MAILBOX0(card->helper) * 2, &mbox0);
    if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
    {
        return r;
    }
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to read UniFi Mailbox0 register\n", card->instance));
        func_exit_r(r);
        return r;
    }

    *paddr = (((CsrUint32) mbox1 << 16) | mbox0);

    func_exit();
    return CSR_RESULT_SUCCESS;
}     /* card_wait_for_firmware_to_start() */

#else /* Hydra */

CsrResult card_wait_for_firmware_to_start(card_t *card, CsrUint32 *paddr)
{
    CsrInt32 i;
    CsrResult r;
    CsrUint8 th_stat;

    func_enter();

    /*
     * Wait for UniFi to initialise its data structures by polling
     * the to-host-scratch-0 register.
     */
    CsrThreadSleep(MAILBOX1_TIMEOUT);

    th_stat = 0;
    CSR_LOG_TEXT_INFO((
                          CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: waiting for TH_SCRATCH0 to be non-zero...\n"
                          , card->instance));
    for (i = 0; i < MAILBOX1_ATTEMPTS; i++)
    {
        r = unifi_read_direct_8_or_16(card, UNIFI_SDIO_CTRL_CFG_R, &th_stat);
        if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
        {
            return r;
        }
        if (r != CSR_RESULT_SUCCESS)
        {
            /* These reads can fail if UniFi isn't up yet, so try again */
            CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to read UniFi TH_SCRATCH0 register\n", card->instance));
        }

        if ((r == CSR_RESULT_SUCCESS) && (th_stat != 0))
        {
            CSR_LOG_TEXT_INFO((
                                  CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: TH_SCRATCH0 ready (0x%02X) in %u millisecs\n",
                                  card->instance,
                                  th_stat, i * MAILBOX1_TIMEOUT));

            /* Read the register again in case we caught the value as it
             * changed. */
            r = unifi_read_direct_8_or_16(card, UNIFI_SDIO_CTRL_CFG_R, &th_stat);
            if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
            {
                return r;
            }
            if (r != CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to read UniFi TH_SCRATCH0 register for second time\n", card->instance));
                func_exit_r(r);
                return r;
            }
            CSR_LOG_TEXT_INFO((
                                  CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: TH_SCRATCH0 value=0x%02X\n",
                                  card->instance,  th_stat));

            break;
        }

        CsrThreadSleep(MAILBOX1_TIMEOUT);
        if ((i % 100) == 99)
        {
            CSR_LOG_TEXT_INFO((
                                  CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG2,  "unifi%d: TH_SCRATCH0 not ready (0x%X), still trying...\n",
                                  card->instance,  th_stat));
        }
    }

    if ((r == CSR_RESULT_SUCCESS) && (th_stat == 0))
    {
        CSR_LOG_TEXT_INFO((
                              CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: Timeout waiting for firmware to start, TH_SCRATCH0 still 0 after %d ms\n",
                              card->instance,
                              MAILBOX1_ATTEMPTS * MAILBOX1_TIMEOUT));
        func_exit_r(CSR_RESULT_FAILURE);
        return CSR_RESULT_FAILURE;
    }

    return CSR_RESULT_SUCCESS;
} /* card_wait_for_firmware_to_start() */

#endif

/*
 * ---------------------------------------------------------------------------
 *  unifi_capture_panic
 *
 *      Attempt to capture panic codes from the firmware. This may involve
 *      warm reset of the chip to regain access following a watchdog reset.
 *
 *  Arguments:
 *      card            Pointer to card struct
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS if panic codes were captured, or none available
 *      CSR_RESULT_FAILURE if the driver could not access function 1
 * ---------------------------------------------------------------------------
 */
CsrResult unifi_capture_panic(card_t *card)
{
#ifdef CSR_WIFI_DRIVER_HYDRA
    /* The Curator and Hydra driver will look after this */
#else
    func_enter();

    /* The firmware must have previously initialised to read the panic addresses
     * from the SLUT
     */
    if (!card->panic_data_phy_addr || !card->panic_data_mac_addr)
    {
        func_exit();
        return CSR_RESULT_SUCCESS;
    }

    /* Ensure we can access function 1 following a panic/watchdog reset */
    if (card_access_panic(card) == CSR_RESULT_SUCCESS)
    {
        /* Read the panic codes */
        unifi_read_panic(card);
    }
    else
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Unable to read panic codes", card->instance));
    }

    func_exit();
#endif
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  card_access_panic
 *      Attempt to read the WLAN SDIO function in order to read panic codes
 *      and perform various reset steps to regain access if the read fails.
 *
 *  Arguments:
 *      card            Pointer to card struct
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS if panic codes can be read
 *      CSR error code if panic codes can not be read
 * ---------------------------------------------------------------------------
 */
#ifndef CSR_WIFI_DRIVER_HYDRA
static CsrResult card_access_panic(card_t *card)
{
    CsrUint16 data_u16 = 0;
    CsrInt32 i;
    CsrResult r, sr;

    func_enter();

    /* A chip version of zero means that the version never got succesfully read
     * during reset. In this case give up because it will not be possible to
     * verify the chip version.
     */
    if (!card->chip_version)
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Unknown chip version\n", card->instance));
        return CSR_RESULT_FAILURE;
    }

    /* Ensure chip is awake or access to function 1 will fail */
    r = unifi_set_host_state(card, UNIFI_HOST_STATE_AWAKE);
    if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
    {
        return r;
    }
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_set_host_state() failed %d\n",
                               card->instance,  r));
        return CSR_RESULT_FAILURE; /* Card is probably unpowered */
    }
    CsrThreadSleep(20);

    for (i = 0; i < 3; i++)
    {
        sr = CsrSdioRead16(card->sdio_if, CHIP_HELPER_UNIFI_GBL_CHIP_VERSION * 2, &data_u16);
        if ((sr != CSR_RESULT_SUCCESS) || (data_u16 != card->chip_version))
        {
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,
                                CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to read valid chip version sr=%d (0x%04x want 0x%04x) try %d\n",
                                card->instance,
                                sr, data_u16, card->chip_version, i));

            /* Set clock speed low */
            sr = CsrSdioMaxBusClockFrequencySet(card->sdio_if, UNIFI_SDIO_CLOCK_SAFE_HZ);
            if (sr != CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrSdioMaxBusClockFrequencySet() failed1 %d\n",
                                       card->instance,  sr));
                r = ConvertCsrSdioToCsrHipResult(card, sr);
            }
            card->sdio_clock_speed = UNIFI_SDIO_CLOCK_SAFE_HZ;

            /* First try re-enabling function in case a f/w watchdog reset disabled it */
            if (i == 0)
            {
                CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Try function enable\n", card->instance));
                sr = CsrSdioFunctionEnable(card->sdio_if);
                if (sr != CSR_RESULT_SUCCESS)
                {
                    r = ConvertCsrSdioToCsrHipResult(card, sr);
                    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                           CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrSdioFunctionEnable failed %d (HIP %d)\n",
                                           card->instance,  sr, r));
                }
                continue;
            }

            /* Second try, set awake */
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Try set awake\n", card->instance));

            /* Ensure chip is awake */
            r = unifi_set_host_state(card, UNIFI_HOST_STATE_AWAKE);
            if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
            {
                return r;
            }
            if (r != CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_set_host_state() failed2 %d\n",
                                       card->instance,  r));
            }

            /* Set clock speed low in case setting the host state raised it, which
             * would only happen if host state was previously TORPID
             */
            sr = CsrSdioMaxBusClockFrequencySet(card->sdio_if, UNIFI_SDIO_CLOCK_SAFE_HZ);
            if (sr != CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrSdioMaxBusClockFrequencySet() failed2 %d\n",
                                       card->instance,  sr));
            }
            card->sdio_clock_speed = UNIFI_SDIO_CLOCK_SAFE_HZ;

            if (i == 1)
            {
                continue;
            }

            /* Perform a s/w reset to preserve as much as the card state as possible,
             * (mainly the preserve RAM). The context will be lost for coredump - but as we
             * were unable to access the WLAN function for panic, the coredump would have
             * also failed without a reset.
             */
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Try s/w reset\n", card->instance));

            r = unifi_card_hard_reset(card);
            if (r != CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_card_hard_reset() failed %d\n",
                                       card->instance,  r));
            }
        }
        else
        {
            if (i > 0)
            {
                CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,
                                    CSR_WIFI_HIP_LOG_DEF, "unifi%d: Read chip version 0x%x after %d retries\n",
                                    card->instance,  data_u16, i));
            }
            break;
        }
    }

    r = ConvertCsrSdioToCsrHipResult(card, sr);
    func_exit_r(r);
    return r;
}

#endif

/*
 * ---------------------------------------------------------------------------
 *  unifi_read_panic
 *      Reads, saves and prints panic codes stored by the firmware in UniFi's
 *      preserve RAM by the last panic that occurred since chip was powered.
 *      Nothing is saved if the panic codes are read as zero.
 *
 *  Arguments:
 *      card            Pointer to card struct
 *
 *  Returns:
 * ---------------------------------------------------------------------------
 */
void unifi_read_panic(card_t *card)
{
#ifdef CSR_WIFI_DRIVER_HYDRA
    /* The Curator should retrieve this information per subsystem startup
     * and display it in the Hydra driver's log
     */
#else
    CsrResult r;
    CsrUint16 p_code, p_arg;

    func_enter();

    /* The firmware must have previously initialised to read the panic addresses
     * from the SLUT
     */
    if (!card->panic_data_phy_addr || !card->panic_data_mac_addr)
    {
        return;
    }

    /* Get the panic data from PHY */
    r = unifi_card_read16(card, card->panic_data_phy_addr, &p_code);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: capture_panic: unifi_read16 %08x failed %d\n",
                               card->instance,  card->panic_data_phy_addr, r));
        p_code = 0;
    }
    if (p_code)
    {
        r = unifi_card_read16(card, card->panic_data_phy_addr + 2, &p_arg);
        if (r != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: capture_panic: unifi_read16 %08x failed %d\n",
                                   card->instance,  card->panic_data_phy_addr + 2, r));
        }
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: Last UniFi PHY PANIC %04x arg %04x\n",
                               card->instance,  p_code, p_arg));
        card->last_phy_panic_code = p_code;
        card->last_phy_panic_arg = p_arg;
    }

    /* Get the panic data from MAC */
    r = unifi_card_read16(card, card->panic_data_mac_addr, &p_code);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: capture_panic: unifi_read16 %08x failed %d\n",
                               card->instance,  card->panic_data_mac_addr, r));
        p_code = 0;
    }
    if (p_code)
    {
        r = unifi_card_read16(card, card->panic_data_mac_addr + 2, &p_arg);
        if (r != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: capture_panic: unifi_read16 %08x failed %d\n",
                                   card->instance,  card->panic_data_mac_addr + 2, r));
        }
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: Last UniFi MAC PANIC %04x arg %04x\n",
                               card->instance,  p_code, p_arg));
        card->last_mac_panic_code = p_code;
        card->last_mac_panic_arg = p_arg;
    }

    func_exit();
#endif /* !CSR_WIFI_DRIVER_HYDRA */
}

/*
 * ---------------------------------------------------------------------------
 *  card_allocate_memory_resources
 *
 *      Allocates memory for the from-host, to-host bulk data slots,
 *      soft queue buffers and bulk data buffers.
 *
 *  Arguments:
 *      card            Pointer to card struct
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success, CSR error code on failure.
 * ---------------------------------------------------------------------------
 */
static CsrResult card_allocate_memory_resources(card_t *card)
{
    CsrUint32 n, i;
    sdio_config_data_t *cfg_data;

    func_enter();

#ifndef CSR_WIFI_HIP_TA_DISABLE
    unifi_ta_sampling_init(card);
#endif
    /* Convenience short-cut */
    cfg_data = &card->config_data;

    /*
     * Allocate memory for the from-host and to-host signal buffers.
     */
    card->fh_buffer_d.buffer = CsrMemAllocDma(UNIFI_FH_BUF_SIZE);
    if (card->fh_buffer_d.buffer == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to allocate memory for fh buffer\n", card->instance));
        func_exit_r(CSR_WIFI_HIP_RESULT_NO_MEMORY);
        return CSR_WIFI_HIP_RESULT_NO_MEMORY;
    }
    card->fh_buffer_d.buffer_ptr = card->fh_buffer_d.buffer;

    card->th_buffer_d.buffer = CsrMemAllocDma(UNIFI_TH_BUF_SIZE);
    if (card->th_buffer_d.buffer == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to allocate memory for th buffer\n", card->instance));
        func_exit_r(CSR_WIFI_HIP_RESULT_NO_MEMORY);
        return CSR_WIFI_HIP_RESULT_NO_MEMORY;
    }
    card->th_buffer_d.buffer_size = UNIFI_TH_BUF_SIZE;

    /*
     * Allocate control buffer.
     */
    card->ctrl_read_buffer_len = card->config_data.block_round_size * card->config_data.ctrl_info_blocks;
    if ((card->ctrl_read_buffer_len == 0) || (card->ctrl_read_buffer_len > 65536))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "Suspicious control buffer length %u\n", card->ctrl_read_buffer_len));
        return CSR_WIFI_HIP_RESULT_INVALID_VALUE;
    }

    card->ctrl_read_buffer = CsrMemAllocDma(card->ctrl_read_buffer_len);
    if (card->ctrl_read_buffer == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to allocate memory for ctrl buffer\n", card->instance));
        func_exit_r(CSR_WIFI_HIP_RESULT_NO_MEMORY);
        return CSR_WIFI_HIP_RESULT_NO_MEMORY;
    }

    /*
     * Allocate memory for the from-host and to-host bulk data slots.
     */

    n = cfg_data->bf_entries;

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG3,  "unifi%d: Alloc from-host resources, %d slots.\n",
                           card->instance,  n));
    card->from_host_data =
        (slot_desc_t *) CsrMemAlloc(n * sizeof(slot_desc_t));
    if (card->from_host_data == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to allocate memory for F-H bulk data array\n", card->instance));
        func_exit_r(CSR_WIFI_HIP_RESULT_NO_MEMORY);
        return CSR_WIFI_HIP_RESULT_NO_MEMORY;
    }
    CsrMemSet(card->from_host_data, 0, n * sizeof(slot_desc_t));

    /*
     * Allocate a buffer for writing HIP (n)acks.
     */
    card->ack_buffer =
        (CsrUint8 *) CsrMemAllocDma((CsrSize) card->config_data.ack_buffer_blocks * (CsrSize) card->config_data.block_round_size);
    if (card->ack_buffer == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to allocate memory for HIP ack buffer\n", card->instance));
        func_exit_r(CSR_WIFI_HIP_RESULT_NO_MEMORY);
        return CSR_WIFI_HIP_RESULT_NO_MEMORY;
    }
    CsrMemSet(card->ack_buffer, 0, (CsrSize) card->config_data.ack_buffer_blocks * (CsrSize) card->config_data.block_round_size);
    card->ack_buffer_ptr = card->ack_buffer;
    card->acks_written = 0;

    /* Initialise from-host bulk data slots */
    card_init_slot_list(card);

    /* Allocate memory for the array of pointers */
    n = cfg_data->bt_entries;

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG3,  "unifi%d: Alloc to-host resources, %d slots.\n",
                           card->instance,  n));
    card->to_host_data = CsrMemAlloc(n * sizeof(CsrWifiHipBulkDataDesc));
    if (card->to_host_data == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to allocate memory for T-H bulk data array\n", card->instance));
        func_exit_r(CSR_WIFI_HIP_RESULT_NO_MEMORY);
        return CSR_WIFI_HIP_RESULT_NO_MEMORY;
    }

    /* Initialise to-host bulk data slots */
    for (i = 0; i < n; i++)
    {
        UNIFI_INIT_BULK_DATA(&card->to_host_data[i]);
    }

    card->memory_resources_allocated = 1;

    func_exit();
    return CSR_RESULT_SUCCESS;
} /* card_allocate_memory_resources() */

/*
 * ---------------------------------------------------------------------------
 *  unifi_free_bulk_data
 *
 *      Free the data associated to a bulk data structure.
 *
 *  Arguments:
 *      card            Pointer to card struct
 *      bulk_data_slot  Pointer to bulk data structure
 *
 *  Returns:
 *      None.
 *
 * ---------------------------------------------------------------------------
 */
static void unifi_free_bulk_data(card_t *card, CsrWifiHipBulkDataDesc *bulk_data_slot)
{
    if (bulk_data_slot->data_length != 0)
    {
        unifi_net_data_free(card->ospriv, bulk_data_slot);
    }
} /* unifi_free_bulk_data() */

/*
 * ---------------------------------------------------------------------------
 *  card_free_memory_resources
 *
 *      Frees memory allocated for the from-host, to-host bulk data slots,
 *      soft queue buffers and bulk data buffers.
 *
 *  Arguments:
 *      card            Pointer to card struct
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
static void card_free_memory_resources(card_t *card)
{
    func_enter();

    CSR_LOG_TEXT_INFO((
                          CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: Freeing card memory resources.\n"
                          , card->instance));

    #ifndef CSR_WIFI_HIP_TA_DISABLE
    unifi_ta_sampling_init(card);
    #endif

    if (card->to_host_data)
    {
        CsrMemFree(card->to_host_data);
        card->to_host_data = NULL;
    }

    if (card->from_host_data)
    {
        CsrMemFree(card->from_host_data);
        card->from_host_data = NULL;
    }


    if (card->fh_buffer_d.buffer)
    {
        CsrMemFree(card->fh_buffer_d.buffer);
    }
    card->fh_buffer_d.buffer_ptr = NULL;
    card->fh_buffer_d.buffer = NULL;

    if (card->th_buffer_d.buffer)
    {
        CsrMemFree(card->th_buffer_d.buffer);
    }
    card->th_buffer_d.buffer = NULL;
    card->th_buffer_d.buffer_size = 0;

    if (card->ack_buffer)
    {
        CsrMemFree(card->ack_buffer);
        card->ack_buffer = NULL;
    }
    card->memory_resources_allocated = 0;

    func_exit();
} /* card_free_memory_resources() */

/*
 * ---------------------------------------------------------------------------
 *  unifi_free_card
 *
 *      Free the memory allocated for the card structure and buffers.
 *
 *  Notes:
 *
 *  Arguments:
 *      card        Pointer to card struct
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
void unifi_free_card(card_t *card)
{
    func_enter();

    /* Free any memory allocated. */
    card_free_memory_resources(card);
    CsrMemFree(card);

    func_exit();
} /* unifi_free_card() */

/*
 * ---------------------------------------------------------------------------
 *  card_init_slots
 *
 *      Allocate memory for host-side slot data and signal queues.
 *
 * Arguments:
 *      card            Pointer to card object
 *
 * Returns:
 *      CSR error code.
 * ---------------------------------------------------------------------------
 */
static CsrResult card_init_slots(card_t *card)
{
    sdio_config_data_t *cfg_data = &card->config_data;
    CsrUint8 cfg_data_buf[SDIO_CONFIG_DATA_RW_SIZE];
    CsrResult r;
    CsrUint8 i;

    func_enter();

    /* Allocate the buffers we need, only once. */
    if (card->memory_resources_allocated == 1)
    {
        card_free_memory_resources(card);
    }

    r = card_allocate_memory_resources(card);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to allocate card memory resources.\n", card->instance));
        card_free_memory_resources(card);
        func_exit_r(r);
        return r;
    }

    /* Get initial signal counts from UniFi, in case it has not been reset. */
#ifdef CSR_WIFI_TRANSPORT_CSPI
#error "CSPI currently unsupported in this HIP version"
    {
        CsrUint16 s;
    }
#endif

    /*
    * Complete initialisation by writing a response to the config response
    * buffer handle
    *
    * We can negotiate up the following:
    *    cfg_data->th_ctrl_round
    *    cfg_data->th_bulk_round
    *    cfg_data->fh_pushed_round
    *    cfg_data->fh_pulled_round
    */
    pack_sdio_config_data(card, cfg_data, cfg_data_buf);

    /* Write the config structure to the firmware */
#ifdef CSR_WIFI_DRIVER_HYDRA
    /* D-30105: For Hydra the auto-rewind bit should be set in the handle when writing the buffer,
     * AND a SDIO block-sized write used.
     */
    r = unifi_bulk_rw_noretry(card,
                              (/*UNIFI_SDIO_AUTO_REWIND |*/ cfg_data->config_response_buffer_h),
                              cfg_data_buf,
                              SDIO_CONFIG_DATA_RW_SIZE,
                              UNIFI_SDIO_WRITE);
#else
    r = unifi_bulk_rw(card,
                      cfg_data->config_response_buffer_h,
                      cfg_data_buf,
                      SDIO_CONFIG_DATA_RW_SIZE,
                      UNIFI_SDIO_WRITE_CONFIG);
#endif
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to write sdio config data to handle %d\n",
                               card->instance,  cfg_data->config_response_buffer_h));
        func_exit();
        return r;
    }
    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,
                       CSR_WIFI_HIP_UDBG2, "unifi%d: HIP CONFIG: Version: 0x%04x\n",
                       card->instance,  card->config_data.version));
    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,
                       CSR_WIFI_HIP_UDBG2, "unifi%d: HIP CONFIG: Sub-version: 0x%04x\n",
                       card->instance,  card->config_data.sub_version));
    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,
                       CSR_WIFI_HIP_UDBG2, "unifi%d: HIP CONFIG: Fw version:0x%x\n",
                       card->instance,  cfg_data->fw_version));
    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,
                       CSR_WIFI_HIP_UDBG2, "unifi%d: HIP CONFIG: ToHost Ctrl Round: %d\n",
                       card->instance,  card->config_data.th_ctrl_round));
    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,
                       CSR_WIFI_HIP_UDBG2, "unifi%d: HIP CONFIG: ToHost Bulk Round: %d\n",
                       card->instance,  card->config_data.th_bulk_round));
    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,
                       CSR_WIFI_HIP_UDBG2, "unifi%d: HIP CONFIG: Signal Chunk Size: %d\n",
                       card->instance,  card->config_data.signal_chunk_size));
    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,
                       CSR_WIFI_HIP_UDBG2, "unifi%d: HIP CONFIG: Handle size: %d (%d bytes)\n",
                       card->instance,  card->config_data.buff_maximum_size_blocks, card->buff_maximum_size_bytes));
    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,
                       CSR_WIFI_HIP_UDBG2, "unifi%d: HIP CONFIG: Total Round: %d\n",
                       card->instance,  card->config_data.block_round_size));
    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,
                       CSR_WIFI_HIP_UDBG2, "unifi%d: HIP CONFIG: Ctrl Info Blocks: %d\n",
                       card->instance,  card->config_data.ctrl_info_blocks));
    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,
                       CSR_WIFI_HIP_UDBG2, "unifi%d: HIP CONFIG: FromHost Block Bulk Gap: %d\n",
                       card->instance,  card->config_data.fh_pushed_block_gap));
    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,
                       CSR_WIFI_HIP_UDBG2, "unifi%d: HIP CONFIG: ToHost Slot Entries: %d\n",
                       card->instance,  card->config_data.bt_entries));
    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,
                       CSR_WIFI_HIP_UDBG2, "unifi%d: HIP CONFIG: FromHost Slot Entries: %d\n",
                       card->instance,  card->config_data.bf_entries));
    for (i = 0; i < card->fh_buffer_d.num_handles; i++)
    {
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,
                           CSR_WIFI_HIP_UDBG2, "unifi%d: HIP CONFIG: FromHost Handles: %d: %d\n",
                           card->instance,  i, card->fh_buffer_d.handles[i]));
    }

    card->fh_pulled_buffer_d.pipe_ptr = 0;
    card->fh_pulled_buffer_d.winp = 0;

    card->th_buffer_d.winp = 0;
    card->th_buffer_d.th_ptr = 0;

    card->fh_buffer_d.winp = 0;
    card->fh_buffer_d.fh_current_ctrl_ptr = 0;
    card->fh_buffer_d.fh_current_data_ptr = 0;
    card->fh_buffer_d.fh_pipe_current = 0;

    card->fh_signaldata_pending = 0;
    card->fh_bulkdata_pending = 0;

    card->bh_reason_host = 0;
    card->bh_reason_unifi = 0;
    card->bh_reason_unifi_int = 0;

    CsrMemSet((void *) &(card->ctrl_buffer), 0, sizeof(card->ctrl_buffer));

    func_exit();
    return CSR_RESULT_SUCCESS;
} /* card_init_slots() */

/*
 * ---------------------------------------------------------------------------
 *  unifi_set_udi_hook
 *
 *      Registers the udi hook that reports the sent signals to the core.
 *
 *  Arguments:
 *      card            Pointer to the card context struct
 *      udi_fn          Pointer to the callback function.
 *
 *  Returns:
 *      CSR_WIFI_HIP_RESULT_INVALID_VALUE if the card pointer is invalid,
 *      CSR_RESULT_SUCCESS on success.
 * ---------------------------------------------------------------------------
 */
CsrResult unifi_set_udi_hook(card_t *card, udi_func_t udi_fn)
{
    if (card == NULL)
    {
        return CSR_WIFI_HIP_RESULT_INVALID_VALUE;
    }

    if (card->udi_hook == NULL)
    {
        card->udi_hook = udi_fn;
    }

    return CSR_RESULT_SUCCESS;
} /* unifi_set_udi_hook() */

/*
 * ---------------------------------------------------------------------------
 *  unifi_remove_udi_hook
 *
 *      Removes the udi hook that reports the sent signals from the core.
 *
 *  Arguments:
 *      card            Pointer to the card context struct
 *      udi_fn          Pointer to the callback function.
 *
 *  Returns:
 *      CSR_WIFI_HIP_RESULT_INVALID_VALUE if the card pointer is invalid,
 *      CSR_RESULT_SUCCESS on success.
 * ---------------------------------------------------------------------------
 */
CsrResult unifi_remove_udi_hook(card_t *card, udi_func_t udi_fn)
{
    if (card == NULL)
    {
        return CSR_WIFI_HIP_RESULT_INVALID_VALUE;
    }

    if (card->udi_hook == udi_fn)
    {
        card->udi_hook = NULL;
    }

    return CSR_RESULT_SUCCESS;
} /* unifi_remove_udi_hook() */

/*
 * ---------------------------------------------------------------------------
 *  CardClearFromHostDataSlot
 *
 *      Clear a the given data slot, making it available again.
 *
 *  Arguments:
 *      card            Pointer to Card object
 *      slot            Index of the signal slot to clear.
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
void CardClearFromHostDataSlot(card_t *card, const CsrUint16 slot, CsrBool free_net_buf)
{
    CsrWifiHipBulkDataDesc *bd;

    bd = &card->from_host_data[slot].bd;

    func_enter();

    if (bd->data_length == 0)
    {
        CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID,
                              CSR_WIFI_HIP_LOG_DEF, "unifi%d: Surprise request to clear an already free FH data slot: %d\n",
                              card->instance,
                              slot));
        func_exit();
        return;
    }

    if (bd->os_data_ptr == NULL)
    {
        CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID,
                              CSR_WIFI_HIP_LOG_DEF, "unifi%d: Clearing FH data slot %d with null payload, len=%d\n",
                              card->instance,
                              slot, bd->data_length));
    }

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: CardClearFromHostDataSlot: slot %d recycled %p\n",
                           card->instance,  slot, bd->os_data_ptr));

    if (free_net_buf)
    {
        /* Free card->from_host_data[slot].bd.os_net_ptr here. */
        unifi_free_bulk_data(card, bd);
        if (card->from_host_data[slot].qsig)
        {
            (void) csrWifiHipPacketSchedulerFreeSlot(card, card->from_host_data[slot].qsig);
        }
        else
        {
            CSR_LOG_TEXT_CRITICAL((
                                      CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: CardClearFromHostDataSlot: slot %d has no qsig\n",
                                      card->instance,  slot));
        }
    }
    bd->data_length = 0;
    bd->os_data_ptr = NULL;
    bd->net_buf_length = 0;

    card_free_slot(card, slot);

    func_exit();
} /* CardClearFromHostDataSlot() */

CsrUint32 CardGetDataSlotSize(card_t *card)
{
    return card->buff_maximum_size_bytes;
} /* CardGetDataSlotSize() */

/*
 * ---------------------------------------------------------------------------
 *  CardGetFreeFromHostDataSlots
 *
 *      Retrieve the number of from-host bulk data slots available.
 *
 *  Arguments:
 *      card            Pointer to the card context struct
 *
 *  Returns:
 *      Number of free from-host bulk data slots.
 * ---------------------------------------------------------------------------
 */
CsrUint16 CardGetFreeFromHostDataSlots(card_t *card)
{
    return card->slot_desc_list_freecount;
} /* CardGetFreeFromHostDataSlots() */

/*
 * ---------------------------------------------------------------------------
 *  CardAreAllFromHostDataSlotsEmpty
 *
 *      Returns the state of from-host bulk data slots.
 *
 *  Arguments:
 *      card            Pointer to the card context struct
 *
 *  Returns:
 *      1       The from-host bulk data slots are all empty (available).
 *      0       Some or all the from-host bulk data slots are in use.
 * ---------------------------------------------------------------------------
 */
CsrUint16 CardAreAllFromHostDataSlotsEmpty(card_t *card)
{
    if (card->slot_desc_list_freecount == card->config_data.bf_entries)
    {
        return 1;
    }
    return 0;
} /* CardGetFreeFromHostDataSlots() */

static CsrResult unifi_identify_hw(card_t *card)
{
    func_enter();

    card->chip_id = card->sdio_if->sdioId.cardId;
    card->function = card->sdio_if->sdioId.sdioFunction;
    card->sdio_io_block_size = card->sdio_if->blockSize;

    /* If SDIO controller doesn't support byte mode CMD53, pad transfers to block sizes */
    card->sdio_io_block_pad = (card->sdio_if->features & CSR_SDIO_FEATURE_BYTE_MODE) ? FALSE : TRUE;

#ifdef CSR_WIFI_DRIVER_HYDRA
    /* On Hydra chips don't use the chiphelper */
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,
                        CSR_WIFI_HIP_LOG_DEF, "unifi%d: Chip ID 0x%02X  Function %u  Block Size %u\n",
                        card->instance,
                        card->chip_id, card->function, card->sdio_io_block_size));
#else
    /*
     * Setup the chip helper so that we can access the registers (and
     * also tell what sub-type of HIP we should use).
     */
    card->helper = ChipHelper_GetVersionSdio((CsrUint8) card->chip_id);
    if (!card->helper)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Null ChipHelper\n", card->instance));
        return CSR_RESULT_FAILURE;
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,
                        CSR_WIFI_HIP_LOG_DEF, "unifi%d: Chip ID 0x%02X  Function %u  Block Size %u  Name %s(%s)\n",
                        card->instance,
                        card->chip_id, card->function, card->sdio_io_block_size,
                        ChipHelper_MarketingName(card->helper),
                        ChipHelper_FriendlyName(card->helper)));
#endif
    func_exit();
    return CSR_RESULT_SUCCESS;
} /* unifi_identify_hw() */

static CsrResult unifi_prepare_hw(card_t *card)
{
    CsrResult r;
    CsrResult csrResult;
#ifndef CSR_WIFI_DRIVER_HYDRA
    enum unifi_host_state old_state = card->host_state;
#endif

    func_enter();

    r = unifi_identify_hw(card);
    if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
    {
        return r;
    }
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to identify hw\n", card->instance));
        func_exit_r(r);
        return r;
    }

    CSR_LOG_TEXT_INFO((
                          CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                          "unifi%d: %s mode SDIO\n",
                          card->instance,  card->sdio_io_block_pad ? "Block" : "Byte"));
    /*
     * Chip must be a awake or blocks that are asleep may not get
     * reset.  We can only do this after we have read the chip_id.
     */
    r = unifi_set_host_state(card, UNIFI_HOST_STATE_AWAKE);
    if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
    {
        return r;
    }

#ifndef CSR_WIFI_DRIVER_HYDRA
    if (old_state == UNIFI_HOST_STATE_TORPID)
    {
        /* Ensure the initial clock rate is set; if a reset occured when the chip was
         * TORPID, unifi_set_host_state() may have raised it to MAX.
         */
        csrResult = CsrSdioMaxBusClockFrequencySet(card->sdio_if, UNIFI_SDIO_CLOCK_INIT_HZ);
        if (csrResult != CSR_RESULT_SUCCESS)
        {
            r = ConvertCsrSdioToCsrHipResult(card, csrResult);
            func_exit_r(r);
            return r;
        }
        card->sdio_clock_speed = UNIFI_SDIO_CLOCK_INIT_HZ;
    }
#endif

    /*
     * The WLAN function must be enabled to access MAILBOX2 and DEBUG_RST
     * registers.
     */
    csrResult = CsrSdioFunctionEnable(card->sdio_if);
    if (csrResult == CSR_SDIO_RESULT_NO_DEVICE)
    {
        return CSR_WIFI_HIP_RESULT_NO_DEVICE;
    }
    if (csrResult != CSR_RESULT_SUCCESS)
    {
        r = ConvertCsrSdioToCsrHipResult(card, csrResult);
        /* Can't enable WLAN function. Try resetting the SDIO block. */
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to re-enable function %d.\n",
                               card->instance,  card->function));
        func_exit_r(r);
        return r;
    }

#ifndef CSR_WIFI_DRIVER_HYDRA
    /*
     * Poke some registers to make sure the PLL has started,
     * otherwise memory accesses are likely to fail.
     */
    bootstrap_chip_hw(card);
#endif

    /* Try to read the chip version from register. */
    r = unifi_read_chip_version(card);
    if (r != CSR_RESULT_SUCCESS)
    {
        func_exit_r(r);
        return r;
    }

    func_exit();
    return CSR_RESULT_SUCCESS;
} /* unifi_prepare_hw() */

static CsrResult unifi_read_chip_version(card_t *card)
{
    CsrResult r = CSR_RESULT_SUCCESS;
#ifndef CSR_WIFI_DRIVER_HYDRA
    CsrUint16 ver;
    CsrUint32 gbl_chip_version;
#endif

    func_enter();

#ifdef CSR_WIFI_DRIVER_HYDRA
    /* D-30105: Fix this: There needs to be an API to get this from the Curator. For now
     * it will be hardcoded.
     */
    card->chip_version = 0x0044;
#else
    gbl_chip_version = ChipHelper_GBL_CHIP_VERSION(card->helper);

    /* Try to read the chip version from register. */
    if (gbl_chip_version != 0)
    {
        r = unifi_read_direct16(card, gbl_chip_version * 2, &ver);
        if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
        {
            return r;
        }
        if (r != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to read GBL_CHIP_VERSION\n", card->instance));
            func_exit_r(r);
            return r;
        }
        card->chip_version = ver;
    }
    else
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Unknown Chip ID, cannot locate GBL_CHIP_VERSION\n", card->instance));
        r = CSR_RESULT_FAILURE;
    }
#endif

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,
                        CSR_WIFI_HIP_LOG_DEF, "unifi%d: Chip Version 0x%04X\n",
                        card->instance,  card->chip_version));

    func_exit_r(r);
    return r;
} /* unifi_read_chip_version() */

/*
 * ---------------------------------------------------------------------------
 *  unifi_reset_hardware
 *
 *      Execute the UniFi reset sequence.
 *
 *      Note: This may fail if the chip is going TORPID so retry at
 *      least once.
 *
 *  Arguments:
 *      card - pointer to card context structure
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success, CSR error otherwise.
 *
 *  Notes:
 *      Some platforms (e.g. Windows Vista) do not allow access to registers
 *      that are necessary for a software soft reset.
 * ---------------------------------------------------------------------------
 */
static CsrResult unifi_reset_hardware(card_t *card)
{
    CsrResult r;
    CsrUint16 new_block_size = UNIFI_IO_BLOCK_SIZE;
    CsrResult csrResult;

    func_enter();

    /* Errors returned by unifi_prepare_hw() are not critical at this point */
    r = unifi_prepare_hw(card);
    if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
    {
        return r;
    }

#ifdef CSR_WIFI_DRIVER_HYDRA
    /* On Hydra chips, the chip will have been reset as needed and the WLAN
     * subsystem initialised before HIP init has started.
     */
#else
    /* First try SDIO controller reset, which may power cycle the UniFi, assert
     * its reset line, or not be implemented depending on the platform.
     */
    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2, "unifi%d: Calling CsrSdioHardReset\n", card->instance));
    csrResult = CsrSdioHardReset(card->sdio_if);
    if (csrResult == CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrSdioHardReset succeeded on reseting UniFi\n", card->instance));
        r = unifi_prepare_hw(card);
        if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
        {
            return r;
        }
        if (r != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_prepare_hw failed after hard reset\n", card->instance));
            func_exit_r(r);
            return r;
        }
    }
    else if (csrResult == CSR_SDIO_RESULT_NO_DEVICE)
    {
        return CSR_WIFI_HIP_RESULT_NO_DEVICE;
    }
    else
    {
        /* Falling back to software hard reset methods */
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Falling back to software hard reset\n", card->instance));
        r = unifi_card_hard_reset(card);
        if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
        {
            return r;
        }
        if (r != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: software hard reset failed\n", card->instance));
            func_exit_r(r);
            return r;
        }

        /* If we fell back to unifi_card_hard_reset() methods, chip version may
         * not have been read. (Note in the unlikely event that it is zero,
         * it will be harmlessly read again)
         */
        if (card->chip_version == 0)
        {
            r = unifi_read_chip_version(card);
            if (r != CSR_RESULT_SUCCESS)
            {
                func_exit_r(r);
                return r;
            }
        }
    }
#endif  /* !CSR_WIFI_DRIVER_HYDRA */

#ifdef CSR_WIFI_HIP_SDIO_BLOCK_SIZE
    new_block_size = CSR_WIFI_HIP_SDIO_BLOCK_SIZE;
#endif

    /* After hard reset, we need to restore the SDIO block size */
    csrResult = CsrSdioBlockSizeSet(card->sdio_if, new_block_size);
    r = ConvertCsrSdioToCsrHipResult(card, csrResult);

    /* Warn if a different block size was achieved by the transport */
    if (card->sdio_if->blockSize != new_block_size)
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,
                            CSR_WIFI_HIP_LOG_DEF, "unifi%d: Actually got block size %d\n",
                            card->instance,  card->sdio_if->blockSize));
    }

    /* sdio_io_block_size always needs be updated from the achieved block size,
     * as it is used by the OS layer to allocate memory in unifi_net_malloc().
     * Controllers which don't support block mode (e.g. CSPI) will report a
     * block size of zero.
     */
    if (card->sdio_if->blockSize == 0)
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Block size 0, block mode not available\n", card->instance));

        /* Set sdio_io_block_size to 1 so that priv->allocFrame() has a
         * sensible rounding value. Elsewhere padding will already be
         * disabled because the controller supports byte mode.
         */
        card->sdio_io_block_size = 1;

        /* Controller features must declare support for byte mode */
        if (!(card->sdio_if->features & CSR_SDIO_FEATURE_BYTE_MODE))
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Requires byte mode\n", card->instance));
            r = CSR_WIFI_HIP_RESULT_INVALID_VALUE;
        }
    }
    else
    {
        /* Padding will be enabled if CSR_SDIO_FEATURE_BYTE_MODE isn't set */
        card->sdio_io_block_size = card->sdio_if->blockSize;
    }


    func_exit_r(r);
    return r;
} /* unifi_reset_hardware() */

/*
 * ---------------------------------------------------------------------------
 *  card_reset_method_io_enable
 *
 *      Issue a hard reset to the hw writing the IO_ENABLE.
 *
 *  Arguments:
 *      card            Pointer to Card object
 *
 *  Returns:
 *      0 on success,
 *      CSR_WIFI_HIP_RESULT_NO_DEVICE   if the card was ejected
 *      CSR_RESULT_FAILURE         if an SDIO error occurred or if a response
 *                                 was not seen in the expected time
 * ---------------------------------------------------------------------------
 */
#ifndef CSR_WIFI_DRIVER_HYDRA
static CsrResult card_reset_method_io_enable(card_t *card)
{
    CsrResult r;
    CsrResult csrResult;

    func_enter();

    /*
     * This resets only function 1, so should be used in
     * preference to the method below (CSR_FUNC_EN)
     */
    CSR_LOG_TEXT_INFO((
                          CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: Hard reset (IO_ENABLE)\n"
                          , card->instance));

    csrResult = CsrSdioFunctionDisable(card->sdio_if);
    if (csrResult == CSR_SDIO_RESULT_NO_DEVICE)
    {
        return CSR_WIFI_HIP_RESULT_NO_DEVICE;
    }
    if (csrResult != CSR_RESULT_SUCCESS)
    {
        r = ConvertCsrSdioToCsrHipResult(card, csrResult);
        CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID,
                              CSR_WIFI_HIP_LOG_DEF, "unifi%d: SDIO error writing IO_ENABLE: %d\n",
                              card->instance,  r));
    }
    else
    {
        /* Delay here to let the reset take affect. */
        CsrThreadSleep(RESET_SETTLE_DELAY);

        r = card_wait_for_unifi_to_disable(card);
        if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
        {
            return r;
        }

        if (r == CSR_RESULT_SUCCESS)
        {
            r = card_wait_for_unifi_to_reset(card);
            if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
            {
                return r;
            }
        }
    }

    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_INFO((
                              CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: Hard reset (CSR_FUNC_EN)\n"
                              , card->instance));

        r = sdio_write_f0(card, SDIO_CSR_FUNC_EN, 0);
        if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
        {
            return r;
        }
        if (r != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID,
                                  CSR_WIFI_HIP_LOG_DEF, "unifi%d: SDIO error writing SDIO_CSR_FUNC_EN: %d\n",
                                  card->instance,  r));
            func_exit_r(r);
            return r;
        }
        else
        {
            /* Delay here to let the reset take affect. */
            CsrThreadSleep(RESET_SETTLE_DELAY);

            r = card_wait_for_unifi_to_reset(card);
            if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
            {
                return r;
            }
        }
    }

    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: card_reset_method_io_enable failed to reset UniFi\n", card->instance));
    }

    func_exit();
    return r;
} /* card_reset_method_io_enable() */

#endif

/*
 * ---------------------------------------------------------------------------
 *  card_reset_method_dbg_reset
 *
 *      Issue a hard reset to the hw writing the DBG_RESET.
 *
 *  Arguments:
 *      card            Pointer to Card object
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS         on success,
 *      CSR_WIFI_HIP_RESULT_NO_DEVICE   if the card was ejected
 *      CSR_RESULT_FAILURE         if an SDIO error occurred or if a response
 *                                 was not seen in the expected time
 * ---------------------------------------------------------------------------
 */
#ifndef CSR_WIFI_DRIVER_HYDRA
static CsrResult card_reset_method_dbg_reset(card_t *card)
{
    CsrResult r;

    func_enter();

    /*
     * Prepare UniFi for h/w reset
     */
    if (card->host_state == UNIFI_HOST_STATE_TORPID)
    {
        r = unifi_set_host_state(card, UNIFI_HOST_STATE_DROWSY);
        if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
        {
            return r;
        }
        if (r != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to set UNIFI_HOST_STATE_DROWSY\n", card->instance));
            func_exit_r(r);
            return r;
        }
        CsrThreadSleep(5);
    }

    r = unifi_card_stop_processor(card, UNIFI_PROC_BOTH);
    if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
    {
        return r;
    }
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Can't stop processors\n", card->instance));
        func_exit();
        return r;
    }

    CSR_LOG_TEXT_INFO((
                          CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: Hard reset (DBG_RESET)\n"
                          , card->instance));

    /*
     * This register write may fail. The debug reset resets
     * parts of the Function 0 sections of the chip, and
     * therefore the response cannot be sent back to the host.
     */
    r = unifi_write_direct_8_or_16(card, ChipHelper_DBG_RESET(card->helper) * 2, 1);
    if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
    {
        return r;
    }
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID,
                              CSR_WIFI_HIP_LOG_DEF, "unifi%d: SDIO error writing DBG_RESET: %d\n",
                              card->instance,  r));
        func_exit_r(r);
        return r;
    }

    /* Delay here to let the reset take affect. */
    CsrThreadSleep(RESET_SETTLE_DELAY);

    r = card_wait_for_unifi_to_reset(card);
    if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
    {
        return r;
    }
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: card_reset_method_dbg_reset failed to reset UniFi\n", card->instance));
    }

    func_exit();
    return r;
} /* card_reset_method_dbg_reset() */

#endif

/*
 * ---------------------------------------------------------------------------
 *  unifi_card_hard_reset
 *
 *      Issue reset to hardware, by writing to registers on the card.
 *      Power to the card is preserved.
 *
 *  Arguments:
 *      card            Pointer to Card object
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS         on success,
 *      CSR_WIFI_HIP_RESULT_NO_DEVICE   if the card was ejected
 *      CSR_RESULT_FAILURE         if an SDIO error occurred or if a response
 *                                 was not seen in the expected time
 * ---------------------------------------------------------------------------
 */
CsrResult unifi_card_hard_reset(card_t *card)
{
#ifdef CSR_WIFI_DRIVER_HYDRA
    return CSR_RESULT_FAILURE;
#else
    CsrResult r;
    const struct chip_helper_reset_values *init_data;
    CsrUint32 chunks;

    func_enter();

    /* Clear cache of page registers */
    card->proc_select = (CsrUint32) (-1);
    card->dmem_page = (CsrUint32) (-1);
    card->pmem_page = (CsrUint32) (-1);

    /*
     * We need to have a valid card->helper before we use software hard reset.
     * If unifi_identify_hw() fails to get the card ID, it probably means
     * that there is no way to talk to the h/w.
     */
    r = unifi_identify_hw(card);
    if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
    {
        return r;
    }
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_card_hard_reset failed to identify h/w\n", card->instance));
        func_exit();
        return r;
    }

    /* Search for some reset code. */
    chunks = ChipHelper_HostResetSequence(card->helper, &init_data);
    if (chunks != 0)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Hard reset (Code download) is unsupported\n", card->instance));

        func_exit_r(CSR_RESULT_FAILURE);
        return CSR_RESULT_FAILURE;
    }

    if (card->chip_id > SDIO_CARD_ID_UNIFI_2)
    {
        /* The HIP spec considers this a bus-specific reset.
         * This resets only function 1, so should be used in
         * preference to the method below (CSR_FUNC_EN)
         * If this method fails, it means that the f/w is probably
         * not running. In this case, try the DBG_RESET method.
         */
        r = card_reset_method_io_enable(card);
        if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
        {
            return r;
        }
        if (r == CSR_RESULT_SUCCESS)
        {
            func_exit();
            return r;
        }
    }

    /* Software hard reset */
    r = card_reset_method_dbg_reset(card);

    func_exit_r(r);
    return r;
#endif
} /* unifi_card_hard_reset() */

/*
 * ---------------------------------------------------------------------------
 *
 *  CardGenInt
 *
 *      Prod the card.
 *      This function causes an internal interrupt to be raised in the
 *      UniFi chip. It is used to signal the firmware that some action has
 *      been completed.
 *      The UniFi Host Interface asks that the value used increments for
 *      debugging purposes.
 *
 *  Arguments:
 *      card            Pointer to Card object
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS         on success,
 *      CSR_WIFI_HIP_RESULT_NO_DEVICE   if the card was ejected
 *      CSR_RESULT_FAILURE         if an SDIO error occurred or if a response
 *                                 was not seen in the expected time
 * ---------------------------------------------------------------------------
 */
CsrResult CardGenInt(card_t *card)
{
    CsrResult r;

    func_enter();

#ifdef CSR_WIFI_DRIVER_HYDRA
    /* Hydra uses a from-host scratch register to interrupt the f/w */

    /* D-30105: Fix this: For HIP2 this may not be used, this is how f/w interrupts were
     * generated for Hydra HIP1
     */
    r = unifi_write_direct_8_or_16(card, UNIFI_SDIO_HOST_INT_REQ,
                                   (CsrUint8) card->unifi_interrupt_seq);
#else
    r = sdio_write_f0(card, SDIO_CSR_FROM_HOST_SCRATCH0,
                      (CsrUint8) card->unifi_interrupt_seq);
#endif
    if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
    {
        return r;
    }
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: SDIO error writing UNIFI_SHARED_IO_INTERRUPT: %d\n",
                               card->instance,  r));
        func_exit_r(r);
        return r;
    }

    card->unifi_interrupt_seq++;

    func_exit();
    return CSR_RESULT_SUCCESS;
} /* CardGenInt() */

/*
 * ---------------------------------------------------------------------------
 *  CardEnableInt
 *
 *      Enable the outgoing SDIO interrupt from UniFi to the host.
 *
 *  Arguments:
 *      card            Pointer to Card object
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS            on success,
 *      CSR_WIFI_HIP_RESULT_NO_DEVICE      if the card was ejected
 *      CSR_RESULT_FAILURE            if an SDIO error occurred,
 * ---------------------------------------------------------------------------
 */
CsrResult CardEnableInt(card_t *card)
{
    CsrResult r;
    CsrUint8 int_enable;

    r = sdio_read_f0(card, SDIO_INT_ENABLE, &int_enable);
    if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
    {
        return r;
    }
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: SDIO error reading SDIO_INT_ENABLE\n", card->instance));
        return r;
    }

    int_enable |= (CsrUint8) ((1 << card->function) | UNIFI_SD_INT_ENABLE_IENM);

    r = sdio_write_f0(card, SDIO_INT_ENABLE, int_enable);
    if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
    {
        return r;
    }
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: SDIO error writing SDIO_INT_ENABLE\n", card->instance));
        return r;
    }

    return CSR_RESULT_SUCCESS;
} /* CardEnableInt() */

/*
 * ---------------------------------------------------------------------------
 *  CardDisableInt
 *
 *      Disable the outgoing SDIO interrupt from UniFi to the host.
 *
 *  Arguments:
 *      card            Pointer to Card object
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS            on success,
 *      CSR_WIFI_HIP_RESULT_NO_DEVICE      if the card was ejected
 *      CSR_RESULT_FAILURE            if an SDIO error occurred,
 * ---------------------------------------------------------------------------
 */
CsrResult CardDisableInt(card_t *card)
{
    CsrResult r;
    CsrUint8 int_enable;

    r = sdio_read_f0(card, SDIO_INT_ENABLE, &int_enable);
    if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
    {
        return r;
    }
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: SDIO error reading SDIO_INT_ENABLE\n", card->instance));
        return r;
    }

    int_enable &= ~(1 << card->function);

    r = sdio_write_f0(card, SDIO_INT_ENABLE, int_enable);
    if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
    {
        return r;
    }
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: SDIO error writing SDIO_INT_ENABLE\n", card->instance));
        return r;
    }

    return CSR_RESULT_SUCCESS;
} /* CardDisableInt() */

/*
 * ---------------------------------------------------------------------------
 *  CardPendingInt
 *
 *      Determine whether UniFi is currently asserting the SDIO interrupt
 *      request.
 *
 *  Arguments:
 *      card            Pointer to Card object
 *      pintr           Pointer to location to write interrupt status,
 *                          TRUE if interrupt pending,
 *                          FALSE if no interrupt pending.
 *  Returns:
 *      CSR_RESULT_SUCCESS            interrupt status read successfully
 *      CSR_WIFI_HIP_RESULT_NO_DEVICE      if the card was ejected
 *      CSR_RESULT_FAILURE            if an SDIO error occurred,
 * ---------------------------------------------------------------------------
 */
CsrResult CardPendingInt(card_t *card, CsrBool *pintr)
{
    CsrResult r;
    CsrUint8 pending;

    *pintr = FALSE;

    r = sdio_read_f0(card, SDIO_INT_PENDING, &pending);
    if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
    {
        return r;
    }
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: SDIO error reading SDIO_INT_PENDING\n", card->instance));
        return r;
    }

    *pintr = (pending & (1 << card->function)) ? TRUE : FALSE;

    return CSR_RESULT_SUCCESS;
} /* CardPendingInt() */

/*
 * ---------------------------------------------------------------------------
 *  CardClearInt
 *
 *      Clear the UniFi SDIO interrupt request.
 *
 *  Arguments:
 *      card            Pointer to Card object
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS          if pending interrupt was cleared, or no pending interrupt.
 *      CSR_WIFI_HIP_RESULT_NO_DEVICE    if the card was ejected
 *      CSR_RESULT_FAILURE          if an SDIO error occurred,
 * ---------------------------------------------------------------------------
 */
CsrResult CardClearInt(card_t *card)
{
    CsrResult r;

#ifdef CSR_WIFI_DRIVER_HYDRA
    r = sdio_write_f0(card, SDIO_CSR_HOST_INT_CLEAR, (1 << card->function));
#else
    r = sdio_write_f0(card, SDIO_CSR_HOST_INT_CLEAR, 1);
#endif
    if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
    {
        return r;
    }
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: SDIO error writing SDIO_CSR_HOST_INT_CLEAR\n", card->instance));
    }

    return r;
} /* CardClearInt() */

/*
 * ---------------------------------------------------------------------------
 *  CardIntEnabled
 *
 *      Determine whether UniFi is currently asserting the SDIO interrupt
 *      request.
 *
 *  Arguments:
 *      card            Pointer to Card object
 *      enabled         Pointer to location to write interrupt enable status,
 *                          TRUE if interrupts enabled,
 *                          FALSE if interupts disabled.
 *
 *  Returns:
 *      CSR_WIFI_HIP_RESULT_NO_DEVICE      if the card was ejected
 *      CSR_RESULT_FAILURE            if an SDIO error occurred,
 * ---------------------------------------------------------------------------
 */
CsrResult CardIntEnabled(card_t *card, CsrBool *enabled)
{
    CsrResult r;
    CsrUint8 int_enable;

    r = sdio_read_f0(card, SDIO_INT_ENABLE, &int_enable);
    if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
    {
        return r;
    }
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: SDIO error reading SDIO_INT_ENABLE\n", card->instance));
        return r;
    }

    *enabled = (int_enable & (1 << card->function)) ? TRUE : FALSE;

    return CSR_RESULT_SUCCESS;
} /* CardIntEnabled() */

/*
 * ---------------------------------------------------------------------------
 *  card_init_slot_list
 *
 *  Initialise array of slot structures into a free list.
 *
 *  Arguments:
 *      card            Pointer to Card object
 *
 *  Returns:
 *      void
 * ---------------------------------------------------------------------------
 */
static void card_init_slot_list(card_t *card)
{
    CsrInt32 i;

    slot_desc_t *sd;
    CSR_LOG_TEXT_INFO((
                          CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG2,  "unifi%d: card_init_slot_list: card->from_host_data=%p sizeof(*sd)=%d\n",
                          card->instance,  card->from_host_data, sizeof(slot_desc_t)));

    card->slot_desc_list_freecount = 0;
    card->freelist = 0;
    for (i = 0, sd = card->from_host_data; i < card->config_data.bf_entries; i++, sd++)
    {
        sd->slot = (CsrUint16) i;
        sd->qsig = 0;
        UNIFI_INIT_BULK_DATA(&sd->bd);
        card_free_slot(card, i);
    }
}

/*
 * ---------------------------------------------------------------------------
 *  card_allocate_slot
 *
 *  Take the first free slot descriptor from the free list.
 *
 *  Arguments:
 *      card            Pointer to Card object
 *      slot            Slot #
 *
 *  Returns:
 *      void
 * ---------------------------------------------------------------------------
 */
static CsrInt16 card_allocate_slot(card_t *card, CsrUint16 *slot, CsrWifiHipPacketSchedulerQsig *qsig)
{
    slot_desc_t *sd;

    sd = card->freelist;
    if (sd)
    {
        card->freelist = sd->next;
        sd->next = NULL;
        *slot = sd->slot;
        sd->qsig = qsig;
        card->slot_desc_list_freecount--;
        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: card_allocate_slot: slot=%d sd=%p\n",
                               card->instance,  *slot, sd));
        return 0;
    }
    CSR_LOG_TEXT_INFO((
                          CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi : card_allocate_slot: no slot to allocate\n"));
    return -1;
}

static void card_free_slot(card_t *card, CsrUint16 slot)
{
    slot_desc_t *sd;

    if (slot >= card->config_data.bf_entries)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: card_free_slot: slot (%d) out of range\n",
                               card->instance,  slot));
        return;
    }
    sd = &card->from_host_data[slot];

    sd->next = card->freelist;
    card->freelist = sd;
    card->slot_desc_list_freecount++;
    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: card_free_slot: slot=%d sd=%p sd->next=%p freelist=%p\n",
                           card->instance,  slot, sd, sd->next, card->freelist));
}

CsrResult CardClearSlotQsigPtr(card_t *card, CsrUint16 slot, CsrWifiHipPacketSchedulerQsig *qsig)
{
    slot_desc_t *sd;

    if (slot >= card->config_data.bf_entries)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: CardClearSlotQsigPtr: slot (%d) out of range\n",
                               card->instance,  slot));
        return CSR_RESULT_FAILURE;
    }
    sd = &card->from_host_data[slot];
    if (sd->qsig != qsig)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: CardClearSlotQsigPtr: slot qsig (0x%p) does not match parameter (0x%p)\n",
                               card->instance,  sd->qsig, qsig));
        return CSR_RESULT_FAILURE;
    }
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,
                        CSR_WIFI_HIP_UDBG2, "unifi%d: CardClearSlotQsigPtr: cleared slot %d  of qsig (0x%p)\n",
                        card->instance,  slot, sd->qsig));
    sd->qsig = 0;
    return CSR_RESULT_SUCCESS;
}

CsrResult CardWriteBulkData(card_t *card, CsrWifiHipPacketSchedulerQsig *qsig, CsrUint16 slots_p[], CsrUint16 *slots_c)
{
    CsrUint16 i, j = 0;
    CsrUint8 *packed_sigptr = NULL;
    CsrWifiHipBulkDataDesc *bulkdata;
    CsrUint16 slots[UNIFI_MAX_DATA_REFERENCES];
    card_signal_t *csptr;

    func_enter();

    csptr = qsig_get_signal_ptr(qsig);
    bulkdata = csptr->bulkdata;

    /* Count the number of slots required */
    for (i = 0; i < UNIFI_MAX_DATA_REFERENCES; i++)
    {
        if (bulkdata[i].data_length != 0)
        {
            CSR_LOG_TEXT_DEBUG((
                                   CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: CardWriteBulkData: index=%d data_length=%d\n",
                                   card->instance,  i, bulkdata[i].data_length));
            if (card_allocate_slot(card, &slots[i], qsig) != 0)
            {
                for (j = 0; j < i; j++)
                {
                    card_free_slot(card, slots[j]);
                }
                return CSR_RESULT_FAILURE;
            }
        }
    }
    packed_sigptr = csptr->sigbuf;

    /* Fill in the slots with data */
    j = 0;
    for (i = 0; i < UNIFI_MAX_DATA_REFERENCES; i++)
    {
        if (bulkdata[i].data_length == 0)
        {
            /*lint -save -e572*/
            /* Zero-out the DATAREF in the signal */
            SET_PACKED_DATAREF_SLOT(packed_sigptr, i, 0);
            SET_PACKED_DATAREF_LEN(packed_sigptr, i, 0);
            /*lint -restore*/
        }
        else
        {
            /*
             * Fill in the slot number in the SIGNAL structure but
             * preserve the offset already in there
             */
            SET_PACKED_DATAREF_SLOT(packed_sigptr, i, slots[j]);
            SET_PACKED_DATAREF_LEN(packed_sigptr, i, bulkdata[i].data_length);

            CSR_LOG_TEXT_DEBUG((
                                   CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: CardWriteBulkData: slot=%d len=%d\n",
                                   card->instance,  slots[j], bulkdata[i].data_length));
            /* Do not copy the data, just store the information to them */
            card->from_host_data[slots[j]].bd.os_data_ptr = bulkdata[i].os_data_ptr;
            card->from_host_data[slots[j]].bd.os_net_buf_ptr = bulkdata[i].os_net_buf_ptr;
            card->from_host_data[slots[j]].bd.data_length = bulkdata[i].data_length;
            card->from_host_data[slots[j]].bd.net_buf_length = bulkdata[i].net_buf_length;

            slots_p[(*slots_c)++] = slots[j];
            CSR_LOG_TEXT_DEBUG((
                                   CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: CardWriteBulkData sig=0x%x, fh slot %d = %p\n",
                                   card->instance,
                                   GET_SIGNAL_ID(packed_sigptr), i, bulkdata[i].os_data_ptr));

            /* Sanity-check that the bulk data desc being assigned to the slot
             * actually has a payload.
             */
            if (!bulkdata[i].os_data_ptr)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: Assign null os_data_ptr (len=%d) fh slot %d, i=%d, sig=0x%x",
                                       card->instance,
                                       bulkdata[i].data_length, slots[j], i, GET_SIGNAL_ID(packed_sigptr)));
            }

            j++;
        }
    }

    func_exit();

    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  card_find_data_slot
 *
 *      Dereference references to bulk data slots into pointers to real data.
 *
 *  Arguments:
 *      card            Pointer to the card struct.
 *      slot            Slot number from a signal structure
 *
 *  Returns:
 *      Pointer to entry in bulk_data_slot array.
 * ---------------------------------------------------------------------------
 */
CsrWifiHipBulkDataDesc *card_find_data_slot(card_t *card, CsrInt16 slot)
{
    CsrInt16 sn;
    CsrWifiHipBulkDataDesc *bd;

    sn = slot & 0x7FFF;

    /* ?? check sanity of slot number ?? */

    if (slot & SLOT_DIR_TO_HOST)
    {
        bd = &card->to_host_data[sn];
    }
    else
    {
        bd = &card->from_host_data[sn].bd;
    }

    return bd;
} /* card_find_data_slot() */

/*
 * ---------------------------------------------------------------------------
 *  firmware_present_in_flash
 *
 *      Probe for external Flash that looks like it might contain firmware.
 *
 *      If Flash is not present, reads always return 0x0008.
 *      If Flash is present, but empty, reads return 0xFFFF.
 *      Anything else is considered to be firmware.
 *
 *  Arguments:
 *      card        Pointer to card struct
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS                 firmware is present in ROM or flash
 *      CSR_WIFI_HIP_RESULT_NOT_FOUND      firmware is not present in ROM or flash
 *      CSR_WIFI_HIP_RESULT_NO_DEVICE      if the card was ejected
 *      CSR_RESULT_FAILURE                 if an SDIO error occurred
 * ---------------------------------------------------------------------------
 */
#ifndef CSR_WIFI_DRIVER_HYDRA
static CsrResult firmware_present_in_flash(card_t *card)
{
    CsrResult r;
    CsrUint16 m1, m5;

    if (ChipHelper_HasRom(card->helper))
    {
        return CSR_RESULT_SUCCESS;
    }
    if (!ChipHelper_HasFlash(card->helper))
    {
        return CSR_WIFI_HIP_RESULT_NOT_FOUND;
    }

    /*
     * Examine the Flash locations that are the power-on default reset
     * vectors of the XAP processors.
     * These are words 1 and 5 in Flash.
     */
    r = unifi_card_read16(card, UNIFI_MAKE_GP(EXT_FLASH, 2), &m1);
    if (r != CSR_RESULT_SUCCESS)
    {
        return r;
    }

    r = unifi_card_read16(card, UNIFI_MAKE_GP(EXT_FLASH, 10), &m5);
    if (r != CSR_RESULT_SUCCESS)
    {
        return r;
    }

    /* Check for uninitialised/missing flash */
    if ((m1 == 0x0008) || (m1 == 0xFFFF) ||
        (m1 == 0x0004) || (m5 == 0x0004) ||
        (m5 == 0x0008) || (m5 == 0xFFFF))
    {
        return CSR_WIFI_HIP_RESULT_NOT_FOUND;
    }

    return CSR_RESULT_SUCCESS;
} /* firmware_present_in_flash() */

#endif /* !CSR_WIFI_DRIVER_HYDRA */

/*
 * ---------------------------------------------------------------------------
 *  bootstrap_chip_hw
 *
 *      Perform chip specific magic to "Get It Working" TM.  This will
 *      increase speed of PLLs in analogue and maybe enable some
 *      on-chip regulators.
 *
 *  Arguments:
 *      card            Pointer to card struct
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
#ifndef CSR_WIFI_DRIVER_HYDRA
static void bootstrap_chip_hw(card_t *card)
{
    const struct chip_helper_init_values *vals;
    CsrUint32 i, len;
    void *sdio = card->sdio_if;
    CsrResult csrResult;

    len = ChipHelper_ClockStartupSequence(card->helper, &vals);
    if (len != 0)
    {
        for (i = 0; i < len; i++)
        {
            csrResult = CsrSdioWrite16(sdio, vals[i].addr * 2, vals[i].value);
            if (csrResult != CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID,
                                      CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to write bootstrap value %d\n",
                                      card->instance,  i));
                /* Might not be fatal */
            }

            CsrThreadSleep(1);
        }
    }
} /* bootstrap_chip_hw() */

#endif

/*
 * ---------------------------------------------------------------------------
 *  unifi_card_stop_processor
 *
 *      Stop the UniFi XAP processors.
 *
 *  Arguments:
 *      card            Pointer to card struct
 *      which           One of UNIFI_PROC_MAC, UNIFI_PROC_PHY, UNIFI_PROC_BOTH
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS if successful, or CSR error code
 * ---------------------------------------------------------------------------
 */
CsrResult unifi_card_stop_processor(card_t *card, enum unifi_dbg_processors_select which)
{
    CsrResult r = CSR_RESULT_SUCCESS;
#ifndef CSR_WIFI_DRIVER_HYDRA
    CsrUint8 status;
    CsrInt16 retry = 100;

    while (retry--)
    {
        /* Select both XAPs */
        r = unifi_set_proc_select(card, which);
        if (r != CSR_RESULT_SUCCESS)
        {
            break;
        }

        /* Stop processors */
        r = unifi_write_direct16(card, ChipHelper_DBG_EMU_CMD(card->helper) * 2, 2);
        if (r != CSR_RESULT_SUCCESS)
        {
            break;
        }

        /* Read status */
        r = unifi_read_direct_8_or_16(card,
                                      ChipHelper_DBG_HOST_STOP_STATUS(card->helper) * 2,
                                      &status);
        if (r != CSR_RESULT_SUCCESS)
        {
            break;
        }

        if ((status & 1) == 1)
        {
            /* Success! */
            return CSR_RESULT_SUCCESS;
        }

        /* Processors didn't stop, try again */
    }

    if (r != CSR_RESULT_SUCCESS)
    {
        /* An SDIO error occurred */
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to stop processors: SDIO error\n", card->instance));
    }
    else
    {
        /* If we reach here, we didn't the status in time. */
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to stop processors: timeout waiting for stopped status\n", card->instance));
        r = CSR_RESULT_FAILURE;
    }
#endif
    return r;
} /* unifi_card_stop_processor() */

/*
 * ---------------------------------------------------------------------------
 *  card_start_processor
 *
 *      Start the UniFi XAP processors.
 *
 *  Arguments:
 *      card            Pointer to card struct
 *      which           One of UNIFI_PROC_MAC, UNIFI_PROC_PHY, UNIFI_PROC_BOTH
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or CSR error code
 * ---------------------------------------------------------------------------
 */
CsrResult card_start_processor(card_t *card, enum unifi_dbg_processors_select which)
{
#ifndef CSR_WIFI_DRIVER_HYDRA
    CsrResult r;

    /* Select both XAPs */
    r = unifi_set_proc_select(card, which);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_set_proc_select failed: %d.\n",
                               card->instance,  r));
        return r;
    }


    r = unifi_write_direct_8_or_16(card,
                                   ChipHelper_DBG_EMU_CMD(card->helper) * 2, 8);
    if (r != CSR_RESULT_SUCCESS)
    {
        return r;
    }

    r = unifi_write_direct_8_or_16(card,
                                   ChipHelper_DBG_EMU_CMD(card->helper) * 2, 0);
    if (r != CSR_RESULT_SUCCESS)
    {
        return r;
    }
#endif /* !CSR_WIFI_DRIVER_HYDRA */
    return CSR_RESULT_SUCCESS;
} /* card_start_processor() */

/*
 * ---------------------------------------------------------------------------
 *  unifi_set_interrupt_mode
 *
 *      Configure the interrupt processing mode used by the HIP
 *
 *  Arguments:
 *      card            Pointer to card struct
 *      mode            Interrupt mode to apply
 *
 *  Returns:
 *      None
 * ---------------------------------------------------------------------------
 */
void unifi_set_interrupt_mode(card_t *card, CsrUint32 mode)
{
    if (mode == CSR_WIFI_INTMODE_RUN_BH_ONCE)
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Scheduled interrupt mode", card->instance));
    }
    card->intmode = mode;
} /* unifi_set_interrupt_mode() */

/*
 * ---------------------------------------------------------------------------
 *  unifi_start_processors
 *
 *      Start all UniFi XAP processors.
 *
 *  Arguments:
 *      card            Pointer to card struct
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success, CSR error code on error
 * ---------------------------------------------------------------------------
 */
CsrResult unifi_start_processors(card_t *card)
{
    return card_start_processor(card, UNIFI_PROC_BOTH);
} /* unifi_start_processors() */

/*
 * ---------------------------------------------------------------------------
 *  unifi_request_max_sdio_clock
 *
 *      Requests that the maximum SDIO clock rate is set at the next suitable
 *      opportunity (e.g. when the BH next runs, so as not to interfere with
 *      any current operation).
 *
 *  Arguments:
 *      card            Pointer to card struct
 *
 *  Returns:
 *      None
 * ---------------------------------------------------------------------------
 */
void unifi_request_max_sdio_clock(card_t *card)
{
    /* On Hydra chips the safe init clock should be unncessary */
#ifndef CSR_WIFI_DRIVER_HYDRA
    card->request_max_clock = 1;
#endif
} /* unifi_request_max_sdio_clock() */

/*
 * ---------------------------------------------------------------------------
 *  unifi_set_host_state
 *
 *      Set the host deep-sleep state.
 *
 *      If transitioning to TORPID, the SDIO driver will be notified
 *      that the SD bus will be unused (idle) and conversely, when
 *      transitioning from TORPID that the bus will be used (active).
 *
 *  Arguments:
 *      card            Pointer to card struct
 *      state           New deep-sleep state.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS            on success
 *      CSR_WIFI_HIP_RESULT_NO_DEVICE      if the card was ejected
 *      CSR_RESULT_FAILURE            if an SDIO error occurred
 *
 *  Notes:
 *      We need to reduce the SDIO clock speed before trying to wake up the
 *      chip. Actually, in the implementation below we reduce the clock speed
 *      not just before we try to wake up the chip, but when we put the chip to
 *      deep sleep. This means that if the f/w wakes up on its' own, we waste
 *      a reduce/increace cycle. However, trying to eliminate this overhead is
 *      proved difficult, as the current state machine in the HIP lib does at
 *      least a CMD52 to disable the interrupts before we configure the host
 *      state.
 * ---------------------------------------------------------------------------
 */
CsrResult unifi_set_host_state(card_t *card, enum unifi_host_state state)
{
    CsrResult r = CSR_RESULT_SUCCESS;

#ifdef CSR_LOG_ENABLE
    static const CsrCharString *const states[] =
    {
        "AWAKE", "DROWSY", "TORPID"
    };
#endif
#ifndef CSR_WIFI_DRIVER_HYDRA
    static const CsrUint8 state_csr_host_wakeup[] =
    {
        1, 3, 0
    };
    static const CsrUint8 state_io_abort[] =
    {
        0, 2, 3
    };
#endif

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,  "unifi%d: State %s to %s\n",
                        card->instance,
                        states[card->host_state], states[state]));

    if (card->host_state == UNIFI_HOST_STATE_TORPID)
    {
        CsrSdioFunctionActive(card->sdio_if);
    }

#ifndef CSR_WIFI_DRIVER_HYDRA
    /* Write the new state to UniFi. */
    if (card->chip_id > SDIO_CARD_ID_UNIFI_2)
    {
        r = sdio_write_f0(card, SDIO_CSR_HOST_WAKEUP,
                          (CsrUint8) ((card->function << 4) | state_csr_host_wakeup[state]));
    }
    else
    {
        r = sdio_write_f0(card, SDIO_IO_ABORT, state_io_abort[state]);
    }
#endif

    if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
    {
        return r;
    }
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to write UniFi deep sleep state\n", card->instance));
    }
    else
    {
#ifndef CSR_WIFI_DRIVER_HYDRA
        /*
         * If the chip was in state TORPID then we can now increase
         * the maximum bus clock speed.
         */
        if (card->host_state == UNIFI_HOST_STATE_TORPID)
        {
            CsrResult csrResult = CsrSdioMaxBusClockFrequencySet(card->sdio_if,
                                                                 UNIFI_SDIO_CLOCK_MAX_HZ);
            if (csrResult == CSR_RESULT_SUCCESS)
            {
                card->sdio_clock_speed = UNIFI_SDIO_CLOCK_MAX_HZ;
            }
            else
            {
                r = ConvertCsrSdioToCsrHipResult(card, csrResult);
                if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
                {
                    return r;
                }
                /* Non-fatal error */
                CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to increase the SDIO clock speed\n", card->instance));
            }
        }
#endif
        /*
         * Cache the current state in the card structure to avoid
         * unnecessary SDIO reads.
         */
        card->host_state = state;

        if (state == UNIFI_HOST_STATE_TORPID)
        {
#ifndef CSR_WIFI_DRIVER_HYDRA
            /*
             * If the chip is now in state TORPID then we must now decrease
             * the maximum bus clock speed.
             */
            CsrResult csrResult = CsrSdioMaxBusClockFrequencySet(card->sdio_if,
                                                                 UNIFI_SDIO_CLOCK_SAFE_HZ);
            if (csrResult == CSR_RESULT_SUCCESS)
            {
                card->sdio_clock_speed = UNIFI_SDIO_CLOCK_SAFE_HZ;
            }
            else
            {
                r = ConvertCsrSdioToCsrHipResult(card, csrResult);
                if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
                {
                    return r;
                }
                CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to decrease the SDIO clock speed\n", card->instance));
            }
#endif
            CsrSdioFunctionIdle(card->sdio_if);
        }
    }

    return r;
} /* unifi_set_host_state() */

/*
 * ---------------------------------------------------------------------------
 *  unifi_card_info
 *
 *      Update the card information data structure
 *
 *  Arguments:
 *      card            Pointer to card struct
 *      card_info       Pointer to info structure to update
 *
 *  Returns:
 *      None
 * ---------------------------------------------------------------------------
 */
void unifi_card_info(card_t *card, card_info_t *card_info)
{
    card_info->chip_id = card->chip_id;
    card_info->chip_version = card->chip_version;
    card_info->fw_build = card->build_id;
    card_info->fw_hip_version = card->config_data.version;
    card_info->sdio_block_size = card->sdio_io_block_size;
} /* unifi_card_info() */

/*
 * ---------------------------------------------------------------------------
 *  unifi_check_io_status
 *
 *      Check UniFi for spontaneous reset and pending interrupt.
 *
 *  Arguments:
 *      card            Pointer to card struct
 *      status          Pointer to location to write chip status:
 *                        0 if UniFi is running, and no interrupt pending
 *                        1 if UniFi has spontaneously reset
 *                        2 if there is a pending interrupt
 *  Returns:
 *      CSR_RESULT_SUCCESS if OK, or CSR error
 * ---------------------------------------------------------------------------
 */
CsrResult unifi_check_io_status(card_t *card, CsrInt32 *status)
{
    CsrUint8 io_en;
    CsrResult r;
    CsrBool pending;

    *status = 0;

    r = sdio_read_f0(card, SDIO_IO_ENABLE, &io_en);
    if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
    {
        return r;
    }
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to read SDIO_IO_ENABLE to check for spontaneous reset\n", card->instance));
        return r;
    }

    if ((io_en & (1 << card->function)) == 0)
    {
        *status = 1;
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: UniFi has spontaneously reset.\n", card->instance));
        return r;
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,
                        CSR_WIFI_HIP_LOG_DEF, "unifi%d: UniFi function %d is enabled.\n",
                        card->instance,  card->function));

    /* See if we missed an SDIO interrupt */
    r = CardPendingInt(card, &pending);
    if (pending)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: There is an unhandled pending interrupt.\n", card->instance));
        *status = 2;
        return r;
    }

    return r;
} /* unifi_check_io_status() */

CsrResult ConvertCsrSdioToCsrHipResult(card_t *card, CsrResult csrResult)
{
    CsrResult r = CSR_RESULT_FAILURE;

    switch (csrResult)
    {
        case CSR_RESULT_SUCCESS:
            r = CSR_RESULT_SUCCESS;
            break;
        /* Timeout errors */
        case CSR_SDIO_RESULT_TIMEOUT:
        /* Integrity errors */
        case CSR_SDIO_RESULT_CRC_ERROR:
            r = CSR_RESULT_FAILURE;
            break;
        case CSR_SDIO_RESULT_NO_DEVICE:
            r = CSR_WIFI_HIP_RESULT_NO_DEVICE;
            break;
        case CSR_SDIO_RESULT_INVALID_VALUE:
            r = CSR_WIFI_HIP_RESULT_INVALID_VALUE;
            break;
        case CSR_RESULT_FAILURE:
            r = CSR_RESULT_FAILURE;
            break;
        default:
            CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID,
                                  CSR_WIFI_HIP_LOG_DEF, "unifi%d: Unrecognised csrResult error code: %d\n",
                                  card->instance,  csrResult));
            break;
    }

    return r;
} /* ConvertCsrSdioToCsrHipResult() */
