/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#ifndef CSR_WIFI_HIP_LIST_H__
#define CSR_WIFI_HIP_LIST_H__

#include "csr_synergy.h"
#include "csr_types.h"

#ifdef __cplusplus
extern "C" {
#endif


struct listHeader
{
    struct listHeader *next, *prev;
};


/*
 * ---------------------------------------------------------------------------
 *  CSR_WIFI_HIP_LIST_INIT
 *
 *      Initialise a list header by pointing the list header to itself.
 *
 *  Arguments:
 *      list            Head of the list to initialise.
 *
 *  Returns:
 *      None
 *
 * ---------------------------------------------------------------------------
 */
#define CSR_WIFI_HIP_LIST_INIT(list)   (list)->next = (list);(list)->prev = (list);

/*
 * ---------------------------------------------------------------------------
 *  CSR_WIFI_HIP_LIST_LIST_ENTRY
 *
 *      Returns the start of a .
 *
 *  Arguments:
 *      ptr             Pointer to
 *      type            Type of the structure to be .
 *      member
 *
 *  Returns:
 *      None
 *
 * ---------------------------------------------------------------------------
 */
#define CSR_WIFI_HIP_LIST_LIST_ENTRY(ptr, type, member)  ((type *) ((CsrUint8 *) (ptr) - (CsrSize) (&((type *) 0)->member)))

/*
 * ---------------------------------------------------------------------------
 *  CSR_WIFI_HIP_LIST_ADD_ENTRY
 *
 *      Add an entry to the list provided in the macro.
 *
 *  Arguments:
 *      entry           The entry to add to the list.
 *      list            Head of the list to add the entry.
 *
 *  Returns:
 *      None
 *
 * ---------------------------------------------------------------------------
 */
#define CSR_WIFI_HIP_LIST_ADD_ENTRY(entry, list)  ((list)->next)->prev = entry;   \
    (entry)->next = (list)->next;   \
    (entry)->prev = list;           \
    (list)->next = entry;

/*
 * ---------------------------------------------------------------------------
 *  CSR_WIFI_HIP_LIST_ADD_TAIL_ENTRY
 *
 *      Add an entry to the tail of the list provided in the macro.
 *
 *  Arguments:
 *      entry           The entry to add to the tail of the list.
 *      list            Head of the list to add the entry.
 *
 *  Returns:
 *      None
 *
 * ---------------------------------------------------------------------------
 */
#define CSR_WIFI_HIP_LIST_ADD_TAIL_ENTRY(entry, list) (entry)->next = list;     \
    (entry)->prev = (list->prev);   \
    ((list)->prev)->next = entry;     \
    (list)->prev = entry;    \

/*
 * ---------------------------------------------------------------------------
 *  CSR_WIFI_HIP_LIST_DELETE_ENTRY
 *
 *      Delete the entry from a list.
 *
 *  Arguments:
 *      entry           The entry to be deleted.
 *
 *  Returns:
 *      None
 *
 * ---------------------------------------------------------------------------
 */
#define CSR_WIFI_HIP_LIST_DELETE_ENTRY(entry)  ((entry)->prev)->next = (entry)->next;      \
    ((entry)->next)->prev = (entry)->prev;      \
    (entry)->prev = NULL;(entry)->next = NULL;

/*
 * ---------------------------------------------------------------------------
 *  CSR_WIFI_HIP_LIST_IS_LIST_EMPTY
 *
 *      Tests whether a list is empty.
 *
 *  Arguments:
 *      list            Head of the list to test.
 *
 *  Returns:
 *      TRUE            If the list is empty
 *      FALSE           If the list is not empty
 *
 * ---------------------------------------------------------------------------
 */
#define CSR_WIFI_HIP_LIST_IS_LIST_EMPTY(list)  ((list)->next == list)

/*
 * ---------------------------------------------------------------------------
 *  CSR_WIFI_HIP_LIST_LIST_FOR_EACH_SAFE
 *
 *      Iterates through the list provided, safe against removal of list entry.
 *
 *  Arguments:
 *      pos             &struct listHeader to use as the loop variable.
 *      n               &struct listHeader to use as tempory storage
 *      list            The head of the list to iterate.
 *
 *  Returns:
 *      pos             Points at the current listHeader entry.
 *
 * ---------------------------------------------------------------------------
 */
#define CSR_WIFI_HIP_LIST_LIST_FOR_EACH_SAFE(pos, n, list)  for (pos = (list)->next, n = pos->next; pos != (list); \
                                                                 pos = n, n = pos->next)


/*
 * ---------------------------------------------------------------------------
 *  CSR_WIFI_HIP_LIST_FOR_EACH_PREV_SAFE
 *
 *      Iterate backwards through the list provided, safe against removal of list entry.
 *
 *  Arguments:
 *      pos             &struct listHeader to use as the loop variable.
 *      n               &struct listHeader to use as tempory storage
 *      list            The head of the list to iterate.
 *
 *  Returns:
 *      pos             Points at the current listHeader entry.
 *
 * ---------------------------------------------------------------------------
 */
#define CSR_WIFI_HIP_LIST_FOR_EACH_PREV_SAFE(pos, n, list)  for (pos = (list)->prev, n = pos->prev; pos != (list); \
                                                                 pos = n, n = pos->prev)


#ifdef __cplusplus
}
#endif

#endif /* __CSR_WIFI_HIP_LIST_H__ */
