/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"

#include "csr_wifi_hip_log_text.h"
#include "csr_wifi_hip_util.h"
#include "csr_wifi_hip_unifi.h"
#include "csr_wifi_hip_hal_priv.h"
#include "csr_wifi_hip.h"
#include "csr_wifi_router_ctrl_prim.h"


/* Chip power handling */
CsrBool csrWifiHipPowerChipOff(CsrWifiHipHalPriv *priv)
{
#ifndef CSR_WIFI_DRIVER_HYDRA
    return (priv->fw_init > 0) ? FALSE : TRUE;
#else
    /* Always keep the chip powered in Hydra */
    return FALSE;
#endif
}

/* Virtual interface utility functions */
void interfaceTagMapDbDump(CsrWifiHipHalPriv *priv)
{
    CsrUint16 i;

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "InterfaceTagMap\n"));
    for (i = 0; i < CSR_WIFI_HIP_MAX_VIFINDEX; i++)
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                            "entry (vifIndex) %u ==> interfaceTag 0x%02X\n", i, priv->interfaceTagMap[i]));
    }
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "primaryVifIndexMap\n"));
    for (i = 0; i < CSR_WIFI_MAX_INTERFACES; i++)
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                            "entry (interfaceTag) %u ==> vifIndex 0x%02X\n", i, priv->primaryVifIndexMap[i]));
    }
}

static CsrUint8 csrWifiHipVifIndexExists(CsrWifiHipHalPriv *priv, CsrUint16 vifIndex)
{
    return (priv->interfaceTagMap[vifIndex] == 0xFFFF) ? FALSE : TRUE;
}

CsrResult csrWifiHipUpdateInterfaceTagAndVifIndexMap(CsrWifiHipHalPriv *priv, CsrUint16 interfaceTag,
                                                     CsrUint16 vifIndex, CsrBool isPrimaryVifIndex,
                                                     CsrUint8 action, CsrUint16 newVifIndex)
{
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6,
                        "unifi%d: csrWifiHipUpdateInterfaceTagAndVifIndexMap: >>>\n",
                        priv ? priv->instance : 0));

    /* Validate vifIndex, which can't be zero for ADD/DEL actions */
    if ((action != CSR_WIFI_HIP_VIFINDEX_FLUSH_IFACE_DB) && (!vifIndex || (vifIndex > (CSR_WIFI_HIP_MAX_VIFINDEX - 1))))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "csrWifiHipUpdateInterfaceTagAndVifIndexMap: vifIndex can't be 0 \n"));
        return CSR_RESULT_FAILURE;
    }

    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                       "csrWifiHipUpdateInterfaceTagAndVifIndexMap: ifTag %d vifIndex %u isprimaryVifIndex %u action %u newVifIndex %u \n",
                       interfaceTag, vifIndex, isPrimaryVifIndex, action, newVifIndex));
    switch (action)
    {
        case CSR_WIFI_HIP_VIFINDEX_ADD:
            /* Check new VifIndex SME configuring already in data base? */
            if (csrWifiHipVifIndexExists(priv, vifIndex))
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                       "csrWifiHipUpdateInterfaceTagAndVifIndexMap: (DUP)vifIndex %u already exists\n", vifIndex));
                interfaceTagMapDbDump(priv);
                return CSR_RESULT_FAILURE;
            }

            if (isPrimaryVifIndex)
            {
                CSR_WIFI_HIP_SPINLOCK_LOCK(&priv->vifMapLock);
                /* interfaceTag to primary vifIndexMap */
                priv->primaryVifIndexMap[interfaceTag] = vifIndex;
                CSR_WIFI_HIP_SPINLOCK_UNLOCK(&priv->vifMapLock);
            }

            /* primary vifIndex to interfaceTagMap */
            if (priv->interfaceTagMap[vifIndex] == 0xFFFF)
            {
                CSR_WIFI_HIP_SPINLOCK_LOCK(&priv->vifMapLock);
                priv->interfaceTagMap[vifIndex] = interfaceTag;
                CSR_WIFI_HIP_SPINLOCK_UNLOCK(&priv->vifMapLock);
            }
            else
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                       "csrWifiHipUpdateInterfaceTagAndVifIndexMap: (%x)Slot is not free\n", vifIndex));
                return CSR_RESULT_FAILURE;
            }
            break;
        case CSR_WIFI_HIP_VIFINDEX_DEL:
            /* Validate vifIndex, which can't be zero */
            if (!newVifIndex || (newVifIndex > (CSR_WIFI_HIP_MAX_VIFINDEX - 1)))
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                       "csrWifiHipUpdateInterfaceTagAndVifIndexMap: newVifIndex can't be 0 in DEL action\n"));
                return CSR_RESULT_FAILURE;
            }

            if (newVifIndex == vifIndex)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                       "csrWifiHipUpdateInterfaceTagAndVifIndexMap: vifIndex & newVifIndex are same\n"));
                break;
            }

            if ((vifIndex ^ priv->primaryVifIndexMap[interfaceTag]))
            {
                /* Not matching with primary interface, Trying to remove the secondary VifIndex */
                isPrimaryVifIndex = FALSE;
            }

            /* search for vifIndex to be deleted, if not found then panic */
            if (!(csrWifiHipVifIndexExists(priv, vifIndex)))
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                       "csrWifiHipUpdateInterfaceTagAndVifIndexMap: vifIndex %u not found for deletion\n", vifIndex));
                interfaceTagMapDbDump(priv);
                return CSR_RESULT_FAILURE;
            }
            else
            {
                CSR_WIFI_HIP_SPINLOCK_LOCK(&priv->vifMapLock);
                /* remove vifIndex mapping from DB (RX mapping removed) */
                priv->interfaceTagMap[vifIndex] = 0xFFFF;
                CSR_WIFI_HIP_SPINLOCK_UNLOCK(&priv->vifMapLock);
            }

            /* search for newVifIndex to be set as primary, If it not exist in database then panic */
            if (!(csrWifiHipVifIndexExists(priv, newVifIndex)))
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                       "csrWifiHipUpdateInterfaceTagAndVifIndexMap: newNifIndex %u not found for setting primary flag\n", vifIndex));
                interfaceTagMapDbDump(priv);
                return CSR_RESULT_FAILURE;
            }

            if (isPrimaryVifIndex)
            {
                CSR_WIFI_HIP_SPINLOCK_LOCK(&priv->vifMapLock);
                /* Removed interface is primary, so set newvifIndex as primary */
                priv->primaryVifIndexMap[interfaceTag] = newVifIndex;
                CSR_WIFI_HIP_SPINLOCK_UNLOCK(&priv->vifMapLock);
            }
            else
            {
                if (!(vifIndex ^ priv->primaryVifIndexMap[interfaceTag]))
                {
                    /* primary vifIndex removed, BUT newvifIndex not made as primary
                     * corrupts the mapping of RX path */
                    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                           "csrWifiHipUpdateInterfaceTagAndVifIndexMap: primary vifIndex Removed But\
                                           alternate vifIndex not set as primary: Debug\n"));
                    return CSR_RESULT_FAILURE;
                }
            }
            break;
        case CSR_WIFI_HIP_VIFINDEX_FLUSH_IFACE_DB:
        {
            CsrUint8 i;

            /* Flush all the interfaceTag/VifIndex mapping, as interface will
             * going to die
             */
            CsrMemSet(&priv->primaryVifIndexMap[interfaceTag], 0, sizeof(CsrUint16));

            /* loop indexing starts with 1 as vifIndex starts from 1 */
            for (i = 1; i < CSR_WIFI_HIP_MAX_VIFINDEX; i++)
            {
                if (priv->interfaceTagMap[i] == interfaceTag)
                {
                    CSR_WIFI_HIP_SPINLOCK_LOCK(&priv->vifMapLock);
                    priv->interfaceTagMap[i] = 0xFFFF;
                    CSR_WIFI_HIP_SPINLOCK_UNLOCK(&priv->vifMapLock);
                }
            }
            break;
        }

        default:
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "csrWifiHipUpdateInterfaceTagAndVifIndexMap: action not supported\n"));
    }
    return CSR_RESULT_SUCCESS;
}

CsrUint16 CsrWifiHipVifIndexGetReq(void *hipHandle, CsrUint16 interfaceTag)
{
    CsrWifiHipHalPriv *priv = hipHandle;

    /* returns always the primary vifIndex */
    return priv->primaryVifIndexMap[interfaceTag];
}

CsrUint16 CsrWifiHipInterfaceTagGetReq(void *hipHandle, CsrUint16 vifIndex)
{
    CsrUint16 interfaceTag = 0;
    CsrWifiHipHalPriv *priv = hipHandle;


    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3,
                        "CsrWifiHipInterfaceTagGetReq: vifIndex=%d\n", vifIndex));

    if (!vifIndex)
    {
        /* Beacons, probe Res before ModeSetReq received/processed on
         * interfaceTag 0
         */
        return 0;
    }

    /* Validate vifIndex, which can't be zero */
    if ((vifIndex > (CSR_WIFI_HIP_MAX_VIFINDEX - 1)))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "CsrWifiHipInterfaceTagGetReq: vifIndex out of Range(should < %d)\n",
                               (CSR_WIFI_HIP_MAX_VIFINDEX - 1)));
        return 0xFFFF;
    }

    CSR_WIFI_HIP_SPINLOCK_LOCK(&priv->vifMapLock);
    /* contains interfaceTag map for only primary vifIndex */
    interfaceTag = priv->interfaceTagMap[vifIndex];
    CSR_WIFI_HIP_SPINLOCK_UNLOCK(&priv->vifMapLock);

    return interfaceTag;
}

CsrResult CsrWifiHipStatGetReq(void *hipHandle, CsrUint16 interfaceTag, CsrWifiHipReceiveStat *stats)
{
    if (hipHandle != NULL)
    {
        CsrWifiHipHalPriv *priv = hipHandle;

        if (interfaceTag < CSR_WIFI_MAX_INTERFACES)
        {
            CsrWifiHipVifInstance *vif = (CsrWifiHipVifInstance *) &priv->vif[interfaceTag];

            CsrMemCpy(stats, &vif->receiveStats, sizeof(CsrWifiHipReceiveStat));
            return CSR_RESULT_SUCCESS;
        }
        else
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipStatGetReq: Invalid interfaceTag", priv->instance));
            return CSR_RESULT_FAILURE;
        }
    }
    else
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi : CsrWifiHipStatGetReq: Invalid hipHandle"));
        return CSR_RESULT_FAILURE;
    }
}

void CsrWifiHipStatResetReq(void *hipHandle, CsrUint16 interfaceTag)
{
    if (hipHandle != NULL)
    {
        CsrWifiHipHalPriv *priv = hipHandle;

        if (interfaceTag < CSR_WIFI_MAX_INTERFACES)
        {
            CsrWifiHipVifInstance *vif = (CsrWifiHipVifInstance *) &priv->vif[interfaceTag];
            CsrMemSet(&vif->receiveStats, 0, sizeof(CsrWifiHipReceiveStat));
        }
        else
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipStatResetReq: Invalid interfaceTag", priv->instance));
        }
    }
    else
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi : CsrWifiHipStatResetReq: Invalid hipHandle"));
    }
}

CsrBool csrWifiHipGetProtectionBitFromInterfaceMode(CsrWifiHipHalPriv *priv, CsrWifiHipVifInstance *vif, const CsrUint8 *daddr, CsrWifiHipStaPeerInfo *destPeerInfo)
{
    CsrBool protection = 0;

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  "unifi%d: csrWifiHipGetProtectionBitFromInterfaceMode: >>> \n"
                           , priv ? priv->instance : 0));
    switch (vif->interfaceMode)
    {
        case CSR_WIFI_ROUTER_CTRL_MODE_STA:
        case CSR_WIFI_ROUTER_CTRL_MODE_P2PCLI:
        case CSR_WIFI_ROUTER_CTRL_MODE_AMP:
        case CSR_WIFI_ROUTER_CTRL_MODE_IBSS:
            protection = vif->staInfo.protect;
            break;
        case CSR_WIFI_ROUTER_CTRL_MODE_AP:
        case CSR_WIFI_ROUTER_CTRL_MODE_P2PGO:
        {
            if (daddr[0] & 0x1)
            {
                CSR_LOG_TEXT_DEBUG((
                                       CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG3,  "unifi%d: csrWifiHipGetProtectionBitFromInterfaceMode: broadcast/multicast packet\n"
                                       , priv ? priv->instance : 0));
                /* In this mode, the protect member of priv structure has an information of how
                 * AP/P2PGO has started, & the member updated in set mode request for AP/P2PGO
                 */
                protection = vif->staInfo.protect;
            }
            else
            {
                /* fetch the destination record from staion record database */
                CSR_LOG_TEXT_DEBUG((
                                       CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG3,  "unifi%d: csrWifiHipGetProtectionBitFromInterfaceMode: unicast packet\n"
                                       , priv ? priv->instance : 0));
                if (!destPeerInfo)
                {
                    CSR_LOG_TEXT_DEBUG((
                                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG3,  "unifi%d: csrWifiHipGetProtectionBitFromInterfaceMode: peer not found in station records\n"
                                           , priv ? priv->instance : 0));
                    return FALSE;
                }
                protection = destPeerInfo->protection;
            }
            break;
        }
        default:
            CSR_LOG_TEXT_INFO((
                                  CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG2,  "unifi%d: csrWifiHipGetProtectionBitFromInterfaceMode: mode unknown\n"
                                  , priv ? priv->instance : 0));
    }
    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  "unifi%d: csrWifiHipGetProtectionBitFromInterfaceMode: %u <<< \n",
                           priv ? priv->instance : 0,  protection));
    return protection;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPrepareAndAddMacheader
 *
 *
 *      This function prepares and add a MAC header to the bulkdata buffer
 *      provided as an argument.
 *
 *  Arguments:
 *      priv            Pointer to device private context struct
 *      vif             Pointer to virtual interface instance data
 *      priority        Priority to be added in MAC header
 *      protocol        The protocol/EtherType for the higher layer payload
 *                      that is about to be added MAC header for
 *      daddr           Destination MAC address
 *      saddr           Source MAC address
 *      macHeaderMem    The number of bytes allocated for the MAC header
 *                      before the network stack payload in the bulkdata buf.
 *      hostTag         The hostTag is among other things used to identify M4
 *                      frames, which needs special handling.
 *      bulkdata        Pointer to the network payload with macHeaderMem pre-
 *                      allocated.
 *      dstStaInfo      Pointer to the station record
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success or CSR_RESULT_FAILURE on error.
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPrepareAndAddMacheader(CsrWifiHipHalPriv       *priv,
                                           CsrWifiHipVifInstance   *vif,
                                           CSR_PRIORITY             priority,
                                           CsrUint16                protocol,
                                           const CsrUint8          *daddr,
                                           const CsrUint8          *saddr,
                                           CsrUint8                 macHeaderMem,
                                           CsrUint32                hostTag,
                                           CsrWifiHipBulkDataParam *bulkdata,
                                           CsrWifiHipStaPeerInfo   *dstStaInfo)
{
    CsrUint16 frameCtrl = 0;
    CsrUint16 qosCtrl = 0;
    CsrUint8 macHeaderLengthInBytes = MAC_HEADER_SIZE, *bufPtr = NULL;
    CsrUint8 direction = 0;
    CsrUint8 *addressOne = NULL;
    CsrBool protection = FALSE;
    CsrWifiHipQosType qosType;

    qosType = CSR_WIFI_HIP_QOS_TYPE(priority);

    /* Add a MAC header refer: 7.1.3.1 Frame Control field in P802.11REVmb.book */
    frameCtrl |= CSR_WIFI_HIP_IEEE80211_FCTL_TYPE_DATA;

    /* Add QoS control if the frame is QoS type */
    if (CSR_WIFI_HIP_QOS_TYPE_QOS_CONTROL == qosType)
    {
        /* Qos Control Field */
        macHeaderLengthInBytes += QOS_CONTROL_HEADER_SIZE;

        /* If payload is 0, then set as Null frame */
        if ((bulkdata->d[0].data_length) - macHeaderLengthInBytes == 0)
        {
            frameCtrl |= CSR_WIFI_HIP_IEEE80211_FCTL_SUB_TYPE_QOS_NULL;
        }
        else
        {
            frameCtrl |= CSR_WIFI_HIP_IEEE80211_FCTL_SUB_TYPE_QOS_DATA;
        }
    }
    else
    {
        if ((bulkdata->d[0].data_length) - macHeaderLengthInBytes == 0)
        {
            frameCtrl |= CSR_WIFI_HIP_IEEE80211_FCTL_SUB_TYPE_NULL;
        }
        else
        {
            frameCtrl |= CSR_WIFI_HIP_IEEE80211_FCTL_SUB_TYPE_DATA;
        }
    }


    /* The frame is sent to Access point */
    switch (vif->interfaceMode)
    {
        case CSR_WIFI_ROUTER_CTRL_MODE_STA:
        case CSR_WIFI_ROUTER_CTRL_MODE_P2PCLI:
            direction = 2;
            frameCtrl |= CSR_WIFI_HIP_IEEE80211_FCTL_TO_DS;
            frameCtrl &= ~(CSR_WIFI_HIP_IEEE80211_FCTL_FROM_DS);
            break;
        case CSR_WIFI_ROUTER_CTRL_MODE_IBSS:
            /* IBSS this bit will be zero */
            direction = 0;
            frameCtrl &= ~(CSR_WIFI_HIP_IEEE80211_FCTL_TO_DS);
            frameCtrl &= ~(CSR_WIFI_HIP_IEEE80211_FCTL_FROM_DS);
            break;
        case CSR_WIFI_ROUTER_CTRL_MODE_AP:
        case CSR_WIFI_ROUTER_CTRL_MODE_P2PGO:
            direction = 1;
            frameCtrl &= ~(CSR_WIFI_HIP_IEEE80211_FCTL_TO_DS);
            frameCtrl |= CSR_WIFI_HIP_IEEE80211_FCTL_FROM_DS;
            break;
        case CSR_WIFI_ROUTER_CTRL_MODE_AMP:
            if (priority == CSR_MANAGEMENT)
            {
                direction = 2;
                frameCtrl |= CSR_WIFI_HIP_IEEE80211_FCTL_TO_DS;
                frameCtrl &= ~(CSR_WIFI_HIP_IEEE80211_FCTL_FROM_DS);
            }
            else
            {
                /* Data frames have to use WDS 4 address frames */
                direction = 3;
                frameCtrl |= CSR_WIFI_HIP_IEEE80211_FCTL_TO_DS;
                frameCtrl |= CSR_WIFI_HIP_IEEE80211_FCTL_FROM_DS;
            }
            break;

        default:
            CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID,
                                  CSR_WIFI_HIP_LOG_DEF, "unifi%d: csrWifiHipPrepareAndAddMacheader: Unknown mode %d\n",
                                  priv ? priv->instance : 0,  vif->interfaceMode));
            frameCtrl &= ~(CSR_WIFI_HIP_IEEE80211_FCTL_TO_DS);
            frameCtrl &= ~(CSR_WIFI_HIP_IEEE80211_FCTL_FROM_DS);
    }

    /* If interfaceMode is QOS and HTC is supported, then need to set this bit.
     * We do not support HTC now. So this bit is reset
     */
    frameCtrl &= ~(CSR_WIFI_HIP_IEEE80211_FCTL_ORDER);

    /* Do not set protection bit for null frames or WAPI frames. For other frames
     * find the protection required and set accordingly
     */
    if (((bulkdata->d[0].data_length) - macHeaderLengthInBytes) && (protocol != ETH_P_WAI))
    {
        protection = csrWifiHipGetProtectionBitFromInterfaceMode(priv, vif, daddr, dstStaInfo);
    }

    if (protection && ((hostTag & CSR_WIFI_HIP_M4_MESSAGE) != CSR_WIFI_HIP_M4_MESSAGE))
    {
        frameCtrl |= CSR_WIFI_HIP_IEEE80211_FCTL_PROTECTED_FRAME;
    }
    else
    {
        frameCtrl &= ~(CSR_WIFI_HIP_IEEE80211_FCTL_PROTECTED_FRAME);
    }

    if ((CSR_WIFI_HIP_IEEE80211_HAS_TO_DS(frameCtrl)) & (CSR_WIFI_HIP_IEEE80211_HAS_FROM_DS(frameCtrl)))
    {
        macHeaderLengthInBytes += 6;
    }

    /* Check to ensure that the preallocated memory for
       the MAC header is sufficient */
    if (macHeaderLengthInBytes > macHeaderMem)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: csrWifiHipPrepareAndAddMacheader: Insufficient memory has been allocated for the MAC header: \
                    %d bytes was allocated %d bytes were needed \n",
                               priv->instance, macHeaderMem, macHeaderLengthInBytes));
        return CSR_RESULT_FAILURE;
    }

    bufPtr = (CsrUint8 *) bulkdata->d[0].os_data_ptr;

    /* Frame the actual MAC header */
    CsrMemSet(bufPtr, 0, macHeaderLengthInBytes);

    /* Copy frameControl field */
    CSR_COPY_UINT16_TO_LITTLE_ENDIAN(frameCtrl, bufPtr);

    bufPtr += FRAME_CONTROL_HEADER_SIZE;
    macHeaderLengthInBytes -= FRAME_CONTROL_HEADER_SIZE;


    /* Duration/ID field which is 2 bytes */
    bufPtr += 2;
    macHeaderLengthInBytes -= 2;

    /*
     * There is no default for this switch statement because
     * direction is set in this function and cannot be any other
     * value than those in the case statements.
     */
    switch (direction)
    {
        case 0:
            /* Its an Ad-Hoc no need to route it through AP */
            /* Address1: MAC address of the destination from eth header */
            CsrMemCpy(bufPtr, daddr, ETH_ALEN);
            bufPtr += ETH_ALEN;
            macHeaderLengthInBytes -= ETH_ALEN;

            /* Address2: MAC address of the source */
            CsrMemCpy(bufPtr, saddr, ETH_ALEN);
            bufPtr += ETH_ALEN;
            macHeaderLengthInBytes -= ETH_ALEN;

            /* Address3: the BSSID (locally generated in AdHoc (creators Bssid)) */
            CsrMemCpy(bufPtr, &vif->staInfo.bssid, ETH_ALEN);
            bufPtr += ETH_ALEN;
            macHeaderLengthInBytes -= ETH_ALEN;
            break;
        case 1:
            /* Address1: MAC address of the actual destination */
            CsrMemCpy(bufPtr, daddr, ETH_ALEN);
            bufPtr += ETH_ALEN;
            macHeaderLengthInBytes -= ETH_ALEN;
            /* Address2: The MAC address of the AP */
            CsrMemCpy(bufPtr, &vif->staInfo.bssid, ETH_ALEN);
            bufPtr += ETH_ALEN;
            macHeaderLengthInBytes -= ETH_ALEN;

            /* Address3: MAC address of the source from eth header */
            CsrMemCpy(bufPtr, saddr, ETH_ALEN);
            bufPtr += ETH_ALEN;
            macHeaderLengthInBytes -= ETH_ALEN;
            break;
        case  2:
            /* Address1: To AP is the MAC address of the AP to which its associated */
            CsrMemCpy(bufPtr, &vif->staInfo.bssid, ETH_ALEN);
            bufPtr += ETH_ALEN;
            macHeaderLengthInBytes -= ETH_ALEN;

            /* Address2: MAC address of the source from eth header */
            CsrMemCpy(bufPtr, saddr, ETH_ALEN);
            bufPtr += ETH_ALEN;
            macHeaderLengthInBytes -= ETH_ALEN;

            /* Address3: MAC address of the actual destination on the distribution system */
            CsrMemCpy(bufPtr, daddr, ETH_ALEN);
            bufPtr += ETH_ALEN;
            macHeaderLengthInBytes -= ETH_ALEN;
            break;
        case 3:
            CsrMemCpy(bufPtr, &vif->staInfo.bssid, ETH_ALEN);
            bufPtr += ETH_ALEN;
            macHeaderLengthInBytes -= ETH_ALEN;

            /* Address2: MAC address of the source from eth header */
            CsrMemCpy(bufPtr, saddr, ETH_ALEN);
            bufPtr += ETH_ALEN;
            macHeaderLengthInBytes -= ETH_ALEN;

            /* Address3: MAC address of the actual destination on the distribution system */
            CsrMemCpy(bufPtr, daddr, ETH_ALEN);
            bufPtr += ETH_ALEN;
            macHeaderLengthInBytes -= ETH_ALEN;

            break;
    }

    /* 2 bytes of sequence control field, appended by firmware */
    bufPtr += 2;
    macHeaderLengthInBytes -= 2;

    if (3 == direction)
    {
        /* Address4: MAC address of the source */
        CsrMemCpy(bufPtr, saddr, ETH_ALEN);
        bufPtr += ETH_ALEN;
        macHeaderLengthInBytes -= ETH_ALEN;
    }

    /* If Qos Data or Qos Null Data then add QosControl field */
    if (CSR_WIFI_HIP_QOS_TYPE_QOS_CONTROL == qosType)
    {
        if (priority > CSR_QOS_UP7)
        {
            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi%d: csrWifiHipPrepareAndAddMacheader: data packets priority is more than 7, priority = %x\n",
                               priv ? priv->instance : 0, priority));
            qosCtrl |= CSR_QOS_UP7;
        }
        else
        {
            qosCtrl |= (CsrUint16) priority;
        }

        /* Point at address 1: Add duration and frame control header */
        addressOne = (CsrUint8 *) (bulkdata->d[0].os_data_ptr + FRAME_CONTROL_HEADER_SIZE + 2);

        if (addressOne[0] & 0x1)
        {
            /* multicast/broadcast frames, no acknowledgement needed */
            qosCtrl |= CSR_WIFI_HIP_IEEE80211_QOS_CONTROL_ACK_POLICY_NO_ACK;
        }
        else
        {
            /* normal acknowledgement */
            qosCtrl |= CSR_WIFI_HIP_IEEE80211_QOS_CONTROL_ACK_POLICY_NORMAL_ACK;
        }

        /* non-AP mode only for now */
        if ((vif->interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_STA) ||
            (vif->interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_IBSS) ||
            (vif->interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_P2PCLI))
        {
            /* In case of STA and IBSS case eosp and txop limit is 0. */
            qosCtrl &= ~(CSR_WIFI_HIP_IEEE80211_QOS_CONTROL_EOSP);
        }
        else
        {
            if ((frameCtrl & CSR_WIFI_HIP_IEEE80211_FCTL_SUB_TYPE_MASK) == CSR_WIFI_HIP_IEEE80211_FCTL_SUB_TYPE_QOS_NULL)
            {
                qosCtrl |= CSR_WIFI_HIP_IEEE80211_QOS_CONTROL_EOSP;
            }
            else
            {
                qosCtrl &= ~(CSR_WIFI_HIP_IEEE80211_QOS_CONTROL_EOSP);
            }
        }

        /* append Qos control field to mac header */
        CSR_COPY_UINT16_TO_LITTLE_ENDIAN(qosCtrl, bufPtr);
        macHeaderLengthInBytes -= QOS_CONTROL_HEADER_SIZE;
    }

    if (macHeaderLengthInBytes)
    {
        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG3,  "unifi%d: csrWifiHipPrepareAndAddMacheader: MAC header not appended properly\n"
                               , priv ? priv->instance : 0));
        return CSR_RESULT_FAILURE;
    }

    return CSR_RESULT_SUCCESS;
}

CsrResult CsrWifiHipHip2StatsGet(void *hipHandle, CsrWifiHipHip2Stats *stats)
{
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;

    return unifi_hip2_stats_get(priv->card, stats);
}

CsrWifiHipMaPacketPriority csrWifiHipPacketPriorityGet(CsrWifiHipHalPriv *priv, CsrWifiHipVifInstance *vif, CsrUint8 *frame, CsrUint16 proto)
{
    CsrWifiHipMaPacketPriority priority = CSR_WIFI_HIP_MA_PACKET_PRIORITY_CONTENTION;

    func_enter();

    if (frame != NULL)
    {
        switch (proto)
        {
            case 0x0800:        /* IPv4 */
            case 0x814C:        /* SNMP */
            case 0x880C:        /* GSMP */
                priority = (CsrWifiHipMaPacketPriority) (((frame[IP4_OFFSET_TO_TOS_FIELD]) & 0xE0) >> 5);
                break;

            case 0x8100:        /* VLAN */
                priority = (CsrWifiHipMaPacketPriority) ((*frame & 0xE0) >> 5);
                break;

            case 0x86DD:        /* IPv6 */
                priority = (CsrWifiHipMaPacketPriority) ((*frame & 0x0E) >> 1);
                break;

            case ETH_P_PAE:     /* EAPOL and WAPI */
            case ETH_P_WAI:
            /* FALL THROUGH: Send EAPOL and WAPI frames with BE priority */
            default:
                priority = CSR_WIFI_HIP_MA_PACKET_PRIORITY_QOS_UP0;
                break;
        }

        /* Check if we are allowed to transmit on this AC. Because of ACM we may have to downgrade to a lower
         * priority */
        if ((vif->interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_STA) ||
            (vif->interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_P2PCLI))
        {
            unifi_TrafficQueue queue;

            /* Keep trying lower priorities until we find a queue
             * Priority to queue mapping is 1,2 - BK, 0,3 - BE, 4,5 - VI, 6,7 - VO */
            queue = unifi_frame_priority_to_queue((CSR_PRIORITY) priority);

            while ((queue > UNIFI_TRAFFIC_Q_BK) && !vif->staInfo.queueEnabled[queue])
            {
                queue--;
                priority = (CsrWifiHipMaPacketPriority) unifi_get_default_downgrade_priority(queue);
            }
        }
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG5,
                        "unifi%d: packetPriorityGet: Packet priority = %d\n",
                        priv ? priv->instance : 0,  priority));

    func_exit();
    return priority;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipApStaRecordInitialise
 *
 *
 *      This function initialises an access point peer station record.
 *      The caller has to ensure mutual exclusive access to the staRecord.
 *
 *  Arguments:
 *      priv            Pointer to the HAL private instance data
 *      staRecord       The station record to be initialised
 *
 *  Returns:
 *      No return value - the call can not fail.
 * ---------------------------------------------------------------------------
 */
void csrWifiHipApStaRecordInitialise(CsrWifiHipHalPriv *priv, CsrWifiHipStaPeerInfo *staRecord)
{
    CsrMemSet(staRecord, 0, sizeof(CsrWifiHipStaPeerInfo));

    staRecord->entryInUse = FALSE;
    staRecord->wmmOrQosEnabled = FALSE;
    staRecord->protection = FALSE;
#ifdef CSR_WIFI_AP_ENABLE
    staRecord->active = FALSE;
#endif
}
