/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"

#include "csr_wifi_hip_log_text.h"
#include "csr_types.h"
#include "csr_result.h"
#include "csr_wifi_hip_hal_priv.h"

#include "csr_wifi_hip.h"
#include "csr_wifi_router_ctrl_lib.h"
#include "csr_wifi_hip_util.h"
#ifdef CSR_WIFI_AP_ENABLE
#include "csr_wifi_hip_ap_util.h"
#endif

/* The expiration function runs in an unknown context, so limit what we do
   in this function to safe stuff. Processing of the STA peer list will
   subsequently take place in the BH thread (normal thread context). */
void CsrWifiHipTimerExpInactivity(void *pointer)
{
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) pointer;
    (void) CsrEventSet(&priv->bhEventHandle, CSR_WIFI_HIP_EVENT_MASK_STA_INACT);

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                        "unifi : CsrWifiHipTimerExpInactivity: Received exp set event timer time \n"));
}

void csrWifiHipTimerInactivityHandler(CsrWifiHipHalPriv *priv)
{
#ifdef CSR_WIFI_AP_ENABLE
    CsrWifiHipStaPeerInfo *staRecord = NULL;
    CsrUint8 i = 0, j = 0;
    CsrTime now;
    CsrTime inactiveTime;
    CsrWifiHipVifInstance *vif = NULL;

    /* Check inactivity duration for each connected station */
    now = CsrTimeGet(NULL);

    (void) CsrMutexLock(&priv->configurationMutex);
    for (j = 0; j < CSR_WIFI_MAX_INTERFACES; j++)
    {
        if ((priv->vif[j].interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_AP) ||
            (priv->vif[j].interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_P2PGO))
        {
            vif = &priv->vif[j];

            if (vif->inactiveDetectEnabled == FALSE)
            {
                continue;
            }

            for (i = 0; i < CSR_WIFI_HIP_PEER_CONNECTIONS_MAX; i++)
            {
                if (vif->staPeerInfo[i].entryInUse == TRUE)
                {
                    staRecord = &vif->staPeerInfo[i];

                    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6,
                                        "unifi%d: csrWifiHipTimerInactivityHandler: staInfo %u active %u now %u lastActivityTime %u\n",
                                        priv->instance,
                                        staRecord->associationId,
                                        staRecord->active,
                                        now,
                                        staRecord->lastActivityTime));

                    if (staRecord->active == TRUE)
                    {
                        staRecord->active = FALSE;
                        staRecord->lastActivityTime = now;
                        continue;
                    }

                    if (staRecord->lastActivityTime > now)
                    {
                        /* Simple timer wrap (for 1 wrap) */
                        inactiveTime = CsrTimeAdd((CsrTime) CsrTimeSub(CSR_SCHED_TIME_MAX, staRecord->lastActivityTime), now);
                    }
                    else
                    {
                        inactiveTime = (CsrTime) CsrTimeSub(now, staRecord->lastActivityTime);
                    }

                    if (inactiveTime >= CSR_WIFI_HIP_STA_INACTIVE_TIMEOUT_VAL)
                    {
                        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3,
                                            "unifi%d: csrWifiHipTimerInactivityHandler: STA is Inactive - AID = %d inactiveTime = %d\n",
                                            priv->instance, staRecord->associationId, inactiveTime));

                        /* Station is in-active,  time to probe the station to determine it is
                        * alive or not. Send a null frame to the station and check the status
                        * of null frame transmission. If it fails with error RETRY_LIMIT
                        * then throw out the station.
                        */

                        if (staRecord->hostTagNullFrame != CSR_WIFI_HIP_HOST_TAG_INVALID)
                        {
                            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3,
                                                "unifi%d: csrWifiHipTimerInactivityHandler: Probe null frame already sent\n",
                                                priv->instance));

                            /* If null frame is already sent and we do not have any activity for
                            * more than 3 listen intervals then send a disconnected
                            * indication to SME, to delete the station from station
                            * record list.
                            * The inactivity is already more than CSR_WIFI_HIP_STA_INACTIVE_TIMEOUT_VAL
                            * and this check ensures if the listen interval is a larger
                            * value than CSR_WIFI_HIP_STA_INACTIVE_TIMEOUT_VAL.
                            */
                            if (inactiveTime > (CsrTime) (3 * (staRecord->listenIntervalInTUs * 1024)))
                            {
                                /* The SME/NME may be waiting for confirmation for requested frames to this station.
                                * So disassociate the queue set in packet scheduler that triggers the auto
                                * confirmation for undelivered packets
                                */
                                CsrWifiHipPacketSchedulerQueueSetDisassociate(priv->card, vif->interfaceTag, staRecord->associationId);

                                CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                                      "unifi%d: csrWifiHipTimerInactivityHandler: Router Disconnected IND Peer (%x-%x-%x-%x-%x-%x)\n",
                                                      priv->instance,
                                                      staRecord->peerMacAddress.a[0],
                                                      staRecord->peerMacAddress.a[1],
                                                      staRecord->peerMacAddress.a[2],
                                                      staRecord->peerMacAddress.a[3],
                                                      staRecord->peerMacAddress.a[4],
                                                      staRecord->peerMacAddress.a[5]));

                                CsrWifiRouterCtrlConnectedIndSend(priv->smeAppHandle,
                                                                  0,
                                                                  vif->interfaceTag,
                                                                  staRecord->peerMacAddress,
                                                                  CSR_WIFI_ROUTER_CTRL_PEER_DISCONNECTED);
                            }
                        }
                        else
                        {
                            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3,
                                                "unifi%d: csrWifiHipTimerInactivityHandler: Send a null frame to probe STA presence\n",
                                                priv->instance));
                            csrWifiHipApFrameAndsendNullFrame(priv, vif, staRecord, CSR_CONTENTION);
                        }
                    }
                }
            }

            /* Schedule a new timer */
            CsrTimerStart(&vif->inActivityTimerHandle, CSR_WIFI_HIP_STA_INACTIVE_DETECTION_TIMER_INTERVAL);
        }
    }

    (void) CsrMutexUnlock(&priv->configurationMutex);
#endif
}
