/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"

#include "csr_wifi_hip_log_text.h"
#include "csr_wifi_hip_hal_priv.h"
#include "csr_wifi_hip_sigs.h"
#include "csr_wifi_hip_list.h"
#include "csr_wifi_hip_conversions.h"
#include "csr_wifi_hip_util.h"
#include "csr_wifi_hip_ap_util.h"
#include "csr_wifi_router_ctrl_lib.h"
#include "csr_framework_ext.h"


#ifdef CSR_WIFI_AP_ENABLE

/***************************** Inactivity detection *******************************/
void csrWifiHipApFrameAndsendNullFrame(CsrWifiHipHalPriv     *priv,
                                       CsrWifiHipVifInstance *vif,
                                       CsrWifiHipStaPeerInfo *staInfo,
                                       CSR_PRIORITY           priority)
{
    CsrWifiHipBulkDataParam data_ptrs;
    CsrResult result;
    CSR_SIGNAL signal;
    CSR_MA_PACKET_REQUEST *req = &signal.u.MaPacketRequest;
    CSR_TRANSMISSION_CONTROL transmissionControl = (CSR_TRANSMISSION_CONTROL) 0;
    CSR_RATE transmitRate = 0;
    CsrUint16 vifIndex;

    result = priv->allocFrame(priv->osLayerContext, &data_ptrs.d[0], MAC_HEADER_SIZE);
    if (result != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_LOG_DEF,
                              "unifi%d: csrWifiHipApFrameAndsendNullFrame: Failed to allocate memory for NULL frame\n",
                              priv->instance));
        return;
    }

    data_ptrs.d[1].os_data_ptr = NULL;
    data_ptrs.d[1].os_net_buf_ptr = NULL;
    data_ptrs.d[1].net_buf_length = data_ptrs.d[1].data_length = 0;

    if (csrWifiHipPrepareAndAddMacheader(priv, vif, priority, 0, staInfo->peerMacAddress.a,
                                         vif->staInfo.bssid.a, MAC_HEADER_SIZE, 0, &data_ptrs,
                                         staInfo))
    {
        CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_LOG_DEF,
                              "unifi%d: csrWifiHipApFrameAndsendNullFrame: Failed to create MAC header\n",
                              priv->instance));
        priv->freeFrame(priv->osLayerContext, &data_ptrs.d[0]);
        return;
    }


    switch (vif->interfaceMode)
    {
        case CSR_WIFI_ROUTER_CTRL_MODE_AP:
            /* Send the null frame at 1 Mbps data rate */
            transmitRate = 2;
            break;
        case CSR_WIFI_ROUTER_CTRL_MODE_P2PGO:
            /* Send the null frame at 6 Mbps data rate */
            transmitRate = 12;
            break;
        default:
            transmitRate = 0;
    }

    transmissionControl &= ~(CSR_NO_CONFIRM_REQUIRED);
    vifIndex = CsrWifiHipVifIndexGetReq(priv, vif->interfaceTag);

    /* Prepares Ma-packet.req signal */
    CsrWifiHipConstructMaPacketReqSignal(priv, vif->interfaceTag, priority, transmitRate,
                                         CSR_WIFI_HIP_HOST_TAG_INVALID, transmissionControl,
                                         CSR_WIFI_HIP_HAL_INTERNAL_CLIENT_TAG, (((CsrUint8 *) data_ptrs.d[0].os_data_ptr) + 4),
                                         vifIndex, (CsrUint8 *) &signal, CSR_WIFI_HIP_FRAMETYPE_IEEE80211_DATA, 0);

    /* Save host tag to check the status on reception of MA packet confirm */
    staInfo->hostTagNullFrame = req->HostTag;

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3,
                        "unifi%d: csrWifiHipApFrameAndsendNullFrame: STA AID = %d hostTag = %x\n",
                        priv->instance,
                        staInfo->associationId,
                        req->HostTag));

    result = csrWifiHipPackAndSendFrame(priv, vif, staInfo->associationId,
                                        CSR_WIFI_HIP_PORT_TYPE_CONTROLLED,
                                        CSR_WIFI_HIP_FRAMETYPE_IEEE80211_DATA,
                                        CSR_WIFI_HIP_UNICAST_PDU,
                                        &signal, &data_ptrs);

    if (result)
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                            "unifi%d: csrWifiHipApFrameAndsendNullFrame: Failed to send Null frame Error = %d\n",
                            priv->instance,
                            result));
        priv->freeFrame(priv->osLayerContext, &data_ptrs.d[0]);
        staInfo->hostTagNullFrame = CSR_WIFI_HIP_HOST_TAG_INVALID;
    }
}

CsrResult csrWifiHipProcessMaPacketConfirm(CsrWifiHipHalPriv *priv, CsrUint16 interfaceTag, CSR_SIGNAL *signal)
{
    CsrWifiHipVifInstance *vif = NULL;
    CsrUint8 i;
    CsrWifiHipStaPeerInfo *staRecord = NULL;

    const CSR_MA_PACKET_CONFIRM *maPacketCfm = &signal->u.MaPacketConfirm;

    /* Get HIP vif instance */
    vif = (CsrWifiHipVifInstance *) &priv->vif[interfaceTag];

    if ((vif->interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_AP) ||
        (vif->interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_P2PGO))
    {
        /* Check if this is a confirm for null data frame sent to detect STA presence */
        for (i = 0; i < CSR_WIFI_HIP_PEER_CONNECTIONS_MAX; i++)
        {
            staRecord = (CsrWifiHipStaPeerInfo *) &(vif->staPeerInfo[i]);
            if ((staRecord->entryInUse == TRUE) && (staRecord->hostTagNullFrame == maPacketCfm->HostTag))
            {
                CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3,
                                    "unifi%d: csrWifiHipProcessMaPacketConfirm: CFM for Inactive probe Null frame (tag = %x, status = %d)\n",
                                    priv ? priv->instance : 0,
                                    maPacketCfm->HostTag,
                                    maPacketCfm->TransmissionStatus));

                staRecord->hostTagNullFrame = CSR_WIFI_HIP_HOST_TAG_INVALID;

                if ((CSR_TX_RETRY_LIMIT == maPacketCfm->TransmissionStatus) ||
                    (CSR_TX_LIFETIME == maPacketCfm->TransmissionStatus))
                {
                    CsrTime now;
                    CsrTime inactiveTime;

                    CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                          "unifi%d: csrWifiHipProcessMaPacketConfirm: Null frame to probe STA (AID = %u) presence Failed with transmission status %u\n",
                                          priv ? priv->instance : 0,
                                          staRecord->associationId,
                                          maPacketCfm->TransmissionStatus));

                    /* Recheck if there is some activity after null data is sent.
                    *
                    * If still there is no activity then send a disconnected indication
                    * to SME to delete the station record.
                    */
                    if (staRecord->active)
                    {
                        return CSR_RESULT_SUCCESS;
                    }
                    now = CsrTimeGet(NULL);

                    if (staRecord->lastActivityTime > now)
                    {
                        /* Simple timer wrap (for 1 wrap) */
                        inactiveTime = CsrTimeAdd((CsrTime) CsrTimeSub(CSR_SCHED_TIME_MAX, staRecord->lastActivityTime), now);
                    }
                    else
                    {
                        inactiveTime = (CsrTime) CsrTimeSub(now, staRecord->lastActivityTime);
                    }

                    if (inactiveTime >= CSR_WIFI_HIP_STA_INACTIVE_TIMEOUT_VAL)
                    {
                        CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                              "unifi%d: csrWifiHipProcessMaPacketConfirm: Router Disconnected IND Peer (%x-%x-%x-%x-%x-%x)\n",
                                              priv ? priv->instance : 0,
                                              staRecord->peerMacAddress.a[0],
                                              staRecord->peerMacAddress.a[1],
                                              staRecord->peerMacAddress.a[2],
                                              staRecord->peerMacAddress.a[3],
                                              staRecord->peerMacAddress.a[4],
                                              staRecord->peerMacAddress.a[5]));

                        CsrWifiRouterCtrlConnectedIndSend(priv->smeAppHandle,
                                                          0,
                                                          interfaceTag,
                                                          staRecord->peerMacAddress,
                                                          CSR_WIFI_ROUTER_CTRL_PEER_DISCONNECTED);
                    }
                }
                else if (maPacketCfm->TransmissionStatus == CSR_TX_SUCCESSFUL)
                {
                    staRecord->active = TRUE;
                }
                return CSR_RESULT_SUCCESS;
            }
        }
    }
    return CSR_RESULT_FAILURE;
}

/*
 * ---------------------------------------------------------------------------
 *  updateMacHeader
 *
 *
 *      This function updates MAC header for intra BSS frame routing.
 *
 *  Arguments:
 *      priv            Pointer to device private context struct
 *      vif             VIF instance data
 *      bulkdata        Bulkdata containing data frame to be forwarded
 *      priority        to append QoS control header in MAC header
 *      macHeaderLength Number of bytes of MAC header in received frame
 *      dMac            Destination MAC address
 *      qosDestination  Used to append QoS control field
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success or CSR_RESULT_FAILURE on error.
 * ---------------------------------------------------------------------------
 */
static CsrResult updateMacHeader(CsrWifiHipHalPriv *priv, CsrWifiHipVifInstance *vif,
                                 CsrWifiHipBulkDataParam *bulkdata, CSR_PRIORITY priority,
                                 CsrUint8 macHeaderLength, CsrUint8 **dMac, CsrBool qosDestination)
{
    CsrUint16 *frameCtrl = NULL;
    CsrUint8 direction = 0, toDS = 0, fromDs = 0, *bufPtr = NULL;
    CsrUint8 sa[ETH_ALEN], da[ETH_ALEN], macHeaderBuf[CSR_WIFI_HIP_80211_MAC_HEADER_MAX_LEN] = {0};
    CsrWifiHipBulkDataParam data_ptrs;
    CsrResult result;
    CsrWifiHipBulkDataParam recvedPayload;

    func_enter();

    /* Temporary buffer for the MAC header storage */
    CsrMemCpy(macHeaderBuf, bulkdata->d[0].os_data_ptr, macHeaderLength);

    /* Get the layer 2 payload reference from bulkdata */
    recvedPayload.d[0].os_data_ptr = (CsrUint8 *) bulkdata->d[0].os_data_ptr + macHeaderLength;
    recvedPayload.d[0].data_length = bulkdata->d[0].data_length - macHeaderLength;

    /* Pointer to frame control field */
    frameCtrl = (CsrUint16 *) macHeaderBuf;

    toDS = CSR_WIFI_HIP_IEEE80211_HAS_TO_DS(*frameCtrl);
    fromDs = CSR_WIFI_HIP_IEEE80211_HAS_FROM_DS(*frameCtrl);

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5, "unifi%d: updateMacHeader: fromDs = %x, toDs = %x\n",
                        priv ? priv->instance : 0, fromDs, toDS));
    direction = ((fromDs | (toDS << 1)) & 0x3);

    /* Address 1 or 3 from the MAC header */
    CsrMemCpy(da, macHeaderBuf + 4 + toDS * 12, ETH_ALEN);
    /* Address 2, 3 or 4 from the MAC header */
    CsrMemCpy(sa, macHeaderBuf + 10 + fromDs * (6 + CSR_WIFI_HIP_IEEE80211_HAS_TO_DS(*frameCtrl) * 8), ETH_ALEN);

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3, "unifi%d: updateMacHeader: Direction = %x\n",
                        priv ? priv->instance : 0, direction));

    /* Update toDs, fromDs and address fields in MAC header */
    if (direction == 2)
    {
        /* Access point received frame with: toDs = 1 and fromDs = 0 from peer (toAP).
         * The access point has to update the MAC header fields before forwarding to
         * the destination to fromDs = 1 and toDs = 0 (fromAp).
         */
        (*frameCtrl) &= ~(CSR_WIFI_HIP_IEEE80211_FCTL_TO_DS);
        (*frameCtrl) |= CSR_WIFI_HIP_IEEE80211_FCTL_FROM_DS;

        /* Address 1: MAC address of the actual destination */
        CsrMemCpy((macHeaderBuf + CSR_WIFI_HIP_MAC_HEADER_ADDR1_OFFSET), da, ETH_ALEN);

        /* Address 2: The MAC address of the AP */
        CsrMemCpy((macHeaderBuf + CSR_WIFI_HIP_MAC_HEADER_ADDR1_OFFSET + ETH_ALEN), &vif->staInfo.bssid, ETH_ALEN);

        /* Address 3: MAC address of the actual source from MAC header */
        CsrMemCpy((macHeaderBuf + CSR_WIFI_HIP_MAC_HEADER_ADDR1_OFFSET + ETH_ALEN + ETH_ALEN), sa, ETH_ALEN);
    }
    else
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi%d: updateMacHeader: Invalid direction %u. Not support. Dropping frame.\n",
                            priv ? priv->instance : 0, direction));
        return CSR_RESULT_FAILURE;
    }

    /* frameType is Data always, Validation is done before calling this function */

    /* Check for the souce station type */
    switch (*frameCtrl & CSR_WIFI_HIP_IEEE80211_FCTL_SUB_TYPE_MASK)
    {
        case CSR_WIFI_HIP_IEEE80211_FCTL_SUB_TYPE_QOS_DATA:

            /* No need to modify the QoS control field */
            if (!qosDestination)
            {
                /* If source STA is QoS enabled and if order bit is set, then HTC is supported by
                 * peer station and HTC field present in MAC header */
                if (*frameCtrl & CSR_WIFI_HIP_IEEE80211_FCTL_ORDER)
                {
                    /* HT control field present in MAC header */
                    macHeaderLength -= (HT_CONTROL_HEADER_SIZE + QOS_CONTROL_HEADER_SIZE);
                }
                else
                {
                    macHeaderLength -= QOS_CONTROL_HEADER_SIZE;
                }

                /* Destination STA is non QoS so change frame subtype to DATA */
                *frameCtrl &= ~(CSR_WIFI_HIP_IEEE80211_FCTL_SUB_TYPE_MASK);

                result = priv->allocFrame(priv->osLayerContext, &data_ptrs.d[0], recvedPayload.d[0].data_length + macHeaderLength);
                if (result != CSR_RESULT_SUCCESS)
                {
                    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                           "unifi%d: updateMacHeader: Failed to allocate request_data.\n", priv ? priv->instance : 0));
                    return CSR_RESULT_FAILURE;
                }

                CsrMemCpy((void *) (data_ptrs.d[0].os_data_ptr + macHeaderLength),
                          recvedPayload.d[0].os_data_ptr,
                          recvedPayload.d[0].data_length);

                /* Free memory allocated by old bulkdata */
                priv->freeFrame(priv->osLayerContext, &bulkdata->d[0]);

                /* Update bulkdata params */
                bulkdata->d[0].os_data_ptr = data_ptrs.d[0].os_data_ptr;
                bulkdata->d[0].data_length = data_ptrs.d[0].data_length;
                bulkdata->d[0].os_net_buf_ptr = data_ptrs.d[0].os_net_buf_ptr;
                bulkdata->d[0].net_buf_length = data_ptrs.d[0].net_buf_length;

                bufPtr = (CsrUint8 *) bulkdata->d[0].os_data_ptr;
            }
            else
            {
                /* At the moment there is no way to determine if the destination STA supports
                   HTC; hence, should a frame with HTC be received, the field will be removed. */
                CsrUint16 *qosCtrl = NULL;
                CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5, "unifi%d: updateMacHeader: QoS DATA and Destination is QoS station \n",
                                    priv ? priv->instance : 0));
                if (*frameCtrl & CSR_WIFI_HIP_IEEE80211_FCTL_ORDER)
                {
                    qosCtrl = (CsrUint16 *) (macHeaderBuf + (macHeaderLength - (HT_CONTROL_HEADER_SIZE + QOS_CONTROL_HEADER_SIZE)));

                    /* The HTC header is the last part of the received MAC header,
                       so ensure that it is not copied into the new frame */
                    macHeaderLength -= HT_CONTROL_HEADER_SIZE;

                    /* No reason to free and allocate a new bulkdata block since the HTC
                       header is being removed, so the allocated space is sufficient */
                    CsrMemMove(bulkdata->d[0].os_data_ptr + macHeaderLength,
                               bulkdata->d[0].os_data_ptr + macHeaderLength + HT_CONTROL_HEADER_SIZE,
                               bulkdata->d[0].data_length - (macHeaderLength + HT_CONTROL_HEADER_SIZE));
                    bulkdata->d[0].data_length -= HT_CONTROL_HEADER_SIZE;
                }
                else
                {
                    qosCtrl = (CsrUint16 *) (macHeaderBuf + (macHeaderLength - QOS_CONTROL_HEADER_SIZE));
                }

                /* Prepare the QoS control field */
                if (priority >= 7)
                {
                    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3, "unifi%d: updateMacHeader: Data packets priority is more than 7, priority = %x\n",
                                        priv ? priv->instance : 0,  priority));
                    *qosCtrl |= 0x0007;
                }
                else
                {
                    *qosCtrl |= (CsrUint16) priority;
                }

                bufPtr = (CsrUint8 *) bulkdata->d[0].os_data_ptr;
            }
            break;

        default:
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5, "unifi%d: updateMacHeader: Normal Data packet, no QoS \n",
                                priv ? priv->instance : 0));

            if (qosDestination)
            {
                CsrUint16 qosCtrl = 0;

                CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5, "unifi%d: updateMacHeader: Destination is QoS station \n",
                                    priv ? priv->instance : 0));

                /* Prepare the qos control field */
                if (priority >= 7)
                {
                    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3, "unifi%d: updateMacHeader: Data packets priority is more than 7, priority = %x\n",
                                        priv ? priv->instance : 0,  priority));
                    qosCtrl |= 0x0007;
                }
                else
                {
                    qosCtrl |= (CsrUint16) priority;
                }

                /* No Amsdu is in AP buffer */
                qosCtrl &= ~(CSR_WIFI_HIP_IEEE80211_QOS_CONTROL_EOSP);
                if (da[0] & 0x1)
                {
                    /* Multicast/broadcast frames, no acknowledgement needed */
                    qosCtrl |= (CSR_WIFI_HIP_IEEE80211_QOS_CONTROL_ACK_POLICY_NO_ACK);
                }
                else
                {
                    /* Normal acknowledgement */
                    qosCtrl |= CSR_WIFI_HIP_IEEE80211_QOS_CONTROL_ACK_POLICY_NORMAL_ACK;
                }

                /* Update new MAC header length with QoS control header size */
                macHeaderLength += QOS_CONTROL_HEADER_SIZE;

                CSR_COPY_UINT16_TO_LITTLE_ENDIAN(qosCtrl, (macHeaderBuf + macHeaderLength - QOS_CONTROL_HEADER_SIZE));

                /* Received DATA frame but destination is QoS station so update subtype
                 * to QOS_DATA
                 */
                *frameCtrl |= CSR_WIFI_HIP_IEEE80211_FCTL_SUB_TYPE_QOS_DATA;

                /* Create new bulkdata for updated MAC header */
                result = priv->allocFrame(priv->osLayerContext, &data_ptrs.d[0], recvedPayload.d[0].data_length + macHeaderLength);
                if (result != CSR_RESULT_SUCCESS)
                {
                    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: updateMacHeader: Failed to allocate memory for new bulkdata\n", priv ? priv->instance : 0));
                    return CSR_RESULT_FAILURE;
                }

                CsrMemCpy((void *) (data_ptrs.d[0].os_data_ptr + macHeaderLength),
                          recvedPayload.d[0].os_data_ptr,
                          recvedPayload.d[0].data_length);

                /* Free memory allocated by old bulkdata */
                priv->freeFrame(priv->osLayerContext, &bulkdata->d[0]);

                /* Update bulkdata params */
                bulkdata->d[0].os_data_ptr = data_ptrs.d[0].os_data_ptr;
                bulkdata->d[0].data_length = data_ptrs.d[0].data_length;
                bulkdata->d[0].os_net_buf_ptr = data_ptrs.d[0].os_net_buf_ptr;
                bulkdata->d[0].net_buf_length = data_ptrs.d[0].net_buf_length;

                bufPtr = (CsrUint8 *) bulkdata->d[0].os_data_ptr;
            }
            else
            {
                /* Destination station is a none QoS station */
                CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3, "unifi%d: updateMacHeader: Destination is not a QSTA\n",
                                    priv ? priv->instance : 0));
                bufPtr = (CsrUint8 *) bulkdata->d[0].os_data_ptr;
            }
    }

    /* Update the MAC header */
    CsrMemCpy(bufPtr, macHeaderBuf, macHeaderLength);

    /* After MAC header update, update the dMac pointer */
    *dMac = (CsrUint8 *) bulkdata->d[0].os_data_ptr + 4;

    return CSR_RESULT_SUCCESS;
}

static CsrResult processStaRecordsForSendingData(CsrWifiHipHalPriv     *priv,
                                                 CsrWifiHipVifInstance *vif,
                                                 CsrWifiHipStaPeerInfo *srcStaInfo,
                                                 CsrWifiHipStaPeerInfo *dstStaInfo)
{
    if (srcStaInfo->currentPeerState == CSR_WIFI_ROUTER_CTRL_PEER_DISCONNECTED)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: processStaRecordsForSendingData: Peer State not connected AID = %x, handle = %x, control port state = %x\n",
                               priv ? priv->instance : 0,
                               srcStaInfo->associationId, srcStaInfo->assignedHandle, srcStaInfo->peerControlledPort->port_action));
        return CSR_RESULT_FAILURE;
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5,
                        "unifi%d: processStaRecordsForSendingData: Process data\n",
                        priv ? priv->instance : 0));

    switch (dstStaInfo->peerControlledPort->port_action)
    {
        case CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_CLOSED_DISCARD:
        case CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_CLOSED_BLOCK:
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5,
                                "unifi%d: processStaRecordsForSendingData: Destination port is closed/blocked, discarding the packet\n",
                                priv ? priv->instance : 0));
            return CSR_RESULT_FAILURE;
        default:
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5,
                                "unifi%d: processStaRecordsForSendingData: Destination port state is open\n",
                                priv ? priv->instance : 0));
    }
    /* port state is open, destination station record is valid, Power save state is
     * validated in csrWifiHipProcessMaPacketReq function
     */

    return CSR_RESULT_SUCCESS;
}

CsrResult csrWifiHipApProcessDataPdu(CsrWifiHipHalPriv *priv, CsrWifiHipVifInstance *vif,
                                     CsrUint8 *daddr, CsrWifiHipStaPeerInfo *srcStaInfo,
                                     CSR_SIGNAL *signal, CsrWifiHipBulkDataParam *bulkdata,
                                     CsrUint8 macHeaderLengthInBytes)
{
    CsrBool forwardRxFrame = FALSE, qosDestination = FALSE;
    CsrWifiHipMaPacketPriority priority = CSR_WIFI_HIP_MA_PACKET_PRIORITY_QOS_UP0;
    CsrWifiHipStaPeerInfo *dstStaInfo = NULL;
    CsrUint16 prot = 0, protOffset = 0, pktOffset = 0;
    CsrWifiHipBulkDataParam bulkDataCopy;
    CsrWifiHipBulkDataParam bulkDataToUnifi;
    CsrResult result;

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5,
                        "unifi%d: >> csrWifiHipApProcessDataPdu\n", priv->instance));

    /* Fetch the destination record from staion record database */
    dstStaInfo = csrWifiHipGetStationRecordFromPeerMacAddress(priv, vif, daddr);

    if ((vif->interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_P2PGO) && (vif->intraBssEnabled == FALSE))
    {
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                           "unifi%d: csrWifiHipApProcessDataPdu: P2P GO intrabssEnabled?= %d\n",
                           priv->instance, vif->intraBssEnabled));
        /* In P2P GO case, if intraBSS distribution Disabled then don't do IntraBSS routing */

        /* If destination in our BSS then drop otherwise give packet to netdev */
        if (dstStaInfo)
        {
            priv->freeFrame(priv->osLayerContext, &bulkdata->d[0]);
            vif->receiveStats.rxDropped++;
            return CSR_RESULT_FAILURE;
        }
        /* May be associated P2PCLI trying to send the packets on backbone (Netdev) */
        return CSR_RESULT_SUCCESS;
    }

    /* If the packet destination is CSR_AP then, check port status of peer
     * station & route accordingly
     */
    if (!CsrMemCmp(daddr, vif->staInfo.bssid.a, ETH_ALEN))
    {
        /* This packet will be given to the TCP/IP stack since this packet is for us(AP)
         * No routing needed
         */
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3,
                            "unifi%d: csrWifiHipApProcessDataPdu: Destination address is csr_ap\n", priv->instance));
        return CSR_RESULT_SUCCESS;
    }

    bulkDataToUnifi.d[0].os_data_ptr = (CsrUint8 *) bulkdata->d[0].os_data_ptr;
    bulkDataToUnifi.d[0].data_length = bulkdata->d[0].data_length;
    bulkDataToUnifi.d[0].net_buf_length = bulkdata->d[0].net_buf_length;
    bulkDataToUnifi.d[0].os_net_buf_ptr = bulkdata->d[0].os_net_buf_ptr;

    bulkDataToUnifi.d[1].os_data_ptr = (CsrUint8 *) bulkdata->d[1].os_data_ptr;
    bulkDataToUnifi.d[1].data_length = bulkdata->d[1].data_length;
    bulkDataToUnifi.d[1].net_buf_length = bulkdata->d[1].net_buf_length;
    bulkDataToUnifi.d[1].os_net_buf_ptr = bulkdata->d[1].os_net_buf_ptr;


    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                       "unifi%d: csrWifiHipApProcessDataPdu: dMac = %2x-%2x-%2x-%2x-%2x-%2x\n",
                       priv->instance,
                       daddr[0], daddr[1], daddr[2],
                       daddr[3], daddr[4], daddr[5]));

    if (!dstStaInfo)
    {
        if (!(daddr[0] & 0x1))
        {
            /* Destination not in station record and it's a unicast packet, so pass the packet to network stack */
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3,
                                "unifi%d: csrWifiHipApProcessDataPdu: Unicast frame and destination record doesn't exist, forward RX frame\n",
                                priv->instance));
            return CSR_RESULT_SUCCESS;
        }
        else
        {
            /* Packet is multicast/broadcast, copy the bulkdata to bulkdataCopy, send bulkdata to
             * netdev and bulkdataCopy back to Unifi to relay on BSS
             */
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5,
                                "unifi%d: csrWifiHipApProcessDataPdu: Copy bulkdata to xmit on BSS\n", priv->instance));
            result = priv->allocFrame(priv->osLayerContext, &bulkDataCopy.d[0], bulkdata->d[0].data_length);
            if (result != CSR_RESULT_SUCCESS)
            {
                /* We don't have memory to send the frame in BSS */
                CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                                   "unifi%d: csrWifiHipApProcessDataPdu: No memory - Broadcast/multicast frame can't be sent in BSS\n",
                                   priv->instance));

                /* We return success to send the frame to net interface */
                return CSR_RESULT_SUCCESS;
            }

            CsrMemCpy((void *) (bulkDataCopy.d[0].os_data_ptr),
                      bulkdata->d[0].os_data_ptr,
                      bulkdata->d[0].data_length);

            bulkDataToUnifi.d[0].os_data_ptr = (CsrUint8 *) bulkDataCopy.d[0].os_data_ptr;
            bulkDataToUnifi.d[0].data_length = bulkDataCopy.d[0].data_length;
            bulkDataToUnifi.d[0].net_buf_length = bulkDataCopy.d[0].net_buf_length;
            bulkDataToUnifi.d[0].os_net_buf_ptr = bulkDataCopy.d[0].os_net_buf_ptr;

            forwardRxFrame = TRUE;
        }
    }
    else
    {
        /* Validate the Peer and Destination Station record */
        if (processStaRecordsForSendingData(priv, vif, srcStaInfo, dstStaInfo))
        {
            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: csrWifiHipApProcessDataPdu: Station record validation failed\n",
                               priv->instance));

            vif->receiveStats.rxDropped++;
            priv->freeFrame(priv->osLayerContext, &bulkdata->d[0]);
            return CSR_RESULT_FAILURE;
        }

        if (dstStaInfo->wmmOrQosEnabled)
        {
            /* Destination station association is QOS */
            qosDestination = TRUE;
        }
    }


    /* Obtain QoS Type and priority. First identify protocol type */
    protOffset = macHeaderLengthInBytes + CSR_WIFI_LLC_SNAP_HDR_PROT_OFFSET;
    pktOffset = protOffset + 2; /* 2 octets for the protocol field */
    prot = (bulkDataToUnifi.d[0].os_data_ptr[protOffset] << 8 |
            *(bulkDataToUnifi.d[0].os_data_ptr + protOffset + 1));
    /*
     * csrWifiHipPacketPriorityGet() expects the data pointer to after the Ethernet/SNAP header - i.e. pointing at the IP packet (or whatever).
     */
    if (!(*daddr & 0x01) && dstStaInfo && dstStaInfo->wmmOrQosEnabled)
    {
        priority = csrWifiHipPacketPriorityGet(priv, vif, (CsrUint8 *) &(bulkDataToUnifi.d[0].os_data_ptr[pktOffset]), prot);
    }
    /* Packet is allowed to send to unifi, update the MAC header */
    if (updateMacHeader(priv, vif, &bulkDataToUnifi, (CSR_PRIORITY) priority, macHeaderLengthInBytes, &daddr, qosDestination) != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: csrWifiHipApProcessDataPdu: Failed to update the Mac header\n", priv->instance));
        vif->receiveStats.rxDropped++;
        priv->freeFrame(priv->osLayerContext, &bulkDataToUnifi.d[0]);

        if (forwardRxFrame)
        {
            /* We still need to pass the frame to net device */
            return CSR_RESULT_SUCCESS;
        }
        else
        {
            return CSR_RESULT_FAILURE;
        }
    }

    /* Send the frame as MA_PACKET_REQ */
    CsrWifiHipConstructMaPacketReqSignal(priv,
                                         vif->interfaceTag,
                                         (CSR_PRIORITY) priority,
                                         (CSR_RATE) 0,
                                         CSR_WIFI_HIP_HOST_TAG_INVALID,
                                         CSR_NO_CONFIRM_REQUIRED,
                                         CSR_WIFI_HIP_HAL_INTERNAL_CLIENT_TAG,
                                         daddr,
                                         signal->u.MaPacketIndication.VirtualInterfaceIndex,
                                         (CsrUint8 *) signal, CSR_WIFI_HIP_FRAMETYPE_IEEE80211_DATA, 0);

    if (csrWifiHipProcessMaPacketReq(priv, vif,
                                     CSR_WIFI_HIP_PORT_TYPE_CONTROLLED,
                                     CSR_WIFI_HIP_FRAMETYPE_IEEE80211_DATA,
                                     daddr, (CSR_PRIORITY) priority,
                                     signal, &bulkDataToUnifi, dstStaInfo))
    {
        /* If MA-PACKET.req fails free the bulkdata reference, for
         * unicast frames original bulkdata is freed and for broadcast frames
         * bulkdataCopy memory is freed and original bulkdata is kept to be
         * passed to net device.
         */
        priv->freeFrame(priv->osLayerContext, &bulkDataToUnifi.d[0]);
    }

    if (forwardRxFrame)
    {
        /* The packet is multicast/broadcast, so after AP processing packet has to
         * be forwarded, if peer port state is open
         */
        return CSR_RESULT_SUCCESS;
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5,
                        "unifi%d: -- csrWifiHipApProcessDataPdu\n",
                        priv->instance));
    /* AP handled the packet and its a unicast packet, no need to forward frame */
    return CSR_RESULT_FAILURE;
}

#endif

CsrWifiHipStaPeerInfo *csrWifiHipGetStationRecordFromPeerMacAddress(CsrWifiHipHalPriv     *priv,
                                                                    CsrWifiHipVifInstance *vif,
                                                                    const CsrUint8        *peerMacAddress)
{
    CsrUint8 i;

    CSR_WIFI_HIP_SPINLOCK_LOCK(&priv->peerInfoLock);
    for (i = 0; i < CSR_WIFI_HIP_PEER_CONNECTIONS_MAX; i++)
    {
        if (vif->staPeerInfo[i].entryInUse == TRUE)
        {
            if (!CsrMemCmp(vif->staPeerInfo[i].peerMacAddress.a,
                           peerMacAddress,
                           ETH_ALEN))
            {
                CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5, "unifi%d: csrWifiHipGetStationRecordFromPeerMacAddress: Peer entry found in station records\n",
                                    priv ? priv->instance : 0));

                CSR_WIFI_HIP_SPINLOCK_UNLOCK(&priv->peerInfoLock);
                return (CsrWifiHipStaPeerInfo *) (&vif->staPeerInfo[i]);
            }
        }
    }

    CSR_WIFI_HIP_SPINLOCK_UNLOCK(&priv->peerInfoLock);
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5, "unifi%d: csrWifiHipGetStationRecordFromPeerMacAddress: Peer entry not found in station records\n",
                        priv ? priv->instance : 0));
    return NULL;
}

CsrResult csrWifiHipProcessMaPacketReq(CsrWifiHipHalPriv *priv, CsrWifiHipVifInstance *vif,
                                       CsrWifiHipPortType portType,
                                       CsrWifiHipFrameType frameType,
                                       CsrUint8 *peerMacAddress, CSR_PRIORITY priority,
                                       CSR_SIGNAL *signal, CsrWifiHipBulkDataParam *bulkdata,
                                       CsrWifiHipStaPeerInfo *staPeerRecord)
{
    CsrResult result = CSR_RESULT_FAILURE;
#ifdef CSR_WIFI_AP_ENABLE
    CsrWifiHipPacketType packetType;
    CsrUint32 aid = 0;
#endif

    switch (vif->interfaceMode)
    {
        case CSR_WIFI_ROUTER_CTRL_MODE_AP:
        case CSR_WIFI_ROUTER_CTRL_MODE_P2PGO:
            /* For this mode processing done below */
            break;
        default:
            if (vif->interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_NONE)
            {
                CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: csrWifiHipProcessMaPacketReq: Interface Mode is NONE", priv ? priv->instance : 0));
            }
            return result = csrWifiHipPackAndSendFrame(priv, vif, 0, portType, frameType, CSR_WIFI_HIP_PKT_TYPE_RESERVED, signal, bulkdata);
    }

    /* Only AP/P2P GO mode handling falls below */

#ifdef CSR_WIFI_AP_ENABLE
    /* Determine destination type (unicast or multicast/broadcast) */
    if (*peerMacAddress & 0x01)
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG5,
                            "unifi%d: csrWifiHipProcessMaPacketReq : multicast PDU; frameType %d\n",
                            priv ? priv->instance : 0, frameType));
        packetType = CSR_WIFI_HIP_MULTICAST_PDU;
    }
    else
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG5,
                            "unifi%d: csrWifiHipProcessMaPacketReq : unicast PDU; frameType %d\n",
                            priv ? priv->instance : 0, frameType));
        packetType = CSR_WIFI_HIP_UNICAST_PDU;
    }

    if (staPeerRecord)
    {
        /* update station activity flag */
        staPeerRecord->active = TRUE;
        aid = staPeerRecord->associationId;
    }
    else if ((packetType == CSR_WIFI_HIP_UNICAST_PDU) && (frameType == CSR_WIFI_HIP_FRAMETYPE_IEEE80211_DATA))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: csrWifiHipProcessMaPacketReq: Unicast PDU but staPeerRecord = NULL\n",
                               priv ? priv->instance : 0));
        return CSR_RESULT_FAILURE;
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG5,
                        "unifi%d: csrWifiHipProcessMaPacketReq: packetType = %d, frameType = %d, aid = %d\n",
                        priv ? priv->instance : 0, packetType, frameType, aid));
    result = csrWifiHipPackAndSendFrame(priv, vif, aid,
                                        portType, frameType, packetType,
                                        signal, bulkdata);

    if (result == CSR_WIFI_HIP_RESULT_NO_SPACE)
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                            "unifi%d: csrWifiHipProcessMaPacketReq: Failed to queue frame to packet scheduler result = %u\n",
                            priv ? priv->instance : 0, result));
    }
#endif

    return result;
}
