/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"

#include "csr_wifi_hip_log_text.h"
#include "csr_wifi_hip_unifi.h"
#include "csr_wifi_hip_hal_priv.h"
#include "csr_types.h"
#include "csr_sdio.h"
#include "csr_framework_ext.h"
#include "csr_wifi_hip.h"
#include "csr_wifi_hip_card.h"
#include "csr_wifi_hip_io.h"

static CsrSdioFunctionId csrWifiHipSdioUnifiIds[] =
{
#ifdef CSR_WIFI_DRIVER_HYDRA
    {
        SDIO_MANF_ID_CSR,
        SDIO_CARD_ID_WIFI_HYDRA,
        SDIO_WLAN_FUNC_ID_WIFI_HYDRA,
        CSR_SDIO_ANY_SDIO_INTERFACE,
    }
#else
    {
        SDIO_MANF_ID_CSR,
        SDIO_CARD_ID_UNIFI_3,
        SDIO_WLAN_FUNC_ID_UNIFI_3,
        CSR_SDIO_ANY_SDIO_INTERFACE,
    },
    {
        SDIO_MANF_ID_CSR,
        SDIO_CARD_ID_UNIFI_4,
        SDIO_WLAN_FUNC_ID_UNIFI_4,
        CSR_SDIO_ANY_SDIO_INTERFACE,
    }
#endif
};

static void csrWifiHipSdioInserted(CsrSdioFunction *function);
static void csrWifiHipSdioRemoved(CsrSdioFunction *function);
static CsrSdioInterruptDsrCallback csrWifiHipSdioInterrupt(CsrSdioFunction *function);
static void csrWifiHipSdioSuspend(CsrSdioFunction *function);
static void csrWifiHipSdioResume(CsrSdioFunction *function);

/*
 * Structure to register with SDIO driver.
 */
static CsrSdioFunctionDriver csrWifiHipSdioFuncDrv =
{
    csrWifiHipSdioInserted,
    csrWifiHipSdioRemoved,
    csrWifiHipSdioInterrupt,
    csrWifiHipSdioSuspend,
    csrWifiHipSdioResume,
    csrWifiHipSdioUnifiIds,
    sizeof(csrWifiHipSdioUnifiIds) / sizeof(csrWifiHipSdioUnifiIds[0]),
    NULL
};


static void csrWifiHipSdioResume(CsrSdioFunction *function)
{
#ifdef CSR_WIFI_HIP_SUSPEND_ENABLE
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) function->driverData;

    priv->suspend = FALSE;

#ifdef CSR_WIFI_HIP_SUSPEND_WOL
    if (priv->bhThreadStarted)
    {
        (void) CsrEventSet(&priv->bhEventHandle, CSR_WIFI_HIP_EVENT_MASK_BH_RESUME);
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi : csrWifiHipSdioResume with wifi on"));
    }
    else
    {
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi : csrWifiHipSdioResume with wifi off"));
        CsrSdioResumeAcknowledge(function, CSR_RESULT_SUCCESS); /* presumably wifi is off, nothing to do */
    }
#else
    if (priv->bh_thread_before_suspend)
    {
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi : csrWifiHipSdioResume wifi was on"));

        /* BH was terminated, tell the SME to resume which will cause a wifi-on */
        CsrWifiHipResumeInd(priv->osLayerContext, 0, FALSE);
    }
    else
    {
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi : csrWifiHipSdioResume wifi was off"));
        CsrSdioResumeAcknowledge(function, CSR_RESULT_SUCCESS); /* presumably wifi is off, nothing to do */
    }
#endif
#else
    CsrSdioResumeAcknowledge(function, CSR_RESULT_SUCCESS);
#endif
}

static void csrWifiHipSdioSuspend(CsrSdioFunction *function)
{
#ifdef CSR_WIFI_HIP_SUSPEND_ENABLE
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) function->driverData;

    /* save because in hard suspend cases SME will now turn wifi off, stopping bh */
    priv->bh_thread_before_suspend = priv->bhThreadStarted;

    if (priv->bhThreadStarted)
    {
        (void) CsrEventSet(&priv->bhEventHandle, CSR_WIFI_HIP_EVENT_MASK_BH_SUSPEND);

        /* This will cause the packet scheduler to pause every queue it knows about. */
        CsrWifiHipPacketSchedulerSuspend(priv->card);
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi : csrWifiHipSdioSuspend with wifi on"));
    }
    else
    {
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi : csrWifiHipSdioSuspend with wifi off"));
        CsrSdioSuspendAcknowledge(function, CSR_RESULT_SUCCESS);
    }
#else
    CsrSdioSuspendAcknowledge(function, CSR_RESULT_SUCCESS);
#endif
}

static void csrWifiHipDsrCb(CsrSdioFunction *function)
{
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) function->driverData;

#ifdef CSR_WIFI_HIP_SUSPEND_ENABLE
    /* SDIO Interrupts are not delivered to the BH while in suspend. The interrupt
     * should trigger a host wake up, and a resume callback from the SDIO layer.
     */
    if (priv->suspend == FALSE)
    {
        unifi_sdio_interrupt_handler(priv->card);
    }
    else
    {
        /* Remember that an interrupt occurred, because it must be acknowledged once
         * the resume is underway.
         */
        priv->resumeIrq = TRUE;
    }
#else
    unifi_sdio_interrupt_handler(priv->card);
#endif
}

static CsrSdioInterruptDsrCallback csrWifiHipSdioInterrupt(CsrSdioFunction *function)
{
    return (CsrSdioInterruptDsrCallback) csrWifiHipDsrCb;
}

static void halPrivFree(CsrWifiHipHalPriv *priv)
{
    if (priv != NULL)
    {
        if (priv->card != NULL)
        {
            unifi_free_card(priv->card);
            priv->card = NULL;
        }

        CsrMemFree(priv);
        priv = NULL;
    }
}

static void csrWifiHipSdioRemoved(CsrSdioFunction *function)
{
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) function->driverData;

    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3,
                       "unifi : csrWifiHipSdioRemoved\n"));

    if (priv)
    {
#if defined CSR_WIFI_DRIVER_HYDRA && defined CSR_WIFI_DRIVER_USE_HYDRA_DRIVER
        /* Unregister as a Hydra SDIO function driver */
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                           "unifi : calling CsrWifiHipHydraUnregister(osLayerContext=0x%x, function=0x%x)\n",
                           priv->osLayerContext,
                           function
                           ));
        CsrWifiHipHydraUnregister(priv->osLayerContext, function);
#endif
        /* Tell the OS layer to clean up */
        CsrWifiHipRemovedInd(priv->osLayerContext, priv->instance);
        halPrivFree(priv);
    }

#ifdef CSR_WIFI_DRIVER_HYDRA
    CsrWifiHipRemoveOsDevice(function);
#endif

    CsrSdioRemovedAcknowledge(function);
}

static void csrWifiHipSdioInserted(CsrSdioFunction *function)
{
    CsrWifiHipHalPriv *priv = NULL;

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3,
                        "unifi : csrWifiHipSdioInserted\n"));

    priv = CsrMemAlloc(sizeof(CsrWifiHipHalPriv));
    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi : csrWifiHipSdioInserted: Could not allocate memory for priv\n"));
        CsrSdioInsertedAcknowledge(function, CSR_RESULT_FAILURE);
        return;
    }

    CsrMemSet(priv, 0, sizeof(CsrWifiHipHalPriv));

    /* Save SDIO function handle */
    priv->sdio = function;

    /* Save HIP driver context */
    function->driverData = priv;

    /* Alloc card structure */
    priv->card = unifi_alloc_card(priv->sdio, priv);
    if (priv->card == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi : csrWifiHipSdioInserted: Could not allocate memory for card\n"));
        halPrivFree(priv);
        function->driverData = NULL;
        CsrSdioInsertedAcknowledge(function, CSR_RESULT_FAILURE);
        return;
    }

#ifdef CSR_WIFI_DRIVER_HYDRA
    /* There is no glue layer callback in which to add the OS device,
     * so it must be done from the HAL
     */
    CsrWifiHipAddOsDevice(function);
#endif

    CsrWifiHipInsertedInd((void *) priv);
}

void CsrWifiHipInsertedRes(void *hipHandle, void *osLayerContext, CsrResult result, CsrUint32 instance)
{
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi : Received invalid hipHandle. Can not call CsrSdioInsertedAcknowledge\n"));
        return;
    }

    if (result != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi : Failure during insertedInd in os layer\n"));
        CsrSdioInsertedAcknowledge(priv->sdio, CSR_RESULT_FAILURE);
        return;
    }

    /* Store platform specific information in HAL's instance data. */
    priv->instance = instance;
    priv = (CsrWifiHipHalPriv *) hipHandle;
    priv->osLayerContext = osLayerContext;

    CsrSdioInsertedAcknowledge(priv->sdio, CSR_RESULT_SUCCESS);
#if defined CSR_WIFI_DRIVER_HYDRA && defined CSR_WIFI_DRIVER_USE_HYDRA_DRIVER
    /* Register as a Hydra SDIO function driver */
    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                       "unifi : calling CsrWifiHipHydraRegister: osLayerContext=0x%x, priv->sdio=0x%x\n",
                       priv->osLayerContext,
                       priv->sdio));
    CsrWifiHipHydraRegister(priv->osLayerContext, priv->sdio);
#endif
}

CsrResult csrWifiHipSdioRegister(void)
{
    CsrResult r;

    r = CsrSdioFunctionDriverRegister(&csrWifiHipSdioFuncDrv);

    /* Return generic error code to avoid confusion by returning a CSR SDIO one */
    return (r == CSR_RESULT_SUCCESS) ? CSR_RESULT_SUCCESS : CSR_RESULT_FAILURE;
}

void csrWifiHipSdioUnregister(void)
{
    CsrSdioFunctionDriverUnregister(&csrWifiHipSdioFuncDrv);
}
