/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"

#include "csr_wifi_hip_log_text.h"
#include "csr_wifi_hip_hal_priv.h"
#include "csr_wifi_hip_unifi.h"
#include "csr_wifi_common.h"
#include "csr_wifi_hip.h"
#include "csr_wifi_hip_list.h"
#include "csr_wifi_hip_util.h"
#include "csr_wifi_router_ctrl_prim.h"

static void ctrlPortQueueFree(CsrWifiHipHalPriv *priv, CsrWifiHipVifInstance *vif, struct listHeader *queue, CsrUint32 *counter)
{
    CsrWifiHipCtrlPortRxElement *element = NULL;
    struct listHeader *listHead, *placeHolder;
    CsrUint8 freeElm;

    (void) CsrMutexLock(&priv->configurationMutex);

    if (CSR_WIFI_HIP_LIST_IS_LIST_EMPTY(queue))
    {
        (void) CsrMutexUnlock(&priv->configurationMutex);
        return;
    }

    CSR_WIFI_HIP_LIST_LIST_FOR_EACH_SAFE(listHead, placeHolder, queue)
    {
        element = (CsrWifiHipCtrlPortRxElement *) CSR_WIFI_HIP_LIST_LIST_ENTRY(listHead, CsrWifiHipCtrlPortRxElement, header);
        CSR_WIFI_HIP_LIST_DELETE_ENTRY(listHead);

        *counter = *counter - 1;
        for (freeElm = 0; freeElm < CSR_WIFI_HIP_MAX_DATA_REFERENCES; freeElm++)
        {
            priv->freeFrame(priv->osLayerContext, &element->bulkdata.d[freeElm]);
        }

        CsrMemFree(element);
    }
    (void) CsrMutexUnlock(&priv->configurationMutex);
}

void csrWifiHipCtrlPortQueueFreeAll(CsrWifiHipHalPriv *priv, CsrWifiHipVifInstance *vif)
{
    ctrlPortQueueFree(priv,
                      vif,
                      &vif->uncontrolledPortQueue,
                      &vif->numOfControlledFrames);

    ctrlPortQueueFree(priv,
                      vif,
                      &vif->controlledPortQueue,
                      &vif->numOfControlledFrames);
}

/* It is a requirement that the caller ensures mutual exclusive access to
   shared data structures. */
void csrWifiHipCtrlPortQueueProcess(CsrWifiHipHalPriv *priv, CsrWifiHipVifInstance *vif,
                                    CsrWifiHipPortType portType, CsrWifiMacAddress *macAddress, CsrBool indicate)
{
    CsrTime timeStamp = 0;
    CsrWifiHipCtrlPortRxElement *element = NULL;
    struct listHeader *listHead, *placeHolder, *queue;
    static const CsrWifiMacAddress broadcastAddress = {{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF}};
    CSR_MA_PACKET_INDICATION *ind = NULL;
    CsrResult receptionResult;

    if (portType == CSR_WIFI_HIP_PORT_TYPE_CONTROLLED)
    {
        queue = &vif->controlledPortQueue;
    }
    else
    {
        queue = &vif->uncontrolledPortQueue;
    }

    if (CSR_WIFI_HIP_LIST_IS_LIST_EMPTY(queue))
    {
        CSR_LOG_TEXT_DEBUG((CsrWifiHalLto, CSR_WIFI_HAL_LTSO_PORT,
                            "unifi%d: csrWifiHipCtrlPortQueueProcess: No buffered RX frame (port %d)\n",
                            priv ? priv->instance : 0, portType));
        return;
    }


    CSR_LOG_TEXT_DEBUG((CsrWifiHalLto, CSR_WIFI_HAL_LTSO_PORT,
                        "unifi%d: csrWifiHipCtrlPortQueueProcess: priv %p, vif %p, macAddress %02X:%02X:%02X:%02X:%02X:%02X indicate %s\n",
                        priv->instance, priv, vif, macAddress->a[0], macAddress->a[1], macAddress->a[2], macAddress->a[3], macAddress->a[4], macAddress->a[5], indicate ? "Yes" : "No"));

    CSR_WIFI_HIP_LIST_LIST_FOR_EACH_SAFE(listHead, placeHolder, queue)
    {
        element = (CsrWifiHipCtrlPortRxElement *) CSR_WIFI_HIP_LIST_LIST_ENTRY(listHead, CsrWifiHipCtrlPortRxElement, header);

        if (!CsrMemCmp(broadcastAddress.a, macAddress->a, ETH_ALEN) ||
            !CsrMemCmp(element->sMac.a, macAddress->a, ETH_ALEN))
        {
            CSR_WIFI_HIP_LIST_DELETE_ENTRY(listHead);

            if (indicate)
            {
                ind = &element->signal.u.MaPacketIndication;

                if (ind->ReceptionStatus == CSR_RX_SUCCESS)
                {
                    receptionResult = CSR_RESULT_SUCCESS;
                }
                else
                {
                    receptionResult = CSR_RESULT_FAILURE;
                }

                CSR_LOG_TEXT_DEBUG((CsrWifiHalLto, CSR_WIFI_HAL_LTSO_PORT,
                                    "unifi%d: csrWifiHipCtrlPortQueueProcess: Forwarding frame for further processing\n",
                                    priv->instance));
                CsrWifiHipMaPacketInd(priv->osLayerContext,
                                      ((CsrWifiHipVifInstance *) element->vif)->interfaceTag,
                                      element->signal.SignalPrimitiveHeader.ReceiverProcessId,
                                      sizeof(CSR_SIGNAL),
                                      (CsrUint8 *) &element->signal,
                                      (CsrUint8 *) element->dMac.a,
                                      (CsrUint8 *) element->sMac.a,
                                      CSR_WIFI_HIP_FRAMETYPE_IEEE80211_DATA,
                                      element->macHeaderOffset,
                                      (CsrWifiHipBulkDataParam *) &element->bulkdata,
                                      receptionResult,
                                      ind->Rssi,
                                      ind->Snr,
                                      ind->ReceivedRate);

                /* Notify TA about the RX frame */
                timeStamp = (CsrTimeGet(NULL) / 1000);

                unifi_ta_sample_rx(priv->card, ((CsrWifiHipVifInstance *) element->vif)->interfaceTag,
                                   (const CsrWifiHipBulkDataDesc *) &element->bulkdata.d[0],
                                   (const CsrUint8 *) element->sMac.a,
                                   (const CsrUint8 *) element->dMac.a,
                                   timeStamp,
                                   ind->ReceivedRate);
            }

            if (portType == CSR_WIFI_HIP_PORT_TYPE_CONTROLLED)
            {
                vif->numOfControlledFrames--;
            }
            else
            {
                vif->numOfUncontrolledFrames--;
            }

            CsrMemFree(element);
        }
    }

    CSR_LOG_TEXT_DEBUG((CsrWifiHalLto, CSR_WIFI_HAL_LTSO_PORT,
                        "unifi%d: csrWifiHipCtrlPortQueueProcess: Buffered RX frames (port %d) has been processed\n",
                        priv ? priv->instance : 0, portType));
}

/* The wrapper ensures that locks are taken as required to protect the queue. */
CsrResult CsrWifiHipCtrlPortQueueProcessReq(void *hipHandle, CsrUint16 interfaceTag, CsrWifiHipPortType portType,
                                            CsrWifiMacAddress *macAddress, CsrBool indicate)
{
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;
    CsrWifiHipVifInstance *vif = NULL;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHalLto, CSR_WIFI_HAL_LTSO_PORT, "unifi%d: CsrWifiHipCtrlPortQueueProcessReq: Invalid hipHandle\n", 0));
        return CSR_RESULT_FAILURE;
    }

    /* Controlled port lists are only initialised when the BH is initialised */
    if (priv->wifiOnState != CSR_WIFI_HIP_WIFI_ON_DONE)
    {
        return CSR_RESULT_SUCCESS; /* nothing to do */
    }

    if (interfaceTag < CSR_WIFI_MAX_INTERFACES)
    {
        vif = (CsrWifiHipVifInstance *) &priv->vif[interfaceTag];
    }
    else
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHalLto, CSR_WIFI_HAL_LTSO_PORT,
                            "unifi%d: CsrWifiHipCtrlPortQueueProcessReq: Invalid interfaceTag: %u \n", priv->instance, interfaceTag));
        return CSR_RESULT_FAILURE;
    }

    (void) CsrMutexLock(&priv->configurationMutex);
    csrWifiHipCtrlPortQueueProcess(priv, vif, portType, macAddress, indicate);
    (void) CsrMutexUnlock(&priv->configurationMutex);

    return CSR_RESULT_SUCCESS;
}

/* It is a requirement that the caller ensures mutual exclusive access to
   shared data structures. */
void csrWifiHipCtrlPortQueueEntryAdd(CsrWifiHipHalPriv *priv, CsrWifiHipVifInstance *vif, CsrWifiHipPortType portType,
                                     CSR_SIGNAL *signal, CsrWifiHipBulkDataParam *bulkdata, CsrUint8 *dMac,
                                     CsrUint8 *sMac, CsrUint8 macHeaderOffset)
{
    /*lint -save -e550 */
    CsrWifiHipCtrlPortRxElement *element = NULL;
    CsrUint8 freeBulkDataCounter;
    CsrUint32 *counter;
    struct listHeader *queue;

    if (portType == CSR_WIFI_HIP_PORT_TYPE_CONTROLLED)
    {
        CSR_LOG_TEXT_DEBUG((CsrWifiHalLto, CSR_WIFI_HAL_LTSO_PORT,
                            "unifi%d: csrWifiHipCtrlPortQueueEntryAdd: Buffer RX frame received on the controlled port\n",
                            priv ? priv->instance : 0));

        if (vif->numOfControlledFrames < CSR_WIFI_HIP_QUEUE_CONTROLLED_MAX)
        {
            queue = &vif->controlledPortQueue;
            counter = &vif->numOfControlledFrames;
        }
        else
        {
            CSR_LOG_TEXT_INFO((CsrWifiHalLto, CSR_WIFI_HAL_LTSO_PORT,
                               "unifi%d: csrWifiHipCtrlPortQueueEntryAdd: Discarding RX frame received on the controlled port - the queue is full\n",
                               priv ? priv->instance : 0));
            for (freeBulkDataCounter = 0; freeBulkDataCounter < CSR_WIFI_HIP_MAX_DATA_REFERENCES; freeBulkDataCounter++)
            {
                priv->freeFrame(priv->osLayerContext, &bulkdata->d[freeBulkDataCounter]);
            }

            return;
        }
    }
    else
    {
        CSR_LOG_TEXT_DEBUG((CsrWifiHalLto, CSR_WIFI_HAL_LTSO_PORT,
                            "unifi%d: csrWifiHipCtrlPortQueueEntryAdd: Buffer RX frame received on the uncontrolled port\n",
                            priv ? priv->instance : 0));

        queue = &vif->uncontrolledPortQueue;
        counter = &vif->numOfUncontrolledFrames;

        if (vif->numOfUncontrolledFrames >= CSR_WIFI_HIP_QUEUE_UNCONTROLLED_MAX)
        {
            struct listHeader *listHead, *placeHolder;

            CSR_WIFI_HIP_LIST_LIST_FOR_EACH_SAFE(listHead, placeHolder, queue)
            {
                element = (CsrWifiHipCtrlPortRxElement *) CSR_WIFI_HIP_LIST_LIST_ENTRY(listHead, CsrWifiHipCtrlPortRxElement, header);
                CSR_WIFI_HIP_LIST_DELETE_ENTRY(listHead);

                vif->numOfUncontrolledFrames--;

                for (freeBulkDataCounter = 0; freeBulkDataCounter < CSR_WIFI_HIP_MAX_DATA_REFERENCES; freeBulkDataCounter++)
                {
                    priv->freeFrame(priv->osLayerContext, &element->bulkdata.d[freeBulkDataCounter]);
                }

                CsrMemFree(element);
            }
        }
    }

    element = CsrMemAlloc(sizeof(CsrWifiHipCtrlPortRxElement));
    CSR_WIFI_HIP_LIST_INIT(&element->header);

    element->bulkdata = *bulkdata;
    element->signal = *signal;
    element->vif = vif;
    CsrMemCpy(element->dMac.a, dMac, ETH_ALEN);
    CsrMemCpy(element->sMac.a, sMac, ETH_ALEN);
    element->macHeaderOffset = macHeaderOffset;

    CSR_WIFI_HIP_LIST_ADD_ENTRY(&element->header, queue);
    *counter = *counter + 1;

    CSR_LOG_TEXT_DEBUG((CsrWifiHalLto, CSR_WIFI_HAL_LTSO_PORT,
                        "unifi%d: csrWifiHipCtrlPortQueueEntryAdd: Frame buffered, dMac %02X:%02X:%02X:%02X:%02X:%02X, sMac %02X:%02X:%02X:%02X:%02X:%02X\n",
                        priv->instance, dMac[0], dMac[1], dMac[2], dMac[3], dMac[4], dMac[5], sMac[0], sMac[1], sMac[2], sMac[3], sMac[4], sMac[5]));
    /*lint -restore */
}

/*
 * ---------------------------------------------------------------------------
 * csrWifiHipPortConfigure
 *
 *      Store the new controlled port configuration.
 *
 * Arguments:
 *      priv            Pointer to device private context struct
 *      port_cfg        Pointer to the port configuration
 *
 * Returns:
 *      CSR result code
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPortConfigure(CsrWifiHipHalPriv *priv, CsrWifiHipVifInstance *vif,
                                  CsrWifiRouterCtrlPortAction port_action,
                                  CsrWifiMacAddress *macAddress, CsrWifiHipPortType queue)
{
    const CsrUint8 broadcast_mac_address[] = {0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF};
    unifi_port_config_t *port;
    CsrWifiHipStaPeerInfo *staRecord = NULL;
#ifdef CSR_LOG_ENABLE
    const CsrCharString *controlledString; /* cosmetic "controlled"/"uncontrolled" for trace */
#endif
    CsrUint8 i;

    if (queue == CSR_WIFI_HIP_PORT_TYPE_CONTROLLED)
    {
        port = &vif->staInfo.controlled_data_port;
#ifdef CSR_LOG_ENABLE
        controlledString = "controlled";
#endif
    }
    else
    {
        port = &vif->staInfo.uncontrolled_data_port;
#ifdef CSR_LOG_ENABLE
        controlledString = "uncontrolled";
#endif
    }


    /* If the new configuration has the broadcast MAC address or if we are in infrastructure mode then clear the list first and set port overide mode */
    if ((CSR_WIFI_ROUTER_CTRL_MODE_STA == vif->interfaceMode) ||
        (CSR_WIFI_ROUTER_CTRL_MODE_P2PCLI == vif->interfaceMode) ||
        (CSR_WIFI_ROUTER_CTRL_MODE_AMP == vif->interfaceMode) ||
        !CsrMemCmp(macAddress->a, broadcast_mac_address, ETH_ALEN))
    {
        (void) CsrMutexLock(&priv->configurationMutex);

        port->port_cfg[0].port_action = port_action;
        port->port_cfg[0].mac_address = *macAddress;
        port->port_cfg[0].in_use = TRUE;
        port->entries_in_use = 1;
        port->overide_action = UF_DATA_PORT_OVERIDE;

        /* Discard the remaining entries in the port config table */
        for (i = 1; i < CSR_WIFI_HIP_PEER_CONNECTIONS_MAX; i++)
        {
            port->port_cfg[i].in_use = FALSE;
        }
        (void) CsrMutexUnlock(&priv->configurationMutex);

        CSR_LOG_TEXT_INFO((
                              CsrWifiHalLto,  CSR_WIFI_HAL_LTSO_PORT,  "unifi%d: csrWifiHipPortConfigure: Override on %s port \n",
                              priv ? priv->instance : 0,
                              (queue == CSR_WIFI_HIP_PORT_TYPE_CONTROLLED) ? "Controlled" : "Uncontrolled"));

        if (port_action == CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_OPEN)
        {
            CSR_LOG_TEXT_INFO((CsrWifiHalLto, CSR_WIFI_HAL_LTSO_PORT, "unifi%d: csrWifiHipPortConfigure: %s port broadcast set to open.\n",
                               priv ? priv->instance : 0,
                               (queue == CSR_WIFI_HIP_PORT_TYPE_CONTROLLED) ? "Controlled" : "Uncontrolled"));

            if (vif->interfaceMode != CSR_WIFI_ROUTER_CTRL_MODE_NONE)
            {
                if (queue == CSR_WIFI_HIP_PORT_TYPE_CONTROLLED)
                {
                    CsrWifiHipPacketSchedulerQueueSetACQState(priv->card, vif->interfaceTag, 0, QS_ACQ_ENABLED);
                }
                else
                {
                    CsrWifiHipPacketSchedulerQueueSetAuthQState(priv->card, vif->interfaceTag, 0, QS_AUTHQ_ENABLED);
                }
            }
        }
        else
        {
            CSR_LOG_TEXT_INFO((CsrWifiHalLto, CSR_WIFI_HAL_LTSO_PORT, "unifi%d: csrWifiHipPortConfigure: %s port broadcast set to %s.\n",
                               priv ? priv->instance : 0,
                               controlledString,
                               (port_action == CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_CLOSED_DISCARD) ? "discard" : "closed"));

            if (vif->interfaceMode != CSR_WIFI_ROUTER_CTRL_MODE_NONE)
            {
                if (queue == CSR_WIFI_HIP_PORT_TYPE_CONTROLLED)
                {
                    if (port_action == CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_CLOSED_DISCARD)
                    {
                        CsrWifiHipPacketSchedulerQueueSetACQState(priv->card, vif->interfaceTag, 0, QS_ACQ_DISABLED);
                        CsrWifiHipPacketSchedulerQueueSetACQPurge(priv->card, vif->interfaceTag, 0);
                    }
                    else
                    {
                        CsrWifiHipPacketSchedulerQueueSetACQState(priv->card, vif->interfaceTag, 0, QS_ACQ_DISABLED);
                    }
                }
                else
                {
                    if (port_action == CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_CLOSED_DISCARD)
                    {
                        CsrWifiHipPacketSchedulerQueueSetAuthQState(priv->card, vif->interfaceTag, 0, QS_AUTHQ_DISABLED);
                        CsrWifiHipPacketSchedulerQueueSetAuthQPurge(priv->card, vif->interfaceTag, 0);
                    }
                    else
                    {
                        CsrWifiHipPacketSchedulerQueueSetAuthQState(priv->card, vif->interfaceTag, 0, QS_AUTHQ_DISABLED);
                    }
                }
            }
        }
    }
    else
    {
        /* store the new configuration, either in the entry with matching mac address (if already present),
         * otherwise in a new entry
         */

        CsrInt32 found_entry_flag;
        CsrInt32 first_free_slot = -1;

        CSR_LOG_TEXT_INFO((CsrWifiHalLto, CSR_WIFI_HAL_LTSO_PORT,
                           "unifi%d: csrWifiHipPortConfigure: port config %02x-%02x-%02x-%02x-%02x-%02x with port_action %d.\n",
                           priv ? priv->instance : 0,
                           *(macAddress->a + 0), *(macAddress->a + 1), *(macAddress->a + 2),
                           *(macAddress->a + 3), *(macAddress->a + 4), *(macAddress->a + 5),
                           port_action));

        /* Fetch the station record for corresponding peer mac address */
        staRecord = csrWifiHipGetStationRecordFromPeerMacAddress(priv, vif, macAddress->a);

        if (!staRecord)
        {
            CSR_LOG_TEXT_ERROR((CsrWifiHalLto,
                                CSR_WIFI_HAL_LTSO_PORT, "unifi%d: csrWifiHipPortConfigure: no staRecord found matching the MAC address\n",
                                priv ? priv->instance : 0));
            return CSR_RESULT_FAILURE;
        }

        (void) CsrMutexLock(&priv->configurationMutex);

        /* If leaving override mode, free the port entry used for override */
        if (port->overide_action == UF_DATA_PORT_OVERIDE)
        {
            port->port_cfg[0].in_use = FALSE;
            port->entries_in_use = 0;
            port->overide_action = UF_DATA_PORT_NOT_OVERIDE;

            CSR_LOG_TEXT_INFO((
                                  CsrWifiHalLto,  CSR_WIFI_HAL_LTSO_PORT,  "unifi%d: csrWifiHipPortConfigure: %s port override off\n",
                                  priv ? priv->instance : 0,  controlledString));
        }

        found_entry_flag = 0;
        for (i = 0; i < CSR_WIFI_HIP_PEER_CONNECTIONS_MAX; i++)
        {
            if (port->port_cfg[i].in_use)
            {
                if (!CsrMemCmp(port->port_cfg[i].mac_address.a, macAddress->a, ETH_ALEN))
                {
                    /* We've seen this address before, reconfigure it */
                    port->port_cfg[i].port_action = port_action;
                    found_entry_flag = 1;
                    break;
                }
            }
            else if (first_free_slot == -1)
            {
                /* Remember the first free slot on the way past so it can be claimed
                 * if this turns out to be a new MAC address (to save walking the list again).
                 */
                first_free_slot = i;
            }
        }

        /* At this point we found an existing entry and have updated it, or need to
         * add a new entry. If all slots are allocated, give up and return an error.
         */
        if (!found_entry_flag)
        {
            if (first_free_slot == -1)
            {
                CSR_LOG_TEXT_ERROR((CsrWifiHalLto,
                                    CSR_WIFI_HAL_LTSO_PORT, "unifi%d: csrWifiHipPortConfigure: no free slot found in port config array (%d used)\n",
                                    priv ? priv->instance : 0,  port->entries_in_use));
                (void) CsrMutexUnlock(&priv->configurationMutex);
                return CSR_RESULT_FAILURE;
            }
            else
            {
                port->entries_in_use++;
            }

            CSR_LOG_TEXT_DEBUG((
                                   CsrWifiHalLto,  CSR_WIFI_HAL_LTSO_PORT,  "unifi%d: csrWifiHipPortConfigure: port config index assigned = %d\n",
                                   priv ? priv->instance : 0,  first_free_slot));
            port->port_cfg[first_free_slot].in_use = TRUE;
            port->port_cfg[first_free_slot].port_action = port_action;
            port->port_cfg[first_free_slot].mac_address = *macAddress;
        }

        (void) CsrMutexUnlock(&priv->configurationMutex);

        if (port_action == CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_OPEN)
        {
            CSR_LOG_TEXT_INFO((CsrWifiHalLto, CSR_WIFI_HAL_LTSO_PORT,
                               "unifi%d: csrWifiHipPortConfigure: Non-STA queued frames for %s port has been forwarded\n",
                               priv ? priv->instance : 0,  controlledString));

            if (vif->interfaceMode != CSR_WIFI_ROUTER_CTRL_MODE_NONE)
            {
                if (queue == CSR_WIFI_HIP_PORT_TYPE_CONTROLLED)
                {
                    CsrWifiHipPacketSchedulerQueueSetACQState(priv->card, vif->interfaceTag, staRecord->associationId, QS_ACQ_ENABLED);
                }
                else
                {
                    CsrWifiHipPacketSchedulerQueueSetAuthQState(priv->card, vif->interfaceTag, staRecord->associationId, QS_AUTHQ_ENABLED);
                }
            }
        }
        else
        {
            if (port_action == CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_CLOSED_DISCARD)
            {
                CSR_LOG_TEXT_INFO((CsrWifiHalLto, CSR_WIFI_HAL_LTSO_PORT,
                                   "unifi%d: csrWifiHipPortConfigure: Non-STA queued frames for %s port has dropped \n",
                                   priv ? priv->instance : 0, controlledString));
            }
            if (vif->interfaceMode != CSR_WIFI_ROUTER_CTRL_MODE_NONE)
            {
                if (queue == CSR_WIFI_HIP_PORT_TYPE_CONTROLLED)
                {
                    if (port_action == CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_CLOSED_DISCARD)
                    {
                        CsrWifiHipPacketSchedulerQueueSetACQState(priv->card, vif->interfaceTag, staRecord->associationId, QS_ACQ_DISABLED);
                        CsrWifiHipPacketSchedulerQueueSetACQPurge(priv->card, vif->interfaceTag, staRecord->associationId);
                    }
                    else
                    {
                        CsrWifiHipPacketSchedulerQueueSetACQState(priv->card, vif->interfaceTag, staRecord->associationId, QS_ACQ_DISABLED);
                    }
                }
                else
                {
                    if (port_action == CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_CLOSED_DISCARD)
                    {
                        CsrWifiHipPacketSchedulerQueueSetACQState(priv->card, vif->interfaceTag, staRecord->associationId, QS_ACQ_DISABLED);
                        CsrWifiHipPacketSchedulerQueueSetAuthQPurge(priv->card, vif->interfaceTag, staRecord->associationId);
                    }
                    else
                    {
                        CsrWifiHipPacketSchedulerQueueSetAuthQState(priv->card, vif->interfaceTag, staRecord->associationId, QS_AUTHQ_DISABLED);
                    }
                }
            }
        }
    }
    return CSR_RESULT_SUCCESS;
} /* csrWifiHipPortConfigure() */

CsrResult CsrWifiHipPortConfigureReq(void *hipHandle, CsrUint16 interfaceTag,
                                     CsrWifiRouterCtrlPortAction uncontrolledPortAction,
                                     CsrWifiRouterCtrlPortAction controlledPortAction,
                                     CsrWifiMacAddress *macAddress, CsrBool setProtection)
{
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;
    CsrResult r, rr = CSR_RESULT_SUCCESS;
    CsrWifiHipVifInstance *vif = NULL;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHalLto, CSR_WIFI_HAL_LTSO_PORT, "unifi%d: CsrWifiHipPortConfigureReq: Failed - Invalid hipHandle\n", priv ? priv->instance : 0));
        return CSR_RESULT_FAILURE;
    }

    vif = (CsrWifiHipVifInstance *) &priv->vif[interfaceTag];

    (void) CsrMutexLock(&priv->configurationMutex);

    /* To update the protection status of the peer/station */
    switch (vif->interfaceMode)
    {
        /* Since for Unifi as a station, the station record not maintained & interfaceID is
         * only needed to update the peer protection status
         *
         * As all stations in an IBSS must use the same security settings record it against
         * current station rather than its peers.
         */
        case CSR_WIFI_ROUTER_CTRL_MODE_STA:
        case CSR_WIFI_ROUTER_CTRL_MODE_AMP:
        case CSR_WIFI_ROUTER_CTRL_MODE_P2PCLI:
        case CSR_WIFI_ROUTER_CTRL_MODE_IBSS:
            vif->staInfo.protect = setProtection;
            break;
        case CSR_WIFI_ROUTER_CTRL_MODE_AP:
        case CSR_WIFI_ROUTER_CTRL_MODE_P2PGO:
        {
#ifdef CSR_WIFI_AP_ENABLE
            CsrUint8 i;
            CsrWifiHipStaPeerInfo *staPeerRecord;

            /* If controlled port is open means, the peer has been added to station record
             * so that the protection corresponding to the peer is valid in this req
             */
            if (controlledPortAction == CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_OPEN)
            {
                for (i = 0; i < CSR_WIFI_HIP_PEER_CONNECTIONS_MAX; i++)
                {
                    staPeerRecord = (CsrWifiHipStaPeerInfo *) &(vif->staPeerInfo[i]);
                    if (staPeerRecord->entryInUse == TRUE)
                    {
                        /* Find the matching station record & set the protection type */
                        if (!CsrMemCmp(macAddress->a, staPeerRecord->peerMacAddress.a, ETH_ALEN))
                        {
                            staPeerRecord->protection = setProtection;
                            break;
                        }
                    }
                }
            }
#else
            CSR_LOG_TEXT_ERROR((CsrWifiHalLto, CSR_WIFI_HAL_LTSO_PORT,
                                "unifi%d: CsrWifiHipPortConfigureReq: Trying to process a CsrWifiHipPortConfigureReq on an interface configured for AP or P2PGO but AP code is not enabled \n",
                                priv ? priv->instance : 0));
#endif
            break;
        }

        default:
            CSR_LOG_TEXT_INFO((
                                  CsrWifiHalLto,  CSR_WIFI_HAL_LTSO_PORT,  "unifi%d: CsrWifiHipPortConfigureReq: Unknown mode\n"
                                  , priv ? priv->instance : 0));
    }
    (void) CsrMutexUnlock(&priv->configurationMutex);

    r = csrWifiHipPortConfigure(priv,
                                vif,
                                uncontrolledPortAction,
                                macAddress,
                                CSR_WIFI_HIP_PORT_TYPE_UNCONTROLLED);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_ERROR((CsrWifiHalLto,
                            CSR_WIFI_HAL_LTSO_PORT, "unifi%d: CsrWifiHipPortConfigureReq: Configuring the uncontrolled port failed, error code %u \n",
                            priv ? priv->instance : 0,  r));
        rr = CSR_RESULT_FAILURE;
    }

    r = csrWifiHipPortConfigure(priv,
                                vif,
                                controlledPortAction,
                                macAddress,
                                CSR_WIFI_HIP_PORT_TYPE_CONTROLLED);
    if ((r != CSR_RESULT_SUCCESS) || (rr != CSR_RESULT_SUCCESS))
    {
        if (r)
        {
            CSR_LOG_TEXT_ERROR((CsrWifiHalLto,
                                CSR_WIFI_HAL_LTSO_PORT, "unifi%d: CsrWifiHipPortConfigureReq: Configuring the controlled port failed, error code %d \n",
                                priv ? priv->instance : 0,  r));
        }

        if (rr)
        {
            CSR_LOG_TEXT_ERROR((CsrWifiHalLto,
                                CSR_WIFI_HAL_LTSO_PORT, "unifi%d: CsrWifiHipPortConfigureReq: Configuring the uncontrolled port failed, error code %d \n",
                                priv ? priv->instance : 0,  rr));
        }

        return CSR_RESULT_FAILURE;
    }

    CSR_LOG_TEXT_INFO((
                          CsrWifiHalLto,  CSR_WIFI_HAL_LTSO_PORT,  "unifi%d: CsrWifiHipPortConfigureReq: Controlled port set to %d. Uncontrolled port set to %d \n",
                          priv ? priv->instance : 0,
                          controlledPortAction, uncontrolledPortAction));

    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 * csrWifiHipControlledPortState
 *
 *      Return the state of the controlled port.
 *
 * Arguments:
 *      priv       Pointer to device private context struct
 *      address    Pointer to the destination for tx or sender for rx address
 *      queue      Controlled or uncontrolled queue
 *
 * Returns:
 *      An unifi_ControlledPortAction value.
 * ---------------------------------------------------------------------------
 */
CsrWifiRouterCtrlPortAction csrWifiHipControlledPortState(CsrWifiHipHalPriv *priv, CsrWifiHipVifInstance *vif, CsrUint8 *address, CsrWifiHipPortType queue)
{
    CsrUint16 i;
    unifi_port_config_t *port;

    if (queue == CSR_WIFI_HIP_PORT_TYPE_CONTROLLED)
    {
        port = &vif->staInfo.controlled_data_port;
    }
    else
    {
        port = &vif->staInfo.uncontrolled_data_port;
    }


    if (!port->entries_in_use)
    {
        CSR_LOG_TEXT_DEBUG((
                               CsrWifiHalLto,  CSR_WIFI_HAL_LTSO_PORT,  "unifi%d: csrWifiHipControlledPortState: No port configurations, return Discard.\n"
                               , priv ? priv->instance : 0));
        return CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_CLOSED_DISCARD;
    }

    /* If the port configuration is common for all destinations, return it. */
    if (port->overide_action == UF_DATA_PORT_OVERIDE)
    {
        CSR_LOG_TEXT_DEBUG((
                               CsrWifiHalLto,  CSR_WIFI_HAL_LTSO_PORT,  "unifi%d: csrWifiHipControlledPortState: Single port configuration (%u).\n",
                               priv ? priv->instance : 0,
                               port->port_cfg[0].port_action));
        return port->port_cfg[0].port_action;
    }

    CSR_LOG_TEXT_DEBUG((
                           CsrWifiHalLto,  CSR_WIFI_HAL_LTSO_PORT,  "unifi%d: csrWifiHipControlledPortState: Multiple (%d) port configurations.\n",
                           priv ? priv->instance : 0,  port->entries_in_use));

    /* If multiple configurations exist.. */
    for (i = 0; i < CSR_WIFI_HIP_PEER_CONNECTIONS_MAX; i++)
    {
        /* .. go through the list and match the destination address. */
        if ((port->port_cfg[i].in_use) &&
            (CsrMemCmp(address, port->port_cfg[i].mac_address.a, ETH_ALEN) == 0))
        {
            /* Return the desired action. */
            return port->port_cfg[i].port_action;
        }
    }

    /* Could not find any information, return Open. */
    CSR_LOG_TEXT_DEBUG((
                           CsrWifiHalLto,  CSR_WIFI_HAL_LTSO_PORT,  "unifi%d: csrWifiHipControlledPortState: port configuration not found, return Open.\n"
                           , priv ? priv->instance : 0));
    return CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_OPEN;
}

/*
 * ---------------------------------------------------------------------------
 * CsrWifiHipControlledPortHandle
 *
 *      Return the index handle corresponding to peer from port config array.
 *
 * Arguments:
 *      priv       Pointer to device private context struct
 *      address    Pointer to the destination for tx or sender for rx address
 *      queue      Controlled or uncontrolled queue
 *
 * Returns:
 *      An pointer to unifi_port_cfg_t
 * ---------------------------------------------------------------------------
 */
unifi_port_cfg_t *CsrWifiHipControlledPortHandle(CsrWifiHipHalPriv *priv, CsrWifiHipVifInstance *vif, CsrUint8 *address, CsrUint32 queue)
{
    CsrUint16 i;
    unifi_port_config_t *port;


    if (queue == CSR_WIFI_HIP_PORT_TYPE_CONTROLLED)
    {
        port = &vif->staInfo.controlled_data_port;
    }
    else
    {
        port = &vif->staInfo.uncontrolled_data_port;
    }


    if (!port->entries_in_use)
    {
        CSR_LOG_TEXT_DEBUG((
                               CsrWifiHalLto,  CSR_WIFI_HAL_LTSO_PORT,  "unifi%d: CsrWifiHipControlledPortHandle: No port configurations, return Discard.\n"
                               , priv ? priv->instance : 0));
        return NULL;
    }

    /* If the port configuration is common for all destinations, return it. */
    if (port->overide_action == UF_DATA_PORT_OVERIDE)
    {
        CSR_LOG_TEXT_DEBUG((
                               CsrWifiHalLto,  CSR_WIFI_HAL_LTSO_PORT,  "unifi%d: CsrWifiHipControlledPortHandle: Single port configuration (%u).\n",
                               priv ? priv->instance : 0,
                               port->port_cfg[0].port_action));
        return &port->port_cfg[0];
    }

    CSR_LOG_TEXT_DEBUG((
                           CsrWifiHalLto,  CSR_WIFI_HAL_LTSO_PORT,  "unifi%d: CsrWifiHipControlledPortHandle: Multiple port configurations.\n"
                           , priv ? priv->instance : 0));

    /* If multiple configurations exist.. */
    for (i = 0; i < CSR_WIFI_HIP_PEER_CONNECTIONS_MAX; i++)
    {
        /* .. go through the list and match the destination address. */
        if ((port->port_cfg[i].in_use) &&
            (CsrMemCmp(address, port->port_cfg[i].mac_address.a, ETH_ALEN) == 0))
        {
            /* Return the desired action. */
            return &port->port_cfg[i];
        }
    }

    /* Could not find any information, NULL. */
    CSR_LOG_TEXT_DEBUG((
                           CsrWifiHalLto,  CSR_WIFI_HAL_LTSO_PORT,  "unifi%d: CsrWifiHipControlledPortHandle: port configuration not found, return NULL (debug).\n"
                           , priv ? priv->instance : 0));
    return NULL;
}
