/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"

#include "csr_wifi_hip_log_text.h"
#include "csr_types.h"
#include "csr_result.h"
#include "csr_wifi_hip_hal_priv.h"
#include "csr_wifi_router_ctrl_lib.h"
#include "csr_wifi_hip_list.h"
#include "csr_wifi_hip_conversions.h"
#include "csr_wifi_hip_util.h"
#include "csr_wifi_hip_unifi_udi.h"
#include "csr_wifi_ps_if.h"


void csrWifiHipLogUdiProxyHook(void *ospriv, CsrUint8 *sigdata, CsrUint32 signal_len,
                               const CsrWifiHipBulkDataParam *bulkdata, CsrWifiHipLogUdiDirection direction)
{
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) ospriv;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "csrWifiHipLogUdiProxyHook: Invalid priv \n"));
        return;
    }

    if (priv->udiProxyHook != NULL)
    {
        priv->udiProxyHook(priv->osLayerContext, sigdata, signal_len, bulkdata, direction);
    }
    else
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: csrWifiHipLogUdiProxyHook: No UDI function registered \n", priv ? priv->instance : 0));
    }
}

CsrResult CsrWifiHipLogUdiRegistrationReq(void *hipHandle, CsrWifiHipLogUdiFunc logFunction)
{
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;
    CsrResult result;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "CsrWifiHipLogUdiRegistrationReq: Invalid priv \n"));
        return CSR_RESULT_FAILURE;
    }

    if (logFunction == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "CsrWifiHipLogUdiRegistrationReq: Invalid UDI function \n"));
        return CSR_RESULT_FAILURE;
    }

    result = unifi_set_udi_hook(priv->card, csrWifiHipLogUdiProxyHook);
    if (result != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "CsrWifiHipLogUdiRegistrationReq: Failed to register UDI log hook in lib_hip\n"));
        return CSR_RESULT_FAILURE;
    }

    priv->udiProxyHook = (udi_func_t) logFunction;
    return result;
}

CsrResult CsrWifiHipLogUdiUnregistrationReq(void *hipHandle)
{
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;
    CsrResult result;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "CsrWifiHipLogUdiUnregistrationReq: Invalid priv \n"));
        return CSR_RESULT_FAILURE;
    }

    if (priv->udiProxyHook == NULL)
    {
        CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                              "unifi : CsrWifiHipLogUdiUnregistrationReq: UDI log proxy hook was not registered. Nothing to unregister.\n"));
        return CSR_RESULT_SUCCESS;
    }

    result = unifi_remove_udi_hook(priv->card, csrWifiHipLogUdiProxyHook);
    priv->udiProxyHook = NULL;
    return result;
}

CsrInt32 CsrWifiHipDriverInfoGetReq(void *hipHandle, CsrCharString *data, CsrInt32 *remaining, CsrWifiHipDriverInfoDebugInfo *debugInfo)
{
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "CsrWifiHipDriverInfoGetReq: Invalid priv \n"));
        return 0;
    }

    if (debugInfo == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "CsrWifiHipDriverInfoGetReq: Received a NULL debugInfo pointer\n"));
        return 0;
    }

    CsrMemCpy(debugInfo->last_debug_string, priv->debugInfo.last_debug_string, CSR_WIFI_HIP_LAST_DEBUG_STRING_LENGTH);
    CsrMemCpy(debugInfo->last_debug_word16, priv->debugInfo.last_debug_word16, CSR_WIFI_HIP_LAST_DEBUG_WORD16_LENGTH);
    return unifi_print_status(priv->card, data, remaining);
}

void CsrWifiHipCapabilitiesReq(void *hipHandle, CsrSchedQid appHandle, CsrUint16 clientData)
{
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;
    CsrInt32 bufferSize;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi: CsrWifiHipCapabilitiesReq: Invalid priv \n"));

        CsrWifiRouterCtrlCapabilitiesCfmSend(appHandle,
                                             clientData,
                                             0,
                                             0);
        return;
    }

    bufferSize = CsrWifiHipPacketSchedulerCapabilitiesQueueLen(priv->card);
    if (bufferSize <= 0)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi: CsrWifiHipCapabilitiesReq: call to Packet Scheduler failed \n"));

        CsrWifiRouterCtrlCapabilitiesCfmSend(appHandle,
                                             clientData,
                                             0,
                                             0);
        return;
    }

    CsrWifiRouterCtrlCapabilitiesCfmSend(appHandle,
                                         clientData,
                                         bufferSize,
                                         bufferSize);

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG5,  "unifi%d: CsrWifiHipCapabilitiesReq was successful \n", priv->instance));
}

static CsrBool calculateRelativeQueuePriorities(CsrWifiRouterCtrlQueueSettings queueConfigs[], CsrInt32 relativePriorities[])
{
    CsrInt32 ac, swapped;
    CsrInt32 queueConfigMap[4];
    CsrWifiRouterCtrlQueueSettings *queueConfigUpper, *queueConfigLower;
    CsrBool rc = FALSE;
    /*
     * queueConfig contains the AC EDCA parameters in the standard order: BE, BK, VI, VO.
     * For technical reasons. the driver has definitions that are in the order: BK, BE, VI, VO.
     * queueConfigMap is indexed in the driver order and contains the standard index for accessing queueConfigs.
     */
    queueConfigMap[UNIFI_TRAFFIC_Q_BK] = 1;
    queueConfigMap[UNIFI_TRAFFIC_Q_BE] = 0;
    queueConfigMap[UNIFI_TRAFFIC_Q_VI] = 2;
    queueConfigMap[UNIFI_TRAFFIC_Q_VO] = 3;

    /*
     * Initialise priority map to default values.
     */
    relativePriorities[UNIFI_TRAFFIC_Q_BK] = UNIFI_TRAFFIC_Q_BK;
    relativePriorities[UNIFI_TRAFFIC_Q_BE] = UNIFI_TRAFFIC_Q_BE;
    relativePriorities[UNIFI_TRAFFIC_Q_VI] = UNIFI_TRAFFIC_Q_VI;
    relativePriorities[UNIFI_TRAFFIC_Q_VO] = UNIFI_TRAFFIC_Q_VO;

    /*
     * Sort the queues based on their EDCA parameters.
     */
    do
    {
        swapped = 0;
        for (ac = 3; ac > 0; ac--)
        {
            CsrInt32 qcIndexUpper, qcIndexLower;

            qcIndexUpper = relativePriorities[ac];
            qcIndexLower = relativePriorities[ac - 1];

            queueConfigUpper = &queueConfigs[queueConfigMap[qcIndexUpper]];
            queueConfigLower = &queueConfigs[queueConfigMap[qcIndexLower]];
            if (queueConfigUpper->aifsn > queueConfigLower->aifsn)
            {
                relativePriorities[ac] = qcIndexLower;
                relativePriorities[ac - 1] = qcIndexUpper;
                swapped = 1;
            }
            else if (queueConfigUpper->aifsn == queueConfigLower->aifsn)
            {
                if (queueConfigUpper->cWmin > queueConfigLower->cWmin)
                {
                    relativePriorities[ac] = qcIndexLower;
                    relativePriorities[ac - 1] = qcIndexUpper;
                    swapped = 1;
                }
            }
        }
        if (swapped)
        {
            rc = TRUE;
        }
    } while (swapped);

    for (ac = 0; ac < 4; ac++)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "CsrWifiHipQosControlReq: Token bucket=%d aifsn=%d cWmin=%d cWmax=%d txOpLimit=%d PSQ=%d\n",
                               ac, queueConfigs[queueConfigMap[relativePriorities[ac]]].aifsn,
                               queueConfigs[queueConfigMap[relativePriorities[ac]]].cWmin,
                               queueConfigs[queueConfigMap[relativePriorities[ac]]].cWmax,
                               queueConfigs[queueConfigMap[relativePriorities[ac]]].txopLimit, relativePriorities[ac]));
    }
    return rc;
}

void CsrWifiHipQosControlReq(void *hipHandle, CsrUint16 interfaceTag, CsrWifiRouterCtrlQoSControl mode, CsrWifiRouterCtrlQueueConfigMask queueConfig, CsrWifiRouterCtrlQueueSettings queueConfigs[])
{
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;
    CsrWifiHipVifInstance *vif = NULL;
    CsrInt32 relativePriorities[4];

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipQosControlReq: Invalid priv \n", priv ? priv->instance : 0));
        return;
    }

    if (interfaceTag < CSR_WIFI_MAX_INTERFACES)
    {
        vif = (CsrWifiHipVifInstance *) &priv->vif[interfaceTag];
    }
    else
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipQosControlReq: Invalid interfaceTag \n", priv ? priv->instance : 0));
        return;
    }


    if (mode == CSR_WIFI_ROUTER_CTRL_QOS_CONTROL_WMM_ON)
    {
        vif->staInfo.wmmControl |= CSR_WIFI_HIP_QOS_CAPABILITY_WMM_ENABLED;
        CSR_LOG_TEXT_INFO((
                              CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: CsrWifiHipQosControlReq: WMM enabled, queue config %x\n",
                              priv ? priv->instance : 0,  queueConfig));

        vif->staInfo.queueEnabled[UNIFI_TRAFFIC_Q_BK] = (queueConfig & CSR_WIFI_ROUTER_CTRL_QUEUE_BK_ENABLE) ? 1 : 0;
        vif->staInfo.queueEnabled[UNIFI_TRAFFIC_Q_BE] = (queueConfig & CSR_WIFI_ROUTER_CTRL_QUEUE_BE_ENABLE) ? 1 : 0;
        vif->staInfo.queueEnabled[UNIFI_TRAFFIC_Q_VI] = (queueConfig & CSR_WIFI_ROUTER_CTRL_QUEUE_VI_ENABLE) ? 1 : 0;
        vif->staInfo.queueEnabled[UNIFI_TRAFFIC_Q_VO] = (queueConfig & CSR_WIFI_ROUTER_CTRL_QUEUE_VO_ENABLE) ? 1 : 0;

        if (calculateRelativeQueuePriorities(queueConfigs, relativePriorities) == TRUE)
        {
            if (CsrWifiHipPacketSchedulerVIFACRelativePriorities(priv->card, interfaceTag, relativePriorities) == CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipQosControlReq: PSched queues relative priority\n", priv->instance));
            }
        }
    }
    else
    {
        vif->staInfo.wmmControl = 0;
        CSR_LOG_TEXT_INFO((
                              CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: CsrWifiHipQosControlReq: WMM disabled\n"
                              , priv ? priv->instance : 0));
    }
}

void CsrWifiHipSetMacReq(void *hipHandle, CsrWifiMacAddress *macAddresses, CsrUint16 interfaceTag)
{
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipSetMacReq: Invalid hipHandle\n", priv ? priv->instance : 0));
        return;
    }

    CsrMemCpy(priv->vif[interfaceTag].staInfo.sta_mac_addresses.a, macAddresses, sizeof(CsrWifiMacAddress));
    priv->vif[interfaceTag].interfaceTag = interfaceTag;
}

/******************* Auto confirms from HAL for un-delivered packets *******************/
#ifdef CSR_WIFI_AP_ENABLE
void csrWifiHipAutoConfirmSend(void *ospriv, CsrInt32 interfaceTag, struct card_signal *signal)
{
    CsrWifiHipVifInstance *vif = NULL;
    CSR_SIGNAL sigPtr;
    const CSR_MA_PACKET_REQUEST *req;
    CSR_SIGNAL unpackedSignal;
    CsrUint8 packedSigBuf[UNIFI_PACKED_SIGBUF_SIZE];
    CsrUint16 packedSigLen;
    CsrWifiHipBulkDataParam dummyBulkData;
    CsrResult r;
    CsrUint8 i;

    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) ospriv;

    if (!priv)
    {
        CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                              "unifi : csrWifiHipAutoConfirmSend: Invalid ospriv\n"));
        return;
    }

    if (interfaceTag < CSR_WIFI_MAX_INTERFACES)
    {
        vif = (CsrWifiHipVifInstance *) &priv->vif[interfaceTag];
    }
    else
    {
        CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID,
                              CSR_WIFI_HIP_LOG_DEF, "unifi%d: csrWifiHipAutoConfirmSend: Invalid interfaceTag: %u\n",
                              priv ? priv->instance : 0,  interfaceTag));
        return;
    }

    if ((vif->interfaceMode != CSR_WIFI_ROUTER_CTRL_MODE_NONE) &&
        (priv->wifiOnState == CSR_WIFI_HIP_WIFI_ON_DONE))
    {
        /* Unpack the signal and check if it is a MA-Packet.request */
        r = read_unpack_signal(signal->sigbuf, &sigPtr);
        if (r)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: csrWifiHipAutoConfirmSend: Unpack failed (%u)\n",
                                   priv ? priv->instance : 0,  r));
            return;
        }

        if (sigPtr.SignalPrimitiveHeader.SignalId == CSR_MA_PACKET_REQUEST_ID)
        {
            /* Check if a confirm is requested. HAL can not check who the sender is, but MaPacketCfms should not be
               generated for MaPacketReq originating from the network stack. Should a MaPacketCfm be forwarded to
               the network stack, the platform specific code (which is aware of the origin) will discard the
               MaPacketCfm */
            req = &sigPtr.u.MaPacketRequest;
            if (!(req->TransmissionControlBitmap & CSR_NO_CONFIRM_REQUIRED))
            {
                CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                      "unifi%d: csrWifiHipAutoConfirmSend: Send Auto confirm (tag=%x vif=%x sender=%x hostTag=%x txControl=%x)\n",
                                      priv ? priv->instance : 0,
                                      interfaceTag,
                                      CsrWifiHipVifIndexGetReq(priv, interfaceTag),
                                      sigPtr.SignalPrimitiveHeader.SenderProcessId,
                                      req->HostTag,
                                      req->TransmissionControlBitmap));

                /* Construct auto MA-PACKET.confirm signal */
                CsrMemSet(&unpackedSignal, 0, sizeof(unpackedSignal));
                unpackedSignal.SignalPrimitiveHeader.SignalId = CSR_MA_PACKET_CONFIRM_ID;
                unpackedSignal.SignalPrimitiveHeader.ReceiverProcessId = sigPtr.SignalPrimitiveHeader.SenderProcessId;
                unpackedSignal.SignalPrimitiveHeader.SenderProcessId = CSR_WIFI_ROUTER_IFACEQUEUE;
                unpackedSignal.u.MaPacketConfirm.VirtualInterfaceIndex = CsrWifiHipVifIndexGetReq(priv, interfaceTag);
                unpackedSignal.u.MaPacketConfirm.TransmissionStatus = CSR_TX_NO_BSS;
                unpackedSignal.u.MaPacketConfirm.HostTag = req->HostTag;

                (void) write_pack(&unpackedSignal, packedSigBuf, &packedSigLen);

                /* Construct dummy bulk data as required by the HAL API */
                for (i = 0; i < CSR_WIFI_HIP_MAX_DATA_REFERENCES; i++)
                {
                    dummyBulkData.d[i].os_data_ptr = NULL;
                    dummyBulkData.d[i].data_length = 0;
                    dummyBulkData.d[i].net_buf_length = 0;
                    dummyBulkData.d[i].os_net_buf_ptr = NULL;
                }

                CsrWifiHipMaPacketCfm(priv->osLayerContext,
                                      interfaceTag,
                                      unpackedSignal.SignalPrimitiveHeader.ReceiverProcessId,
                                      CSR_RESULT_FAILURE,
                                      0,
                                      unpackedSignal.u.MaPacketConfirm.HostTag,
                                      packedSigLen,
                                      packedSigBuf,
                                      &dummyBulkData);
            }
        }
    }
}

#endif


static void resetStationRecordList(CsrWifiHipHalPriv *priv, CsrWifiHipVifInstance *vif)
{
    CsrUint8 i;
    CsrWifiHipStaPeerInfo *staPeerInfo = NULL;
    CsrResult result = CSR_RESULT_SUCCESS;

    CSR_WIFI_HIP_SPINLOCK_LOCK(&priv->peerInfoLock);
    /* Interface independent, no of packet queued, in case of mode is None or AP set to 0 */
    for (i = 0; i < CSR_WIFI_HIP_PEER_CONNECTIONS_MAX; i++)
    {
        staPeerInfo = &vif->staPeerInfo[i];
        if (staPeerInfo->entryInUse == TRUE)
        {
            result = CsrWifiHipPacketSchedulerQueueSetDisassociate(priv->card, vif->interfaceTag, staPeerInfo->associationId);
            if (CSR_RESULT_SUCCESS != result)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                       "unifi%d: resetStationRecordList: Failed to Disassociate QueueSet (VIF %d AID %d) result = %u\n",
                                       priv ? priv->instance : 0,
                                       vif->interfaceTag,
                                       staPeerInfo->associationId,
                                       result));
            }

            /* resets mac address, in_use flags */
            CsrMemSet(staPeerInfo->peerControlledPort, 0, sizeof(unifi_port_cfg_t));
            CsrMemSet(staPeerInfo->peerUnControlledPort, 0, sizeof(unifi_port_cfg_t));

            staPeerInfo->peerControlledPort->port_action =
                CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_CLOSED_DISCARD;
            staPeerInfo->peerUnControlledPort->port_action =
                CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_CLOSED_DISCARD;

            vif->staInfo.controlled_data_port.entries_in_use--;
            vif->staInfo.uncontrolled_data_port.entries_in_use--;

            csrWifiHipApStaRecordInitialise(priv, staPeerInfo);
        }
    }
    CSR_WIFI_HIP_SPINLOCK_UNLOCK(&priv->peerInfoLock);

    if (((vif->staInfo.controlled_data_port.entries_in_use != 0) || (vif->staInfo.uncontrolled_data_port.entries_in_use != 0))
        && (vif->interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_NONE))
    {
        /* Print in case if the value of entries goes to -ve/+ve (apart from 0)
         * we expect the entries should be zero here if mode is set as NONE
         */
        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG3,  "unifi%d: resetStationRecordList: controlled port entries = %d, uncontrolled port entries = %d\n",
                               priv ? priv->instance : 0,
                               vif->staInfo.controlled_data_port.entries_in_use,
                               vif->staInfo.uncontrolled_data_port.entries_in_use));
    }
}

void csrWifiHipResetVif(CsrWifiHipHalPriv *priv, CsrWifiHipVifInstance *vif, CsrBool flushControlledPortQueues)
{
    CsrUint8 baSessionIdx = 0;
    CsrWifiHipBaSessionTx *baSessionTx = NULL;
    CsrWifiHipBaSessionRx *baSessionRx = NULL;
    CsrResult r;

    /* StaInfo init */
    vif->staInfo.wmmControl = 0;
    CsrMemSet(vif->staInfo.bssid.a, 0, ETH_ALEN);
    vif->staInfo.protect = FALSE;

#if defined(CSR_WIFI_SECURITY_WAPI_ENABLE)
    vif->staInfo.wapi_multicast_mic_filter = FALSE;
    vif->staInfo.wapi_unicast_mic_filter = FALSE;
#endif

    /* Enable all queues by default */
    vif->staInfo.queueEnabled[0] = 1;
    vif->staInfo.queueEnabled[1] = 1;
    vif->staInfo.queueEnabled[2] = 1;
    vif->staInfo.queueEnabled[3] = 1;

    if (flushControlledPortQueues)
    {
        csrWifiHipCtrlPortQueueFreeAll(priv, vif);
    }

    /* Reset the station record to NULL if mode is tried to set as NONE */
    switch (vif->interfaceMode)
    {
        case CSR_WIFI_ROUTER_CTRL_MODE_STA:
        case CSR_WIFI_ROUTER_CTRL_MODE_P2PCLI:
        case CSR_WIFI_ROUTER_CTRL_MODE_AMP:

            /* Station records not available in these modes */
            r = CsrWifiHipPacketSchedulerQueueSetDisassociate(priv->card, vif->interfaceTag, 0);
            if (r != CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: csrWifiHipResetVif: Disassociate queue set failed, interfaceTag %u, aId %u, error %u\n",
                                       priv ? priv->instance : 0, vif->interfaceTag, 0, r));
            }

            break;
        default:
            resetStationRecordList(priv, vif);
    }

    /* Remove all the Peer database, before going down */
    (void) CsrMutexLock(&priv->configurationMutex);
    for (baSessionIdx = 0; baSessionIdx < CSR_WIFI_HIP_BA_SUPPORTED_SESSIONS_RX_MAX; baSessionIdx++)
    {
        baSessionRx = vif->baSessionRx[baSessionIdx];
        if (baSessionRx)
        {
            r = csrWifiHipBaSessionStop(priv,
                                        vif,
                                        CSR_WIFI_ROUTER_CTRL_BLOCK_ACK_RECIPIENT,
                                        baSessionRx->trafficStreamId,
                                        baSessionRx->macAddress);
            if (r != CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: csrWifiHipResetVif: Failed to delete RX block ACK session (%u)",
                                       priv ? priv->instance : 0,  baSessionIdx));
            }
        }
    }

    for (baSessionIdx = 0; baSessionIdx < CSR_WIFI_HIP_BA_SUPPORTED_SESSIONS_TX_MAX; baSessionIdx++)
    {
        baSessionTx = vif->baSessionTx[baSessionIdx];
        if (baSessionIdx)
        {
            r = csrWifiHipBaSessionStop(priv,
                                        vif,
                                        CSR_WIFI_ROUTER_CTRL_BLOCK_ACK_ORIGINATOR,
                                        baSessionRx->trafficStreamId,
                                        baSessionTx->macAddress);
            if (r != CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: csrWifiHipResetVif: Failed to delete TX block ACK session (%u)",
                                       priv ? priv->instance : 0,  baSessionIdx));
            }
        }
    }

#ifdef CSR_WIFI_AP_ENABLE
    vif->numPeers = 0;
    /* Clear if a timer is running for inactivity detection */
    if (vif->inactiveDetectEnabled)
    {
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                           "unifi%d: csrWifiHipResetVif: Stopping the inactivity detection timer\n",
                           priv ? priv->instance : 0));

        vif->inactiveDetectEnabled = FALSE;
        CsrTimerDestroy(&vif->inActivityTimerHandle);
    }
#endif

    (void) CsrMutexUnlock(&priv->configurationMutex);
}

void CsrWifiHipVifModeSetReq(void *hipHandle, CsrUint16 interfaceTag,
                             CsrWifiRouterCtrlMode mode, CsrWifiMacAddress macAddress,
                             CsrBool protect, CsrBool intraBssDistEnabled, CsrUint16 vifIndex)
{
    CsrWifiHipHalPriv *priv = NULL;
    CsrWifiHipVifInstance *vif = NULL;
    CsrUint8 nullMacAddr[ETH_ALEN] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
    CsrResult result;

    if (hipHandle == NULL)
    {
        return;
    }

    priv = (CsrWifiHipHalPriv *) hipHandle;

    if (interfaceTag >= CSR_WIFI_MAX_INTERFACES)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d:  CsrWifiHipVifModeSetReq: Invalid interfaceTag (%u)\n", priv->instance, interfaceTag));
        return;
    }

    vif = (CsrWifiHipVifInstance *) &priv->vif[interfaceTag];
    if ((vif->interfaceMode != CSR_WIFI_ROUTER_CTRL_MODE_NONE) && mode)
    {
        /* Without setting mode NONE, trying to set new value */
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiHipVifModeSetReq: prevmode=%x, newmode=%x",
                               priv ? priv->instance : 0, vif->interfaceMode, mode));
        return;
    }

    csrWifiHipResetVif(priv, vif, FALSE);

#ifdef CSR_WIFI_AP_ENABLE
    /* For modes other than AP/P2PGO, set below member FALSE */
    vif->intraBssEnabled = FALSE;
#endif
    vif->interfaceMode = mode;

    if (mode != CSR_WIFI_ROUTER_CTRL_MODE_NONE)
    {
        /* update the interfaceTag & vifIndex mapping database */
        result = csrWifiHipUpdateInterfaceTagAndVifIndexMap(priv, interfaceTag, vifIndex,
                                                            TRUE, CSR_WIFI_HIP_VIFINDEX_ADD,
                                                            0);
        if (result != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi%d: CsrWifiHipVifModeSetReq: vifIndex mapping failed\n",
                                   priv->instance));
        }
        CsrMemCpy(vif->staInfo.bssid.a, macAddress.a, ETH_ALEN);
    }
    else
    {
        /* Mode is none, expecting InterfaceDel Req, so Invalidate the
         * InterfaceTag & VifIndex mapping here
         */
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3,
                            "unifi%d: CsrWifiHipVifModeSetReq: Flushing VIF tables, interfacetag %u\n",
                            priv ? priv->instance : 0, interfaceTag));

        result = csrWifiHipUpdateInterfaceTagAndVifIndexMap(priv, interfaceTag, 0,
                                                            FALSE, CSR_WIFI_HIP_VIFINDEX_FLUSH_IFACE_DB,
                                                            0);
        if (result != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi%d: CsrWifiHipVifModeSetReq: vifIndex mapping failed\n",
                                   priv->instance));
        }
        CsrMemCpy(vif->staInfo.bssid.a, nullMacAddr, ETH_ALEN);
    }


    if ((mode == CSR_WIFI_ROUTER_CTRL_MODE_AP) ||
        (mode == CSR_WIFI_ROUTER_CTRL_MODE_P2PGO))
    {
        vif->staInfo.protect = protect;
#ifdef CSR_WIFI_AP_ENABLE
        /* For AP/P2PGO mode SME sending intraBssDistEnabled
         * i.e. for AP: intraBssDistEnabled = TRUE, for P2PGO
         * intraBssDistEnabled = TRUE/FALSE on requirement
         */
        vif->intraBssEnabled = intraBssDistEnabled;

        result = CsrWifiHipPacketSchedulerVIFAssociate(priv->card, interfaceTag, (CsrWifiHipPacketSchedulerDiscardCb) csrWifiHipAutoConfirmSend);
        if (result != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipVifModeSetReq: Failed to associate VIF (VIF tag %u error %u)\n", priv->instance, interfaceTag, result));
        }
        else
        {
            #ifdef CSR_WIFI_AP_USE_PSCHED_MCAST_BURST_MODE
            CsrInt32 enableFlag = 1;
            #else
            CsrInt32 enableFlag = 0;
            #endif
            /*
             * Configure the multicast queue burst mode: 0 = disable, 1 enable.
             * If multicast burst mode is disabled, multicast traffic will be forwarded to the chip immediately. This
             * will give the highest multicast throughput at the cost of consuming buffer space in the chip between
             * DTIM. If burst mode is enabled, multicast traffic is held in the host until a DTIM trigger signal is
             * received, at which point a number of multicast frames (a burst) are forwarded to the chip.
             */
            if (CsrWifiHipPacketSchedulerQueueMulticastBurstEnable(priv->card, interfaceTag, enableFlag) != CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                       "unifi%d: CsrWifiHipVifModeSetReq: Failed to set Multicast BurstMax mode (VIF tag %u)\n", priv->instance, interfaceTag));
            }
        }
#endif

        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5,
                            "unifi%d: CsrWifiHipVifModeSetReq: Settings installed\n",
                            priv->instance));
    }
    else if ((mode == CSR_WIFI_ROUTER_CTRL_MODE_STA) ||
             (mode == CSR_WIFI_ROUTER_CTRL_MODE_P2P) ||
             (mode == CSR_WIFI_ROUTER_CTRL_MODE_AMP) ||
             (mode == CSR_WIFI_ROUTER_CTRL_MODE_P2PCLI))
    {
        result = CsrWifiHipPacketSchedulerVIFAssociate(priv->card, interfaceTag, (CsrWifiHipPacketSchedulerDiscardCb) NULL);
        if (result != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipVifModeSetReq: Failed to associate VIF (VIF tag %u error %u)\n", priv->instance, interfaceTag, result));
        }

        if (CsrWifiHipPacketSchedulerQueueSetAssociate(priv->card, interfaceTag, 0, QS_PRI_NORMAL, 0, priv->pauseCb, priv->resumeCb, priv->osLayerContext) == CSR_RESULT_SUCCESS)
        {
            if (CsrWifiHipPacketSchedulerQueueSetActivate(priv->card, interfaceTag, 0) != CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                       "unifi%d: CsrWifiHipVifModeSetReq: Failed to set SMod (VIF tag %u)\n",
                                       priv->instance, interfaceTag));
            }
        }
        else
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi%d: CsrWifiHipVifModeSetReq: Failed to set packet scheduler queues (VIF tag %u error 0x%02X)\n",
                                   priv->instance, interfaceTag, result));
        }
    }
    else if (mode == CSR_WIFI_ROUTER_CTRL_MODE_NONE)
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5, "unifi%d: CsrWifiHipVifModeSetReq: None mode",
                            priv ? priv->instance : 0));

        /* Disassociate VIF in Packet Scheduler */
        result = CsrWifiHipPacketSchedulerVIFDisassociate(priv->card, vif->interfaceTag);
        if (result != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi%d: CsrWifiHipVifModeSetReq: Failed to disassociate VIF in packet scheduler 0x%02X\n",
                                   priv->instance, result));
        }
    }
    else
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5,
                            "unifi%d: CsrWifiHipVifModeSetReq: Settings installed. No further actions.",
                            priv ? priv->instance : 0));
    }
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPeerAssociateQueueSetReq
 *
 *      Helper to associate a peer's AID with a packet scheduler Queue Set
 *
 *  Arguments:
 *      priv            - instance data
 *      interfaceTag    - interface we are dealing with
 *      associationId   - AID to connect to the interface's queue set
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPeerAssociateQueueSetReq(CsrWifiHipHalPriv *priv,
                                             CsrUint16          interfaceTag,
                                             CsrUint16          associationId)
{
    CsrResult r;

    r = CsrWifiHipPacketSchedulerQueueSetAssociate(priv->card, interfaceTag, associationId, QS_PRI_NORMAL, 0, priv->pauseCb, priv->resumeCb, priv->osLayerContext);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: csrWifiHipPeerAssociateQueueSetReq: Failed to associate queue set (VIF tag %u aId %u error %u)\n",
                               priv->instance, interfaceTag, associationId, r));
        return r;
    }

    r = CsrWifiHipPacketSchedulerQueueSetActivate(priv->card, interfaceTag, associationId);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: csrWifiHipPeerAssociateQueueSetReq: Failed to set SMod (VIF tag %u aId %u error %u)\n",
                               priv->instance, interfaceTag, associationId, r));
    }
    return r;
}

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipPeerAddReq
 *
 *      Add a peer to the list of peers.
 *
 *  Arguments:
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
CsrResult CsrWifiHipPeerAddReq(void *hipHandle, CsrUint16 interfaceTag,
                               CsrWifiMacAddress peerMacAddress, CsrUint16 associationId,
                               CsrWifiRouterCtrlStaInfo *staInfo, CsrUint32 *peerHandle)
{
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;
#ifdef CSR_WIFI_AP_ENABLE
    CsrUint8 powerModeTemp = 0;
#endif
    CsrResult r;
    CsrUint8 i;
    CsrBool freeSlotFound = FALSE;
    CsrWifiHipVifInstance *vif = NULL;
    *peerHandle = 0;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipPeerAddReq: Invalid hipHandle", priv ? priv->instance : 0));
        return CSR_RESULT_FAILURE;
    }

    if (interfaceTag >= CSR_WIFI_MAX_INTERFACES)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipPeerAddReq: Invalid interfaceTag (%u)\n",
                               priv ? priv->instance : 0,  interfaceTag));
        return CSR_RESULT_FAILURE;
    }

    vif = (CsrWifiHipVifInstance *) &priv->vif[interfaceTag];

    switch (vif->interfaceMode)
    {
        case CSR_WIFI_ROUTER_CTRL_MODE_AP:
        case CSR_WIFI_ROUTER_CTRL_MODE_IBSS:
        case CSR_WIFI_ROUTER_CTRL_MODE_P2PGO:

            /* Add station record */
            break;
        case CSR_WIFI_ROUTER_CTRL_MODE_STA:
        case CSR_WIFI_ROUTER_CTRL_MODE_AMP:
        case CSR_WIFI_ROUTER_CTRL_MODE_P2PCLI:
        default:
            /* No station record to maintain in these modes */
            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2, "unifi%d: CsrWifiHipPeerAddReq: Mode was %u so hasn't added a peer instance",
                               priv ? priv->instance : 0,  vif->interfaceMode));
            return CSR_RESULT_SUCCESS;
    }

    CSR_WIFI_HIP_SPINLOCK_LOCK(&priv->peerInfoLock);
    for (i = 0; i < CSR_WIFI_HIP_PEER_CONNECTIONS_MAX; i++)
    {
        if (vif->staPeerInfo[i].entryInUse == FALSE)
        {
            CsrWifiHipStaPeerInfo *newRecord = NULL;

            /* Slot is empty, so can be used for station record */
            freeSlotFound = TRUE;
            *peerHandle = i;

            newRecord = &vif->staPeerInfo[i];

            /* Initialize the record */
            csrWifiHipApStaRecordInitialise(priv, newRecord);
            newRecord->entryInUse = TRUE;

            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2, "unifi%d: CsrWifiHipPeerAddReq: New peer address = %02x-%02x-%02x-%02x-%02x-%02x\n",
                               priv ? priv->instance : 0,
                               peerMacAddress.a[0], peerMacAddress.a[1], peerMacAddress.a[2],
                               peerMacAddress.a[3], peerMacAddress.a[4], peerMacAddress.a[5]));

            /* Update the station record */
            CsrMemCpy(newRecord->peerMacAddress.a, peerMacAddress.a, ETH_ALEN);
            newRecord->wmmOrQosEnabled = staInfo->wmmOrQosEnabled;
            newRecord->assignedHandle = i;
            newRecord->associationId = associationId;

#ifdef  CSR_WIFI_AP_ENABLE
            /* maxSpLength is bit map in qosInfo field, so converting accordingly */
            newRecord->maxSpLength = staInfo->maxSpLength * 2;

            /* Max SP 0 mean any number of packets. since we buffer only 512
             * packets we are hard coding this to zero for the moment
             */
            if (newRecord->maxSpLength == 0)
            {
                newRecord->maxSpLength = 512;
            }

            /* Copy power save mode of all access catagory (Trigger/Delivery/both enabled/disabled) */
            powerModeTemp = (CsrUint8) ((staInfo->powersaveMode >> 4) & 0xff);

            if (!(staInfo->powersaveMode & 0x0001))
            {
                newRecord->powersaveMode[UNIFI_TRAFFIC_Q_BK] = CSR_WIFI_HIP_AC_LEGACY_POWER_SAVE;
            }
            else
            {
                newRecord->powersaveMode[UNIFI_TRAFFIC_Q_BK] = powerModeTemp & 0x03;
            }

            if (!(staInfo->powersaveMode & 0x0002))
            {
                newRecord->powersaveMode[UNIFI_TRAFFIC_Q_BE] = CSR_WIFI_HIP_AC_LEGACY_POWER_SAVE;
            }
            else
            {
                newRecord->powersaveMode[UNIFI_TRAFFIC_Q_BE] = ((powerModeTemp & 0x0C) >> 2);
            }

            if (!(staInfo->powersaveMode & 0x0004))
            {
                newRecord->powersaveMode[UNIFI_TRAFFIC_Q_VI] = CSR_WIFI_HIP_AC_LEGACY_POWER_SAVE;
            }
            else
            {
                newRecord->powersaveMode[UNIFI_TRAFFIC_Q_VI] = ((powerModeTemp & 0x30) >> 4);
            }

            if (!(staInfo->powersaveMode & 0x0008))
            {
                newRecord->powersaveMode[UNIFI_TRAFFIC_Q_VO] = CSR_WIFI_HIP_AC_LEGACY_POWER_SAVE;
            }
            else
            {
                newRecord->powersaveMode[UNIFI_TRAFFIC_Q_VO] = ((powerModeTemp & 0xC0) >> 6);
            }

            {
                CsrUint8 k;
                for (k = 0; k < CSR_WIFI_HIP_MAX_ACCESS_CATOGORY; k++)
                {
                    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                                       "unifi%d: CsrWifiHipPeerAddReq: WMM: %d, AC: %d, powersaveMode: %x\n",
                                       priv->instance, staInfo->wmmOrQosEnabled, k, newRecord->powersaveMode[k]));
                }
            }

            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3,
                                "unifi%d: CsrWifiHipPeerAddReq: wmmOrQosEnabled: %d, MAX SP: %d\n",
                                priv ? priv->instance : 0,
                                newRecord->wmmOrQosEnabled, newRecord->maxSpLength));
#endif

            CSR_WIFI_HIP_SPINLOCK_UNLOCK(&priv->peerInfoLock);

            r = csrWifiHipPeerAssociateQueueSetReq(priv, interfaceTag, associationId);
            if (r != CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                       "unifi%d: CsrWifiHipPeerAddReq: Failed to set packet scheduler queues (VIF tag %u aId %u error 0x%02X)\n",
                                       priv->instance, interfaceTag, associationId, r));

                csrWifiHipApStaRecordInitialise(priv, newRecord);
                return CSR_RESULT_FAILURE;
            }

            /* First time port actions are set for the peer with below information,
             * As we will get PeerAddReq & then PortConfigureReq from SME.
             * Inorder to keep our station record Data base up to date we are adding
             * the entry here in port config array & updating pointers
             */
            (void) csrWifiHipPortConfigure(priv,
                                           vif,
                                           CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_OPEN,
                                           &peerMacAddress,
                                           CSR_WIFI_HIP_PORT_TYPE_UNCONTROLLED);

            if (vif->interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_IBSS)
            {
                (void) csrWifiHipPortConfigure(priv,
                                               vif,
                                               CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_OPEN,
                                               &peerMacAddress,
                                               CSR_WIFI_HIP_PORT_TYPE_CONTROLLED);
            }
            else
            {
                (void) csrWifiHipPortConfigure(priv,
                                               vif,
                                               CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_CLOSED_DISCARD,
                                               &peerMacAddress,
                                               CSR_WIFI_HIP_PORT_TYPE_CONTROLLED);
            }


            CSR_WIFI_HIP_SPINLOCK_LOCK(&priv->peerInfoLock);
            /* The protection bit is updated once the port opens for corresponding peer in
             * routerPortConfigure request */
            newRecord->peerControlledPort = CsrWifiHipControlledPortHandle(priv,
                                                                           vif,
                                                                           newRecord->peerMacAddress.a,
                                                                           CSR_WIFI_HIP_PORT_TYPE_CONTROLLED);

            newRecord->peerUnControlledPort = CsrWifiHipControlledPortHandle(priv,
                                                                             vif,
                                                                             newRecord->peerMacAddress.a,
                                                                             CSR_WIFI_HIP_PORT_TYPE_UNCONTROLLED);

            if (!newRecord->peerControlledPort || !newRecord->peerUnControlledPort)
            {
                CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID,
                                      CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipPeerAddReq: Un/ControlledPort record not found in port configuration array index = %d\n",
                                      priv ? priv->instance : 0,  i));
                csrWifiHipApStaRecordInitialise(priv, newRecord);
                CSR_WIFI_HIP_SPINLOCK_UNLOCK(&priv->peerInfoLock);
                CsrWifiHipPacketSchedulerQueueSetDisassociate(priv->card, interfaceTag, associationId);
                return CSR_RESULT_FAILURE;
            }

            newRecord->currentPeerState = CSR_WIFI_ROUTER_CTRL_PEER_CONNECTED_ACTIVE;

            CSR_WIFI_HIP_SPINLOCK_UNLOCK(&priv->peerInfoLock);

#ifdef CSR_WIFI_AP_ENABLE
            vif->numPeers++;

            /* Inactivity timer initializations */
            newRecord->active = TRUE;
            newRecord->listenIntervalInTUs = staInfo->listenIntervalInTus;
            newRecord->hostTagNullFrame = CSR_WIFI_HIP_HOST_TAG_INVALID;

            (void) CsrMutexLock(&priv->configurationMutex);
            if (!(vif->inactiveDetectEnabled) &&
                (vif->numPeers >= CSR_WIFI_HIP_STA_INACTIVE_DETECTION_TRIGGER_THRESHOLD))
            {
                CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi%d: CsrWifiHipPeerAddReq: Starting the inactivity detection timer (numPeers = %d)\n",
                                   priv->instance, vif->numPeers));

                r = CsrTimerCreate(CsrWifiHipTimerExpInactivity, (void *) priv, &vif->inActivityTimerHandle);
                if (r != CSR_RESULT_SUCCESS)
                {
                    (void) CsrMutexUnlock(&priv->configurationMutex);
                    CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                          "unifi%d: CsrWifiHipPeerAddReq: Failed to add timer\n",
                                          priv->instance));
                    csrWifiHipApStaRecordInitialise(priv, newRecord);
                    vif->numPeers--;
                    CsrWifiHipPacketSchedulerQueueSetDisassociate(priv->card, interfaceTag, associationId);
                    return CSR_RESULT_FAILURE;
                }

                CsrTimerStart(&vif->inActivityTimerHandle, CSR_WIFI_HIP_STA_INACTIVE_DETECTION_TIMER_INTERVAL);
                vif->inactiveDetectEnabled = TRUE;
            }
            (void) CsrMutexUnlock(&priv->configurationMutex);
#endif
            break;
        }
    }

    if (!freeSlotFound)
    {
        /* Release the lock */
        CSR_WIFI_HIP_SPINLOCK_UNLOCK(&priv->peerInfoLock);
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiHipPeerAddReq: Limited connectivity, Free slot not found for station record addition",
                               priv->instance));
        return CSR_RESULT_FAILURE;
    }

    return CSR_RESULT_SUCCESS;
}

CsrResult CsrWifiHipPeerUpdateReq(void *hipHandle, CsrUint16 interfaceTag, CsrUint32 peerHandle, CsrResult *result)
{
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;
    CsrWifiHipVifInstance *vif = NULL;
    CsrWifiHipStaPeerInfo *staPeerInfo = NULL;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipPeerUpdateReq: Invalid hipHandle", priv ? priv->instance : 0));
        return CSR_RESULT_FAILURE;
    }

    vif = (CsrWifiHipVifInstance *) &priv->vif[interfaceTag];

    if (peerHandle < CSR_WIFI_HIP_PEER_CONNECTIONS_MAX)
    {
        staPeerInfo = (CsrWifiHipStaPeerInfo *) &vif->staPeerInfo[peerHandle];
        if (staPeerInfo->entryInUse == TRUE)
        {
            *result = staPeerInfo->currentPeerState;
        }
        else
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipPeerUpdateReq: PeerHandle (%d) is incorrect",
                                   priv ? priv->instance : 0,  peerHandle));
            return CSR_RESULT_FAILURE;
        }
    }
    else
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipPeerUpdateReq: PeerHandle (%d) is incorrect",
                               priv ? priv->instance : 0,  peerHandle));
        return CSR_RESULT_FAILURE;
    }

    return CSR_RESULT_SUCCESS;
}

CsrResult CsrWifiHipPeerDelReq(void *hipHandle, CsrUint16 interfaceTag, CsrUint32 peerHandle)
{
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;
    CsrWifiHipVifInstance *vif = NULL;
    CsrWifiHipStaPeerInfo *staPeerInfo = NULL;

    unifi_port_config_t *controlledPort;
    unifi_port_config_t *unControlledPort;
    CsrUint32 baSessionIdx = 0;
    CsrWifiHipBaSessionRx *baSessionRx = NULL;
    CsrWifiHipBaSessionTx *baSessionTx = NULL;
    CsrResult r;


    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipPeerDelReq: Invalid hipHandle\n", priv ? priv->instance : 0));
        return CSR_RESULT_FAILURE;
    }

    if (interfaceTag >= CSR_WIFI_MAX_INTERFACES)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipPeerDelReq: Invalid interfaceTag %d\n",
                               priv ? priv->instance : 0,  interfaceTag));
        return CSR_RESULT_FAILURE;
    }

    vif = (CsrWifiHipVifInstance *) &priv->vif[interfaceTag];

    switch (vif->interfaceMode)
    {
        case CSR_WIFI_ROUTER_CTRL_MODE_AP:
        case CSR_WIFI_ROUTER_CTRL_MODE_IBSS:
        case CSR_WIFI_ROUTER_CTRL_MODE_P2PGO:

            /* remove the station from station record data base */
            break;
        case CSR_WIFI_ROUTER_CTRL_MODE_STA:
        case CSR_WIFI_ROUTER_CTRL_MODE_P2PCLI:
        case CSR_WIFI_ROUTER_CTRL_MODE_AMP:
        default:
            /* No station record to maintain in these modes */
            return CSR_RESULT_SUCCESS;
    }

    if (peerHandle < CSR_WIFI_HIP_PEER_CONNECTIONS_MAX)
    {
        staPeerInfo = (CsrWifiHipStaPeerInfo *) &vif->staPeerInfo[peerHandle];

        /* Remove the station record & make it NULL */
        if (staPeerInfo->entryInUse == TRUE)
        {
            /* Clear the port configure array info, for the corresponding peer entry */
            controlledPort = &vif->staInfo.controlled_data_port;
            unControlledPort = &vif->staInfo.uncontrolled_data_port;

            /* Flush any queued frames */
            (void) CsrMutexLock(&priv->configurationMutex);
            csrWifiHipCtrlPortQueueProcess(priv,
                                           vif,
                                           CSR_WIFI_HIP_PORT_TYPE_CONTROLLED,
                                           &staPeerInfo->peerMacAddress,
                                           FALSE);

            csrWifiHipCtrlPortQueueProcess(priv,
                                           vif,
                                           CSR_WIFI_HIP_PORT_TYPE_UNCONTROLLED,
                                           &staPeerInfo->peerMacAddress,
                                           FALSE);
            (void) CsrMutexUnlock(&priv->configurationMutex);

            CSR_WIFI_HIP_SPINLOCK_LOCK(&priv->peerInfoLock);
            CsrMemSet(staPeerInfo->peerControlledPort, 0, sizeof(unifi_port_cfg_t));
            staPeerInfo->peerControlledPort->port_action = CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_CLOSED_DISCARD;
            staPeerInfo->peerControlledPort->in_use = FALSE;
            if (controlledPort->entries_in_use)
            {
                controlledPort->entries_in_use--;
            }
            else
            {
                CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipPeerDelReq: Number of controlled port entries is zero, trying to decrement\n", priv ? priv->instance : 0));
            }

            CsrMemSet(staPeerInfo->peerUnControlledPort, 0, sizeof(unifi_port_cfg_t));
            staPeerInfo->peerUnControlledPort->port_action = CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_CLOSED_DISCARD;
            staPeerInfo->peerUnControlledPort->in_use = FALSE;
            if (unControlledPort->entries_in_use)
            {
                unControlledPort->entries_in_use--;
            }
            else
            {
                CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipPeerDelReq: Number of uncontrolled port entries is zero, trying to decrement\n", priv ? priv->instance : 0));
            }
            CSR_WIFI_HIP_SPINLOCK_UNLOCK(&priv->peerInfoLock);

#ifdef CSR_WIFI_AP_ENABLE
            CSR_WIFI_HIP_SPINLOCK_LOCK(&priv->peerInfoLock);
            vif->numPeers--;

            staPeerInfo->hostTagNullFrame = CSR_WIFI_HIP_HOST_TAG_INVALID;

            CSR_WIFI_HIP_SPINLOCK_UNLOCK(&priv->peerInfoLock);
#endif

            /* Stop inactivity timer and BA session if it is active, for this peer
               address all BA sessions (per tID per role) are closed. */
            (void) CsrMutexLock(&priv->configurationMutex);
#ifdef CSR_WIFI_AP_ENABLE
            if ((vif->inactiveDetectEnabled) &&
                (vif->numPeers < CSR_WIFI_HIP_STA_INACTIVE_DETECTION_TRIGGER_THRESHOLD))
            {
                CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi%d: CsrWifiHipPeerDelReq: Stopping the inactivity detection timer (numPeers = %d)\n",
                                   priv->instance, vif->numPeers));

                vif->inactiveDetectEnabled = FALSE;
                CsrTimerDestroy(&vif->inActivityTimerHandle);
            }
#endif
            for (baSessionIdx = 0; baSessionIdx < CSR_WIFI_HIP_BA_SUPPORTED_SESSIONS_RX_MAX; baSessionIdx++)
            {
                baSessionRx = vif->baSessionRx[baSessionIdx];
                if (baSessionRx)
                {
                    if (!CsrMemCmp(baSessionRx->macAddress.a, staPeerInfo->peerMacAddress.a, ETH_ALEN))
                    {
                        r = csrWifiHipBaSessionStop(priv,
                                                    vif,
                                                    CSR_WIFI_ROUTER_CTRL_BLOCK_ACK_RECIPIENT,
                                                    baSessionRx->trafficStreamId,
                                                    baSessionRx->macAddress);
                        if (r != CSR_RESULT_SUCCESS)
                        {
                            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipPeerDelReq: Failed to stop RX BA session for traffic stream id %u\n",
                                                   priv ? priv->instance : 0,
                                                   baSessionRx->trafficStreamId));
                        }
                    }
                }
            }

            for (baSessionIdx = 0; baSessionIdx < CSR_WIFI_HIP_BA_SUPPORTED_SESSIONS_TX_MAX; baSessionIdx++)
            {
                baSessionTx = vif->baSessionTx[baSessionIdx];
                if (baSessionTx)
                {
                    if (!CsrMemCmp(baSessionTx->macAddress.a, staPeerInfo->peerMacAddress.a, ETH_ALEN))
                    {
                        r = csrWifiHipBaSessionStop(priv,
                                                    vif,
                                                    CSR_WIFI_ROUTER_CTRL_BLOCK_ACK_ORIGINATOR,
                                                    baSessionTx->trafficStreamId,
                                                    baSessionTx->macAddress);
                        if (r != CSR_RESULT_SUCCESS)
                        {
                            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipPeerDelReq: Failed to stop TX BA session for traffic stream id %u\n",
                                                   priv ? priv->instance : 0,
                                                   baSessionTx->trafficStreamId));
                        }
                    }
                }
            }

            r = CsrWifiHipPacketSchedulerQueueSetDisassociate(priv->card, interfaceTag, staPeerInfo->associationId);
            if (r != CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                       "unifi%d: CsrWifiHipPeerDelReq: Disassociate queue set failed, interfaceTag %u, aId %u, error %u\n",
                                       priv ? priv->instance : 0,
                                       interfaceTag,
                                       staPeerInfo->associationId, r));
            }
            (void) CsrMutexUnlock(&priv->configurationMutex);

            CSR_WIFI_HIP_SPINLOCK_LOCK(&priv->peerInfoLock);
            /* Release the peer station record */
            csrWifiHipApStaRecordInitialise(priv, staPeerInfo);

            CSR_WIFI_HIP_SPINLOCK_UNLOCK(&priv->peerInfoLock);
        }
        else
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipPeerDelReq: peerHandle (%d) is invalid\n",
                                   priv ? priv->instance : 0,  peerHandle));
            return CSR_RESULT_FAILURE;
        }
    }
    else
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipPeerDelReq: peerHandle (%d) is invalid",
                               priv ? priv->instance : 0,  peerHandle));
        return CSR_RESULT_FAILURE;
    }

    return CSR_RESULT_SUCCESS;
}

CsrResult CsrWifiHipBlockAckDisableReq(void *hipHandle, CsrUint16 interfaceTag,
                                       CsrWifiMacAddress macAddress, CsrWifiRouterCtrlTrafficStreamId trafficStreamID,
                                       CsrWifiRouterCtrlBlockAckRole role)
{
    CsrResult r;
    CsrWifiHipHalPriv *priv = NULL;

    CsrWifiHipVifInstance *vif = NULL;

    if (hipHandle == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipBlockAckDisableReq: Invalid hipHandle", priv ? priv->instance : 0));
        return CSR_RESULT_FAILURE;
    }

    priv = (CsrWifiHipHalPriv *) hipHandle;

    if (interfaceTag >= CSR_WIFI_MAX_INTERFACES)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipBlockAckDisableReq: Invalid interfaceTag %d",
                               priv ? priv->instance : 0,  interfaceTag));
        return CSR_RESULT_FAILURE;
    }

    vif = (CsrWifiHipVifInstance *) &priv->vif[interfaceTag];

    (void) CsrMutexLock(&priv->configurationMutex);
    r = csrWifiHipBaSessionStop(priv,
                                vif,
                                role,
                                trafficStreamID,
                                macAddress);
    (void) CsrMutexUnlock(&priv->configurationMutex);

    return r;
}

CsrResult CsrWifiHipBlockAckEnableReq(void *hipHandle, CsrUint16 interfaceTag, CsrWifiMacAddress macAddress,
                                      CsrWifiRouterCtrlTrafficStreamId trafficStreamID, CsrWifiRouterCtrlBlockAckRole role,
                                      CsrUint16 bufferSize, CsrUint16 timeout, CsrUint16 ssn)
{
    CsrResult r;
    CsrWifiHipHalPriv *priv = NULL;
    CsrWifiHipVifInstance *vif = NULL;

    if (hipHandle == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipBlockAckEnableReq: Invalid hipHandle", priv ? priv->instance : 0));
        return CSR_RESULT_FAILURE;
    }

    priv = (CsrWifiHipHalPriv *) hipHandle;

    if (interfaceTag >= CSR_WIFI_MAX_INTERFACES)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipBlockAckEnableReq: Invalid interfaceTag %d",
                               priv ? priv->instance : 0,  interfaceTag));
        return CSR_RESULT_FAILURE;
    }

    vif = (CsrWifiHipVifInstance *) &priv->vif[interfaceTag];

    (void) CsrMutexLock(&priv->configurationMutex);
    r = csrWifiHipBaSessionStart(priv,
                                 vif,
                                 trafficStreamID,
                                 timeout,
                                 role,
                                 bufferSize,
                                 ssn,
                                 macAddress);
    (void) CsrMutexUnlock(&priv->configurationMutex);

    return r;
}

CsrResult CsrWifiHipInterfaceAddReq(void *hipHandle, CsrUint16 interfaceTag, CsrWifiMacAddress macAddress)
{
    CsrResult r;
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;

    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                       "CsrWifiHipInterfaceAddReq: ifTag=%d\n", interfaceTag));

    if (interfaceTag >= CSR_WIFI_MAX_INTERFACES)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d:  CsrWifiHipInterfaceAddReq: Invalid interfaceTag (%u)\n",
                               priv->instance, interfaceTag));
        return CSR_RESULT_FAILURE;
    }

    r = CsrWifiHipInterfaceRegisterInd(priv->osLayerContext, interfaceTag, macAddress);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipInterfaceAddReq: FAILED\n", priv->instance));
        return r;
    }

    return r;
}

CsrResult CsrWifiHipInterfaceDelReq(void *hipHandle, CsrUint16 interfaceTag)
{
    CsrResult r;
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                        "CsrWifiHipInterfaceDelReq: ifTag=%d\n", interfaceTag));

    if (interfaceTag >= CSR_WIFI_MAX_INTERFACES)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d:  CsrWifiHipInterfaceDelReq: Invalid interfaceTag (%u)\n",
                               priv->instance, interfaceTag));
        return CSR_RESULT_FAILURE;
    }

    r = CsrWifiHipInterfaceUnregisterInd(priv->osLayerContext, interfaceTag);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipInterfaceDelReq: FAILED ", priv->instance));
        return r;
    }

    return r;
}

CsrResult CsrWifiHipVifAddReq(void *hipHandle, CsrUint16 interfaceTag, CsrUint16 vifIndex, CsrBool isPrimaryVifIndex)
{
    CsrResult r;
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;

    if (interfaceTag >= CSR_WIFI_MAX_INTERFACES)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiHipVifAddReq: Invalid interfaceTag (%u)\n",
                               priv->instance, interfaceTag));
        return CSR_RESULT_FAILURE;
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                        "CsrWifiHipVifAddReq: ifTag=%d\n", interfaceTag));

    r = csrWifiHipUpdateInterfaceTagAndVifIndexMap(priv, interfaceTag, vifIndex,
                                                   isPrimaryVifIndex, CSR_WIFI_HIP_VIFINDEX_ADD,
                                                   0);
    if (r)
    {
        CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                            "unifi(%d): CsrWifiHipVifAddReq: vifIndex mapping failed\n",
                            priv->instance));
    }
    return r;
}

CsrResult CsrWifiHipVifDelReq(void *hipHandle, CsrUint16 interfaceTag, CsrUint16 vifIndex, CsrUint16 vifIndexPrimary)
{
    CsrResult r;
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) hipHandle;

    if (interfaceTag >= CSR_WIFI_MAX_INTERFACES)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d:  CsrWifiHipVifDelReq: Invalid interfaceTag (%u)\n",
                               priv->instance, interfaceTag));
        return CSR_RESULT_FAILURE;
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                        "CsrWifiHipVifDelReq: ifTag=%d\n", interfaceTag));

    r = csrWifiHipUpdateInterfaceTagAndVifIndexMap(priv, interfaceTag, vifIndex,
                                                   TRUE, CSR_WIFI_HIP_VIFINDEX_DEL,
                                                   vifIndexPrimary);
    if (r)
    {
        CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                            "unifi(%d): CsrWifiHipVifDelReq: vifIndex mapping failed\n",
                            priv->instance));
    }
    return r;
}
