/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"
#include "csr_wifi_hip_log_text.h"

#include "os_linux_priv.h"
#include "csr_wifi_hip_unifi.h"
#include "csr_wifi_hip_conversions.h"


int convert_sme_error(CsrResult error)
{
    switch (error)
    {
        case CSR_RESULT_SUCCESS:
            return 0;
        case CSR_RESULT_FAILURE:
        case CSR_WIFI_RESULT_NOT_FOUND:
        case CSR_WIFI_RESULT_TIMED_OUT:
        case CSR_WIFI_RESULT_CANCELLED:
        case CSR_WIFI_RESULT_UNAVAILABLE:
            return -EIO;
        case CSR_WIFI_RESULT_NO_ROOM:
            return -EBUSY;
        case CSR_WIFI_RESULT_INVALID_PARAMETER:
            return -EINVAL;
        case CSR_WIFI_RESULT_UNSUPPORTED:
            return -EOPNOTSUPP;
        default:
            return -EIO;
    }
}

void uf_multicast_list_wq(struct work_struct *work)
{
    os_linux_priv_t *priv = container_of(work, os_linux_priv_t,
                                         multicast_list_task);
    int i;
    CsrWifiMacAddress *multicast_address_list = NULL;
    int mc_count;
    u8 *mc_list;
    netInterface_priv_t *interfacePriv = priv->interfacePriv[0];

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG5,
                           "unifi%d: uf_multicast_list_wq: list count = %d\n",
                           priv ? priv->instance : 0,
                           interfacePriv->mc_list_count));

    /* Flush the current list */
    CsrWifiRouterCtrlMulticastAddressIndSend(priv->CSR_WIFI_SME_IFACEQUEUE, 0, interfacePriv->InterfaceTag, CSR_WIFI_SME_LIST_ACTION_FLUSH, 0, NULL);

    mc_count = interfacePriv->mc_list_count;
    mc_list = interfacePriv->mc_list;
    /*
     * Allocate a new list, need to free it later
     * in unifi_mgt_multicast_address_cfm().
     */
    multicast_address_list = CsrPmemAlloc(mc_count * sizeof(CsrWifiMacAddress));

    if (multicast_address_list == NULL)
    {
        return;
    }

    for (i = 0; i < mc_count; i++)
    {
        memcpy(multicast_address_list[i].a, mc_list, ETH_ALEN);
        mc_list += ETH_ALEN;
    }

    if (priv->smepriv == NULL)
    {
        CsrPmemFree(multicast_address_list);
        return;
    }

    CsrWifiRouterCtrlMulticastAddressIndSend(priv->CSR_WIFI_SME_IFACEQUEUE, 0,
                                             interfacePriv->InterfaceTag,
                                             CSR_WIFI_SME_LIST_ACTION_ADD,
                                             mc_count, multicast_address_list);

    /* The SME will take a copy of the addreses*/
    CsrPmemFree(multicast_address_list);
}

int unifi_cfg_power(os_linux_priv_t *priv, unsigned char *arg)
{
    unifi_cfg_power_t cfg_power;
    int rc;

    if (get_user(cfg_power, (unifi_cfg_power_t *) (((unifi_cfg_command_t *) arg) + 1)))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: UNIFI_CFG: Failed to get the argument\n", priv ? priv->instance : 0));
        return -EFAULT;
    }

    switch (cfg_power)
    {
        case UNIFI_CFG_POWER_OFF:
            rc = sme_sys_suspend(priv);
            if (rc)
            {
                return rc;
            }
            break;
        case UNIFI_CFG_POWER_ON:
            rc = sme_sys_resume(priv);
            if (rc)
            {
                return rc;
            }
            break;
        default:
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: WIFI POWER: Unknown value.\n", priv ? priv->instance : 0));
            return -EINVAL;
    }

    return 0;
}

int unifi_cfg_power_save(os_linux_priv_t *priv, CsrUint16 InterfaceTag, unsigned char *arg)
{
    unifi_cfg_powersave_t cfg_power_save;
    CsrWifiDataBlock mibData = {0, NULL};
    CsrUint32 powerSaveLevel;
    int rc;
    CsrUint8 *data;

    if (get_user(cfg_power_save, (unifi_cfg_powersave_t *) (((unifi_cfg_command_t *) arg) + 1)))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: UNIFI_CFG: Failed to get the argument\n", priv ? priv->instance : 0));
        return -EFAULT;
    }

    switch (cfg_power_save)
    {
        case UNIFI_CFG_POWERSAVE_NONE:
            powerSaveLevel = CSR_WIFI_SME_POWER_SAVE_LEVEL_LOW;
            break;
        case UNIFI_CFG_POWERSAVE_FAST:
            powerSaveLevel = CSR_WIFI_SME_POWER_SAVE_LEVEL_MED;
            break;
        case UNIFI_CFG_POWERSAVE_FULL:
            powerSaveLevel = CSR_WIFI_SME_POWER_SAVE_LEVEL_HIGH;
            break;
        case UNIFI_CFG_POWERSAVE_AUTO:
            powerSaveLevel = CSR_WIFI_SME_POWER_SAVE_LEVEL_AUTO;
            break;
        default:
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: POWERSAVE: Unknown value.\n", priv ? priv->instance : 0));
            return -EINVAL;
    }

    if (CsrWifiMibEncodeUint(&mibData, CSR_WIFI_SME_PSID_POWER_SAVE_LEVEL, powerSaveLevel, 0) != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: POWERSAVE: CsrWifiMibEncodeUint() Failed.\n", priv ? priv->instance : 0));
        return -EINVAL;
    }
    data = mibData.data; /* allocated by CsrWifiMibEncodeUint, and we need to free this later */
    rc = sme_mgt_mib_set(priv, InterfaceTag, &mibData);
    CsrPmemFree(data);
    if (rc)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: UNIFI_CFG: Set unifi_PowerConfigValue failed.\n", priv ? priv->instance : 0));
    }

    return rc;
}

int unifi_cfg_power_supply(os_linux_priv_t *priv, CsrUint16 InterfaceTag, unsigned char *arg)
{
    unifi_cfg_powersupply_t cfg_power_supply;
    CsrWifiDataBlock mibData = {0, NULL};
    CsrUint32 powerMode;
    int rc;
    CsrUint8 *data;

    if (get_user(cfg_power_supply, (unifi_cfg_powersupply_t *) (((unifi_cfg_command_t *) arg) + 1)))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: UNIFI_CFG: Failed to get the argument\n", priv ? priv->instance : 0));
        return -EFAULT;
    }

    switch (cfg_power_supply)
    {
        case UNIFI_CFG_POWERSUPPLY_MAINS:
            powerMode = CSR_WIFI_SME_HOST_POWER_MODE_ACTIVE;
            break;
        case UNIFI_CFG_POWERSUPPLY_BATTERIES:
            powerMode = CSR_WIFI_SME_HOST_POWER_MODE_POWER_SAVE;
            break;
        default:
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: POWERSUPPLY: Unknown value.\n", priv ? priv->instance : 0));
            return -EINVAL;
    }

    if (CsrWifiMibEncodeUint(&mibData, CSR_WIFI_SME_PSID_HOST_POWER_MODE, powerMode, 0) != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: POWERSUPPLY: CsrWifiMibEncodeUint() Failed.\n", priv ? priv->instance : 0));
        return -EINVAL;
    }
    data = mibData.data; /* allocated by CsrWifiMibEncodeUint, and we need to free this later */
    rc = sme_mgt_mib_set(priv, InterfaceTag, &mibData);
    CsrPmemFree(data);
    if (rc)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: UNIFI_CFG: Set unifi_HostConfigValue failed.\n", priv ? priv->instance : 0));
    }

    return rc;
}

int unifi_cfg_packet_filters(os_linux_priv_t *priv, CsrUint16 InterfaceTag, unsigned char *arg)
{
    unsigned char *tclas_buffer;
    unsigned int tclas_buffer_length;
    tclas_t *dhcp_tclas;
    int rc;
    netInterface_priv_t *interfacePriv = priv->interfacePriv[InterfaceTag];

    /* Free any TCLASs previously allocated */
    if (interfacePriv->packet_filters.tclas_ies_length)
    {
        CsrPmemFree(interfacePriv->filter_tclas_ies);
        interfacePriv->filter_tclas_ies = NULL;
    }

    tclas_buffer = ((unsigned char *) arg) + sizeof(unifi_cfg_command_t) + sizeof(unsigned int);
    if (copy_from_user(&interfacePriv->packet_filters, (void *) tclas_buffer,
                       sizeof(uf_cfg_bcast_packet_filter_t)))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: UNIFI_CFG: Failed to get the filter struct\n", priv ? priv->instance : 0));
        return -EFAULT;
    }

    tclas_buffer_length = interfacePriv->packet_filters.tclas_ies_length;

    /* Allocate TCLASs if necessary */
    if (interfacePriv->packet_filters.dhcp_filter)
    {
        interfacePriv->packet_filters.tclas_ies_length += sizeof(tclas_t);
    }
    if (interfacePriv->packet_filters.tclas_ies_length > 0)
    {
        interfacePriv->filter_tclas_ies = CsrPmemAlloc(interfacePriv->packet_filters.tclas_ies_length);
        if (interfacePriv->filter_tclas_ies == NULL)
        {
            return -ENOMEM;
        }
        if (tclas_buffer_length)
        {
            tclas_buffer += sizeof(uf_cfg_bcast_packet_filter_t) - sizeof(unsigned char *);
            if (copy_from_user(interfacePriv->filter_tclas_ies,
                               tclas_buffer,
                               tclas_buffer_length))
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: UNIFI_CFG: Failed to get the TCLAS buffer\n", priv ? priv->instance : 0));
                return -EFAULT;
            }
        }
    }

    if (interfacePriv->packet_filters.dhcp_filter)
    {
        /* Append the DHCP tclas IE */
        dhcp_tclas = (tclas_t *) (interfacePriv->filter_tclas_ies + tclas_buffer_length);
        memset(dhcp_tclas, 0, sizeof(tclas_t));
        dhcp_tclas->element_id = 14;
        dhcp_tclas->length = sizeof(tcpip_clsfr_t) + 1;
        dhcp_tclas->user_priority = 0;
        dhcp_tclas->tcp_ip_cls_fr.cls_fr_type = 1;
        dhcp_tclas->tcp_ip_cls_fr.version = 4;
        ((CsrUint8 *) (&dhcp_tclas->tcp_ip_cls_fr.source_port))[0] = 0x00;
        ((CsrUint8 *) (&dhcp_tclas->tcp_ip_cls_fr.source_port))[1] = 0x44;
        ((CsrUint8 *) (&dhcp_tclas->tcp_ip_cls_fr.dest_port))[0] = 0x00;
        ((CsrUint8 *) (&dhcp_tclas->tcp_ip_cls_fr.dest_port))[1] = 0x43;
        dhcp_tclas->tcp_ip_cls_fr.protocol = 0x11;
        dhcp_tclas->tcp_ip_cls_fr.cls_fr_mask = 0x58; /* bits: 3,4,6 */
    }

    rc = sme_mgt_packet_filter_set(priv, InterfaceTag);

    return rc;
}

int unifi_cfg_wmm_qos_info(os_linux_priv_t *priv, CsrUint16 InterfaceTag, unsigned char *arg)
{
    CsrUint8 wmm_qos_info;
    CsrWifiDataBlock mibData = {0, NULL};
    int rc = 0;
    CsrUint8 *data;

    if (get_user(wmm_qos_info, (CsrUint8 *) (((unifi_cfg_command_t *) arg) + 1)))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: UNIFI_CFG: Failed to get the argument\n", priv ? priv->instance : 0));
        return -EFAULT;
    }

    if (CsrWifiMibEncodeUint(&mibData, CSR_WIFI_SME_PSID_WMM_QOS_INFO, wmm_qos_info, 0) != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: WMM_QOS_INFO: CsrWifiMibEncodeUint() Failed.\n", priv ? priv->instance : 0));
        return -EINVAL;
    }
    data = mibData.data; /* allocated by CsrWifiMibEncodeUint, and we need to free this later */
    rc = sme_mgt_mib_set(priv, InterfaceTag, &mibData);
    CsrPmemFree(data);
    if (rc)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: WMM_QOS_INFO: Set CsrWifiSmeSetReq failed.\n", priv ? priv->instance : 0));
    }
    return rc;
}

int unifi_cfg_wmm_addts(os_linux_priv_t *priv, CsrUint16 InterfaceTag, unsigned char *arg)
{
    CsrUint32 addts_tid;
    CsrUint8 addts_ie_length;
    CsrUint8 *addts_ie;
    CsrUint8 *addts_params;
    CsrWifiDataBlock tspec;
    CsrWifiDataBlock tclas;
    int rc;

    addts_params = (CsrUint8 *) (((unifi_cfg_command_t *) arg) + 1);
    if (get_user(addts_tid, (CsrUint32 *) addts_params))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_cfg_wmm_addts: Failed to get the argument\n", priv ? priv->instance : 0));
        return -EFAULT;
    }

    addts_params += sizeof(CsrUint32);
    if (get_user(addts_ie_length, (CsrUint8 *) addts_params))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_cfg_wmm_addts: Failed to get the argument\n", priv ? priv->instance : 0));
        return -EFAULT;
    }

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: addts: tid = 0x%x ie_length = %d\n",
                           priv ? priv->instance : 0,
                           addts_tid, addts_ie_length));

    addts_ie = CsrPmemAlloc(addts_ie_length);
    if (addts_ie == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_cfg_wmm_addts: Failed to malloc %d bytes for addts_ie buffer\n",
                               priv ? priv->instance : 0,
                               addts_ie_length));
        return -ENOMEM;
    }

    addts_params += sizeof(CsrUint8);
    rc = copy_from_user(addts_ie, addts_params, addts_ie_length);
    if (rc)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_cfg_wmm_addts: Failed to get the addts buffer\n", priv ? priv->instance : 0));
        CsrPmemFree(addts_ie);
        return -EFAULT;
    }

    tspec.data = addts_ie;
    tspec.dataLength = addts_ie_length;
    tclas.data = NULL;
    tclas.dataLength = 0;

    rc = sme_mgt_tspec(priv, InterfaceTag, CSR_WIFI_SME_LIST_ACTION_ADD, addts_tid,
                       &tspec, &tclas);

    CsrPmemFree(addts_ie);
    return rc;
}

int unifi_cfg_wmm_delts(os_linux_priv_t *priv, CsrUint16 InterfaceTag, unsigned char *arg)
{
    CsrUint32 delts_tid;
    CsrUint8 *delts_params;
    CsrWifiDataBlock tspec;
    CsrWifiDataBlock tclas;
    int rc;

    delts_params = (CsrUint8 *) (((unifi_cfg_command_t *) arg) + 1);
    if (get_user(delts_tid, (CsrUint32 *) delts_params))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_cfg_wmm_delts: Failed to get the argument\n", priv ? priv->instance : 0));
        return -EFAULT;
    }

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: delts: tid = 0x%x\n",
                           priv ? priv->instance : 0,  delts_tid));

    tspec.data = tclas.data = NULL;
    tspec.dataLength = tclas.dataLength = 0;

    rc = sme_mgt_tspec(priv, InterfaceTag, CSR_WIFI_SME_LIST_ACTION_REMOVE, delts_tid,
                       &tspec, &tclas);

    return rc;
}

int unifi_cfg_strict_draft_n(os_linux_priv_t *priv, CsrUint16 InterfaceTag, unsigned char *arg)
{
    CsrWifiDataBlock mibData = {0, NULL};
    CsrBool strict_draft_n;
    CsrUint8 *strict_draft_n_params;
    int rc;
    CsrUint8 *data;

    strict_draft_n_params = (CsrUint8 *) (((unifi_cfg_command_t *) arg) + 1);
    if (get_user(strict_draft_n, (CsrBool *) strict_draft_n_params))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_cfg_strict_draft_n: Failed to get the argument\n", priv ? priv->instance : 0));
        return -EFAULT;
    }

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: strict_draft_n: = %s\n",
                           priv ? priv->instance : 0,  ((strict_draft_n) ? "yes" : "no")));

    if (CsrWifiMibEncodeBool(&mibData, CSR_WIFI_SME_PSID_ENABLE_STRICT_RULES, strict_draft_n, 0) != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_cfg_strict_draft_n: Get unifi_SMEConfigValue failed.\n", priv ? priv->instance : 0));
        return -EFAULT;
    }
    data = mibData.data; /* need to free this later */
    rc = sme_mgt_mib_set(priv, InterfaceTag, &mibData);
    CsrPmemFree(data);
    if (rc)
    {
        CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_cfg_strict_draft_n: Set unifi_SMEConfigValue failed.\n", priv ? priv->instance : 0));
    }

    return rc;
}

int unifi_cfg_get_info(os_linux_priv_t *priv, CsrUint16 InterfaceTag, unsigned char *arg)
{
    CsrWifiDataBlock mibData = {0, NULL};
    unifi_cfg_get_t get_cmd;
    char inst_name[IFNAMSIZ];
    int rc;

    if (get_user(get_cmd, (unifi_cfg_get_t *) (((unifi_cfg_command_t *) arg) + 1)))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: UNIFI_CFG: Failed to get the argument\n", priv ? priv->instance : 0));
        return -EFAULT;
    }

    switch (get_cmd)
    {
        case UNIFI_CFG_GET_COEX:
        {
            CsrWifiMibValue *mibValues;
            csr_wifi_cfg_coex_info_t coexInfo;
            CsrUint8 *data;

            CsrWifiMibGetEntry getValues[] =
            {
                {CSR_WIFI_SME_PSID_COEX_INFO_HAS_TRAFFIC_DATA,  {0, 0}},
                {CSR_WIFI_SME_PSID_CURRENT_TRAFFIC_TYPE,  {0, 0}},
                {CSR_WIFI_SME_PSID_CURRENT_PERIOD_MS,  {0, 0}},
                {CSR_WIFI_SME_PSID_CURRENT_POWER_SAVE,  {0, 0}},
                {CSR_WIFI_SME_PSID_COEX_INFO_HAS_BT_DEVICE,  {0, 0}},
                {CSR_WIFI_SME_PSID_COEX_INFO_CURRENT_BLACKOUT_DURATION_US,  {0, 0}},
                {CSR_WIFI_SME_PSID_COEX_INFO_CURRENT_BLACKOUT_PERIOD_US,  {0, 0}},
                {CSR_WIFI_HIP_PSID_UNIFI_COEX_SCHEME,  {0, 0}},
            };

            CsrWifiMibEncodeGet(&mibData, CSR_WIFI_SME_PSID_COEX_INFO_HAS_TRAFFIC_DATA, 0);
            CsrWifiMibEncodeGet(&mibData, CSR_WIFI_SME_PSID_CURRENT_TRAFFIC_TYPE, 0);
            CsrWifiMibEncodeGet(&mibData, CSR_WIFI_SME_PSID_CURRENT_PERIOD_MS, 0);
            CsrWifiMibEncodeGet(&mibData, CSR_WIFI_SME_PSID_CURRENT_POWER_SAVE, 0);
            CsrWifiMibEncodeGet(&mibData, CSR_WIFI_SME_PSID_COEX_INFO_HAS_BT_DEVICE, 0);
            CsrWifiMibEncodeGet(&mibData, CSR_WIFI_SME_PSID_COEX_INFO_CURRENT_BLACKOUT_DURATION_US, 0);
            CsrWifiMibEncodeGet(&mibData, CSR_WIFI_SME_PSID_COEX_INFO_CURRENT_BLACKOUT_PERIOD_US, 0);
            CsrWifiMibEncodeGet(&mibData, CSR_WIFI_HIP_PSID_UNIFI_COEX_SCHEME, 0);

            data = mibData.data; /* allocated by CsrWifiMibEncodeGet, and we need to free this later */
            rc = sme_mgt_mib_get(priv, InterfaceTag, &mibData);
            CsrPmemFree(data);

            if (rc)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: UNIFI_CFG: Get CSR_WIFI_SME_PSID_POWER_SAVE_LEVEL failed.\n", priv ? priv->instance : 0));
                return rc;
            }


            mibValues = CsrWifiMibDecodeGetList(&mibData,
                                                sizeof(getValues) / sizeof(CsrWifiMibGetEntry),
                                                getValues);

            coexInfo.hasTrafficData = mibValues[0].u.boolValue;
            coexInfo.currentTrafficType = mibValues[1].u.uintValue;
            coexInfo.currentPeriodMs = mibValues[2].u.uintValue;
            coexInfo.currentPowerSave = mibValues[3].u.uintValue;
            coexInfo.currentCoexLatencyMs = 0; /* There is no mapping MIB for this in SME */
            coexInfo.hasBtDevice = mibValues[4].u.boolValue;
            coexInfo.currentBlackOutDurationUs = mibValues[5].u.uintValue;
            coexInfo.currentCoexPeriodMs = coexInfo.currentBlackOutDurationUs / 1000;
            coexInfo.currentBlackOutPeriodUs = mibValues[6].u.uintValue;
            coexInfo.currentCoexScheme = mibValues[7].u.uintValue;

            CsrPmemFree(mibData.data);
            CsrPmemFree(mibValues);
            if (copy_to_user((void *) arg,
                             &coexInfo,
                             sizeof(csr_wifi_cfg_coex_info_t)))
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: UNIFI_CFG: Failed to copy the coex info\n", priv ? priv->instance : 0));
                return -EFAULT;
            }
            break;
        }
        case UNIFI_CFG_GET_POWER_MODE:
        {
            CsrWifiMibEntry mibValue;
            CsrWifiSmePowerSaveLevel powerSaveLevel;
            CsrUint8 *data;

            CsrWifiMibEncodeGet(&mibData, CSR_WIFI_SME_PSID_POWER_SAVE_LEVEL, 0);
            data = mibData.data; /* allocated by CsrWifiMibEncodeGet, and we need to free this later */
            rc = sme_mgt_mib_get(priv, InterfaceTag, &mibData);
            CsrPmemFree(data);
            if (rc)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: UNIFI_CFG: Get CSR_WIFI_SME_PSID_POWER_SAVE_LEVEL failed.\n", priv ? priv->instance : 0));
                return rc;
            }

            (void) CsrWifiMibDecode(mibData.data, &mibValue);
            CsrPmemFree(mibData.data);
            powerSaveLevel = (CsrWifiSmePowerSaveLevel) mibValue.value.u.uintValue;

            /* Copy the info to the out buffer */
            if (copy_to_user((void *) arg, &powerSaveLevel, sizeof(CsrWifiSmePowerSaveLevel)))
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: UNIFI_CFG: Failed to copy the power save info\n", priv ? priv->instance : 0));
                return -EFAULT;
            }
            break;
        }
        case UNIFI_CFG_GET_POWER_SUPPLY:
        {
            CsrWifiMibEntry mibValue;
            CsrWifiSmeHostPowerMode powerMode;
            CsrUint8 *data;

            CsrWifiMibEncodeGet(&mibData, CSR_WIFI_SME_PSID_HOST_POWER_MODE, 0);
            data = mibData.data; /* allocated by CsrWifiMibEncodeGet, and we need to free this later */
            rc = sme_mgt_mib_get(priv, InterfaceTag, &mibData);
            CsrPmemFree(data);
            if (rc)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: UNIFI_CFG: Get unifi_HostConfigValue failed.\n", priv ? priv->instance : 0));
                return rc;
            }

            (void) CsrWifiMibDecode(mibData.data, &mibValue);
            CsrPmemFree(mibData.data);
            powerMode = (CsrWifiSmeHostPowerMode) mibValue.value.u.uintValue;

            /* Copy the info to the out buffer */
            if (copy_to_user((void *) arg, &powerMode, sizeof(CsrWifiSmeHostPowerMode)))
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: UNIFI_CFG: Failed to copy the host power mode\n", priv ? priv->instance : 0));
                return -EFAULT;
            }
            break;
        }
        case UNIFI_CFG_GET_VERSIONS:
            break;
        case UNIFI_CFG_GET_INSTANCE:
        {
            uf_net_get_name(priv->netdev[InterfaceTag], &inst_name[0], sizeof(inst_name));

            /* Copy the info to the out buffer */
            if (copy_to_user((void *) arg, &inst_name[0], sizeof(inst_name)))
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: UNIFI_CFG: Failed to copy the instance name\n", priv ? priv->instance : 0));
                return -EFAULT;
            }
            break;
        }

        case UNIFI_CFG_GET_AP_CONFIG:
        {
#ifdef CSR_SUPPORT_WEXT_AP
            uf_cfg_ap_config_t cfg_ap_config;
            cfg_ap_config.channel = priv->ap_config.channel;
            cfg_ap_config.beaconInterval = priv->ap_mac_config.beaconInterval;
            cfg_ap_config.wmmEnabled = priv->ap_mac_config.wmmEnabled;
            cfg_ap_config.dtimPeriod = priv->ap_mac_config.dtimPeriod;
            cfg_ap_config.phySupportedBitmap = priv->ap_mac_config.phySupportedBitmap;
            if (copy_to_user((void *) arg, &cfg_ap_config, sizeof(uf_cfg_ap_config_t)))
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: UNIFI_CFG: Failed to copy the AP configuration\n", priv ? priv->instance : 0));
                return -EFAULT;
            }
#else
            return -EPERM;
#endif
            break;
        }


        default:
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_cfg_get_info: Unknown value.\n", priv ? priv->instance : 0));
            return -EINVAL;
    }

    return 0;
}

#ifdef CSR_SUPPORT_WEXT_AP
int unifi_cfg_set_ap_config(os_linux_priv_t *priv, unsigned char *arg)
{
    uf_cfg_ap_config_t cfg_ap_config;
    char *buffer;

    buffer = ((unsigned char *) arg) + sizeof(unifi_cfg_command_t) + sizeof(unsigned int);
    if (copy_from_user(&cfg_ap_config, (void *) buffer,
                       sizeof(uf_cfg_ap_config_t)))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: UNIFI_CFG: Failed to get the ap config struct\n", priv ? priv->instance : 0));
        return -EFAULT;
    }
    priv->ap_config.channel = cfg_ap_config.channel;
    priv->ap_mac_config.dtimPeriod = cfg_ap_config.dtimPeriod;
    priv->ap_mac_config.beaconInterval = cfg_ap_config.beaconInterval;
    priv->group_sec_config.apGroupkeyTimeout = cfg_ap_config.groupkeyTimeout;
    priv->group_sec_config.apStrictGtkRekey = cfg_ap_config.strictGtkRekeyEnabled;
    priv->group_sec_config.apGmkTimeout = cfg_ap_config.gmkTimeout;
    priv->group_sec_config.apResponseTimeout = cfg_ap_config.responseTimeout;
    priv->group_sec_config.apRetransLimit = cfg_ap_config.retransLimit;

    priv->ap_mac_config.shortSlotTimeEnabled = cfg_ap_config.shortSlotTimeEnabled;
    priv->ap_mac_config.ctsProtectionType = cfg_ap_config.ctsProtectionType;

    priv->ap_mac_config.wmmEnabled = cfg_ap_config.wmmEnabled;

    priv->ap_mac_config.apHtParams.rxStbc = cfg_ap_config.rxStbc;
    priv->ap_mac_config.apHtParams.rifsModeAllowed = cfg_ap_config.rifsModeAllowed;

    priv->ap_mac_config.phySupportedBitmap = cfg_ap_config.phySupportedBitmap;
    priv->ap_mac_config.maxListenInterval = cfg_ap_config.maxListenInterval;

    return 0;
}

#endif
#ifdef CSR_SUPPORT_WEXT

void uf_sme_config_wq(struct work_struct *work)
{
    /* Register to receive indications from the SME */
    CsrWifiSmeEventMaskSetReqSend(0, CSR_WIFI_SME_INDICATIONS_WIFIOFF |
                                  CSR_WIFI_SME_INDICATIONS_CONNECTIONQUALITY |
                                  CSR_WIFI_SME_INDICATIONS_MEDIASTATUS |
                                  CSR_WIFI_SME_INDICATIONS_MICFAILURE);
} /* uf_sme_config_wq() */

#endif /* CSR_SUPPORT_WEXT */


/*
 * ---------------------------------------------------------------------------
 *  uf_ta_ind_wq
 *
 *      Deferred work queue function to send Traffic Analysis protocols
 *      indications to the SME.
 *      These are done in a deferred work queue for two reasons:
 *       - the CsrWifiRouterCtrl...Send() functions are not safe for atomic context
 *       - we want to load the main driver data path as lightly as possible
 *
 *      The TA classifications already come from a workqueue.
 *
 *  Arguments:
 *      work    Pointer to work queue item.
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
void uf_ta_ind_wq(struct work_struct *work)
{
    struct ta_ind *ind = container_of(work, struct ta_ind, task);
    os_linux_priv_t *priv = container_of(ind, os_linux_priv_t, ta_ind_work);


    CsrWifiRouterCtrlTrafficProtocolIndSend(priv->CSR_WIFI_SME_IFACEQUEUE, 0,
                                            ind->interfaceTag,
                                            ind->packet_type,
                                            ind->direction,
                                            ind->src_addr);
    ind->in_use = 0;
} /* uf_ta_ind_wq() */

/*
 * ---------------------------------------------------------------------------
 *  uf_ta_sample_ind_wq
 *
 *      Deferred work queue function to send Traffic Analysis sample
 *      indications to the SME.
 *      These are done in a deferred work queue for two reasons:
 *       - the CsrWifiRouterCtrl...Send() functions are not safe for atomic context
 *       - we want to load the main driver data path as lightly as possible
 *
 *      The TA classifications already come from a workqueue.
 *
 *  Arguments:
 *      work    Pointer to work queue item.
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
void uf_ta_sample_ind_wq(struct work_struct *work)
{
    struct ta_sample_ind *ind = container_of(work, struct ta_sample_ind, task);
    os_linux_priv_t *priv = container_of(ind, os_linux_priv_t, ta_sample_ind_work);


    CsrWifiRouterCtrlTrafficSampleIndSend(priv->CSR_WIFI_SME_IFACEQUEUE, 0, ind->interfaceTag, ind->stats);

    ind->in_use = 0;
} /* uf_ta_sample_ind_wq() */

/*
 * ---------------------------------------------------------------------------
 *  uf_send_m4_ready_wq
 *
 *      Deferred work queue function to send M4 ReadyToSend inds to the SME.
 *      These are done in a deferred work queue for two reasons:
 *       - the CsrWifiRouterCtrl...Send() functions are not safe for atomic context
 *       - we want to load the main driver data path as lightly as possible
 *
 *  Arguments:
 *      work    Pointer to work queue item.
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
void uf_send_m4_ready_wq(struct work_struct *work)
{
    netInterface_priv_t *interfacePriv = container_of(work, netInterface_priv_t, send_m4_ready_task);
    CsrUint16 iface = interfacePriv->InterfaceTag;
    os_linux_priv_t *priv = interfacePriv->privPtr;
    CsrWifiMacAddress peer;
    unsigned long flags;

    func_enter();

    /* The peer address was stored in the signal */
    spin_lock_irqsave(&priv->m4_lock, flags);
    memcpy(peer.a, interfacePriv->m4_peer_mac_addr.a, sizeof(peer.a));
    spin_unlock_irqrestore(&priv->m4_lock, flags);

    /* Send a signal to SME */
    CsrWifiRouterCtrlM4ReadyToSendIndSend(priv->CSR_WIFI_SME_IFACEQUEUE, 0, iface, peer);

    UF_TRACE_MAC_UDBG1(priv, "uf_send_m4_ready_wq: M4ReadyToSendInd sent for peer", peer.a);

    func_exit();
} /* uf_send_m4_ready_wq() */
