/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2012
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"
#include "csr_wifi_hip_log_text.h"
#include "csr_wifi_hip.h"
#include "os_linux_priv.h"
#ifdef CSR_SUPPORT_WEXT_AP
#include "sme_csr/csr_wifi_sme_sef.h"
#endif

/*
#if defined CSR_WIFI_DRIVER_HYDRA && defined CSR_WIFI_DRIVER_USE_HYDRA_DRIVER
#include <hydra/service.h>
#include <hydra/service_driver.h>
#endif
*/

/*
 * This file implements the SME SYS API and contains the following functions:
 * CsrWifiRouterCtrlMediaStatusReqHandler()
 * CsrWifiRouterCtrlHipReqHandler()
 * CsrWifiRouterCtrlPortConfigureReqHandler()
 * CsrWifiRouterCtrlWifiOnReqHandler()
 * CsrWifiRouterCtrlWifiOffReqHandler()
 * CsrWifiRouterCtrlSuspendResHandler()
 * CsrWifiRouterCtrlResumeResHandler()
 * CsrWifiRouterCtrlQosControlReqHandler()
 * CsrWifiRouterCtrlConfigurePowerModeReqHandler()
 * CsrWifiRouterCtrlWifiOnResHandler()
 * CsrWifiRouterCtrlWifiOffRspHandler()
 * CsrWifiRouterCtrlMulticastAddressResHandler()
 * CsrWifiRouterCtrlTrafficConfigReqHandler()
 * CsrWifiRouterCtrlTrafficClassificationReqHandler()
 * CsrWifiRouterCtrlTclasAddReqHandler()
 * CsrWifiRouterCtrlTclasDelReqHandler()
 * CsrWifiRouterCtrlSetModeReqHandler()
 * CsrWifiRouterCtrlWapiBroadcastFilterReqHandler()
 * CsrWifiRouterCtrlWapiBroadcastReqHandler()
 * CsrWifiRouterCtrlInterfaceAddReqHandler()
 * CsrWifiRouterCtrlInterfaceDelReqHandler()
 * CsrWifiRouterCtrlVifAddReqHandler()
 * CsrWifiRouterCtrlVifDelReqHandler()
 */


void CsrWifiRouterCtrlMediaStatusReqHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
    os_linux_priv_t *priv = (os_linux_priv_t *) drvpriv;
    CsrWifiRouterCtrlMediaStatusReq *req = (CsrWifiRouterCtrlMediaStatusReq *) msg;
    netInterface_priv_t *interfacePriv = priv->interfacePriv[req->interfaceTag];
    unsigned long flags;

    if (priv->smepriv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiRouterCtrlMediaStatusReqHandler: invalid smepriv\n",
                               priv ? priv->instance : 0));
        return;
    }

    if (req->interfaceTag >= CSR_WIFI_MAX_INTERFACES)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiRouterCtrlMediaStatusReqHandler: invalid interfaceTag\n",
                               priv ? priv->instance : 0));
        return;
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3,
                        "unifi%d: CsrWifiRouterCtrlMediaStatusReqHandler: Mode = %d req->mediaStatus = %d\n",
                        priv ? priv->instance : 0, interfacePriv->interfaceMode, req->mediaStatus));

    if (interfacePriv->interfaceMode != CSR_WIFI_ROUTER_CTRL_MODE_AMP)
    {
        CsrWifiHipBulkDataParam bulk_data;

        bulk_data.d[0].data_length = 0;

        spin_lock_irqsave(&priv->m4_lock, flags);
        if (interfacePriv->m4_bulk_data.d[0].data_length > 0)
        {
            bulk_data = interfacePriv->m4_bulk_data;
            interfacePriv->m4_bulk_data.d[0].net_buf_length = 0;
            interfacePriv->m4_bulk_data.d[0].data_length = 0;
            interfacePriv->m4_bulk_data.d[0].os_data_ptr = NULL;
            interfacePriv->m4_bulk_data.d[0].os_net_buf_ptr = NULL;
            memset(&interfacePriv->m4_info, 0, sizeof(os_linux_store_m4_info));
        }
        spin_unlock_irqrestore(&priv->m4_lock, flags);

        if (bulk_data.d[0].data_length != 0)
        {
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5,
                                "unifi%d: CsrWifiRouterCtrlMediaStatusReqHandler: free M4\n",
                                priv ? priv->instance : 0));
            os_linux_net_data_free_all(priv, &bulk_data);
        }

        if ((req->mediaStatus == CSR_WIFI_SME_MEDIA_STATUS_CONNECTED) &&
            (interfacePriv->connected != UnifiConnected))
        {
            switch (interfacePriv->interfaceMode)
            {
                case CSR_WIFI_ROUTER_CTRL_MODE_AP:
                case CSR_WIFI_ROUTER_CTRL_MODE_P2PGO:
                    interfacePriv->connected = UnifiConnected;
                    netif_carrier_on(priv->netdev[req->interfaceTag]);
#ifdef CSR_SUPPORT_WEXT
                    wext_send_started_event(priv, req->interfaceTag);
#endif
                    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                                       "unifi%d: CsrWifiRouterCtrlMediaStatusReqHandler: AP/P2PGO setting netif_carrier_on\n",
                                       priv ? priv->instance : 0));
                    UF_NETIF_TX_WAKE_ALL_QUEUES(priv->netdev[req->interfaceTag]);
                    break;

                default:
#ifdef CSR_SUPPORT_WEXT
                    /* In the WEXT builds (sme and native), the userspace is not ready
                     * to process any EAPOL or WAPI packets, until it has been informed
                     * of the NETDEV_CHANGE.
                     */
                    if (interfacePriv->netdev_callback_registered && (interfacePriv->interfaceMode != CSR_WIFI_ROUTER_CTRL_MODE_P2PCLI))
                    {
                        interfacePriv->wait_netdev_change = TRUE;
                        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                                           "unifi%d: CsrWifiRouterCtrlMediaStatusReqHandler: waiting for NETDEV_CHANGE\n",
                                           priv ? priv->instance : 0));
                        /*
                         * Carrier can go to on, only after wait_netdev_change is set to TRUE.
                         * Otherwise there can be a race in uf_netdev_event().
                         */
                        netif_carrier_on(priv->netdev[req->interfaceTag]);

                        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                                           "unifi%d: CsrWifiRouterCtrlMediaStatusReqHandler: STA/P2PCLI setting netif_carrier_on\n",
                                           priv ? priv->instance : 0));
                    }
                    else
#endif
                    {
                        /* In the NME build, the userspace does not wait for the NETDEV_CHANGE
                         * so it is ready to process all the EAPOL or WAPI packets.
                         * At this point, we enable all the Tx queues, and we indicate any packets
                         * that are queued (and the respective port is opened).
                         */
                        CsrWifiMacAddress broadcast_address = {{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF}};
                        CsrResult result;
                        interfacePriv->connected = UnifiConnected;
                        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                                           "unifi%d: CsrWifiRouterMediaStatusReqHandler: UnifiConnected && netif_carrier_on\n",
                                           priv ? priv->instance : 0));
                        netif_carrier_on(priv->netdev[req->interfaceTag]);
                        UF_NETIF_TX_WAKE_ALL_QUEUES(priv->netdev[req->interfaceTag]);
                        result = CsrWifiHipCtrlPortQueueProcessReq(priv->hip_handle, req->interfaceTag, CSR_WIFI_HIP_PORT_TYPE_UNCONTROLLED, &broadcast_address, TRUE);
                        if (result != CSR_RESULT_SUCCESS)
                        {
                            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                                   "unifi%d: CsrWifiRouterCtrlMediaStatusReqHandler: Failed to process port type %u's queue\n",
                                                   priv ? priv->instance : 0, CSR_WIFI_HIP_PORT_TYPE_UNCONTROLLED));
                        }

                        result = CsrWifiHipCtrlPortQueueProcessReq(priv->hip_handle, req->interfaceTag, CSR_WIFI_HIP_PORT_TYPE_CONTROLLED, &broadcast_address, TRUE);
                        if (result != CSR_RESULT_SUCCESS)
                        {
                            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                                   "unifi%d: CsrWifiRouterCtrlMediaStatusReqHandler: Failed to process port type %u's queue\n",
                                                   priv ? priv->instance : 0, CSR_WIFI_HIP_PORT_TYPE_CONTROLLED));
                        }
                    }
                    break;
            }
        }
        else if ((req->mediaStatus == CSR_WIFI_SME_MEDIA_STATUS_CONNECTED) &&
                 (interfacePriv->interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_STA) &&
                 (interfacePriv->connected == UnifiConnected))
        {
            /* Waking up network interface queue when media status indication received as
             * connected and previous driver state is already connected,
             * as this is the cases for roaming where the STA-DUT roams from AP1 to AP2 without
             * transition to disconnected state
             */
            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                               "unifi%d: CsrWifiRouterCtrlMediaStatusReqHandler: STA wake all queues \n",
                               priv ? priv->instance : 0));
            UF_NETIF_TX_WAKE_ALL_QUEUES(priv->netdev[req->interfaceTag]);
        }

        if (req->mediaStatus == CSR_WIFI_SME_MEDIA_STATUS_DISCONNECTED)
        {
#ifdef CSR_SUPPORT_WEXT
            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                               "unifi%d: CsrWifiRouterMediaStatusReqHandler: cancel waiting for NETDEV_CHANGE\n",
                               priv ? priv->instance : 0));
            interfacePriv->wait_netdev_change = FALSE;
#endif
            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                               "unifi%d: CsrWifiRouterMediaStatusReqHandler: setting netif_carrier_off\n",
                               priv ? priv->instance : 0));
            netif_carrier_off(priv->netdev[req->interfaceTag]);
#ifdef CSR_SUPPORT_WEXT
            switch (interfacePriv->interfaceMode)
            {
                case CSR_WIFI_ROUTER_CTRL_MODE_AP:
                case CSR_WIFI_ROUTER_CTRL_MODE_P2PGO:
                    wext_send_started_event(priv, req->interfaceTag);
                    break;
                default:
                    break;
            }
#endif
            interfacePriv->connected = UnifiNotConnected;
        }
    }
    else
    {
        /* For AMP, just update the L2 connected flag */
        if (req->mediaStatus == CSR_WIFI_SME_MEDIA_STATUS_CONNECTED)
        {
            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                               "unifi%d: CsrWifiRouterCtrlMediaStatusReqHandler: AMP connected\n",
                               priv ? priv->instance : 0));
            interfacePriv->connected = UnifiConnected;
        }
        else
        {
            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                               "unifi%d: CsrWifiRouterCtrlMediaStatusReqHandler: AMP disconnected\n",
                               priv ? priv->instance : 0));
            interfacePriv->connected = UnifiNotConnected;
        }
    }
}

void CsrWifiRouterCtrlHipReqHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
    os_linux_priv_t *priv = drvpriv;
    CsrWifiRouterCtrlHipReq *hipreq = (CsrWifiRouterCtrlHipReq *) msg;
    CsrWifiHipBulkDataParam bulk;

    if (priv == NULL)
    {
        return;
    }

    CsrMemSet(&bulk, 0, sizeof bulk);

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG4, "unifi%d: CsrWifiRouterCtrlHipReqHandler: 0x04%X ---->\n", priv->instance,
                        *((CsrUint16 *) hipreq->mlmeCommand)));

    if (hipreq->dataRef1 && hipreq->dataRef1Length)
    {
        if (CSR_RESULT_SUCCESS != os_linux_net_data_malloc(priv, &bulk.d[0], hipreq->dataRef1Length))
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: signal not sent down - alloc failed for bulk data1\n", priv->instance));
            return;
        }
        CsrMemCpy(bulk.d[0].os_data_ptr, hipreq->dataRef1, hipreq->dataRef1Length);
    }

    if (hipreq->dataRef2)
    {
        if (CSR_RESULT_SUCCESS != os_linux_net_data_malloc(priv, &bulk.d[1], hipreq->dataRef2Length))
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: signal not sent down - alloc failed for bulk data2\n", priv->instance));
            os_linux_net_data_free(priv, &bulk.d[0]);
            return;
        }
        CsrMemCpy(bulk.d[1].os_data_ptr, hipreq->dataRef2, hipreq->dataRef2Length);
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3, "unifi%d: SME SEND: Signal 0x%.4X \n", priv->instance, *((CsrUint16 *) hipreq->mlmeCommand)));


    /*
     * The MSB of the sender ID needs to be set to the client ID.
     * The LSB is controlled by the SME.
     */
    hipreq->mlmeCommand[5] = (priv->sme_cli->sender_id >> 8) & 0xff;

    if (CSR_RESULT_SUCCESS != CsrWifiHipPackedSignalReq(priv->hip_handle, 0, hipreq->mlmeCommandLength, CsrMemDup(hipreq->mlmeCommand, hipreq->mlmeCommandLength), &bulk))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiRouterCtrlHipReqHandler: Failed to send signal\n", priv->instance));
        CsrWifiRouterCtrlWifiOffIndSend(priv->sme_synergy_sched_queue, 0, CSR_WIFI_SME_CONTROL_INDICATION_ERROR);
        return;
    }
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG4, "unifi%d: CsrWifiRouterCtrlHipReqHandler: <----\n",
                        priv ? priv->instance : 0));
}

#ifndef CSR_NATIVE_LINUX

static void uf_send_gratuitous_arp(os_linux_priv_t *priv, CsrUint16 interfaceTag)
{
    netInterface_priv_t *interfacePriv = priv->interfacePriv[interfaceTag];
    struct ethhdr ehdr;
    CsrResult result;
    struct sk_buff *skb;
    CsrWifiHipMaPacketPriority priority;
    CsrInt32 aid = 0;
    static const CsrUint8 arp_req[] = {0, 0x01, 0x08, 0, ETH_ALEN, sizeof interfacePriv->sta_ip_address, 0, 0x01};  /* arp req up to just before sender hw address */

    func_enter();

    /* ASCH-DL_20140821 OIL 147: countermeasure for CSR Nullpointer exception issue. */
    /* enough space must be allocated here for what is copied in below; there are no checks */
    /* skb = dev_alloc_skb(CSR_WIFI_HIP_80211_MAC_HEADER_MAX_LEN + CSR_WIFI_HIP_LLC_HEADER_LEN
                        + sizeof(arp_req) + ETH_ALEN + sizeof interfacePriv->sta_ip_address + ETH_ALEN + sizeof interfacePriv->sta_ip_address); */
    skb = __dev_alloc_skb(CSR_WIFI_HIP_80211_MAC_HEADER_MAX_LEN + CSR_WIFI_HIP_LLC_HEADER_LEN
                        + sizeof(arp_req) + ETH_ALEN + sizeof interfacePriv->sta_ip_address + ETH_ALEN + sizeof interfacePriv->sta_ip_address, GFP_KERNEL);

    if (!skb)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: in uf_send_gratuitous_arp alloc skb failed.\n", priv->instance));
        return;
    }
    skb_reserve(skb, CSR_WIFI_HIP_80211_MAC_HEADER_MAX_LEN + CSR_WIFI_HIP_LLC_HEADER_LEN);

    memcpy(skb_tail_pointer(skb), arp_req, sizeof(arp_req));
    skb_put(skb, sizeof arp_req);

    memcpy(skb_tail_pointer(skb), priv->netdev[interfaceTag]->dev_addr, ETH_ALEN); /* sender hardware address */
    skb_put(skb, ETH_ALEN);

    memcpy(skb_tail_pointer(skb), &interfacePriv->sta_ip_address, sizeof interfacePriv->sta_ip_address); /* sender protocol address */
    skb_put(skb, sizeof interfacePriv->sta_ip_address);

    memset(skb_tail_pointer(skb), 0xff, ETH_ALEN);
    skb_put(skb, ETH_ALEN);

    memcpy(skb_tail_pointer(skb), &interfacePriv->sta_ip_address, sizeof interfacePriv->sta_ip_address); /* target protocol address */
    skb_put(skb, sizeof interfacePriv->sta_ip_address);

    memset(ehdr.h_dest, 0xff, sizeof(ehdr.h_dest));
    memcpy(ehdr.h_source, priv->netdev[interfaceTag]->dev_addr, sizeof(ehdr.h_source));
    ehdr.h_proto = htons(ETH_P_ARP);

    result = CsrWifiHipPriorityGetReq(priv->hip_handle, interfaceTag, 0, skb->data, ehdr.h_dest, &priority, &aid);
    if (result == CSR_RESULT_SUCCESS)
    {
        skb->priority = priority;
        result = send_ma_pkt_request(priv, interfacePriv, skb, &ehdr);
        if (result != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to send gratuitous arp", priv->instance));
        }
    }
    else
    {
        CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to get priority for gratuitous arp", priv->instance));
    }
    func_exit();
}

#endif

static void portConfigureHandleDataPlane(os_linux_priv_t *priv, CsrUint16 interfaceTag, CsrWifiRouterCtrlPortAction portState, CsrWifiHipPortType portType, CsrWifiMacAddress *macAddress)
{
    if (CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_OPEN == portState)
    {
        /*
         * Ask stack to schedule for transmission any packets queued
         * while port was not open.
         * Use netif_schedule() instead of netif_wake_queue() because
         * transmission should be already enabled at this point. If it
         * is not, probably the interface is down and should remain as is.
         */
        uf_resume_data_plane(priv, interfaceTag, portType, macAddress);
    }
    else
    {
        /* Free all queued frames in HAL since the port has been set to something else
           than opened. The frames will not be indicated to higher layers */
        CsrResult ret;
        ret = CsrWifiHipCtrlPortQueueProcessReq(priv->hip_handle, interfaceTag, portType, macAddress, FALSE);
        if (ret != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi%d: CsrWifiRouterCtrlPortConfigureReqHandler: Failed to process port type %u's queue\n",
                                   priv ? priv->instance : 0, portType));
        }
    }
}

void CsrWifiRouterCtrlPortConfigureReqHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
    os_linux_priv_t *priv = drvpriv;
    CsrWifiRouterCtrlPortConfigureReq *req = (CsrWifiRouterCtrlPortConfigureReq *) msg;
    CsrResult result;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi : CsrWifiRouterCtrlPortConfigureReqHandler: Invalid priv.\n"));
        return;
    }

    if (req->interfaceTag >= CSR_WIFI_MAX_INTERFACES)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiRouterCtrlPortConfigureReqHandler: interfaceTag >= CSR_WIFI_MAX_INTERFACES.\n",
                               priv ? priv->instance : 0));
        return;
    }


    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                        "unifi%d: entering CsrWifiRouterCtrlPortConfigureReqHandler (%x %x %d)\n",
                        priv->instance, req->uncontrolledPortAction, req->controlledPortAction, req->setProtection));

    result = CsrWifiHipPortConfigureReq(priv->hip_handle, req->interfaceTag, req->uncontrolledPortAction, req->controlledPortAction, &req->macAddress, req->setProtection);
    if (CSR_RESULT_SUCCESS != result)
    {
        CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,  "unifi%d: failed HipPortConfigureReq\n", priv->instance));

        /* Free any queued frame in the HAL. The frames will not be indicated to higher layers */
        CsrWifiHipCtrlPortQueueProcessReq(priv->hip_handle, req->interfaceTag, CSR_WIFI_HIP_PORT_TYPE_UNCONTROLLED, &req->macAddress, FALSE);
        CsrWifiHipCtrlPortQueueProcessReq(priv->hip_handle, req->interfaceTag, CSR_WIFI_HIP_PORT_TYPE_CONTROLLED, &req->macAddress, FALSE);

        CsrWifiRouterCtrlPortConfigureCfmSend(msg->source, req->clientData, req->interfaceTag, result, req->macAddress);

        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3,
                            "unifi%d: leaving CsrWifiRouterCtrlPortConfigureReqHandler\n",
                            priv->instance));
        return;
    }


#ifndef CSR_NATIVE_LINUX
    /* If supplicant uses the subscription API, process the uncontrolled port as soon as the port is open.
       The MaPacket subscription API/supplicant is ready to transmit and receive frames - no reason to
       wait on the netdevice to be ready. */
    if (req->uncontrolledPortAction == CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_OPEN)
    {
        CsrUint8 oui[3] = {0, 0, 0};
        CsrUint32 i;

        for (i = 0; i < MAX_MA_PACKET_IND_FILTERS; i++)
        {
            if (priv->sme_ma_packet_ind_filters[i].in_use)
            {
                if (!memcmp(oui, priv->sme_ma_packet_ind_filters[i].oui, 3) &&
                    (ETH_P_PAE == priv->sme_ma_packet_ind_filters[i].protocol))
                {
                    CsrWifiHipCtrlPortQueueProcessReq(priv->hip_handle, req->interfaceTag, CSR_WIFI_HIP_PORT_TYPE_UNCONTROLLED, &req->macAddress, TRUE);
                    break;
                }
            }
        }
    }
#endif

    /* After having configured the (un)controlled ports, resume the data plane
     * and indicate any buffered frames received while the ports were in
     * blocked state. */
    portConfigureHandleDataPlane(priv, req->interfaceTag, req->uncontrolledPortAction, CSR_WIFI_HIP_PORT_TYPE_UNCONTROLLED, &req->macAddress);
    portConfigureHandleDataPlane(priv, req->interfaceTag, req->controlledPortAction, CSR_WIFI_HIP_PORT_TYPE_CONTROLLED, &req->macAddress);

#ifndef CSR_NATIVE_LINUX
    if (CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_OPEN == req->controlledPortAction)
    {
        netInterface_priv_t *interfacePriv = priv->interfacePriv[req->interfaceTag];

        if ((CSR_WIFI_ROUTER_CTRL_MODE_STA == interfacePriv->interfaceMode) && (interfacePriv->sta_ip_address != 0xFFFFFFFF))
        {
            uf_send_gratuitous_arp(priv, req->interfaceTag);
        }
    }
#endif

    CsrWifiRouterCtrlPortConfigureCfmSend(msg->source, req->clientData, req->interfaceTag, result, req->macAddress);
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3,
                        "unifi%d: leaving CsrWifiRouterCtrlPortConfigureReqHandler\n",
                        priv->instance));
}

void CsrWifiRouterCtrlWifiOnReqHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
    os_linux_priv_t *linux_priv = drvpriv;
    CsrWifiRouterCtrlWifiOnReq *req = (CsrWifiRouterCtrlWifiOnReq *) msg;
    CsrResult result;
    CsrWifiHipInstanceConfig config;
    CsrBool cmanrTestMode = FALSE;
    CsrWifiDataBlock buffer = {req->valuesLength, req->values};
    CsrWifiMibEntry decodedMib;
    CsrSize bufferIndex = 0;

    if (linux_priv == NULL)
    {
        CsrWifiRouterCtrlWifiOnCfmSend(msg->source, req->clientData, CSR_RESULT_FAILURE);
        return;
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                        "unifi%d: CsrWifiRouterCtrlWifiOnReqHandler: SME appHandle = 0x%.4X valuesLength=%d values=0x%x\n",
                        linux_priv->instance, msg->source, req->valuesLength, req->values));


    if (linux_priv->state != OS_LINUX_STATE_ACTIVATED)
    {
        CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                            "unifi%d: CsrWifiRouterCtrlWifiOnReqHandler: os_linux was not in activated state (%d)\n",
                            linux_priv->instance, linux_priv->state));

        CsrWifiRouterCtrlWifiOnCfmSend(msg->source, req->clientData, CSR_RESULT_FAILURE);
        return;
    }

#ifdef ANDROID_BUILD
    /* Take the wakelock while Wi-Fi On is in progress */
    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi%d: CsrWifiRouterCtrlWifiOnReqHandler: take wakelock", linux_priv->instance));
    wake_lock(&unifi_sdio_wake_lock);
#endif

    linux_priv->state = OS_LINUX_STATE_WIFI_ON_IN_PROGRESS;

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: CsrWifiRouterCtrlWifiOnReqHandler: SME appHandle = 0x%.4X valuesLength=%d values=0x%x\n"
                           , linux_priv->instance, msg->source, req->valuesLength, req->values));

#if defined CSR_WIFI_DRIVER_HYDRA && defined CSR_WIFI_DRIVER_USE_HYDRA_DRIVER
    /* Start the Hydra WLAN subsystem */
    result = uf_start_hydra_wlan_service(linux_priv);
    if (result != CSR_RESULT_SUCCESS)
    {
        linux_priv->state = OS_LINUX_STATE_ACTIVATED;
        CsrWifiRouterCtrlWifiOnCfmSend(msg->source, req->clientData, CSR_RESULT_FAILURE);
        return;
    }
#endif

    CsrMemSet(&config, 0, sizeof(CsrWifiHipInstanceConfig));

    while (bufferIndex < buffer.dataLength)
    {
        bufferIndex += CsrWifiMibDecode(&buffer.data[bufferIndex], &decodedMib);
        CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: PSID=0x%x Value=%d\n", linux_priv->instance,
                              decodedMib.psid,
                              decodedMib.value.u.uintValue
                              ));
        switch (decodedMib.psid)
        {
            case CSR_WIFI_SME_PSID_ENABLE_STRICT_RULES:
                if (decodedMib.value.u.uintValue & CSR_WIFI_STRICT_RULES_ENABLE_CMANR_TEST_MODE)
                {
                    cmanrTestMode = TRUE;
                    CSR_LOG_TEXT_DEBUG((
                                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: CsrWifiRouterCtrlWifiOnReqHandler: cmanrTestMode=%d\n"
                                           , linux_priv ? linux_priv->instance : 0, cmanrTestMode));
                }
                else
                {
                    cmanrTestMode = FALSE;
                }
                break;
            case CSR_WIFI_ROUTER_PSID_ZERO_COPY_MODE:
                config_params.zero_copy = decodedMib.value.u.uintValue;
                break;
            case CSR_WIFI_ROUTER_PSID_POLL_PERIOD:
                config_params.poll_period = decodedMib.value.u.uintValue;
                break;
            case CSR_WIFI_ROUTER_PSID_TX_WINDOW_SEGMENT_SIZE:
                config_params.tx_window_segment_size = decodedMib.value.u.uintValue;
                break;
            case CSR_WIFI_ROUTER_PSID_TX_FORCE_PULL_MODE:
                config_params.tx_force_pull_mode = decodedMib.value.u.uintValue;
                break;
            case CSR_WIFI_ROUTER_PSID_MPARAM_SLOT_COUNT1:
                config_params.mparam_slot_count[0] = decodedMib.value.u.uintValue;
                break;
            case CSR_WIFI_ROUTER_PSID_MPARAM_SLOT_COUNT2:
                config_params.mparam_slot_count[1] = decodedMib.value.u.uintValue;
                break;
            case CSR_WIFI_ROUTER_PSID_MIN_VIF_DURATION_EAPOL:
                if (decodedMib.value.u.uintValue)
                {
                    config_params.minVifDurationEapol = decodedMib.value.u.uintValue;
                    config_params.minVifDurationPacketFilter = config_params.minVifDurationPacketFilter | CSR_WIFI_ROUTER_CTRL_TRAFFIC_PACKET_TYPE_EAPOL;
                }
                break;
            case CSR_WIFI_ROUTER_PSID_MIN_VIF_DURATION_DHCP:
                if (decodedMib.value.u.uintValue)
                {
                    config_params.minVifDurationDhcp = decodedMib.value.u.uintValue;
                    config_params.minVifDurationPacketFilter = config_params.minVifDurationPacketFilter | CSR_WIFI_ROUTER_CTRL_TRAFFIC_PACKET_TYPE_DHCP;
                }
                break;
            case CSR_WIFI_ROUTER_PSID_MIN_VIF_DURATION_ARP:
                if (decodedMib.value.u.uintValue)
                {
                    config_params.minVifDurationArp = decodedMib.value.u.uintValue;
                    config_params.minVifDurationPacketFilter = config_params.minVifDurationPacketFilter | CSR_WIFI_ROUTER_CTRL_TRAFFIC_PACKET_TYPE_ARP;
                }
                break;
            default:
                CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Unknown MIB PSID: %x\n", linux_priv->instance, decodedMib.psid));
                break;
        }
    }

    /* Save the Synergy Scheduler queue for the SME */
    linux_priv->sme_synergy_sched_queue = msg->source;
    config.hipMode = CSR_WIFI_HIP_WIFI_STACK_MODE;
    config.allocFunc = os_linux_net_data_malloc;
    config.freeFunc = os_linux_net_data_free;
    config.flowControlPauseCb = os_linux_flow_control_pause_cb;
    config.flowControlResumeCb = os_linux_flow_control_resume_cb;
    if ((linux_priv->instance >= 0) && (linux_priv->instance < MAX_UNIFI_DEVS))
    {
        config.fw_init = fw_init[linux_priv->instance];
    }
    else
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiRouterCtrlWifiOnReqHandler: The instance number in os_linux priv is invalid (%u). Defaulting to instance 0\n",
                               linux_priv->instance, linux_priv->instance));
        config.fw_init = fw_init[0];
    }
    config.cardParams = config_params;
    config.cmanrTestMode = cmanrTestMode;

    result = CsrWifiHipWifiOnReq(linux_priv->hip_handle,
                                 linux_priv,
                                 msg->source,
                                 config);
    if (result != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiRouterCtrlWifiOnReqHandler: Failed to do WifiOnReq (%u)\n",
                               linux_priv->instance, result));
#if defined CSR_WIFI_DRIVER_HYDRA && defined CSR_WIFI_DRIVER_USE_HYDRA_DRIVER
        uf_stop_hydra_wlan_service(linux_priv);
#endif
        linux_priv->state = OS_LINUX_STATE_ACTIVATED;
        CsrWifiRouterCtrlWifiOnCfmSend(msg->source, req->clientData, CSR_RESULT_FAILURE);
        return;
    }

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG3,  "unifi%d: CsrWifiRouterCtrlWifiOnReqHandler: WifiOnReq in progress\n"
                           , linux_priv ? linux_priv->instance : 0));
}

void CsrWifiHipWifiOnInd(void *osLayerContext, CsrResult result, CsrUint32 chipId, CsrUint32 chipVersion, CsrUint32 firmwareBuild,
                         CsrUint32 firmwareHip, CsrCharString *routerBuild, CsrUint32 routerHip, CsrUint32 sdioBlockSize)
{
    os_linux_priv_t *os_linux = (os_linux_priv_t *) osLayerContext;
    CsrWifiRouterCtrlVersions versions;

    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG2,  "unifi%d: CsrWifiHipWifiOnInd received WifiOnInd\n",
                       os_linux ? os_linux->instance : 0));

    /* Parse the version information on to the SME. */
    versions.chipId = chipId;
#ifndef CSR_WIFI_DRIVER_HYDRA
    versions.chipVersion = chipVersion;
#else
    versions.chipVersion = 0x0044;
#endif
    versions.firmwareBuild = firmwareBuild;
    versions.firmwareHip = firmwareHip;
    versions.routerBuild = routerBuild;
    versions.routerHip = routerHip;
    os_linux->fw_build = firmwareBuild;

    os_linux->sdio_block_size = sdioBlockSize;
    os_linux->state = OS_LINUX_STATE_WIFI_ON_DONE;
    CsrWifiRouterCtrlWifiOnIndSend(os_linux->sme_synergy_sched_queue, 0, CSR_RESULT_SUCCESS, versions, 0);
}

/*
 * os_linux_wifi_off
 *      The common os specific WiFi off handling happens here.
 */
static void os_linux_wifi_off(os_linux_priv_t *priv)
{
    int i;


    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                       "unifi%d: os_linux specific WiFi off steps\n",
                       priv ? priv->instance : 0));

    /* Destroy the Traffic Analysis Module */
    cancel_work_sync(&priv->ta_ind_work.task);
    cancel_work_sync(&priv->ta_sample_ind_work.task);

#ifdef CSR_SUPPORT_WEXT
    cancel_work_sync(&priv->sme_config_task);
#endif

    /* Cancel pending M4 stuff */
    for (i = 0; i < CSR_WIFI_MAX_INTERFACES; i++)
    {
        if (priv->netdev[i])
        {
            netInterface_priv_t *interfacePriv = (netInterface_priv_t *) netdev_priv(priv->netdev[i]);
            cancel_work_sync(&interfacePriv->send_m4_ready_task);
            interfacePriv->connected = UnifiNotConnected;
#ifdef CSR_SUPPORT_WEXT
            if (interfacePriv->interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_STA)
            {
                wext_send_disassoc_event(priv, i);
            }
#endif
            interfacePriv->interfaceMode = CSR_WIFI_ROUTER_CTRL_MODE_NONE;
        }
    }

    flush_workqueue(priv->unifi_workqueue);

#if defined CSR_WIFI_DRIVER_HYDRA && defined CSR_WIFI_DRIVER_USE_HYDRA_DRIVER
    /* Stop the Hydra WLAN subsystem firmware */
    uf_stop_hydra_wlan_service(priv);
#endif

    /* Consider UniFi to be uninitialised */
    priv->init_progress = UNIFI_INIT_NONE;
} /* os_linux_wifi_off() */

/*
 * hal_wifi_off
 *      The common HAL WiFi off handling happens here.
 */
static void hal_wifi_off(os_linux_priv_t *priv)
{
    CsrUint8 i;
    CsrWifiMacAddress nullMac = {{0x00, 0x00, 0x00, 0x00, 0x00, 0x00}};

    /* Stop the network traffic. */
    for (i = 0; i < CSR_WIFI_MAX_INTERFACES; i++)
    {
        netInterface_priv_t *interfacePriv = priv->interfacePriv[i];

        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                           "unifi%d: hal_wifi_off: Handling interface %u (%u)\n",
                           priv ? priv->instance : 0, i, interfacePriv->netdev_state));

        if ((interfacePriv->netdev_state == OS_LINUX_NETDEV_STATE_REGISTERED) ||
            (interfacePriv->netdev_state == OS_LINUX_NETDEV_STATE_REGISTERED_STOPPED))
        {
            /* CsrWifiHipInterfaceDelReq will turn the net_carrier off and pause
                all the transmit queues for the netdev, but it won't unregister
                the netdev since it can confuse wpa_supplicants under some
                circumstances.
                The supplicant or the 'user' will use ifconfig down as required. */
            CsrWifiHipInterfaceDelReq(priv->hip_handle, i);
        }

        /* Each VIF has to be set to mode none to ensure that
           VIF tables etc. are flushed */
        CsrWifiHipVifModeSetReq(priv->hip_handle, i,
                                CSR_WIFI_ROUTER_CTRL_MODE_NONE,
                                nullMac, FALSE, FALSE, 0);
    }

    if (priv->hip_handle != NULL)
    {
        CsrWifiHipWifiOffReq(TRUE, priv->hip_handle);
    }
}

void CsrWifiRouterCtrlWifiOffReqHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
    os_linux_priv_t *priv = drvpriv;
    CsrWifiRouterCtrlWifiOffReq *req = (CsrWifiRouterCtrlWifiOffReq *) msg;

    if (priv == NULL)
    {
        return;
    }

    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                       "unifi%d: CsrWifiRouterCtrlWifiOffReqHandler(0x%.4X)\n",
                       priv ? priv->instance : 0,  msg->source));

    if ((priv->state != OS_LINUX_STATE_IDLE) && (priv->state != OS_LINUX_STATE_ACTIVATED))
    {
        hal_wifi_off(priv);
        os_linux_wifi_off(priv);
        priv->state = OS_LINUX_STATE_ACTIVATED;

        CsrWifiRouterCtrlWifiOffCfmSend(msg->source, req->clientData);

        /* If this is called in response to closing the character device, the
         * caller must use uf_sme_cancel_request() to terminate any pending SME
         * blocking request or there will be a delay while the operation times out.
         */

#if defined CSR_WIFI_DRIVER_HYDRA && defined CSR_WIFI_DRIVER_USE_HYDRA_DRIVER
        /* Stop Hydra WLAN subsystem */
        uf_stop_hydra_wlan_service(priv);
#endif
    }
    else
    {
        CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                              "unifi%d: CsrWifiRouterCtrlWifiOffReqHandler: State/event error: Received WifiOffReq in state %u\n",
                              priv ? priv->instance : 0, priv->state));
    }
}

void CsrWifiRouterCtrlQosControlReqHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
    os_linux_priv_t *priv = drvpriv;
    CsrWifiRouterCtrlQosControlReq *req = (CsrWifiRouterCtrlQosControlReq *) msg;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiRouterCtrlQosControlReqHandler: invalid priv\n",
                               priv ? priv->instance : 0));
        return;
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,
                        "unifi%d: CsrWifiRouterCtrlQosControlReqHandler: control=%d, queueConfig=%d",
                        priv ? priv->instance : 0,  req->control, req->queueConfig));

    if (req->interfaceTag >= CSR_WIFI_MAX_INTERFACES)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiRouterCtrlQosControlReqHandler: interfaceID >= CSR_WIFI_MAX_INTERFACES.\n",
                               priv ? priv->instance : 0));
        return;
    }

    CsrWifiHipQosControlReq(priv->hip_handle, req->interfaceTag, req->control, req->queueConfig, req->queueConfigs);
}

void CsrWifiRouterCtrlTclasAddReqHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
    os_linux_priv_t *priv = drvpriv;
    CsrWifiRouterCtrlTclasAddReq *req = (CsrWifiRouterCtrlTclasAddReq *) msg;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiRouterCtrlTclasAddReqHandler: invalid smepriv\n", priv ? priv->instance : 0));
        return;
    }

    CsrWifiRouterCtrlTclasAddCfmSend(msg->source, req->clientData, req->interfaceTag, CSR_RESULT_SUCCESS);
}

void CsrWifiRouterCtrlTclasDelReqHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
    os_linux_priv_t *priv = drvpriv;
    CsrWifiRouterCtrlTclasDelReq *req = (CsrWifiRouterCtrlTclasDelReq *) msg;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiRouterCtrlTclasDelReqHandler: invalid smepriv\n", priv ? priv->instance : 0));
        return;
    }

    CsrWifiRouterCtrlTclasDelCfmSend(msg->source, req->clientData, req->interfaceTag, CSR_RESULT_SUCCESS);
}

void CsrWifiRouterCtrlConfigurePowerModeReqHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
    os_linux_priv_t *priv = drvpriv;
    CsrWifiRouterCtrlConfigurePowerModeReq *req = (CsrWifiRouterCtrlConfigurePowerModeReq *) msg;
    CsrWifiRouterCtrlLowPowerMode lowPowerMode;
    CsrBool wakeHost;

    lowPowerMode = req->mode;
    wakeHost = req->wakeHost;

    (void) CsrWifiHipLowPowerModeReq(priv->hip_handle, lowPowerMode, wakeHost);

    CSR_LOG_TEXT_INFO((
                          CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG2,
                          "unifi%d: CsrWifiRouterCtrlConfigurePowerModeReqHandler (mode=%d, wake=%d)\n",
                          priv ? priv->instance : 0,
                          req->mode, req->wakeHost));
}

void CsrWifiRouterCtrlWifiOnResHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
    os_linux_priv_t *priv = drvpriv;
    CsrWifiRouterCtrlWifiOnRes *res = (CsrWifiRouterCtrlWifiOnRes *) msg;
    CsrResult result;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi : CsrWifiRouterCtrlWifiOnResHandler: Invalid ospriv.\n"));
        goto error;
    }

    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                       "unifi%d: CsrWifiRouterCtrlWifiOnResHandler: status %d (Firmware patch %u)\n",
                       priv->instance,
                       res->status, res->smeVersions.firmwarePatch));

    /* UniFi is now initialised, complete the init. */
    if (res->status == CSR_RESULT_SUCCESS)
    {
        CsrBool intmode = FALSE;

        /* Copy version structure into the private versions field */
        priv->sme_versions = res->smeVersions;

        /* If the MIB has selected f/w scheduled interrupt mode, apply it now
         * but let module param override.
         */
        if (run_bh_once != -1)
        {
            intmode = TRUE;
        }
        else if (res->scheduledInterrupt)
        {
            intmode = TRUE;
        }

        result = CsrWifiHipPostInitReq(priv->hip_handle, intmode);
        if (result != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiRouterCtrlWifiOnResHandler: Failed to perform HAL post init.\n", priv->instance));
            goto error;
        }

        /* Acknowledge the CsrWifiRouterCtrlWifiOnReq now */
        CsrWifiRouterCtrlWifiOnCfmSend(msg->source, res->clientData, CSR_RESULT_SUCCESS);

        /* The first MLME-RESET.cfm has been processed, declare UniFi ready */
        priv->state = OS_LINUX_STATE_UF_READY;

        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                               "unifi%d: UniFi ready\n", priv->instance)); /* SME */
#ifdef ANDROID_BUILD
        /* Release the initialisation/resume wakelock */
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi%d: Got RouterCtrlWifiOn.resp so release wakelock", priv->instance));
        wake_unlock(&unifi_sdio_wake_lock);
#endif
        return;
    }
    else
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiRouterCtrlWifiOnResHandler: Status %u\n", priv->instance, res->status));
        goto error;
    }


error:
    {
        hal_wifi_off(priv);
        os_linux_wifi_off(priv);
        priv->state = OS_LINUX_STATE_ACTIVATED;
        CsrWifiRouterCtrlWifiOnCfmSend(msg->source, res->clientData, CSR_RESULT_FAILURE);
    }
}

void CsrWifiRouterCtrlWifiOffResHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
    os_linux_priv_t *priv = drvpriv;

    if ((priv->state != OS_LINUX_STATE_ACTIVATED) &&
        (priv->state != OS_LINUX_STATE_IDLE))
    {
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                           "unifi%d: CsrWifiRouterCtrlWifiOffResHandler (0x%.4X)\n",
                           priv ? priv->instance : 0, msg->source));

        hal_wifi_off(priv);
        os_linux_wifi_off(priv);
        priv->state = OS_LINUX_STATE_ACTIVATED;
    }
    else
    {
        CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                              "unifi%d: CsrWifiRouterCtrlWifiOffResHandler: State/event error: Received WifiOffRes in state %u\n",
                              priv ? priv->instance : 0, priv->state));
    }
}

void CsrWifiRouterCtrlMulticastAddressResHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
}

void CsrWifiRouterMaPacketSubscribeReqHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
    os_linux_priv_t *priv = drvpriv;
    CsrWifiRouterMaPacketSubscribeReq *req = (CsrWifiRouterMaPacketSubscribeReq *) msg;
    CsrUint8 i;
    CsrResult result;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiRouterMaPacketSubscribeReqHandler: invalid priv\n", priv ? priv->instance : 0));
        return;
    }

    /* Look for an unused filter */

    result = CSR_WIFI_RESULT_NO_ROOM;
    for (i = 0; i < MAX_MA_PACKET_IND_FILTERS; i++)
    {
        if (!priv->sme_ma_packet_ind_filters[i].in_use)
        {
            priv->sme_ma_packet_ind_filters[i].in_use = 1;
            priv->sme_ma_packet_ind_filters[i].appHandle = msg->source;
            priv->sme_ma_packet_ind_filters[i].encapsulation = req->encapsulation;
            priv->sme_ma_packet_ind_filters[i].protocol = req->protocol;

            priv->sme_ma_packet_ind_filters[i].oui[2] = (CsrUint8) (req->oui & 0xFF);
            priv->sme_ma_packet_ind_filters[i].oui[1] = (CsrUint8) ((req->oui >> 8) & 0xFF);
            priv->sme_ma_packet_ind_filters[i].oui[0] = (CsrUint8) ((req->oui >> 16) & 0xFF);

            priv->num_of_ma_packet_subscriptions++;

            result = CSR_RESULT_SUCCESS;
            break;
        }
    }

    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                       "unifi%d: CsrWifiRouterMaPacketSubscribeReqHandler: encap=%d, handle=%d, prot %d, result=%d\n",
                       priv ? priv->instance : 0,
                       req->encapsulation, i, req->protocol, result));
    CsrWifiRouterMaPacketSubscribeCfmSend(msg->source, req->interfaceTag, i, result, 0);
}

void CsrWifiRouterMaPacketUnsubscribeReqHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
    os_linux_priv_t *priv = drvpriv;
    CsrWifiRouterMaPacketUnsubscribeReq *req = (CsrWifiRouterMaPacketUnsubscribeReq *) msg;
    CsrResult result;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiRouterMaPacketUnsubscribeReqHandler: invalid priv\n", priv ? priv->instance : 0));
        return;
    }

    result = CSR_WIFI_RESULT_NOT_FOUND;

    if (req->subscriptionHandle < MAX_MA_PACKET_IND_FILTERS)
    {
        if (priv->sme_ma_packet_ind_filters[req->subscriptionHandle].in_use)
        {
            priv->sme_ma_packet_ind_filters[req->subscriptionHandle].in_use = 0;
            priv->num_of_ma_packet_subscriptions ? priv->num_of_ma_packet_subscriptions-- : 0;
            result = CSR_RESULT_SUCCESS;
        }
        else
        {
            result = CSR_WIFI_RESULT_NOT_FOUND;
        }
    }

    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                       "unifi%d: unsubscribe_req: handle=%d, result=%d\n",
                       priv ? priv->instance : 0,
                       req->subscriptionHandle, result));
    CsrWifiRouterMaPacketUnsubscribeCfmSend(msg->source, req->interfaceTag, result);
}

void CsrWifiRouterCtrlCapabilitiesReqHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
    os_linux_priv_t *priv = drvpriv;

    CsrWifiRouterCtrlCapabilitiesReq *req = (CsrWifiRouterCtrlCapabilitiesReq *) msg;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiRouterCtrlCapabilitiesReqHandler: invalid priv\n", priv ? priv->instance : 0));
        return;
    }
    CsrWifiHipCapabilitiesReq(priv->hip_handle, msg->source, req->clientData);
}

void CsrWifiRouterCtrlSuspendResHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
    os_linux_priv_t *priv = drvpriv;
    CsrWifiRouterCtrlSuspendRes *res = (CsrWifiRouterCtrlSuspendRes *) msg;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi: CsrWifiRouterCtrlSuspendResHandler: invalid priv\n"));
        return;
    }
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi%d: Got RouterCtrl Suspend response with result %u", priv->instance, res->status));

    if ((priv->expected_reply == SME_REPLY_SUSPEND) && (priv->sme_reply.request_status == SME_REQUEST_PENDING))
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi%d: Got RouterCtrl Suspend response for fake suspend", priv->instance));

        /* a thread is blocked in sme_sys_suspend() waiting for this response */
        (void) sme_complete_request(priv, res->status);
    }
    else
    {
        /* This is a response to a true suspend ind from the HIP */
        CsrWifiHipSuspendRes(priv->hip_handle, res->status);
    }
}

void CsrWifiRouterCtrlResumeResHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
    os_linux_priv_t *priv = drvpriv;
    CsrWifiRouterCtrlResumeRes *res = (CsrWifiRouterCtrlResumeRes *) msg;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi: CsrWifiRouterCtrlResumeResHandler: invalid priv\n"));
        return;
    }
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi%d: Got RouterCtrl Resume response with result %u", priv->instance, res->status));

    if ((priv->expected_reply == SME_REPLY_RESUME) && (priv->sme_reply.request_status == SME_REQUEST_PENDING))
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi%d: Got RouterCtrl Resume response for fake resume", priv->instance));

        /* a thread is blocked in sme_sys_resume() waiting for this response */
        (void) sme_complete_request(priv, res->status);
    }
#ifdef CSR_WIFI_HIP_SUSPEND_ENABLE
    else
    {
        /* This is a response to a true resume ind from the HIP */
        CsrWifiHipResumeRes(priv->hip_handle, CSR_RESULT_SUCCESS); /* always succeeds */

#ifdef CSR_WIFI_HIP_SUSPEND_WOL
#ifdef ANDROID_BUILD
        /* In the case of cold suspend the lock will instead be released in the wifi-on */
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1, "unifi%d: Got RouterCtrlWifiResume.resp so release wakelock", priv->instance));
        wake_unlock(&unifi_sdio_wake_lock);
#endif
#else
        /* cold suspend */
        priv->init_progress = priv->init_progress_save;
#endif
    }
#endif
}

void CsrWifiRouterCtrlTrafficConfigReqHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
    os_linux_priv_t *priv = drvpriv;
    CsrWifiRouterCtrlTrafficConfigReq *req = (CsrWifiRouterCtrlTrafficConfigReq *) msg;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi : CsrWifiRouterCtrlTrafficConfigReqHandler: invalid priv\n"));
        return;
    }
    if (req->trafficConfigType == CSR_WIFI_ROUTER_CTRL_TRAFFIC_CONFIG_TYPE_FILTER)
    {
        req->config.packetFilter |= CSR_WIFI_ROUTER_CTRL_TRAFFIC_PACKET_TYPE_CUSTOM;
    }
    if (CSR_RESULT_SUCCESS != CsrWifiHipTaConfigureReq(priv->hip_handle, req->trafficConfigType, req->config))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiRouterCtrlTrafficClassificationReqHandler: hip req failed\n", priv->instance));
    }
}

void CsrWifiRouterCtrlTrafficClassificationReqHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
    os_linux_priv_t *priv = drvpriv;
    CsrWifiRouterCtrlTrafficClassificationReq *req = (CsrWifiRouterCtrlTrafficClassificationReq *) msg;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiRouterCtrlTrafficClassificationReqHandler: invalid smepriv\n", priv ? priv->instance : 0));
        return;
    }

    if (CSR_RESULT_SUCCESS != CsrWifiHipTaClassificationReq(priv->hip_handle, req->trafficType, req->period))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipTaClassificationReq: hip req failed\n", priv->instance));
    }
}

void CsrWifiRouterMaPacketReqHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
    os_linux_priv_t *os_linux = (os_linux_priv_t *) drvpriv;
    CsrWifiRouterMaPacketReq *request = (CsrWifiRouterMaPacketReq *) msg;
    CsrUint16 interfaceTag = request->interfaceTag & 0x00ff;
    CsrResult result;
    CsrUint8 *daddr, *saddr, macHeaderOffset;
    CsrBool eapolStore = FALSE;
    netInterface_priv_t *interfacePriv;
    const sme_ma_packet_ind_filter_t *subscription;
    unsigned long spin_lock_flag;
    CsrWifiHipBulkDataParam bulkdata;
    CsrWifiHipQosType qosType;
    CsrWifiHipMaPacketPriority priority = CSR_WIFI_HIP_MA_PACKET_PRIORITY_CONTENTION;
    CsrUint16 transmitRate = 0;

    if (!request->frame || !os_linux || !os_linux->smepriv)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiRouterMaPacketReqHandler: Invalid frame/os_linux/os_linux->smepriv\n",
                               os_linux ? os_linux->instance : 0));
    }

    if (interfaceTag >= CSR_WIFI_MAX_INTERFACES)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiRouterMaPacketReqHandler: interfaceID >= CSR_WIFI_MAX_INTERFACES.\n",
                               os_linux->instance));
    }

    interfacePriv = os_linux->interfacePriv[interfaceTag];

    if (request->subscriptionHandle > MAX_MA_PACKET_IND_FILTERS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiRouterMaPacketReqHandler: Invalid subscriptionHandle.\n",
                               os_linux->instance));
    }

    if (!os_linux->sme_ma_packet_ind_filters[request->subscriptionHandle].in_use)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiRouterMaPacketReqHandler: Unknown subscription.\n",
                               os_linux->instance));
    }

    subscription = &os_linux->sme_ma_packet_ind_filters[request->subscriptionHandle];

    if (CSR_WIFI_ROUTER_PRIORITY_MANAGEMENT == request->priority)
    {
        result = CsrWifiHipPriorityGetReq(os_linux->hip_handle,
                                          interfaceTag,
                                          subscription->protocol,
                                          (CsrUint8 *) request->frame + CSR_WIFI_LLC_SNAP_HDR_LEN + (2 * ETH_ALEN),
                                          request->frame,
                                          &priority,
                                          (CsrInt32 *) 0);
        if (result != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi%d: CsrWifiRouterMaPacketReqHandler: Failed to get frame priority\n",
                                   os_linux ? os_linux->instance : 0));
        }
    }
    else
    {
        priority = request->priority;
    }

    qosType = CSR_WIFI_HIP_QOS_TYPE(priority);

    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                       "unifi%d: CsrWifiRouterMaPacketReqHandler: Subscription API user priority is 0x%x (QoS %s)\n",
                       os_linux->instance,
                       priority,
                       (qosType == CSR_WIFI_HIP_QOS_TYPE_QOS_CONTROL) ? "WMM" : "Legacy"));

    CsrWifiHipFrameBufferSizeReq(os_linux->hip_handle,
                                 interfaceTag,
                                 qosType,
                                 &macHeaderOffset);

    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                       "unifi%d: CsrWifiRouterMaPacketReqHandler: macHeaderOffset %u\n",
                       os_linux->instance, macHeaderOffset));

    /* Allocate space for the MAC header but no reason to keep the 2 MAC addresses */
    result = os_linux_net_data_malloc(os_linux, &bulkdata.d[0],
                                      request->frameLength + macHeaderOffset - 2 * ETH_ALEN);
    if (result != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiRouterMaPacketReqHandler: failed to allocate bulkdata.\n",
                               os_linux->instance));
    }

    /* Determine if we need to add SNAP header.
       During the MaPacketSubscriptionReq, it has been verified that the encapsulation type is valid */
    if (subscription->encapsulation == CSR_WIFI_ROUTER_ENCAPSULATION_ETHERNET)
    {
        struct sk_buff *skb = NULL;
        int r;

        memcpy((void *) bulkdata.d[0].os_data_ptr + macHeaderOffset, request->frame, request->frameLength);

        /* The translation is performed on the skb */
        skb = (struct sk_buff *) bulkdata.d[0].os_net_buf_ptr;

        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                           "unifi%d: CsrWifiRouterMaPacketReqHandler: skb_add_llc_snap -->\n",
                           os_linux->instance));
        r = skb_add_llc_snap(os_linux->netdev[interfaceTag], skb, subscription->protocol);
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                           "unifi%d: CsrWifiRouterMaPacketReqHandler: skb_add_llc_snap <--\n",
                           os_linux->instance));
        if (r)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi%d: CsrWifiRouterMaPacketReqHandler: failed to translate eth frame.\n",
                                   os_linux->instance));
            os_linux_net_data_free(os_linux, &bulkdata.d[0]);
        }

        bulkdata.d[0].data_length = skb->len;

        /* Get the destination and source MAC addresses.
           An allocationOffset of size SNAP header has been prepended */
        daddr = request->frame + CSR_WIFI_LLC_SNAP_HDR_LEN;
        saddr = (request->frame + ETH_ALEN + CSR_WIFI_LLC_SNAP_HDR_LEN);
    }
    else
    {
        /* Crop the MAC addresses from the packet - don't have to update the length of the skb since
           the allocation already ensured that the lengths were correct */
        memcpy((void *) bulkdata.d[0].os_data_ptr + macHeaderOffset, request->frame + 2 * ETH_ALEN, request->frameLength - 2 * ETH_ALEN);

        /* Get the destination and source MAC addresses */
        daddr = request->frame;
        saddr = (request->frame + ETH_ALEN);
    }

    bulkdata.d[1].os_data_ptr = NULL;
    bulkdata.d[1].os_net_buf_ptr = NULL;
    bulkdata.d[1].data_length = 0;


    /* check for m4 detection */
    if (0 == uf_verify_m4(os_linux, bulkdata.d[0].os_data_ptr + macHeaderOffset, bulkdata.d[0].data_length - macHeaderOffset))
    {
        eapolStore = TRUE;
    }

    if (eapolStore)
    {
        CsrWifiMacAddress peerMacAddress;
        CsrUint8 m4_already_queued;

        /* get the peer Mac address */
        memcpy(&peerMacAddress, daddr, ETH_ALEN);

        /* Store the EAPOL M4 packet for later */

        /*
         * There is a chance that there is already an M4 queued.
         * If this is true, then this existing M4 will be replaced by the new one,
         * but an MA-PACKET.cfm for the existing one must be sent to the SME.
         */
        spin_lock_irqsave(&os_linux->m4_lock, spin_lock_flag);
        m4_already_queued = 0;

        if (interfacePriv->m4_hostTag != CSR_WIFI_EAPOL_M4_NULL_HOST_TAG)
        {
            /* Confirm the existing packet before it is replaced with the new one */

            m4_already_queued = 1;
        }

        /* Store the packet */
        interfacePriv->m4_bulk_data.d[0].net_buf_length = bulkdata.d[0].net_buf_length;
        interfacePriv->m4_bulk_data.d[0].data_length = bulkdata.d[0].data_length;
        interfacePriv->m4_bulk_data.d[0].os_data_ptr = bulkdata.d[0].os_data_ptr;
        interfacePriv->m4_bulk_data.d[0].os_net_buf_ptr = bulkdata.d[0].os_net_buf_ptr;

        /* Update the HostTag */
        interfacePriv->m4_hostTag = (request->hostTag | CSR_WIFI_HIP_M4_MESSAGE);
        memcpy(interfacePriv->m4_peer_mac_addr.a, peerMacAddress.a, ETH_ALEN);

        memset(&interfacePriv->m4_info, 0, sizeof(os_linux_store_m4_info));

        /*
         * CsrMaPacketCfm is mandatory for the M4 handling mechanism,
         * so ensure that a confirm is requested
         */
        if (!request->cfmRequested)
        {
            CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                                "unifi%d: CsrWifiRouterMaPacketReqHandler: ERROR Cfm is not requested\n",
                                os_linux->instance));
        }
        interfacePriv->m4_info.cfmRequested = request->cfmRequested;

        /* Update the M4 info structure */
        memcpy(interfacePriv->m4_info.dmac.a, daddr, ETH_ALEN);
        memcpy(interfacePriv->m4_info.smac.a, saddr, ETH_ALEN);
        interfacePriv->m4_info.interfaceTag = interfaceTag;
        interfacePriv->m4_info.macHeaderOffset = macHeaderOffset;
        interfacePriv->m4_info.priority = priority;
        interfacePriv->m4_info.protocol = subscription->protocol;
        interfacePriv->m4_info.senderProcessId = ((os_linux->sme_cli->sender_id & 0xff00) | (unsigned int) msg->source);

        spin_unlock_irqrestore(&os_linux->m4_lock, spin_lock_flag);

        /* Send a signal to SME */
        if (!m4_already_queued)
        {
            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                               "unifi%d: CsrWifiRouterMaPacketReqHandler: Sending CsrWifiRouterCtrlM4ReadyToSendInd\n",
                               os_linux->instance));
            CsrWifiRouterCtrlM4ReadyToSendIndSend(os_linux->CSR_WIFI_SME_IFACEQUEUE, 0, interfaceTag, peerMacAddress);
        }
        else
        {
            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                               "unifi%d: CsrWifiRouterMaPacketReqHandler: Skip CsrWifiRouterCtrlM4ReadyToSendInd\n",
                               os_linux->instance));
        }
        return;
    }
    /*Firmware transmits EAPOL packets at the highest rate which causes IOP issues with
     few devices. Setting the min rate viz 6Mbps is safe for p2p*/
    if (subscription->protocol == ETH_P_PAE)
    {
        if ((interfacePriv->interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_P2PGO) ||
            (interfacePriv->interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_P2PCLI))
        {
            transmitRate = 0x000c;
        }
    }

    /* The final CsrWifiRouterMaPacketCfmSend() will called when the actual MA-PACKET.cfm is received from the chip */
    result = CsrWifiHipMaPacketReq(os_linux->hip_handle,
                                   daddr,
                                   saddr,
                                   ((os_linux->sme_cli->sender_id & 0xff00) | (unsigned int) msg->source),
                                   interfaceTag,
                                   CSR_WIFI_HIP_FRAMETYPE_IEEE80211_DATA,
                                   subscription->protocol,
                                   &bulkdata,
                                   priority,
                                   transmitRate,
                                   request->hostTag,
                                   request->cfmRequested,
                                   macHeaderOffset,
                                   CsrWifiHipVifIndexGetReq(os_linux->hip_handle, interfaceTag),
                                   0);
    if (result != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiRouterMaPacketReqHandler: failed to send MaPacketReq\n",
                               os_linux->instance));
        os_linux_net_data_free(os_linux, &bulkdata.d[0]);

        if (request->cfmRequested)
        {
            CsrWifiRouterMaPacketCfmSend(msg->source, interfaceTag,
                                         CSR_RESULT_FAILURE,
                                         request->hostTag, 0);
        }
    }
}

void CsrWifiRouterMaPacketCancelReqHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
}

void CsrWifiRouterCtrlM4TransmitReqHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
    os_linux_priv_t *os_linux = (os_linux_priv_t *) drvpriv;
    CsrWifiRouterCtrlM4TransmitReq *request = (CsrWifiRouterCtrlM4TransmitReq *) msg;
    CsrWifiHipBulkDataParam bulkdata;
    netInterface_priv_t *interfacePriv;
    unsigned long flags;
    CsrResult result;
    os_linux_store_m4_info tmp_m4_store;
    CsrUint16 transmitRate = 0;

    if (os_linux == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiRouterCtrlM4TransmitReqHandler: invalid priv\n",
                               os_linux ? os_linux->instance : 0));
        return;
    }

    if (request->interfaceTag >= CSR_WIFI_MAX_INTERFACES)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiRouterCtrlM4TransmitReqHandler: interfaceTag >= CSR_WIFI_MAX_INTERFACES\n",
                               os_linux ? os_linux->instance : 0));
        return;
    }

    interfacePriv = os_linux->interfacePriv[request->interfaceTag];

    spin_lock_irqsave(&os_linux->m4_lock, flags);
    if (interfacePriv->m4_bulk_data.d[0].data_length == 0)
    {
        spin_unlock_irqrestore(&os_linux->m4_lock, flags);
        CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                              "unifi%d: CsrWifiRouterCtrlM4TransmitReqHandler: invalid buffer\n",
                              os_linux ? os_linux->instance : 0));
        /*
         * The driver can hold only one M4 message at a time.
         * The SME can queue more than one at a time and it will ask for all of them
         * to be transmitted at some point. It is safe to just return success
         * as the message has already gone out.
         */
        CsrWifiRouterCtrlM4TransmittedIndSend(msg->source, 0,
                                              interfacePriv->InterfaceTag,
                                              interfacePriv->m4_peer_mac_addr,
                                              CSR_RESULT_SUCCESS);

        return;
    }

    /* Fetch the packet from the buffer */
    memcpy(&bulkdata.d[0], &interfacePriv->m4_bulk_data.d[0], sizeof(bulkdata.d[0]));

    /* Clear the buffered packet */
    interfacePriv->m4_bulk_data.d[0].net_buf_length = 0;
    interfacePriv->m4_bulk_data.d[0].data_length = 0;
    interfacePriv->m4_bulk_data.d[0].os_data_ptr = NULL;
    interfacePriv->m4_bulk_data.d[0].os_net_buf_ptr = NULL;

    /* Fetch the M4 info */
    memcpy(&tmp_m4_store, &interfacePriv->m4_info, sizeof(os_linux_store_m4_info));

    /* Clear the M4 info */
    memset(&interfacePriv->m4_info, 0, sizeof(os_linux_store_m4_info));

    /*
     * Store the hostTag to the pending so can match the confirm later.
     * If between sending this packet and getting a confirm the SME queues
     * a new M4, then we will still be able to match the confirm
     */
    interfacePriv->m4_pending_hostTag = interfacePriv->m4_hostTag;
    interfacePriv->m4_hostTag = CSR_WIFI_EAPOL_M4_NULL_HOST_TAG;

    spin_unlock_irqrestore(&os_linux->m4_lock, flags);

    bulkdata.d[1].os_net_buf_ptr = NULL;
    bulkdata.d[1].os_data_ptr = NULL;
    bulkdata.d[1].net_buf_length = 0;
    bulkdata.d[1].data_length = 0;

    /*Firmware transmits EAPOL packets at the highest rate which causes IOP issues with
     few devices. Setting the min rate viz. 6Mbps is safe for p2p*/
    if (tmp_m4_store.protocol == ETH_P_PAE)
    {
        if ((interfacePriv->interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_P2PGO) ||
            (interfacePriv->interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_P2PCLI))
        {
            transmitRate = 0x000c;
        }
    }

    result = CsrWifiHipMaPacketReq(os_linux->hip_handle,
                                   (CsrUint8 *) tmp_m4_store.dmac.a,
                                   (CsrUint8 *) tmp_m4_store.smac.a,
                                   tmp_m4_store.senderProcessId,
                                   tmp_m4_store.interfaceTag,
                                   CSR_WIFI_HIP_FRAMETYPE_IEEE80211_DATA,
                                   tmp_m4_store.protocol,
                                   &bulkdata,
                                   tmp_m4_store.priority,
                                   transmitRate,
                                   interfacePriv->m4_pending_hostTag,
                                   tmp_m4_store.cfmRequested,
                                   tmp_m4_store.macHeaderOffset,
                                   CsrWifiHipVifIndexGetReq(os_linux->hip_handle, tmp_m4_store.interfaceTag),
                                   0);

    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                       "unifi%d: CsrWifiRouterCtrlM4TransmitReqHandler: sent with HostTag=0x%x\n",
                       os_linux ? os_linux->instance : 0, interfacePriv->m4_pending_hostTag));
    if (result != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiRouterCtrlM4TransmitReqHandler: failed to send signal.\n",
                               os_linux ? os_linux->instance : 0));
        os_linux_net_data_free(os_linux, &bulkdata.d[0]);
    }
}

void CsrWifiRouterCtrlModeSetReqHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
    os_linux_priv_t *priv = drvpriv;
    CsrWifiRouterCtrlModeSetReq *req = (CsrWifiRouterCtrlModeSetReq *) msg;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiRouterCtrlModeSetReqHandler: invalid smepriv\n", priv ? priv->instance : 0));
        return;
    }

    if (req->interfaceTag < CSR_WIFI_MAX_INTERFACES)
    {
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2, "unifi%d: CsrWifiRouterCtrlModeSetReqHandler: tag = %d, mode = %d\n", priv->instance, req->interfaceTag, req->mode));
        priv->interfacePriv[req->interfaceTag]->interfaceMode = req->mode; /* keep a copy, only for benefit of gratuitous arp */
        CsrWifiHipVifModeSetReq(priv->hip_handle, req->interfaceTag,
                                req->mode, req->bssid, req->protection,
                                req->intraBssDistEnabled, req->vifIndex);
    }
    else
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiRouterCtrlModeSetReqHandler: invalid interfaceTag :%d\n",
                               priv ? priv->instance : 0, req->interfaceTag));
    }
}

void CsrWifiRouterCtrlPeerDelReqHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
    CsrWifiRouterCtrlPeerDelReq *req = (CsrWifiRouterCtrlPeerDelReq *) msg;
    CsrResult status;
    os_linux_priv_t *priv = drvpriv;

    CSR_LOG_TEXT_INFO((
                          CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG2,  "unifi%d: entering CsrWifiRouterCtrlPeerDelReqHandler \n"
                          , priv ? priv->instance : 0));
    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiRouterCtrlPeerDelReqHandler: invalid priv\n", priv->instance));
        return;
    }

    if (req->interfaceTag >= CSR_WIFI_MAX_INTERFACES)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiRouterCtrlPeerDelReqHandler: bad interfaceTag\n", priv ? priv->instance : 0));
        return;
    }

    status = CsrWifiHipPeerDelReq(priv->hip_handle, req->interfaceTag, req->peerRecordHandle);
    CsrWifiRouterCtrlPeerDelCfmSend(msg->source, req->clientData, req->interfaceTag, status);
    CSR_LOG_TEXT_INFO((
                          CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG2,  "unifi%d: leaving CsrWifiRouterCtrlPeerDelReqHandler \n"
                          , priv ? priv->instance : 0));
}

void CsrWifiRouterCtrlPeerAddReqHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
    CsrWifiRouterCtrlPeerAddReq *req = (CsrWifiRouterCtrlPeerAddReq *) msg;
    CsrResult status;
    os_linux_priv_t *priv = drvpriv;
    CsrUint32 handle = 0;

    CSR_LOG_TEXT_INFO((
                          CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG2,  "unifi%d: entering CsrWifiRouterCtrlPeerAddReqHandler \n"
                          , priv ? priv->instance : 0));
    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi : CsrWifiRouterCtrlPeerAddReqHandler: invalid priv\n"));
        return;
    }

    if (req->interfaceTag >= CSR_WIFI_MAX_INTERFACES)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiRouterCtrlPeerAddReqHandler: bad interfaceTag\n", priv ? priv->instance : 0));
        return;
    }

    status = CsrWifiHipPeerAddReq(priv->hip_handle, req->interfaceTag, req->peerMacAddress, req->associationId, &req->staInfo, &handle);

    CsrWifiRouterCtrlPeerAddCfmSend(msg->source, req->clientData, req->interfaceTag, req->peerMacAddress, handle, status);
    CSR_LOG_TEXT_INFO((
                          CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG2,  "unifi%d: leaving CsrWifiRouterCtrlPeerAddReqHandler \n"
                          , priv->instance));
}

void CsrWifiRouterCtrlPeerUpdateReqHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
    CsrWifiRouterCtrlPeerUpdateReq *req = (CsrWifiRouterCtrlPeerUpdateReq *) msg;
    CsrResult status;
    os_linux_priv_t *priv = drvpriv;
    CsrUint32 handle = 0;

    CSR_LOG_TEXT_INFO((
                          CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG2,  "unifi%d: entering CsrWifiRouterCtrlPeerUpdateReqHandler \n"
                          , priv ? priv->instance : 0));
    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi : CsrWifiRouterCtrlPeerUpdateReqHandler: invalid priv\n"));
        return;
    }

    if (CSR_RESULT_SUCCESS != CsrWifiHipPeerUpdateReq(priv->hip_handle, req->interfaceTag, handle, &status))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipPeerUpdateReq failed\n", priv->instance));
        return;
    }
    CsrWifiRouterCtrlPeerUpdateCfmSend(msg->source, req->clientData, req->interfaceTag, status);
    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG2,  "unifi%d: leaving CsrWifiRouterCtrlPeerUpdateReqHandler\n", priv->instance));
}

void CsrWifiRouterCtrlRawSdioDeinitialiseReqHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
    /* This will never be called as it is intercepted in the Userspace */
}

void CsrWifiRouterCtrlRawSdioInitialiseReqHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
    /* This will never be called as it is intercepted in the Userspace */
}

void CsrWifiRouterCtrlBlockAckDisableReqHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
    CsrWifiRouterCtrlBlockAckDisableReq *req = (CsrWifiRouterCtrlBlockAckDisableReq *) msg;
    CsrResult result;
    os_linux_priv_t *priv = drvpriv;

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  "unifi%d: %s: in ok\n",
                           priv ? priv->instance : 0,  __FUNCTION__));

    result = CsrWifiHipBlockAckDisableReq(priv->hip_handle, req->interfaceTag, req->macAddress, req->trafficStreamID, req->role);

    CsrWifiRouterCtrlBlockAckDisableCfmSend(msg->source,
                                            req->clientData,
                                            req->interfaceTag,
                                            result);

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  "unifi%d: %s: out ok\n",
                           priv ? priv->instance : 0,  __FUNCTION__));
}

void CsrWifiRouterCtrlBlockAckEnableReqHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
    CsrWifiRouterCtrlBlockAckEnableReq *req = (CsrWifiRouterCtrlBlockAckEnableReq *) msg;
    CsrResult result;
    os_linux_priv_t *priv = drvpriv;

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  "unifi%d: >>%s\n",
                           priv ? priv->instance : 0,  __FUNCTION__));
    result = CsrWifiHipBlockAckEnableReq(priv->hip_handle,  req->interfaceTag, req->macAddress, req->trafficStreamID, req->role, req->bufferSize, req->timeout, req->ssn);
    CsrWifiRouterCtrlBlockAckEnableCfmSend(msg->source,
                                           req->clientData,
                                           req->interfaceTag,
                                           result);
    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  "unifi%d: <<%s: r=%d\n",
                           priv ? priv->instance : 0,  __FUNCTION__, result));
}

void CsrWifiRouterCtrlWapiMicFilterReqHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
#if defined(CSR_WIFI_SECURITY_WAPI_ENABLE)
    os_linux_priv_t *priv = drvpriv;
    CsrWifiRouterCtrlWapiMicFilterReq *req = (CsrWifiRouterCtrlWapiMicFilterReq *) msg;

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  "unifi%d: >>%s\n",
                           priv ? priv->instance : 0,  __FUNCTION__));
    CSR_LOG_TEXT_INFO((
                          CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: %s: req->isUnicast = (%d), req->status = (%d), \n",
                          priv ? priv->instance : 0,  __FUNCTION__, req->isUnicast, req->status));

    /*Update the respective filter in the HAL*/
    CsrWifiHipWapiMicFilterReq(priv->hip_handle, req->interfaceTag, req->status, req->isUnicast);

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,  "unifi%d: <<%s\n",
                           priv ? priv->instance : 0,  __FUNCTION__));

#elif defined(UNIFI_DEBUG) && defined(CSR_LOG_ENABLE)
    os_linux_priv_t *priv = drvpriv;

    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                           "unifi%d: CsrWifiRouterCtrlWapiMicFilterReqHandler: called when WAPI isn't enabled\n",
                           priv ? priv->instance : 0));
#endif
}

void CsrWifiRouterCtrlWapiRxPktReqHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
#if defined(CSR_WIFI_SECURITY_WAPI_ENABLE)
    os_linux_priv_t *priv = drvpriv;
    CsrWifiRouterCtrlWapiRxPktReq *req = (CsrWifiRouterCtrlWapiRxPktReq *) msg;
    CsrWifiHipBulkDataParam bulkdata;
    CsrResult res;

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi%d: >>%s\n",
                        priv ? priv->instance : 0, __FUNCTION__));

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiRouterCtrlWapiRxPktReq : invalid priv\n",
                               priv ? priv->instance : 0));
        return;
    }

    if (priv->smepriv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiRouterCtrlWapiRxPktReq : invalid sme priv\n",
                               priv->instance));
        return;
    }

    if ((req->dataLength == 0) || (req->data == NULL))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiRouterCtrlWapiRxPktReq: invalid request\n",
                               priv->instance));
        return;
    }

    if (req->interfaceTag >= CSR_WIFI_MAX_INTERFACES)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiRouterCtrlWapiRxPktReq: interfaceID >= CSR_WIFI_MAX_INTERFACES.\n",
                               priv->instance));
        return;
    }

    res = os_linux_net_data_malloc(priv, &bulkdata.d[0], req->dataLength);
    if (res != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiRouterCtrlWapiRxPktReq: Could not allocate net data\n",
                               priv->instance));
        return;
    }

    memset(&bulkdata.d[1], 0, sizeof(CsrWifiHipBulkDataDesc));
    memcpy((void *) bulkdata.d[0].os_data_ptr, req->data, req->dataLength);

    CsrWifiHipWapiMicVerifiedFrameRxReq(priv->hip_handle, req->interfaceTag, req->signalLength, req->signal, &bulkdata);

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6, "unifi%d: <<%s\n",
                        priv->instance, __FUNCTION__));

#elif defined(UNIFI_DEBUG) && defined(CSR_LOG_ENABLE)
    os_linux_priv_t *priv = drvpriv;

    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                           "unifi%d: CsrWifiRouterCtrlWapiRxPktReqHandler: called when WAPI isn't enabled\n",
                           priv ? priv->instance : 0));
#endif
}

void CsrWifiHipWifiOffCfm(void *osLayerContext, CsrResult result)
{
    /* WifiOff handling is always blocking on Linux; hence, this signal should
       never be received from HAL. */
    CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                        "unifi%d: CsrWifiHipWifiOffCfm: Received a WifiOffCfm from HAL during a blocking wifiOff procedure (unexpected signal)\n",
                        osLayerContext ? ((os_linux_priv_t *) osLayerContext)->instance : 0));
}

CsrResult CsrWifiHipInterfaceRegisterInd(void *osLayerContext, CsrUint16 interfaceTag, CsrWifiMacAddress macAddress)
{
    os_linux_priv_t *priv = (os_linux_priv_t *) osLayerContext;
    netInterface_priv_t *interfacePriv = NULL;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: (%s) : invalid priv\n",
                               priv ? priv->instance : 0, __FUNCTION__));
        return CSR_RESULT_FAILURE;
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d:%s:ifTag=%d \n",
                        priv ? priv->instance : 0, __FUNCTION__, interfaceTag));

    /* Store the MAC address in GHAL database */
    CsrWifiHipSetMacReq(priv->hip_handle, &macAddress, interfaceTag);

    /* Register the netdevs for requested interface. */
    interfacePriv = (netInterface_priv_t *) priv->interfacePriv[interfaceTag];
    if (interfacePriv->netdev_state == OS_LINUX_NETDEV_STATE_UNREGISTERED)
    {
        int r;

        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3,
                            "unifi%d: registering net device %d\n",
                            priv ? priv->instance : 0, interfaceTag));

        /* Store the MAC address in the netdev structure */
        memcpy(priv->netdev[interfaceTag]->dev_addr, macAddress.a, ETH_ALEN);

        r = uf_register_netdev(priv, interfaceTag);
        if (r)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi%d: CsrWifiHipInterfaceRegisterInd: Failed to register the network device.\n",
                                   priv ? priv->instance : 0));

            hal_wifi_off(priv);
            os_linux_wifi_off(priv);
            priv->state = OS_LINUX_STATE_ACTIVATED;
            return CSR_RESULT_FAILURE;
        }
    }
    else
    {
        /* If interface Mac Address changes, then update accordingly */
        if (CsrMemCmp(macAddress.a, priv->netdev[interfaceTag]->dev_addr, ETH_ALEN))
        {
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,
                                "unifi%d:%s: Trying to set new Mac address iface=%s \n",
                                priv ? priv->instance : 0, __FUNCTION__, priv->netdev[interfaceTag]->name));
            /* Trying to set new Mac address for the interface */
            if (netif_running(priv->netdev[interfaceTag]))
            {
                UF_NETIF_TX_STOP_ALL_QUEUES(priv->netdev[interfaceTag]);
            }

            netif_carrier_off(priv->netdev[interfaceTag]);
            /* Store the MAC address in the netdev */
            memcpy(priv->netdev[interfaceTag]->dev_addr, macAddress.a, ETH_ALEN);
            netif_carrier_on(priv->netdev[interfaceTag]);
        }
    }

    /* The chip is initialised, but don't declare UniFi ready until the first
     * MLME-RESET.cfm is complete.
     */
    priv->init_progress = UNIFI_INIT_COMPLETED;
    interfacePriv->netdev_state = OS_LINUX_NETDEV_STATE_REGISTERED;
#ifdef CSR_SUPPORT_WEXT
    /* Notify the Android wpa_supplicant that we are ready */
    wext_send_started_event(priv, interfaceTag);

    queue_work(priv->unifi_workqueue, &priv->sme_config_task);
#endif
    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                           "unifi%d: Interface %d ready\n", interfaceTag, priv->instance));
    return CSR_RESULT_SUCCESS;
}

CsrResult CsrWifiHipInterfaceUnregisterInd(void *osLayerContext, CsrUint16 interfaceTag)
{
    os_linux_priv_t *priv = (os_linux_priv_t *) osLayerContext;
    netInterface_priv_t *interfacePriv = NULL;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi : CsrWifiHipInterfaceUnregisterInd: Invalid priv\n"));
        return CSR_RESULT_FAILURE;
    }

    if (interfaceTag >= CSR_WIFI_MAX_INTERFACES)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi : CsrWifiHipInterfaceUnregisterInd: Invalid interfaceTag %u\n",
                               interfaceTag));
        return CSR_RESULT_FAILURE;
    }

    interfacePriv = (netInterface_priv_t *) priv->interfacePriv[interfaceTag];

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG4, "unifi%d:%s:ifTag=%d netdev state %u\n",
                        priv ? priv->instance : 0, __FUNCTION__, interfaceTag, interfacePriv->netdev_state));

    if ((interfacePriv->netdev_state == OS_LINUX_NETDEV_STATE_REGISTERED) ||
        (interfacePriv->netdev_state == OS_LINUX_NETDEV_STATE_REGISTERED_STOPPED))
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3, "unifi%d: Stopping netdev interface %s\n",
                            priv ? priv->instance : 0, priv->netdev[interfaceTag]->name));

#ifdef CSR_SUPPORT_WEXT
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                           "unifi%d: CsrWifiHipInterfaceUnregisterInd: Cancel waiting for NETDEV_CHANGE\n"
                           , priv ? priv->instance : 0));
        interfacePriv->wait_netdev_change = FALSE;
#endif

        netif_carrier_off(priv->netdev[interfaceTag]);
        UF_NETIF_TX_STOP_ALL_QUEUES(priv->netdev[interfaceTag]);

        /* Mark the netdev as registered but stopped. */
        interfacePriv->netdev_state = OS_LINUX_NETDEV_STATE_REGISTERED_STOPPED;
    }

    return CSR_RESULT_SUCCESS;
}

void CsrWifiRouterCtrlInterfaceAddReqHandler(void *osLayerContext, CsrWifiFsmEvent *msg)
{
    CsrResult r;
    os_linux_priv_t *priv = (os_linux_priv_t *) osLayerContext;
    CsrWifiRouterCtrlInterfaceAddReq *req = (CsrWifiRouterCtrlInterfaceAddReq *) msg;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d:(%s) : invalid priv\n",
                               priv->instance, __FUNCTION__));
        return;
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG4, "unifi%d:%s:ifTag=%d \n",
                        priv->instance, __FUNCTION__, req->interfaceTag));
    /* registers the statically created netdevice interface with network sub system */
    if ((r = CsrWifiHipInterfaceAddReq(priv->hip_handle, req->interfaceTag, req->macAddress)))
    {
        CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                            "unifi(%d):%s: netdev registering failed\n", req->interfaceTag, __FUNCTION__));
    }

    CsrWifiRouterCtrlInterfaceAddCfmSend(msg->source, req->clientData, req->interfaceTag, r);
}

void CsrWifiRouterCtrlInterfaceDelReqHandler(void *osLayerContext, CsrWifiFsmEvent *msg)
{
    CsrResult r;
    os_linux_priv_t *priv = (os_linux_priv_t *) osLayerContext;
    CsrWifiRouterCtrlInterfaceDelReq *req = (CsrWifiRouterCtrlInterfaceDelReq *) msg;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d:(%s) : invalid priv\n",
                               priv ? priv->instance : 0, __FUNCTION__));
        return;
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d:%s:ifTag=%d\n",
                        priv ? priv->instance : 0, __FUNCTION__, req->interfaceTag));

    /* Deregisters the statically created netdevice interface with network sub system */
    if ((r = CsrWifiHipInterfaceDelReq(priv->hip_handle, req->interfaceTag)))
    {
        CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                            "unifi(%d):%s: netdev unregistering failed\n", req->interfaceTag,  __FUNCTION__));
    }
    CsrWifiRouterCtrlInterfaceDelCfmSend(msg->source, req->clientData, req->interfaceTag, r);
}

void CsrWifiRouterCtrlVifAddReqHandler(void *osLayerContext, CsrWifiFsmEvent *msg)
{
    CsrResult r;
    os_linux_priv_t *priv = (os_linux_priv_t *) osLayerContext;
    CsrWifiRouterCtrlVifAddReq *req = (CsrWifiRouterCtrlVifAddReq *) msg;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d:(%s) : invalid priv\n",
                               priv ? priv->instance : 0, __FUNCTION__));
        return;
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d:%s:ifTag=%d\n",
                        priv ? priv->instance : 0, __FUNCTION__, req->interfaceTag));

    if ((r = CsrWifiHipVifAddReq(priv->hip_handle, req->interfaceTag, req->vifIndex, FALSE)))
    {
        CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                            "unifi(%d):%s: vifIndex mapping failed\n", req->interfaceTag, __FUNCTION__));
    }
    CsrWifiRouterCtrlVifAddCfmSend(msg->source, req->clientData, req->interfaceTag, r);
}

void CsrWifiRouterCtrlVifDelReqHandler(void *osLayerContext, CsrWifiFsmEvent *msg)
{
    CsrResult r;
    os_linux_priv_t *priv = (os_linux_priv_t *) osLayerContext;
    CsrWifiRouterCtrlVifDelReq *req = (CsrWifiRouterCtrlVifDelReq *) msg;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d:(%s) : invalid priv\n",
                               priv ? priv->instance : 0, __FUNCTION__));
        return;
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d:%s:ifTag=%d\n",
                        priv ? priv->instance : 0, __FUNCTION__, req->interfaceTag));

    if ((r = CsrWifiHipVifDelReq(priv->hip_handle, req->interfaceTag, req->vifIndex, req->vifIndexPrimary)))
    {
        CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                            "unifi(%d):%s: vifIndex mapping failed\n", req->interfaceTag, __FUNCTION__));
    }
    CsrWifiRouterCtrlVifDelCfmSend(msg->source, req->clientData, req->interfaceTag, r);
}

void CsrWifiHipWifiOffInd(void *osLayerContext, CsrWifiHipControlIndication controlIndication)
{
    os_linux_priv_t *os_linux = (os_linux_priv_t *) osLayerContext;
    CsrUint16 interfaceTag = 0;
    u8 conf_param = CONFIG_IND_ERROR;

    /* Stop the network traffic */
    for (interfaceTag = 0; interfaceTag < CSR_WIFI_MAX_INTERFACES; interfaceTag++)
    {
        netInterface_priv_t *interfacePriv = os_linux->interfacePriv[interfaceTag];
        if (interfacePriv->netdev_state == OS_LINUX_NETDEV_STATE_REGISTERED)
        {
            /* Stop the data plane as soon as possible. The WifiOffReq/Res will
               delete the interface later. */
            UF_NETIF_TX_STOP_ALL_QUEUES(os_linux->netdev[interfaceTag]);
            netif_carrier_off(os_linux->netdev[interfaceTag]);
            interfacePriv->netdev_state = OS_LINUX_NETDEV_STATE_REGISTERED_STOPPED;
        }
    }

    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                           "unifi%d: CsrWifiHipWifiOffInd: fatal error is reported to the SME.\n",
                           os_linux ? os_linux->instance : 0));
    /* Notify the clients (SME or unifi_manager) for the error. */
    ul_log_config_ind(os_linux, &conf_param, sizeof(u8));

    /* Consider UniFi to be uninitialised */
    os_linux->init_progress = UNIFI_INIT_NONE;
}

void CsrWifiHipTaIndicateProtocolInd(void *osLayerContext, CsrUint16 interfaceTag, CsrWifiRouterCtrlTrafficPacketType packetType, CsrWifiRouterCtrlProtocolDirection direction, CsrWifiMacAddress srcAddr)
{
    linux_ta_indicate_protocol(osLayerContext, interfaceTag, packetType, direction, &srcAddr);
}

void CsrWifiHipTaIndicateSamplingInd(void *osLayerContext, CsrUint16 interfaceTag, CsrWifiRouterCtrlTrafficStats *trafficStats)
{
    linux_ta_indicate_sampling(osLayerContext, interfaceTag, trafficStats);
}

void CsrWifiHipSuspendInd(void     *osLayerContext,
                          CsrUint16 clientData,
                          CsrBool   hardSuspend,
                          CsrBool   d3Suspend)
{
#ifdef CSR_WIFI_HIP_SUSPEND_ENABLE
    os_linux_priv_t *os_linux = (os_linux_priv_t *) osLayerContext;

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                        "unifi%d: CsrWifiHipSuspendInd hardSuspend=%d, d3Suspend=%d\n",
                        os_linux->instance,
                        hardSuspend,
                        d3Suspend));

    os_linux->init_progress_save = os_linux->init_progress; /* will be used in cold suspend only */
    CsrWifiRouterCtrlSuspendIndSend(os_linux->CSR_WIFI_SME_IFACEQUEUE, clientData, hardSuspend, d3Suspend);
#endif
}

void CsrWifiHipResumeInd(void     *osLayerContext,
                         CsrUint16 clientData,
                         CsrBool   powerMaintained)
{
#ifdef CSR_WIFI_HIP_SUSPEND_ENABLE
    os_linux_priv_t *os_linux = (os_linux_priv_t *) osLayerContext;

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                        "unifi%d: CsrWifiHipResumeInd powerMaintained=%d\n",
                        os_linux->instance,
                        powerMaintained));

    CsrWifiRouterCtrlResumeIndSend(os_linux->CSR_WIFI_SME_IFACEQUEUE, clientData, powerMaintained);
#endif
}

void CsrWifiHipWifiOnCfm(void *osLayerContext, CsrResult result)
{
    os_linux_priv_t *os_linux = (os_linux_priv_t *) osLayerContext;

    /* HIP will only issue a WifiOnCfm in case of an error during init, so
       ensure that state information is updated so that a new WifiOnReq can
       handled */
    os_linux->state = OS_LINUX_STATE_ACTIVATED;
    CsrWifiRouterCtrlWifiOnCfmSend(os_linux->sme_synergy_sched_queue, 0, result);
}
