/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"
#include "csr_wifi_hip_log_text.h"

#include "os_linux_priv.h"
#include "csr_wifi_hip_conversions.h"
/*
 * This file implements the SME MGT API. It contains the following functions:
 * CsrWifiSmeWifiFlightmodeCfmSend()
 * CsrWifiSmeWifiOnCfmSend()
 * CsrWifiSmeWifiOffCfmSend()
 * CsrWifiSmeWifiOffIndSend()
 * CsrWifiSmeScanFullCfmSend()
 * CsrWifiSmeScanResultsGetCfmSend()
 * CsrWifiSmeScanResultIndSend()
 * CsrWifiSmeScanResultsFlushCfmSend()
 * CsrWifiSmeConnectCfmSend()
 * CsrWifiSmeMediaStatusIndSend()
 * CsrWifiSmeDisconnectCfmSend()
 * CsrWifiSmeKeyCfmSend()
 * CsrWifiSmeMulticastAddressCfmSend()
 * CsrWifiSmeSetCfmSend()
 * CsrWifiSmeGetCfmSend()
 * CsrWifiSmeMicFailureIndSend()
 * CsrWifiSmePmkidCfmSend()
 * CsrWifiSmePmkidCandidateListIndSend()
 * CsrWifiSmeConnectionQualityIndSend()
 * CsrWifiSmePacketFilterSetCfmSend()
 * CsrWifiSmeTspecCfmSend()
 * CsrWifiSmeTspecIndSend()
 * CsrWifiSmeBlacklistCfmSend()
 * CsrWifiSmeEventMaskSetCfmSend()
 * CsrWifiSmeRoamStartIndSend()
 * CsrWifiSmeRoamCompleteIndSend()
 * CsrWifiSmeAssociationStartIndSend()
 * CsrWifiSmeAssociationCompleteIndSend()
 * CsrWifiSmeIbssStationIndSend()
 */


void CsrWifiSmeMicFailureIndHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
#ifdef CSR_SUPPORT_WEXT
    os_linux_priv_t *priv = drvpriv;
    CsrWifiSmeMicFailureInd *ind = (CsrWifiSmeMicFailureInd *) msg;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "unifi : CsrWifiSmeMicFailureIndSend: invalid priv\n"));
        return;
    }

    CSR_LOG_TEXT_INFO((
                          CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                          "unifi%d: CsrWifiSmeMicFailureIndSend: count=%d, KeyType=%d\n",
                          priv ? priv->instance : 0,
                          ind->count, ind->keyType));
    if (ind->interfaceTag >= CSR_WIFI_MAX_INTERFACES)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiSmeMicFailureIndHandler: bad interfaceTag=%d\n",
                               priv ? priv->instance : 0, ind->interfaceTag));
        return;
    }

    wext_send_michaelmicfailure_event(priv, ind->count, ind->address, ind->keyType, ind->interfaceTag);
#endif
}

void CsrWifiSmePmkidCfmHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
#ifdef CSR_SUPPORT_WEXT
    os_linux_priv_t *priv = drvpriv;
    CsrWifiSmePmkidCfm *cfm = (CsrWifiSmePmkidCfm *) msg;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "unifi : CsrWifiSmePmkidCfmSend: Invalid ospriv.\n"));
        return;
    }

    /*
     * WEXT never does a GET operation the PMKIDs, so we don't need
     * handle data returned in pmkids.
     */

    (void) sme_complete_request(priv, cfm->status);
#endif
}

void CsrWifiSmePmkidCandidateListIndHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
#ifdef CSR_SUPPORT_WEXT
    os_linux_priv_t *priv = drvpriv;
    CsrWifiSmePmkidCandidateListInd *ind = (CsrWifiSmePmkidCandidateListInd *) msg;
    int i;

    if (priv->smepriv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiSmePmkidCandidateListIndSend: invalid smepriv\n", priv ? priv->instance : 0));
        return;
    }

    if (ind->interfaceTag >= CSR_WIFI_MAX_INTERFACES)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiSmePmkidCandidateListIndHandler: bad interfaceTag=%d\n",
                               priv ? priv->instance : 0, ind->interfaceTag));
        return;
    }

    for (i = 0; i < ind->pmkidCandidatesCount; i++)
    {
        wext_send_pmkid_candidate_event(priv, ind->pmkidCandidates[i].bssid,
                                        ind->pmkidCandidates[i].preAuthAllowed, ind->interfaceTag);
    }
#endif
}

void CsrWifiSmeScanResultsFlushCfmHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
}

void CsrWifiSmeScanResultsGetCfmHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
#ifdef CSR_SUPPORT_WEXT
    os_linux_priv_t *priv = drvpriv;
    CsrWifiSmeScanResultsGetCfm *cfm = (CsrWifiSmeScanResultsGetCfm *) msg;
    int bytesRequired = cfm->scanResultsCount * sizeof(CsrWifiSmeScanResult);
    int i;
    CsrUint8 *current_buff;
    CsrWifiSmeScanResult *scanCopy;
    int result;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "unifi : CsrWifiSmeScanResultsGetCfmSend: Invalid ospriv.\n"));
        return;
    }

    /* Calc the size of the buffer reuired */
    for (i = 0; i < cfm->scanResultsCount; ++i)
    {
        const CsrWifiSmeScanResult *scan_result = &cfm->scanResults[i];
        bytesRequired += scan_result->informationElementsLength;
    }

    /* Take a Copy of the scan Results :-) */
    scanCopy = CsrPmemAlloc(bytesRequired);
    memcpy(scanCopy, cfm->scanResults, sizeof(CsrWifiSmeScanResult) * cfm->scanResultsCount);

    /* Take a Copy of the Info Elements AND update the scan result pointers */
    current_buff = (CsrUint8 *) &scanCopy[cfm->scanResultsCount];
    for (i = 0; i < cfm->scanResultsCount; ++i)
    {
        CsrWifiSmeScanResult *scan_result = &scanCopy[i];
        CsrMemCpy(current_buff, scan_result->informationElements, scan_result->informationElementsLength);
        scan_result->informationElements = current_buff;
        current_buff += scan_result->informationElementsLength;
    }

    priv->sme_reply.reply_scan_results_count = cfm->scanResultsCount;
    priv->sme_reply.reply_scan_results = scanCopy;

    result = sme_complete_request(priv, cfm->status);
    if (result == -EINVAL)
    {
        CsrPmemFree(scanCopy);
    }
#endif
}

void CsrWifiSmeScanFullCfmHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
#ifdef CSR_SUPPORT_WEXT
    os_linux_priv_t *priv = drvpriv;
    CsrWifiSmeScanFullCfm *cfm = (CsrWifiSmeScanFullCfm *) msg;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "unifi : CsrWifiSmeScanFullCfmSend: Invalid ospriv.\n"));
        return;
    }

    (void) sme_complete_request(priv, cfm->status);
#endif
}

void CsrWifiSmeScanResultIndHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
}

void CsrWifiSmeConnectCfmHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
#ifdef CSR_SUPPORT_WEXT
    os_linux_priv_t *priv = drvpriv;
    CsrWifiSmeConnectCfm *cfm = (CsrWifiSmeConnectCfm *) msg;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "unifi : CsrWifiSmeConnectCfmSend: Invalid ospriv.\n"));
        return;
    }

    (void) sme_complete_request(priv, cfm->status);
#endif
}

void CsrWifiSmeDisconnectCfmHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
#ifdef CSR_SUPPORT_WEXT
    os_linux_priv_t *priv = drvpriv;
    CsrWifiSmeDisconnectCfm *cfm = (CsrWifiSmeDisconnectCfm *) msg;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "unifi : CsrWifiSmeDisconnectCfmSend: Invalid ospriv.\n"));
        return;
    }

    (void) sme_complete_request(priv, cfm->status);
#endif
}

void CsrWifiSmeKeyCfmHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
#ifdef CSR_SUPPORT_WEXT
    os_linux_priv_t *priv = drvpriv;
    CsrWifiSmeKeyCfm *cfm = (CsrWifiSmeKeyCfm *) msg;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "unifi : CsrWifiSmeKeyCfmSend: Invalid ospriv.\n"));
        return;
    }

    (void) sme_complete_request(priv, cfm->status);
#endif
}

void CsrWifiSmeMulticastAddressCfmHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
#ifdef CSR_SUPPORT_WEXT
    os_linux_priv_t *priv = drvpriv;
    CsrWifiSmeMulticastAddressCfm *cfm = (CsrWifiSmeMulticastAddressCfm *) msg;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "unifi : CsrWifiSmeMulticastAddressCfmSend: Invalid ospriv.\n"));
        return;
    }

    (void) sme_complete_request(priv, cfm->status);
#endif
}

void CsrWifiSmeWifiFlightmodeCfmHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
#ifdef CSR_SUPPORT_WEXT
    os_linux_priv_t *priv = drvpriv;
    CsrWifiSmeWifiFlightmodeCfm *cfm = (CsrWifiSmeWifiFlightmodeCfm *) msg;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "unifi : CsrWifiSmeWifiFlightmodeCfmSend: Invalid ospriv.\n"));
        return;
    }

    (void) sme_complete_request(priv, cfm->status);
#endif
}

void CsrWifiSmeWifiOnCfmHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
#ifdef CSR_SUPPORT_WEXT
    os_linux_priv_t *priv = drvpriv;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "unifi : CsrWifiSmeWifiOnCfmSend: Invalid ospriv.\n"));
        return;
    }

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,
                           "unifi%d: CsrWifiSmeWifiOnCfmSend: wake up status %d\n",
                           priv ? priv->instance : 0,  ((CsrWifiSmeWifiOnCfm *) msg)->status));
#ifdef CSR_SUPPORT_WEXT_AP
    (void) sme_complete_request(priv, ((CsrWifiSmeWifiOnCfm *) msg)->status);
#endif
#endif
}

void CsrWifiSmeWifiOffCfmHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
#ifdef CSR_SUPPORT_WEXT
    os_linux_priv_t *priv = drvpriv;
    CsrWifiSmeWifiOffCfm *cfm = (CsrWifiSmeWifiOffCfm *) msg;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "unifi : CsrWifiSmeWifiOffCfmSend: Invalid ospriv.\n"));
        return;
    }

    (void) sme_complete_request(priv, cfm->status);
#endif
}

void CsrWifiSmeWifiOffIndHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
#ifdef CSR_SUPPORT_WEXT
    os_linux_priv_t *priv = drvpriv;
    CsrWifiSmeWifiOffInd *ind = (CsrWifiSmeWifiOffInd *) msg;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "unifi : CsrWifiRouterCtrlStoppedReqSend: Invalid ospriv.\n"));
        return;
    }

    if (priv->smepriv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiRouterCtrlStoppedReqSend: invalid smepriv\n", priv ? priv->instance : 0));
        return;
    }

    /*
     * If the status indicates an error, the SME is in a stopped state.
     * We need to start it again in order to reinitialise UniFi.
     */
    switch (ind->reason)
    {
        case CSR_WIFI_SME_CONTROL_INDICATION_ERROR:
            CSR_LOG_TEXT_INFO((
                                  CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                                  "unifi%d: CsrWifiRouterCtrlStoppedReqSend: Restarting SME (ind:%d)\n",
                                  priv ? priv->instance : 0,
                                  ind->reason));

            /* On error, restart the SME */
            sme_mgt_wifi_on(priv);
            break;
        case CSR_WIFI_SME_CONTROL_INDICATION_EXIT:
#ifdef CSR_SUPPORT_WEXT_AP
            (void) sme_complete_request(priv, 0);
#endif
            break;
        default:
            break;
    }
#endif
}

void CsrWifiSmeSetCfmHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
#ifdef CSR_SUPPORT_WEXT
    os_linux_priv_t *priv = drvpriv;
    CsrWifiSmeSetCfm *cfm = (CsrWifiSmeSetCfm *) msg;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "unifi : CsrWifiSmeSetCfm: Invalid ospriv.\n"));
        return;
    }
    if (cfm->errorValuesLength != 0)
    {
        CsrWifiMibEntry error;
        CsrSize offset = 0;
        while (offset < cfm->errorValuesLength)
        {
            offset += CsrWifiMibDecode(&cfm->errorValues[offset], &error);
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiSmeSetCfm: PSID:%d, ErrorCode:%d.\n",
                                   priv ? priv->instance : 0, error.psid, error.value.u.uintValue));
        }
    }

    (void) sme_complete_request(priv, cfm->errorValuesLength == 0 ? CSR_RESULT_SUCCESS : CSR_RESULT_FAILURE);
#endif
}

void CsrWifiSmeGetCfmHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
#ifdef CSR_SUPPORT_WEXT
    os_linux_priv_t *priv = drvpriv;
    CsrWifiSmeGetCfm *cfm = (CsrWifiSmeGetCfm *) msg;
    int result;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "unifi : CsrWifiSmeGetCfm: Invalid ospriv.\n"));
        return;
    }

    if (cfm->errorValuesLength == 0)
    {
        priv->sme_reply.mibData.dataLength = cfm->valuesLength;
        priv->sme_reply.mibData.data = cfm->values;

        cfm->valuesLength = 0;
        cfm->values = 0;
    }
    else
    {
        CsrWifiMibEntry error;
        CsrSize offset = 0;
        while (offset < cfm->errorValuesLength)
        {
            offset += CsrWifiMibDecode(&cfm->errorValues[offset], &error);
            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                               "unifi%d: CsrWifiSmeGetCfm: PSID:%d, ErrorCode:%d.\n",
                               priv ? priv->instance : 0, error.psid, error.value.u.uintValue));
        }
    }
    result = sme_complete_request(priv, cfm->errorValuesLength == 0 ? CSR_RESULT_SUCCESS : CSR_RESULT_FAILURE);
    if (result == -EINVAL)
    {
        CsrPmemFree(priv->sme_reply.mibData.data);
    }
#endif
}

void CsrWifiSmeConnectionQualityIndHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
#ifdef CSR_SUPPORT_WEXT
    os_linux_priv_t *priv = drvpriv;
    CsrWifiSmeConnectionQualityInd *ind = (CsrWifiSmeConnectionQualityInd *) msg;
    int signal, noise, snr;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "unifi : CsrWifiSmeConnectionQualityIndSend: Invalid ospriv.\n"));
        return;
    }

    /*
     * level and noise below are mapped into an unsigned 8 bit number,
     * ranging from [-192; 63]. The way this is achieved is simply to
     * add 0x100 onto the number if it is negative,
     * once clipped to the correct range.
     */
    signal = ind->linkQuality.unifiRssi;
    /* Clip range of snr */
    snr = (ind->linkQuality.unifiSnr > 0) ? ind->linkQuality.unifiSnr : 0;    /* In dB relative, from 0 - 255 */
    snr = (snr < 255) ? snr : 255;
    noise = signal - snr;

    /* Clip range of signal */
    signal = (signal < 63) ? signal : 63;
    signal = (signal > -192) ? signal : -192;

    /* Clip range of noise */
    noise = (noise < 63) ? noise : 63;
    noise = (noise > -192) ? noise : -192;

    /* Make u8 */
    signal = (signal < 0) ? signal + 0x100 : signal;
    noise = (noise < 0) ? noise + 0x100 : noise;

    priv->wext_wireless_stats.qual.level = (u8) signal;  /* -192 : 63 */
    priv->wext_wireless_stats.qual.noise = (u8) noise;   /* -192 : 63 */
    priv->wext_wireless_stats.qual.qual = snr;            /* 0 : 255 */
    priv->wext_wireless_stats.qual.updated = 0;

#if WIRELESS_EXT > 16
    priv->wext_wireless_stats.qual.updated |= IW_QUAL_LEVEL_UPDATED |
                                              IW_QUAL_NOISE_UPDATED |
                                              IW_QUAL_QUAL_UPDATED;
#if WIRELESS_EXT > 18
    priv->wext_wireless_stats.qual.updated |= IW_QUAL_DBM;
#endif
#endif
#endif
}

void CsrWifiSmePacketFilterSetCfmHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
    os_linux_priv_t *priv = drvpriv;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "unifi : CsrWifiSmePacketFilterSetCfmSend: Invalid ospriv.\n"));
        return;
    }

    /* The packet filter set request does not block for a reply */
}

void CsrWifiSmeTspecCfmHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
    os_linux_priv_t *priv = drvpriv;
    CsrWifiSmeTspecCfm *cfm = (CsrWifiSmeTspecCfm *) msg;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "unifi : CsrWifiSmeTspecCfmSend: Invalid ospriv.\n"));
        return;
    }

    (void) sme_complete_request(priv, cfm->status);
}

void CsrWifiSmeTspecIndHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
}

void CsrWifiSmeBlacklistCfmHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
}

void CsrWifiSmeEventMaskSetCfmHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
}

void CsrWifiSmeRoamStartIndHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
}

void CsrWifiSmeRoamCompleteIndHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
}

void CsrWifiSmeAssociationStartIndHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
}

void CsrWifiSmeAssociationCompleteIndHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
}

void CsrWifiSmeIbssStationIndHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
}

void CsrWifiSmeWifiOnIndHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
}

void CsrWifiSmeRestrictedAccessEnableCfmHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
}

void CsrWifiSmeRestrictedAccessDisableCfmHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
}

void CsrWifiSmeCalibrationDataGetCfmHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
}

void CsrWifiSmeCloakedSsidsGetCfmHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
}

void CsrWifiSmeCloakedSsidsSetCfmHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
#ifdef CSR_SUPPORT_WEXT
    os_linux_priv_t *priv = drvpriv;
    CsrWifiSmeCloakedSsidsSetCfm *cfm = (CsrWifiSmeCloakedSsidsSetCfm *) msg;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "unifi : CsrWifiSmeSmeConfigSetCfmSend: Invalid ospriv.\n"));
        return;
    }

    (void) sme_complete_request(priv, cfm->status);
#endif
}

#ifdef CSR_SUPPORT_WEXT
static void check_and_set_tx_rate(CsrUint8 *apIe, CsrUint32 length, CsrWifiFixedTxRate *rateCfg)
{
    CsrUint32 ie_index = 0, ie_length = 0;
    CsrUint8 i;
    rateCfg->txRate = 0;

    while (ie_index < length)
    {
        ie_length = apIe[ie_index + 1];
        switch (apIe[ie_index])
        {
            case 0x01: /* Supported Rate IE */
            case 0x32: /* Extended Supported Rate IE */
                for (i = 0; i < ie_length; i++)
                {
                    if ((apIe[ie_index + 2 + i] & 0x7F) == rateCfg->setRate)
                    {
                        rateCfg->txRate = rateCfg->setRate;
                        return;
                    }
                }
                break;
            case 0x2D: /* HT Capability IE */
                /*
                 * We support MCS0-7, which is mandatory and hence need
                 * not check for the Rx MCS bit mask.
                 * In future we may need to check
                 */
                if (rateCfg->type == CSR_WIFI_WEXT_RATE_TYPE_MCS)
                {
                    rateCfg->txRate = rateCfg->setRate;
                    return;
                }
                break;
            default:
                break;
        }
        ie_index += (ie_length + 2);
    }
}

#endif

void CsrWifiSmeMediaStatusIndHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
#ifdef CSR_SUPPORT_WEXT
    os_linux_priv_t *priv = drvpriv;
    CsrWifiSmeMediaStatusInd *ind = (CsrWifiSmeMediaStatusInd *) msg;

    if (priv->smepriv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiSmeMediaStatusIndSend: invalid smepriv\n", priv ? priv->instance : 0));
        return;
    }

    if (ind->interfaceTag >= CSR_WIFI_MAX_INTERFACES)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiSmeMediaStatusIndHandler: bad interfaceTag=%d\n", priv ? priv->instance : 0, ind->interfaceTag));
        return;
    }

    if (ind->mediaStatus == CSR_WIFI_SME_MEDIA_STATUS_CONNECTED)
    {
        /*
         * Send wireless-extension event up to userland to announce
         * connection.
         */
        wext_send_assoc_event(priv,
                              ind->interfaceTag,
                              (unsigned char *) ind->connectionInfo.bssid.a,
                              (unsigned char *) ind->connectionInfo.assocReqInfoElements,
                              ind->connectionInfo.assocReqInfoElementsLength,
                              (unsigned char *) ind->connectionInfo.assocRspInfoElements,
                              ind->connectionInfo.assocRspInfoElementsLength,
                              (unsigned char *) ind->connectionInfo.assocScanInfoElements,
                              ind->connectionInfo.assocScanInfoElementsLength);

        CSR_LOG_TEXT_INFO((
                              CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG2,
                              "unifi%d: CsrWifiSmeMediaStatusIndSend: IBSS=%02X:%02X:%02X:%02X:%02X:%02X\n",
                              priv ? priv->instance : 0,
                              ind->connectionInfo.bssid.a[0],
                              ind->connectionInfo.bssid.a[1],
                              ind->connectionInfo.bssid.a[2],
                              ind->connectionInfo.bssid.a[3],
                              ind->connectionInfo.bssid.a[4],
                              ind->connectionInfo.bssid.a[5]));

        sme_mgt_packet_filter_set(priv, ind->interfaceTag);
        /* Find the supported rates  */
        if (priv->interfacePriv[ind->interfaceTag]->fixed_tx_rate.setRate && priv->interfacePriv[ind->interfaceTag]->fixed_tx_rate.fixedTxRate)
        {
            check_and_set_tx_rate(ind->connectionInfo.assocRspInfoElements,
                                  ind->connectionInfo.assocRspInfoElementsLength,
                                  &priv->interfacePriv[ind->interfaceTag]->fixed_tx_rate);

            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                                "unifi%d: setRate = %d, txRate = %d\n",
                                priv ? priv->instance : 0,
                                priv->interfacePriv[ind->interfaceTag]->fixed_tx_rate.setRate,
                                priv->interfacePriv[ind->interfaceTag]->fixed_tx_rate.txRate));
        }
    }
    else
    {
        /*
         * Send wireless-extension event up to userland to announce
         * connection lost to a BSS.
         */
        wext_send_disassoc_event(priv, ind->interfaceTag);
        /* Currently siwrate is only supported for STA mode
         * due to unavailability of the SME API to set the rate
         */
        priv->interfacePriv[ind->interfaceTag]->fixed_tx_rate.txRate = 0;
    }
#endif
}

void CsrWifiSmeCoreDumpIndHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
}

void CsrWifiSmeAmpStatusChangeIndHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
}

void CsrWifiSmeActivateCfmHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
}

void CsrWifiSmeDeactivateCfmHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
}

void CsrWifiSmeInterfaceAddIndHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
}

void CsrWifiSmeInterfaceAddCfmHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
}

void CsrWifiSmeInterfaceDelIndHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
}

void CsrWifiSmeInterfaceDelCfmHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
}

void CsrWifiSmeSuspendIndHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
}

void CsrWifiSmeResumeIndHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
}

#ifdef CSR_SUPPORT_WEXT
#ifdef CSR_SUPPORT_WEXT_AP
void CsrWifiNmeApStartCfmHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
    os_linux_priv_t *priv = drvpriv;
    CsrWifiNmeApStartCfm *cfm = (CsrWifiNmeApStartCfm *) msg;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "unifi : CsrWifiNmeApStartCfmSend: Invalid ospriv.\n"));
        return;
    }

    (void) sme_complete_request(priv, cfm->status);
}

void CsrWifiNmeApStopCfmHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
    os_linux_priv_t *priv = drvpriv;
    CsrWifiNmeApStopCfm *cfm = (CsrWifiNmeApStopCfm *) msg;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "unifi : CsrWifiNmeApStopCfmSend: Invalid ospriv.\n"));
        return;
    }

    (void) sme_complete_request(priv, cfm->status);
}

void CsrWifiNmeApConfigSetCfmHandler(void *drvpriv, CsrWifiFsmEvent *msg)
{
    os_linux_priv_t *priv = drvpriv;
    CsrWifiNmeApConfigSetCfm *cfm = (CsrWifiNmeApConfigSetCfm *) msg;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "unifi : CsrWifiNmeApConfigSetCfmSend: Invalid ospriv.\n"));
        return;
    }
    (void) sme_complete_request(priv, cfm->status);
}

#endif
#endif
