/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"
#include "csr_wifi_hip_log_text.h"

#include "os_linux_priv.h"

/*
 * It is OK to set the values to 0xFFFF because RemoteRouter
 * knows to route the messages to the right place
 */
CsrSchedQid CSR_WIFI_ROUTER_IFACEQUEUE = 0xFFFF;
CsrSchedQid CSR_WIFI_SME_IFACEQUEUE = 0xFFFF;
#ifdef CSR_WIFI_NME_ENABLE
CsrSchedQid CSR_WIFI_NME_IFACEQUEUE = 0xFFFF;
#endif
int uf_sme_init(os_linux_priv_t *priv)
{
    int i;

    CsrWifiRouterTransportInit(priv);

    priv->smepriv = priv;

    init_waitqueue_head(&priv->sme_request_wq);

#ifdef CSR_SUPPORT_WEXT
    priv->ignore_bssid_join = FALSE;

    uf_sme_wext_set_defaults(priv);
#endif /* CSR_SUPPORT_WEXT*/

    sema_init(&priv->sme_sem, 1);
    memset(&priv->sme_reply, 0, sizeof(sme_reply_t));
    priv->expected_reply = SME_REPLY_NONE;

    priv->ta_ind_work.in_use = 0;
    priv->ta_sample_ind_work.in_use = 0;

    priv->CSR_WIFI_SME_IFACEQUEUE = 0xFFFF;

    for (i = 0; i < MAX_MA_PACKET_IND_FILTERS; i++)
    {
        priv->sme_ma_packet_ind_filters[i].in_use = 0;
    }
    priv->num_of_ma_packet_subscriptions = 0;

    /* Create a work queue item for Traffic Analysis indications to SME */
    INIT_WORK(&priv->ta_ind_work.task, uf_ta_ind_wq);
    INIT_WORK(&priv->ta_sample_ind_work.task, uf_ta_sample_ind_wq);
#ifdef CSR_SUPPORT_WEXT
    INIT_WORK(&priv->sme_config_task, uf_sme_config_wq);
#endif

    for (i = 0; i < CSR_WIFI_MAX_INTERFACES; i++)
    {
        netInterface_priv_t *interfacePriv = priv->interfacePriv[i];
        interfacePriv->filter_tclas_ies = NULL;
        memset(&interfacePriv->packet_filters, 0, sizeof(uf_cfg_bcast_packet_filter_t));
        interfacePriv->m4_hostTag = CSR_WIFI_EAPOL_M4_NULL_HOST_TAG;
        interfacePriv->m4_pending_hostTag = CSR_WIFI_EAPOL_M4_NULL_HOST_TAG;
        interfacePriv->m4_bulk_data.d[0].net_buf_length = 0;
        interfacePriv->m4_bulk_data.d[0].data_length = 0;
        interfacePriv->m4_bulk_data.d[0].os_data_ptr = NULL;
        interfacePriv->m4_bulk_data.d[0].os_net_buf_ptr = NULL;
        memset(&interfacePriv->m4_info, 0, sizeof(os_linux_store_m4_info));
        interfacePriv->sta_ip_address = 0xFFFFFFFF;
    }


    return 0;
} /* uf_sme_init() */

void uf_sme_deinit(os_linux_priv_t *priv)
{
    int i;
    netInterface_priv_t *interfacePriv;

    for (i = 0; i < MAX_MA_PACKET_IND_FILTERS; i++)
    {
        priv->sme_ma_packet_ind_filters[i].in_use = 0;
    }
    priv->num_of_ma_packet_subscriptions = 0;

    /* Remove all the Peer database, before going down */
    for (i = 0; i < CSR_WIFI_MAX_INTERFACES; i++)
    {
        interfacePriv = priv->interfacePriv[i];
        interfacePriv->interfaceMode = CSR_WIFI_ROUTER_CTRL_MODE_NONE;
        /* Free any TCLASs previously allocated */
        if (interfacePriv->packet_filters.tclas_ies_length)
        {
            interfacePriv->packet_filters.tclas_ies_length = 0;
            CsrPmemFree(interfacePriv->filter_tclas_ies);
            interfacePriv->filter_tclas_ies = NULL;
        }
    }
} /* uf_sme_deinit() */

/*
 * ---------------------------------------------------------------------------
 *  linux_ta_indicate_protocol
 *
 *      Report that a packet of a particular type has been seen
 *
 *  Arguments:
 *      drv_priv        The device context pointer passed to ta_init.
 *      protocol        The protocol type enum value.
 *      direction       Whether the packet was a tx or rx.
 *      src_addr        The source MAC address from the data packet.
 *
 *  Returns:
 *      None.
 *
 *  Notes:
 *      We defer the actual sending to a background workqueue,
 *      see uf_ta_ind_wq().
 * ---------------------------------------------------------------------------
 */
void linux_ta_indicate_protocol(os_linux_priv_t *priv, CsrUint16 interfaceTag,
                                CsrWifiRouterCtrlTrafficPacketType packet_type,
                                CsrWifiRouterCtrlProtocolDirection direction,
                                const CsrWifiMacAddress *src_addr)
{
    if (priv->ta_ind_work.in_use)
    {
        CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: linux_ta_indicate_protocol: workqueue item still in use, not sending\n",
                              priv->instance));
        return;
    }

    if (CSR_WIFI_ROUTER_CTRL_PROTOCOL_DIRECTION_RX == direction)
    {
        CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID,
                              CSR_WIFI_HIP_LOG_DEF,
                              "unifi%d: linux_ta_indicate_protocol: interfaceTag = %d",
                              priv->instance,
                              interfaceTag));

        CsrWifiRouterCtrlTrafficProtocolIndSend(priv->CSR_WIFI_SME_IFACEQUEUE, 0,
                                                interfaceTag,
                                                packet_type,
                                                direction,
                                                *src_addr);
    }
    else
    {
        priv->ta_ind_work.packet_type = packet_type;
        priv->ta_ind_work.direction = direction;
        priv->ta_ind_work.src_addr = *src_addr;
        priv->ta_ind_work.interfaceTag = interfaceTag;

        queue_work(priv->unifi_workqueue, &priv->ta_ind_work.task);
    }
} /* linux_ta_indicate_protocol() */

/*
 * ---------------------------------------------------------------------------
 * linux_ta_indicate_sampling
 *
 *      Send the TA sampling information to the SME.
 *
 *  Arguments:
 *      drv_priv        The device context pointer passed to ta_init.
 *      stats   The TA sampling data to send.
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
void linux_ta_indicate_sampling(os_linux_priv_t *priv, CsrUint16 interfaceTag, CsrWifiRouterCtrlTrafficStats *stats)
{
    if (!priv)
    {
        return;
    }

    if (priv->ta_sample_ind_work.in_use)
    {
        CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: linux_ta_indicate_sampling: workqueue item still in use, not sending\n",
                              priv->instance));
        return;
    }

    priv->ta_sample_ind_work.stats = *stats;
    priv->ta_sample_ind_work.interfaceTag = interfaceTag;

    queue_work(priv->unifi_workqueue, &priv->ta_sample_ind_work.task);
} /* linux_ta_indicate_sampling() */
