/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"
#include "csr_wifi_hip_log_text.h"

#include "os_linux_priv.h"
#include "os_linux_priv.h"

#include "csr_types.h"
#include "csr_sched.h"
#include "csr_msgconv.h"

#include "sme_userspace.h"

#include "csr_wifi_hostio_prim.h"
#include "csr_wifi_router_lib.h"
#include "csr_wifi_router_sef.h"
#include "csr_wifi_router_converter_init.h"
#include "csr_wifi_router_ctrl_lib.h"
#include "csr_wifi_router_ctrl_sef.h"
#include "csr_wifi_router_ctrl_converter_init.h"
#include "csr_wifi_router_ctrl_serialize.h"
#include "csr_wifi_sme_prim.h"
#include "csr_wifi_sme_sef.h"
#include "csr_wifi_sme_converter_init.h"
#ifdef CSR_SUPPORT_WEXT
#ifdef CSR_SUPPORT_WEXT_AP
#include "csr_wifi_nme_ap_prim.h"
#include "csr_wifi_nme_ap_sef.h"
#include "csr_wifi_nme_ap_converter_init.h"
#endif
#endif

#define CSR_WIFI_UDI_STRIP_HEADER_SIZE 6

static os_linux_priv_t *drvpriv = NULL;
void CsrWifiRouterTransportInit(os_linux_priv_t *priv)
{
    CSR_LOG_TEXT_INFO((
                          CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: CsrWifiRouterTransportInit: \n"
                          , priv ? priv->instance : 0));

    drvpriv = priv;
    (void) CsrMsgConvInit();
    CsrWifiRouterConverterInit();
    CsrWifiRouterCtrlConverterInit();
    CsrWifiSmeConverterInit();
#ifdef CSR_SUPPORT_WEXT
#ifdef CSR_SUPPORT_WEXT_AP
    CsrWifiNmeApConverterInit();
#endif
#endif
}

void CsrWifiRouterTransportDeinit(os_linux_priv_t *priv)
{
    CSR_LOG_TEXT_INFO((
                          CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: CsrWifiRouterTransportDeinit: \n"
                          , priv ? priv->instance : 0));
    if (priv == drvpriv)
    {
        CsrMsgConvDeinit();
        drvpriv = NULL;
    }
}

void CsrWifiRouterTransportRecv(void *inst, CsrUint8 *buffer, CsrSize bufferLength)
{
    CsrMsgConvMsgEntry *msgEntry;
    CsrUint16 primType;
    CsrSchedQid src;
    CsrSchedQid dest;
    CsrUint16 msgType;
    CsrSize offset = 0;
    CsrWifiFsmEvent *msg;
    os_linux_priv_t *priv = inst;

    /* Decode the prim and message type */
    CsrUint16Des(&primType, buffer, &offset);
    CsrUint16Des(&src, buffer, &offset);
    CsrUint16Des(&dest, buffer, &offset);
    CsrUint16Des(&msgType, buffer, &offset);
    offset -= 2; /* Adjust as the Deserialise Function will read this as well */

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: CsrWifiRouterTransportRecv: primType=0x%.4X, msgType=0x%.4X, bufferLength=%d\n",
                        priv ? priv->instance : 0,
                        primType, msgType, bufferLength));

    /* Special handling for HOSTIO messages.... */
    if (primType == CSR_WIFI_HOSTIO_PRIM)
    {
        CsrWifiRouterCtrlHipReq req = {{CSR_WIFI_ROUTER_CTRL_HIP_REQ, CSR_WIFI_ROUTER_CTRL_PRIM, dest, src, NULL}, 0, NULL, 0, NULL, 0, NULL};

        req.mlmeCommandLength = bufferLength;
        req.mlmeCommand = buffer;

        offset += 8; /* Skip the id, src, dest and slot number */
        CsrUint16Des(&req.dataRef1Length, buffer, &offset);
        offset += 2; /* Skip the slot number */
        CsrUint16Des(&req.dataRef2Length, buffer, &offset);

        if (req.dataRef1Length)
        {
            CsrUint16 dr1Offset = (bufferLength - req.dataRef2Length) - req.dataRef1Length;
            req.dataRef1 = &buffer[dr1Offset];
        }

        if (req.dataRef2Length)
        {
            CsrUint16 dr2Offset = bufferLength - req.dataRef2Length;
            req.dataRef2 = &buffer[dr2Offset];
        }

        /* Copy the hip data but strip off the prim type */
        req.mlmeCommandLength -= (req.dataRef1Length + req.dataRef2Length + 6);
        req.mlmeCommand = &buffer[6];

        /* Log Host signals to the UDI logger if that mode is set */
        if (priv->logging_client && (priv->logging_client->configuration & CLI_LOG_HOST_ALL))
        {
            /* Special for HipReq messages. */
            static const CsrWifiHipBulkDataParam bulkdata = {{{NULL, 0, NULL, 0}, {NULL, 0, NULL, 0}}};
            CsrSize hipReqSize = CsrWifiRouterCtrlHipReqSizeof(&req);
            CsrUint8 *hipReq = CsrPmemAlloc(hipReqSize);
            CsrWifiRouterCtrlHipReqSer(hipReq, &hipReqSize, &req);
            logging_handler(priv,
                            hipReq,
                            (CsrUint32) hipReqSize,
                            &bulkdata,
                            UDI_FROM_HOST);
            CsrPmemFree(hipReq);
        }

        CsrWifiRouterCtrlHipReqHandler(priv, &req.common);
        return;
    }

    /* Log Host signals to the UDI logger if that mode is set */
    if (priv->logging_client && (priv->logging_client->configuration & CLI_LOG_HOST))
    {
        static const CsrWifiHipBulkDataParam bulkdata = {{{NULL, 0, NULL, 0}, {NULL, 0, NULL, 0}}};
        logging_handler(priv,
                        &buffer[CSR_WIFI_UDI_STRIP_HEADER_SIZE],
                        bufferLength - CSR_WIFI_UDI_STRIP_HEADER_SIZE,
                        &bulkdata,
                        UDI_FROM_HOST);
    }


    msgEntry = CsrMsgConvFindEntry(primType, msgType);
    if (!msgEntry)
    {
        CSR_LOG_TEXT_BUFFER_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, bufferLength, buffer, "unifi%d: CsrWifiRouterTransportDeserialiseAndSend can not process the message. primType=0x%.4X, msgType=0x%.4X\n",
                                      priv ? priv->instance : 0, primType, msgType));
        return;
    }

    msg = (CsrWifiFsmEvent *) (msgEntry->deserFunc)(&buffer[offset], bufferLength - offset);

    msg->primtype = primType;
    msg->type = msgType;
    msg->source = src;
    msg->destination = dest;

    switch (primType)
    {
        case CSR_WIFI_ROUTER_CTRL_PRIM:
            CsrWifiRouterCtrlDownstreamStateHandlers[msg->type - CSR_WIFI_ROUTER_CTRL_PRIM_DOWNSTREAM_LOWEST](priv, msg);
            CsrWifiRouterCtrlFreeDownstreamMessageContents(CSR_WIFI_ROUTER_CTRL_PRIM, msg);
            break;
        case CSR_WIFI_ROUTER_PRIM:
            CsrWifiRouterDownstreamStateHandlers[msg->type - CSR_WIFI_ROUTER_PRIM_DOWNSTREAM_LOWEST](priv, msg);
            CsrWifiRouterFreeDownstreamMessageContents(CSR_WIFI_ROUTER_PRIM, msg);
            break;
        case CSR_WIFI_SME_PRIM:
            CsrWifiSmeUpstreamStateHandlers[msg->type - CSR_WIFI_SME_PRIM_UPSTREAM_LOWEST](priv, msg);
            CsrWifiSmeFreeUpstreamMessageContents(CSR_WIFI_SME_PRIM, msg);
            break;
#ifdef CSR_SUPPORT_WEXT
#ifdef CSR_SUPPORT_WEXT_AP
        case CSR_WIFI_NME_AP_PRIM:
            CsrWifiNmeApUpstreamStateHandlers(priv, msg);
            CsrWifiNmeApFreeUpstreamMessageContents(CSR_WIFI_NME_AP_PRIM, msg);
            break;
#endif
#endif
        default:
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiRouterTransportDeserialiseAndSend unhandled prim type 0x%.4X\n",
                                   priv ? priv->instance : 0,  primType));
            break;
    }
    CsrPmemFree(msg);
}

static void CsrWifiRouterTransportSerialiseAndSend(CsrUint16 primType, void *msg)
{
    CsrWifiFsmEvent *evt = (CsrWifiFsmEvent *) msg;
    CsrMsgConvMsgEntry *msgEntry;
    CsrSize msgSize;
    CsrSize encodeBufferLen = 0;
    CsrSize offset = 0;
    CsrUint8 *encodeBuffer;

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: CsrWifiRouterTransportSerialiseAndSend: primType=0x%.4X, msgType=0x%.4X\n",
                        drvpriv ? drvpriv->instance : 0,
                        primType, evt->type));

    msgEntry = CsrMsgConvFindEntry(primType, evt->type);
    if (!msgEntry)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiRouterTransportSerialiseAndSend can not process the message. primType=0x%.4X, msgType=0x%.4X\n",
                               drvpriv ? drvpriv->instance : 0,
                               primType, evt->type));
        return;
    }

    msgSize = 6 + (msgEntry->sizeofFunc)((void *) msg);

    encodeBuffer = CsrPmemAlloc(msgSize);

    /* Encode PrimType */
    CsrUint16Ser(encodeBuffer, &encodeBufferLen, primType);
    CsrUint16Ser(encodeBuffer, &encodeBufferLen, evt->source);
    CsrUint16Ser(encodeBuffer, &encodeBufferLen, evt->destination);

    (void) (msgEntry->serFunc)(&encodeBuffer[encodeBufferLen], &offset, msg);
    encodeBufferLen += offset;

    /* Log Host signals to the UDI logger if that mode is set */
    if (drvpriv->logging_client && (drvpriv->logging_client->configuration & CLI_LOG_HOST))
    {
        if ((drvpriv->logging_client->configuration & CLI_LOG_HOST_ALL) ||
            !((primType == CSR_WIFI_ROUTER_CTRL_PRIM) && (evt->type == CSR_WIFI_ROUTER_CTRL_HIP_IND)))
        {
            static const CsrWifiHipBulkDataParam bulkdata = {{{NULL, 0, NULL, 0}, {NULL, 0, NULL, 0}}};
            logging_handler(drvpriv,
                            &encodeBuffer[CSR_WIFI_UDI_STRIP_HEADER_SIZE],
                            encodeBufferLen - CSR_WIFI_UDI_STRIP_HEADER_SIZE,
                            &bulkdata,
                            UDI_TO_HOST);
        }
    }

    uf_sme_queue_message(drvpriv, encodeBuffer, encodeBufferLen);

    /* Do not use msgEntry->freeFunc because the memory is owned by the driver */
    CsrPmemFree(msg);
}

#if defined(CSR_LOG_ENABLE) && defined(CSR_LOG_INCLUDE_FILE_NAME_AND_LINE_NUMBER)
void CsrSchedMessagePutStringLog(CsrSchedQid          q,
                                 CsrUint16            mi,
                                 void                *mv,
                                 CsrUint32            line,
                                 const CsrCharString *file)
#define CsrSchedMessagePut(q, mi, mv) CsrSchedMessagePutStringLog((q), (mi), (mv), __LINE__, __FILE__)
#else
void CsrSchedMessagePut(CsrSchedQid q,
                        CsrUint16   mi,
                        void       *mv)
#endif
{
    CsrWifiFsmEvent *evt = (CsrWifiFsmEvent *) mv;
    evt->destination = q;
    CsrWifiRouterTransportSerialiseAndSend(mi, mv);
}
