/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"
#include "csr_wifi_hip_log_text.h"
#include "os_linux_priv.h"
#include "csr_wifi_hip.h"

/*
 * Porting Notes:
 * This file implements the data plane of the UniFi linux driver.
 *
 * All the Tx packets are passed to the HIP core lib, using the
 * unifi_send_signal() API. For EAPOL packets use the MLME-EAPOL.req
 * signal, for all other use the MLME-UNITDATA.req. The unifi_send_signal()
 * expects the wire-formatted (packed) signal. For convenience, in the OS
 * layer we only use the native (unpacked) signal structures. The HIP core lib
 * provides the write_pack() helper function to convert to the packed signal.
 * The packet is stored in the bulk data of the signal. We do not need to
 * allocate new memory to store the packet, because unifi_net_data_malloc()
 * is implemented to return a skb, which is the format of packet in Linux.
 * The HIP core lib frees the bulk data buffers, so we do not need to do
 * this in the OS layer.
 *
 * All the Rx packets are MLME-UNITDATA.ind signals, passed by the HIP core lib
 * in unifi_receive_event(). We do not need to allocate an skb and copy the
 * received packet because the HIP core lib has used memory allocated by
 * CsrWifiHipBulkDataAlloc function pointer we passed in CsrWifiHipWifiOnReq(). Also, we can perform the 802.11 to Ethernet
 * translation in-place because we allocate the extra memory in
 * os_linux_net_data_malloc().
 *
 * If possible, the porting exercise should appropriately implement
 * CsrWifiHipBulkDataAlloc and CsrWifiHipBulkDataFree to save copies between
 * network and driver buffers.
 */

#include <linux/types.h>
#include <linux/etherdevice.h>
#include <linux/mutex.h>
#include <linux/semaphore.h>

#include <linux/vmalloc.h>
#include "csr_wifi_hip_unifi.h"
#include "csr_wifi_hip_conversions.h"
#include "os_linux_priv.h"
#include <net/pkt_sched.h>

#define UNIFI_NET_IF_Q(aid, ac) ((aid == 0) ? ac : (((aid - CSR_WIFI_HIP_AID_START) * UNIFI_TRAFFIC_Q_MAX) + ac))

#ifdef UNIFI_NET_NAME
#define UF_ALLOC_NETDEV(_dev, _size, _name, _setup, _num_of_queues)     \
    do {                                                                \
        static char name[8];                                           \
        sprintf(name, "%s%s", UNIFI_NET_NAME, _name);                   \
        _dev = alloc_netdev_mq(_size, name, _setup, _num_of_queues);    \
    } while (0);
#else
#define UF_ALLOC_NETDEV(_dev, _size, _name, _setup, _num_of_queues)     \
    do {                                                                \
        _dev = alloc_etherdev_mq(_size, _num_of_queues);                \
    } while (0);
#endif /* UNIFI_NET_NAME */


/* Wext handler is suported only if CSR_SUPPORT_WEXT is defined */
#ifdef CSR_SUPPORT_WEXT
extern struct iw_handler_def unifi_iw_handler_def;
#endif /* CSR_SUPPORT_WEXT */
static int uf_net_open(struct net_device *dev);
static int uf_net_ioctl(struct net_device *dev, struct ifreq *rq, int cmd);
static int uf_net_stop(struct net_device *dev);
static struct net_device_stats *uf_net_get_stats(struct net_device *dev);
static u16 uf_net_select_queue(struct net_device *dev, struct sk_buff *skb);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 32)
static netdev_tx_t uf_net_xmit(struct sk_buff *skb, struct net_device *dev);
#else
static int uf_net_xmit(struct sk_buff *skb, struct net_device *dev);
#ifndef NETDEV_TX_OK
#define NETDEV_TX_OK        0
#endif
#ifndef NETDEV_TX_BUSY
#define NETDEV_TX_BUSY      1
#endif
#endif
#if LINUX_VERSION_CODE < KERNEL_VERSION(3, 2, 0)
static void uf_set_multicast_list(struct net_device *dev);
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 29)
static const struct net_device_ops uf_netdev_ops =
{
    .ndo_open = uf_net_open,
    .ndo_stop = uf_net_stop,
    .ndo_start_xmit = uf_net_xmit,
    .ndo_do_ioctl = uf_net_ioctl,
    .ndo_get_stats = uf_net_get_stats, /* called by /proc/net/dev */
#if LINUX_VERSION_CODE < KERNEL_VERSION(3, 2, 0)
    .ndo_set_multicast_list = uf_set_multicast_list,
#endif
    .ndo_select_queue = uf_net_select_queue,
};
#endif

static u8 oui_rfc1042[P80211_OUI_LEN] = {0x00, 0x00, 0x00};
static u8 oui_8021h[P80211_OUI_LEN] = {0x00, 0x00, 0xf8};


#ifdef CSR_SUPPORT_WEXT
/* Declare netdev_notifier block which will contain the state change
 * handler callback function
 */
static struct notifier_block uf_netdev_notifier;
#endif


/* Shared function for allocating and configuring a netdev */
static struct net_device *alloc_and_configure_netdev(int allocSize)
{
    struct net_device *dev = NULL;

    if (allocSize <= 0)
    {
        return NULL;
    }

    UF_ALLOC_NETDEV(dev, allocSize, "%d", ether_setup, UNIFI_TRAFFIC_Q_MAX * CSR_WIFI_HIP_PEER_CONNECTIONS_MAX);
    if (dev == NULL)
    {
        return NULL;
    }

    /* Setup / override net_device fields */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 29)
    dev->netdev_ops = &uf_netdev_ops;
#else
    dev->open = uf_net_open;
    dev->stop = uf_net_stop;
    dev->hard_start_xmit = uf_net_xmit;
    dev->do_ioctl = uf_net_ioctl;

    /* called by /proc/net/dev */
    dev->get_stats = uf_net_get_stats;

    dev->set_multicast_list = uf_set_multicast_list;
    dev->select_queue = uf_net_select_queue;
#endif

#ifdef CSR_SUPPORT_WEXT
    dev->wireless_handlers = &unifi_iw_handler_def;
#if IW_HANDLER_VERSION < 6
    dev->get_wireless_stats = unifi_get_wireless_stats;
#endif /* IW_HANDLER_VERSION */
#endif /* CSR_SUPPORT_WEXT */

    /* This gives us enough headroom to add the 802.11 header */
    dev->needed_headroom = 32;

    return dev;
}

/*
 * ---------------------------------------------------------------------------
 *  uf_alloc_netdevice
 *
 *      Allocate memory for the net_device and device private structs
 *      for this interface.
 *      Fill in the fields, but don't register the interface yet.
 *      We need to configure the UniFi first.
 *
 *  Arguments:
 *      bus_id          A small number indicating the SDIO card position on the
 *                      bus. Typically this is the slot number, e.g. 0, 1 etc.
 *                      Valid values are 0 to MAX_UNIFI_DEVS-1.
 *
 *  Returns:
 *      Pointer to device private struct.
 *
 *  Notes:
 *      The net_device and device private structs are allocated together
 *      and should be freed by freeing the net_device pointer.
 * ---------------------------------------------------------------------------
 */
os_linux_priv_t *uf_alloc_netdevice(int bus_id)
{
    struct net_device *dev;
    os_linux_priv_t *priv;
    netInterface_priv_t *interfacePriv;
#ifdef CSR_SUPPORT_WEXT
    int rc;
#endif
    unsigned char i; /* loop index */

    /*
     * Allocate netdevice struct, assign name template and
     * setup as an ethernet device.
     * The net_device and private structs are zeroed. Ether_setup() then
     * sets up ethernet handlers and values.
     * The RedHat 9 redhat-config-network tool doesn't recognise wlan* devices,
     * so use "eth*" (like other wireless extns drivers).
     */
    dev = alloc_and_configure_netdev(sizeof(os_linux_priv_t) + sizeof(netInterface_priv_t));
    if (dev == NULL)
    {
        return NULL;
    }

    /* Set up back pointer from priv to netdev */
    interfacePriv = (netInterface_priv_t *) netdev_priv(dev);
    priv = (os_linux_priv_t *) (interfacePriv + 1);
    interfacePriv->privPtr = priv;
    interfacePriv->InterfaceTag = 0;


    /* Initialize all supported netdev interface to be NULL */
    for (i = 0; i < CSR_WIFI_MAX_INTERFACES; i++)
    {
        priv->netdev[i] = NULL;
        priv->interfacePriv[i] = NULL;
    }
    priv->netdev[0] = dev;
    priv->interfacePriv[0] = interfacePriv;
#ifdef CSR_SUPPORT_WEXT
    CsrMemSet(&interfacePriv->fixed_tx_rate, 0, sizeof(interfacePriv->fixed_tx_rate));
#endif

    /* Use bus_id as instance number */
    priv->instance = bus_id;

    /* Consider UniFi to be uninitialised */
    priv->init_progress = UNIFI_INIT_NONE;

    priv->prev_queue = 0;

    /*
     * Initialise the clients structure array.
     * We do not need protection around ul_init_clients() because
     * the character device can not be used until uf_alloc_netdevice()
     * returns and Unifi_instances[bus_id]=priv is set, since unifi_open()
     * will return -ENODEV.
     */
    ul_init_clients(priv);

    /*
     * Register a new ul client to send the multicast list signals.
     * Note: priv->instance must be set before calling this.
     * The netdev client is not used for routing signals to the network
     * stack, so the call back function is set to NULL. However, the
     * clientId is used for routing signals, so the client must be registered.
     */
    priv->netdev_client = ul_register_client(priv,
                                             0,
                                             NULL);
    if (priv->netdev_client == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: Failed to register a unifi client for background netdev processing\n",
                               priv->instance));
        free_netdev(priv->netdev[0]);
        return NULL;
    }

    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                       "unifi%d: Netdev %p client (id:%d s:0x%X) is registered\n",
                       priv->instance, dev, priv->netdev_client->client_id, priv->netdev_client->sender_id));

    /*
     * Initialise the OS private struct.
     */
    /* reset the connected state for the interface */
    interfacePriv->connected = UnifiConnectedUnknown;  /* -1 unknown, 0 no, 1 yes */

    spin_lock_init(&priv->m4_lock);

    /* Create the Traffic Analysis workqueue */
    priv->unifi_workqueue = create_singlethread_workqueue("unifi_workq");
    if (priv->unifi_workqueue == NULL)
    {
        /* Deregister priv->netdev_client */
        ul_deregister_client(priv->netdev_client);
        free_netdev(priv->netdev[0]);
        return NULL;
    }

#ifdef CSR_SUPPORT_SME
    /* Create the Multicast Addresses list work structure */
    INIT_WORK(&priv->multicast_list_task, uf_multicast_list_wq);
#endif

    priv->ref_count = 1;


#ifndef CSR_WIFI_DRIVER_HYDRA
    priv->coredump_mode = 0;
    priv->ptest_mode = 0;
#endif /* CSR_WIFI_DRIVER_HYDRA */

#ifdef CSR_SUPPORT_WEXT
    interfacePriv->netdev_callback_registered = FALSE;
    interfacePriv->wait_netdev_change = FALSE;
    /* Register callback for netdevice state changes */
    if ((rc = register_netdevice_notifier(&uf_netdev_notifier)) == 0)
    {
        interfacePriv->netdev_callback_registered = TRUE;
    }
    else
    {
        CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                              "unifi%d: Failed to register netdevice notifier : %d %p\n",
                              priv->instance, rc, dev));
    }
#endif /* CSR_SUPPORT_WEXT */

    return priv;
} /* uf_alloc_netdevice() */

/*
 *---------------------------------------------------------------------------
 *  uf_alloc_netdevice_for_other_interfaces
 *
 *      Allocate memory for the net_device and device private structs
 *      for this interface.
 *      Fill in the fields, but don't register the interface yet.
 *      We need to configure the UniFi first.
 *
 *  Arguments:
 *      priv            Instance data for the os linux instance.
 *      interfaceTag    Interface number.
 *
 *  Returns:
 *      Pointer to device private struct.
 *
 *  Notes:
 *      The device private structure contains the interfaceTag and pointer to the os_linux_priv_t
 *      structure created allocated by net_device od interface0.
 *      The net_device and device private structs are allocated together
 *      and should be freed by freeing the net_device pointer.
 * ---------------------------------------------------------------------------
 */
CsrBool uf_alloc_netdevice_for_other_interfaces(os_linux_priv_t *priv, CsrUint16 interfaceTag)
{
    struct net_device *dev;
    netInterface_priv_t *interfacePriv;

    if (interfaceTag >= CSR_WIFI_MAX_INTERFACES)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: uf_alloc_netdevice_for_other_interfaces bad interfaceTag\n",
                               priv->instance));
        return FALSE;
    }

    /*
     * Allocate netdevice struct, assign name template and
     * setup as an ethernet device.
     * The net_device and private structs are zeroed. Ether_setup() then
     * sets up ethernet handlers and values.
     * The RedHat 9 redhat-config-network tool doesn't recognise wlan* devices,
     * so use "eth*" (like other wireless extns drivers).
     */
    dev = alloc_and_configure_netdev(sizeof(netInterface_priv_t));
    if (dev == NULL)
    {
        return FALSE;
    }

    /* Set up back pointer from priv to netdev */
    interfacePriv = (netInterface_priv_t *) netdev_priv(dev);
    interfacePriv->privPtr = priv;
    interfacePriv->InterfaceTag = interfaceTag;
    priv->netdev[interfaceTag] = dev;
    priv->interfacePriv[interfacePriv->InterfaceTag] = interfacePriv;


    /* reset the connected state for the interface */
    interfacePriv->connected = UnifiConnectedUnknown;  /* -1 unknown, 0 no, 1 yes */

    return TRUE;
} /* uf_alloc_netdevice() */

/*
 * ---------------------------------------------------------------------------
 *  uf_free_netdevice
 *
 *      Unregister the network device and free the memory allocated for it.
 *      NB This includes the memory for the priv struct.
 *
 *  Arguments:
 *      priv            Device private pointer.
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
int uf_free_netdevice(os_linux_priv_t *priv)
{
    int i;
    unsigned long flags;

    func_enter();

    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                       "unifi%d: uf_free_netdevice\n",
                       priv ? priv->instance : 0));

    if (!priv)
    {
        return -EINVAL;
    }

#if (defined CSR_SUPPORT_SME) && (defined CSR_SUPPORT_WEXT)
    /* Shouldn't this stuff be per-interface? */
    if (priv->connection_config.mlmeAssociateReqInformationElements)
    {
        kfree(priv->connection_config.mlmeAssociateReqInformationElements);
    }
    priv->connection_config.mlmeAssociateReqInformationElements = NULL;
    priv->connection_config.mlmeAssociateReqInformationElementsLength = 0;

    if (priv->scanInformationElements)
    {
        kfree(priv->scanInformationElements);
    }
    priv->scanInformationElementsLength = 0;
    priv->scanInformationElements = NULL;
#endif /* CSR_SUPPORT_SME && CSR_SUPPORT_WEXT*/



    /* Free any bulkdata buffers allocated for M4 caching */
    spin_lock_irqsave(&priv->m4_lock, flags);
    for (i = 0; i < CSR_WIFI_MAX_INTERFACES; i++)
    {
        netInterface_priv_t *interfacePriv = priv->interfacePriv[i];
        memset(&interfacePriv->m4_info, 0, sizeof(os_linux_store_m4_info));
        if (interfacePriv->m4_bulk_data.d[0].data_length > 0)
        {
            CSR_LOG_TEXT_DEBUG((
                                   CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG5,  "unifi%d: uf_free_netdevice: free M4 bulkdata %d\n",
                                   priv ? priv->instance : 0,  i));
            os_linux_net_data_free_all(priv, &interfacePriv->m4_bulk_data);
        }
    }
    spin_unlock_irqrestore(&priv->m4_lock, flags);

#ifdef CSR_SUPPORT_WEXT
    /* Unregister callback for netdevice state changes */
    unregister_netdevice_notifier(&uf_netdev_notifier);
#endif /* CSR_SUPPORT_WEXT */

#ifdef CSR_SUPPORT_SME
    /* Cancel work items and destroy the workqueue */
    cancel_work_sync(&priv->multicast_list_task);
#endif
    /* Destroy the workqueues. */
    flush_workqueue(priv->unifi_workqueue);
    destroy_workqueue(priv->unifi_workqueue);

    /* Free up netdev in reverse order: priv is allocated with netdev[0].
     * So, netdev[0] should be freed after all other netdevs are freed up
     */
    for (i = CSR_WIFI_MAX_INTERFACES - 1; i >= 0; i--)
    {
        /*Free the netdev struct and priv, which are all one lump*/
        if (priv->netdev[i])
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: uf_free_netdevice: netdev %d %p\n",
                                   priv ? priv->instance : 0,  i, priv->netdev[i]));
            free_netdev(priv->netdev[i]);
        }
    }

    func_exit();
    return 0;
} /* uf_free_netdevice() */

/*
 * ---------------------------------------------------------------------------
 *  uf_net_open
 *
 *      Called when userland does "ifconfig wlan0 up".
 *
 *  Arguments:
 *      dev             Device pointer.
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
static int uf_net_open(struct net_device *dev)
{
    netInterface_priv_t *interfacePriv = (netInterface_priv_t *) netdev_priv(dev);
    os_linux_priv_t *priv = interfacePriv->privPtr;

    func_enter();

    /* If we haven't finished UniFi initialisation, we can't start */
    if (priv->init_progress != UNIFI_INIT_COMPLETED)
    {
        CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID,
                              CSR_WIFI_HIP_LOG_DEF, "unifi%d: %s: unifi not ready, failing net_open\n",
                              priv ? priv->instance : 0,  __FUNCTION__));
        return -EINVAL;
    }

#ifdef CSR_SUPPORT_WEXT
    if (interfacePriv->wait_netdev_change)
    {
        CSR_LOG_TEXT_INFO((
                              CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: %s: Waiting for NETDEV_CHANGE, assume connected\n",
                              priv ? priv->instance : 0,
                              __FUNCTION__));
        interfacePriv->connected = UnifiConnected;
        interfacePriv->wait_netdev_change = FALSE;
    }
#endif

#ifdef CSR_NATIVE_SOFTMAC
    interfacePriv->connected = UnifiConnected;
#endif

    UF_NETIF_TX_START_ALL_QUEUES(dev);

    func_exit();
    return 0;
} /* uf_net_open() */

static int uf_net_stop(struct net_device *dev)
{
    func_enter();

    UF_NETIF_TX_STOP_ALL_QUEUES(dev);

    func_exit();
    return 0;
} /* uf_net_stop() */

/* This is called after the WE handlers */
static int uf_net_ioctl(struct net_device *dev, struct ifreq *rq, int cmd)
{
    int rc;

    rc = -EOPNOTSUPP;

    return rc;
} /* uf_net_ioctl() */

static struct net_device_stats *uf_net_get_stats(struct net_device *dev)
{
    netInterface_priv_t *interfacePriv = (netInterface_priv_t *) netdev_priv(dev);
    os_linux_priv_t *priv = NULL;
    CsrResult result;
    CsrWifiHipReceiveStat halStats;

    priv = (os_linux_priv_t *) interfacePriv->privPtr;
    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi : uf_net_get_stats: priv is NULL\n"));
        return &interfacePriv->stats;
    }

    memset(&halStats, 0, sizeof(CsrWifiHipReceiveStat));

    /* Query HAL for RX stats */
    result = CsrWifiHipStatGetReq(priv->hip_handle, interfacePriv->InterfaceTag, &halStats);
    if (result != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: uf_net_get_stats: CsrWifiHipStatGetReq failed\n", priv->instance));
        return &interfacePriv->stats;
    }

    /* Since received frames can be dropped both in the os_linux specific code
       and in HAL, we have to query stats from both entities. The transmit stats
       can be handled directly in the os_linux specific code.
       Stats for good cases, i.e. number of transmitted/received frames, number
       of transmitted/received bytes etc. are handled directly in the os_linux
       specific code as well as number of dropped transmit frames. */
    interfacePriv->stats.rx_dropped = interfacePriv->rx_stats_dropped_frames + halStats.rxDropped;

    return &interfacePriv->stats;
} /* uf_net_get_stats() */

/*
 * ---------------------------------------------------------------------------
 *  uf_net_select_queue
 *
 *      Called by the kernel to select which queue to put the packet in
 *
 *  Arguments:
 *      dev             Device pointer
 *      skb             Packet
 *
 *  Returns:
 *      Queue index
 * ---------------------------------------------------------------------------
 */
static u16 uf_net_select_queue(struct net_device *dev, struct sk_buff *skb)
{
    netInterface_priv_t *interfacePriv;
    os_linux_priv_t *priv = 0;
    unifi_TrafficQueue queue;
    CsrWifiHipMaPacketPriority priority = CSR_WIFI_HIP_MA_PACKET_PRIORITY_CONTENTION;
    CsrInt32 aid = 0;
    CsrUint16 netif_q;

    func_enter();

    if (!dev)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi0: uf_net_select_queue: !dev\n"));
        goto out;
    }

    interfacePriv = (netInterface_priv_t *) netdev_priv(dev);
    if (!interfacePriv)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi0: uf_net_select_queue: !interfacePriv\n"));

        goto out;
    }

    priv = (os_linux_priv_t *) interfacePriv->privPtr;
    if (!priv)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi0: uf_net_select_queue: !priv\n"));

        goto out;
    }

    if (interfacePriv->netdev_state == OS_LINUX_NETDEV_STATE_REGISTERED)
    {
        struct ethhdr *ehdr;
        int proto;
        CsrResult result;

        ehdr = (struct ethhdr *) skb->data;
        proto = ntohs(skb->protocol);

        result = CsrWifiHipPriorityGetReq(priv->hip_handle,
                                          interfacePriv->InterfaceTag,
                                          proto,
                                          (CsrUint8 *) skb->data + ETH_HLEN,
                                          (CsrUint8 *) ehdr->h_dest,
                                          &priority,
                                          &aid);
        if (result != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi%d: Failed to get QoS type and or frame priority.\n",
                                   priv->instance));
            priority = CSR_WIFI_HIP_MA_PACKET_PRIORITY_CONTENTION;
        }
        if ((aid < CSR_WIFI_HIP_AID_START) || (aid > CSR_WIFI_HIP_AID_END))
        {
            aid = 0;
        }
    }

out:
    skb->priority = priority;
    queue = unifi_frame_priority_to_queue((CSR_PRIORITY) priority);
    netif_q = UNIFI_NET_IF_Q(aid, queue);
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG7, "unifi%d: uf_net_select_queue: queue=%d\n", priv ? priv->instance : 0, netif_q));
    return netif_q;
}

int skb_add_llc_snap(struct net_device *dev, struct sk_buff *skb, int proto)
{
    llc_snap_hdr_t *snap;
    int headroom;
    #ifdef CSR_LOG_ENABLE
    netInterface_priv_t *interfacePriv = (netInterface_priv_t *) netdev_priv(dev);
    #endif

    /* get the headroom available in skb */
    headroom = skb_headroom(skb);
    /* step 1: classify ether frame, DIX or 802.3? */

    if (proto < 0x600)
    {
        /* codes <= 1500 reserved for 802.3 lengths */
        /* it's 802.3, pass ether payload unchanged,  */
        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG3,  "unifi%d: 802.3 len: %d\n",
                               interfacePriv->privPtr ? ((os_linux_priv_t *) interfacePriv->privPtr)->instance : 0,  skb->len));

        /*   leave off any PAD octets.  */
        skb_trim(skb, proto);
    }
    else if (proto == ETH_P_8021Q)
    {
        /* Store the VLAN SNAP (should be 87-65). */
        u16 vlan_snap = *(u16 *) skb->data;
        /* check for headroom availability before skb_push 14 = (4 + 10) */
        if (headroom < 14)
        {
            CSR_LOG_TEXT_DEBUG((
                                   CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG3,  "unifi%d: cant append vlan snap: debug\n",
                                   interfacePriv->privPtr ? ((os_linux_priv_t *) interfacePriv->privPtr)->instance : 0));
            return -1;
        }
        /* Add AA-AA-03-00-00-00 */
        snap = (llc_snap_hdr_t *) skb_push(skb, 4);
        snap->dsap = snap->ssap = 0xAA;
        snap->ctrl = 0x03;
        memcpy(snap->oui, oui_rfc1042, P80211_OUI_LEN);

        /* Add AA-AA-03-00-00-00 */
        snap = (llc_snap_hdr_t *) skb_push(skb, 10);
        snap->dsap = snap->ssap = 0xAA;
        snap->ctrl = 0x03;
        memcpy(snap->oui, oui_rfc1042, P80211_OUI_LEN);

        /* Add the VLAN specific information */
        snap->protocol = htons(proto);
        *(u16 *) (snap + 1) = vlan_snap;
    }
    else
    {
        /* it's DIXII, time for some conversion */
        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG3,  "unifi%d: DIXII len: %d\n",
                               interfacePriv->privPtr ? ((os_linux_priv_t *) interfacePriv->privPtr)->instance : 0,  skb->len));

        /* check for headroom availability before skb_push */
        if (headroom < CSR_WIFI_LLC_SNAP_HDR_LEN)
        {
            CSR_LOG_TEXT_DEBUG((
                                   CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG3,  "unifi%d: cant append snap: debug\n",
                                   interfacePriv->privPtr ? ((os_linux_priv_t *) interfacePriv->privPtr)->instance : 0));
            return -1;
        }
        /* tack on SNAP */
        snap = (llc_snap_hdr_t *) skb_push(skb, CSR_WIFI_LLC_SNAP_HDR_LEN);
        snap->dsap = snap->ssap = 0xAA;
        snap->ctrl = 0x03;
        /* Use the appropriate OUI. */
        if ((proto == ETH_P_AARP) || (proto == ETH_P_IPX))
        {
            memcpy(snap->oui, oui_8021h, P80211_OUI_LEN);
        }
        else
        {
            memcpy(snap->oui, oui_rfc1042, P80211_OUI_LEN);
        }
        snap->protocol = htons(proto);
    }

    return 0;
} /* skb_add_llc_snap() */

#ifdef CSR_SUPPORT_SME
static int _identify_sme_ma_pkt_ind(os_linux_priv_t *priv,
                                    CsrUint16 ifTag,
                                    const CsrInt8 *oui, CsrUint16 protocol,
                                    CsrUint8 *daddr, CsrUint8 *saddr, CsrWifiHipBulkDataParam *bulkdata,
                                    CsrResult receptionResult, CsrInt16 rssi, CsrInt16 snr, CsrUint16 rate)
{
    int r;
    CsrUint8 i;

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG5,
                        "unifi%d: _identify_sme_ma_pkt_ind -->\n",
                        priv ? priv->instance : 0));

    for (i = 0; i < MAX_MA_PACKET_IND_FILTERS; i++)
    {
        if (priv->sme_ma_packet_ind_filters[i].in_use)
        {
            if (!memcmp(oui, priv->sme_ma_packet_ind_filters[i].oui, 3) &&
                (protocol == priv->sme_ma_packet_ind_filters[i].protocol))
            {
                /* Send to client */
                if (priv->sme_cli)
                {
                    /*
                     * Pass the packet to the SME, using unifi_sys_ma_unitdata_ind().
                     * The frame needs to be converted according to the encapsulation.
                     */
                    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                                       "unifi%d: _identify_sme_ma_pkt_ind: handle=%d, encap=%d, proto=%x\n",
                                       priv ? priv->instance : 0,
                                       i, priv->sme_ma_packet_ind_filters[i].encapsulation,
                                       priv->sme_ma_packet_ind_filters[i].protocol));
                    if (priv->sme_ma_packet_ind_filters[i].encapsulation == CSR_WIFI_ROUTER_ENCAPSULATION_ETHERNET)
                    {
                        struct sk_buff *skb;
                        /* The translation is performed on skb... */
                        skb = (struct sk_buff *) bulkdata->d[0].os_net_buf_ptr;
                        skb->len = bulkdata->d[0].data_length;

                        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                                           "unifi%d: _identify_sme_ma_pkt_ind: skb_80211_to_ether -->\n",
                                           priv ? priv->instance : 0));
                        r = skb_80211_to_ether(priv, skb, daddr, saddr, bulkdata);
                        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                                           "unifi%d: _identify_sme_ma_pkt_ind: skb_80211_to_ether <--\n",
                                           priv ? priv->instance : 0));
                        if (r)
                        {
                            return -EINVAL;
                        }

                        /* ... but we indicate buffer and length */
                        bulkdata->d[0].os_data_ptr = skb->data;
                        bulkdata->d[0].data_length = skb->len;
                    }
                    else
                    {
                        /* Add the MAC addresses before the SNAP */
                        bulkdata->d[0].os_data_ptr -= 2 * ETH_ALEN;
                        bulkdata->d[0].data_length += 2 * ETH_ALEN;
                        memcpy((void *) bulkdata->d[0].os_data_ptr, daddr, ETH_ALEN);
                        memcpy((void *) bulkdata->d[0].os_data_ptr + ETH_ALEN, saddr, ETH_ALEN);
                    }

                    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                                       "unifi%d: _identify_sme_ma_pkt_ind: unifi_sys_ma_pkt_ind --> (appHandle 0x%02X)\n",
                                       priv ? priv->instance : 0, priv->sme_ma_packet_ind_filters[i].appHandle));

                    CsrWifiRouterMaPacketIndSend(priv->sme_ma_packet_ind_filters[i].appHandle,
                                                 ifTag,
                                                 i,
                                                 receptionResult,
                                                 bulkdata->d[0].data_length,
                                                 (CsrUint8 *) bulkdata->d[0].os_data_ptr,
                                                 NULL,
                                                 rssi,
                                                 snr,
                                                 rate);

                    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                                       "unifi%d: _identify_sme_ma_pkt_ind: unifi_sys_ma_pkt_ind <--\n",
                                       priv ? priv->instance : 0));
                }

                return 1;
            }
        }
    }

    return -1;
}

#endif /* CSR_SUPPORT_SME */

/*
 * ---------------------------------------------------------------------------
 *  skb_80211_to_ether
 *
 *      Make sure the received frame is in Ethernet (802.3) form.
 *      De-encapsulates SNAP if necessary, adds a ethernet header.
 *      The source buffer should not contain an 802.11 MAC header
 *
 *  Arguments:
 *      payload         Pointer to packet data received from UniFi.
 *      payload_length  Number of bytes of data received from UniFi.
 *      daddr           Destination MAC address.
 *      saddr           Source MAC address.
 *
 *  Returns:
 *      0 on success, -1 if the packet is bad and should be dropped,
 *      1 if the packet was forwarded to the SME or AMP client.
 * ---------------------------------------------------------------------------
 */
int skb_80211_to_ether(os_linux_priv_t *priv, struct sk_buff *skb,
                       const unsigned char *daddr, const unsigned char *saddr,
                       CsrWifiHipBulkDataParam *bulkdata)
{
    unsigned char *payload;
    int payload_length;
    struct ethhdr *eth;
    llc_snap_hdr_t *snap;
    int headroom;
#define UF_VLAN_LLC_HEADER_SIZE     18
    static const u8 vlan_inner_snap[] = {0xAA, 0xAA, 0x03, 0x00, 0x00, 0x00};

    if ((skb == NULL) || (daddr == NULL) || (saddr == NULL))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: skb_80211_to_ether: PBC fail\n", priv ? priv->instance : 0));
        return 1;
    }

    payload = skb->data;
    payload_length = skb->len;

    snap = (llc_snap_hdr_t *) payload;
    eth = (struct ethhdr *) payload;

    /* get the skb headroom size */
    headroom = skb_headroom(skb);

    /*
     * Test for the various encodings
     */
    if ((payload_length >= CSR_WIFI_LLC_SNAP_HDR_LEN) &&
        (snap->dsap == 0xAA) &&
        (snap->ssap == 0xAA) &&
        (snap->ctrl == 0x03) &&
        (snap->oui[0] == 0) &&
        (snap->oui[1] == 0) &&
        ((snap->oui[2] == 0) || (snap->oui[2] == 0xF8)))
    {
        u16 tmp_prot = 0;

        /* AppleTalk AARP (2) or IPX SNAP */
        if ((snap->oui[2] == 0) &&
            ((ntohs(snap->protocol) == ETH_P_AARP) || (ntohs(snap->protocol) == ETH_P_IPX)))
        {
            u16 len;

            CSR_LOG_TEXT_DEBUG((
                                   CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG3,  "unifi%d: %s len: %d\n",
                                   priv ? priv->instance : 0,
                                   (ntohs(snap->protocol) == ETH_P_AARP) ? "ETH_P_AARP" : "ETH_P_IPX",
                                   payload_length));

            /* check for headroom availability before skb_push */
            if (headroom < (2 * ETH_ALEN + 2))
            {
                CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: headroom not available to skb_push ether header\n", priv ? priv->instance : 0));
                return -1;
            }

            /* Add 802.3 header and leave full payload */
            len = htons(skb->len);
            memcpy(skb_push(skb, 2), &len, 2);
            memcpy(skb_push(skb, ETH_ALEN), saddr, ETH_ALEN);
            memcpy(skb_push(skb, ETH_ALEN), daddr, ETH_ALEN);

            return 0;
        }
        /* VLAN-tagged IP */
        if ((snap->oui[2] == 0) && (ntohs(snap->protocol) == ETH_P_8021Q))
        {
            /*
             * The translation doesn't change the packet length, so is done in-place.
             *
             * Example header (from Std 802.11-2007 Annex M):
             * AA-AA-03-00-00-00-81-00-87-65-AA-AA-03-00-00-00-08-06
             * -------SNAP-------p1-p1-ll-ll-------SNAP--------p2-p2
             * dd-dd-dd-dd-dd-dd-aa-aa-aa-aa-aa-aa-p1-p1-ll-ll-p2-p2
             * dd-dd-dd-dd-dd-dd-aa-aa-aa-aa-aa-aa-81-00-87-65-08-06
             */
            u16 vlan_snap;

            if (payload_length < UF_VLAN_LLC_HEADER_SIZE)
            {
                CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID,
                                      CSR_WIFI_HIP_LOG_DEF, "unifi%d: VLAN SNAP header too short: %d bytes\n",
                                      priv ? priv->instance : 0,  payload_length));
                return -1;
            }

            if (memcmp(payload + 10, vlan_inner_snap, 6))
            {
                CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: VLAN malformatted SNAP header.\n", priv ? priv->instance : 0));
                return -1;
            }

            CSR_LOG_TEXT_DEBUG((
                                   CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG3,  "unifi%d: VLAN SNAP: %02x-%02x\n",
                                   priv ? priv->instance : 0,  payload[8], payload[9]));
            CSR_LOG_TEXT_DEBUG((
                                   CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG3,  "unifi%d: VLAN len: %d\n",
                                   priv ? priv->instance : 0,  payload_length));

            /* Create the 802.3 header */

            vlan_snap = *((u16 *) (payload + 8));

            /* Create LLC header without byte-swapping */
            eth->h_proto = snap->protocol;

            memcpy(eth->h_dest, daddr, ETH_ALEN);
            memcpy(eth->h_source, saddr, ETH_ALEN);
            *(u16 *) (eth + 1) = vlan_snap;
            return 0;
        }

        /* it's a SNAP + RFC1042 frame */
        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG3,  "unifi%d: SNAP+RFC1042 len: %d\n",
                               priv ? priv->instance : 0,  payload_length));

        tmp_prot = snap->protocol;

        /* chop SNAP+llc header from skb. */
        skb_pull(skb, CSR_WIFI_LLC_SNAP_HDR_LEN);

        /* Since skb_pull called above to chop snap+llc, no need to check for headroom
         * availability before skb_push
         */
        /* create 802.3 header at beginning of skb. */
        eth = (struct ethhdr *) skb_push(skb, ETH_HLEN);
        memcpy(eth->h_dest, daddr, ETH_ALEN);
        memcpy(eth->h_source, saddr, ETH_ALEN);

        /* Copy protocol field without byte-swapping */
        eth->h_proto = tmp_prot;
    }
    else
    {
        u16 len;

        /* check for headroom availability before skb_push */
        if (headroom < (2 * ETH_ALEN + 2))
        {
            CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: headroom not available to skb_push ether header\n", priv ? priv->instance : 0));
            return -1;
        }
        /* Add 802.3 header and leave full payload */
        len = htons(skb->len);
        memcpy(skb_push(skb, 2), &len, 2);
        memcpy(skb_push(skb, ETH_ALEN), saddr, ETH_ALEN);
        memcpy(skb_push(skb, ETH_ALEN), daddr, ETH_ALEN);

        return 1;
    }

    return 0;
} /* skb_80211_to_ether() */

#ifndef CSR_NATIVE_SOFTMAC
/*
 * ---------------------------------------------------------------------------
 *  send_ma_pkt_request
 *
 *      These functions send a data packet to UniFi for transmission.
 *      EAP protocol packets are also sent as send_ma_pkt_request().
 *
 *  Arguments:
 *      dev             Pointer to struct net_device to send on
 *      skb             Socket buffer containing data packet to transmit
 *      ehdr            Pointer to Ethernet header within skb.
 *
 *  Returns:
 *      Zero on success or error code.
 * ---------------------------------------------------------------------------
 */
CsrResult send_ma_pkt_request(os_linux_priv_t *os_linux, netInterface_priv_t *interfacePriv, struct sk_buff *skb, const struct ethhdr *ehdr)
{
    CsrResult result;
    CsrBool eapolStore = FALSE;
    CsrWifiHipBulkDataParam bulkdata;
    const int proto = ntohs(ehdr->h_proto);
    CsrUint16 interfaceTag;
    CsrUint8 headerOffset = 0;
    CsrWifiHipMaPacketPriority prio;
    CsrWifiHipQosType qosType;
    CsrUint32 hostTag = CSR_WIFI_HIP_HOST_TAG_INVALID;
    int headroom = 0;
    CsrUint16 transmitRate = 0;

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5, "unifi%d: entering send_ma_pkt_request\n"
                        , os_linux ? os_linux->instance : 0));

    if (interfacePriv == NULL)
    {
        /* No match found - error */
        CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Mac address not matching ... debugging needed\n", os_linux ? os_linux->instance : 0));
        kfree_skb(skb);
        return -1;
    }

    interfaceTag = interfacePriv->InterfaceTag;

    /* Add a SNAP header if necessary */
    if (skb_add_llc_snap(os_linux->netdev[interfaceTag], skb, proto) != 0)
    {
        /* convert failed */
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: skb_add_llc_snap failed.\n", os_linux ? os_linux->instance : 0));
        kfree_skb(skb);
        return -1;
    }

    if ((proto == ETH_P_PAE)
#ifdef CSR_WIFI_SECURITY_WAPI_ENABLE
        || (proto == ETH_P_WAI)
#endif
        )
    {
        /* check for m4 detection */
        if (0 == uf_verify_m4(os_linux, skb->data, skb->len))
        {
            eapolStore = TRUE;
            hostTag = CSR_WIFI_HIP_M4_MESSAGE;
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                                "unifi%d: EAPOL M4 detected\n",
                                os_linux ? os_linux->instance : 0));
        }
    }

    prio = skb->priority;
    qosType = CSR_WIFI_HIP_QOS_TYPE(prio);

    CsrWifiHipFrameBufferSizeReq(os_linux->hip_handle,
                                 interfaceTag,
                                 qosType,
                                 &headerOffset);

    headroom = skb_headroom(skb);
    if (headroom < headerOffset)
    {
        CsrResult ret;

        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3, "unifi%d: send_ma_pkt_request: Not sufficient headroom (%u bytes) to add MAC header (%u bytes). Realloc needed\n",
                            os_linux ? os_linux->instance : 0, headroom, headerOffset));

        ret = os_linux_net_data_malloc(os_linux, &bulkdata.d[0], skb->len + headerOffset);
        if (ret != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: send_ma_pkt_request: Failed to allocate memory \n", os_linux ? os_linux->instance : 0));
            kfree_skb(skb);
            return CSR_RESULT_FAILURE;
        }

        bulkdata.d[0].net_buf_length = bulkdata.d[0].data_length = (skb->len + headerOffset);
        memcpy(bulkdata.d[0].os_data_ptr + headerOffset, skb->data, skb->len);
        kfree_skb(skb);
    }
    else
    {
        skb_push(skb, headerOffset);
        bulkdata.d[0].os_data_ptr = skb->data;
        bulkdata.d[0].os_net_buf_ptr = (unsigned char *) skb;
        bulkdata.d[0].net_buf_length = bulkdata.d[0].data_length = skb->len;
    }

    bulkdata.d[1].os_data_ptr = NULL;
    bulkdata.d[1].os_net_buf_ptr = NULL;
    bulkdata.d[1].net_buf_length = bulkdata.d[1].data_length = 0;


    if ((proto == ETH_P_PAE)
#ifdef CSR_WIFI_SECURITY_WAPI_ENABLE
        || (proto == ETH_P_WAI)
#endif
        )
    {
#ifdef CSR_SUPPORT_SME
        if (eapolStore)
        {
            /* Store the M4-PACKET.req for later */
            spin_lock(&os_linux->m4_lock);
            interfacePriv->m4_hostTag = hostTag;

            /* RA adrress must contain the immediate destination MAC address that is similiar to
             * the Address 1 field of 802.11 Mac header here 4 is: (sizeof(framecontrol) + sizeof (durationID))
             * which is address 1 field.
             * The packet header is not available at this point, as it gets populated in
             * CsrWifiHipMaPacketReq(). We use the ehdr->h_dest instead. This is correct for all M4 packets.
             */
            memcpy(interfacePriv->m4_peer_mac_addr.a, ehdr->h_dest, ETH_ALEN);
            interfacePriv->m4_bulk_data.d[0].net_buf_length = bulkdata.d[0].net_buf_length;
            interfacePriv->m4_bulk_data.d[0].data_length = bulkdata.d[0].data_length;
            interfacePriv->m4_bulk_data.d[0].os_data_ptr = bulkdata.d[0].os_data_ptr;
            interfacePriv->m4_bulk_data.d[0].os_net_buf_ptr = bulkdata.d[0].os_net_buf_ptr;

            interfacePriv->m4_info.interfaceTag = interfaceTag;
            memcpy(interfacePriv->m4_info.dmac.a, ehdr->h_dest, ETH_ALEN);
            memcpy(interfacePriv->m4_info.smac.a, ehdr->h_source, ETH_ALEN);
            interfacePriv->m4_info.senderProcessId = os_linux->netdev_client->sender_id;
            interfacePriv->m4_info.priority = prio;
            interfacePriv->m4_info.protocol = proto;

            /* Always request a MaPacketConfirm for M4 messages since this is required for
             * the M4 capture mechanism */
            interfacePriv->m4_info.cfmRequested = TRUE;
            interfacePriv->m4_info.macHeaderOffset = headerOffset;

            spin_unlock(&os_linux->m4_lock);

            /* Signal the workqueue to call CsrWifiRouterCtrlM4ReadyToSendIndSend().
             * It cannot be called directly from the tx path because it
             * does a non-atomic kmalloc via the framework's CsrPmemAlloc().
             */
            queue_work(os_linux->unifi_workqueue, &interfacePriv->send_m4_ready_task);

            return CSR_RESULT_SUCCESS;
        }
#endif
    }
#ifdef CSR_SUPPORT_WEXT
    /*
     * Fixed transmit rate can be set by the WEXT commands
     * fixed transmit rate is supported for only STA mode
     */
    transmitRate = interfacePriv->fixed_tx_rate.txRate;
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6,
                        "unifi%d: send_ma_pkt_request: transmitRate %d\n",
                        os_linux->instance, transmitRate));
#endif

    result = CsrWifiHipMaPacketReq(os_linux->hip_handle,
                                   (CsrUint8 *) ehdr->h_dest,
                                   (CsrUint8 *) ehdr->h_source,
                                   os_linux->netdev_client->sender_id,
                                   interfaceTag,
                                   CSR_WIFI_HIP_FRAMETYPE_IEEE80211_DATA,
                                   proto,
                                   &bulkdata,
                                   prio,
                                   transmitRate,
                                   hostTag,
                                   FALSE,
                                   headerOffset,
                                   CsrWifiHipVifIndexGetReq(os_linux->hip_handle, interfaceTag),
                                   0);

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3,
                        "unifi%d: leaving send_ma_pkt_request, UNITDATA result code = %d\n",
                        os_linux ? os_linux->instance : 0, result));

    return result;
} /* send_ma_pkt_request() */

#endif

/*
 * ---------------------------------------------------------------------------
 *  uf_net_xmit
 *
 *      This function is called by the higher level stack to transmit an
 *      ethernet packet.
 *
 *  Arguments:
 *      skb     Ethernet packet to send.
 *      dev     Pointer to the linux net device.
 *
 *  Returns:
 *      0   on success (packet was consumed, not necessarily transmitted)
 *      1   if packet was requeued
 *     -1   on error
 *
 *
 *  Notes:
 *      The controlled port is handled in the qdisc dequeue handler.
 * ---------------------------------------------------------------------------
 */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 32)
static netdev_tx_t
#else
static int
#endif
uf_net_xmit(struct sk_buff *skb, struct net_device *dev)
{
    netInterface_priv_t *interfacePriv = (netInterface_priv_t *) netdev_priv(dev);
    os_linux_priv_t *priv = interfacePriv->privPtr;
    struct ethhdr ehdr;
    CsrResult result;
    unsigned int packetLength = 0;

    func_enter();

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5, "unifi%d: unifi_net_xmit: skb = %x\n",
                        priv ? priv->instance : 0,  skb));

    memcpy(&ehdr, skb->data, ETH_HLEN);


    /* Keep the packet length. The packet length will be used to increment
       stats for the netdev if the packet was successfully transmitted.
       The ownership of the SKB is parsed to HAL, so we should
       not refer the SKB after this point */
    packetLength = skb->len;

#ifdef CSR_NATIVE_SOFTMAC
#else
    /* Remove the ethernet header */
    skb_pull(skb, ETH_HLEN);
    result = send_ma_pkt_request(priv, interfacePriv, skb, &ehdr);
#endif

    if (result == CSR_RESULT_SUCCESS)
    {
        dev->trans_start = jiffies;
        interfacePriv->stats.tx_packets++;
        interfacePriv->stats.tx_bytes += packetLength;
    }
    else
    {
        /* Failed to send: fh queue was full, and the skb was discarded.
         * Return OK to indicate that the buffer was consumed, to stop the
         * kernel re-transmitting the freed buffer.
         */
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi%d: unifi_net_xmit: (Packet Drop)\n",
                           priv ? priv->instance : 0));
        interfacePriv->stats.tx_dropped++;
        result = NETDEV_TX_OK;
    }

    func_exit();
    return result;
} /* uf_net_xmit() */

/*
 * ---------------------------------------------------------------------------
 *  os_linux_flow_control_pause_cb
 *  os_linux_flow_control_resume_cb
 *
 *      These functions are called from the UniFi driver to control the flow
 *      of packets from the upper layers.
 *      os_linux_flow_control_pause_cb is called when the internal queue is
 *      full.
 *      When the queue has drained, os_linux_flow_control_resume_cb will be
 *      called to re-enable the flow of packets for transmission.
 *
 *  Arguments:
 *      osLayerContext    OS linux private context pointer.
 *      interfaceTag      The virtual interface tag
 *      aid               Association ID - used in access point mode
 *      queue             The traffic queue, which is affected by the action.
 *
 * ---------------------------------------------------------------------------
 */
void os_linux_flow_control_resume_cb(void *osLayerContext, CsrUint16 interfaceTag, CsrUint16 aid, CsrWifiHipTrafficQueue queue)
{
    os_linux_priv_t *priv = (os_linux_priv_t *) osLayerContext;
    CsrUint16 net_if_q;

    net_if_q = UNIFI_NET_IF_Q(aid, queue);

    func_enter();
    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6,
                       "unifi%d: os_linux_flow_control_resume_cb: Waking queue %d\n",
                       priv->instance, queue));

    if (netif_running(priv->netdev[interfaceTag]))
    {
        netif_wake_subqueue(priv->netdev[interfaceTag], net_if_q);
    }

    func_exit();
}

void os_linux_flow_control_pause_cb(void *osLayerContext, CsrUint16 interfaceTag, CsrUint16 aid, CsrWifiHipTrafficQueue queue)
{
    os_linux_priv_t *priv = (os_linux_priv_t *) osLayerContext;
    CsrUint16 net_if_q;

    net_if_q = UNIFI_NET_IF_Q(aid, queue);

    func_enter();
    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6,
                       "unifi%d: os_linux_flow_control_pause_cb: Pausing queue %d\n",
                       priv->instance, queue));

    if (netif_running(priv->netdev[interfaceTag]))
    {
        netif_stop_subqueue(priv->netdev[interfaceTag], net_if_q);
    }

    func_exit();
}

void CsrWifiHipFlowControlPauseAllInd(void *osLayerContext)
{
    os_linux_priv_t *priv = (os_linux_priv_t *) osLayerContext;
    int i = 0; /* used as a loop counter */

    func_enter();

    for (i = 0; i < CSR_WIFI_MAX_INTERFACES; i++)
    {
        if (netif_running(priv->netdev[i]))
        {
            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiHipFlowControlPauseAllInd: Interface %s Stopping all queues\n",
                               priv->instance, priv->netdev[i]->name));
            UF_NETIF_TX_STOP_ALL_QUEUES(priv->netdev[i]);
        }
    }
    func_exit();
}

void indicate_rx_skb(os_linux_priv_t *priv, CsrUint16 ifTag, CsrUint8 *dst_a, CsrUint8 *src_a, struct sk_buff *skb,
                     CsrWifiHipBulkDataParam *bulkdata, CsrResult receptionResult, CsrInt16 rssi, CsrInt16 snr, CsrUint16 rate)
{
    int r, sr = 0;
    struct net_device *dev;
    CsrUint8 tmp_dmac[ETH_ALEN], tmp_smac[ETH_ALEN];

    memcpy(tmp_dmac, dst_a, ETH_ALEN);
    memcpy(tmp_smac, src_a, ETH_ALEN);

#ifdef CSR_SUPPORT_SME
    /* Only check for MaPacket subscriptions if a subscription has been registered. */
    if (priv->num_of_ma_packet_subscriptions > 0)
    {
        llc_snap_hdr_t *snap;

        snap = (llc_snap_hdr_t *) skb->data;
        sr = _identify_sme_ma_pkt_ind(priv,
                                      ifTag,
                                      snap->oui, ntohs(snap->protocol),
                                      tmp_dmac, tmp_smac,
                                      bulkdata,
                                      receptionResult, rssi, snr, rate);
    }
#endif

    /*
     * Decapsulate any SNAP header and
     * prepend an ethernet header so that the skb manipulation and ARP
     * stuff works.
     */
    r = skb_80211_to_ether(priv, skb, tmp_dmac, tmp_smac, bulkdata);
    if (r == -1)
    {
        /* Drop the packet and return */
        priv->interfacePriv[ifTag]->rx_stats_dropped_frames++;
        os_linux_net_data_free(priv, &bulkdata->d[0]);
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: indicate_rx_skb: Discard unknown frame.\n", priv ? priv->instance : 0));
        func_exit();
        return;
    }

    /* Handle the case where packet is sent up through the subscription
     * API but should not be given to the network stack (AMP PAL case)
     * LLC header is different from WiFi and the packet has been subscribed for
     */
    if ((r == 1) && (sr == 1))
    {
        os_linux_net_data_free(priv, &bulkdata->d[0]);
        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG5,  "unifi%d: indicate_rx_skb: Data given to subscription" "API, not being given to kernel\n"
                               , priv ? priv->instance : 0));
        func_exit();
        return;
    }

    /* Now we look like a regular ethernet frame */
    /* Fill in SKB meta data */
    dev = priv->netdev[ifTag];
    skb->dev = dev;
    skb->protocol = eth_type_trans(skb, dev);
    skb->ip_summed = CHECKSUM_UNNECESSARY;

    /* Test for an overlength frame */
    if (skb->len > (dev->mtu + ETH_HLEN))
    {
        /* A bogus length ethfrm has been encap'd. */
        /* Is someone trying an oflow attack? */
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: %s: oversize frame (%d > %d)\n",
                               priv ? priv->instance : 0,
                               dev->name,
                               skb->len, dev->mtu + ETH_HLEN));

        /* Drop the packet and return */
        priv->interfacePriv[ifTag]->rx_stats_dropped_frames++;
        priv->interfacePriv[ifTag]->stats.rx_length_errors++;
        os_linux_net_data_free(priv, &bulkdata->d[0]);
        func_exit();
        return;
    }


    /* Pass SKB up the stack */
#ifdef CSR_WIFI_DRIVER_USE_NETIF_RX
    netif_rx(skb);
#else
    netif_rx_ni(skb);
#endif

    if (dev != NULL)
    {
        dev->last_rx = jiffies;
    }

    /* Bump rx stats */
    priv->interfacePriv[ifTag]->stats.rx_packets++;
    priv->interfacePriv[ifTag]->stats.rx_bytes += bulkdata->d[0].data_length;

    func_exit();
}

/*
 * ---------------------------------------------------------------------------
 *  uf_set_multicast_list
 *
 *      This function is called by the higher level stack to set
 *      a list of multicast rx addresses.
 *
 *  Arguments:
 *      dev             Network Device pointer.
 *
 *  Returns:
 *      None.
 *
 *  Notes:
 * ---------------------------------------------------------------------------
 */
#if LINUX_VERSION_CODE < KERNEL_VERSION(3, 2, 0)
static void uf_set_multicast_list(struct net_device *dev)
{
    netInterface_priv_t *interfacePriv = (netInterface_priv_t *) netdev_priv(dev);
    os_linux_priv_t *priv = interfacePriv->privPtr;

#ifdef CSR_NATIVE_LINUX
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3,
                        "unifi%d: uf_set_multicast_list unsupported\n",
                        priv ? priv->instance : 0));
    return;
#else
    u8 *mc_list = interfacePriv->mc_list;
#if LINUX_VERSION_CODE > KERNEL_VERSION(2, 6, 34)
    struct netdev_hw_addr *mc_addr;
    int mc_addr_count;
#else
    struct dev_mc_list *p;      /* Pointer to the addresses structure. */
    int i;
#endif

    if (priv->init_progress != UNIFI_INIT_COMPLETED)
    {
        return;
    }

#if LINUX_VERSION_CODE > KERNEL_VERSION(2, 6, 34)
    mc_addr_count = netdev_mc_count(dev);

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG3,
                           "unifi%d: uf_set_multicast_list (count=%d)\n",
                           priv ? priv->instance : 0,  mc_addr_count));


    /* Not enough space? */
    if (mc_addr_count > UNIFI_MAX_MULTICAST_ADDRESSES)
    {
        return;
    }

    /* Store the list to be processed by the work item. */
    interfacePriv->mc_list_count = mc_addr_count;
    netdev_hw_addr_list_for_each(mc_addr, &dev->mc)
    {
        memcpy(mc_list, mc_addr->addr, ETH_ALEN);
        mc_list += ETH_ALEN;
    }

#else
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3,
                        "uf_set_multicast_list (count=%d)\n", dev->mc_count));

    /* Not enough space? */
    if (dev->mc_count > UNIFI_MAX_MULTICAST_ADDRESSES)
    {
        return;
    }

    /* Store the list to be processed by the work item. */
    interfacePriv->mc_list_count = dev->mc_count;
    p = dev->mc_list;
    for (i = 0; i < dev->mc_count; i++)
    {
        CSR_LOG_TEXT_BUFFER_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG4, ETH_ALEN, p->dmi_addr, "uf_set_multicast_list: addr:"));
        memcpy(mc_list, p->dmi_addr, ETH_ALEN);
        p = p->next;
        mc_list += ETH_ALEN;
    }
#endif

    /* Send a message to the workqueue */
    queue_work(priv->unifi_workqueue, &priv->multicast_list_task);
#endif
} /* uf_set_multicast_list() */

#endif


/*
 * ---------------------------------------------------------------------------
 *  uf_net_get_name
 *
 *      Retrieve the name (e.g. eth1) associated with this network device
 *
 *  Arguments:
 *      dev             Pointer to the network device.
 *      name            Buffer to write name
 *      len             Size of buffer in bytes
 *
 *  Returns:
 *      None
 *
 *  Notes:
 * ---------------------------------------------------------------------------
 */
void uf_net_get_name(struct net_device *dev, char *name, int len)
{
    *name = '\0';
    if (dev)
    {
        strlcpy(name, dev->name, (len > IFNAMSIZ) ? IFNAMSIZ : len);
    }
} /* uf_net_get_name */

/*
 * ---------------------------------------------------------------------------
 *  uf_resume_data_plane
 *
 *      Is called when the (un)controlled port is set to open,
 *      to notify the network stack to schedule for transmission
 *      any packets queued while port was closed is indicated to
 *      the stack.
 *
 *  Arguments:
 *      priv        Pointer to device private struct
 *      interfaceTag The interface tag
 *      portType    Indicates if the controlled or uncontrolled ports/queues
 *                  shall be resumed.
 *      peer_address The peer_address will be used while processing the
 *                  queue.
 *
 *  Returns:
 * ---------------------------------------------------------------------------
 */
void uf_resume_data_plane(os_linux_priv_t *priv,  CsrUint16 interfaceTag, CsrWifiHipPortType portType,
                          CsrWifiMacAddress *peer_address)
{
#ifdef CSR_SUPPORT_WEXT
    netInterface_priv_t *interfacePriv = NULL;
#endif

    if (interfaceTag >= CSR_WIFI_MAX_INTERFACES)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: uf_resume_data_plane: Invalid interfaceTag %u\n",
                               priv ? priv->instance : 0, interfaceTag));
        return;
    }

#ifdef CSR_SUPPORT_WEXT
    interfacePriv = priv->interfacePriv[interfaceTag];
#endif

    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                       "unifi : Resuming netif %s\n", priv->netdev[interfaceTag]->name));

    /*
     * If we are waiting for the net device to enter the up state, don't
     * process the rx queue yet as it will be done by the callback when
     * the device is ready.
     */
#ifdef CSR_SUPPORT_WEXT
    if (!interfacePriv->wait_netdev_change)
#endif
    {
        CsrResult result;
#ifdef CONFIG_NET_SCHED
        if (netif_running(priv->netdev[interfaceTag]))
        {
            netif_tx_schedule_all(priv->netdev[interfaceTag]);
        }
#endif

        result = CsrWifiHipCtrlPortQueueProcessReq(priv->hip_handle, interfaceTag, portType, peer_address, TRUE);
        if (result != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi%d: uf_resume_data_plane: Failed to process port type %u's queue\n",
                                   priv ? priv->instance : 0, portType));
        }
    }
} /* uf_resume_data_plane() */

#ifdef CSR_SUPPORT_WEXT

/*
 * ---------------------------------------------------------------------------
 *  uf_netdev_event
 *
 *     Callback function to handle netdev state changes
 *
 *  Arguments:
 *      notif           Pointer to a notifier_block.
 *      event           Event prompting notification
 *      ptr             net_device pointer
 *
 *  Returns:
 *      Negative value in case of error.
 *
 *  Notes:
 *   The event handler is global, and may occur on non-UniFi netdevs.
 * ---------------------------------------------------------------------------
 */
static int uf_netdev_event(struct notifier_block *notif, unsigned long event, void *ptr)
{
    struct net_device *netdev = ptr;
    netInterface_priv_t *interfacePriv = (netInterface_priv_t *) netdev_priv(netdev);
    os_linux_priv_t *priv = NULL;

    /* Check that the event is for a UniFi netdev. If it's not, the netdev_priv
     * structure is not safe to use.
     */
    if (uf_find_netdev_priv(interfacePriv))
    {
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi : uf_netdev_event: ignore e=%d, ptr=%p, priv=%p %s\n",
                           event, ptr, interfacePriv, netdev->name));
        return 0;
    }

    switch (event)
    {
        case NETDEV_CHANGE:
            priv = interfacePriv->privPtr;
            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi%d: NETDEV_CHANGE: %p %s %s waiting for it\n",
                               priv ? priv->instance : 0, ptr, netdev->name,
                               interfacePriv->wait_netdev_change ? "" : "not"));

            if (interfacePriv->wait_netdev_change)
            {
                CsrResult result;
                CsrWifiMacAddress broadcast_address = {{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF}};
                UF_NETIF_TX_WAKE_ALL_QUEUES(netdev);
                interfacePriv->connected = UnifiConnected;
                interfacePriv->wait_netdev_change = FALSE;

                /* Note: passing the broadcast address here will allow anyone to attempt to join our adhoc network */
                result = CsrWifiHipCtrlPortQueueProcessReq(priv->hip_handle, interfacePriv->InterfaceTag, CSR_WIFI_HIP_PORT_TYPE_UNCONTROLLED, &broadcast_address, TRUE);
                if (result != CSR_RESULT_SUCCESS)
                {
                    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                           "unifi%d: uf_netdev_event: Failed to process port type %u's queue\n",
                                           priv ? priv->instance : 0, CSR_WIFI_HIP_PORT_TYPE_UNCONTROLLED));
                }

                result = CsrWifiHipCtrlPortQueueProcessReq(priv->hip_handle, interfacePriv->InterfaceTag, CSR_WIFI_HIP_PORT_TYPE_CONTROLLED, &broadcast_address, TRUE);
                if (result != CSR_RESULT_SUCCESS)
                {
                    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                           "unifi%d: uf_netdev_event: Failed to process port type %u's queue\n",
                                           priv ? priv->instance : 0, CSR_WIFI_HIP_PORT_TYPE_CONTROLLED));
                }
            }
            break;

        default:
            break;
    }

    return 0;
}

static struct notifier_block uf_netdev_notifier =
{
    .notifier_call = uf_netdev_event,
};
#endif /* CSR_SUPPORT_WEXT */
