/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"

#include "os_linux_priv.h"
#include "csr_wifi_hip_unifi_udi.h"   /* for unifi_print_status() */
#include "os_linux_priv.h"
#include "csr_wifi_hip_log_text.h"

#include "csr_wifi_ps_if.h"
#include <linux/proc_fs.h>
#include <linux/wait.h>
#include <linux/semaphore.h>
#include <linux/netdevice.h>

#define PACKET_SCHEDULER_STATS 1
/*
 * Array of pointers to context structs for unifi devices that are present.
 * The index in the array corresponds to the wlan interface number
 * (if "wlan*" is used). If "eth*" is used, the eth* numbers are allocated
 * after any Ethernet cards.
 *
 * The Arasan PCI-SDIO controller card supported by this driver has 2 slots,
 * hence a max of 2 devices.
 */
static os_linux_priv_t *Unifi_instances[MAX_UNIFI_DEVS];

/* Array of pointers to netdev objects used by the UniFi driver, as there
 * are now many per instance. This is used to determine which netdev events
 * are for UniFi as opposed to other net interfaces.
 */
static netInterface_priv_t *Unifi_netdev_instances[MAX_UNIFI_DEVS * CSR_WIFI_MAX_INTERFACES];

/*
 * Array to hold the status of each unifi device in each slot.
 * We only process an insert event when In_use[] for the slot is
 * UNIFI_DEV_NOT_IN_USE. Otherwise, it means that the slot is in use or
 * we are in the middle of a cleanup (the action on unplug).
 */
#define UNIFI_DEV_NOT_IN_USE    0
#define UNIFI_DEV_IN_USE        1
#define UNIFI_DEV_CLEANUP       2
static int In_use[MAX_UNIFI_DEVS];

static int active_slot = MAX_UNIFI_DEVS;
static struct device *os_devices[MAX_UNIFI_DEVS];

/*
 * Mutex to prevent UDI clients to open the character device before the priv
 * is created and initialised.
 */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 37)
DEFINE_SEMAPHORE(os_linux_inst_mutex);
#else
DECLARE_MUTEX(os_linux_inst_mutex);
#endif

/*
 * When the device is removed, unregister waits on Unifi_cleanup_wq
 * until all the UDI clients release the character device.
 */
DECLARE_WAIT_QUEUE_HEAD(Unifi_cleanup_wq);


static void ask_unifi_sdio_cleanup(os_linux_priv_t *os_linux);

/*
 * ---------------------------------------------------------------------------
 *  uf_find_instance
 *
 *      Find the context structure for a given UniFi device instance.
 *
 *  Arguments:
 *      inst            The instance number to look for.
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
os_linux_priv_t *uf_find_instance(int inst)
{
    if ((inst < 0) || (inst >= MAX_UNIFI_DEVS))
    {
        return NULL;
    }

    return Unifi_instances[inst];
} /* uf_find_instance() */

/*
 * ---------------------------------------------------------------------------
 *  uf_find_priv
 *
 *      Find the device instance for a given context structure.
 *
 *  Arguments:
 *      priv            The context structure pointer to look for.
 *
 *  Returns:
 *      index of instance, -1 otherwise.
 * ---------------------------------------------------------------------------
 */
int uf_find_priv(os_linux_priv_t *priv)
{
    int inst;

    if (!priv)
    {
        return -1;
    }

    for (inst = 0; inst < MAX_UNIFI_DEVS; inst++)
    {
        if (Unifi_instances[inst] == priv)
        {
            return inst;
        }
    }

    return -1;
} /* uf_find_priv() */

/*
 * ---------------------------------------------------------------------------
 *  uf_find_netdev_priv
 *
 *      Find the device instance for a given netdev context structure.
 *
 *  Arguments:
 *      priv            The context structure pointer to look for.
 *
 *  Returns:
 *      index of instance, -1 otherwise.
 * ---------------------------------------------------------------------------
 */
int uf_find_netdev_priv(netInterface_priv_t *priv)
{
    int inst;

    if (!priv)
    {
        return -1;
    }

    for (inst = 0; inst < MAX_UNIFI_DEVS * CSR_WIFI_MAX_INTERFACES; inst++)
    {
        if (Unifi_netdev_instances[inst] == priv)
        {
            return inst;
        }
    }

    return -1;
} /* uf_find_netdev_priv() */

/*
 * ---------------------------------------------------------------------------
 *  uf_get_instance
 *
 *      Find the context structure for a given UniFi device instance
 *      and increment the reference count.
 *
 *  Arguments:
 *      inst            The instance number to look for.
 *
 *  Returns:
 *      Pointer to the instance or NULL if no instance exists.
 * ---------------------------------------------------------------------------
 */
os_linux_priv_t *uf_get_instance(int inst)
{
    os_linux_priv_t *os_linux;

    down(&os_linux_inst_mutex);

    os_linux = uf_find_instance(inst);
    if (os_linux)
    {
        os_linux->ref_count++;
    }

    up(&os_linux_inst_mutex);

    return os_linux;
}

/*
 * ---------------------------------------------------------------------------
 *  uf_put_instance
 *
 *      Decrement the context reference count, freeing resources and
 *      shutting down the driver when the count reaches zero.
 *
 *  Arguments:
 *      inst            The instance number to look for.
 *
 *  Returns:
 *      Pointer to the instance or NULL if no instance exists.
 * ---------------------------------------------------------------------------
 */
void uf_put_instance(int inst)
{
    os_linux_priv_t *os_linux;

    down(&os_linux_inst_mutex);
    os_linux = uf_find_instance(inst);
    if (os_linux)
    {
        os_linux->ref_count--;
        if (os_linux->ref_count == 0)
        {
            ask_unifi_sdio_cleanup(os_linux);
        }
    }
    up(&os_linux_inst_mutex);
}

/*
 * ---------------------------------------------------------------------------
 *  uf_release_instance
 *
 *      Release an os_linux instance.
 *
 *  Arguments:
 *      Instance number to release.
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
void uf_release_instance(int inst)
{
    if ((inst < 0) || (inst >= MAX_UNIFI_DEVS))
    {
        return;
    }

    down(&os_linux_inst_mutex);
    Unifi_instances[inst] = NULL;
    /*
     * Now clear the flag that says the old instance is in use.
     * This is used to prevent a new instance being started before old
     * one has finshed closing down, for example if bounce makes the card
     * appear to be ejected and re-inserted quickly.
     */
    In_use[inst] = UNIFI_DEV_NOT_IN_USE;
    up(&os_linux_inst_mutex);
}

/*
 * ---------------------------------------------------------------------------
 *  uf_add_instance
 *
 *      Verify that the entry corresponding to requestedInstanceNum is
 *      available in the instance table.
 *      If the instance entry is already in use - return error, otherwise
 *      mark the entry as in use.
 *      The active_slot variable is a static variable, which is updated by
 *      platform specific linux code.
 *
 *  Arguments:
 *      Instance    Pointer, which will be updated with the instance number.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS if successful otherwise CSR_RESULT_FAILURE.
 * ---------------------------------------------------------------------------
 */
CsrResult uf_add_instance(CsrUint32 *instance)
{
    down(&os_linux_inst_mutex);

    if ((active_slot >= 0) && (active_slot < MAX_UNIFI_DEVS))
    {
        if (In_use[active_slot] == UNIFI_DEV_NOT_IN_USE)
        {
            In_use[active_slot] = UNIFI_DEV_IN_USE;
            Unifi_instances[active_slot] = NULL; /* defensive */
            *instance = active_slot;
            up(&os_linux_inst_mutex);
            return CSR_RESULT_SUCCESS;
        }
    }

    *instance = 0;
    up(&os_linux_inst_mutex);
    return CSR_RESULT_FAILURE;
}

void uf_add_os_device(int bus_id, struct device *os_device)
{
    if ((bus_id < 0) || (bus_id >= MAX_UNIFI_DEVS))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "unifi : uf_add_os_device: invalid device %d\n",
                               bus_id));
        return;
    }

    active_slot = bus_id;
    os_devices[bus_id] = os_device;
} /* uf_add_os_device() */

void uf_remove_os_device(int bus_id)
{
    if ((bus_id < 0) || (bus_id >= MAX_UNIFI_DEVS))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "unifi : uf_remove_os_device: invalid device %d\n",
                               bus_id));
        return;
    }

    active_slot = bus_id;
    os_devices[bus_id] = NULL;
} /* uf_remove_os_device() */

/*
 * ---------------------------------------------------------------------------
 *  uf_read_proc
 *
 *      Read method for driver node in /proc/driver/unifi0
 *
 *  Arguments:
 *      page
 *      start
 *      offset
 *      count
 *      eof
 *      data
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
#ifdef CONFIG_PROC_FS
static int uf_read_proc(char *page, char **start, off_t offset, int count,
                        int *eof, void *data)
{
#define UNIFI_DEBUG_TXT_BUFFER 8 * 1024
    os_linux_priv_t *priv;
    int actual_amount_to_copy;
    char *p, *orig_p;
    CsrInt32 remain = UNIFI_DEBUG_TXT_BUFFER;
    CsrInt32 written;
    int i;
    CsrResult r;
    CsrWifiHipHip2Stats stats;
    CsrWifiHipDriverInfoDebugInfo *debugInfo = NULL;

    /*
    * The following complex casting is in place in order to eliminate 64-bit compilation warning
    * "cast to/from pointer from/to integer of different size"
    */

    priv = uf_find_instance((int) (long) data);
    if (!priv)
    {
        return 0;
    }

    p = kmalloc(UNIFI_DEBUG_TXT_BUFFER, GFP_KERNEL);

    orig_p = p;

    written = CsrSnprintf(p, remain, "UniFi SDIO Driver: %s %s %s\n",
                          CSR_WIFI_VERSION, __DATE__, __TIME__);
    UNIFI_SNPRINTF_RET(p, remain, written);
#ifdef CSR_SME_USERSPACE
    written = CsrSnprintf(p, remain, "SME: CSR userspace ");
    UNIFI_SNPRINTF_RET(p, remain, written);
#ifdef CSR_SUPPORT_WEXT
    written = CsrSnprintf(p, remain, "with WEXT support\n");
#else
    written = CsrSnprintf(p, remain, "\n");
#endif /* CSR_SUPPORT_WEXT */
    UNIFI_SNPRINTF_RET(p, remain, written);
#endif /* CSR_SME_USERSPACE */
#ifdef CSR_NATIVE_LINUX
    written = CsrSnprintf(p, remain, "SME: native\n");
    UNIFI_SNPRINTF_RET(p, remain, written);
#endif

#ifdef CSR_SUPPORT_SME
    written = CsrSnprintf(p, remain,
                          "Firmware (ROM) build:%lu, Patch:%lu\n",
                          priv->fw_build,
                          priv->sme_versions.firmwarePatch);
    UNIFI_SNPRINTF_RET(p, remain, written);
#endif

    debugInfo = kmalloc(sizeof(CsrWifiHipDriverInfoDebugInfo), GFP_KERNEL);
    if (debugInfo == NULL)
    {
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                           "unifi%d: uf_read_proc: Could not allocate memory for debugInfo struct\n",
                           priv->instance));
        goto end;
    }

    memset(debugInfo, 0, sizeof(CsrWifiHipDriverInfoDebugInfo));
    p += CsrWifiHipDriverInfoGetReq(priv->hip_handle, p, &remain, debugInfo);
    written = CsrSnprintf(p, remain, "Last dbg str: %s\n",
                          debugInfo->last_debug_string);
    UNIFI_SNPRINTF_RET(p, remain, written);

    written = CsrSnprintf(p, remain, "Last dbg16:");
    UNIFI_SNPRINTF_RET(p, remain, written);
    for (i = 0; i < 8; i++)
    {
        written = CsrSnprintf(p, remain, " %04X",
                              debugInfo->last_debug_word16[i]);
        UNIFI_SNPRINTF_RET(p, remain, written);
    }
    written = CsrSnprintf(p, remain, "\n");
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "           ");
    UNIFI_SNPRINTF_RET(p, remain, written);
    for ( ; i < 16; i++)
    {
        written = CsrSnprintf(p, remain, " %04X",
                              debugInfo->last_debug_word16[i]);
        UNIFI_SNPRINTF_RET(p, remain, written);
    }
    written = CsrSnprintf(p, remain, "\n");
    UNIFI_SNPRINTF_RET(p, remain, written);

    r = CsrWifiHipHip2StatsGet(priv->hip_handle, &stats);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,
                           CSR_WIFI_HIP_LOG_DEF, "unifi%d: uf_read_proc: CsrWifiHipHip2StatsGet failed\n",
                           priv ? priv->instance : 0));
        goto end;
    }

    /* HIP config */
    written = CsrSnprintf(p, remain, "ctrl_buffer_h: 0x%x\n", stats.ctrl_buffer_h);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "ctrl_re_read_buffer_h: 0x%x\n", stats.ctrl_re_read_buffer_h);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "ack_buffer_h: 0x%x\n", stats.ack_buffer_h);
    UNIFI_SNPRINTF_RET(p, remain, written);

    written = CsrSnprintf(p, remain, "fh_pushed_h:\n   ");
    UNIFI_SNPRINTF_RET(p, remain, written);
    for (i = 0; i < sizeof(stats.fh_pushed_buffer_h) / sizeof(stats.fh_pushed_buffer_h[0]); i++)
    {
        written = CsrSnprintf(p, remain, " 0x%x", stats.fh_pushed_buffer_h[i]);
        UNIFI_SNPRINTF_RET(p, remain, written);
    }
    written = CsrSnprintf(p, remain, "\n");
    UNIFI_SNPRINTF_RET(p, remain, written);

    /* HIP status */
    written = CsrSnprintf(p, remain, "Rx interrupts: %u\n", stats.hip2_stats_rx_interrupts);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Rx wake-only interrupts: %u\n", stats.hip2_stats_rx_wake_only_interrupts);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Rx wake interrupts with data pending: %u\n", stats.hip2_stats_rx_wake_with_data_interrupts);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Rx non-hip interrupts: %u\n", stats.hip2_stats_rx_spurious_interrupts);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Rx ctrl frames: %u\n", stats.hip2_stats_rx_ctrl_frames);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Rx ctrl frames read errors: %u\n", stats.hip2_stats_rx_ctrl_frames_read_error);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Rx ctrl frames missed: %u\n", stats.hip2_stats_rx_ctrl_frames_missed);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Rx ctrl frames invalid: %u\n", stats.hip2_stats_rx_ctrl_frames_invalid);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Rx ctrl frames value errors: %u\n", stats.hip2_stats_rx_ctrl_frames_value_error);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Rx ctrl frame magic: %d\n", stats.pm);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Rx last ack magic: %u\n", stats.hip2_stats_rx_last_ack_magic);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Rx last implicit ack magic: %u\n", stats.hip2_stats_rx_last_implicit_ack_magic);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Rx last nack magic: %u\n", stats.hip2_stats_rx_last_nack_magic);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Rx cycles: %u (clrs only %u pkts %u)\n", stats.hip2_stats_rx_cycles, stats.hip2_stats_rx_cycles_clrs_only, stats.hip2_stats_rx_cycles_pkts);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Rx pkts per cycle: ");
    UNIFI_SNPRINTF_RET(p, remain, written);
    for (i = 0; i < sizeof(stats.hip2_stats_rx_pkts_per_cycle) / sizeof(unsigned long); i++)
    {
        written = CsrSnprintf(p, remain, "%u ", stats.hip2_stats_rx_pkts_per_cycle[i]);
        UNIFI_SNPRINTF_RET(p, remain, written);
    }
    written = CsrSnprintf(p, remain, "\nRx clrs per cycle: ");
    UNIFI_SNPRINTF_RET(p, remain, written);
    for (i = 0; i < sizeof(stats.hip2_stats_rx_pkts_per_cycle_clrs) / sizeof(unsigned long); i++)
    {
        written = CsrSnprintf(p, remain, "%u ", stats.hip2_stats_rx_pkts_per_cycle_clrs[i]);
        UNIFI_SNPRINTF_RET(p, remain, written);
    }
    written = CsrSnprintf(p, remain, "\nRx bytes: %u (ctrl %u)\n", stats.hip2_stats_rx_total_bytes, stats.hip2_stats_rx_ctrl_bytes);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Rx window efficiency: %u\n", stats.hip2_stats_rx_window_efficiency);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Rx data efficiency: %u\n", 100 - ((stats.hip2_stats_rx_ctrl_bytes * 100) / (stats.hip2_stats_rx_total_bytes ? stats.hip2_stats_rx_total_bytes : 1)));
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Rx signals: %u (unknown %u unsupported %u)\n", stats.hip2_stats_rx_signals, stats.hip2_stats_rx_unknown_signals, stats.hip2_stats_rx_unsupported_signals);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Rx data bytes: %u\n", stats.hip2_stats_rx_data_bytes);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Rx mac frame discards: %u\n", stats.hip2_stats_rx_mac_discards);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Tx queue lengths: c=%d t0=%d t1=%d t2=%d t3=%d\n", stats.cmd_q, stats.t0_q, stats.t1_q, stats.t2_q, stats.t3_q);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Tx queue pause counts: t0=%u t1=%u t2=%u t3=%u\n", stats.hip2_stats_tx_q_pause[3], stats.hip2_stats_tx_q_pause[2], stats.hip2_stats_tx_q_pause[1], stats.hip2_stats_tx_q_pause[0]);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Tx queue pkts: c=%u t0=%u t1=%u t2=%u t3=%u\n", stats.hip2_stats_tx_q_pkts[4], stats.hip2_stats_tx_q_pkts[3], stats.hip2_stats_tx_q_pkts[2], stats.hip2_stats_tx_q_pkts[1], stats.hip2_stats_tx_q_pkts[0]);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Tx pkts re-aligned: %u\n", stats.hip2_stats_tx_realignment);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Tx pkts realloc: %u\n", stats.hip2_stats_tx_realloc);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Tx slots available: %d\n", stats.free_slots);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Tx slot congested events: %u\n", stats.hip2_stats_tx_slot_congested);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Tx cycles: %u\n", stats.hip2_stats_tx_cycles);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Tx cycles starved: %u\n", stats.hip2_stats_tx_cycles_starved);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Tx pkts per cycle: ");
    UNIFI_SNPRINTF_RET(p, remain, written);
    for (i = 0; i < sizeof(stats.hip2_stats_tx_pkts_per_cycle) / sizeof(unsigned long); i++)
    {
        written = CsrSnprintf(p, remain, "%u ", stats.hip2_stats_tx_pkts_per_cycle[i]);

        UNIFI_SNPRINTF_RET(p, remain, written);
    }
    written = CsrSnprintf(p, remain, "\nTx pkts confim msgs: %u\n", stats.hip2_stats_tx_pkt_cfm);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Tx bulk pkts: %u (clrs %u)\n", stats.hip2_stats_tx_bulk_data_pkts, stats.hip2_stats_tx_bulk_data_clrs);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Tx pulled bulk pkts: %u (bytes %u)\n", stats.hip2_stats_tx_pull_bulk_data_pkts, stats.hip2_stats_tx_pull_bulk_data_bytes);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Tx ctrl: %u (pad:%u)\n", stats.hip2_stats_tx_ctrl_bytes, stats.hip2_stats_tx_pad_bytes);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Tx data: %u\n", stats.hip2_stats_tx_data_bytes);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Tx pending: ctrl %u data %u\n", stats.ctrl_data, stats.bulk_data);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Tx window req: ");
    UNIFI_SNPRINTF_RET(p, remain, written);
    for (i = 0; i < sizeof(stats.hip2_stats_tx_window_req) / sizeof(unsigned long); i++)
    {
        written = CsrSnprintf(p, remain, "%u ", stats.hip2_stats_tx_window_req[i]);
        UNIFI_SNPRINTF_RET(p, remain, written);
    }
    written = CsrSnprintf(p, remain, "\nTx window rec: ");
    UNIFI_SNPRINTF_RET(p, remain, written);
    for (i = 0; i < sizeof(stats.hip2_stats_tx_window_rec) / sizeof(unsigned long); i++)
    {
        written = CsrSnprintf(p, remain, "%u ", stats.hip2_stats_tx_window_rec[i]);
        UNIFI_SNPRINTF_RET(p, remain, written);
    }
    written = CsrSnprintf(p, remain, "\nTx pipe ptr: %d ( limit %d )\n", stats.fh_pipe_current, stats.fh_pipe_limit);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Tx window starvation events: %u\n", stats.hip2_stats_tx_window_starved);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Tx window congestion events: %u\n", stats.hip2_stats_tx_window_congested);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Tx window flags: pending %d ack %d cread %d\n", stats.fwin, stats.ack, stats.cread);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Tx window efficiency: %u\n", stats.hip2_stats_tx_window_efficiency);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Tx window deferred: %u\n", stats.hip2_stats_tx_window_deferred);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Tx pkts pending: %d\n", stats.packets_pending);
    UNIFI_SNPRINTF_RET(p, remain, written);

#ifdef PACKET_SCHEDULER_STATS
    {
        CsrWifiHipPacketSchedulerStats psched_qstats;
        CsrWifiHipPacketSchedulerACPri ac;
        CsrInt32 vif, aid;
        written = CsrSnprintf(p, remain, "\nPSched available QSIG: %d\n", CsrWifiHipPacketSchedulerQSigCount(priv->hip_handle));
        UNIFI_SNPRINTF_RET(p, remain, written);
        if (CsrWifiHipPacketSchedulerCtrlStats(priv->hip_handle, &psched_qstats) == CSR_RESULT_SUCCESS)
        {
            written = CsrSnprintf(p, remain, " ctrl: enabled %u (%u) queued:%u (%u) rej: %u discards: %u data: %u\n",
                                  psched_qstats.enabled, psched_qstats.disabled,
                                  psched_qstats.queued_signals, psched_qstats.dequeued_signals,
                                  psched_qstats.rejected_signals, psched_qstats.discarded_signals,
                                  psched_qstats.dequeued_data);
            UNIFI_SNPRINTF_RET(p, remain, written);
        }
        for (vif = PS_MIN_VIF_ID; vif <= PS_MAX_VIF_ID; vif++)
        {
            if (CsrWifiHipPacketSchedulerMgmtStats(priv->hip_handle, vif, &psched_qstats) == CSR_RESULT_SUCCESS)
            {
                written = CsrSnprintf(p, remain, " vif=%2d mgmt: enabled %u (%u) queued:%u (%u) rej: %u discards: %u data: %u\n",
                                      vif,
                                      psched_qstats.enabled, psched_qstats.disabled,
                                      psched_qstats.queued_signals, psched_qstats.dequeued_signals,
                                      psched_qstats.rejected_signals, psched_qstats.discarded_signals,
                                      psched_qstats.dequeued_data);

                UNIFI_SNPRINTF_RET(p, remain, written);
                if (CsrWifiHipPacketSchedulerMulticastStats(priv->hip_handle, vif, &psched_qstats) == CSR_RESULT_SUCCESS)
                {
                    written = CsrSnprintf(p, remain, "        mcast: enabled %u (%u) queued:%u (%u) rej: %u discards: %u data: %u\n",
                                          psched_qstats.enabled, psched_qstats.disabled,
                                          psched_qstats.queued_signals, psched_qstats.dequeued_signals,
                                          psched_qstats.rejected_signals, psched_qstats.discarded_signals,
                                          psched_qstats.dequeued_data);
                    UNIFI_SNPRINTF_RET(p, remain, written);
                }
                for (aid = PS_MIN_AID; aid < PS_MAX_AID; aid++)
                {
                    CsrInt32 station_state = -1;
                    CsrUint32 station_state_transitions = 0;
                    CsrUint32 smod = 0, scod = 0;
                    if (CsrWifiHipPacketSchedulerQueueSetSMODGet(priv->hip_handle, vif, aid, &smod, &scod) == CSR_RESULT_SUCCESS)
                    {
                        written = CsrSnprintf(p, remain, "        aid=%2d smod=%u scod=%u\n",
                                              aid,
                                              smod,
                                              scod);
                        UNIFI_SNPRINTF_RET(p, remain, written);
                        if (CsrWifiHipPacketSchedulerQueueSetStationStateGet(priv->hip_handle, vif, aid, &station_state, &station_state_transitions) == CSR_RESULT_SUCCESS)
                        {
                            written = CsrSnprintf(p, remain, "        aid=%2d state=%s transitions=%u\n",
                                                  aid,
                                                  station_state ? "active" : "powersave",
                                                  station_state_transitions);
                            UNIFI_SNPRINTF_RET(p, remain, written);
                            if (CsrWifiHipPacketSchedulerQueueSetMgmtStats(priv->hip_handle, vif, aid, &psched_qstats) == CSR_RESULT_SUCCESS)
                            {
                                written = CsrSnprintf(p, remain, "        aid=%2d mgmt: enabled %u (%u) queued:%u (%u) rej: %u discards: %u data: %u\n",
                                                      aid,
                                                      psched_qstats.enabled, psched_qstats.disabled,
                                                      psched_qstats.queued_signals, psched_qstats.dequeued_signals,
                                                      psched_qstats.rejected_signals, psched_qstats.discarded_signals,
                                                      psched_qstats.dequeued_data);
                                UNIFI_SNPRINTF_RET(p, remain, written);

                                if (CsrWifiHipPacketSchedulerQueueSetAuthStats(priv->hip_handle, vif, aid, &psched_qstats) == CSR_RESULT_SUCCESS)
                                {
                                    written = CsrSnprintf(p, remain, "        aid=%2d auth: enabled %u (%u) queued:%u (%u) rej: %u discards: %u data: %u\n",
                                                          aid,
                                                          psched_qstats.enabled, psched_qstats.disabled,
                                                          psched_qstats.queued_signals, psched_qstats.dequeued_signals,
                                                          psched_qstats.rejected_signals, psched_qstats.discarded_signals,
                                                          psched_qstats.dequeued_data);
                                    UNIFI_SNPRINTF_RET(p, remain, written);
                                }
                                for (ac = AC_PRI_0; ac <= AC_PRI_3; ac++)
                                {
                                    CsrWifiHipPacketSchedulerQSACQState acq_state;

                                    acq_state = 0;
                                    if (CsrWifiHipPacketSchedulerQueueSetACStats(priv->hip_handle, vif, aid, ac, &psched_qstats, &acq_state) == CSR_RESULT_SUCCESS)
                                    {
                                        written = CsrSnprintf(p, remain, "               ac=%d: qstate=%s enabled %u (%u) queued:%u (%u) rej: %u discards: %u data: %u\n",
                                                              ac,
                                                              acq_state == QS_ACQ_ENABLED ? "up" : "dn",
                                                              psched_qstats.enabled, psched_qstats.disabled,
                                                              psched_qstats.queued_signals, psched_qstats.dequeued_signals,
                                                              psched_qstats.rejected_signals, psched_qstats.discarded_signals,
                                                              psched_qstats.dequeued_data);
                                        UNIFI_SNPRINTF_RET(p, remain, written);
                                    }
                                    else
                                    {
                                        break;
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        written = CsrSnprintf(p, remain, "DWRR: AC   Scheduled     Missed CurrTokens TotalTokens\n");
        UNIFI_SNPRINTF_RET(p, remain, written);
        for (ac = AC_PRI_0; ac <= AC_PRI_3; ac++)
        {
            CsrWifiHipPacketSchedulerDwrrStats dwrr_stats = {0, 0, 0};
            if (CsrWifiHipPacketSchedulerDwrrStatsGet(priv->hip_handle, QS_PRI_NORMAL, ac, &dwrr_stats) == CSR_RESULT_SUCCESS)
            {
                written = CsrSnprintf(p, remain, "      %2d %11u %11u %11d %11u\n",
                                      ac,
                                      dwrr_stats.scheduled,
                                      dwrr_stats.missed,
                                      dwrr_stats.current_tokens,
                                      dwrr_stats.accumulated_tokens);
                UNIFI_SNPRINTF_RET(p, remain, written);
            }
        }
    }
    #endif
    written = CsrSnprintf(p, remain, "\nCmd53 errors:");
    UNIFI_SNPRINTF_RET(p, remain, written);
    for (i = 0; i < sizeof(stats.hip2_sdio_cmd53_errors) / sizeof(stats.hip2_sdio_cmd53_errors[0]); i++)
    {
        written = CsrSnprintf(p, remain, "%u ", stats.hip2_sdio_cmd53_errors[i]);
        UNIFI_SNPRINTF_RET(p, remain, written);
    }

    written = CsrSnprintf(p, remain, "\nState torpid=%u\n", stats.hip2_stats_state_torpid);
    UNIFI_SNPRINTF_RET(p, remain, written);
    written = CsrSnprintf(p, remain, "Bh sched: intr %u host %u\n", stats.hip2_stats_bh_sched[0], stats.hip2_stats_bh_sched[1]);
    UNIFI_SNPRINTF_RET(p, remain, written);
    for (i = 0; i < CSR_WIFI_MAX_INTERFACES; i++)
    {
        struct net_device *dev;

        dev = priv->netdev[i];
        if (dev && netif_running(dev))
        {
            struct net_device_stats *stats;
            if (dev->netdev_ops && dev->netdev_ops->ndo_get_stats)
            {
                stats = dev->netdev_ops->ndo_get_stats(dev);
                if (stats)
                {
                    written = CsrSnprintf(p, remain, "%s: txpkts %lu txbytes %lu rxpkts %lu rxbytes %lu\n",
                                          dev->name, stats->tx_packets, stats->tx_bytes, stats->rx_packets, stats->rx_bytes);
                    UNIFI_SNPRINTF_RET(p, remain, written);
                }
            }
        }
    }

    written = CsrSnprintf(p, remain, "\n");
    UNIFI_SNPRINTF_RET(p, remain, written);

end:
    *start = page;

    written = UNIFI_DEBUG_TXT_BUFFER - remain;

    if (offset >= written)
    {
        *eof = 1;
        kfree(orig_p);

        if (debugInfo != NULL)
        {
            kfree(debugInfo);
        }
        return 0;
    }

    if (offset + count > written)
    {
        actual_amount_to_copy = written - offset;
        *eof = 1;
    }
    else
    {
        actual_amount_to_copy = count;
    }

    memcpy(page, &(orig_p[offset]), actual_amount_to_copy);

    kfree(orig_p);

    if (debugInfo != NULL)
    {
        kfree(debugInfo);
    }

    return actual_amount_to_copy;
} /* uf_read_proc() */

#endif


/*
 * ---------------------------------------------------------------------------
 *  register_unifi_linux
 *
 *      This function is called from the the SDIO insert call back of the
 *      SDIO driver when a UniFi card is detected.
 *      We allocate the Linux net_device struct, create the char device
 *      nodes and start the userspace helper to initialise the device.
 *
 *  Arguments:
 *      hip_handle      Handle to the HAL instance to use for all calls to
 *                      HAL.
 *      bus_id          The instance number. Typically this is the slot number,
 *                      e.g. 0, 1 etc. Valid values are 0 to MAX_UNIFI_DEVS-1.
 *
 *  Returns:
 *      The result of the registration.
 * ---------------------------------------------------------------------------
 */
CsrResult register_unifi_linux(void *hip_handle, int bus_id)
{
    int r = -1;
    os_linux_priv_t *priv = NULL;

    func_enter();

    if ((bus_id < 0) || (bus_id >= MAX_UNIFI_DEVS))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi : register_unifi_sdio: invalid device %d\n", bus_id));
        return CSR_RESULT_FAILURE;
    }

    down(&os_linux_inst_mutex);

    /* Allocate device private and net_device structs */
    priv = uf_alloc_netdevice(bus_id);
    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to allocate driver private\n", bus_id));
        goto failed0;
    }

    priv->hip_handle = hip_handle;
    priv->state = OS_LINUX_STATE_ACTIVATED;
    priv->unifi_device = os_devices[bus_id];

#ifdef CSR_SUPPORT_WEXT
    CsrMemSet(&priv->wext_wireless_stats, 0, sizeof(struct iw_statistics));
#endif

    SET_NETDEV_DEV(priv->netdev[0], os_devices[bus_id]);

    /* We are not ready to send data yet. */
    netif_carrier_off(priv->netdev[0]);

    if (Unifi_instances[bus_id])
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi : Internal error: instance for slot %d is already taken\n", bus_id));
    }

    /* Store priv */
    Unifi_instances[bus_id] = priv;

    /* Save the netdev_priv for use by the netdev event callback mechanism */
    Unifi_netdev_instances[bus_id * CSR_WIFI_MAX_INTERFACES] = netdev_priv(priv->netdev[0]);

    /* Create the character device nodes */
    r = uf_create_device_nodes(priv, bus_id);
    if (r)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to create device node\n", bus_id));
        goto failed1;
    }

    /*
     * We use the slot number as unifi device index.
     */
    snprintf(priv->proc_entry_name, 64, "driver/unifi%d", priv->instance);
    /*
     * The following complex casting is in place in order to eliminate 64-bit compilation warning
     * "cast to/from pointer from/to integer of different size"
     */
    if (!create_proc_read_entry(priv->proc_entry_name, 0, 0,
                                uf_read_proc, (void *) (long) priv->instance))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi: can't create /proc/driver/unifi\n", bus_id));
    }

    /* Allocate the net_device for interfaces other than 0. */
    {
        int i;
        priv->totalInterfaceCount = 0;

        for (i = 1; i < CSR_WIFI_MAX_INTERFACES; i++)
        {
            if (!uf_alloc_netdevice_for_other_interfaces(priv, i))
            {
                /* error occured while allocating the net_device for interface[i]. The net_device are
                 * allocated for the interfaces with id<i. Dont worry, all the allocated net_device will
                 * be releasing chen the control goes to the label failed0.
                 */
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to allocate driver private for interface[%d]\n",
                                       priv ? priv->instance : 0, i));
                goto failed0;
            }
            else
            {
                SET_NETDEV_DEV(priv->netdev[i], os_devices[bus_id]);

                /* We are not ready to send data yet. */
                netif_carrier_off(priv->netdev[i]);

                /* Save the netdev_priv for use by the netdev event callback mechanism */
                Unifi_netdev_instances[bus_id * CSR_WIFI_MAX_INTERFACES + i] = netdev_priv(priv->netdev[i]);
            }
        }

        for (i = 0; i < CSR_WIFI_MAX_INTERFACES; i++)
        {
            netInterface_priv_t *interfacePriv = priv->interfacePriv[i];
            interfacePriv->netdev_state = OS_LINUX_NETDEV_STATE_UNREGISTERED;
#ifdef CSR_SUPPORT_SME
            /* Create m4 buffering work structure for other interfaces since it is used for multi interface combination of STA + P2PCli*/
            INIT_WORK(&interfacePriv->send_m4_ready_task, uf_send_m4_ready_wq);
#endif
        }
    }

    /* Initialise the SME related threads and parameters */
    r = uf_sme_init(priv);
    if (r)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: SME initialisation failed.\n", bus_id));
        goto failed4;
    }

    /*
     * Run the userspace helper program (unififw) to perform
     * the device initialisation.
     */
/*ALSI_DL 20140206*/
    #if 0 /* not used in ALPS prj */
    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi%d: run UniFi helper app...\n", bus_id));
    r = uf_run_unifihelper(priv);
    if (r)
    {
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: unable to run UniFi helper app\n", bus_id));
        /* Not a fatal error. */
    }
    #endif /* not used in ALPS prj */

    up(&os_linux_inst_mutex);

    func_exit();
    return CSR_RESULT_SUCCESS;

failed4:
    /* Remove the device nodes */
    uf_destroy_device_nodes(priv);
failed1:
    /* Deregister priv->netdev_client */
    ul_deregister_client(priv->netdev_client);

failed0:
    if (priv)
    {
        uf_free_netdevice(priv);
    }

    up(&os_linux_inst_mutex);

    func_exit();

    return CSR_RESULT_FAILURE;
} /* register_unifi_sdio() */

/*
 * ---------------------------------------------------------------------------
 *  cleanup_unifi_sdio
 *
 *      Release any resources owned by a unifi instance.
 *
 *  Arguments:
 *      priv          Pointer to the instance to free.
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
static void cleanup_unifi_sdio(os_linux_priv_t *priv)
{
    int priv_instance;
    CsrUint16 i;

    func_enter();

    /* Remove the device nodes */
    uf_destroy_device_nodes(priv);

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5,
                        "unifi%d: cleanup_unifi_sdio: remove_proc_entry\n",
                        priv ? priv->instance : 0));
    /*
     * Free the children of priv before unifi_free_netdevice() frees
     * the priv struct
     */
    remove_proc_entry(priv->proc_entry_name, 0);


    /* Unregister netdev as a client. */
    if (priv->netdev_client)
    {
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                           "unifi%d: Netdev client (id:%d s:0x%X) is unregistered\n",
                           priv->instance, priv->netdev_client->client_id, priv->netdev_client->sender_id));
        ul_deregister_client(priv->netdev_client);
    }

    /* Destroy the SME related threads and parameters */
    uf_sme_deinit(priv);

#ifdef CSR_SME_USERSPACE
    priv->smepriv = NULL;
#endif


    /* Clear the table of registered netdev_priv's */
    for (i = 0; i < CSR_WIFI_MAX_INTERFACES; i++)
    {
        /*
         * Unregister the network device.
         * We can not unregister the netdev before we release
         * all pending packets in the core.
         */
        uf_unregister_netdev(priv, i);
        Unifi_netdev_instances[priv->instance * CSR_WIFI_MAX_INTERFACES + i] = NULL;
    }

    priv->totalInterfaceCount = 0;

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5,
                        "unifi%d: cleanup_unifi_sdio: uf_free_netdevice\n",
                        priv->instance));
    /*
     * When uf_free_netdevice() returns, the priv is invalid
     * so we need to remember the instance to clear the global flag later.
     */
    priv_instance = priv->instance;

    /* Priv is freed as part of the net_device */
    uf_free_netdevice(priv);

    /* Mark the instance as released */
    uf_release_instance(priv_instance);

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5, "unifi%d: cleanup_unifi_sdio: DONE.\n", priv_instance));

    func_exit();
} /* cleanup_unifi_sdio() */

/*
 * ---------------------------------------------------------------------------
 *  unregister_unifi_linux
 *
 *      Call from SDIO driver when it detects that UniFi has been removed.
 *
 *  Arguments:
 *      bus_id          Number of the card that was ejected.
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
void unregister_unifi_linux(int bus_id)
{
    os_linux_priv_t *priv;
    int interfaceTag = 0;
    u8 reason = CONFIG_IND_EXIT;
    CsrResult result;

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG5,
                        "unifi : unregister_unifi_linux %d\n", bus_id));

    if ((bus_id < 0) || (bus_id >= MAX_UNIFI_DEVS))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "unifi : unregister_unifi_linux: invalid device %d\n",
                               bus_id));
        return;
    }

    priv = Unifi_instances[bus_id];
    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: unregister_unifi_sdio: device %d is not registered\n",
                               priv ? priv->instance : 0, bus_id));
        func_exit();
        return;
    }

    /* Stop the network traffic before freeing the core. */
    for (interfaceTag = 0; interfaceTag < CSR_WIFI_MAX_INTERFACES; interfaceTag++)
    {
        netInterface_priv_t *interfacePriv = priv->interfacePriv[interfaceTag];
        if ((interfacePriv->netdev_state == OS_LINUX_NETDEV_STATE_REGISTERED) ||
            (interfacePriv->netdev_state == OS_LINUX_NETDEV_STATE_REGISTERED_STOPPED))
        {
            netif_carrier_off(priv->netdev[interfaceTag]);
            UF_NETIF_TX_STOP_ALL_QUEUES(priv->netdev[interfaceTag]);
        }
    }

#ifdef CSR_NATIVE_LINUX
    /*
     * If the unifi thread was started, signal it to stop.  This
     * should cause any userspace processes with open unifi device to
     * close them.
     */
    if ((priv->hip_handle != NULL) && (priv->init_progress != UNIFI_INIT_NONE))
    {
        CsrWifiHipWifiOffReq(TRUE, priv->hip_handle);
    }

    priv->state = OS_LINUX_STATE_ACTIVATED;

    /* Ensure no MLME functions are waiting on a the mlme_event semaphore. */
    uf_abort_mlme(priv);
#endif /* CSR_NATIVE_LINUX */

    ul_log_config_ind(priv, &reason, sizeof(u8));

    /* Unregister the UDI log hook from the core. */
    result = CsrWifiHipLogUdiUnregistrationReq(priv->hip_handle);
    if (result != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                            "unifi%d: unregister_unifi_linux: Failed to unregister UDI log hook\n", priv ? priv->instance : 0));
    }

    uf_put_instance(bus_id);

    /*
     * Wait until the device is cleaned up. i.e., when all userspace
     * processes have closed any open unifi devices.
     */
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                        "unifi%d: Waiting for clean up event\n", priv ? priv->instance : 0));

    wait_event(Unifi_cleanup_wq, In_use[bus_id] == UNIFI_DEV_CLEANUP);

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                        "unifi%d: Received clean up event\n", priv ? priv->instance : 0));

    /* Now we can free the private context and the char device nodes */
    cleanup_unifi_sdio(priv);
} /* unregister_unifi_linux() */

/*
 * ---------------------------------------------------------------------------
 *  uf_register_netdev
 *
 *      Registers the network interface, installes the qdisc,
 *      and registers the inet handler.
 *      In the porting exercise, register the driver to the network
 *      stack if necessary.
 *
 *  Arguments:
 *      priv          Pointer to driver context.
 *
 *  Returns:
 *      O on success, non-zero otherwise.
 *
 *  Notes:
 *      We will only unregister when the card is ejected, so we must
 *      only do it once.
 * ---------------------------------------------------------------------------
 */
int uf_register_netdev(os_linux_priv_t *priv, CsrUint16 interfaceTag)
{
    int r;
    netInterface_priv_t *interfacePriv = priv->interfacePriv[interfaceTag];

    if (interfaceTag >= CSR_WIFI_MAX_INTERFACES)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: uf_register_netdev bad interfaceTag\n", priv ? priv->instance : 0));
        return -EINVAL;
    }

    /*
     * Allocates a device number and registers device with the network
     * stack.
     */
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5,
                        "unifi%d: uf_register_netdev: netdev %d - 0x%p\n", priv ? priv->instance : 0,
                        interfaceTag, priv->netdev[interfaceTag]));

    r = register_netdev(priv->netdev[interfaceTag]);
    if (r)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to register net device\n", priv ? priv->instance : 0));
        return -EINVAL;
    }

    /* The device is registed */
    interfacePriv->netdev_state = OS_LINUX_NETDEV_STATE_REGISTERED;

#ifdef CSR_SUPPORT_SME
    /*
     * Register the inet handler; it notifies us for changes in the IP address.
     */
    uf_register_inet_notifier();
#endif /* CSR_SUPPORT_SME */

    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                           CSR_WIFI_HIP_UDBG1, "unifi%d: unifi%d is %s\n",
                           priv ? priv->instance : 0,
                           priv->instance, priv->netdev[interfaceTag]->name));

    priv->totalInterfaceCount++;
    return 0;
} /* uf_register_netdev */

/*
 * ---------------------------------------------------------------------------
 *  uf_unregister_netdev
 *
 *      Unregisters the network interface and the inet handler.
 *
 *  Arguments:
 *      priv          Pointer to driver context.
 *
 *  Returns:
 *      None.
 *
 * ---------------------------------------------------------------------------
 */
void uf_unregister_netdev(os_linux_priv_t *priv, CsrUint16 interfaceTag)
{
    netInterface_priv_t *interfacePriv = priv->interfacePriv[interfaceTag];

#ifdef CSR_SUPPORT_SME
    /* Unregister the inet handler... */
    uf_unregister_inet_notifier();
#endif /* CSR_SUPPORT_SME */

    if ((interfacePriv->netdev_state == OS_LINUX_NETDEV_STATE_REGISTERED) ||
        (interfacePriv->netdev_state == OS_LINUX_NETDEV_STATE_REGISTERED_STOPPED))
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5,
                            "unifi%d: uf_unregister_netdev: netdev %d - 0x%p\n",
                            priv ? priv->instance : 0,
                            interfaceTag, priv->netdev[interfaceTag]));

        /* ... and the netdev */
        unregister_netdev(priv->netdev[interfaceTag]);
        interfacePriv->netdev_state = OS_LINUX_NETDEV_STATE_UNREGISTERED;
    }

    interfacePriv->interfaceMode = CSR_WIFI_ROUTER_CTRL_MODE_NONE;
    priv->totalInterfaceCount--;
} /* uf_unregister_netdev() */

/*
 * ---------------------------------------------------------------------------
 *  ask_unifi_sdio_cleanup
 *
 *      We can not free our private context, until all the char device
 *      clients have closed the file handles. unregister_unifi_sdio() which
 *      is called when a card is removed, waits on Unifi_cleanup_wq until
 *      the reference count becomes zero. It is time to wake it up now.
 *
 *  Arguments:
 *      priv          Pointer to driver context.
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
static void ask_unifi_sdio_cleanup(os_linux_priv_t *os_linux)
{
    func_enter();

    /*
     * Now clear the flag that says the old instance is in use.
     * This is used to prevent a new instance being started before old
     * one has finshed closing down, for example if bounce makes the card
     * appear to be ejected and re-inserted quickly.
     */
    In_use[os_linux->instance] = UNIFI_DEV_CLEANUP;

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5, "unifi : ask_unifi_sdio_cleanup: wake up cleanup workqueue.\n"));
    wake_up(&Unifi_cleanup_wq);

    func_exit();
} /* ask_unifi_sdio_cleanup() */
