/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"

#include "csr_wifi_hip_log_text.h"
#include "csr_result.h"
#include "csr_wifi_firmware_patch.h"
#include "os_linux_priv.h"
#include "csr_pmem.h"

#include <linux/firmware.h>

#define UNIFI_MAX_FW_PATH_LEN       32


/*
 * ---------------------------------------------------------------------------
 *  uf_release_firmware
 *
 *      Release specific buffer used to store firmware
 *
 *  Arguments:
 *      priv            Pointer to OS private struct.
 *      to_free         Pointer to specific buffer to release
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
static int uf_release_firmware(os_linux_priv_t *os_linux, const struct firmware *to_free)
{
    if (to_free != NULL)
    {
        release_firmware(to_free);
    }

    return 0;
}

CsrResult CsrWifiFirmwarePatchAcquire(void *osLayerContext, CsrUint32 fwBuild, CsrSize *length, CsrUint8 **firmware)
{
    os_linux_priv_t *os_linux = NULL;
    const struct firmware *fw_entry;
    int postfix;
    char fw_name[UNIFI_MAX_FW_PATH_LEN];
    int r;

    if (osLayerContext == NULL)
    {
        return CSR_RESULT_FAILURE;
    }

    os_linux = (os_linux_priv_t *) osLayerContext;

    if (os_linux->unifi_device == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: NULL unifi_device context\n", os_linux ? os_linux->instance : 0));
        return CSR_RESULT_FAILURE;
    }

    postfix = os_linux->instance;
    scnprintf(fw_name, UNIFI_MAX_FW_PATH_LEN, "unifi-sdio-%d/%s",
              postfix, "sta.xbv");
    r = request_firmware(&fw_entry, fw_name, os_linux->unifi_device);
    if (r == 0)
    {
        *length = fw_entry->size;
        *firmware = CsrPmemAlloc(*length);
        memcpy(*firmware, fw_entry->data, *length);

        r = uf_release_firmware(os_linux, fw_entry);
        if (r)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to free firmware structure\n",
                                   os_linux ? os_linux->instance : 0));
            return CSR_RESULT_FAILURE;
        }
    }
    else
    {
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi%d: Firmware file not available\n",
                           os_linux ? os_linux->instance : 0));

        /* Still have to release firmware since the request_firmware call registers various things */
        r = uf_release_firmware(os_linux, fw_entry);
        if (r)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Failed to release firmware\n",
                                   os_linux ? os_linux->instance : 0));
            return CSR_RESULT_FAILURE;
        }

        return CSR_RESULT_FAILURE;
    }

    return CSR_RESULT_SUCCESS;
}

#define UNIFIHELPER_INIT_MODE_SMEEMB    0
#define UNIFIHELPER_INIT_MODE_SMEUSER   2
#define UNIFIHELPER_INIT_MODE_NATIVE    1

/*
 * ---------------------------------------------------------------------------
 *  uf_run_unifihelper_cleanup
 *
 *      Function used to deallocate argv in the sp_info structure.
 *
 *  Arguments:
 *      sp_info         Pointer to structure allocated by call_usermodehelper_setup().
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 34) || defined(CSR_WIFI_DRIVER_COMPAT_OLD_USERMODEHELPER_API)) && !defined(CSR_WIFI_DRIVER_COMPAT_NEW_USERMODEHELPER_API)
static void uf_run_unifihelper_cleanup(char **argv, char **envp)
{
    if (!argv)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_LOG_DEF,
                               "unifi: uf_run_unifihelper_cleanup: argv is null"));
        return;
    }
    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                       "unifi: uf_run_unifihelper_cleanup: 0x%p\n", argv));
    argv_free(argv);
}

#else
static void uf_run_unifihelper_cleanup(struct subprocess_info *sp_info)
{
    if (!sp_info)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_LOG_DEF,
                               "unifi: uf_run_unifihelper_cleanup: sp_info is null"));
        return;
    }
    if (!sp_info->argv)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_LOG_DEF,
                               "unifi: uf_run_unifihelper_cleanup: argv is null"));
        return;
    }
    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                       "unifi: uf_run_unifihelper_cleanup: 0x%p\n", sp_info->argv));
    argv_free(sp_info->argv);
}

#endif
/*
 * ---------------------------------------------------------------------------
 *  uf_run_unifihelper
 *
 *      Ask userspace to send us firmware for download by running
 *      '/usr/sbin/unififw'.
 *      The same script starts the SME userspace application.
 *      Derived from net_run_sbin_hotplug().
 *
 *  Arguments:
 *      priv            Pointer to OS private struct.
 *
 *  Returns:
 *      zero success, negative otherwise.
 * ---------------------------------------------------------------------------
 */
int uf_run_unifihelper(os_linux_priv_t *priv)
{
#ifdef CONFIG_HOTPLUG
#ifdef ANDROID_BUILD
    char *prog = "/system/bin/unififw";
#else
    char *prog = "/usr/sbin/unififw";
#endif /* ANDROID_BUILD */
#if (defined CSR_SME_USERSPACE)
    int user = UNIFIHELPER_INIT_MODE_SMEUSER;
#else
    int user = UNIFIHELPER_INIT_MODE_NATIVE;
#endif /* CSR_SME_USERSPACE */

    const int exec_string_buffer_len = 64;
    const int exec_string_args = 3;
    char **argv;
    static char *envp[] = {"HOME=/", "PATH=/sbin:/bin:/usr/sbin:/usr/bin", NULL};
    char argv_str[exec_string_buffer_len];
    int argc, r, len;

#if (defined CSR_SME_USERSPACE) && (!defined CSR_SUPPORT_WEXT)
    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                       "unifi%d: SME userspace build: run unifi_helper manually\n", priv ? priv->instance : 0));
    return 0;
#endif

    /* Helper was disabled via module parameter */
    if (no_helper)
    {
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                           "unifi%d: Run userspace helper manually\n", priv ? priv->instance : 0));
        return 0;
    }

    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                       "unifi%d: starting %s\n", priv ? priv->instance : 0,  prog));

    len = snprintf(argv_str, exec_string_buffer_len, "%s %d %d", prog, priv->instance, user);
    if (len >= exec_string_buffer_len)
    {
        /* snprintf() returns a value of buffer size of greater if it had to truncate the format string. */
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: exec string buffer insufficient (buffer size=%d, actual string=%d)\n", priv->instance, exec_string_buffer_len, len));
        return -1;
    }

    /* Kernel library function argv_split() will allocate memory for argv. */
    argc = 0;
    argv = argv_split(GFP_ATOMIC, argv_str, &argc);
    if (!argv)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: failed to allocate argv for userspace helper\n", priv->instance));
        return -1;
    }
    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                       "unifi: argv allocated 0x%p\n", argv));
    /* Check the argument count - should be exec_string_args. */
    if (argc != exec_string_args)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: exec string has the wrong number of arguments (has %d, should be %d\n", priv->instance, argc, exec_string_args));
        argv_free(argv);
        return -1;
    }
    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG2,
                       "unifi%d: running %s %s %s\n", priv ? priv->instance : 0,  argv[0], argv[1], argv[2]));

    #if LINUX_VERSION_CODE < KERNEL_VERSION(3, 5, 0)
    {
        struct subprocess_info *sp_info;

        /* Allocate sp_info and initialise pointers to argv and envp. */
        sp_info = call_usermodehelper_setup(argv[0], argv, envp, GFP_KERNEL);
        if (!sp_info)
        {
            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: call_usermodehelper_setup() failed\n", priv->instance));
            argv_free(argv);
            return -1;
        }
        /*
        * Set the cleanup function for the memory allocated for argv. This function
        * is called according to the UMH flag passed to call_usermodehelper_exec().
        * Note: The USE_OLD_USERMODEHELPER_API allows the configuration system to force
        * the driver to use the old api - this only appears to be necessary for patched
        * versions of 2.6.34 where the KERNEL_VERSION is not enough information to make
        * a decision.
        */
        #if (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 34) || defined(CSR_WIFI_DRIVER_COMPAT_OLD_USERMODEHELPER_API)) && !defined(CSR_WIFI_DRIVER_COMPAT_NEW_USERMODEHELPER_API)
        call_usermodehelper_setcleanup(sp_info, uf_run_unifihelper_cleanup);
        #else
        call_usermodehelper_setfns(sp_info, NULL, uf_run_unifihelper_cleanup, NULL);
        #endif

        /*
        * Put sp_info into work queue for processing by khelper.
        * Beware that UMH_WAIT_EXEC means that the calling context
        * for this function will be blocked until the khelper thread
        * has successfully created a new thread for the execution
        * of the user space process. It does not mean that the calling
        * thread will be blocked until the user space process has actually
        * been executed.
        */
        r = call_usermodehelper_exec(sp_info, UMH_WAIT_EXEC);
        if (r != 0)
        {
            /*
            * call_usermodehelper_exec() will free sp_info and call any cleanup function
            * whether it succeeds or fails, so do not free argv.
            */
            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: call_usermodehelper() failed with %d\n", priv->instance, r));
        }
    }
    #else
    r = call_usermodehelper_fns(argv[0], argv, envp, UMH_WAIT_EXEC, NULL, uf_run_unifihelper_cleanup, NULL);
    #endif /* KERNEL_VERSION(3, 5, 0) */

    return r;
#else
    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                       "unifi%d: Can't automatically download firmware because kernel does not have HOTPLUG\n", priv ? priv->instance : 0));
    return -1;
#endif
} /* uf_run_unifihelper() */
