/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"
#include "csr_types.h"
#include "csr_log_text_2.h"
#include "csr_wifi_hip_log_text.h"
#include "csr_pmem.h"

#ifdef UNIFI_DEBUG
#ifdef CSR_LOG_ENABLE

static const int trace_critical_mask = 1;
static const int trace_error_mask = 2;
static const int trace_warn_mask = 4;
static const int trace_info_mask = 8;
static const int trace_debug_mask = 16;

/* A Lep is a way of storing a string as a LEngth and a Pointer, without needing the traditional C '\0' terminator */
typedef struct
{
    CsrUint16            len;
    const CsrCharString *ptr;
} Lep;

/* This is an abstract representation of each trace level specified in the string
   A zero length for origin means the level applies to all origins
   A zero length for suborigin means the level applies to all suborigins
*/
typedef struct Levelspec
{
    Lep               origin;
    Lep               suborigin;
    int               level_mask;
    struct Levelspec *next;
} Levelspec;

/* returns 0 if they are the same */
static int string_and_lep_are_different(const CsrCharString *string, Lep lep)
{
    if (lep.len == CsrStrLen(string))
    {
        return CsrMemCmp(lep.ptr, string, lep.len);
    }
    return 1;
}

/* Applies the passed trace level specification (passed in ordered list Levelspec*)
   to the passed handle
*/
static void apply_settings_to_handle(CsrLogTextHandle *handle, Levelspec *settings)
{
    while (settings)
    {
        if (!settings->origin.len || !string_and_lep_are_different(handle->origin, settings->origin))
        {
            int i;

            for (i = 0; i < handle->suborigin_n; i++)
            {
                if (!settings->suborigin.len || !string_and_lep_are_different(handle->suborigins[i], settings->suborigin))
                {
                    handle->level_masks[i] = settings->level_mask;
                }
            }
        }
        settings = settings->next;
    }
}

/* Whether passed char is acceptable in an origin (and suborigin) name */
static CsrInt16 is_nameworthy(CsrInt16 c)
{
    return (c >= 'A' && c <= 'Z') || ('_' == c) || (c >= '0' && c <= '9');
}

static Lep Str2lep(const CsrCharString *string)
{
    Lep leppy;

    leppy.ptr = string;
    leppy.len = (CsrUint16) CsrStrLen(leppy.ptr);

    return leppy;
}

static void free_level_spec_list(Levelspec *head)
{
    while (head)
    {
        Levelspec *current = head;

        head = current->next;
        CsrPmemFree(current);
    }
}

static void remove_first_char(Lep *input)
{
    if (input->len > 0)
    {
        input->len--;
        input->ptr++;
    }
}

static void remove_optional_leading_spaces(Lep *lep)
{
    while (lep->len && ' ' == *lep->ptr)
    {
        lep->len--;
        lep->ptr++;
    }
}

/*
   This function parses the trace level (CEWID) string from the start of the passed string.
   A level string must be one (and only one) of the following:
      zero length
      the single char "A"
      any sequence of chars from the set {CEWID} (repetitions are allowed for now, although they make no sense)

   This function can never encounter a syntax error because it assumes a zero length mask if an there is unacceptable char at
   the start of the string.  And in all cases the mask ends just before the first unacceptable character.

   Returns the mask value, which is bitwise OR of all the bit mask values specified by the string.
   Updates the passed string by removing the part that it has parsed from the front.
*/
static int get_level_token(Lep *remaining_ip)
{
    int mask = 0;

    if (!remaining_ip->len)
    {
        return 0;
    }

    if ('A' == *remaining_ip->ptr)
    {
        remove_first_char(remaining_ip);
        mask = 0xffff;
    }
    else
    {
        CsrInt16 finished = 0;

        do
        {
            switch (*remaining_ip->ptr)
            {
                case 'C':
                    remove_first_char(remaining_ip);
                    mask |= trace_critical_mask;
                    break;
                case 'E':
                    remove_first_char(remaining_ip);
                    mask |= trace_error_mask;
                    break;
                case 'W':
                    remove_first_char(remaining_ip);
                    mask |= trace_warn_mask;
                    break;
                case 'I':
                    remove_first_char(remaining_ip);
                    mask |= trace_info_mask;
                    break;
                case 'D':
                    remove_first_char(remaining_ip);
                    mask |= trace_debug_mask;
                    break;
                default:
                    finished = 1;
            }
        } while (!finished && remaining_ip->len);
    }
    return mask;
}

static void flag_syntax_error(Lep remaining, const CsrCharString *message)
{
    printk("unifi : trace level string syntax error (%s) remaining string is (%s)", message, remaining.ptr);
}

/*
   This function parses a trace level spec at the start of the passed string.
   tracelevelspec => origin '.' suborigin '.' levelsettings

   Examples of a substring representing a single level spec.
   ".."
   "..C"
   "OS.TX.CEW"

   Returns 0 if the string is parsed without syntax error.  Returns 1 otherwise.

   It creates an abstract representation of the meaning in *level.
   It updates the passed Lep to remove the substring that it has parsed.
   If there is an error, it may have written something into *level and updated the Lep anyway.

   Does not use any static or global vars.
*/
int parse_level_spec(Levelspec *level, Lep *remaining_input)
{
    level->origin.ptr = remaining_input->ptr;
    level->origin.len = 0;

    while (remaining_input->len && is_nameworthy(*remaining_input->ptr))
    {
        level->origin.len++;
        remove_first_char(remaining_input);
    }

    if (!remaining_input->len || (*remaining_input->ptr != '.'))
    {
        flag_syntax_error(*remaining_input, "missing separator dot after origin");
        return 1;
    }
    remove_first_char(remaining_input);

    level->suborigin.ptr = remaining_input->ptr;
    level->suborigin.len = 0;
    while (remaining_input->len && is_nameworthy(*remaining_input->ptr))
    {
        level->suborigin.len++;
        remove_first_char(remaining_input);
    }

    if (!remaining_input->len || (*remaining_input->ptr != '.'))
    {
        flag_syntax_error(*remaining_input, "missing separator dot after suborigin");
        return 1;
    }
    remove_first_char(remaining_input);

    level->level_mask = get_level_token(remaining_input);
    return 0;
}

/*
   This parses the passed string, which represents a list of trace level specs.
   See parse_level_spec() for a description of the substring representing a single level spec.
   Each level spec in the list is separated either by a single colon or by an arbitrary number of spaces.
   Examples of a list that could be passed in:
   "..C:OS.TX.CEW:OS.RX.A:HIP.TX.CEWI"
   "..C OS.TX.CEW OS.RX.A HIP.TX.CEWI"
   "..C OS.TX.CEW:OS.RX.A   HIP.TX.CEWI"

   Returns 0 if the string is parsed without syntax error.  Returns 1 otherwise.

   If it returns 0, it has created an abstract representation of the meaning.
   The representation is a linked list of level specifiers in the order they were in the string. (Note order is significant).
   The passed Levelspec pointer *lpp is set to point to the head of the list.
   The list is in allocated memory (and the caller will have to free it).

   If it does not return 0, *lpp is not altered and no memory is left allocated.
   Does not use any static or global vars
*/
int parse_level_specs(const char *level_settings_string, Levelspec **lpp)
{
    Lep settings;
    Levelspec *tail = NULL; /* init is unnecessary but needed to quieten would-be clever gnu compiler */
    Levelspec *head = NULL;
    int error_flag = 0;

    settings = Str2lep(level_settings_string);
    remove_optional_leading_spaces(&settings);

    while (1)
    {
        Levelspec temp;
        Levelspec *plevel;

        if (parse_level_spec(&temp, &settings))
        {
            error_flag = 1;
            break;
        }

        /* parsed ok, so add to the end of the list of level specs */
        plevel = CsrPmemAlloc(sizeof(*plevel));

        *plevel = temp;
        plevel->next = NULL;
        if (head)
        {
            tail->next = plevel; /* add to end of the list */
        }
        else
        {
            head = plevel;
        }
        tail = plevel;

        if (settings.len)
        {
            if (':' == *settings.ptr)
            {
                remove_first_char(&settings);
            }
            else if (' ' == *settings.ptr)
            {
                remove_optional_leading_spaces(&settings);
                if (!settings.len)
                {
                    break; /* trailing spaces at end of string are allowed */
                }
            }
            else
            {
                flag_syntax_error(settings, "unacceptable trailing chars");
                error_flag = 1;
                break;
            }
        }
        else /* parsing is finished */
        {
            break;
        }
    }

    if (error_flag)
    {
        free_level_spec_list(head);
    }
    else
    {
        *lpp = head;
    }
    return error_flag;
}

static const char *const default_trace_levels = "..CE";
static Levelspec *levels_p = NULL;
static CsrLogTextHandle *handle_list = NULL;

void deinit_log_text(void)
{
    free_level_spec_list(levels_p);
    levels_p = NULL;
    while (handle_list)
    {
        CsrLogTextHandle *handle = handle_list;

        handle_list = handle_list->next;
        CsrPmemFree(handle->level_masks);
        CsrPmemFree(handle);
    }
    printk("unifi : debug deinit_log_text\n");
}

void init_log_text(const char *trace_levels, int unifi_debug)
{
    static const CsrCharString *subOrigins[] = CSR_WIFI_HIP_LOG_LEGACY_SUBORIGINS;

    printk("unifi : debug debug debug trace_levels =(%s), unifi_debug=%d\n", trace_levels, unifi_debug);
    /* Use of unifi_debug is deprecated and is for support of transition away from legacy messages only.
       After this, this whole "if" construct can be deleted
       Meanwhile this provides equivalent behaviour to the legacy behaviour for those messages that
       remain in the legacy origin "LEG" - that is legacy handle CSR_WIFI_HIP_LOG_ID.
       If the user sets unifi_debug to a value greater than 0 then a corresponding
       list of level specifications is generated.  This is prepended to the list generated from the specifications
       that the user may supply in  the module parameter "trace_levels"
     */
    if (unifi_debug)
    {
        Levelspec *preload = NULL;
        const char *preload_spec;

        switch (unifi_debug)
        {
            case 1:
                preload_spec = "LEG.DEFAULT.A LEG.UDBG1.A";
                break;
            case 2:
                preload_spec = "LEG.DEFAULT.A LEG.UDBG1.A LEG.UDBG2.A";
                break;
            case 3:
                preload_spec = "LEG.DEFAULT.A LEG.UDBG1.A LEG.UDBG2.A LEG.UDBG3.A";
                break;
            case 4:
                preload_spec = "LEG.DEFAULT.A LEG.UDBG1.A LEG.UDBG2.A LEG.UDBG3.A LEG.UDBG4.A";
                break;
            case 5:
                preload_spec = "LEG.DEFAULT.A LEG.UDBG1.A LEG.UDBG2.A LEG.UDBG3.A LEG.UDBG4.A LEG.UDBG5.A";
                break;
            case 6:
                preload_spec = "LEG.DEFAULT.A LEG.UDBG1.A LEG.UDBG2.A LEG.UDBG3.A LEG.UDBG4.A LEG.UDBG5.A LEG.UDBG6.A";
                break;
            default:
                preload_spec = "LEG.DEFAULT.A LEG.UDBG1.A LEG.UDBG2.A LEG.UDBG3.A LEG.UDBG4.A LEG.UDBG5.A LEG.UDBG6.A LEG.UDBG7.A";
                break;
        }
        if (parse_level_specs(preload_spec, &preload))
        {
            /* This cannot happen if the above preload strings are sensible. If it does it must be immediately fixed */
            printk(KERN_WARNING "unifi : trace internal error cannot parse own level specs (%s)", preload_spec);
        }
        else if (!*trace_levels || parse_level_specs(trace_levels, &levels_p))
        {
            printk(KERN_WARNING "unifi : using only default trace levels implied by unifi_debug setting");
            levels_p = preload;
        }
        else if (preload) /* insert preload list at the front of the user specified list */
        {
            Levelspec *tail = preload->next;

            while (tail->next)
            {
                tail = tail->next;
            }
            tail->next = levels_p;
            levels_p = preload;
        }
    }
    else if (!*trace_levels || parse_level_specs(trace_levels, &levels_p))
    {
        printk(KERN_WARNING "unifi : using default trace levels (%s)", default_trace_levels);
        (void) parse_level_specs(default_trace_levels, &levels_p);
    }
    /* Immediately register the legacy origin.  Use of this origin is deprecated and it will be removed.
       Immediate registration here in linux means that all subsequent legacy messages in the linux component will work.
       HAL will later also attempt to register this handle but the attempt will be ignored because
       CSR_WIFI_HIP_LOG_ID will already not be NULL.
       (The HAL registration will of course work in the non-linux case.)
    */
    CsrLogTextRegister2(&CSR_WIFI_HIP_LOG_ID, "LEG", CSR_ARRAY_SIZE(subOrigins), subOrigins);
}

void CsrLogTextRegister2(CsrLogTextHandle **pphandle, const CsrCharString *originName, CsrUint16 subOriginsCount, const CsrCharString *subOrigins[])
{
    if (NULL == *pphandle)
    {
        CsrLogTextHandle *handle = CsrPmemAlloc(sizeof(*handle));

        handle->origin = originName;
        handle->suborigin_n = subOriginsCount;
        handle->suborigins = subOrigins;
        handle->level_masks = CsrPmemZalloc(sizeof(*handle->level_masks) * subOriginsCount);
        apply_settings_to_handle(handle, levels_p);
        *pphandle = handle;
        /* now add to head of handle list */
        handle->next = handle_list;
        handle_list = handle;
    }
}

#ifdef ANDROID_TIMESTAMP
#define PRINT_FORMAT_STRING "%s : ", print_time()
#else
#define PRINT_FORMAT_STRING
#endif /* ANDROID_TIMESTAMP */

#define LOG_BUFFER_SIZE 120

void CsrLogTextCritical2(CsrLogTextHandle *handle, CsrUint16 subOrigin, const CsrCharString *formatString, ...)
{
    subOrigin--;
    if (handle && (subOrigin < handle->suborigin_n) && handle->level_masks[subOrigin] & trace_critical_mask)
    {
        char s[LOG_BUFFER_SIZE];
        va_list args;
        unsigned int len;

        va_start(args, formatString);
        len = vsnprintf(s, sizeof(s), formatString, args);
        va_end(args);
        if (len >= sizeof(s))
        {
            s[sizeof(s) - 1] = 0;
        }
        printk(KERN_ERR "%s", s);
    }
}

void CsrLogTextError2(CsrLogTextHandle *handle, CsrUint16 subOrigin, const CsrCharString *formatString, ...)
{
    subOrigin--;
    if (handle && (subOrigin < handle->suborigin_n) && handle->level_masks[subOrigin] & trace_error_mask)
    {
        char s[LOG_BUFFER_SIZE];
        va_list args;
        unsigned int len;

        va_start(args, formatString);
        len = vsnprintf(s, sizeof(s), formatString, args);
        va_end(args);
        if (len >= sizeof(s))
        {
            s[sizeof(s) - 1] = 0;
        }
        printk(KERN_ERR "%s", s);
    }
}

void CsrLogTextWarning2(CsrLogTextHandle *handle, CsrUint16 subOrigin, const CsrCharString *formatString, ...)
{
    subOrigin--;
    if (handle && (subOrigin < handle->suborigin_n) && handle->level_masks[subOrigin] & trace_warn_mask)
    {
        char s[LOG_BUFFER_SIZE];
        va_list args;
        unsigned int len;

        va_start(args, formatString);
        len = vsnprintf(s, sizeof(s), formatString, args);
        va_end(args);
        if (len >= sizeof(s))
        {
            s[sizeof(s) - 1] = 0;
        }
        printk(KERN_WARNING "%s", s);
    }
}

void CsrLogTextInfo2(CsrLogTextHandle *handle, CsrUint16 subOrigin, const CsrCharString *formatString, ...)
{
    subOrigin--;
    if (handle && (subOrigin < handle->suborigin_n) && handle->level_masks[subOrigin] & trace_info_mask)
    {
        char s[LOG_BUFFER_SIZE];
        va_list args;
        unsigned int len;

        va_start(args, formatString);
        len = vsnprintf(s, sizeof(s), formatString, args);
        va_end(args);
        if (len >= sizeof(s))
        {
            s[sizeof(s) - 1] = 0;
        }
        printk(KERN_NOTICE "%s", s);
    }
}

void CsrLogTextDebug2(CsrLogTextHandle *handle, CsrUint16 subOrigin, const CsrCharString *formatString, ...)
{
    subOrigin--;
    if (handle && (subOrigin < handle->suborigin_n) && handle->level_masks[subOrigin] & trace_debug_mask)
    {
        char s[LOG_BUFFER_SIZE];
        va_list args;
        unsigned int len;

        va_start(args, formatString);
        len = vsnprintf(s, sizeof(s), formatString, args);
        va_end(args);
        if (len >= sizeof(s))
        {
            s[sizeof(s) - 1] = 0;
        }
        printk(KERN_INFO "%s", s);
    }
}

static const CsrCharString *hexstring = "0123456789ABCDEF";

void CsrLogTextBufferCritical2(CsrLogTextHandle *handle, CsrUint16 subOrigin, CsrSize bufferLength, const void *buffer, const CsrCharString *formatString, ...)
{
    subOrigin--;
    if (handle && (subOrigin < handle->suborigin_n) && handle->level_masks[subOrigin] & trace_critical_mask)
    {
        char s[LOG_BUFFER_SIZE];
        char *p;
        va_list args;
        int format_len;
        int max_per_line; /* maximum bytes from the passed buffer that may be printed on a line */
        int bytes_on_this_line; /* bytes from passed buffer printed on current line so far */

        va_start(args, formatString);
        format_len = vsnprintf(s, sizeof(s), formatString, args);
        va_end(args);

        max_per_line = ((signed) sizeof(s) - 1 - format_len) / 2;
        if (max_per_line < 1)
        {
            s[sizeof(s) - 1] = 0;
            printk(KERN_ERR "%s", s);
            return;
        }
        if (max_per_line > 16)
        {
            max_per_line = 16;
        }

        p = s + format_len;
        bytes_on_this_line = 0;
        while (bufferLength--)
        {
            *p++ = hexstring[(*(CsrCharString *) buffer >> 4) & 0x0f];
            *p++ = hexstring[*(CsrCharString *) buffer++ & 0x0f];
            bytes_on_this_line++;
            if (bytes_on_this_line >= max_per_line)
            {
                *p = '\0';
                printk(KERN_ERR "%s", s);
                p = s + format_len;
                bytes_on_this_line = 0;
            }
        }
        if (bytes_on_this_line)
        {
            *p = '\0';
            printk(KERN_ERR "%s", s);
        }
    }
}

void CsrLogTextBufferError2(CsrLogTextHandle *handle, CsrUint16 subOrigin, CsrSize bufferLength, const void *buffer, const CsrCharString *formatString, ...)
{
    subOrigin--;
    if (handle && (subOrigin < handle->suborigin_n) && handle->level_masks[subOrigin] & trace_error_mask)
    {
        char s[LOG_BUFFER_SIZE];
        char *p;
        va_list args;
        int format_len;
        int max_per_line; /* maximum bytes from the passed buffer that may be printed on a line */
        int bytes_on_this_line; /* bytes from passed buffer printed on current line so far */

        va_start(args, formatString);
        format_len = vsnprintf(s, sizeof(s), formatString, args);
        va_end(args);

        max_per_line = ((signed) sizeof(s) - 1 - format_len) / 2;
        if (max_per_line < 1)
        {
            s[sizeof(s) - 1] = 0;
            printk(KERN_ERR "%s", s);
            return;
        }
        if (max_per_line > 16)
        {
            max_per_line = 16;
        }

        p = s + format_len;
        bytes_on_this_line = 0;
        while (bufferLength--)
        {
            *p++ = hexstring[(*(CsrCharString *) buffer >> 4) & 0x0f];
            *p++ = hexstring[*(CsrCharString *) buffer++ & 0x0f];
            bytes_on_this_line++;
            if (bytes_on_this_line >= max_per_line)
            {
                *p = '\0';
                printk(KERN_ERR "%s", s);
                p = s + format_len;
                bytes_on_this_line = 0;
            }
        }
        if (bytes_on_this_line)
        {
            *p = '\0';
            printk(KERN_ERR "%s", s);
        }
    }
}

void CsrLogTextBufferWarning2(CsrLogTextHandle *handle, CsrUint16 subOrigin, CsrSize bufferLength, const void *buffer, const CsrCharString *formatString, ...)
{
    subOrigin--;
    if (handle && (subOrigin < handle->suborigin_n) && handle->level_masks[subOrigin] & trace_warn_mask)
    {
        char s[LOG_BUFFER_SIZE];
        char *p;
        va_list args;
        int format_len;
        int max_per_line; /* maximum bytes from the passed buffer that may be printed on a line */
        int bytes_on_this_line; /* bytes from passed buffer printed on current line so far */

        va_start(args, formatString);
        format_len = vsnprintf(s, sizeof(s), formatString, args);
        va_end(args);

        max_per_line = ((signed) sizeof(s) - 1 - format_len) / 2;
        if (max_per_line < 1)
        {
            s[sizeof(s) - 1] = 0;
            printk(KERN_WARNING "%s", s);
            return;
        }
        if (max_per_line > 16)
        {
            max_per_line = 16;
        }

        p = s + format_len;
        bytes_on_this_line = 0;
        while (bufferLength--)
        {
            *p++ = hexstring[(*(CsrCharString *) buffer >> 4) & 0x0f];
            *p++ = hexstring[*(CsrCharString *) buffer++ & 0x0f];
            bytes_on_this_line++;
            if (bytes_on_this_line >= max_per_line)
            {
                *p = '\0';
                printk(KERN_WARNING "%s", s);
                p = s + format_len;
                bytes_on_this_line = 0;
            }
        }
        if (bytes_on_this_line)
        {
            *p = '\0';
            printk(KERN_WARNING "%s", s);
        }
    }
}

void CsrLogTextBufferInfo2(CsrLogTextHandle *handle, CsrUint16 subOrigin, CsrSize bufferLength, const void *buffer, const CsrCharString *formatString, ...)
{
    subOrigin--;
    if (handle && (subOrigin < handle->suborigin_n) && handle->level_masks[subOrigin] & trace_info_mask)
    {
        char s[LOG_BUFFER_SIZE];
        char *p;
        va_list args;
        int format_len;
        int max_per_line; /* maximum bytes from the passed buffer that may be printed on a line */
        int bytes_on_this_line; /* bytes from passed buffer printed on current line so far */

        va_start(args, formatString);
        format_len = vsnprintf(s, sizeof(s), formatString, args);
        va_end(args);

        max_per_line = ((signed) sizeof(s) - 1 - format_len) / 2;
        if (max_per_line < 1)
        {
            s[sizeof(s) - 1] = 0;
            printk(KERN_NOTICE "%s", s);
            return;
        }
        if (max_per_line > 16)
        {
            max_per_line = 16;
        }

        p = s + format_len;
        bytes_on_this_line = 0;
        while (bufferLength--)
        {
            *p++ = hexstring[(*(CsrCharString *) buffer >> 4) & 0x0f];
            *p++ = hexstring[*(CsrCharString *) buffer++ & 0x0f];
            bytes_on_this_line++;
            if (bytes_on_this_line >= max_per_line)
            {
                *p = '\0';
                printk(KERN_NOTICE "%s", s);
                p = s + format_len;
                bytes_on_this_line = 0;
            }
        }
        if (bytes_on_this_line)
        {
            *p = '\0';
            printk(KERN_NOTICE "%s", s);
        }
    }
}

void CsrLogTextBufferDebug2(CsrLogTextHandle *handle, CsrUint16 subOrigin, CsrSize bufferLength, const void *buffer, const CsrCharString *formatString, ...)
{
    subOrigin--;
    if (handle && (subOrigin < handle->suborigin_n) && handle->level_masks[subOrigin] & trace_debug_mask)
    {
        char s[LOG_BUFFER_SIZE];
        char *p;
        va_list args;
        int format_len;
        int max_per_line; /* maximum bytes from the passed buffer that may be printed on a line */
        int bytes_on_this_line; /* bytes from passed buffer printed on current line so far */

        va_start(args, formatString);
        format_len = vsnprintf(s, sizeof(s), formatString, args);
        va_end(args);

        max_per_line = ((signed) sizeof(s) - 1 - format_len) / 2;
        if (max_per_line < 1)
        {
            s[sizeof(s) - 1] = 0;
            printk(KERN_INFO "%s", s);
            return;
        }
        if (max_per_line > 16)
        {
            max_per_line = 16;
        }

        p = s + format_len;
        bytes_on_this_line = 0;
        while (bufferLength--)
        {
            *p++ = hexstring[(*(CsrCharString *) buffer >> 4) & 0x0f];
            *p++ = hexstring[*(CsrCharString *) buffer++ & 0x0f];
            bytes_on_this_line++;
            if (bytes_on_this_line >= max_per_line)
            {
                *p = '\0';
                printk(KERN_INFO "%s", s);
                p = s + format_len;
                bytes_on_this_line = 0;
            }
        }
        if (bytes_on_this_line)
        {
            *p = '\0';
            printk(KERN_INFO "%s", s);
        }
    }
}

#endif /* CSR_LOG_ENABLE */
#endif /* UNIFI_DEBUG */
