/*
 * dev-fake.c --  netdevice for InterNodeCommunication
 *
 *   Copyright (c) 2012 Robert Bosch GmbH, Hildesheim
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 * based on snull from
 * Copyright (C) 2001 Alessandro Rubini and Jonathan Corbet
 * Copyright (C) 2001 O'Reilly & Associates
 *
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/moduleparam.h>

#include <linux/sched.h>
#include <linux/kernel.h> /* printk() */
#include <linux/slab.h> /* kmalloc() */
#include <linux/errno.h>  /* error codes */
#include <linux/types.h>  /* size_t */
#include <linux/interrupt.h> /* mark_bh */
#include <linux/ratelimit.h>
#include <linux/in.h>
#include <linux/netdevice.h>   /* struct device, and other headers */
#include <linux/etherdevice.h> /* eth_type_trans */
#include <linux/ip.h>          /* struct iphdr */
#include <linux/tcp.h>         /* struct tcphdr */
#include <linux/skbuff.h>
#include <linux/inc.h>

#include "dev-fake.h"

MODULE_AUTHOR("Peter Waechtler <external.Peter.Waechtler@de.bosch.com");
MODULE_LICENSE("GPL");

/*
 * Transmitter lockup simulation, normally disabled.
 */
static int lockup;
module_param(lockup, int, 0);

static int timeout = DEVINC_TIMEOUT;
module_param(timeout, int, 0);

/*
 * Do we run in NAPI mode?
 */
static int use_napi;
module_param(use_napi, int, 0);

/*
 * A structure representing an in-flight packet.
 */
struct inc_packet {
	struct inc_packet *next;
	struct net_device *dev;
	int	datalen;
	u8 data[ETH_DATA_LEN];
};

static int pool_size = 8;
module_param(pool_size, int, 0);

/*
 * This structure is private to each device. It is used to pass
 * packets in and out, so there is place for a packet
 */

struct devinc_priv {
	struct net_device_stats stats;
	int status;
	struct inc_packet *ppool;
	struct inc_packet *rx_queue;  /* List of incoming packets */
	int rx_int_enabled;
	int tx_packetlen;
	u8 *tx_packetdata;
	struct sk_buff *skb;
	spinlock_t lock;
	struct net_device *dev;
	struct napi_struct napi;
};

/*
 * The devices
 */
struct net_device *devinc_devs[2];

static void devinc_tx_timeout(struct net_device *dev);
static void (*devinc_interrupt)(int, void *, struct pt_regs *);

/*
 * Set up a device's packet pool.
 */
static void devinc_setup_pool(struct net_device *dev)
{
	struct devinc_priv *priv = netdev_priv(dev);
	int i;
	struct inc_packet *pkt;

	priv->ppool = NULL;
	for (i = 0; i < pool_size; i++) {
		pkt = kmalloc(sizeof(struct inc_packet), GFP_KERNEL);
		if (pkt == NULL) {
			printk(KERN_NOTICE "allocating packet pool failed\n");
			return;
		}
		pkt->dev = dev;
		pkt->next = priv->ppool;
		priv->ppool = pkt;
	}
}

static void devinc_teardown_pool(struct net_device *dev)
{
	struct devinc_priv *priv = netdev_priv(dev);
	struct inc_packet *pkt;

	while ((pkt = priv->ppool)) {
		priv->ppool = pkt->next;
		kfree(pkt);
		/* FIXME - in-flight packets ? */
	}
}

/*
 * Buffer/pool management.
 */
static struct inc_packet *devinc_get_tx_buffer(struct net_device *dev)
{
	struct devinc_priv *priv = netdev_priv(dev);
	unsigned long flags;
	struct inc_packet *pkt;

	spin_lock_irqsave(&priv->lock, flags);
	pkt = priv->ppool;
	priv->ppool = pkt->next;
	if (priv->ppool == NULL) {
		printk(KERN_WARNING "devinc: Pool empty\n");
		netif_stop_queue(dev);
	}
	spin_unlock_irqrestore(&priv->lock, flags);
	return pkt;
}


static void devinc_release_buffer(struct inc_packet *pkt)
{
	unsigned long flags;
	struct devinc_priv *priv = netdev_priv(pkt->dev);

	spin_lock_irqsave(&priv->lock, flags);
	pkt->next = priv->ppool;
	priv->ppool = pkt;
	spin_unlock_irqrestore(&priv->lock, flags);
	if (netif_queue_stopped(pkt->dev) && pkt->next == NULL)
		netif_wake_queue(pkt->dev);
}

static void devinc_enqueue_buf(struct net_device *dev, struct inc_packet *pkt)
{
	unsigned long flags;
	struct devinc_priv *priv = netdev_priv(dev);

	PDEBUG("devinc_enqueue_buf: %p\n", dev);

	spin_lock_irqsave(&priv->lock, flags);
	pkt->next = priv->rx_queue;  /* FIXME - misorders packets */
	priv->rx_queue = pkt;
	spin_unlock_irqrestore(&priv->lock, flags);
}

static struct inc_packet *devinc_dequeue_buf(struct net_device *dev)
{
	struct devinc_priv *priv = netdev_priv(dev);
	struct inc_packet *pkt;
	unsigned long flags;

	spin_lock_irqsave(&priv->lock, flags);
	pkt = priv->rx_queue;
	if (pkt != NULL)
		priv->rx_queue = pkt->next;
	spin_unlock_irqrestore(&priv->lock, flags);
	return pkt;
}

/*
 * Enable and disable receive interrupts.
 */
static void devinc_rx_ints(struct net_device *dev, int enable)
{
	struct devinc_priv *priv = netdev_priv(dev);
	priv->rx_int_enabled = enable;
}

/*
 * This function is called to fill up an eth header, since arp is not
 * available on the interface
 */
static int devinc_rebuild_header(struct sk_buff *skb)
{
	PDEBUG("device_rebuild_header");
	return 0;
}

static int devinc_header(struct sk_buff *skb, struct net_device *dev,
		unsigned short type, const void *daddr, const void *saddr,
		unsigned len)
{

	PDEBUG("device_header");
	return dev->hard_header_len;
}

/*
 * Open and close
 */
static int devinc_open(struct net_device *dev)
{
	/* request_region(), request_irq(), ....  (like fops->open) */

	netif_start_queue(dev);
	return 0;
}

static int devinc_release(struct net_device *dev)
{
    /* release ports, irq and such -- like fops->close */

	netif_stop_queue(dev); /* can't transmit any more */
	return 0;
}

/*
 * Configuration changes (passed on by ifconfig)
 */
static int devinc_config(struct net_device *dev, struct ifmap *map)
{
	if (dev->flags & IFF_UP) /* can't act on a running interface */
		return -EBUSY;

	/* Don't allow changing the I/O address */
	if (map->base_addr != dev->base_addr) {
		printk(KERN_WARNING "devinc: Can't change I/O address\n");
		return -EOPNOTSUPP;
	}

	/* Allow changing the IRQ */
	if (map->irq != dev->irq) {
		dev->irq = map->irq;
		/* request_irq() is delayed to open-time */
	}

	/* ignore other fields */
	return 0;
}

/*
 * Receive a packet: retrieve, encapsulate and pass over to upper levels
 */
static void devinc_rx(struct net_device *dev, struct inc_packet *pkt)
{
	struct sk_buff *skb;
	struct devinc_priv *priv = netdev_priv(dev);

	PDEBUG("%s: %s: pkt->len: %d\n", __func__, dev->name, pkt->datalen);
	/*
	 * The packet has been retrieved from the transmission
	 * medium. Build an skb around it, so upper layers can handle it
	 */
	skb = dev_alloc_skb(pkt->datalen + 2);
	if (!skb) {
		printk_ratelimited(KERN_NOTICE "inc rx: packet dropped\n");
		priv->stats.rx_dropped++;
		goto out;
	}
	skb_reserve(skb, 2); /* align IP on 16B boundary */
	memcpy(skb_put(skb, pkt->datalen), pkt->data, pkt->datalen);

	/* Write metadata, and then pass to the receive level */
	skb->dev = dev;
	skb->protocol = htons(ETH_P_INC);

	skb->pkt_type = PACKET_HOST;

	skb->ip_summed = CHECKSUM_UNNECESSARY;
	priv->stats.rx_packets++;
	priv->stats.rx_bytes += pkt->datalen;
	PDEBUG("%s: %s: skb->protocol: 0x%X; len:%d\n", __func__,
		dev->name, ntohs(skb->protocol), skb->len);
	if (0) { /* enable this conditional to look at the data */
		int i;
		PDEBUG("len is %i\n" KERN_DEBUG "data:", skb->len);
		for (i = 0 ; i < skb->len; i++)
			printk(" %02x", skb->data[i]&0xff);
			/* no loglevel to prevent line break */
		printk(KERN_DEBUG "\n");
	}
	netif_rx(skb);
out:
	return;
}


/*
 * The poll implementation.
 */
static int devinc_poll(struct napi_struct *napi, int budget)
{
	int npackets = 0;
	struct sk_buff *skb;
	struct devinc_priv *priv = container_of(napi,
					struct devinc_priv, napi);
	struct net_device *dev = priv->dev;
	struct inc_packet *pkt;

	while (npackets < budget && priv->rx_queue) {
		pkt = devinc_dequeue_buf(dev);
		skb = dev_alloc_skb(pkt->datalen + 2);
		if (!skb) {
			printk_ratelimited(KERN_NOTICE "inc: packet dropped\n");
			priv->stats.rx_dropped++;
			devinc_release_buffer(pkt);
			continue;
		}
		skb_reserve(skb, 2); /* align IP on 16B boundary */
		memcpy(skb_put(skb, pkt->datalen), pkt->data, pkt->datalen);
		skb->dev = dev;
		skb->protocol = htons(ETH_P_INC);
		skb->ip_summed = CHECKSUM_UNNECESSARY; /* don't check it */
		netif_receive_skb(skb);

		/* Maintain stats */
		npackets++;
		priv->stats.rx_packets++;
		priv->stats.rx_bytes += pkt->datalen;
		devinc_release_buffer(pkt);
	}
	/** If we processed all packets, we're done;
	 * tell the kernel and reenable ints */
	if (!priv->rx_queue) {
		napi_complete(napi);
		devinc_rx_ints(dev, 1);
		return 0;
	}
	/* We couldn't process everything. */
	return npackets;
}


/*
 * The typical interrupt entry point
 */
static void devinc_regular_interrupt(int irq, void *dev_id,
	struct pt_regs *regs)
{
	int statusword;
	struct devinc_priv *priv;
	struct inc_packet *pkt = NULL;
	/*
	 * As usual, check the "device" pointer to be sure it is
	 * really interrupting.
	 * Then assign "struct device *dev"
	 */
	struct net_device *dev = (struct net_device *)dev_id;
	/* ... and check with hw if it's really ours */

	/* paranoid */
	if (!dev)
		return;

	/* Lock the device */
	priv = netdev_priv(dev);
	spin_lock(&priv->lock);

	/* retrieve statusword: real netdevices use I/O instructions */
	statusword = priv->status;
	priv->status = 0;
	if (statusword & DEVINC_RX_INTR) {
		/* send it to devinc_rx for handling */
		pkt = priv->rx_queue;
		if (pkt) {
			priv->rx_queue = pkt->next;
			devinc_rx(dev, pkt);
		}
	}
	if (statusword & DEVINC_TX_INTR) {
		/* a transmission is over: free the skb */
		priv->stats.tx_packets++;
		priv->stats.tx_bytes += priv->tx_packetlen;
		dev_kfree_skb(priv->skb);
	}

	/* Unlock the device and we are done */
	spin_unlock(&priv->lock);
	if (pkt)
		devinc_release_buffer(pkt); /* Do this outside the lock! */
	return;
}

/*
 * A NAPI interrupt handler.
 */
static void devinc_napi_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	int statusword;
	struct devinc_priv *priv;

	/*
	 * As usual, check the "device" pointer for shared handlers.
	 * Then assign "struct device *dev"
	 */
	struct net_device *dev = (struct net_device *)dev_id;
	/* ... and check with hw if it's really ours */

	/* paranoid */
	if (!dev)
		return;

	/* Lock the device */
	priv = netdev_priv(dev);
	spin_lock(&priv->lock);

	/* retrieve statusword: real netdevices use I/O instructions */
	statusword = priv->status;
	priv->status = 0;
	if (statusword & DEVINC_RX_INTR) {
		devinc_rx_ints(dev, 0);  /* Disable further interrupts */
		napi_schedule(&priv->napi);
	}
	if (statusword & DEVINC_TX_INTR) {
		/* a transmission is over: free the skb */
		priv->stats.tx_packets++;
		priv->stats.tx_bytes += priv->tx_packetlen;
		dev_kfree_skb(priv->skb);
	}

	/* Unlock the device and we are done */
	spin_unlock(&priv->lock);
	return;
}

/*
 * Transmit a packet (low level interface)
 */
static void devinc_hw_tx(char *buf, int len, struct net_device *dev)
{
	/*
	 * This function deals with hw details. This interface loops
	 * back the packet to the other inc interface (if any).
	 * In other words, this function implements the inc behaviour,
	 * while all other procedures are rather device-independent
	 */
	struct inc_hdr *ih;
	struct net_device *dest;
	struct devinc_priv *priv;
	struct inc_packet *tx_buffer;

	if (0) { /* enable this conditional to look at the data */
		int i;
		PDEBUG("len is %i\n" KERN_DEBUG "data:", len);
		for (i = 0 ; i < len; i++)
			printk(" %02x", buf[i]&0xff);
			/* no loglevel to prevent line break */
		printk(KERN_DEBUG "\n");
	}
	/*
	 * Ethhdr is 14 bytes, but the kernel arranges for iphdr
	 * to be aligned (i.e., ethhdr is unaligned)
	 */
	ih = (struct inc_hdr *)(buf+dev->hard_header_len);

	if (dev == devinc_devs[0])
		PDEBUG("%08x:%05i --> %08x:%05i\n",
				ih->src_node, ih->src_lun,
				ih->dest_node, ih->dest_lun);
	else
		PDEBUG("%08x:%05i <-- %08x:%05i\n",
				ih->dest_node, ih->dest_lun,
				ih->src_node, ih->src_lun);

	/*
	 * Ok, now the packet is ready for transmission: first simulate a
	 * receive interrupt on the twin device, then  a
	 * transmission-done on the transmitting device
	 */
	dest = devinc_devs[dev == devinc_devs[0] ? 1 : 0];
	priv = netdev_priv(dest);
	tx_buffer = devinc_get_tx_buffer(dev);
	tx_buffer->datalen = len;
	memcpy(tx_buffer->data, buf, len);
	devinc_enqueue_buf(dest, tx_buffer);
	if (priv->rx_int_enabled) {
		priv->status |= DEVINC_RX_INTR;
		devinc_interrupt(0, dest, NULL);
	}

	priv = netdev_priv(dev);
	priv->tx_packetlen = len;
	priv->tx_packetdata = buf;
	priv->status |= DEVINC_TX_INTR;
	if (lockup && ((priv->stats.tx_packets + 1) % lockup) == 0) {
		/* Simulate a dropped transmit interrupt */
		netif_stop_queue(dev);
		PDEBUG("Simulate lockup at %ld, txp %ld\n", jiffies,
				(unsigned long) priv->stats.tx_packets);
	} else
		devinc_interrupt(0, dev, NULL);
}

/*
 * Transmit a packet (called by the kernel)
 */
static int devinc_tx(struct sk_buff *skb, struct net_device *dev)
{
	int len;
	char *data;
	struct devinc_priv *priv = netdev_priv(dev);

	PDEBUG("devinc_tx\n");
	data = skb->data;
	len = skb->len;

	dev->trans_start = jiffies; /* save the timestamp */

	/* Remember the skb, so we can free it at interrupt time */
	priv->skb = skb;

	devinc_rebuild_header(skb);
	/* actual deliver of data is device-specific, and not shown here */
	devinc_hw_tx(data, len, dev);

	return 0; /* Our simple device can not fail */
}

/*
 * Deal with a transmit timeout.
 */
static void devinc_tx_timeout(struct net_device *dev)
{
	struct devinc_priv *priv = netdev_priv(dev);

	PDEBUG("Transmit timeout at %ld, latency %ld\n", jiffies,
			jiffies - dev->trans_start);
	/* Simulate a transmission interrupt to get things moving */
	priv->status = DEVINC_TX_INTR;
	devinc_interrupt(0, dev, NULL);
	priv->stats.tx_errors++;
	netif_wake_queue(dev);
	return;
}



/*
 * Ioctl commands
 */
static int devinc_ioctl(struct net_device *dev, struct ifreq *rq, int cmd)
{
	PDEBUG("ioctl\n");
	return 0;
}

/*
 * Return statistics to the caller
 */
static struct net_device_stats *devinc_stats(struct net_device *dev)
{
	struct devinc_priv *priv = netdev_priv(dev);
	return &priv->stats;
}

/*
 * The "change_mtu" method is usually not needed.
 * If you need it, it must be like this.
 */
static int devinc_change_mtu(struct net_device *dev, int new_mtu)
{
	unsigned long flags;
	struct devinc_priv *priv = netdev_priv(dev);
	spinlock_t *lock = &priv->lock;

	/* check ranges */
	if ((new_mtu < 68) || (new_mtu > 1500))
		return -EINVAL;
	/*
	 * Do anything you need, and the accept the value
	 */
	spin_lock_irqsave(lock, flags);
	dev->mtu = new_mtu;
	spin_unlock_irqrestore(lock, flags);
	return 0; /* success */
}

static const struct header_ops devinc_header_ops = {
	.create  = devinc_header,
	.rebuild = devinc_rebuild_header
};

static const struct net_device_ops devinc_netdev_ops = {
	.ndo_open            = devinc_open,
	.ndo_stop            = devinc_release,
	.ndo_start_xmit      = devinc_tx,
	.ndo_do_ioctl        = devinc_ioctl,
	.ndo_set_config      = devinc_config,
	.ndo_get_stats       = devinc_stats,
	.ndo_change_mtu      = devinc_change_mtu,
	.ndo_tx_timeout      = devinc_tx_timeout
};

/*
 * The init function (sometimes called probe).
 * It is invoked by register_netdev()
 */
static void devinc_init(struct net_device *dev)
{
	struct devinc_priv *priv;
#if 0
	/*
	 * Make the usual checks: check_region(), probe irq, ...  -ENODEV
	 * should be returned if no device found.  No resource should be
	 * grabbed: this is done on open().
	 */
#endif

	/*
	 * Then, assign other fields in dev, some
	 * hand assignments
	 */
	dev->type		= ETH_P_INC;
	dev->hard_header_len	= 0;
	dev->mtu		= ETH_DATA_LEN;
	dev->addr_len		= 2;
	dev->tx_queue_len	= 10;	/* Ethernet: 1000 */
	dev->flags		= 0;


	dev->watchdog_timeo = timeout;
	dev->netdev_ops = &devinc_netdev_ops;
	dev->header_ops = &devinc_header_ops;
	/* keep the default flags, just add NOARP */
	dev->flags           |= IFF_NOARP;

	/*
	 * Then, initialize the priv field. This encloses the statistics
	 * and a few private fields.
	 */
	priv = netdev_priv(dev);
	memset(priv, 0, sizeof(struct devinc_priv));
	if (use_napi)
		netif_napi_add(dev, &priv->napi, devinc_poll, 2);

	spin_lock_init(&priv->lock);
	devinc_rx_ints(dev, 1);		/* enable receive interrupts */
	devinc_setup_pool(dev);
}

/*
 * Finally, the module stuff
 */
static void devinc_cleanup(void)
{
	int i;

	for (i = 0; i < 2;  i++) {
		if (devinc_devs[i]) {
			unregister_netdev(devinc_devs[i]);
			devinc_teardown_pool(devinc_devs[i]);
			free_netdev(devinc_devs[i]);
		}
	}
	return;
}

static int devinc_init_module(void)
{
	int result, i, ret = -ENOMEM;

	devinc_interrupt = use_napi ?
		devinc_napi_interrupt : devinc_regular_interrupt;

	/* Allocate the devices */
	devinc_devs[0] = alloc_netdev(sizeof(struct devinc_priv),
		"inc-fake%d", devinc_init);
	devinc_devs[1] = alloc_netdev(sizeof(struct devinc_priv),
		"inc-fake%d", devinc_init);
	if (devinc_devs[0] == NULL || devinc_devs[1] == NULL)
		goto out;

	ret = 0;
	for (i = 0; i < 2;  i++) {
		result = register_netdev(devinc_devs[i]);
		if (result) {
			printk(KERN_ERR
				"inc: error %i registering device \"%s\"\n",
				result, devinc_devs[i]->name);
			ret = result;
		}
	}
out:
	if (ret)
		devinc_cleanup();
	return ret;
}


module_init(devinc_init_module);
module_exit(devinc_cleanup);
