/*
 * Copyright (C) 2012 Mentor Graphics Inc.
 * Copyright (C) 2005-2009 Freescale Semiconductor, Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */
#include <linux/export.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/io.h>
#include <linux/err.h>
#include <linux/platform_device.h>
#include <uapi/linux/v4l2-mediabus.h>
#include <linux/clk.h>
#include <linux/clk-provider.h>
#include <linux/clkdev.h>

#include "ipu-prv.h"

struct ipu_csi {
	void __iomem *base;
	int id;
	u32 module;
	struct clk *clk_ipu;    /* IPU bus clock */
	spinlock_t lock;
	int use_count;
	struct ipu_soc *ipu;
};

/* CSI Register Offsets */
#define CSI_SENS_CONF           0x0000
#define CSI_SENS_FRM_SIZE       0x0004
#define CSI_ACT_FRM_SIZE        0x0008
#define CSI_OUT_FRM_CTRL        0x000C
#define CSI_TST_CTRL            0x0010
#define CSI_CCIR_CODE_1         0x0014
#define CSI_CCIR_CODE_2         0x0018
#define CSI_CCIR_CODE_3         0x001C
#define CSI_MIPI_DI             0x0020
#define CSI_SKIP                0x0024
#define CSI_CPD_CTRL            0x0028
#define CSI_CPD_RC(n)           (0x002C + ((n)*4))
#define CSI_CPD_RS(n)           (0x004C + ((n)*4))
#define CSI_CPD_GRC(n)          (0x005C + ((n)*4))
#define CSI_CPD_GRS(n)          (0x007C + ((n)*4))
#define CSI_CPD_GBC(n)          (0x008C + ((n)*4))
#define CSI_CPD_GBS(n)          (0x00AC + ((n)*4))
#define CSI_CPD_BC(n)           (0x00BC + ((n)*4))
#define CSI_CPD_BS(n)           (0x00DC + ((n)*4))
#define CSI_CPD_OFFSET1         0x00EC
#define CSI_CPD_OFFSET2         0x00F0

/* CSI Register Fields */
#define CSI_SENS_CONF_DATA_FMT_SHIFT       8
#define CSI_SENS_CONF_DATA_FMT_MASK        0x00000700
#define CSI_SENS_CONF_DATA_FMT_RGB_YUV444  0L
#define CSI_SENS_CONF_DATA_FMT_YUV422_YUYV 1L
#define CSI_SENS_CONF_DATA_FMT_YUV422_UYVY 2L
#define CSI_SENS_CONF_DATA_FMT_BAYER       3L
#define CSI_SENS_CONF_DATA_FMT_RGB565      4L
#define CSI_SENS_CONF_DATA_FMT_RGB555      5L
#define CSI_SENS_CONF_DATA_FMT_RGB444      6L
#define CSI_SENS_CONF_DATA_FMT_JPEG        7L

#define CSI_SENS_CONF_VSYNC_POL_SHIFT      0
#define CSI_SENS_CONF_HSYNC_POL_SHIFT      1
#define CSI_SENS_CONF_DATA_POL_SHIFT       2
#define CSI_SENS_CONF_PIX_CLK_POL_SHIFT    3
#define CSI_SENS_CONF_SENS_PRTCL_MASK      0x00000070L
#define CSI_SENS_CONF_SENS_PRTCL_SHIFT     4
#define CSI_SENS_CONF_PACK_TIGHT_SHIFT     7
#define CSI_SENS_CONF_DATA_WIDTH_SHIFT     11
#define CSI_SENS_CONF_EXT_VSYNC_SHIFT      15
#define CSI_SENS_CONF_DIVRATIO_SHIFT       16

#define CSI_SENS_CONF_DIVRATIO_MASK        0x00FF0000L
#define CSI_SENS_CONF_DATA_DEST_SHIFT      24
#define CSI_SENS_CONF_DATA_DEST_MASK       0x07000000L
#define CSI_SENS_CONF_JPEG8_EN_SHIFT       27
#define CSI_SENS_CONF_JPEG_EN_SHIFT        28
#define CSI_SENS_CONF_FORCE_EOF_SHIFT      29
#define CSI_SENS_CONF_DATA_EN_POL_SHIFT    31

#define CSI_DATA_DEST_ISP                  1L
#define CSI_DATA_DEST_IC                   2L
#define CSI_DATA_DEST_IDMAC                4L

#define CSI_CCIR_ERR_DET_EN                0x01000000L
#define CSI_HORI_DOWNSIZE_EN               0x80000000L
#define CSI_VERT_DOWNSIZE_EN               0x40000000L
#define CSI_TEST_GEN_MODE_EN               0x01000000L

#define CSI_HSC_MASK                       0x1FFF0000
#define CSI_HSC_SHIFT                      16
#define CSI_VSC_MASK                       0x00000FFF
#define CSI_VSC_SHIFT                      0

#define CSI_TEST_GEN_R_MASK                0x000000FFL
#define CSI_TEST_GEN_R_SHIFT               0
#define CSI_TEST_GEN_G_MASK                0x0000FF00L
#define CSI_TEST_GEN_G_SHIFT               8
#define CSI_TEST_GEN_B_MASK                0x00FF0000L
#define CSI_TEST_GEN_B_SHIFT               16

#define CSI_MAX_RATIO_SKIP_ISP_MASK        0x00070000L
#define CSI_MAX_RATIO_SKIP_ISP_SHIFT       16
#define CSI_SKIP_ISP_MASK                  0x00F80000L
#define CSI_SKIP_ISP_SHIFT                 19
#define CSI_MAX_RATIO_SKIP_SMFC_MASK       0x00000007L
#define CSI_MAX_RATIO_SKIP_SMFC_SHIFT      0
#define CSI_SKIP_SMFC_MASK                 0x000000F8L
#define CSI_SKIP_SMFC_SHIFT                3
#define CSI_ID_2_SKIP_MASK                 0x00000300L
#define CSI_ID_2_SKIP_SHIFT                8

#define CSI_COLOR_FIRST_ROW_MASK           0x00000002L
#define CSI_COLOR_FIRST_COMP_MASK          0x00000001L

/* MIPI CSI-2 data types */
#define MIPI_DT_YUV420		0x18 /* YYY.../UYVY.... */
#define MIPI_DT_YUV420_LEGACY	0x1a /* UYY.../VYY...   */
#define MIPI_DT_YUV422		0x1e /* UYVY...		*/
#define MIPI_DT_RGB444		0x20
#define MIPI_DT_RGB555		0x21
#define MIPI_DT_RGB565		0x22
#define MIPI_DT_RGB666		0x23
#define MIPI_DT_RGB888		0x24
#define MIPI_DT_RAW6		0x28
#define MIPI_DT_RAW7		0x29
#define MIPI_DT_RAW8		0x2a
#define MIPI_DT_RAW10		0x2b
#define MIPI_DT_RAW12		0x2c
#define MIPI_DT_RAW14		0x2d


static inline u32 ipu_csi_read(struct ipu_csi *csi, unsigned offset)
{
	return readl(csi->base + offset);
}

static inline void ipu_csi_write(struct ipu_csi *csi, u32 value,
				 unsigned offset)
{
	writel(value, csi->base + offset);
}

/*
 * Set mclk division ratio for generating test mode mclk. Only used
 * for test generator.
 */
static int ipu_csi_set_testgen_mclk(struct ipu_csi *csi, u32 pixel_clk,
				    u32 ipu_clk)
{
	u32 temp;
	u32 div_ratio;

	div_ratio = (ipu_clk / pixel_clk) - 1;

	if (div_ratio > 0xFF || div_ratio < 0) {
		dev_err(csi->ipu->dev,
			"value of pixel_clk extends normal range\n");
		return -EINVAL;
	}

	temp = ipu_csi_read(csi, CSI_SENS_CONF);
	temp &= ~CSI_SENS_CONF_DIVRATIO_MASK;
	ipu_csi_write(csi, temp | (div_ratio << CSI_SENS_CONF_DIVRATIO_SHIFT),
		      CSI_SENS_CONF);

	return 0;
}

/*
 * ipu_csi_ccir_err_detection_en
 *	Enable error detection and correction for
 *	CCIR interlaced mode with protection bit.
 */
static void ipu_csi_ccir_err_detection_enable(struct ipu_csi *csi)
{
	u32 temp;

	temp = ipu_csi_read(csi, CSI_CCIR_CODE_1);
	temp |= CSI_CCIR_ERR_DET_EN;
	ipu_csi_write(csi, temp, CSI_CCIR_CODE_1);

}

/*
 * ipu_csi_ccir_err_detection_disable
 *	Disable error detection and correction for
 *	CCIR interlaced mode with protection bit.
 */
static void ipu_csi_ccir_err_detection_disable(struct ipu_csi *csi)
{
	u32 temp;

	temp = ipu_csi_read(csi, CSI_CCIR_CODE_1);
	temp &= ~CSI_CCIR_ERR_DET_EN;
	ipu_csi_write(csi, temp, CSI_CCIR_CODE_1);

}

#define GPR1_IPU_CSI_MUX_CTL_SHIFT 19
#define GPR13_IPUV3HDL_CSI_MUX_CTL_SHIFT 0

int ipu_csi_set_src(struct ipu_csi *csi, u32 vc, bool select_mipi_csi2)
{
	struct ipu_soc *ipu = csi->ipu;
	int ipu_id = ipu_get_num(ipu);
	u32 val, bit;

	/*
	 * Set the muxes that choose between mipi-csi2 and parallel inputs
	 * to the CSI's.
	 */
	if (csi->ipu->ipu_type == IPUV3HDL) {
		/*
		 * On D/L, the mux is in GPR13. The TRM is unclear,
		 * but it appears the mux allows selecting the MIPI
		 * CSI-2 virtual channel number to route to the CSIs.
		 */
		bit = GPR13_IPUV3HDL_CSI_MUX_CTL_SHIFT + csi->id * 3;
		val = select_mipi_csi2 ? vc << bit : 4 << bit;
		regmap_update_bits(ipu->gp_reg, IOMUXC_GPR13,
				   0x7 << bit, val);
	} else if (csi->ipu->ipu_type == IPUV3H) {
		/*
		 * For Q/D, the mux only exists on IPU0-CSI0 and IPU1-CSI1,
		 * and the routed virtual channel numbers are fixed at 0 and
		 * 3 respectively.
		 */
		if ((ipu_id == 0 && csi->id == 0) ||
		    (ipu_id == 1 && csi->id == 1)) {
			bit = GPR1_IPU_CSI_MUX_CTL_SHIFT + csi->ipu->id;
			val = select_mipi_csi2 ? 0 : 1 << bit;
			regmap_update_bits(ipu->gp_reg, IOMUXC_GPR1,
					   0x1 << bit, val);
		}
	} else {
		dev_err(csi->ipu->dev,
			"ERROR: %s: unsupported CPU version!\n",
			__func__);
		return -EINVAL;
	}

	/*
	 * there is another mux in the IPU config register that
	 * must be set as well
	 */
	ipu_set_csi_src_mux(ipu, csi->id, select_mipi_csi2);

	return 0;
}
EXPORT_SYMBOL_GPL(ipu_csi_set_src);

/*
 * ipu_csi_init_interface
 *	Sets initial values for the CSI registers.
 *	The width and height of the sensor and the actual frame size will be
 *	set to the same values.
 *
 * @param	width		Sensor width
 * @param       height		Sensor height
 * @param       cfg	        ipu_csi_signal_cfg structure
 *
 * @return      0 for success, -EINVAL for error
 */
int ipu_csi_init_interface(struct ipu_csi *csi, u16 width, u16 height,
			   struct ipu_csi_signal_cfg *cfg)
{
	unsigned long flags;
	u32 data = 0;
	u32 ipu_clk;

	/* Set the CSI_SENS_CONF register remaining fields */
	data |= cfg->data_width << CSI_SENS_CONF_DATA_WIDTH_SHIFT |
		cfg->data_fmt << CSI_SENS_CONF_DATA_FMT_SHIFT |
		cfg->data_pol << CSI_SENS_CONF_DATA_POL_SHIFT |
		cfg->vsync_pol << CSI_SENS_CONF_VSYNC_POL_SHIFT |
		cfg->hsync_pol << CSI_SENS_CONF_HSYNC_POL_SHIFT |
		cfg->pixclk_pol << CSI_SENS_CONF_PIX_CLK_POL_SHIFT |
		cfg->ext_vsync << CSI_SENS_CONF_EXT_VSYNC_SHIFT |
		cfg->clk_mode << CSI_SENS_CONF_SENS_PRTCL_SHIFT |
		cfg->pack_tight << CSI_SENS_CONF_PACK_TIGHT_SHIFT |
		cfg->force_eof << CSI_SENS_CONF_FORCE_EOF_SHIFT |
		cfg->data_en_pol << CSI_SENS_CONF_DATA_EN_POL_SHIFT;

	ipu_clk = clk_get_rate(csi->clk_ipu);

	spin_lock_irqsave(&csi->lock, flags);

	ipu_csi_write(csi, data, CSI_SENS_CONF);

	/* Setup sensor frame size */
	ipu_csi_write(csi, (width - 1) | (height - 1) << 16,
		      CSI_SENS_FRM_SIZE);

	/* Set CCIR registers */
	if (cfg->clk_mode == IPU_CSI_CLK_MODE_CCIR656_PROGRESSIVE) {
		ipu_csi_write(csi, 0x40030, CSI_CCIR_CODE_1);
		ipu_csi_write(csi, 0xFF0000, CSI_CCIR_CODE_3);
	} else if (cfg->clk_mode == IPU_CSI_CLK_MODE_CCIR656_INTERLACED) {
		if (width == 720 && height == 625) {
			/* PAL case */
			/*
			 * Field0BlankEnd = 0x6, Field0BlankStart = 0x2,
			 * Field0ActiveEnd = 0x4, Field0ActiveStart = 0
			 */
			ipu_csi_write(csi, 0x40596, CSI_CCIR_CODE_1);
			/*
			 * Field1BlankEnd = 0x7, Field1BlankStart = 0x3,
			 * Field1ActiveEnd = 0x5, Field1ActiveStart = 0x1
			 */
			ipu_csi_write(csi, 0xD07DF, CSI_CCIR_CODE_2);
			ipu_csi_write(csi, 0xFF0000, CSI_CCIR_CODE_3);

		} else if (width == 720 && height == 525) {
			/* NTSC case */
			/*
			 * Field0BlankEnd = 0x7, Field0BlankStart = 0x3,
			 * Field0ActiveEnd = 0x5, Field0ActiveStart = 0x1
			 */
			ipu_csi_write(csi, 0xD07DF, CSI_CCIR_CODE_1);
			/*
			 * Field1BlankEnd = 0x6, Field1BlankStart = 0x2,
			 * Field1ActiveEnd = 0x4, Field1ActiveStart = 0
			 */
			ipu_csi_write(csi, 0x405A6, CSI_CCIR_CODE_2);
			ipu_csi_write(csi, 0xFF0000, CSI_CCIR_CODE_3);
		} else {
			dev_err(csi->ipu->dev,
					"Unsupported CCIR656 interlaced " \
					"video mode\n");
			spin_unlock_irqrestore(&csi->lock, flags);
			return -EINVAL;
		}
		ipu_csi_ccir_err_detection_enable(csi);
	} else if ((cfg->clk_mode ==
			IPU_CSI_CLK_MODE_CCIR1120_PROGRESSIVE_DDR) ||
		(cfg->clk_mode ==
			IPU_CSI_CLK_MODE_CCIR1120_PROGRESSIVE_SDR) ||
		(cfg->clk_mode ==
			IPU_CSI_CLK_MODE_CCIR1120_INTERLACED_DDR) ||
		(cfg->clk_mode ==
			IPU_CSI_CLK_MODE_CCIR1120_INTERLACED_SDR)) {
		ipu_csi_write(csi, 0x40030, CSI_CCIR_CODE_1);
		ipu_csi_write(csi, 0xFF0000, CSI_CCIR_CODE_3);
		ipu_csi_ccir_err_detection_enable(csi);
	} else if ((cfg->clk_mode == IPU_CSI_CLK_MODE_GATED_CLK) ||
		   (cfg->clk_mode == IPU_CSI_CLK_MODE_NONGATED_CLK)) {
		ipu_csi_ccir_err_detection_disable(csi);
	}

	dev_dbg(csi->ipu->dev, "CSI_SENS_CONF = 0x%08X\n",
		ipu_csi_read(csi, CSI_SENS_CONF));
	dev_dbg(csi->ipu->dev, "CSI_ACT_FRM_SIZE = 0x%08X\n",
		ipu_csi_read(csi, CSI_ACT_FRM_SIZE));

	spin_unlock_irqrestore(&csi->lock, flags);

	return 0;
}
EXPORT_SYMBOL_GPL(ipu_csi_init_interface);

/*
 * Find the CSI data format and data width codes for the given media
 * bus pixel format code.
 */
int ipu_csi_mbus_fmt_to_sig_cfg(struct ipu_csi_signal_cfg *cfg, u32 mbus_code)
{
	switch (mbus_code) {
	case V4L2_MBUS_FMT_BGR565_2X8_BE:
	case V4L2_MBUS_FMT_BGR565_2X8_LE:
	case V4L2_MBUS_FMT_RGB565_2X8_BE:
	case V4L2_MBUS_FMT_RGB565_2X8_LE:
		cfg->data_fmt = CSI_SENS_CONF_DATA_FMT_RGB565;
		cfg->mipi_dt = MIPI_DT_RGB565;
		cfg->data_width = IPU_CSI_DATA_WIDTH_8;
		break;
	case V4L2_MBUS_FMT_RGB444_2X8_PADHI_BE:
	case V4L2_MBUS_FMT_RGB444_2X8_PADHI_LE:
		cfg->data_fmt = CSI_SENS_CONF_DATA_FMT_RGB444;
		cfg->mipi_dt = MIPI_DT_RGB444;
		cfg->data_width = IPU_CSI_DATA_WIDTH_8;
		break;
	case V4L2_MBUS_FMT_RGB555_2X8_PADHI_BE:
	case V4L2_MBUS_FMT_RGB555_2X8_PADHI_LE:
		cfg->data_fmt = CSI_SENS_CONF_DATA_FMT_RGB555;
		cfg->mipi_dt = MIPI_DT_RGB555;
		cfg->data_width = IPU_CSI_DATA_WIDTH_8;
		break;
	case V4L2_MBUS_FMT_UYVY8_2X8:
		cfg->data_fmt = CSI_SENS_CONF_DATA_FMT_YUV422_UYVY;
		cfg->mipi_dt = MIPI_DT_YUV422;
		cfg->data_width = IPU_CSI_DATA_WIDTH_8;
		break;
	case V4L2_MBUS_FMT_YUYV8_2X8:
		cfg->data_fmt = CSI_SENS_CONF_DATA_FMT_YUV422_YUYV;
		cfg->mipi_dt = MIPI_DT_YUV422;
		cfg->data_width = IPU_CSI_DATA_WIDTH_8;
		break;
	case V4L2_MBUS_FMT_UYVY8_1X16:
	case V4L2_MBUS_FMT_YUYV8_1X16:
		cfg->data_fmt = CSI_SENS_CONF_DATA_FMT_BAYER;
		cfg->mipi_dt = MIPI_DT_YUV422;
		cfg->data_width = IPU_CSI_DATA_WIDTH_16;
		break;
	case V4L2_MBUS_FMT_SBGGR8_1X8:
	case V4L2_MBUS_FMT_SGBRG8_1X8:
	case V4L2_MBUS_FMT_SGRBG8_1X8:
	case V4L2_MBUS_FMT_SRGGB8_1X8:
		cfg->data_fmt = CSI_SENS_CONF_DATA_FMT_BAYER;
		cfg->mipi_dt = MIPI_DT_RAW8;
		cfg->data_width = IPU_CSI_DATA_WIDTH_8;
		break;
	case V4L2_MBUS_FMT_SBGGR10_DPCM8_1X8:
	case V4L2_MBUS_FMT_SGBRG10_DPCM8_1X8:
	case V4L2_MBUS_FMT_SGRBG10_DPCM8_1X8:
	case V4L2_MBUS_FMT_SRGGB10_DPCM8_1X8:
	case V4L2_MBUS_FMT_SBGGR10_2X8_PADHI_BE:
	case V4L2_MBUS_FMT_SBGGR10_2X8_PADHI_LE:
	case V4L2_MBUS_FMT_SBGGR10_2X8_PADLO_BE:
	case V4L2_MBUS_FMT_SBGGR10_2X8_PADLO_LE:
		cfg->data_fmt = CSI_SENS_CONF_DATA_FMT_BAYER;
		cfg->mipi_dt = MIPI_DT_RAW10;
		cfg->data_width = IPU_CSI_DATA_WIDTH_8;
		break;
	case V4L2_MBUS_FMT_SBGGR10_1X10:
	case V4L2_MBUS_FMT_SGBRG10_1X10:
	case V4L2_MBUS_FMT_SGRBG10_1X10:
	case V4L2_MBUS_FMT_SRGGB10_1X10:
		cfg->data_fmt = CSI_SENS_CONF_DATA_FMT_BAYER;
		cfg->mipi_dt = MIPI_DT_RAW10;
		cfg->data_width = IPU_CSI_DATA_WIDTH_10;
		break;
	case V4L2_MBUS_FMT_SBGGR12_1X12:
	case V4L2_MBUS_FMT_SGBRG12_1X12:
	case V4L2_MBUS_FMT_SGRBG12_1X12:
	case V4L2_MBUS_FMT_SRGGB12_1X12:
		cfg->data_fmt = CSI_SENS_CONF_DATA_FMT_BAYER;
		cfg->mipi_dt = MIPI_DT_RAW12;
		cfg->data_width = IPU_CSI_DATA_WIDTH_12;
		break;
	case V4L2_MBUS_FMT_JPEG_1X8:
		/* TODO */
		cfg->data_fmt = CSI_SENS_CONF_DATA_FMT_JPEG;
		cfg->mipi_dt = MIPI_DT_RAW8;
		cfg->data_width = IPU_CSI_DATA_WIDTH_8;
		break;
	default:
		return -EINVAL;
	}

	return 0;
}
EXPORT_SYMBOL_GPL(ipu_csi_mbus_fmt_to_sig_cfg);

/*
 * ipu_csi_get_sensor_protocol
 *
 * @return	Returns sensor protocol
 */
int ipu_csi_get_sensor_protocol(struct ipu_csi *csi)
{
	unsigned long flags;
	int ret;

	spin_lock_irqsave(&csi->lock, flags);
	ret = (ipu_csi_read(csi, CSI_SENS_CONF) &
		CSI_SENS_CONF_SENS_PRTCL_MASK) >>
		CSI_SENS_CONF_SENS_PRTCL_SHIFT;
	spin_unlock_irqrestore(&csi->lock, flags);
	return ret;
}
EXPORT_SYMBOL_GPL(ipu_csi_get_sensor_protocol);

/*
 * ipu_csi_is_interlaced
 *
 * @return	Returns if sensor produces interlaced image
 */
bool ipu_csi_is_interlaced(struct ipu_csi *csi)
{
	int sensor_protocol = ipu_csi_get_sensor_protocol(csi);
	switch (sensor_protocol) {
	case IPU_CSI_CLK_MODE_GATED_CLK:
	case IPU_CSI_CLK_MODE_NONGATED_CLK:
	case IPU_CSI_CLK_MODE_CCIR656_PROGRESSIVE:
	case IPU_CSI_CLK_MODE_CCIR1120_PROGRESSIVE_DDR:
	case IPU_CSI_CLK_MODE_CCIR1120_PROGRESSIVE_SDR:
		return false;
		break;
	case IPU_CSI_CLK_MODE_CCIR656_INTERLACED:
	case IPU_CSI_CLK_MODE_CCIR1120_INTERLACED_DDR:
	case IPU_CSI_CLK_MODE_CCIR1120_INTERLACED_SDR:
		return true;
		break;
	default:
		dev_err(csi->ipu->dev, "CSI %d sensor protocol unsupported\n",
				csi->id);
		return false;
	}
}
EXPORT_SYMBOL_GPL(ipu_csi_is_interlaced);


/*
 * ipu_csi_get_window_size
 *
 * @param	width	pointer to window width
 * @param	height	pointer to window height
 */
void ipu_csi_get_window_size(struct ipu_csi *csi, u32 *width, u32 *height)
{
	unsigned long flags;
	u32 reg;

	spin_lock_irqsave(&csi->lock, flags);

	reg = ipu_csi_read(csi, CSI_ACT_FRM_SIZE);
	*width = (reg & 0xFFFF) + 1;
	*height = (reg >> 16 & 0xFFFF) + 1;

	spin_unlock_irqrestore(&csi->lock, flags);
}
EXPORT_SYMBOL_GPL(ipu_csi_get_window_size);

/*
 * ipu_csi_set_window_size
 *
 * @param	width	window width
 * @param       height	window height
 */
void ipu_csi_set_window_size(struct ipu_csi *csi, u32 width, u32 height)
{
	unsigned long flags;

	spin_lock_irqsave(&csi->lock, flags);

	ipu_csi_write(csi, (width - 1) | (height - 1) << 16, CSI_ACT_FRM_SIZE);

	spin_unlock_irqrestore(&csi->lock, flags);
}
EXPORT_SYMBOL_GPL(ipu_csi_set_window_size);

/*
 * ipu_csi_set_window_pos
 *
 * @param       left	window x start
 * @param       top	window y start
 */
void ipu_csi_set_window_pos(struct ipu_csi *csi, u32 left, u32 top)
{
	unsigned long flags;
	u32 temp;

	spin_lock_irqsave(&csi->lock, flags);
	temp = ipu_csi_read(csi, CSI_OUT_FRM_CTRL);
	temp &= ~(CSI_HSC_MASK | CSI_VSC_MASK);
	temp |= ((top << CSI_VSC_SHIFT) | (left << CSI_HSC_SHIFT));
	ipu_csi_write(csi, temp, CSI_OUT_FRM_CTRL);
	spin_unlock_irqrestore(&csi->lock, flags);
}
EXPORT_SYMBOL_GPL(ipu_csi_set_window_pos);

/*
 * ipu_csi_horizontal_downsize_enable
 *	Enable horizontal downsizing(decimation) by 2.
 */
void ipu_csi_horizontal_downsize_enable(struct ipu_csi *csi)
{
	unsigned long flags;
	u32 temp;

	spin_lock_irqsave(&csi->lock, flags);
	temp = ipu_csi_read(csi, CSI_OUT_FRM_CTRL);
	temp |= CSI_HORI_DOWNSIZE_EN;
	ipu_csi_write(csi, temp, CSI_OUT_FRM_CTRL);
	spin_unlock_irqrestore(&csi->lock, flags);
}
EXPORT_SYMBOL_GPL(ipu_csi_horizontal_downsize_enable);

/*
 * ipu_csi_horizontal_downsize_disable
 *	Disable horizontal downsizing(decimation) by 2.
 */
void ipu_csi_horizontal_downsize_disable(struct ipu_csi *csi)
{
	unsigned long flags;
	u32 temp;

	spin_lock_irqsave(&csi->lock, flags);
	temp = ipu_csi_read(csi, CSI_OUT_FRM_CTRL);
	temp &= ~CSI_HORI_DOWNSIZE_EN;
	ipu_csi_write(csi, temp, CSI_OUT_FRM_CTRL);
	spin_unlock_irqrestore(&csi->lock, flags);
}
EXPORT_SYMBOL_GPL(ipu_csi_horizontal_downsize_disable);

/*
 * ipu_csi_vertical_downsize_enable
 *	Enable vertical downsizing(decimation) by 2.
 */
void ipu_csi_vertical_downsize_enable(struct ipu_csi *csi)
{
	unsigned long flags;
	u32 temp;

	spin_lock_irqsave(&csi->lock, flags);
	temp = ipu_csi_read(csi, CSI_OUT_FRM_CTRL);
	temp |= CSI_VERT_DOWNSIZE_EN;
	ipu_csi_write(csi, temp, CSI_OUT_FRM_CTRL);
	spin_unlock_irqrestore(&csi->lock, flags);
}
EXPORT_SYMBOL_GPL(ipu_csi_vertical_downsize_enable);

/*
 * ipu_csi_vertical_downsize_disable
 *	Disable vertical downsizing(decimation) by 2.
 */
void ipu_csi_vertical_downsize_disable(struct ipu_csi *csi)
{
	unsigned long flags;
	u32 temp;

	spin_lock_irqsave(&csi->lock, flags);
	temp = ipu_csi_read(csi, CSI_OUT_FRM_CTRL);
	temp &= ~CSI_VERT_DOWNSIZE_EN;
	ipu_csi_write(csi, temp, CSI_OUT_FRM_CTRL);
	spin_unlock_irqrestore(&csi->lock, flags);
}
EXPORT_SYMBOL_GPL(ipu_csi_vertical_downsize_disable);

/*
 * ipu_csi_set_test_generator
 *
 * @param	active       1 for active and 0 for inactive
 * @param       r_value	     red value for the generated pattern of even pixel
 * @param       g_value      green value for the generated pattern of even
 *			     pixel
 * @param       b_value      blue value for the generated pattern of even pixel
 * @param	pixel_clk   desired pixel clock frequency in Hz
 */
void ipu_csi_set_test_generator(struct ipu_csi *csi, bool active,
				u32 r_value, u32 g_value, u32 b_value,
				u32 pix_clk)
{
	unsigned long flags;
	u32 temp;
	u32 ipu_clk = clk_get_rate(csi->clk_ipu);

	spin_lock_irqsave(&csi->lock, flags);

	temp = ipu_csi_read(csi, CSI_TST_CTRL);

	if (active == false) {
		temp &= ~CSI_TEST_GEN_MODE_EN;
		ipu_csi_write(csi, temp, CSI_TST_CTRL);
	} else {
		/* Set sensb_mclk div_ratio*/
		ipu_csi_set_testgen_mclk(csi, pix_clk, ipu_clk);

		temp &= ~(CSI_TEST_GEN_R_MASK | CSI_TEST_GEN_G_MASK |
			CSI_TEST_GEN_B_MASK);
		temp |= CSI_TEST_GEN_MODE_EN;
		temp |= (r_value << CSI_TEST_GEN_R_SHIFT) |
			(g_value << CSI_TEST_GEN_G_SHIFT) |
			(b_value << CSI_TEST_GEN_B_SHIFT);
		ipu_csi_write(csi, temp, CSI_TST_CTRL);
	}

	spin_unlock_irqrestore(&csi->lock, flags);
}
EXPORT_SYMBOL_GPL(ipu_csi_set_test_generator);

/*
 * ipu_csi_set_mipi_datatype
 *
 * @param	vc	MIPI CSI2 virtual channel number (0-3)
 * @param	pixfmt	V4L2 pixel format from a sensor
 *
 * @return	Returns 0 on success or negative error code on fail
 */
int ipu_csi_set_mipi_datatype(struct ipu_csi *csi, u32 vc,
			      struct ipu_csi_signal_cfg *cfg)
{
	unsigned long flags;
	u32 temp;

	if (vc > 3)
		return -EINVAL;

	spin_lock_irqsave(&csi->lock, flags);

	temp = ipu_csi_read(csi, CSI_MIPI_DI);
	temp &= ~(0xff << (vc * 8));
	temp |= (cfg->mipi_dt << (vc * 8));
	ipu_csi_write(csi, temp, CSI_MIPI_DI);

	spin_unlock_irqrestore(&csi->lock, flags);

	return 0;
}
EXPORT_SYMBOL_GPL(ipu_csi_set_mipi_datatype);

/*
 * ipu_csi_set_skip_isp
 *
 * @param	skip		select frames to be skipped and set the
 *				correspond bits to 1
 * @param	max_ratio	number of frames in a skipping set and the
 *				maximum value of max_ratio is 5
 *
 * @return	Returns 0 on success or negative error code on fail
 */
int ipu_csi_set_skip_isp(struct ipu_csi *csi, u32 skip, u32 max_ratio)
{
	unsigned long flags;
	u32 temp;
	int retval = 0;

	if (max_ratio > 5) {
		retval = -EINVAL;
		goto err;
	}

	spin_lock_irqsave(&csi->lock, flags);

	temp = ipu_csi_read(csi, CSI_SKIP);
	temp &= ~(CSI_MAX_RATIO_SKIP_ISP_MASK | CSI_SKIP_ISP_MASK);
	temp |= (max_ratio << CSI_MAX_RATIO_SKIP_ISP_SHIFT) |
		(skip << CSI_SKIP_ISP_SHIFT);
	ipu_csi_write(csi, temp, CSI_SKIP);

	spin_unlock_irqrestore(&csi->lock, flags);
err:
	return retval;
}
EXPORT_SYMBOL_GPL(ipu_csi_set_skip_isp);

/*
 * ipu_csi_set_skip_smfc
 *
 * @param	skip		select frames to be skipped and set the
 *				correspond bits to 1
 * @param	max_ratio	number of frames in a skipping set and the
 *				maximum value of max_ratio is 5
 * @param	id		csi to smfc skipping id
 *
 * @return	Returns 0 on success or negative error code on fail
 */
int ipu_csi_set_skip_smfc(struct ipu_csi *csi, u32 skip,
			  u32 max_ratio, u32 id)
{
	unsigned long flags;
	u32 temp;
	int retval = 0;

	if (max_ratio > 5 || id > 3) {
		retval = -EINVAL;
		goto err;
	}

	spin_lock_irqsave(&csi->lock, flags);

	temp = ipu_csi_read(csi, CSI_SKIP);
	temp &= ~(CSI_MAX_RATIO_SKIP_SMFC_MASK | CSI_ID_2_SKIP_MASK |
			CSI_SKIP_SMFC_MASK);
	temp |= (max_ratio << CSI_MAX_RATIO_SKIP_SMFC_SHIFT) |
			(id << CSI_ID_2_SKIP_SHIFT) |
			(skip << CSI_SKIP_SMFC_SHIFT);
	ipu_csi_write(csi, temp, CSI_SKIP);

	spin_unlock_irqrestore(&csi->lock, flags);
err:
	return retval;
}
EXPORT_SYMBOL_GPL(ipu_csi_set_skip_smfc);

/*
 * ipu_csi_set_dest
 *
 * @param	csi_dest
 *
 * @return	Returns 0 on success or negative error code on fail
 */
int ipu_csi_set_dest(struct ipu_csi *csi, enum ipu_csi_dest csi_dest)
{
	unsigned long flags;
	u32 csi_sens_conf, dest;

	if (csi_dest == IPU_CSI_DEST_IDMAC)
		dest = CSI_DATA_DEST_IDMAC;
	else
		dest = CSI_DATA_DEST_IC; /* IC or VDIC */

	spin_lock_irqsave(&csi->lock, flags);

	csi_sens_conf = ipu_csi_read(csi, CSI_SENS_CONF);
	csi_sens_conf &= ~CSI_SENS_CONF_DATA_DEST_MASK;
	csi_sens_conf |= (dest << CSI_SENS_CONF_DATA_DEST_SHIFT);
	ipu_csi_write(csi, csi_sens_conf, CSI_SENS_CONF);

	spin_unlock_irqrestore(&csi->lock, flags);

	return 0;
}
EXPORT_SYMBOL_GPL(ipu_csi_set_dest);

int ipu_csi_enable(struct ipu_csi *csi)
{
	unsigned long flags;

	spin_lock_irqsave(&csi->lock, flags);

	if (!csi->use_count)
		ipu_module_enable(csi->ipu, csi->module);

	csi->use_count++;

	spin_unlock_irqrestore(&csi->lock, flags);

	return 0;
}
EXPORT_SYMBOL_GPL(ipu_csi_enable);

int ipu_csi_disable(struct ipu_csi *csi)
{
	unsigned long flags;

	spin_lock_irqsave(&csi->lock, flags);

	csi->use_count--;

	if (!csi->use_count)
		ipu_module_disable(csi->ipu, csi->module);

	if (csi->use_count < 0)
		csi->use_count = 0;

	spin_unlock_irqrestore(&csi->lock, flags);

	return 0;
}
EXPORT_SYMBOL_GPL(ipu_csi_disable);

int ipu_csi_get_num(struct ipu_csi *csi)
{
	return csi->id;
}
EXPORT_SYMBOL_GPL(ipu_csi_get_num);

struct ipu_csi *ipu_csi_get(struct ipu_soc *ipu, int id)
{
	if (id > 1)
		return ERR_PTR(-EINVAL);

	return ipu->csi_priv[id];
}
EXPORT_SYMBOL_GPL(ipu_csi_get);

void ipu_csi_put(struct ipu_csi *csi)
{
}
EXPORT_SYMBOL_GPL(ipu_csi_put);

int ipu_csi_init(struct ipu_soc *ipu, struct device *dev, int id,
		 unsigned long base, u32 module, struct clk *clk_ipu)
{
	struct ipu_csi *csi;

	if (id > 1)
		return -ENODEV;

	csi = devm_kzalloc(dev, sizeof(*csi), GFP_KERNEL);
	if (!csi)
		return -ENOMEM;

	ipu->csi_priv[id] = csi;

	spin_lock_init(&csi->lock);
	csi->module = module;
	csi->id = id;
	csi->clk_ipu = clk_ipu;
	csi->base = devm_ioremap(dev, base, PAGE_SIZE);
	if (!csi->base)
		return -ENOMEM;

	dev_dbg(dev, "CSI%d base: 0x%08lx remapped to %p\n",
		id, base, csi->base);
	csi->ipu = ipu;

	return 0;
}

void ipu_csi_exit(struct ipu_soc *ipu, int id)
{
}

void ipu_csi_dump(struct ipu_csi *csi)
{
	dev_dbg(csi->ipu->dev, "CSI_SENS_CONF:     %08x\n",
		ipu_csi_read(csi, CSI_SENS_CONF));
	dev_dbg(csi->ipu->dev, "CSI_SENS_FRM_SIZE: %08x\n",
		ipu_csi_read(csi, CSI_SENS_FRM_SIZE));
	dev_dbg(csi->ipu->dev, "CSI_ACT_FRM_SIZE:  %08x\n",
		ipu_csi_read(csi, CSI_ACT_FRM_SIZE));
	dev_dbg(csi->ipu->dev, "CSI_OUT_FRM_CTRL:  %08x\n",
		ipu_csi_read(csi, CSI_OUT_FRM_CTRL));
	dev_dbg(csi->ipu->dev, "CSI_TST_CTRL:      %08x\n",
		ipu_csi_read(csi, CSI_TST_CTRL));
	dev_dbg(csi->ipu->dev, "CSI_CCIR_CODE_1:   %08x\n",
		ipu_csi_read(csi, CSI_CCIR_CODE_1));
	dev_dbg(csi->ipu->dev, "CSI_CCIR_CODE_2:   %08x\n",
		ipu_csi_read(csi, CSI_CCIR_CODE_2));
	dev_dbg(csi->ipu->dev, "CSI_CCIR_CODE_3:   %08x\n",
		ipu_csi_read(csi, CSI_CCIR_CODE_3));
	dev_dbg(csi->ipu->dev, "CSI_MIPI_DI:       %08x\n",
		ipu_csi_read(csi, CSI_MIPI_DI));
	dev_dbg(csi->ipu->dev, "CSI_SKIP:          %08x\n",
		ipu_csi_read(csi, CSI_SKIP));
}
EXPORT_SYMBOL_GPL(ipu_csi_dump);
