/*
 * Copyright (C) 2013 Mentor Graphics, Inc. All Rights Reserved.
 * Copyright 2008-2012 Freescale Semiconductor, Inc. All Rights Reserved.
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/kernel.h>
#include <linux/clk.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/of_address.h>
#include <linux/of_irq.h>
#include <linux/of_platform.h>
#include <linux/errno.h>
#include <linux/mxc_asrc_core_reg.h>
#include <linux/mutex.h>
#include <linux/mxc_asrc_core_internal.h>
#include <linux/platform_data/dma-imx.h>
#include <linux/debugfs.h>

#define STARTUP_TIME_NS	500000

enum asrc_channel_limit_idx {
	CHANNEL_LIMIT_MIN,
	CHANNEL_LIMIT_MAX,
	CHANNEL_LIMIT_LAST
};

static struct class *asrc_core_class;
static struct class *asrc_pair_class;
static int configure_pair(struct asrc_section_params *);

/*
 * Defaults for min/max channel configuration for each of the 3
 * conversion pairs of the ASRC interface.
 * Following limitations should be taken care of while
 * adjusting the defaults:
 *   1) Min/Max should be equal for pair A because reconfiguration of pair A's
 *	channels while pair B is in use would result in audio quality
 *	issues on pair B.
 *   2) The sum of max values for all pairs should not exceed 10.
 */
static const unsigned char asrc_channel_mm_defaults[ASRC_PAIR_NUM]
						   [CHANNEL_LIMIT_LAST] = {
	{2, 2},
	{1, 2},
	{1, 6}
};

/* NOTE - update the table below before adding/deleting items */
static const unsigned int input_frequency[] = {
	5512,
	8000,
	11025,
	12000,
	16000,
	22050,
	24000,
	32000,
	44100,
	48000,
	64000,
	88200,
	96000,
	128000,
	176000,
	192000
};

/* NOTE - update the table below before adding/deleting items */
static const unsigned int output_frequency[] = {
	8000,
	16000,
	32000,
	44100,
	48000,
	64000,
	88200,
	96000,
	128000,
	176000,
	192000
};

/* constants for latency computation of pre and post filter */
static const unsigned int pre_filter_latency[MXC_ASRC_PRE_FILTERS] = {
		39000000,
		78500000,
		235000000
};
static const unsigned int post_filter_latency[MXC_ASRC_POST_FILTERS] = {
		42500000,
		8500000,
		172000000
};

/* scaled ratio of the pre-filter
 * (input_rate * MXC_ASRC_PRE_FILTER_MAX_UPSAMPLING /
 * input_rate polyphase filter )
 */
static const unsigned int pre_filter_ratio[MXC_ASRC_PRE_FILTERS] = {
		1,
		2,
		4
};

/* scaled ideal bandwidth of the signal at the input of polyphase filter
 * (input_rate * MXC_ASRC_PRE_FILTER_MAX_UPSAMPLING /
 * ideal bandwidth at polyphase filter )
 */
static const unsigned int pre_filter_ideal_bw[MXC_ASRC_PRE_FILTERS] = {
		4,
		8,
		16
};

/* scaled ratio of the post-filter
 * (output_rate * MXC_ASRC_POST_FILTER_MAX_UPSAMPLING /
 * output_rate polyphase filter )
 */
static const unsigned int post_filter_ratio[MXC_ASRC_POST_FILTERS] = {
		4,
		2,
		1
};

/* scaled ideal bandwidth of the signal at the output
 * (output_rate * MXC_ASRC_POST_FILTER_MAX_UPSAMPLING /
 * ideal bandwidth at polyphase filter )
 */
static const unsigned int post_filter_ideal_bw[MXC_ASRC_POST_FILTERS] = {
		8,
		4,
		4
};

/* Sample rates are aligned with that defined in pcm.h file */
static const unsigned char asrc_process_table[ARRAY_SIZE(input_frequency)]
				      [ARRAY_SIZE(output_frequency)]
				      [2] = {
/* Across   - output frequencies    */
/* Down     - input frequencies     */
/* 8kHz  16kHz   32kHz   44.1kHz 48kHz   64kHz
 *    88.2kHz 96kHz  128kHz   176kHz   192kHz */
/*5512Hz*/
	{{0, 1}, {0, 1}, {0, 0}, {0, 0}, {0, 0}, {0, 0},
			{0, 0}, {0, 0}, {0, 0}, {0, 0}, {0, 0},},
/*8kHz*/
	{{0, 1}, {0, 1}, {0, 0}, {0, 0}, {0, 0}, {0, 0},
			{0, 0}, {0, 0}, {0, 0}, {0, 0}, {0, 0},},
/*11025Hz*/
	{{0, 2}, {0, 1}, {0, 0}, {0, 0}, {0, 0}, {0, 0},
			{0, 0}, {0, 0}, {0, 0}, {0, 0}, {0, 0},},
/*12000Hz*/
	{{0, 2}, {0, 1}, {0, 1}, {0, 0}, {0, 0}, {0, 0},
			{0, 0}, {0, 0}, {0, 0}, {0, 0}, {0, 0},},
/*16kHz*/
	{{1, 2}, {0, 1}, {0, 1}, {0, 1}, {0, 1}, {0, 0},
			{0, 0}, {0, 0}, {0, 0}, {0, 0}, {0, 0},},
/*22050Hz*/
	{{1, 2}, {0, 2}, {0, 1}, {0, 1}, {0, 1}, {0, 0},
			{0, 0}, {0, 0}, {0, 0}, {0, 0}, {0, 0},},
/*24000Hz*/
	{{1, 2}, {0, 2}, {0, 1}, {0, 1}, {0, 1}, {0, 1},
			{0, 0}, {0, 0}, {0, 0}, {0, 0}, {0, 0},},
/*32kHz*/
	{{1, 2}, {1, 2}, {0, 1}, {0, 1}, {0, 1}, {0, 1},
			{0, 1}, {0, 0}, {0, 0}, {0, 0}, {0, 0},},
/*44.1kHz*/
	{{2, 2}, {1, 2}, {0, 2}, {0, 1}, {0, 1}, {0, 1},
			{0, 1}, {0, 1}, {0, 0}, {0, 0}, {0, 0},},
/*48kHz*/
	{{2, 2}, {1, 2}, {0, 2}, {0, 2}, {0, 1}, {0, 1},
			{0, 1}, {0, 1}, {0, 1}, {0, 0}, {0, 0},},
/*64kHz*/
	{{2, 2}, {1, 2}, {1, 2}, {0, 2}, {0, 2}, {0, 1},
			{0, 1}, {0, 1}, {0, 1}, {0, 1}, {0, 0},},
/*88.2kHz*/
	{{MXC_ASRC_FILTER_UNSUPPORTED, MXC_ASRC_FILTER_UNSUPPORTED},
			{2, 2}, {1, 2}, {1, 2}, {1, 2}, {1, 1},
			{1, 1}, {1, 1}, {1, 1}, {1, 1}, {1, 1},},
/*96kHz*/
	{{MXC_ASRC_FILTER_UNSUPPORTED, MXC_ASRC_FILTER_UNSUPPORTED},
			{2, 2}, {1, 2}, {1, 2}, {1, 2}, {1, 1},
			{1, 1}, {1, 1}, {1, 1}, {1, 1}, {1, 1},},
/*128000Hz*/
	{{MXC_ASRC_FILTER_UNSUPPORTED, MXC_ASRC_FILTER_UNSUPPORTED},
			{2, 2}, {1, 2}, {1, 2}, {1, 2}, {1, 1},
			{1, 1}, {1, 1}, {1, 1}, {1, 1}, {1, 1},},
/*176kHz*/
	{{MXC_ASRC_FILTER_UNSUPPORTED, MXC_ASRC_FILTER_UNSUPPORTED},
		{MXC_ASRC_FILTER_UNSUPPORTED, MXC_ASRC_FILTER_UNSUPPORTED},
				{2, 2}, {2, 2}, {2, 2}, {2, 1},
				{2, 1}, {2, 1}, {2, 1}, {2, 1}, {2, 1},},
/*192kHz*/
	{{MXC_ASRC_FILTER_UNSUPPORTED, MXC_ASRC_FILTER_UNSUPPORTED},
		{MXC_ASRC_FILTER_UNSUPPORTED, MXC_ASRC_FILTER_UNSUPPORTED},
				{2, 2}, {2, 2}, {2, 2}, {2, 1},
				{2, 1}, {2, 1}, {2, 1}, {2, 1}, {2, 1},},
};

static const unsigned char asrc_fixed_prefilter_table[FILT_CNT] = {
		0xff, 0xff, 0, 0, 0, 1, 1, 1, 2, 2, 2, 3
};

static const unsigned char asrc_fixed_postfilter_table[FILT_CNT] = {
		0xff, 0xff, 0, 1, 2, 0, 1, 2, 0, 1, 2, 0
};

static unsigned int get_min_latency(unsigned char pre_filter,
		unsigned char post_filter,
		unsigned int input_samplerate,
		unsigned int output_samplerate,
		struct device *dev);

static int get_asrc_bw(unsigned char pre_filter,
		unsigned char post_filter,
		unsigned int input_samplerate,
		unsigned int output_samplerate,
		unsigned int *bandwidth,
		unsigned int *alias_start,
		unsigned int *quality,
		struct device *dev);

static void compute_best_filter_settings(unsigned int input_samplerate,
		unsigned int output_samplerate,
		unsigned char *pre_filter,
		unsigned char *post_filter,
		struct device *dev);

/* compute latency estimation of ASRC (ignoring content in FIFOs) */
/* in case of invalid filter settings or sample-rates 0 is returned */
/* unit of returned latency is us */
static unsigned int get_min_latency(unsigned char pre_filter,
		unsigned char post_filter,
		unsigned int input_samplerate,
		unsigned int output_samplerate,
		struct device *dev)
{
	unsigned int min_latency;

	if ((pre_filter >= MXC_ASRC_PRE_FILTERS) ||
		(post_filter >= MXC_ASRC_POST_FILTERS)) {
		dev_warn(dev, "get_min_latency called for invalid filter config (%d, %d)\n",
				pre_filter, post_filter);

		return 0;
	}

	if ((input_samplerate == 0) || (output_samplerate == 0)) {
		dev_warn(dev, "get_min_latency called for invalid sample-rates (%d, %d)\n",
				input_samplerate, output_samplerate);

		return 0;
	}

	min_latency = pre_filter_latency[pre_filter] / input_samplerate;
	min_latency += post_filter_latency[post_filter] / output_samplerate;

	return min_latency;
}

/* get the bandwidth of data processed by ASRC
 * in case of invalid filter settings or sample-rates
 *    MXC_ASRC_GET_ASRC_BW_INVALID_PARAM is returned
 * in case of an invalid polyphase filter ratio
 *    MXC_ASRC_GET_ASRC_BW_INVALID_POLYPHASE_FILTER_RATIO is returned
 * in case strong alias effects are expected
 *    MXC_ASRC_GET_ASRC_BW_STRONG_ALIAS is returned
 * in case everything is fine MXC_ASRC_GET_ASRC_BW_OK is returned
 * bandwidth returns is the ideal bandwidth for the conversion
 * alias_start returns is an estimate of the frequency alias
 *    effects are expected to start
 *    In case of no alias effects expected it will be equal to bandwidth
 *    As filters are not ideal bandwidth of filters is increased by a
 *    certain estimated percentage for alias estimation
 * quality returns an estimate of the quality (undesired frequencies)
 *    smaller is better
 * pointer to device structure
 * */
static int get_asrc_bw(unsigned char pre_filter,
		unsigned char post_filter,
		unsigned int input_samplerate,
		unsigned int output_samplerate,
		unsigned int *bandwidth,
		unsigned int *alias_start,
		unsigned int *quality,
		struct device *dev)
{
	unsigned int polyphase_input_rate;
	unsigned int polyphase_output_rate;
	unsigned int polyphase_output_bw;
	unsigned int prefilter_bw;
	unsigned int postfilter_bw;
	unsigned int postfilter_bw_p_save;
	unsigned int output_bw;
	unsigned int _alias_start;
	unsigned int alias_start_tmp;
	unsigned int max_freq;

	*quality = 0;

	if ((pre_filter >= MXC_ASRC_PRE_FILTERS) ||
		(post_filter >= MXC_ASRC_POST_FILTERS)) {
		dev_warn(dev, "%s called for invalid filter config (%d, %d)\n",
				__func__, pre_filter, post_filter);

		return MXC_ASRC_GET_ASRC_BW_INVALID_PARAM;
	}

	if ((input_samplerate == 0) || (output_samplerate == 0)) {
		dev_warn(dev, "%s called for invalid sample-rates (%d, %d)\n",
				__func__, input_samplerate, output_samplerate);

		return MXC_ASRC_GET_ASRC_BW_INVALID_PARAM;
	}

	polyphase_input_rate = input_samplerate *
			MXC_ASRC_PRE_FILTER_MAX_UPSAMPLING /
			pre_filter_ratio[pre_filter];
	polyphase_output_rate = output_samplerate *
			MXC_ASRC_POST_FILTER_MAX_UPSAMPLING /
			post_filter_ratio[post_filter];

	if (polyphase_input_rate >
		polyphase_output_rate *
		MXC_ASRC_MAX_POLYFILTER_DOWNSAMPLING_RATIO) {
		return MXC_ASRC_GET_ASRC_BW_INVALID_POLYPHASE_FILTER_RATIO;
	}

	/* bandwidth of filters */
	prefilter_bw = input_samplerate *
			MXC_ASRC_PRE_FILTER_MAX_UPSAMPLING /
			pre_filter_ideal_bw[pre_filter];

	postfilter_bw = output_samplerate *
			MXC_ASRC_POST_FILTER_MAX_UPSAMPLING /
			post_filter_ideal_bw[post_filter];


	/* after prefilter */
	max_freq = prefilter_bw *
		(100 + MXC_ASRC_FORMULA_SAFETY_DEFAULT) / 100;

	_alias_start = prefilter_bw;
	if (pre_filter == MXC_ASRC_PRE_FILTER_ZERO_INS_UP2_ID)
		*quality = max_freq - prefilter_bw;

	/* at polyphase output */
	if (max_freq > polyphase_output_rate)
		return MXC_ASRC_GET_ASRC_BW_STRONG_ALIAS;

	if (post_filter != MXC_ASRC_POST_FILTER_DIRECT_ID) {
		postfilter_bw_p_save = postfilter_bw *
				(100 + MXC_ASRC_FORMULA_SAFETY_DEFAULT) / 100;
		polyphase_output_bw = polyphase_output_rate/2;
		postfilter_bw_p_save = min(polyphase_output_bw,
				postfilter_bw_p_save);

		if (max_freq > polyphase_output_bw) {
			_alias_start = polyphase_output_rate - max_freq;
			*quality += postfilter_bw_p_save - _alias_start;
		}

		if (_alias_start >= postfilter_bw_p_save) {
			_alias_start = ~0;
			*quality = 0;
		}

		max_freq = min(max_freq, postfilter_bw_p_save);
	}

	/* at output */
	output_bw = output_samplerate/2;
	if (max_freq > output_samplerate)
		return MXC_ASRC_GET_ASRC_BW_STRONG_ALIAS;

	if (max_freq > output_bw) {
		alias_start_tmp = output_samplerate - max_freq;
		*quality += output_bw - alias_start_tmp;

		_alias_start = min(_alias_start, alias_start_tmp);
		max_freq = output_bw;
	}

	/* compute ideal bw */
	*bandwidth = min(prefilter_bw,
			postfilter_bw);

	/* alias_start returned shall never be greater than bw */
	*alias_start = min(_alias_start, *bandwidth);

	if (*alias_start < *bandwidth *
			(100 - MXC_ASRC_FORMULA_STRONG_ALIAS_THRESHOLD) / 100)
		return MXC_ASRC_GET_ASRC_BW_STRONG_ALIAS;

	return MXC_ASRC_GET_ASRC_BW_OK;
}

/* search for the best filter settings for given input output sample-rates
 * in case none are found pre_filter and post_filter will be
 * MXC_ASRC_FILTER_UNSUPPORTED
 */
static void compute_best_filter_settings(unsigned int input_samplerate,
		unsigned int output_samplerate,
		unsigned char *pre_filter,
		unsigned char *post_filter,
		struct device *dev)
{
	unsigned int best_latency_found = 0;
	unsigned int cur_latency;
	unsigned int best_bandwidth_found = 0;
	unsigned int cur_bandwidth;
	unsigned int best_alias_found = 0;
	unsigned int cur_alias;
	unsigned int best_qual_found = 0;
	unsigned int cur_qual;
	unsigned char pre_it, post_it;

	*pre_filter = MXC_ASRC_FILTER_UNSUPPORTED;
	*post_filter = MXC_ASRC_FILTER_UNSUPPORTED;

	for (pre_it = 0; pre_it < MXC_ASRC_PRE_FILTERS; pre_it++) {
		for (post_it = 0; post_it < MXC_ASRC_POST_FILTERS; post_it++) {
			if (MXC_ASRC_GET_ASRC_BW_OK == get_asrc_bw(
					pre_it, post_it,
					input_samplerate, output_samplerate,
					&cur_bandwidth, &cur_alias,
					&cur_qual, dev)) {

				cur_latency = get_min_latency(
						pre_it, post_it,
						input_samplerate,
						output_samplerate,
						dev);

				if ((*pre_filter ==
						MXC_ASRC_FILTER_UNSUPPORTED) ||
					(*post_filter ==
						MXC_ASRC_FILTER_UNSUPPORTED) ||
					(cur_bandwidth >
						best_bandwidth_found) ||
					((cur_bandwidth ==
							best_bandwidth_found) &&
						(cur_alias >
							best_alias_found)) ||
					((cur_bandwidth ==
							best_bandwidth_found) &&
						(cur_alias ==
							best_alias_found) &&
						(cur_qual <
							best_qual_found)) ||
					((cur_bandwidth ==
							best_bandwidth_found) &&
						(cur_alias ==
							best_alias_found) &&
						(cur_qual ==
							best_qual_found) &&
						(cur_latency <
							best_latency_found))) {

					*pre_filter = pre_it;
					*post_filter = post_it;
					best_bandwidth_found = cur_bandwidth;
					best_latency_found = cur_latency;
					best_alias_found = cur_alias;
					best_qual_found = cur_qual;
				}
			}
		}
	}
}

static int find_nearest_in_clk(const unsigned int clk_rate)
{
	int i = 0;
	int in_clk_max = ARRAY_SIZE(input_frequency);

	/* start at the largest supported frequency */
	for (i = in_clk_max-1; i > 0; i--) {
		if (clk_rate >= input_frequency[i])
			return i;
	}
	/* else return lowest frequency */
	return 0;
}

static int find_nearest_out_clk(const unsigned int clk_rate)
{
	int i = 0;
	int out_clk_max = ARRAY_SIZE(output_frequency);

	/* start at the largest supported frequency */
	for (i = out_clk_max-1; i > 0; i--) {
		if (clk_rate >= output_frequency[i])
			return i;
	}
	/* else return lowest frequency */
	return 0;
}

static int set_ideal_ratio(struct fsl_asrc_core_private *asrc_priv,
				enum asrc_pair_index index,
				unsigned int input_sample_rate,
				unsigned int output_sample_rate)
{
	int i;
	int integ = 0;
	unsigned long reg_val = 0, lock_flags;
	unsigned int adjusted_input_rate = input_sample_rate;

	if (output_sample_rate == ASRC_INVALID_RATE)
		return -EINVAL;
	while (adjusted_input_rate >= output_sample_rate) {
		adjusted_input_rate -= output_sample_rate;
		integ++;
	}
	reg_val |= (integ << ASRC_RATIO_DECIMAL_DEPTH);

	for (i = 1; i <= ASRC_RATIO_DECIMAL_DEPTH; i++) {
		if ((adjusted_input_rate * 2) >= output_sample_rate) {
			reg_val |= (1 << (ASRC_RATIO_DECIMAL_DEPTH - i));
			adjusted_input_rate =
					(adjusted_input_rate * 2) -
					output_sample_rate;
		} else
			adjusted_input_rate = adjusted_input_rate << 1;

		if (adjusted_input_rate == 0)
			break;
	}

	spin_lock_irqsave(&asrc_priv->core_lock, lock_flags);
	__raw_writel((reg_val & 0x00FFFFFF),
		(asrc_priv->vaddr + ASRC_ASRIDRLA_REG + (index << 3)));
	__raw_writel((reg_val >> 24),
		(asrc_priv->vaddr + ASRC_ASRIDRHA_REG + (index << 3)));
	spin_unlock_irqrestore(&asrc_priv->core_lock, lock_flags);

	dev_dbg(asrc_priv->dev,
			"%s ideal ratio:[target=(%u/%u),actual=0x%08lx]\n",
			__func__,
			input_sample_rate,
			output_sample_rate,
			reg_val);

	return 0;
}

/* set the pre/post processing filter settings */
static int set_process_configuration(struct asrc_pair_params *pair_params)
{
	int i = 0, j = 0, index = pair_params->index;
	unsigned char pre, post;
	unsigned long reg = 0;
	unsigned long lock_flags;
	struct fsl_asrc_core_private *asrc_priv =
			pair_params->asrc_pair->asrc_priv;
	enum filter_settings filter;

	spin_lock_irqsave(&pair_params->asrc_pair->pair_lock, lock_flags);
	filter = pair_params->filter;
	spin_unlock_irqrestore(&pair_params->asrc_pair->pair_lock, lock_flags);

	if (filter == FILT_FORMULA) {
		compute_best_filter_settings(
				pair_params->section[SECTION_INPUT].rate,
				pair_params->section[SECTION_OUTPUT].rate,
				&pre,
				&post,
				asrc_priv->dev);
	} else if (filter == FILT_TABLE) {
		i = find_nearest_in_clk(
				pair_params->section[SECTION_INPUT].rate);
		j = find_nearest_out_clk(
				pair_params->section[SECTION_OUTPUT].rate);
		pre = asrc_process_table[i][j][0];
		post = asrc_process_table[i][j][1];
	} else if (filter < FILT_CNT) {
		pre = asrc_fixed_prefilter_table[filter];
		post = asrc_fixed_postfilter_table[filter];
	} else
		return -EINVAL;

	if ((pre == MXC_ASRC_FILTER_UNSUPPORTED) ||
			(post == MXC_ASRC_FILTER_UNSUPPORTED)) {
		dev_err(asrc_priv->dev, "Conversion ratios are not supported : filter type = %d, input rate = %d, output rate = %d\n",
			filter, pair_params->section[SECTION_INPUT].rate,
			pair_params->section[SECTION_OUTPUT].rate);
		return -EINVAL;
	}
	spin_lock_irqsave(&asrc_priv->core_lock, lock_flags);
	reg = __raw_readl(asrc_priv->vaddr + ASRC_ASRCFG_REG);
	reg &= ~(0x0f << (6 + (index << 2)));
	reg |= ((pre << (6 + (index << 2))) |
		 (post << (8 + (index << 2))));
	__raw_writel(reg, asrc_priv->vaddr + ASRC_ASRCFG_REG);
	spin_unlock_irqrestore(&asrc_priv->core_lock, lock_flags);
	dev_dbg(asrc_priv->dev, "%s setting in:%d out:%d i:%d j:%d\n",
			__func__, pre, post, i, j);

	return 0;
}

static int get_clock_divider(unsigned long bitclk, unsigned int samplerate)
{
	unsigned int prescaler, divider;
	unsigned int i;
	unsigned int ratio, ra;

	if (samplerate == 0)
		return 0;

	ra = bitclk/samplerate;
	ratio = ra;
	/*calculate the prescaler*/
	i = 0;
	while (ratio > 8) {
		i++;
		ratio = ratio >> 1;
	}
	prescaler = i;
	/*calculate the divider*/
	if (i > 7)
		return (0x7 << 3) + 0x7;
	else if (i >= 1)
		divider = ((ra + (1 << (i - 1)) - 1) >> i) - 1;
	else
		return 0;
	/*the total divider is (2^prescaler)*divider*/
	return (divider << 3) + prescaler;
}

/* no restrictions on when it can be called */
int asrc_set_period(struct asrc_section_params *params,
		unsigned char value)
{
	unsigned long lock_flags;
	struct asrc_pair *pair =
			params->asrc_pair_params->asrc_pair;

	/* validate value */
	if ((value < MIN_PERIOD) || (value > MAX_PERIOD))
		return -EINVAL;

	spin_lock_irqsave(&pair->pair_lock, lock_flags);
	params->hw_params.period = value;
	spin_unlock_irqrestore(&pair->pair_lock, lock_flags);

	return 0;
}
EXPORT_SYMBOL(asrc_set_period);

unsigned char asrc_get_period(struct asrc_section_params *params)
{
	unsigned long lock_flags;
	unsigned char period;
	struct asrc_pair *pair = params->asrc_pair_params->asrc_pair;

	spin_lock_irqsave(&pair->pair_lock, lock_flags);
	period = params->hw_params.period;
	spin_unlock_irqrestore(&pair->pair_lock, lock_flags);

	return period;
}
EXPORT_SYMBOL(asrc_get_period);

/* no restrictions on when it can be called */
int asrc_set_filter_settings(struct asrc_section_params *params,
		enum filter_settings settings)
{
	struct asrc_pair *pair = params->asrc_pair_params->asrc_pair;
	unsigned long lock_flags;

	/* validate settings */
	if ((settings < FILT_START) || (settings >= FILT_CNT))
		return -EINVAL;

	spin_lock_irqsave(&pair->pair_lock, lock_flags);
	params->asrc_pair_params->filter = settings;
	spin_unlock_irqrestore(&pair->pair_lock, lock_flags);

	return 0;
}
EXPORT_SYMBOL(asrc_set_filter_settings);

const char const **asrc_fetch_clock_list(
		struct fsl_asrc_core_private *asrc_core_private, int *clk_cnt)
{
	if (asrc_core_private && clk_cnt) {
		*clk_cnt = asrc_core_private->ideal_table_length;
		return asrc_core_private->clock_text;
	}

	if (clk_cnt)
		*clk_cnt = 0;

	return NULL;
}
EXPORT_SYMBOL_GPL(asrc_fetch_clock_list);


/**
 * locate_ideal_clock_entry() locates ideal clock entry matching clock name
 *
 * @asrc_priv			struct fsl_asrc_core_private
 * @ideal_stream_clk_name	struct ideal_stream_clk_name
 *
 * returns an ideal clock entry matching the given ideal clock name with prefix
 * Note: matching assumes that ideal_stream_clk_name will always contain a
 * stream-name based prefixed ideal clock name.
 */
static struct asrc_ideal_clock_entry *
locate_ideal_clock_entry(struct fsl_asrc_core_private *asrc_priv,
		const char *ideal_stream_clk_name)
{
	int i = 0;
	const char *match_name = NULL, *ideal_clk_name;
	unsigned int idl_strm_clk_len, idl_clk_len, match_name_len;
	unsigned int playback_prefix_len = strlen(IDEAL_PLAYBACK_PREFIX);
	unsigned int capture_prefix_len = strlen(IDEAL_CAPTURE_PREFIX);

	dev_dbg(asrc_priv->dev, "%s ideal table entries:%d\n",
			__func__,
			asrc_priv->ideal_table_length);

	idl_strm_clk_len = strnlen(ideal_stream_clk_name, ASRC_MAX_CLKNAME_LEN);

	/* locate clock table entry */
	for (i = 0; i < asrc_priv->ideal_table_length; i++) {
		/* this is safe because the clock names are populated once */
		ideal_clk_name = asrc_priv->ideal_clock_table[i].ideal_clk_name;

		/* Note: matching assumes that ideal_stream_clk_name will
		 * always contain a prefixed ideal clock name.
		 */
		match_name = strnstr(ideal_stream_clk_name,
			ideal_clk_name,
			ASRC_MAX_CLKNAME_LEN);
		dev_dbg(asrc_priv->dev,
			"%s table:%s, lookup:%s, table-bclk:%lu, ref-cnt:%d\n",
			__func__,
			ideal_clk_name,
			ideal_stream_clk_name,
			asrc_priv->ideal_clock_table[i].bit_clock,
			asrc_priv->ideal_clock_table[i].ideal_clk_cnt);
		if (match_name != NULL) {
			idl_clk_len = strnlen(ideal_clk_name,
					ASRC_MAX_CLKNAME_LEN);
			match_name_len = strnlen(match_name,
					ASRC_MAX_CLKNAME_LEN);

			if (/* prefix based match */
				((match_name == ideal_stream_clk_name) &&
				 (match_name_len == idl_strm_clk_len)) ||
				/* suffix based match */
				(((match_name_len + playback_prefix_len) ==
				  idl_strm_clk_len) &&
				 (match_name_len == idl_clk_len)) ||
				(((match_name_len + capture_prefix_len) ==
				  idl_strm_clk_len) &&
				 (match_name_len == idl_clk_len))
			)
				return (struct asrc_ideal_clock_entry *)
					&asrc_priv->ideal_clock_table[i];
		}
	}

	return NULL;
}

int asrc_clear_xrun_callback(struct asrc_section_params *params)
{
	struct asrc_pair *pair;
	unsigned long lock_flags;
	struct fsl_asrc_core_private *asrc_priv;
	int ret = -EAGAIN;

	if (!params)
		return -EINVAL;

	pair = params->asrc_pair_params->asrc_pair;
	asrc_priv = pair->asrc_priv;

	spin_lock_irqsave(&asrc_priv->xrun_lock, lock_flags);
	if (params->asrc_pair_params->xrun_func) {
		params->asrc_pair_params->xrun_func = NULL;
		params->asrc_pair_params->xrun_data = NULL;
		ret = 0;
	} else
		ret = -EINVAL;
	spin_unlock_irqrestore(&asrc_priv->xrun_lock, lock_flags);

	return ret;
}
EXPORT_SYMBOL(asrc_clear_xrun_callback);

int asrc_set_xrun_callback(struct asrc_section_params *params,
		void (*func)(void *), void *data)
{
	struct asrc_pair *pair;
	int ret = 0;
	unsigned long lock_flags;
	struct fsl_asrc_core_private *asrc_priv;

	if (!params)
		return -EINVAL;

	pair = params->asrc_pair_params->asrc_pair;
	asrc_priv = pair->asrc_priv;

	spin_lock_irqsave(&asrc_priv->xrun_lock, lock_flags);
	if (!params->asrc_pair_params->xrun_func) {
		params->asrc_pair_params->xrun_func = func;
		params->asrc_pair_params->xrun_data = data;
	} else if ((params->asrc_pair_params->xrun_func != func) ||
			(params->asrc_pair_params->xrun_data != data))
		ret = -EBUSY;

	spin_unlock_irqrestore(&asrc_priv->xrun_lock, lock_flags);

	if (ret)
		dev_err(asrc_priv->dev,
		"xrun callback must be cleared first\n");

	return ret;
}
EXPORT_SYMBOL(asrc_set_xrun_callback);

int asrc_set_callback(struct asrc_section_params *params,
		void (*func)(unsigned long), unsigned long data)
{
	struct asrc_pair *pair;
	unsigned long lock_flags;

	if (params == NULL)
		return -EINVAL;

	pair = params->asrc_pair_params->asrc_pair;

	spin_lock_irqsave(&pair->pair_lock, lock_flags);
	params->callback_arg = data;
	params->callback_func = func;
	spin_unlock_irqrestore(&pair->pair_lock, lock_flags);

	return 0;
}
EXPORT_SYMBOL(asrc_set_callback);

/* no restrictions on when it can be called */
int asrc_set_clock_reference_by_idx(struct asrc_section_params *params,
		int ideal_clk_idx)
{
	struct fsl_asrc_core_private *asrc_priv =
			params->asrc_pair_params->asrc_pair->asrc_priv;
	struct asrc_pair *pair = params->asrc_pair_params->asrc_pair;
	unsigned long lock_flags;
	int result;
	struct asrc_ideal_clock_entry const *clk_entry = NULL;

	if (ideal_clk_idx >= 0 && ideal_clk_idx < asrc_priv->ideal_table_length)
		clk_entry = &asrc_priv->ideal_clock_table[ideal_clk_idx];

	if (clk_entry) {
		result = strncmp(ASRC_CLK_NONE,
				clk_entry->ideal_clk_name,
				ASRC_MAX_CLKNAME_LEN);
		if ((params->direction == SECTION_INPUT) || (result != 0)) {
			spin_lock_irqsave(&pair->pair_lock, lock_flags);
			params->clock_ref = clk_entry;
			spin_unlock_irqrestore(&pair->pair_lock, lock_flags);
			return 0;
		}
	}

	return -EINVAL;
}
EXPORT_SYMBOL(asrc_set_clock_reference_by_idx);

int asrc_set_clock_reference(struct asrc_section_params *params,
		const char *ideal_clk_name)
{
	struct fsl_asrc_core_private *asrc_priv =
			params->asrc_pair_params->asrc_pair->asrc_priv;
	struct asrc_pair *pair = params->asrc_pair_params->asrc_pair;
	unsigned long lock_flags;
	int result;

	struct asrc_ideal_clock_entry const *clk_entry =
			locate_ideal_clock_entry(asrc_priv, ideal_clk_name);

	if (clk_entry) {
		result = strncmp(ASRC_CLK_NONE,
				ideal_clk_name, ASRC_MAX_CLKNAME_LEN);
		if ((params->direction == SECTION_INPUT) || (result != 0)) {
			spin_lock_irqsave(&pair->pair_lock, lock_flags);
			params->clock_ref = clk_entry;
			spin_unlock_irqrestore(&pair->pair_lock, lock_flags);
			return 0;
		}
	}

	return -EINVAL;
}
EXPORT_SYMBOL(asrc_set_clock_reference);

/* no restrictions on when it can be called */
int asrc_get_section_from_pair(
		struct fsl_asrc_core_private *asrc_priv,
		enum asrc_pair_index pair,
		struct asrc_section_params **asrc_section,
		enum asrc_section_direction direction)
{

	*asrc_section = &(asrc_priv->asrc_pair[pair].params.section[direction]);

	dev_dbg(asrc_priv->dev, "%s section addr:%p\n",
			__func__,
			*asrc_section);

	return 0;
}
EXPORT_SYMBOL(asrc_get_section_from_pair);

/* no restrictions on when it can be called */
int asrc_get_capabilities(struct asrc_section_params *params,
		struct asrc_section_capabilities *capabilities)
{
	struct asrc_pair_capabilities *param_caps =
			     &params->asrc_pair_params->asrc_pair->capabilities;

	/* This is only written to during probe so does not need a mutex */
	capabilities->formats = param_caps->formats[params->direction];
	capabilities->channels_max = param_caps->channels_max;
	capabilities->channels_min = param_caps->channels_min;
	capabilities->rate_max = param_caps->rate_max;
	capabilities->rate_min = param_caps->rate_min;

	return 0;
}
EXPORT_SYMBOL(asrc_get_capabilities);

/* no restrictions on when it can be called */
int asrc_get_buffer_capabilities(struct asrc_section_params *params,
		struct buffer_capabilities *capabilities)
{
	/* This is only written to during probe so does not need a mutex */
	capabilities->bufbytes_max = MAX_BUFFER;
	capabilities->periods_min = 2;
	capabilities->periods_max = 255; /* arbitrary */
	capabilities->periodbytes_min = 2;
	capabilities->bufbytes_min = capabilities->periods_min *
			capabilities->periodbytes_min;
	capabilities->periodbytes_max = 65535; /* SDMA hard limit */

	return 0;
}
EXPORT_SYMBOL(asrc_get_buffer_capabilities);

/* enforces state transition behavior */
/* returns 0 on successful state change */
/* this function must be called from behind a section lock */
static int attempt_state_change(struct asrc_section_params *params,
		enum asrc_section_state state)
{
	enum asrc_section_state current_state;
	int err = 0;

	current_state = params->state;
	if ((current_state != state) &&
			(state != NEXT_STATE(current_state)) &&
			(state != SECTION_IRQ_ERR)) {

		/* if states are added, pay attention to the logic here*/
		switch (current_state) {
		case SECTION_CLOSED:
		case SECTION_OPENED:
		case SECTION_SETUP:
			if (state != PREV_STATE(current_state))
				err = -EINVAL;
			break;
		case SECTION_PREPARE:
		case SECTION_READY:
			if ((state != SECTION_SETUP) &&
					(state != SECTION_SOFT_XRUN))
				err = -EINVAL;
			break;
		case SECTION_RUN:
		case SECTION_SOFT_XRUN:
		case SECTION_HARD_XRUN:
		case SECTION_IRQ_ERR:
			if ((state != SECTION_SETUP) &&
					(state != SECTION_HARD_XRUN))
				err = -EINVAL;
			break;

		default:
			err = -EINVAL;
		}
	}

	return err;
}

/* this function must be called from a lock on all sections in the pair */
/* 0 = match; -EPERM = not allowed */
static int compare_pair_state(struct asrc_pair_params *params,
		enum asrc_section_state state)
{
	enum asrc_section_direction i;
	int err = 0;

	for (i = 0; i < SECTION_CNT; i++) {
		if (state != params->section[i].state) {
			err = -EPERM;
			break;
		}
	}

	return err;
}

/* this function must be called from a section lock */
static int set_section_state(struct asrc_section_params *params,
		enum asrc_section_state state)
{
	int err;

	err = attempt_state_change(params, state);
	if (!err)
		params->state = state;

	return err;
}

/* this function must be called from a lock on all sections in the pair */
static int set_pair_state(struct asrc_pair_params *params,
		enum asrc_section_state state)
{
	enum asrc_section_direction i;
	int err;

	/* make sure all sections can transition to requested state */
	for (i = 0; i < SECTION_CNT; i++) {
		err = attempt_state_change((&params->section[i]), state);
		if (err)
			break;
	}

	/* both sections can be successfully switched */
	if (!err)
		for (i = 0; i < SECTION_CNT; i++) {
			params->section[i].state = state;
			/* WARNING: This must be atomic */
			if (params->section[i].callback_func)
				params->section[i].callback_func(
					params->section[i].callback_arg);
		}

	return err;
}

enum asrc_section_state
asrc_get_section_state(struct asrc_section_params *params)
{
	enum asrc_section_state state;
	unsigned long lock_flags;
	struct asrc_pair *pair = params->asrc_pair_params->asrc_pair;

	spin_lock_irqsave(&pair->pair_lock, lock_flags);
	state = params->state;
	spin_unlock_irqrestore(&pair->pair_lock, lock_flags);

	return state;
}
EXPORT_SYMBOL(asrc_get_section_state);

int asrc_open_section(struct asrc_section_params *params)
{
	unsigned long lock_flags;
	int err;
	struct asrc_pair *pair = params->asrc_pair_params->asrc_pair;

	spin_lock_irqsave(&pair->pair_lock, lock_flags);
	err = set_section_state(params, SECTION_OPENED);
	spin_unlock_irqrestore(&pair->pair_lock, lock_flags);

	return err;
}
EXPORT_SYMBOL(asrc_open_section);

/* not mandatory. Used to reset a section to req_pair state */
int asrc_init_section(struct asrc_section_params *params)
{
	unsigned long lock_flags;
	struct asrc_pair *pair = params->asrc_pair_params->asrc_pair;

	spin_lock_irqsave(&pair->pair_lock, lock_flags);
	params->hw_params.period = DEFAULT_FIFO_PERIOD;
	params->channels = 0;
	params->format = 0;
	params->rate = ASRC_INVALID_RATE;
	spin_unlock_irqrestore(&pair->pair_lock, lock_flags);

	if (params->direction == SECTION_INPUT)
		asrc_set_clock_reference(params, "NONE");
	else
		asrc_set_clock_reference(params, "ASRCK1");

	return 0;
}
EXPORT_SYMBOL(asrc_init_section);

int asrc_req_pair(struct fsl_asrc_core_private *asrc_priv,
			unsigned char mask,
			enum asrc_pair_index *index)
{
	int err = -EBUSY, i;
	struct asrc_pair *tmp_pair;
	struct asrc_pair_params *params;
	unsigned long lock_flags;

	if (!asrc_priv)
		return -EINVAL;

	/* determine if our channel mask is set and channel is available */
	for (i = ASRC_PAIR_A; ((i < ASRC_PAIR_NUM) && (err != 0)); i++) {
		tmp_pair = &asrc_priv->asrc_pair[i];
		params = &tmp_pair->params;

		if ((mask & CHANNEL_MASK(i)) &&
		    (params->pair_hold == 0) &&
		    (tmp_pair->capabilities.channels_max >= ASRC_CHANNEL_MIN)) {

			spin_lock_irqsave(&tmp_pair->pair_lock, lock_flags);
			params->channel_cnt =
				tmp_pair->capabilities.channels_max;
			params->pair_hold = 1;
			*index = i;
			/* reset pair stats */
			tmp_pair->stats.aool =
					tmp_pair->stats.aodo =
					tmp_pair->stats.aiol =
					tmp_pair->stats.aidu =
					tmp_pair->stats.aodf =
					tmp_pair->stats.aide = 0;

			params->filter = FILT_TABLE;

			params->section[SECTION_INPUT].callback_func = NULL;
			params->section[SECTION_OUTPUT].callback_func = NULL;

			spin_unlock_irqrestore(&tmp_pair->pair_lock,
					lock_flags);

			asrc_init_section(&params->section[SECTION_INPUT]);
			asrc_init_section(&params->section[SECTION_OUTPUT]);

			err = 0;
		}
	}

	dev_dbg(asrc_priv->dev, "%s requested pair[%d]\n",
			__func__, *index);

	if (i == ASRC_PAIR_NUM)
		dev_dbg(asrc_priv->dev,
			"%s , requested pair(s) busy or disabled!\n", __func__);

	return err;
}
EXPORT_SYMBOL(asrc_req_pair);

int asrc_release_pair(struct asrc_section_params *params)
{
	unsigned long reg;
	unsigned long lock_flags;
	int err;

	struct asrc_pair *pair = params->asrc_pair_params->asrc_pair;
	struct fsl_asrc_core_private *asrc_priv =
			params->asrc_pair_params->asrc_pair->asrc_priv;
	struct asrc_pair_params *pair_params = params->asrc_pair_params;
	enum asrc_pair_index index = pair_params->index;

	spin_lock_irqsave(&pair->pair_lock, lock_flags);
	err = set_section_state(params, SECTION_OPENED);
	spin_unlock_irqrestore(&pair->pair_lock, lock_flags);

	if (err < 0)
		return err;

	spin_lock_irqsave(&pair->pair_lock, lock_flags);
	/* make sure both sides have closed before disabling the pair */
	err = set_pair_state(pair_params, SECTION_CLOSED);
	params->callback_arg = 0;
	params->callback_func = NULL;
	spin_unlock_irqrestore(&pair->pair_lock, lock_flags);

	spin_lock_irqsave(&asrc_priv->xrun_lock, lock_flags);
	pair_params->xrun_func = NULL;
	pair_params->xrun_data = NULL;
	spin_unlock_irqrestore(&asrc_priv->xrun_lock, lock_flags);

	if (err == 0) {
		dev_dbg(asrc_priv->dev, "%s released pair[%d]\n",
				__func__, index);

		spin_lock_irqsave(&asrc_priv->core_lock, lock_flags);
		/********Disable PAIR*************/
		reg = __raw_readl(asrc_priv->vaddr + ASRC_ASRCTR_REG);
		reg &= ~(1 << (index + 1));
		__raw_writel(reg, asrc_priv->vaddr + ASRC_ASRCTR_REG);
		spin_unlock_irqrestore(&asrc_priv->core_lock, lock_flags);

		spin_lock_irqsave(&pair_params->asrc_pair->pair_lock,
				lock_flags);
		pair_params->pair_hold = 0;
		spin_unlock_irqrestore(&pair_params->asrc_pair->pair_lock,
				lock_flags);

	} else
		dev_dbg(asrc_priv->dev, "%s Waiting on open section for %p\n",
				__func__, pair_params);

	return 0;
}
EXPORT_SYMBOL(asrc_release_pair);

int asrc_prepare(struct asrc_section_params *params)
{
	int err;
	unsigned long lock_flags;
	struct asrc_pair *pair = params->asrc_pair_params->asrc_pair;

	err = mutex_lock_interruptible(&pair->pair_mutex);
	if (err)
		return err;

	spin_lock_irqsave(&pair->pair_lock, lock_flags);
	err = set_section_state(params, SECTION_PREPARE);
	if (!err) {
		/* reset pair stats */
		pair->stats.aool =
		pair->stats.aodo =
		pair->stats.aiol =
		pair->stats.aidu =
		pair->stats.aodf =
		pair->stats.aide = 0;
	}
	spin_unlock_irqrestore(&pair->pair_lock, lock_flags);

	if (err == 0)
		err = configure_pair(params);
	mutex_unlock(&pair->pair_mutex);

	return err;
}
EXPORT_SYMBOL(asrc_prepare);

/* use with caution */
int asrc_setup(struct asrc_section_params *params)
{
	int err;
	unsigned long lock_flags;
	struct asrc_pair *pair = params->asrc_pair_params->asrc_pair;

	spin_lock_irqsave(&pair->pair_lock, lock_flags);
	err = set_section_state(params, SECTION_SETUP);
	spin_unlock_irqrestore(&pair->pair_lock, lock_flags);

	return err;
}
EXPORT_SYMBOL(asrc_setup);

int asrc_xrun(struct asrc_section_params *params)
{
	int err;
	struct asrc_pair *pair = params->asrc_pair_params->asrc_pair;
	unsigned long lock_flags;

	spin_lock_irqsave(&pair->pair_lock, lock_flags);
	err = set_pair_state(&pair->params, SECTION_SOFT_XRUN);
	spin_unlock_irqrestore(&pair->pair_lock, lock_flags);

	return err;
}
EXPORT_SYMBOL(asrc_xrun);

static int get_word_width_from_format(u64 format)
{
	switch (format) {
	case ASRC_SND_FORMAT_S8:
		return 8;
	case ASRC_SND_FORMAT_S16_LE:
		return 16;
	case ASRC_SND_FORMAT_S24_LE:
	case ASRC_SND_FORMAT_S32_LE:
		return 32;
	};

	return 0;
}

static int get_alignment_from_format(u64 format)
{
	if (format == ASRC_SND_FORMAT_S32_LE)
		return MSB_ALIGNED;

	/* everything else is LSB aligned */
	return LSB_ALIGNED;
}

unsigned int
asrc_get_frame_bytewidth(struct asrc_section_params *params)
{
	return asrc_get_format_data_bytewidth(params) * params->channels;
}
EXPORT_SYMBOL(asrc_get_frame_bytewidth);

unsigned int
asrc_get_format_data_bytewidth(struct asrc_section_params *params)
{
	unsigned int bitwidth;
	unsigned long lock_flags;
	struct asrc_pair *pair = params->asrc_pair_params->asrc_pair;

	spin_lock_irqsave(&pair->pair_lock, lock_flags);
	bitwidth = get_word_width_from_format(params->format);
	spin_unlock_irqrestore(&pair->pair_lock, lock_flags);

	return ((bitwidth >> 3) + ((bitwidth & 0x00000007) > 0 ? 1 : 0));
}
EXPORT_SYMBOL(asrc_get_format_data_bytewidth);

int asrc_get_channels(struct asrc_section_params *params)
{
	int ret;
	unsigned long lock_flags;
	struct asrc_pair *pair = params->asrc_pair_params->asrc_pair;

	spin_lock_irqsave(&pair->pair_lock, lock_flags);
	ret = params->channels;
	spin_unlock_irqrestore(&pair->pair_lock, lock_flags);

	return ret;
}
EXPORT_SYMBOL(asrc_get_channels);

static int configure_pair(struct asrc_section_params *params)
{
	struct fsl_asrc_core_private *asrc_priv =
			params->asrc_pair_params->asrc_pair->asrc_priv;
	int err = 0;
	unsigned int reg, tmp, tmp2, word_width;
	enum asrc_pair_index index = params->asrc_pair_params->index;
	unsigned int channel_num, channel_bits;
	unsigned long lock_flags, bitclk_in, bitclk_out;
	struct asrc_pair_params *pair_params = params->asrc_pair_params;
	struct asrc_section_params *in_params =
			&pair_params->section[SECTION_INPUT];
	struct asrc_section_params *out_params =
			&pair_params->section[SECTION_OUTPUT];

	/* dereference clock mux id for in and out clocks */
	unsigned char in_clk = in_params->clock_ref->clk_mux_id;
	unsigned char out_clk = out_params->clock_ref->clk_mux_id;
	unsigned char in_alignment =
			get_alignment_from_format(in_params->format);
	unsigned char out_alignment =
			get_alignment_from_format(out_params->format);

	dev_dbg(asrc_priv->dev, "%s configuring pair[%d]\n", __func__, index);
	dev_dbg(asrc_priv->dev, "%s In clock: %s\n", __func__,
			in_params->clock_ref->ideal_clk_name);
	dev_dbg(asrc_priv->dev, "%s Out clock: %s\n", __func__,
			out_params->clock_ref->ideal_clk_name);
	dev_dbg(asrc_priv->dev, "%s In samplerate: %d\n",
			__func__, in_params->rate);
	dev_dbg(asrc_priv->dev, "%s Out samplerate: %d\n",
			__func__, out_params->rate);
	dev_dbg(asrc_priv->dev, "%s In channels: %d\n",
			__func__, in_params->channels);
	dev_dbg(asrc_priv->dev, "%s Out channels: %d\n",
			__func__, out_params->channels);
	dev_dbg(asrc_priv->dev, "%s vaddr: %p\n",
			__func__, asrc_priv->vaddr);

	/* Set the channel number */
	channel_bits = asrc_priv->channel_bits;

	channel_num = (params->direction == SECTION_INPUT) ?
		in_params->channels : out_params->channels;

	if (channel_bits <= 3)
		channel_num = (channel_num + 1) / 2;

	tmp = channel_num << (channel_bits * index);

	spin_lock_irqsave(&asrc_priv->core_lock, lock_flags);
	reg = __raw_readl(asrc_priv->vaddr + ASRC_ASRCNCR_REG);

	tmp2 = ((0xFFFFFFFF >> (32 - channel_bits)) << (channel_bits * index));
	/* Write the register only if it is different from what's already set */
	if ((reg & tmp2) != tmp) {
		reg &= ~tmp2;
		reg |= tmp;
		__raw_writel(reg, asrc_priv->vaddr + ASRC_ASRCNCR_REG);
	}
	spin_unlock_irqrestore(&asrc_priv->core_lock, lock_flags);

	dev_dbg(asrc_priv->dev, "%s ASRCNCR:0x%08x\n", __func__, reg);

	/* Set the clock source */
	tmp = ~(0x0f << (index << 2));
	tmp2 = ~(0x0f << (12 + (index << 2)));

	spin_lock_irqsave(&asrc_priv->core_lock, lock_flags);
	reg = __raw_readl(asrc_priv->vaddr + ASRC_ASRCSR_REG);
	reg &= tmp;
	reg &= tmp2;
	reg |= ((in_clk << (index << 2)) |
		(out_clk << (12 + (index << 2))));
	__raw_writel(reg, asrc_priv->vaddr + ASRC_ASRCSR_REG);
	spin_unlock_irqrestore(&asrc_priv->core_lock, lock_flags);

	dev_dbg(asrc_priv->dev, "%s ASRCSR:0x%08x\n", __func__, reg);

	/* pre/post filter selection for processing mode */
	spin_lock_irqsave(&asrc_priv->core_lock, lock_flags);
	reg = __raw_readl(asrc_priv->vaddr + ASRC_ASRCTR_REG);
	/* set manual pre/post processing */
	reg &= ~(0x1 << (20 + index));
	if (in_clk != CLK_NONE) {
		/* use ratio */
		reg &= ~(0x3 << (13 + (index << 1)));
		reg |= (0x1 << (14 + (index << 1)));
	} else {
		/* use ideal ratio */
		reg |= (0x3 << (13 + (index << 1)));
	}
	__raw_writel(reg, asrc_priv->vaddr + ASRC_ASRCTR_REG);
	spin_unlock_irqrestore(&asrc_priv->core_lock, lock_flags);
	dev_dbg(asrc_priv->dev, "%s:ASRCTR:0x%08x\n", __func__, reg);

	err = set_process_configuration(pair_params);

	/* process configuration failed */
	if (err < 0) {
		dev_err(asrc_priv->dev, "%s failed to set process configuration\n",
				__func__);
		return err;
	}

	/* zeroing for compatibility? */
	spin_lock_irqsave(&asrc_priv->core_lock, lock_flags);
	reg = __raw_readl(asrc_priv->vaddr + ASRC_ASRCTR_REG);
	reg = reg & (~(1 << 23));
	__raw_writel(reg, asrc_priv->vaddr + ASRC_ASRCTR_REG);
	spin_unlock_irqrestore(&asrc_priv->core_lock, lock_flags);
	dev_dbg(asrc_priv->dev, "%s:ASRCTR:0x%08x\n", __func__, reg);

	/* get the bit clocks for input/outputs */
	bitclk_in = in_params->clock_ref->bit_clock;
	bitclk_out = out_params->clock_ref->bit_clock;
	if (((in_clk != CLK_NONE) && (bitclk_in == 0))
			|| ((out_clk != CLK_NONE) && (bitclk_out == 0))
			|| (out_clk == CLK_NONE)) {
		dev_err(asrc_priv->dev, "%s invalid input/output bit clock\n",
				__func__);
		return -EINVAL;
	}

	tmp = get_clock_divider(bitclk_in, in_params->rate);
	tmp2 = get_clock_divider(bitclk_out, out_params->rate);

	dev_dbg(asrc_priv->dev,
			"%s clk:in=[target=%u,matched=%lu],"\
			"out=[target=%u,matched=%lu]\n",
			__func__, in_params->rate,
			(bitclk_in / (((tmp >> 3) + 1) *
					(0x1 << (tmp & 0x00000007)))),
			out_params->rate,
			(bitclk_out / (((tmp2 >> 3) + 1) *
					(0x1 << (tmp2 & 0x00000007)))));

	/* set prescaler/divider regardless of mode */
	spin_lock_irqsave(&asrc_priv->core_lock, lock_flags);

	reg = __raw_readl(asrc_priv->vaddr + ASRC_ASRCDR_REG(index));
	/* clear out pair dividers */
	reg &= ASRC_ASRCDR_MASK(index);
	/* Input Part */
	reg |= tmp << CDR_AICP(index);
	/* Output Part */
	reg |= tmp2 << CDR_AOCP(index);
	__raw_writel(reg, asrc_priv->vaddr + ASRC_ASRCDR_REG(index));

	/* Set in/out FIFO data width, alignment, signedness */
	reg = __raw_readl(asrc_priv->vaddr + ASRC_ASRMCR1_REG(index));

	/* clear in/out data alignment settings */
	reg &= ~((0x00000001 << 8) | (0x00000001 << 2));
	reg |= (in_alignment << 8) | (out_alignment << 2);
	/* clear in/out data width settings */
	reg &= ~((0x00000007 << 9) | (0x00000001 << 0));

	word_width = get_word_width_from_format(in_params->format);
	if (word_width >= 24)
		reg |= (DATA_WIDTH_24 << 9);
	else if (word_width >= 16)
		reg |= (DATA_WIDTH_16 << 9);
	else
		reg |= (DATA_WIDTH_8 << 9);

	word_width = get_word_width_from_format(out_params->format);
	if (word_width >= 24)
		reg |= (DATA_WIDTH_24 << 0);
	else
		reg |= (DATA_WIDTH_16 << 0);

	__raw_writel(reg, asrc_priv->vaddr + ASRC_ASRMCR1_REG(index));

	tmp = __raw_readl(asrc_priv->vaddr + ASRC_ASRMCR_REG(index));

	/* clear in/out fifo watermarks */
	tmp &= ~(CCR_SET(MAX_PERIOD,
			MCRX_OUTFIFO_THRESH) | CCR_SET(MAX_PERIOD,
					MCRX_INFIFO_THRESH));

	tmp |= CCR_SET(out_params->hw_params.period,
		MCRX_OUTFIFO_THRESH);
	tmp |= CCR_SET(in_params->hw_params.period,
		MCRX_INFIFO_THRESH);

	__raw_writel(tmp, asrc_priv->vaddr + ASRC_ASRMCR_REG(index));

	spin_unlock_irqrestore(&asrc_priv->core_lock, lock_flags);

	dev_dbg(asrc_priv->dev, "%s ASRMCR1:%08x FORMAT:0x%016llx\n",
			__func__, reg, in_params->format);

	dev_dbg(asrc_priv->dev, "%s ASRMCR:%08x\n", __func__, tmp);

	/* check whether ideal ratio is a must */
	if (in_clk == CLK_NONE) {
		err = set_ideal_ratio(asrc_priv,
					index, in_params->rate,
					out_params->rate);
	}

	if (err < 0)
		dev_err(asrc_priv->dev, "%s failed to set ideal ratio\n",
				__func__);

	return err;
}

/* set channels, format and rate */
int asrc_setup_section(struct asrc_section_params *params,
		struct section_config *config)
{
	struct fsl_asrc_core_private *asrc_priv =
			params->asrc_pair_params->asrc_pair->asrc_priv;
	struct asrc_pair *pair = params->asrc_pair_params->asrc_pair;
	struct asrc_section_capabilities capabilities;
	int err;
	unsigned long lock_flags;

	asrc_get_capabilities(params, &capabilities);

	/* make sure we are not asking for more channels than assigned */
	if ((config->channels > capabilities.channels_max) ||
			(config->channels < capabilities.channels_min)) {
		dev_err(asrc_priv->dev,
				"Channel configuration exceeds channel limit "\
				"allocated for pair: min=%d, max=%d,"\
				"config=%d\n",
				capabilities.channels_min,
				capabilities.channels_max,
				config->channels);
		return -EINVAL;
	}

	/* make sure the format is known and accepted */
	if (!(capabilities.formats & (1ULL << config->format))) {
		dev_err(asrc_priv->dev,
				"ASRC Audio format configuration is not"
				" compatible with the section\n");
		return -EINVAL;
	}

	/* make sure the rate is compatible */
	if ((config->rate > capabilities.rate_max) ||
			(config->rate < capabilities.rate_min)) {
		dev_err(asrc_priv->dev,
				"Rate configuration exceeds rate limit "\
				"allocated for pair: max=%d, config=%d\n",
				capabilities.rate_max,
				config->rate);
		return -EINVAL;
	}

	spin_lock_irqsave(&pair->pair_lock, lock_flags);
	err = set_section_state(params, SECTION_SETUP);
	spin_unlock_irqrestore(&pair->pair_lock, lock_flags);

	if (!err) {
		params->channels = config->channels;
		params->format = config->format;
		params->rate = config->rate;
	}

	return err;
}
EXPORT_SYMBOL(asrc_setup_section);

int asrc_start_conversion(struct asrc_section_params *params)
{
	int reg = 0, err = 0;
	struct asrc_pair_params *pair_params =
			&params->asrc_pair_params->asrc_pair->params;
	enum asrc_pair_index index = pair_params->index;
	struct fsl_asrc_core_private *asrc_priv =
			params->asrc_pair_params->asrc_pair->asrc_priv;
	unsigned long lock_flags;
	struct asrc_pair *pair = params->asrc_pair_params->asrc_pair;
	ktime_t ktime_int;

	spin_lock_irqsave(&pair->pair_lock, lock_flags);
	err = set_section_state(params, SECTION_READY);
	spin_unlock_irqrestore(&pair->pair_lock, lock_flags);
	if (err < 0)
		return err;

	/* check the state on both sides*/
	spin_lock_irqsave(&pair->pair_lock, lock_flags);
	err = set_pair_state(pair_params, SECTION_RUN);
	spin_unlock_irqrestore(&pair->pair_lock, lock_flags);
	if (err == 0) {
		/* setup our startup timer */
		dev_dbg(asrc_priv->dev, "enabling startup timer for pair %d",
			index);
		pair->xrun_enabled = false;
		ktime_int = ktime_set(0, STARTUP_TIME_NS);
		hrtimer_start(&pair->startup_timer,
					ktime_int, HRTIMER_MODE_REL);

		/* enable pair */
		spin_lock_irqsave(&asrc_priv->core_lock, lock_flags);
		reg = __raw_readl(asrc_priv->vaddr + ASRC_ASRCTR_REG);
		reg |= (1 << (1 + index));
		__raw_writel(reg, asrc_priv->vaddr + ASRC_ASRCTR_REG);

		/* wait until pair is initialized */
		reg = __raw_readl(asrc_priv->vaddr + ASRC_ASRCFG_REG);
		while (!(reg & (1 << (index + 21))))
			reg = __raw_readl(asrc_priv->vaddr +
					ASRC_ASRCFG_REG);
		spin_unlock_irqrestore(&asrc_priv->core_lock, lock_flags);
	}

	dev_dbg(asrc_priv->dev, "%s pair[%d] conversion %s: ASRC_ASRCFG:%08x\n",
			__func__, index, (err < 0) ?
					"attempt failed" : "started", reg);

	return 0;
}
EXPORT_SYMBOL(asrc_start_conversion);

int asrc_stop_conversion(struct asrc_section_params *params)
{
	int err;
	unsigned long reg;
	struct asrc_pair_params *pair_params =
				&params->asrc_pair_params->asrc_pair->params;
	enum asrc_pair_index index = pair_params->index;
	struct fsl_asrc_core_private *asrc_priv =
			params->asrc_pair_params->asrc_pair->asrc_priv;
	struct asrc_pair *pair = params->asrc_pair_params->asrc_pair;
	unsigned long lock_flags;

	hrtimer_cancel(&pair->startup_timer);

	spin_lock_irqsave(&pair->pair_lock, lock_flags);
	/* check to see if we are stopping a running pair */
	err = compare_pair_state(pair_params, SECTION_RUN);
	if (err == 0) {
		/* if so bring the pair to setup condition */
		err = set_pair_state(pair_params, SECTION_SETUP);
	} else {
		/* if not bring the section to setup condition */
		err = set_section_state(params, SECTION_SETUP);
	}
	spin_unlock_irqrestore(&pair->pair_lock, lock_flags);

	if (err == 0) {
		spin_lock_irqsave(&asrc_priv->core_lock, lock_flags);
		reg = __raw_readl(asrc_priv->vaddr + ASRC_ASRCTR_REG);
		reg &= ~(1 << (1 + index));
		__raw_writel(reg, asrc_priv->vaddr + ASRC_ASRCTR_REG);
		spin_unlock_irqrestore(&asrc_priv->core_lock, lock_flags);
	}

	return err;
}
EXPORT_SYMBOL(asrc_stop_conversion);

int asrc_get_dma_request(struct asrc_section_params *params)
{
	/* data will not change from probe time and does not need a mutex */
	/* return dma request line */
	return params->hw_params.dma_irq;
}
EXPORT_SYMBOL(asrc_get_dma_request);

static void dump_stats(struct asrc_pair *asrc_pair)
{
	dev_err(asrc_pair->asrc_priv->dev,
			"\tstats pair[%d]\n"
			"\taool:%u\n" /*Pair Output Task Overload*/
			"\taiol:%u\n" /*Pair Input Task Overload*/
			"\taodo:%u\n" /*Output Data Buffer has overflowed*/
			"\taidu:%u\n" /*Input Data Buffer has underflowed*/
			"\taodf:%u\n" /* Output Data Buffer A > threshold */
			"\taide:%u\n", /* Input Data Buffer A < threshold*/
			asrc_pair->params.index,
			asrc_pair->stats.aool,
			asrc_pair->stats.aiol,
			asrc_pair->stats.aodo,
			asrc_pair->stats.aidu,
			asrc_pair->stats.aodf,
			asrc_pair->stats.aide);
}

/**
* Creates the clock table
*
* @pdev: the platfrom device
* @core_private_data: the asrc core private data
*
* @return the error value
*/
static int setup_clock_table(struct platform_device *pdev,
		struct fsl_asrc_core_private *core_private_data)
{
	struct device_node *np = pdev->dev.of_node;
	int ret = 0;
	struct property *prop;
	struct asrc_ideal_clock_entry *clk_table;
	int num_muxids, num_names, num_rates, i, len;
	u32 *muxid_values = NULL, *rate_values = NULL;

	/* check values are present */
	prop = of_find_property(np, "clock-table-muxid", &len);

	/* get the count */
	if (NULL == prop)
		num_muxids = 0;
	else
		num_muxids = len/sizeof(u32);

	prop = of_find_property(np, "clock-table-rate", &len);

	/* get the count */
	if (NULL == prop)
		num_rates = 0;
	else
		num_rates = len/sizeof(u32);

	num_names = of_property_count_strings(np, "clock-table-name");
	/* get the count */
	if (num_names < 0)
		num_names = 0;

	dev_info(&pdev->dev, "'clock-table-muxid' count = (%d)\n",
			num_muxids);
	dev_info(&pdev->dev, "'clock-table-rate' count = (%d)\n",
			num_rates);
	dev_info(&pdev->dev, "'clock-table-name' count = (%d)\n",
			num_names);

	if (num_muxids != num_rates) {
		dev_err(&pdev->dev,
				"mis-match in count of values for "
				"'clock-table-rate' (%d)\n",
				num_rates);
		goto err_invalid_table;
	}

	if (num_muxids != num_names) {
		dev_err(&pdev->dev,
				"mis-match in count of values for "
				"'clock-table-name' (%d)\n",
				num_names);
		goto err_invalid_table;
	}

	/* allocate our clock table structure
	 * NOTE: (+2) for entries for NONE and ASRCK1 clock */
	clk_table = kzalloc(
			sizeof(struct asrc_ideal_clock_entry)*(num_muxids+2),
			GFP_KERNEL);

	if (NULL == clk_table) {
		dev_err(&pdev->dev,
				"memory allocation for clock table failed\n");
		return -ENOMEM;
	}

	/* read values and fill the clock_table */
	if (num_muxids > 0) {
		/* read clock-muxid values */
		muxid_values = kzalloc(sizeof(u32)*num_muxids, GFP_KERNEL);

		if (NULL == muxid_values) {
			dev_err(&pdev->dev, "memory allocation failed\n");
			goto err_alloc_failed;
		}

		ret = of_property_read_u32_array(np, "clock-table-muxid",
				muxid_values, num_muxids);
		if (ret) {
			dev_err(&pdev->dev,
					"error (%d) reading 'clock-table-muxid'"
					" values\n", ret);
			goto err_data_muxid;
		}

		/* read clock-rate values */
		rate_values = kzalloc(sizeof(u32)*num_muxids, GFP_KERNEL);

		if (NULL == rate_values) {
			dev_err(&pdev->dev, "memory allocation failed\n");
			goto err_alloc_failed;
		}

		ret = of_property_read_u32_array(np, "clock-table-rate",
				rate_values, num_muxids);
		if (ret) {
			dev_err(&pdev->dev,
					"error (%d) parsing 'clock-table-rate' "
					"values\n", ret);
			goto err_data_rate;
		}

		/* fill the clock table */
		for (i = 0; i < num_muxids; i++) {
			const char *clk_name;

			clk_table[i].clk_mux_id =
					(enum asrc_clk) muxid_values[i];
			clk_table[i].bit_clock = rate_values[i];

			ret = of_property_read_string_index(np,
					"clock-table-name",
					i, &clk_name);

			if (0 > ret) {
				dev_err(&pdev->dev,
					"error (%d) parsing "
					"'clock-table-name' values\n", ret);
				goto err_data_name;
			}

			clk_table[i].ideal_clk_name = (char *)clk_name;
		}

		kfree(rate_values);
		kfree(muxid_values);
	}

	/* + 2 entries for NONE and ASRCK1 clock */
	num_muxids += 2;

	clk_table[num_muxids-TABLE_POSITION_CLK_NONE].clk_mux_id = CLK_NONE;
	clk_table[num_muxids-TABLE_POSITION_CLK_NONE].bit_clock = 0;
	clk_table[num_muxids-TABLE_POSITION_CLK_NONE].ideal_clk_name =
			ASRC_CLK_NONE;
	clk_table[num_muxids-TABLE_POSITION_CLK_ASRCK1].clk_mux_id = CLK_ASRCK1;
	clk_table[num_muxids-TABLE_POSITION_CLK_ASRCK1].bit_clock = 0;
	clk_table[num_muxids-TABLE_POSITION_CLK_ASRCK1].ideal_clk_name =
			ASRC_CLK_ASRCK1;

	/* store the clk_table */
	core_private_data->ideal_table_length = num_muxids;
	core_private_data->ideal_clock_table = clk_table;

	if (num_muxids > MAX_IDEAL_CLOCKS) {
		dev_err(&pdev->dev,
			"DT clock table entry count exceeds max entries.\n");
			goto err_data_name;
	}

	for (i = 0; i < num_muxids; i++) {
		core_private_data->clock_text[i] = clk_table[i].ideal_clk_name;
		dev_info(&pdev->dev, "clk_entry[%d] <%d> <%lu> \"%s\"\n",
				i, clk_table[i].clk_mux_id,
				clk_table[i].bit_clock,
				clk_table[i].ideal_clk_name);
	}

	return 0;

err_data_name:
err_data_rate:
	kfree(rate_values);

err_data_muxid:
	kfree(muxid_values);

err_alloc_failed:
	kfree(clk_table);

err_invalid_table:
	return -EINVAL;
}

static void
handle_pair(struct asrc_pair *asrc_pair, unsigned long status)
{
	struct asrc_pair_params *params = &asrc_pair->params;
	unsigned long reg;
	unsigned long lock_flags;
	unsigned int trigger_xrun = 0;
	ktime_t ktime_int;

	if (status & ASRC_ASRSTR_AOOL(params->index))
		asrc_pair->stats.aool++;

	if (status & ASRC_ASRSTR_AIOL(params->index))
		asrc_pair->stats.aiol++;

	if (status & ASRC_ASRSTR_AODF(params->index))
		asrc_pair->stats.aodf++;

	if (status & ASRC_ASRSTR_AIDE(params->index))
		asrc_pair->stats.aide++;

	/* check for overload error */
	if ((asrc_pair->stats.aool > IRQ_OVERLOAD_THRSHLD) ||
			(asrc_pair->stats.aiol > IRQ_OVERLOAD_THRSHLD)) {
		dump_stats(asrc_pair);

		spin_lock_irqsave(&asrc_pair->asrc_priv->core_lock,
				lock_flags);
		/* disable problem pair */
		reg = __raw_readl(asrc_pair->asrc_priv->vaddr +
				ASRC_ASRCTR_REG);
		reg &= ~(1UL << (params->index + 1));
		__raw_writel(reg, asrc_pair->asrc_priv->vaddr +
				ASRC_ASRCTR_REG);
		spin_unlock_irqrestore(&asrc_pair->asrc_priv->core_lock,
				lock_flags);

		/* set state to IRQ ERROR */
		spin_lock_irqsave(&asrc_pair->pair_lock, lock_flags);
		set_pair_state(params, SECTION_IRQ_ERR);
		spin_unlock_irqrestore(&asrc_pair->pair_lock, lock_flags);
	}

	if (!(status & ASRC_ASRSTR_AODO(params->index)) &&
		!(status & ASRC_ASRSTR_AIDU(params->index)))
		return;

	hrtimer_cancel(&asrc_pair->startup_timer);

	if (!asrc_pair->xrun_enabled) {
		ktime_int = ktime_set(0, STARTUP_TIME_NS);
		hrtimer_start(&asrc_pair->startup_timer,
				ktime_int, HRTIMER_MODE_REL);
		return;
	}

	if (status & ASRC_ASRSTR_AODO(params->index))
		asrc_pair->stats.aodo++;

	if (status & ASRC_ASRSTR_AIDU(params->index))
		asrc_pair->stats.aidu++;

	spin_lock_irqsave(&asrc_pair->pair_lock, lock_flags);
	if ((asrc_pair->stats.aodo > params->xrun_threshold) ||
		(asrc_pair->stats.aidu > params->xrun_threshold)) {
		/* set state to HARDWARE XRUN */
		set_pair_state(params, SECTION_HARD_XRUN);
		trigger_xrun = 1;
	}
	spin_unlock_irqrestore(&asrc_pair->pair_lock, lock_flags);

	if (trigger_xrun) {
		/* the xrun callback must NOT sleep */
		spin_lock_irqsave(&asrc_pair->asrc_priv->xrun_lock,
				lock_flags);
		if (!WARN_ON(!params->xrun_func))
			params->xrun_func(params->xrun_data);
		spin_unlock_irqrestore(&asrc_pair->asrc_priv->xrun_lock,
				lock_flags);
	}
}

static irqreturn_t asrc_isr(int irq, void *dev_data)
{
	struct fsl_asrc_core_private *asrc = dev_data;
	unsigned long status;
	int reg = 0x40, i;

	/* ASRC_ASRSTR_REG is only used here, so no need for core_lock.  */
	status = __raw_readl(asrc->vaddr + ASRC_ASRSTR_REG);

	if (status & ASRC_ASRSTR_AOLE)    /*Overload Error Flag*/
		asrc->stats.aole++;
	if (status & ASRC_ASRSTR_ATQOL)	  /*Task Queue FIFO overload*/
		asrc->stats.atqol++;
	if (status & ASRC_ASRSTR_FPWT)	  /*FP is in wait states*/
		asrc->stats.fpwt++;
	if (status & ASRC_ASRSTR_DSLCNT)  /*DSL Counter Input to FIFO ready*/
		asrc->stats.dslcnt++;

	for (i = 0; i < ASRC_PAIR_NUM; i++) {
		if (asrc->asrc_pair[i].params.pair_hold)
			handle_pair(&asrc->asrc_pair[i], status);
	}

	/* try to clean the overload error  */
	__raw_writel(reg, asrc->vaddr + ASRC_ASRSTR_REG);

	return IRQ_HANDLED;
}

/* returns the input/output physical register address by pair index */
dma_addr_t asrc_get_fifo_addr(struct asrc_section_params *params)
{
	/* data will not change from probe time and does not need a mutex */
	/* return dma request line */
	if (!params)
		return -EINVAL;

	return params->hw_params.fifo_addr;
}
EXPORT_SYMBOL(asrc_get_fifo_addr);

static int asrc_init(struct fsl_asrc_core_private *asrc,
			 struct platform_device *pdev)
{
	unsigned long reg;
	unsigned int channel_bits = asrc->channel_bits;
	struct asrc_pair_params *pair_params;

	/* we do not have asrc->dev yet here */
	dev_dbg(&pdev->dev, "%s initializing asrc\n", __func__);

	/* disable asrc */
	__raw_writel(0x0, asrc->vaddr + ASRC_ASRCTR_REG);

	/* Disable all interrupt */
	__raw_writel(0x00, asrc->vaddr + ASRC_ASRIER_REG);

	/* Parameter Registers recommended settings must be
	 *  set before asrc is enabled */
	__raw_writel(0x7fffff, asrc->vaddr + ASRC_ASRPM1_REG);
	__raw_writel(0x255555, asrc->vaddr + ASRC_ASRPM2_REG);
	__raw_writel(0xff7280, asrc->vaddr + ASRC_ASRPM3_REG);
	__raw_writel(0xff7280, asrc->vaddr + ASRC_ASRPM4_REG);
	__raw_writel(0xff7280, asrc->vaddr + ASRC_ASRPM5_REG);

	/* enable asrc */
	__raw_writel(0x1, asrc->vaddr + ASRC_ASRCTR_REG);

	/* configured based on DT settings */
	__raw_writel(
		(asrc->asrc_pair[2].capabilities.channels_max <<
				channel_bits * 2) |
		(asrc->asrc_pair[1].capabilities.channels_max << channel_bits) |
		asrc->asrc_pair[0].capabilities.channels_max, asrc->vaddr +
		ASRC_ASRCNCR_REG);

	/* Init Task Queue FIFO */
	__raw_writel(0x001f00, asrc->vaddr + ASRC_ASRTFR1);

	/* set thresholds */
	pair_params = &asrc->asrc_pair[0].params;
	reg = CCR_SET(0x1, MCRX_EXTTHRSH);
	reg &= ~CCR_SET(0x1, MCRX_ZEROBUFF);
	reg |= CCR_SET(pair_params->section[SECTION_OUTPUT].hw_params.period,
		MCRX_OUTFIFO_THRESH);
	reg |= CCR_SET(pair_params->section[SECTION_INPUT].hw_params.period,
		MCRX_INFIFO_THRESH);

	__raw_writel(reg, asrc->vaddr + ASRC_ASRMCRA_REG);
	__raw_writel(reg, asrc->vaddr + ASRC_ASRMCRB_REG);
	__raw_writel(reg, asrc->vaddr + ASRC_ASRMCRC_REG);

	/* Set the processing clock for 76KHz, 133M  */
	/* 133 Mhz = 0x6d6, 200 Mhz it should be 0xa47 */
	/* 60 Mhz = 0x30E, 66 Mhz = 0x35A */
	__raw_writel(0x6d6, asrc->vaddr + ASRC_ASR76K_REG);

	/* Set the processing clock for 56KHz, 133M */
	/* 133Mhz = 0x0947, 200Mhz = 0x0DF3 */
	__raw_writel(0x0947, asrc->vaddr + ASRC_ASR56K_REG);

	/* enable interrupts */
	__raw_writel(ASRIER_FLAGS, asrc->vaddr + ASRC_ASRIER_REG);

	return 0;
}

int asrc_invalidate_ideal_clock(struct fsl_asrc_core_private *asrc_priv,
		const char *streamname, int stream_direction)
{
	unsigned long lock_flags;

	char *ideal_clk_name = NULL;
	int ret;
	struct asrc_ideal_clock_entry *clk_entry;

	ret = asrc_get_ideal_clkname_from_stream(
			streamname, &ideal_clk_name, stream_direction);

	if (ret < 0) {
		dev_err(asrc_priv->dev,
			"%s: failed to generate ideal clock name "
			"from stream.\n",
				__func__);
		return ret;
	}

	clk_entry = locate_ideal_clock_entry(asrc_priv, ideal_clk_name);

	if (clk_entry) {
		spin_lock_irqsave(&asrc_priv->core_lock, lock_flags);
		clk_entry->ideal_clk_cnt--;
		if (clk_entry->ideal_clk_cnt == 0)
			clk_entry->bit_clock = ASRC_INVALID_RATE;
		spin_unlock_irqrestore(&asrc_priv->core_lock, lock_flags);
		if (clk_entry->ideal_clk_cnt < 0) {
			dev_crit(asrc_priv->dev,
				"%s: %s clock reference counter negative. Please provide traceback to kernel vendor.\n",
				__func__, ideal_clk_name);
			BUG();
		}
	} else {
		dev_warn(asrc_priv->dev, "%s: '%s' ideal clock not found.\n",
				__func__, ideal_clk_name);
		ret = -EINVAL;
	}

	kfree(ideal_clk_name);
	ideal_clk_name = NULL;

	return ret;
}
EXPORT_SYMBOL_GPL(asrc_invalidate_ideal_clock);

int asrc_set_ideal_clock_rate(struct fsl_asrc_core_private *asrc_priv,
		const char *ideal_clk_name,
		unsigned int clock_rate)
{
	unsigned long lock_flags;
	struct asrc_ideal_clock_entry *clk_entry =
			locate_ideal_clock_entry(asrc_priv, ideal_clk_name);
	int ret = -EINVAL;

	if (clk_entry) {
		spin_lock_irqsave(&asrc_priv->core_lock, lock_flags);
		if ((clk_entry->bit_clock == ASRC_INVALID_RATE) ||
				(clk_entry->bit_clock == clock_rate)) {
			clk_entry->ideal_clk_cnt++;
			clk_entry->bit_clock = clock_rate;
			ret = 0;
		}
		spin_unlock_irqrestore(&asrc_priv->core_lock, lock_flags);
	} else {
		dev_warn(asrc_priv->dev, "%s: '%s' ideal clock not found.\n",
				__func__, ideal_clk_name);
	}

	return ret;
}
EXPORT_SYMBOL_GPL(asrc_set_ideal_clock_rate);

unsigned int asrc_get_ideal_clock_rate(struct asrc_section_params *params,
		const char *ideal_clk_name)
{
	struct fsl_asrc_core_private *asrc_priv =
			params->asrc_pair_params->asrc_pair->asrc_priv;
	unsigned long lock_flags;
	unsigned int clk_rate;
	struct asrc_ideal_clock_entry const *clk_entry =
			locate_ideal_clock_entry(asrc_priv, ideal_clk_name);

	if (clk_entry) {
		spin_lock_irqsave(&asrc_priv->core_lock, lock_flags);
		clk_rate = clk_entry->bit_clock;
		spin_unlock_irqrestore(&asrc_priv->core_lock, lock_flags);
		return 0;
	}

	return -EINVAL;
}
EXPORT_SYMBOL_GPL(asrc_get_ideal_clock_rate);

int asrc_get_base_streamname(const char *streamname,
		char const **basename)
{
	int strlength;

	strlength = strnlen(streamname, ASRC_MAX_CLKNAME_LEN);

	/* determine if we have a FE or BE prefix and strip it off */
	if (strlength > 0)
		if (!strncmp(streamname, BE_PREFIX, strlen(BE_PREFIX))) {
			*basename = streamname + strlen(BE_PREFIX);
			strlength -= strlen(BE_PREFIX);
		} else if (!strncmp(streamname, FE_PREFIX, strlen(FE_PREFIX))) {
			*basename = streamname + strlen(FE_PREFIX);
			strlength -= strlen(FE_PREFIX);
		} else
			*basename = streamname;
	else
		strlength = -1;

	return strlength;
}
EXPORT_SYMBOL_GPL(asrc_get_base_streamname);

/* returns valid streamname pointer or NULL on failure
 *
 * Note: It is the job of the caller to free the pointer after usage
 */
int asrc_get_ideal_clkname_from_stream(const char const *streamname,
		char **clk_name,
		int stream_direction)
{
	const char const *basename = NULL;
	int strlength;

	strlength = asrc_get_base_streamname(streamname, &basename);

	if (strlength < 0)
		return -EINVAL;
	else if (stream_direction == ASRC_PLAYBACK)
		strlength += (strlen(IDEAL_PLAYBACK_PREFIX) + 1);
	else
		strlength += (strlen(IDEAL_CAPTURE_PREFIX) + 1);

	/* allocated basename + prefix length */
	if (clk_name && (*clk_name == NULL))
		*clk_name = kzalloc(strlength, GFP_ATOMIC);
	else
		return -EINVAL;

	if (!(*clk_name))
		return -ENOMEM;

	if (stream_direction == ASRC_PLAYBACK)
		snprintf(*clk_name, strlength,
				"%s%s", IDEAL_PLAYBACK_PREFIX, basename);
	else
		snprintf(*clk_name, strlength,
				"%s%s", IDEAL_CAPTURE_PREFIX, basename);

	return 0;
}
EXPORT_SYMBOL_GPL(asrc_get_ideal_clkname_from_stream);

#define ASRIER_SHOW(name) \
	sprintf(buf, "%u\n", asrc_priv->stats.name)

#define ASRIER_PAIR_SHOW(name) \
	sprintf(buf, "%u\n", asrc_pair->stats.name)

#define REG_SHOW(name) \
	sprintf(buf, "0x%08x\n",\
			__raw_readl(asrc_priv->vaddr + name))

/**
 * sysfs_dsl_show() display dsl task counts
 * @dev		struct device device pointer
 * @attr	struct device_attribute
 * @buf		char * output buffer pointer
 */
static ssize_t sysfs_dsl_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct fsl_asrc_core_private *asrc_priv = dev_get_drvdata(dev);

	return ASRIER_SHOW(dslcnt);	/*DSL Counter Input to FIFO ready*/
}

/**
 * sysfs_queue_ol_show() display task queue overload state
 * @dev		struct device device pointer
 * @attr	struct device_attribute
 * @buf		char * output buffer pointer
 */
static ssize_t sysfs_queue_ol_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct fsl_asrc_core_private *asrc_priv = dev_get_drvdata(dev);

	return ASRIER_SHOW(atqol);	/*Task Queue FIFO overload*/
}

/**
 * sysfs_reg_ctr_show() display the control register in raw hex
 * @dev		struct device device pointer
 * @attr	struct device_attribute
 * @buf		char * output buffer pointer
 */
static ssize_t sysfs_reg_ctr_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct fsl_asrc_core_private *asrc_priv = dev_get_drvdata(dev);

	return REG_SHOW(ASRC_ASRCTR_REG);	/* ASRC Control Register */
}

/**
 * sysfs_reg_str_show() display the control register in raw hex
 * @dev		struct device device pointer
 * @attr	struct device_attribute
 * @buf		char * output buffer pointer
 */
static ssize_t sysfs_reg_str_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct fsl_asrc_core_private *asrc_priv = dev_get_drvdata(dev);

	return REG_SHOW(ASRC_ASRSTR_REG);	/* ASRC Status Register */
}

/**
 * sysfs_reg_ier_show() display interrupt register in raw hex
 * @dev		struct device device pointer
 * @attr	struct device_attribute
 * @buf		char * output buffer pointer
 */
static ssize_t sysfs_reg_ier_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct fsl_asrc_core_private *asrc_priv = dev_get_drvdata(dev);
	/* ASRC Interrupt Enable Register */
	return REG_SHOW(ASRC_ASRIER_REG);
}

/**
 * sysfs_out_olcnt_a_show() display output task overload count
 * @dev		struct device device pointer
 * @attr	struct device_attribute
 * @buf		char * output buffer pointer
 */
static ssize_t sysfs_out_olcnt_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct asrc_pair *asrc_pair = dev_get_drvdata(dev);
	/*Pair Output Task Overload*/
	return ASRIER_PAIR_SHOW(aool);

}

/**
 * sysfs_in_olcnt_show() display input task overloads
 * @dev		struct device device pointer
 * @attr	struct device_attribute
 * @buf		char * output buffer pointer
 */
static ssize_t sysfs_in_olcnt_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct asrc_pair *asrc_pair = dev_get_drvdata(dev);
	/*Pair Input Task Overload*/
	return ASRIER_PAIR_SHOW(aiol);

}

/**
 * sysfs_out_ofcnt_show() display output buffer overflows
 * @dev		struct device device pointer
 * @attr	struct device_attribute
 * @buf		char * output buffer pointer
 */
static ssize_t sysfs_out_ofcnt_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct asrc_pair *asrc_pair = dev_get_drvdata(dev);
	/*Output Data Buffer has overflowed*/
	return ASRIER_PAIR_SHOW(aodo);
}

/**
 * sysfs_in_ufcnt_show() display input buffer underflows
 * @dev		struct device device pointer
 * @attr	struct device_attribute
 * @buf		char * output buffer pointer
 */
static ssize_t sysfs_in_ufcnt_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct asrc_pair *asrc_pair = dev_get_drvdata(dev);
	/*Input Data Buffer has underflowed*/
	return ASRIER_PAIR_SHOW(aidu);
}

/**
 * sysfs_fp_wait_show() display fp wait status
 * @dev		struct device device pointer
 * @attr	struct device_attribute
 * @buf		char * output buffer pointer
 *
 * This will only display data in the case that the interrupt for
 * waiting is enabled
 */
static ssize_t sysfs_fp_wait_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct fsl_asrc_core_private *asrc_priv = dev_get_drvdata(dev);
	/*FP is in wait states*/
	return ASRIER_SHOW(fpwt);

}

/**
 * sysfs_overload_err_show() display overload error
 * @dev		struct device device pointer
 * @attr	struct device_attribute
 * @buf		char * output buffer pointer
 */
static ssize_t sysfs_overload_err_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct fsl_asrc_core_private *asrc_priv = dev_get_drvdata(dev);
	/*Overload Error Flag*/
	return ASRIER_SHOW(aole);
}

/**
 * sysfs_reg_miscctr_show() display all misc control registers in raw hex
 * @dev		struct device device pointer
 * @attr	struct device_attribute
 * @buf		char * output buffer pointer
 */
static ssize_t sysfs_reg_miscctr_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct asrc_pair *asrc_pair = dev_get_drvdata(dev);
	struct fsl_asrc_core_private *asrc_priv = asrc_pair->asrc_priv;
	/* ASRC Misc Control Register for Pair */
	return REG_SHOW(ASRC_ASRMCR_REG(asrc_pair->params.index));
}

/**
 * sysfs_reg_miscctr1_show() display all misc control registers in raw hex
 * @dev		struct device device pointer
 * @attr	struct device_attribute
 * @buf		char * output buffer pointer
 */
static ssize_t sysfs_reg_miscctr1_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct asrc_pair *asrc_pair = dev_get_drvdata(dev);
	struct fsl_asrc_core_private *asrc_priv = asrc_pair->asrc_priv;
	 /* Misc Control Register1 for Pair */
	return REG_SHOW(ASRC_ASRMCR1_REG(asrc_pair->params.index));
}

/**
 * sysfs_reg_fifostat_show() display fifo status registers in raw hex
 * @dev		struct device device pointer
 * @attr	struct device_attribute
 * @buf		char * output buffer pointer
 */
static ssize_t sysfs_reg_fifostat_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct asrc_pair *asrc_pair = dev_get_drvdata(dev);
	struct fsl_asrc_core_private *asrc_priv = asrc_pair->asrc_priv;
	/* ASRC FIFO Status Register for Pair */
	return REG_SHOW(ASRC_ASRFST_REG(asrc_pair->params.index));
}

/**
 * sysfs_reg_cfg_show() display configuration register in raw hex
 * @dev		struct device device pointer
 * @attr	struct device_attribute
 * @buf		char * output buffer pointer
 */
static ssize_t sysfs_reg_cfg_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct fsl_asrc_core_private *asrc_priv = dev_get_drvdata(dev);

	return REG_SHOW(ASRC_ASRCFG_REG); /* ASRC configuration Register */
}

static const char *find_clkname_from_mux_id(
		struct fsl_asrc_core_private *asrc_priv,
		unsigned char clk_mux_id)
{
	char const *clk_name = NULL;
	int i;

	for (i = 0; i < asrc_priv->ideal_table_length; i++) {
		if (asrc_priv->ideal_clock_table[i].clk_mux_id == clk_mux_id)
			clk_name =
				asrc_priv->ideal_clock_table[i].ideal_clk_name;
	}

	return clk_name;
}

/**
 * sysfs_in_clock_show() display asrc clock information in a human-friendly form
 * @dev		struct device device pointer
 * @attr	struct device_attribute
 * @buf		char * output buffer pointer
 *
 * This displays the set incoming and outgoing clock used by each pair
 */
static ssize_t sysfs_in_clock_show(struct device *dev,
	struct device_attribute *attr, char *buf)
{
	struct asrc_pair *asrc_pair = dev_get_drvdata(dev);
	struct fsl_asrc_core_private *asrc_priv = asrc_pair->asrc_priv;
	char const *clk_name;
	unsigned char clk_mux_id;
	unsigned long reg;

	reg = __raw_readl(asrc_priv->vaddr + ASRC_ASRCSR_REG);

	clk_mux_id = CSR_AICS_GET(asrc_pair->params.index, reg);

	clk_name = find_clkname_from_mux_id(asrc_priv, clk_mux_id);

	return sprintf(buf, "%s\n", (clk_name ? clk_name :
			"couldnt locate in clock\n"));
}

/**
 * sysfs_out_clock_show() display asrc clock information in human-friendly form
 * @dev		struct device device pointer
 * @attr	struct device_attribute
 * @buf		char * output buffer pointer
 *
 * This displays the set incoming and outgoing clock used by each pair as well
 * as the ideal frequency of that clock and ideal ratio if in ideal mode.
 */
static ssize_t sysfs_out_clock_show(struct device *dev,
	struct device_attribute *attr, char *buf)
{
	struct asrc_pair *asrc_pair = dev_get_drvdata(dev);
	struct fsl_asrc_core_private *asrc_priv = asrc_pair->asrc_priv;
	unsigned char clk_mux_id;
	char const *clk_name;
	unsigned long reg;

	reg = __raw_readl(asrc_priv->vaddr + ASRC_ASRCSR_REG);


	clk_mux_id = CSR_AOCS_GET(asrc_pair->params.index, reg);

	clk_name = find_clkname_from_mux_id(asrc_priv, clk_mux_id);

	return sprintf(buf, "%s\n", (clk_name ? clk_name :
			"couldnt locate out clock\n"));
}

/**
 * sysfs_channel_show() display device channel settings
 * @dev		struct device device pointer
 * @attr	struct device_attribute
 * @buf		char * output buffer pointer
 *
 * This displays the current number of channels to use for each pair
 */
static ssize_t sysfs_channel_show(struct device *dev,
	struct device_attribute *attr, char *buf)
{
	struct asrc_pair *asrc_pair = dev_get_drvdata(dev);
	struct fsl_asrc_core_private *asrc_priv = asrc_pair->asrc_priv;

	unsigned long reg;
	unsigned int channel_bits = asrc_priv->channel_bits;

	reg = __raw_readl(asrc_priv->vaddr + ASRC_ASRCNCR_REG);

	return sprintf(buf, "%d\n",
		(int)((reg >> (channel_bits * asrc_pair->params.index)) &
			(0xFFFFFFFF >> (32 - channel_bits))));

}

/**
 * sysfs_channel_a_store() set device channel settings
 * @dev		struct device device pointer
 * @attr	struct device_attribute
 * @buffer	char * output buffer pointer
 * @count	number of characters used from the buffer
 * *
 * One should be careful when using this, as any currently processed data the
 * channel may be affected in an unknown way. The proper use is to stop
 * all conversions before setting this value.
 */
static ssize_t sysfs_channel_store(struct device *dev,
		struct device_attribute *attr,
		const char *buffer, size_t count)
{
	struct asrc_pair *asrc_pair = dev_get_drvdata(dev);
	struct fsl_asrc_core_private *asrc_priv = asrc_pair->asrc_priv;
	unsigned long reg, lock_flags;
	int ns[ASRC_PAIR_NUM] = {0};
	int nin, i;
	int total;
	unsigned int channel_bits;

	if (count > 2)
		return count;

	/* Set the channel number */
	channel_bits = asrc_priv->channel_bits;

	sscanf(buffer, "%d\n", &nin);

	spin_lock_irqsave(&asrc_priv->core_lock, lock_flags);
	reg = __raw_readl(asrc_priv->vaddr + ASRC_ASRCNCR_REG);

	for (i = 0; i < ASRC_PAIR_NUM; i++)
		ns[i] = (int)((reg >> (channel_bits * i)) &
			    (0xFFFFFFFF >> (32 - channel_bits)));
	ns[asrc_pair->params.index] = nin;

	if (asrc_priv->channel_bits > 3)
		total = 10;
	else
		total = 5;
	if ((ns[0] + ns[1] + ns[2]) > total) {
		dev_dbg(dev, "Invalid channel setting: "\
				"A=%d B=%d C=%d (A+B+C>%d)\n",
				ns[0], ns[1], ns[2], total);
	} else {
		reg = ns[0] | (ns[1] << asrc_priv->channel_bits) |
				(ns[2] << (asrc_priv->channel_bits * 2));
		__raw_writel(reg, asrc_priv->vaddr + ASRC_ASRCNCR_REG);
	}
	spin_unlock_irqrestore(&asrc_priv->core_lock, lock_flags);

	return count;
}

/**
 * sysfs_xrun_threshold_show() display pair xrun threshold
 * @dev		struct device device pointer
 * @attr	struct device_attribute
 * @buf		char * output buffer pointer
 *
 * This displays the current xrun threshold for each pair
 */
static ssize_t sysfs_xrun_threshold_show(struct device *dev,
	struct device_attribute *attr, char *buf)
{
	struct asrc_pair *asrc_pair = dev_get_drvdata(dev);
	unsigned long lock_flags;
	int ret;

	spin_lock_irqsave(&asrc_pair->pair_lock, lock_flags);
	ret = sprintf(buf, "%u\n", asrc_pair->params.xrun_threshold);
	spin_unlock_irqrestore(&asrc_pair->pair_lock, lock_flags);

	return ret;
}

/**
 * sysfs_xrun_threshold_store() set pair xrun threshold
 * @dev		struct device device pointer
 * @attr	struct device_attribute
 * @buffer	char * output buffer pointer
 * @count	number of characters used from the buffer
 * *
 */
static ssize_t sysfs_xrun_threshold_store(struct device *dev,
		struct device_attribute *attr,
		const char *buffer, size_t count)
{
	struct asrc_pair *asrc_pair = dev_get_drvdata(dev);
	unsigned long lock_flags;
	unsigned int threshold;

	if (count > 3)
		return count;

	sscanf(buffer, "%u\n", &threshold);

	spin_lock_irqsave(&asrc_pair->pair_lock, lock_flags);
	asrc_pair->params.xrun_threshold = threshold;
	spin_unlock_irqrestore(&asrc_pair->pair_lock, lock_flags);

	return count;
}


static struct device_attribute asrc_pair_attrs[] = {
	/* asrc pair channel settings */
	__ATTR(out_task_overload_cnt,
			ASRC_SYSFS_RO, sysfs_out_olcnt_show, NULL),
	__ATTR(in_task_overload_cnt, ASRC_SYSFS_RO, sysfs_in_olcnt_show, NULL),
	__ATTR(out_data_overflow_cnt,
			ASRC_SYSFS_RO, sysfs_out_ofcnt_show, NULL),
	__ATTR(in_data_underflow_cnt, ASRC_SYSFS_RO, sysfs_in_ufcnt_show, NULL),
	__ATTR(channel,
		ASRC_SYSFS_ACCESS, sysfs_channel_show, sysfs_channel_store),
	__ATTR(xrun_threshold, ASRC_SYSFS_ACCESS, sysfs_xrun_threshold_show,
		sysfs_xrun_threshold_store),

	/* misc control registers */
	__ATTR(reg_miscctr, ASRC_SYSFS_RO, sysfs_reg_miscctr_show, NULL),
	__ATTR(reg_miscctr1, ASRC_SYSFS_RO, sysfs_reg_miscctr1_show, NULL),
	/* fifo status registers */
	__ATTR(reg_fifostat, ASRC_SYSFS_RO, sysfs_reg_fifostat_show, NULL),
	/* pair input output clock settings */
	__ATTR(in_clock, ASRC_SYSFS_RO, sysfs_in_clock_show, NULL),
	__ATTR(out_clock, ASRC_SYSFS_RO, sysfs_out_clock_show, NULL),
	__ATTR_NULL,
};

static struct device_attribute asrc_attrs[] = {
	/* asrc counters and statistics */
	__ATTR(dsl_counter, ASRC_SYSFS_RO, sysfs_dsl_show, NULL),
	__ATTR(fp_wait, ASRC_SYSFS_RO, sysfs_fp_wait_show, NULL),
	__ATTR(overload_err, ASRC_SYSFS_RO, sysfs_overload_err_show, NULL),
	__ATTR(task_queue_overload, ASRC_SYSFS_RO, sysfs_queue_ol_show, NULL),

	/* registers */
	__ATTR(reg_str, ASRC_SYSFS_RO, sysfs_reg_str_show, NULL),
	__ATTR(reg_ctr, ASRC_SYSFS_RO, sysfs_reg_ctr_show, NULL),
	__ATTR(reg_ier, ASRC_SYSFS_RO, sysfs_reg_ier_show, NULL),
	__ATTR(reg_cfg, ASRC_SYSFS_RO, sysfs_reg_cfg_show, NULL),
	__ATTR_NULL,
};


#ifdef CONFIG_DEBUG_FS
static void __iomem *asrc_base;
static struct dentry *asrc_debugfs_root;

static ssize_t asrc_read_pair(struct file *file,
				char __user *user_buf,
				size_t count, loff_t *ppos)
{
	u32 mcr, mcr1;
	ssize_t ret = 0;
	char *buf = kmalloc(PAGE_SIZE, GFP_KERNEL);
	int pair_idx = (int)file->private_data;

	if (!buf)
		return -ENOMEM;

	mcr = __raw_readl(asrc_base +
				ASRC_ASRMCR_REG(pair_idx));
	mcr1 = __raw_readl(asrc_base +
				ASRC_ASRMCR1_REG(pair_idx));

	ret += snprintf(buf + ret, PAGE_SIZE - ret, "ASRC Pair %d\n", pair_idx);

	ret += snprintf(buf + ret, PAGE_SIZE - ret,
			"MCR: %08x -- MCR1: %08x\n", mcr, mcr1);

	ret += snprintf(buf + ret, PAGE_SIZE - ret, "\nSection Input\n");
	ret += snprintf(buf + ret, PAGE_SIZE - ret,
		"FIFO Threshold: %d\n",
		CCR_GET(mcr, MCRX_INFIFO_THRESH));

	ret += snprintf(buf + ret, PAGE_SIZE - ret, "FIFO Alignment: ");
	if (CCR_GET(mcr1, MCR1X_INFIFO_ALIGN) > 0)
		ret += snprintf(buf + ret, PAGE_SIZE - ret, "MSB\n");
	else
		ret += snprintf(buf + ret, PAGE_SIZE - ret, "LSB\n");

	ret += snprintf(buf + ret, PAGE_SIZE - ret,
			"FIFO Data Width (bits): ");
	switch (CCR_GET(mcr1, MCR1X_INFIFO_DATA_SIZE)) {
	case DATA_WIDTH_24:
		ret += snprintf(buf + ret, PAGE_SIZE - ret, "24\n");
		break;
	case DATA_WIDTH_16:
		ret += snprintf(buf + ret, PAGE_SIZE - ret, "16\n");
		break;
	case DATA_WIDTH_8:
		ret += snprintf(buf + ret, PAGE_SIZE - ret, "8\n");
		break;
	default:
		ret += snprintf(buf + ret, PAGE_SIZE - ret, "Unknown\n");
		break;
	}

	ret += snprintf(buf + ret, PAGE_SIZE - ret, "\nSection Output\n");
	ret += snprintf(buf + ret, PAGE_SIZE - ret,
		"FIFO Threshold: %d\n",
		CCR_GET(mcr, MCRX_OUTFIFO_THRESH));

	ret += snprintf(buf + ret, PAGE_SIZE - ret, "FIFO Alignment: ");
	if (CCR_GET(mcr1, MCR1X_OUTFIFO_ALIGN) > 0)
		ret += snprintf(buf + ret, PAGE_SIZE - ret, "MSB\n");
	else
		ret += snprintf(buf + ret, PAGE_SIZE - ret, "LSB\n");

	ret += snprintf(buf + ret, PAGE_SIZE - ret,
		"FIFO Data Width (bits): ");
	switch (CCR_GET(mcr1, MCR1X_OUTFIFO_DATA_SIZE)) {
	case DATA_WIDTH_24:
		ret += snprintf(buf + ret, PAGE_SIZE - ret, "24\n");
		break;
	case DATA_WIDTH_16:
		ret += snprintf(buf + ret, PAGE_SIZE - ret, "16\n");
		break;
	default:
		ret += snprintf(buf + ret, PAGE_SIZE - ret, "Unknown\n");
		break;
	}

	ret += snprintf(buf + ret, PAGE_SIZE - ret,
		"FIFO Sign Extension: ");
	if (CCR_GET(mcr1, MCR1X_OUTFIFO_SIGN) > 0)
		ret += snprintf(buf + ret, PAGE_SIZE - ret, "Signed\n");
	else
		ret += snprintf(buf + ret, PAGE_SIZE - ret, "Unsigned\n");

	ret = simple_read_from_buffer(user_buf, count, ppos, buf, ret);

	kfree(buf);

	return ret;
}

static const struct file_operations asrc_debugfs_fops = {
	.open = simple_open,
	.read = asrc_read_pair,
	.llseek = default_llseek,
};

static const char *debugfs_fname_pair(int pair)
{
	switch (pair) {
	case ASRC_PAIR_A:
		return "pair0_stats";
	case ASRC_PAIR_B:
		return "pair1_stats";
	case ASRC_PAIR_C:
		return "pair2_stats";
	default:
		return "UNKNOWN";
	}
}

static void asrc_debugfs_init(struct fsl_asrc_core_private *asrc_priv)
{
	int i;
	asrc_base = asrc_priv->vaddr;
	asrc_debugfs_root = debugfs_create_dir("asrc", NULL);
	if (!asrc_debugfs_root) {
		pr_warn("Failed to create asrc debugfs root\n");
		return;
	}

	for (i = ASRC_PAIR_A; i < ASRC_PAIR_NUM; i++)
		if (!debugfs_create_file(debugfs_fname_pair(i), 0444,
				asrc_debugfs_root, (void *)i,
				&asrc_debugfs_fops))
			pr_warn("ASRC Pair %d failed to create debugfs file\n",
				i);
}

static void asrc_debugfs_remove(void)
{
	debugfs_remove_recursive(asrc_debugfs_root);
}
#else
static inline void asrc_debugfs_init(struct fsl_asrc_core_private *asrc_priv)
{
}

static inline void asrc_debugfs_remove(void)
{
}
#endif

/* Callback for our startup timer
 * Notice that no locking mechanism is used whereas the
 * variable ‘xrun_enabled’ is accessed from the timer as
 * well as other places, that is because we call
 * hrtimer_cancel(…) which is a blocking call on this timer
 * before accessing ‘xrun_enabled’ from other contexts.
 */
static enum hrtimer_restart startup_timer_cb(struct hrtimer *timer)
{
	struct asrc_pair *pair = container_of(timer,
					      struct asrc_pair, startup_timer);
	dev_dbg(pair->asrc_priv->dev,
		"startup timer overflow, enabling xrun handling for pair %d\n",
		pair->params.index);

	pair->xrun_enabled = true;

	return HRTIMER_NORESTART;
}

static int mxc_asrc_core_probe(struct platform_device *pdev)
{
	int err = 0;
	struct resource res;
	struct clk *audio_clk, *ipg_clk;
	int ret = 0, i, j;
	unsigned int asrc_clk_rate;
	struct asrc_ideal_clock_entry *asrc_clk_entry;
	struct device_node *np = pdev->dev.of_node;
	struct fsl_asrc_core_private *asrc_private;
	struct asrc_pair_params *params;
	struct asrc_pair *pair;
	enum asrc_pair_index idx;

	u32 dma_events[ASRC_PAIR_NUM*2];
	int channel_limits[ASRC_PAIR_NUM];
	int channel_mm_limits[ASRC_PAIR_NUM][CHANNEL_LIMIT_LAST];
	const __be32 *ch_mm_vals;
	u32 ch_mm_len;

	/* If the ASRC is not to be used
	 *      status = "disabled"
	 * property in the device tree node.
	 */
	if (!of_device_is_available(np))
		return -ENODEV;

	/* allocate our device data */
	asrc_private = kzalloc(sizeof(struct fsl_asrc_core_private),
			GFP_KERNEL);

	if (asrc_private == NULL) {
		dev_err(&pdev->dev,
				"Failed to allocate private data\n");
		return -ENOMEM;
	}
	platform_set_drvdata(pdev, asrc_private);

	/* initialize spinlocks */
	spin_lock_init(&asrc_private->core_lock);

	/* read the clock table from device-tree */
	err = setup_clock_table(pdev, asrc_private);
	if (err) {
		dev_err(&pdev->dev, "could not setup ideal clock table\n");
		goto error_kmalloc;
	}

	/* Get the device resources */
	ret = of_address_to_resource(np, 0, &res);
	if (ret) {
		dev_err(&pdev->dev,
				"could not determine device resources\n");
		goto error_kmalloc;
	}

	/* map our phy and virt addresses */
	asrc_private->asrc_phys = res.start;
	dev_dbg(&pdev->dev, "%s resource start address:%lx\n",
		__func__, (unsigned long)asrc_private->asrc_phys);

	asrc_private->vaddr = of_iomap(np, 0);
	if (!asrc_private->vaddr) {
		dev_err(&pdev->dev,
				"could not map device resources\n");
		ret = -ENOMEM;
		goto error_kmalloc;
	}

	dev_dbg(&pdev->dev, "%s mapped virtual address:%lx\n",
		__func__, (unsigned long)asrc_private->vaddr);

	/* parse and fill all the DMA events */
	ret = of_property_read_u32_array(pdev->dev.of_node,
				"fsl,asrc-dma-events",
				dma_events, ASRC_PAIR_NUM*2);
	if (ret) {
		dev_err(&pdev->dev,
				"could not get dma events: %d\n", ret);
		goto error_iomap;
	}

	spin_lock_init(&asrc_private->xrun_lock);

	/* initialize all our pointers */
	for (idx = 0; idx < ASRC_PAIR_NUM; idx++) {
		params = &asrc_private->asrc_pair[idx].params;
		params->asrc_pair = &asrc_private->asrc_pair[idx];
		params->xrun_threshold = IRQ_XRUN_THRSHLD;
		spin_lock_init(&params->asrc_pair->pair_lock);
		mutex_init(&params->asrc_pair->pair_mutex);
		hrtimer_init(&params->asrc_pair->startup_timer,
				CLOCK_MONOTONIC, HRTIMER_MODE_REL);
		params->asrc_pair->startup_timer.function =
							&startup_timer_cb;

		for (j = 0; j < SECTION_CNT; j++) {
			spin_lock_init(&params->section[j].section_lock);
			params->section[j].hw_params.dma_irq =
					dma_events[idx*2+j];
			params->section[j].asrc_pair_params =
					params;
			params->section[j].hw_params.fifo_addr =
					GET_PER_ADDR(asrc_private->asrc_phys,
							j, idx);
			params->section[j].hw_params.period =
					DEFAULT_FIFO_PERIOD;
			params->section[j].state = SECTION_CLOSED;
			params->section[j].direction = j;
		}

		dev_dbg(&pdev->dev,
			"OUTPUT[%d]:%u INPUT[%d]:%u\n",
			idx, dma_events[idx*2],
			idx, dma_events[idx*2+1]);
	}

	/* parse and assign the asrc irq */
	asrc_private->irq = irq_of_parse_and_map(np, 0);
	if (asrc_private->irq == NO_IRQ) {
		dev_err(&pdev->dev, "no irq for node %s\n", np->full_name);
		ret = -ENXIO;
		goto error_iomap;
	}
	/* request the irq */
	ret = request_irq(asrc_private->irq,
			asrc_isr, 0, "asrc", asrc_private);
	if (ret < 0) {
		dev_err(&pdev->dev,
			"could not claim irq %u\n", asrc_private->irq);
		goto error_iomap;
	}

	/* parse and fill all the channel configuration */
	ret = of_property_read_u32_array(pdev->dev.of_node,
			"channels", channel_limits, ASRC_PAIR_NUM);

	/* Get the min-max channel (ch-mm) property */
	ch_mm_vals = of_get_property(pdev->dev.of_node, "ch-mm", &ch_mm_len);

	if (ret < 0) {
		if (ch_mm_vals != NULL)	{
			ch_mm_len /= sizeof(*ch_mm_vals);
			if (ch_mm_len != (ASRC_PAIR_NUM * CHANNEL_LIMIT_LAST)) {
				dev_err(&pdev->dev,
				"wrong number of ch-mm configurations\n");
				ret = -EINVAL;
				goto error_iomap;
			}
			for (i = 0, ret = 0; i < ASRC_PAIR_NUM; i++) {
				ch_mm_len = (i * CHANNEL_LIMIT_LAST);
				channel_mm_limits[i][CHANNEL_LIMIT_MIN] =
						be32_to_cpup(ch_mm_vals +
							     ch_mm_len +
							     CHANNEL_LIMIT_MIN);
				channel_mm_limits[i][CHANNEL_LIMIT_MAX] =
						be32_to_cpup(ch_mm_vals +
							     ch_mm_len +
							     CHANNEL_LIMIT_MAX);
				ret += channel_mm_limits[i][CHANNEL_LIMIT_MAX];
				if ((channel_mm_limits[i][CHANNEL_LIMIT_MIN] <
							   ASRC_CHANNEL_MIN) &&
				    (channel_mm_limits[i][CHANNEL_LIMIT_MAX] >=
							   ASRC_CHANNEL_MIN)) {
					dev_err(&pdev->dev,
					"Pair %d channel min can not be < %d",
					i, ASRC_CHANNEL_MIN);
					ret = -EINVAL;
					goto error_iomap;
				}
			}
		} else {
			dev_info(&pdev->dev,
			"using default channel min/max configuration\n");
			for (i = 0; i < ASRC_PAIR_NUM; i++) {
				channel_mm_limits[i][CHANNEL_LIMIT_MIN] =
				 asrc_channel_mm_defaults[i][CHANNEL_LIMIT_MIN];
				channel_mm_limits[i][CHANNEL_LIMIT_MAX] =
				 asrc_channel_mm_defaults[i][CHANNEL_LIMIT_MAX];
			}
		}
	} else {
		if (ch_mm_vals != NULL)	{
			dev_err(&pdev->dev,
			    "'channels' and 'ch-mm' are mutually exclusive!\n");
			goto error_iomap;
		} else {
			for (i = 0, ret = 0; i < ASRC_PAIR_NUM; i++) {
				ret += channel_limits[i];
				channel_mm_limits[i][CHANNEL_LIMIT_MIN] =
				 channel_mm_limits[i][CHANNEL_LIMIT_MAX] =
				  channel_limits[i];
			}
		}
	}

	if (ret > ASRC_CHANNEL_MAX) {
		ret = -EINVAL;
		dev_err(&pdev->dev, "invalid ch-mm/channels configuration\n");
		dev_err(&pdev->dev,
			"the sum of max configurations should be <= %d\n",
			ASRC_CHANNEL_MAX);
		goto error_iomap;
	}

	if (channel_mm_limits[ASRC_PAIR_A][CHANNEL_LIMIT_MIN] !=
			    channel_mm_limits[ASRC_PAIR_A][CHANNEL_LIMIT_MAX]) {
		dev_warn(&pdev->dev,
		    "illegitimate channel min max configurations for pair A\n");
		dev_warn(&pdev->dev,
		  "reconfiguration of pair A might cause issues on pair B\n");
	}

	for (i = ASRC_PAIR_A; i < ASRC_PAIR_NUM; i++) {
		pair = &asrc_private->asrc_pair[i];
		pair->asrc_priv = asrc_private;
		params = &pair->params;
		params->index = i;

		pair->capabilities.channels_max =
					channel_mm_limits[i][CHANNEL_LIMIT_MAX];
		pair->capabilities.channels_min =
					channel_mm_limits[i][CHANNEL_LIMIT_MIN];
		params->channel_cnt = channel_mm_limits[i][CHANNEL_LIMIT_MAX];
		pair->capabilities.formats[SECTION_INPUT] =
						ASRC_SUPPORTED_INPUT_FORMATS;
		pair->capabilities.formats[SECTION_OUTPUT] =
						ASRC_SUPPORTED_OUTPUT_FORMATS;
		pair->capabilities.rate_max = ASRC_MAX_RATE;
		pair->capabilities.rate_min = ASRC_MIN_RATE;

		dev_dbg(&pdev->dev, "Pair %d channels: min=%d max=%d\n",
			i, pair->capabilities.channels_min,
			pair->capabilities.channels_max);

		if (pair->capabilities.channels_min >
					pair->capabilities.channels_max) {
			dev_err(&pdev->dev, "Pair %d channels: min > max\n", i);
			goto error_iomap;
		}
	}

	/* each pair channel register is 4 bits wide to support
	 * ASRC_CHANNEL_MAX channels */
	asrc_private->channel_bits = 0x4;

	/* setup clocking
	 * we are also making the assumption that the ahb_clk_root is
	 * running already this will enabled the ipg_clk and mem_clk
	 * for asrc when asrc_clk_enable is enabled below */

	/* get the clock associated with this device -- should be "asrc" */
	audio_clk = devm_clk_get(&pdev->dev, "asrc");

	if (IS_ERR(audio_clk)) {
		ret = PTR_ERR(audio_clk);
		dev_err(&pdev->dev, "could not get asrc clock: %d\n", ret);
		goto error_irq;
	}

	ipg_clk = devm_clk_get(&pdev->dev, "ipg");

	if (IS_ERR(ipg_clk)) {
		ret = PTR_ERR(ipg_clk);
		dev_err(&pdev->dev, "could not get ipg clock: %d\n", ret);
		goto error_irq;
	}

	/* prepare and turn on the ipg clock */
	ret = clk_prepare_enable(ipg_clk);
	if (ret) {
		dev_err(&pdev->dev, "couldnt prepare the ipg clock:%d\n", ret);
		goto error_irq;
	}

	/* prepare and turn on the asrc clock */
	ret = clk_prepare_enable(audio_clk);
	if (ret) {
		dev_err(&pdev->dev, "couldnt prepare the asrc clock:%d\n", ret);
		goto err_clock_ipg;
	}

	/* make sure we can set the audio_clk rate */
	asrc_clk_rate = clk_get_rate(audio_clk);
	if (clk_set_rate(audio_clk, asrc_clk_rate))
		goto err_clock_ipg;

	dev_info(&pdev->dev, "asrc_d clk rate = %lu\n",
			clk_get_rate(audio_clk));
	dev_info(&pdev->dev, "ipg clk rate = %lu\n",
			clk_get_rate(ipg_clk));

	asrc_clk_entry = locate_ideal_clock_entry(asrc_private, "ASRCK1");
	if (asrc_clk_entry)
		asrc_clk_entry->bit_clock = asrc_clk_rate;

	dev_dbg(&pdev->dev, "%s clock %s:%u Hz\n", __func__,
			asrc_clk_entry->ideal_clk_name, asrc_clk_rate);

	/* initialize asrc */
	ret = asrc_init(asrc_private, pdev);
	if (ret)
		goto err_clock_asrc;

	asrc_private->dev =
		device_create(asrc_core_class, &pdev->dev, 0,
				asrc_private, ASRC_CORE_DEVICE_NAME);
	if (IS_ERR(asrc_private->dev)) {
		dev_err(&pdev->dev, "Unable to create asrc char device\n");
		err = PTR_ERR(asrc_private->dev);
		goto err_clock_asrc;
	}

	for (i = 0; i < ASRC_PAIR_NUM; i++) {
		pair = &asrc_private->asrc_pair[i];
		pair->dev = device_create(asrc_pair_class, asrc_private->dev, 0,
				pair, "asrc_pair%d", i);

		if (IS_ERR(pair->dev)) {
			dev_err(asrc_private->dev,
				"Unable to create asrc pair char device\n");
			err = PTR_ERR(pair->dev);
			pair->dev = NULL;
			goto err_pairs;
		}
	}

	asrc_debugfs_init(asrc_private);

	dev_info(&pdev->dev, "initialized\n");

	return 0;
err_pairs:
	for (i = 0; i < ASRC_PAIR_NUM; i++)
		if (asrc_private->asrc_pair[i].dev)
			device_unregister(asrc_private->asrc_pair[i].dev);
	device_unregister(asrc_private->dev);
err_clock_asrc:
	clk_disable_unprepare(audio_clk);
err_clock_ipg:
	clk_disable_unprepare(ipg_clk);
error_irq:
	free_irq(asrc_private->irq, asrc_private);
error_iomap:
	iounmap(asrc_private->vaddr);
error_kmalloc:
	dev_set_drvdata(&pdev->dev, NULL);
	kfree(asrc_private);
	return ret;
}

/**
 * mxc_asrc_remove() - Exit asrc device
 * @pdev	Pointer to the registered platform device
 * @return	Error code indicating success or failure
 */
static int mxc_asrc_remove(struct platform_device *pdev)
{
	struct fsl_asrc_core_private *asrc_private = platform_get_drvdata(pdev);
	struct clk *audio_clk = devm_clk_get(&pdev->dev, "asrc");
	struct clk *ipg_clk = devm_clk_get(&pdev->dev, "ipg");
	int i;

	for (i = 0; i < ASRC_PAIR_NUM; i++)
		device_unregister(asrc_private->asrc_pair[i].dev);

	device_unregister(asrc_private->dev);

	if (!IS_ERR(ipg_clk))
		clk_disable_unprepare(ipg_clk);

	if (!IS_ERR(audio_clk))
		clk_disable_unprepare(audio_clk);

	free_irq(asrc_private->irq, asrc_private);
	iounmap(asrc_private->vaddr);
	dev_set_drvdata(&pdev->dev, NULL);

	asrc_debugfs_remove();

	/* free the clock table data */
	kfree(asrc_private->ideal_clock_table);

	kfree(asrc_private);
	asrc_private = NULL;

	return 0;
}

static const struct of_device_id asrc_ids[] = {
	{ .compatible = "fsl,imx6-asrc-core", },
	{}
};
MODULE_DEVICE_TABLE(of, asrc_ids);

static struct platform_driver mxc_asrc_core_driver = {
	.driver = {
		   .name = ASRC_CORE_DEVICE_NAME,
		   .owner = THIS_MODULE,
		   .of_match_table = asrc_ids,
		   },
	.probe = mxc_asrc_core_probe,
	.remove = mxc_asrc_remove,
};

#define ASRC_DEVICES 1

static __init int mxc_asrc_init(void)
{
	int ret;

	asrc_core_class = class_create(THIS_MODULE, ASRC_CORE_DEVICE_NAME);
	if (IS_ERR(asrc_core_class))
		return PTR_ERR(asrc_core_class);

	asrc_core_class->dev_attrs = asrc_attrs;

	asrc_pair_class = class_create(THIS_MODULE, ASRC_DEVICE_PAIR_NAME);
	if (IS_ERR(asrc_pair_class))
		return PTR_ERR(asrc_pair_class);

	asrc_pair_class->dev_attrs = asrc_pair_attrs;

	ret = platform_driver_register(&mxc_asrc_core_driver);
	if (ret == 0)
		return ret;

	class_destroy(asrc_pair_class);
	class_destroy(asrc_core_class);

	return ret;
}

static void __exit mxc_asrc_exit(void)
{
	platform_driver_unregister(&mxc_asrc_core_driver);

	class_destroy(asrc_pair_class);
	class_destroy(asrc_core_class);
	return;
}

module_init(mxc_asrc_init);
module_exit(mxc_asrc_exit);
MODULE_AUTHOR("Mentor Graphics, Inc.");
MODULE_DESCRIPTION("Asynchronous Sample Rate Converter Core");
MODULE_LICENSE("GPL");
