/*
 * Freescale eSDHC i.MX controller driver for the platform bus.
 *
 * derived from the OF-version.
 *
 * Copyright (c) 2010 Pengutronix e.K.
 *   Author: Wolfram Sang <w.sang@pengutronix.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License.
 */

#include <linux/io.h>
#include <linux/delay.h>
#include <linux/err.h>
#include <linux/clk.h>
#include <linux/gpio.h>
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/mmc/host.h>
#include <linux/mmc/mmc.h>
#include <linux/mmc/card.h>
#include <linux/mmc/sdio.h>
#include <linux/mmc/sd.h>
#include "../../../arch/arm/mach-imx/hardware.h"	/* for cpu_is_imx6*() */
#include <linux/scatterlist.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/of_gpio.h>
#include <linux/pinctrl/consumer.h>
#include <linux/platform_data/mmc-esdhc-imx.h>
#include <linux/async.h>
#include "sdhci-pltfm.h"
#include "sdhci-esdhc.h"

#define	ESDHC_CTRL_D3CD			0x08
/* VENDOR SPEC register */
#define ESDHC_VENDOR_SPEC		0xc0
#define  ESDHC_VENDOR_SPEC_SDIO_QUIRK	(1 << 1)
#define ESDHC_WTMK_LVL			0x44

#define ESDHC_SYS_CTRL			0x2C
#define ESDHC_SYS_CTRL_RSTA_LSH 24

#define ESDHC_DLL_CTRL			0x60
#define ESDHC_DLL_OVERRIDE_OFFSET	0x9
#define ESDHC_DLL_OVERRIDE_EN_OFFSET	0x8

#define ESDHC_TUNE_CTRL_STATUS         0x68
#define ESDHC_TUNE_CTRL_STEP           0x1
#define ESDHC_TUNE_CTRL_MIN            0x0
#define ESDHC_TUNE_CTRL_MAX            ((1 << 7) - 1)

#define ESDHC_MIX_CTRL_EXE_TUNE                (1 << 22)
#define ESDHC_MIX_CTRL_SMPCLK_SEL      (1 << 23)
#define ESDHC_MIX_CTRL_AUTO_TUNE     (1 << 24)
#define ESDHC_MIX_CTRL_FBCLK_SEL       (1 << 25)

#define  ESDHC_VENDOR_SPEC_VSELECT     (1 << 1)
#define  ESDHC_VENDOR_SPEC_FRC_SDCLK_ON        (1 << 8)

#define  ESDHC_PRESENT_STATE_CLSL      (1 << 23)
#define  ESDHC_PRESENT_STATE_DLSL_L4   (0xF << 24)
#define  ESDHC_PRESENT_STATE_DLSL_H4   (0xF << 28)

#define  ESDHC_MIX_CTRL_AC23EN		(1 << 7)
/* Bits 3 and 6 are not SDHCI standard definitions */
#define  ESDHC_MIX_CTRL_SDHCI_MASK	0xb7

/*
 * Our interpretation of the SDHCI_HOST_CONTROL register
 */
#define ESDHC_CTRL_1BITBUS		(0x0 << 1)
#define ESDHC_CTRL_4BITBUS		(0x1 << 1)
#define ESDHC_CTRL_8BITBUS		(0x2 << 1)
#define ESDHC_CTRL_BUSWIDTH_MASK	(0x3 << 1)
#define ESDHC_CTRL_LCTL			(1 << 0)
#define ESDHC_CTRL_DMASEL_MASK		(3 << 8)

#define SDHCI_TUNING_BLOCK_PATTERN_LEN		64
/*
 * There is an INT DMA ERR mis-match between eSDHC and STD SDHC SPEC:
 * Bit25 is used in STD SPEC, and is reserved in fsl eSDHC design,
 * but bit28 is used as the INT DMA ERR in fsl eSDHC design.
 * Define this macro DMA error INT for fsl eSDHC
 */
#define ESDHC_INT_VENDOR_SPEC_DMA_ERR	(1 << 28)

/*
 * The CMDTYPE of the CMD register (offset 0xE) should be set to
 * "11" when the STOP CMD12 is issued on imx53 to abort one
 * open ended multi-blk IO. Otherwise the TC INT wouldn't
 * be generated.
 * In exact block transfer, the controller doesn't complete the
 * operations automatically as required at the end of the
 * transfer and remains on hold if the abort command is not sent.
 * As a result, the TC flag is not asserted and SW  received timeout
 * exeception. Bit1 of Vendor Spec registor is used to fix it.
 */
#define ESDHC_FLAG_MULTIBLK_NO_INT	(1 << 1)

/* pinctrl state */
#define ESDHC_PINCTRL_STATE_100MHZ             "state_100mhz"
#define ESDHC_PINCTRL_STATE_200MHZ             "state_200mhz"

static void esdhc_prepare_tuning(struct sdhci_host *host, u32 val);
static void esdhc_post_tuning(struct sdhci_host *host);

enum imx_esdhc_type {
	IMX25_ESDHC,
	IMX35_ESDHC,
	IMX51_ESDHC,
	IMX53_ESDHC,
	IMX6Q_USDHC,
};

struct pltfm_imx_data {
	int flags;
	u32 scratchpad;
	/* uhs mode for sdhc host control2 */
	unsigned char uhs_mode;
	enum imx_esdhc_type devtype;
	struct pinctrl *pinctrl;
	struct pinctrl_state *pins_current;
	struct pinctrl_state *pins_default;
	struct pinctrl_state *pins_100mhz;
	struct pinctrl_state *pins_200mhz;
	struct esdhc_platform_data boarddata;
	struct clk *clk_ipg;
	struct clk *clk_ahb;
	struct clk *clk_per;
};

static struct platform_device_id imx_esdhc_devtype[] = {
	{
		.name = "sdhci-esdhc-imx25",
		.driver_data = IMX25_ESDHC,
	}, {
		.name = "sdhci-esdhc-imx35",
		.driver_data = IMX35_ESDHC,
	}, {
		.name = "sdhci-esdhc-imx51",
		.driver_data = IMX51_ESDHC,
	}, {
		.name = "sdhci-esdhc-imx53",
		.driver_data = IMX53_ESDHC,
	}, {
		.name = "sdhci-usdhc-imx6q",
		.driver_data = IMX6Q_USDHC,
	}, {
		/* sentinel */
	}
};
MODULE_DEVICE_TABLE(platform, imx_esdhc_devtype);

static const struct of_device_id imx_esdhc_dt_ids[] = {
	{ .compatible = "fsl,imx25-esdhc", .data = &imx_esdhc_devtype[IMX25_ESDHC], },
	{ .compatible = "fsl,imx35-esdhc", .data = &imx_esdhc_devtype[IMX35_ESDHC], },
	{ .compatible = "fsl,imx51-esdhc", .data = &imx_esdhc_devtype[IMX51_ESDHC], },
	{ .compatible = "fsl,imx53-esdhc", .data = &imx_esdhc_devtype[IMX53_ESDHC], },
	{ .compatible = "fsl,imx6q-usdhc", .data = &imx_esdhc_devtype[IMX6Q_USDHC], },
	{ /* sentinel */ }
};
MODULE_DEVICE_TABLE(of, imx_esdhc_dt_ids);

static inline int is_imx25_esdhc(struct pltfm_imx_data *data)
{
	return data->devtype == IMX25_ESDHC;
}

static inline int is_imx35_esdhc(struct pltfm_imx_data *data)
{
	return data->devtype == IMX35_ESDHC;
}

static inline int is_imx51_esdhc(struct pltfm_imx_data *data)
{
	return data->devtype == IMX51_ESDHC;
}

static inline int is_imx53_esdhc(struct pltfm_imx_data *data)
{
	return data->devtype == IMX53_ESDHC;
}

static inline int is_imx6q_usdhc(struct pltfm_imx_data *data)
{
	return data->devtype == IMX6Q_USDHC;
}

static void request_done(struct mmc_request *mrq)
{
	complete(&mrq->completion);
}

static int esdhc_send_tuning_cmd(struct sdhci_host *host, u32 opcode)
{
	struct mmc_command cmd = {0};
	struct mmc_request mrq = {0};
	struct mmc_data data = {0};
	struct scatterlist sg;
	char tuning_pattern[SDHCI_TUNING_BLOCK_PATTERN_LEN];

	cmd.opcode = opcode;
	cmd.arg = 0;
	cmd.flags = MMC_RSP_R1 | MMC_CMD_ADTC;

	data.blksz = SDHCI_TUNING_BLOCK_PATTERN_LEN;
	data.blocks = 1;
	data.flags = MMC_DATA_READ;
	data.sg = &sg;
	data.sg_len = 1;

	sg_init_one(&sg, tuning_pattern, sizeof(tuning_pattern));

	mrq.cmd = &cmd;
	mrq.cmd->mrq = &mrq;
	mrq.data = &data;
	mrq.data->mrq = &mrq;
	mrq.cmd->data = mrq.data;

	mrq.done = request_done;

	init_completion(&(mrq.completion));
	sdhci_request(host->mmc, &mrq);
	wait_for_completion(&(mrq.completion));

	if (cmd.error)
		return cmd.error;
	if (data.error)
		return data.error;

	return 0;
}

static int esdhc_execute_tuning(struct sdhci_host *host, u32 opcode)
{
	int min, max, avg, ret;

	min = host->tuning_min;
	while (min < host->tuning_max) {
		esdhc_prepare_tuning(host, min);
		if (!esdhc_send_tuning_cmd(host, opcode))
			break;
		min += host->tuning_step;
	}

	max = min + host->tuning_step;
	while (max < host->tuning_max) {
		esdhc_prepare_tuning(host, max);
		if (esdhc_send_tuning_cmd(host, opcode)) {
			max -= host->tuning_step;
			break;
		}
		max += host->tuning_step;
	}

	avg = (min + max) / 2;
	esdhc_prepare_tuning(host, avg);
	ret = esdhc_send_tuning_cmd(host, opcode);
	esdhc_post_tuning(host);

	dev_dbg(mmc_dev(host->mmc), "tunning %s at 0x%x ret %d\n",
			ret ? "failed" : "passed", avg, ret);
	return 0;
}

static int esdhc_change_pinstate(struct sdhci_host *host,
		unsigned int uhs)
{
	struct sdhci_pltfm_host *pltfm_host = sdhci_priv(host);
	struct pltfm_imx_data *imx_data = pltfm_host->priv;
	struct pinctrl_state *pinctrl;
	int ret;

	dev_dbg(mmc_dev(host->mmc), "change pinctrl state for uhs %d\n", uhs);

	if (IS_ERR(imx_data->pinctrl) ||
			IS_ERR(imx_data->pins_default) ||
			IS_ERR(imx_data->pins_100mhz) ||
			IS_ERR(imx_data->pins_200mhz))
		return -EINVAL;

	switch (uhs) {
	case MMC_TIMING_UHS_SDR12:
	case MMC_TIMING_UHS_SDR25:
	case MMC_TIMING_UHS_DDR50:
		pinctrl = imx_data->pins_default;
		break;
	case MMC_TIMING_UHS_SDR50:
		pinctrl = imx_data->pins_100mhz;
		break;
	case MMC_TIMING_UHS_SDR104:
		pinctrl = imx_data->pins_200mhz;
		break;
	default:
		/* back to default state for other legacy timing */
		pinctrl = imx_data->pins_default;
	}

	if (pinctrl == imx_data->pins_current)
		return 0;

	ret = pinctrl_select_state(imx_data->pinctrl, pinctrl);
	if (!ret)
		imx_data->pins_current = pinctrl;

	return ret;
}

static int esdhc_set_uhs_signaling(struct sdhci_host *host, unsigned int uhs)
{
	struct sdhci_pltfm_host *pltfm_host = sdhci_priv(host);
	struct pltfm_imx_data *imx_data = pltfm_host->priv;

	switch (uhs) {
	case MMC_TIMING_UHS_SDR12:
		imx_data->uhs_mode = SDHCI_CTRL_UHS_SDR12;
		break;
	case MMC_TIMING_UHS_SDR25:
		imx_data->uhs_mode = SDHCI_CTRL_UHS_SDR25;
		break;
	case MMC_TIMING_UHS_SDR50:
		imx_data->uhs_mode = SDHCI_CTRL_UHS_SDR50;
		break;
	case MMC_TIMING_UHS_SDR104:
		imx_data->uhs_mode = SDHCI_CTRL_UHS_SDR104;
		break;
	case MMC_TIMING_UHS_DDR50:
	case MMC_TIMING_MMC_DDR52:
		imx_data->uhs_mode = SDHCI_CTRL_UHS_DDR50;
		writel(readl(host->ioaddr + ESDHC_MIX_CTRL) |
				ESDHC_MIX_CTRL_DDREN,
				host->ioaddr + ESDHC_MIX_CTRL);
		break;
	}

	return esdhc_change_pinstate(host, uhs);
}

static inline void esdhc_clrset_le(struct sdhci_host *host, u32 mask, u32 val, int reg)
{
	void __iomem *base = host->ioaddr + (reg & ~0x3);
	u32 shift = (reg & 0x3) * 8;

	writel(((readl(base) & ~(mask << shift)) | (val << shift)), base);
}

static u32 esdhc_readl_le(struct sdhci_host *host, int reg)
{
	struct sdhci_pltfm_host *pltfm_host = sdhci_priv(host);
	struct pltfm_imx_data *imx_data = pltfm_host->priv;
	struct esdhc_platform_data *boarddata = &imx_data->boarddata;

	/* fake CARD_PRESENT flag */
	u32 val = readl(host->ioaddr + reg);

	/*
	 * mx6q: SDHCI_PRESENT_STATE bit 16, CINST is not functional on SD3.
	 * So move the section up, and check GPIO for card presence again in
	 * the following block.
	 */
	if (unlikely(reg == SDHCI_PRESENT_STATE) &&
				 is_imx6q_usdhc(imx_data)) {
		u32 fsl_prss = readl(host->ioaddr + SDHCI_PRESENT_STATE);
		/* save the least 20 bits */
		val = fsl_prss & 0x000FFFFF;
		/* move dat[0-3] line signal bits */
		val |= (fsl_prss & 0x0F000000) >> 4;
		/* move cmd line signal bits */
		val |= (fsl_prss & 0x00800000) << 1;
	}

	if (unlikely((reg == SDHCI_PRESENT_STATE)
			&& gpio_is_valid(boarddata->cd_gpio))) {
		if (gpio_get_value(boarddata->cd_gpio))
			/* no card, if a valid gpio says so... */
			val &= ~SDHCI_CARD_PRESENT;
		else
			/* ... in all other cases assume card is present */
			val |= SDHCI_CARD_PRESENT;
	}

	if (unlikely(reg == SDHCI_CAPABILITIES)) {
		/* In FSL esdhc IC module, only bit20 is used to indicate the
		 * ADMA2 capability of esdhc, but this bit is messed up on
		 * some SOCs (e.g. on MX25, MX35 this bit is set, but they
		 * don't actually support ADMA2). So set the BROKEN_ADMA
		 * uirk on MX25/35 platforms.
		 */

		if (val & SDHCI_CAN_DO_ADMA1) {
			val &= ~SDHCI_CAN_DO_ADMA1;
			val |= SDHCI_CAN_DO_ADMA2;
		}
	}

	if (unlikely(reg == SDHCI_INT_STATUS)) {
		if (is_imx6q_usdhc(imx_data) &&
				(imx_cpu_ver() == IMX_CHIP_REVISION_1_0)) {
			/*
			 * on mx6q TO1.0, there is low possibility that
			 * DATA END interrupt comes ealier than DMA
			 * END interrupt which is conflict with standard
			 * host controller spec. In this case, read the
			 * status register again will workaround this issue.
			 */
			if ((val & SDHCI_INT_DATA_END) && \
				!(val & SDHCI_INT_DMA_END))
				val = readl(host->ioaddr + reg);
		}

		if (val & ESDHC_INT_VENDOR_SPEC_DMA_ERR) {
			val &= ~ESDHC_INT_VENDOR_SPEC_DMA_ERR;
			val |= SDHCI_INT_ADMA_ERROR;
		}
	} else if (unlikely(reg == SDHCI_CAPABILITIES_1) &&
					 is_imx6q_usdhc(imx_data)) {
		/*
		 * on mx6q, no cap_1 available, fake one.
		 */
		val = SDHCI_SUPPORT_DDR50 | SDHCI_SUPPORT_SDR104 | \
		      SDHCI_SUPPORT_SDR50;
	} else if (unlikely(reg == SDHCI_MAX_CURRENT) &&
					is_imx6q_usdhc(imx_data)) {
		/*
		 * on mx6q, no max current available, fake one.
		 */
		val = 0xFF << SDHCI_MAX_CURRENT_330_SHIFT;
		val |= 0xFF << SDHCI_MAX_CURRENT_300_SHIFT;
		val |= 0xFF << SDHCI_MAX_CURRENT_180_SHIFT;
	}

	return val;
}

static void esdhc_writel_le(struct sdhci_host *host, u32 val, int reg)
{
	struct sdhci_pltfm_host *pltfm_host = sdhci_priv(host);
	struct pltfm_imx_data *imx_data = pltfm_host->priv;
	struct esdhc_platform_data *boarddata = &imx_data->boarddata;
	u32 data;

	if (unlikely(reg == SDHCI_INT_ENABLE || reg == SDHCI_SIGNAL_ENABLE)) {
		if (boarddata->cd_type == ESDHC_CD_GPIO)
			/*
			 * These interrupts won't work with a custom
			 * card_detect gpio (only applied to mx25/35)
			 */
			val &= ~(SDHCI_INT_CARD_REMOVE | SDHCI_INT_CARD_INSERT);

		if (!(val & SDHCI_INT_CARD_INT) && is_imx6q_usdhc(imx_data)
				&& (imx_cpu_ver() == IMX_CHIP_REVISION_1_0))
			/*
			 * write 1 to clear card interrupt status bit
			 * (only applied to mx6q TO1.0)
			 * uSDHC used for mx6q has such problem which is
			 * not consistant with standard host controller
			 * definition.
			 * eSDHC used for mx25/35/50/51/53 does not have
			 * such problem.
			 */
			writel(SDHCI_INT_CARD_INT, \
					host->ioaddr + SDHCI_INT_STATUS);

		if (val & SDHCI_INT_CARD_INT && (!is_imx6q_usdhc(imx_data))) {
			/*
			 * Clear and then set D3CD bit to avoid missing the
			 * card interrupt.  This is a eSDHC controller problem
			 * so we need to apply the following workaround: clear
			 * and set D3CD bit will make eSDHC re-sample the card
			 * interrupt. In case a card interrupt was lost,
			 * re-sample it by the following steps.
			 * (applied to all processors except mx6q)
			 * eSDHC controller used for mx25/35/50/51/53
			 * has such issue, so that we need to do following
			 * operation to avoid losing card interrupt.
			 * uSDCH controller used for mx6q and after won't
			 * have such problem.
			 */
			data = readl(host->ioaddr + SDHCI_HOST_CONTROL);
			data &= ~ESDHC_CTRL_D3CD;
			writel(data, host->ioaddr + SDHCI_HOST_CONTROL);
			data |= ESDHC_CTRL_D3CD;
			writel(data, host->ioaddr + SDHCI_HOST_CONTROL);
		}
	}

	if (unlikely((imx_data->flags & ESDHC_FLAG_MULTIBLK_NO_INT)
				&& (reg == SDHCI_INT_STATUS)
				&& (val & SDHCI_INT_DATA_END))) {
			u32 v;
			v = readl(host->ioaddr + ESDHC_VENDOR_SPEC);
			v &= ~ESDHC_VENDOR_SPEC_SDIO_QUIRK;
			writel(v, host->ioaddr + ESDHC_VENDOR_SPEC);
	}

	if (unlikely(reg == SDHCI_INT_ENABLE || reg == SDHCI_SIGNAL_ENABLE)) {
		if (val & SDHCI_INT_ADMA_ERROR) {
			val &= ~SDHCI_INT_ADMA_ERROR;
			val |= ESDHC_INT_VENDOR_SPEC_DMA_ERR;
		}
	}

	writel(val, host->ioaddr + reg);
}

static u16 esdhc_readw_le(struct sdhci_host *host, int reg)
{
	u16 ret;
	u32 val;
	struct sdhci_pltfm_host *pltfm_host = sdhci_priv(host);
	struct pltfm_imx_data *imx_data = pltfm_host->priv;

	if (unlikely(reg == SDHCI_HOST_VERSION)) {
		reg ^= 2;
		if (is_imx6q_usdhc(imx_data)) {
			/*
			 * The usdhc register returns a wrong host version.
			 * Correct it here.
			 */
			return SDHCI_SPEC_300;
		}
	}

	if (unlikely(reg == SDHCI_HOST_CONTROL2)) {
		/* collect bit info from several regs */
		val = readl(host->ioaddr + ESDHC_VENDOR_SPEC);
		ret = (val & ESDHC_VENDOR_SPEC_VSELECT)
			? SDHCI_CTRL_VDD_180 : 0;

		val = readl(host->ioaddr + ESDHC_MIX_CTRL);
		ret |= (val & ESDHC_MIX_CTRL_EXE_TUNE)
			? SDHCI_CTRL_EXEC_TUNING : 0;
		ret |= (val & ESDHC_MIX_CTRL_SMPCLK_SEL)
			? 0 : SDHCI_CTRL_TUNED_CLK ;
		ret |= SDHCI_CTRL_UHS_MASK & imx_data->uhs_mode;
		/* no preset enable available  */
		ret &= ~SDHCI_CTRL_PRESET_VAL_ENABLE;

		return ret;
	}

	return readw(host->ioaddr + reg);
}

static void esdhc_post_tuning(struct sdhci_host *host)
{
	u32 reg;

	reg = readl(host->ioaddr + ESDHC_MIX_CTRL);
	reg &= ~ESDHC_MIX_CTRL_EXE_TUNE;
	writel(reg, host->ioaddr + ESDHC_MIX_CTRL);
}

static void esdhc_reset(struct sdhci_host *host)
{
	struct sdhci_pltfm_host *pltfm_host = sdhci_priv(host);
	struct pltfm_imx_data *imx_data = pltfm_host->priv;
	unsigned long timeout;
	u32 reg;

	reg = readl(host->ioaddr + ESDHC_SYS_CTRL);
	reg |= 1 << ESDHC_SYS_CTRL_RSTA_LSH;
	writel(reg, host->ioaddr + ESDHC_SYS_CTRL);

	/* Wait max 100ms */
	timeout = 100;

	/* hw clears the bit when it's done */
	while (readl(host->ioaddr + ESDHC_SYS_CTRL)
			& (1 << ESDHC_SYS_CTRL_RSTA_LSH)) {
		if (timeout == 0) {
			pr_err("%s: Reset never completed.\n",
					mmc_hostname(host->mmc));
			return;
		}
		timeout--;
		mdelay(1);
	}

	/*
	 * The RSTA, reset all, on usdhc will not clear following regs:
	 * > ESDHC_MIX_CTRL
	 * > ESDHC_TUNE_CTRL_STATUS
	 *
	 * Do it manually here.
	 */
	if (is_imx6q_usdhc(imx_data)) {
		writel(0, host->ioaddr + ESDHC_MIX_CTRL);
		writel(0, host->ioaddr + ESDHC_TUNE_CTRL_STATUS);
	}

}

static void esdhc_prepare_tuning(struct sdhci_host *host, u32 val)
{
	u32 reg;

	esdhc_reset(host);
	mdelay(1);

	reg = readl(host->ioaddr + ESDHC_MIX_CTRL);
	reg |= ESDHC_MIX_CTRL_EXE_TUNE | \
		ESDHC_MIX_CTRL_SMPCLK_SEL | \
		ESDHC_MIX_CTRL_FBCLK_SEL;
	writel(reg, host->ioaddr + ESDHC_MIX_CTRL);
	writel((val << 8), host->ioaddr + ESDHC_TUNE_CTRL_STATUS);

	dev_dbg(mmc_dev(host->mmc),
		"tunning with delay 0x%x ESDHC_TUNE_CTRL_STATUS 0x%x\n",
			val, readl(host->ioaddr + ESDHC_TUNE_CTRL_STATUS));
}

static void esdhc_writew_le(struct sdhci_host *host, u16 val, int reg)
{
	struct sdhci_pltfm_host *pltfm_host = sdhci_priv(host);
	struct pltfm_imx_data *imx_data = pltfm_host->priv;
	u32 orig_reg;

	switch (reg) {
	case SDHCI_CLOCK_CONTROL:
		orig_reg = readl(host->ioaddr + ESDHC_VENDOR_SPEC);
		if (val & SDHCI_CLOCK_CARD_EN) {
			writel(orig_reg | ESDHC_VENDOR_SPEC_FRC_SDCLK_ON, \
				host->ioaddr + ESDHC_VENDOR_SPEC);
		} else {
			writel(orig_reg & ~ESDHC_VENDOR_SPEC_FRC_SDCLK_ON, \
				host->ioaddr + ESDHC_VENDOR_SPEC);
		}

		return;
	case SDHCI_HOST_CONTROL2:
		orig_reg = readl(host->ioaddr + ESDHC_VENDOR_SPEC);
		if (val & SDHCI_CTRL_VDD_180) {
			orig_reg |= ESDHC_VENDOR_SPEC_VSELECT;
			writel(orig_reg, host->ioaddr + ESDHC_VENDOR_SPEC);
		} else {
			orig_reg &= ~ESDHC_VENDOR_SPEC_VSELECT;
			writel(orig_reg, host->ioaddr + ESDHC_VENDOR_SPEC);
		}

		/*
		 * FSL sdhc controls bus and signal voltage via one bit
		 * VSELECT in VENDOR_SPEC, which has been set in
		 * SDHCI_POWER_CONTROL. So we skip the SDHCI_CTRL_VDD_180
		 * here.
		 *
		 * ignore exec_tuning flag written to SDHCI_HOST_CONTROL2,
		 * tuning will be handled differently for FSL SDHC ip.
		 */
		orig_reg = readl(host->ioaddr + ESDHC_MIX_CTRL);
		orig_reg &= ~ESDHC_MIX_CTRL_SMPCLK_SEL;

		orig_reg |= (val & SDHCI_CTRL_TUNED_CLK)
			? 0 : ESDHC_MIX_CTRL_SMPCLK_SEL;

		writel(orig_reg, host->ioaddr + ESDHC_MIX_CTRL);

		/* set clock frequency again */
		esdhc_set_clock(host, host->clock);
		imx_data->uhs_mode = val & SDHCI_CTRL_UHS_MASK;
		return;
	case SDHCI_TRANSFER_MODE:
		if ((imx_data->flags & ESDHC_FLAG_MULTIBLK_NO_INT)
				&& (host->cmd->opcode == SD_IO_RW_EXTENDED)
				&& (host->cmd->data->blocks > 1)
				&& (host->cmd->data->flags & MMC_DATA_READ)) {
			u32 v;
			v = readl(host->ioaddr + ESDHC_VENDOR_SPEC);
			v |= ESDHC_VENDOR_SPEC_SDIO_QUIRK;
			writel(v, host->ioaddr + ESDHC_VENDOR_SPEC);
		}

		if (is_imx6q_usdhc(imx_data)) {
			u32 m = readl(host->ioaddr + ESDHC_MIX_CTRL);
			/* Swap AC23 bit */
			if (val & SDHCI_TRNS_AUTO_CMD23) {
				val &= ~SDHCI_TRNS_AUTO_CMD23;
				val |= ESDHC_MIX_CTRL_AC23EN;
			}
			m = val | (m & ~ESDHC_MIX_CTRL_SDHCI_MASK);
			writel(m, host->ioaddr + ESDHC_MIX_CTRL);
		} else {
			/*
			 * Postpone this write, we must do it together with a
			 * command write that is down below.
			 */
			imx_data->scratchpad = val;
		}
		return;
	case SDHCI_COMMAND:
		if ((host->cmd->opcode == MMC_STOP_TRANSMISSION ||
		     host->cmd->opcode == MMC_SET_BLOCK_COUNT) &&
	            (imx_data->flags & ESDHC_FLAG_MULTIBLK_NO_INT))
			val |= SDHCI_CMD_ABORTCMD;

		if (is_imx6q_usdhc(imx_data))
			writel(val << 16,
			       host->ioaddr + SDHCI_TRANSFER_MODE);
		else
			writel(val << 16 | imx_data->scratchpad,
			       host->ioaddr + SDHCI_TRANSFER_MODE);
		return;
	case SDHCI_BLOCK_SIZE:
		val &= ~SDHCI_MAKE_BLKSZ(0x7, 0);
		break;
	}
	esdhc_clrset_le(host, 0xffff, val, reg);
}

static u8 esdhc_readb_le(struct sdhci_host *host, int reg)
{
	u8 ret;
	u32 reg_val;

	ret = 0;
	switch (reg) {
	case SDHCI_POWER_CONTROL:
		reg_val = readl(host->ioaddr + ESDHC_VENDOR_SPEC);
		ret |= reg_val & ESDHC_VENDOR_SPEC_VSELECT
			? SDHCI_POWER_180 : SDHCI_POWER_300;
		/* could not power off */
		ret |= SDHCI_POWER_ON;
		return ret;
	case SDHCI_HOST_CONTROL:
		reg_val = readl(host->ioaddr + SDHCI_HOST_CONTROL);
		if (reg_val & ESDHC_CTRL_LCTL)
			ret |= SDHCI_CTRL_LED;
		else
			ret &= ~SDHCI_CTRL_LED;

		ret |= (reg_val & ESDHC_CTRL_DMASEL_MASK) >> 5;
		if (ESDHC_CTRL_8BITBUS == (reg_val &
						ESDHC_CTRL_BUSWIDTH_MASK)) {
			ret &= ~SDHCI_CTRL_4BITBUS;
			ret |= SDHCI_CTRL_8BITBUS;
		} else if (ESDHC_CTRL_4BITBUS
				== (reg_val & ESDHC_CTRL_BUSWIDTH_MASK)) {
			ret &= ~SDHCI_CTRL_8BITBUS;
			ret |= SDHCI_CTRL_4BITBUS;
		} else if (ESDHC_CTRL_1BITBUS
				== (reg_val & ESDHC_CTRL_BUSWIDTH_MASK))
			ret &= ~(SDHCI_CTRL_8BITBUS | SDHCI_CTRL_4BITBUS);
		return ret;
	case SDHCI_SOFTWARE_RESET:
		reg_val = readl(host->ioaddr + SDHCI_CLOCK_CONTROL);
		ret = reg_val >> 24;
		return ret;
	case SDHCI_RESPONSE + 3:
		reg_val = readl(host->ioaddr + SDHCI_RESPONSE);
		ret = reg_val >> 24;
		return ret;
	case SDHCI_RESPONSE + 7:
		reg_val = readl(host->ioaddr + SDHCI_RESPONSE + 4);
		ret = reg_val >> 24;
		return ret;
	case SDHCI_RESPONSE + 11:
		reg_val = readl(host->ioaddr + SDHCI_RESPONSE + 8);
		ret = reg_val >> 24;
		return ret;
	default:
		ret = 0;
		break;
	}
	return ret;
}

static void esdhc_writeb_le(struct sdhci_host *host, u8 val, int reg)
{
	struct sdhci_pltfm_host *pltfm_host = sdhci_priv(host);
	struct pltfm_imx_data *imx_data = pltfm_host->priv;
	u32 new_val;
	u32 mask;

	switch (reg) {
	case SDHCI_POWER_CONTROL:
		/*
		 * FSL put some DMA bits here
		 * If your board has a regulator, code should be here
		 */

		if (val == (SDHCI_POWER_ON | SDHCI_POWER_180)) {
			/* stop sd clock */
			new_val = readl(host->ioaddr + ESDHC_VENDOR_SPEC);
			writel(new_val & ~ESDHC_VENDOR_SPEC_FRC_SDCLK_ON, \
				host->ioaddr + ESDHC_VENDOR_SPEC);

			/* switch to 1.8V */
			new_val = readl(host->ioaddr + ESDHC_VENDOR_SPEC);
			new_val |= ESDHC_VENDOR_SPEC_VSELECT;
			writel(new_val, host->ioaddr + ESDHC_VENDOR_SPEC);
		} else {
			new_val = readl(host->ioaddr + ESDHC_VENDOR_SPEC);
			new_val &= ~ESDHC_VENDOR_SPEC_VSELECT;
			writel(new_val, host->ioaddr + ESDHC_VENDOR_SPEC);
		}
		return;
	case SDHCI_HOST_CONTROL:
		/* FSL messed up here, so we can just keep those three */
		new_val = val & (SDHCI_CTRL_LED);
		if (val & SDHCI_CTRL_8BITBUS) {
			new_val |= ESDHC_CTRL_8BITBUS;
			new_val &= ~ESDHC_CTRL_4BITBUS;
		} else if (val & SDHCI_CTRL_4BITBUS) {
			new_val &= ~ESDHC_CTRL_8BITBUS;
			new_val |= ESDHC_CTRL_4BITBUS;
		}
		/* ensure the endianness */
		new_val |= ESDHC_HOST_CONTROL_LE;
		/* bits 8&9 are reserved on mx25 */
		if (!is_imx25_esdhc(imx_data)) {
			/* DMA mode bits are shifted */
			new_val |= (val & SDHCI_CTRL_DMA_MASK) << 5;
		}

		/*
		 * Do not touch buswidth bits here. This is done in
		 * esdhc_pltfm_bus_width.
		 */
		mask = 0xffff & ~ESDHC_CTRL_BUSWIDTH_MASK;

		esdhc_clrset_le(host, mask, new_val, reg);
		return;
	case SDHCI_TIMEOUT_CONTROL:
		/*
		 * On i.MX6 the timeout value DTOCV is 4 bit large. Touch only
		 * these 4 bits (lower nibble of the byte), but not the upper
		 * nibble of the byte. The upper nibble of the byte contains
		 * IPP_RST_N which should keep the reset value, i.e. 1, and
		 * shouldn't be touched here.
		 */
		if (is_imx6q_usdhc(imx_data)) {
			esdhc_clrset_le(host, 0x0f, val, reg);
			return;
		}
	}
	esdhc_clrset_le(host, 0xff, val, reg);

	/*
	 * The esdhc has a design violation to SDHC spec which tells
	 * that software reset should not affect card detection circuit.
	 * But esdhc clears its SYSCTL register bits [0..2] during the
	 * software reset.  This will stop those clocks that card detection
	 * circuit relies on.  To work around it, we turn the clocks on back
	 * to keep card detection circuit functional.
	 */
	if ((reg == SDHCI_SOFTWARE_RESET) && (val & 1)) {
		esdhc_clrset_le(host, 0x7, 0x7, ESDHC_SYSTEM_CONTROL);
		/*
		 * The reset on usdhc fails to clear MIX_CTRL register.
		 * Do it manually here.
		 */
		if (is_imx6q_usdhc(imx_data))
			writel(0, host->ioaddr + ESDHC_MIX_CTRL);
	}
}

static void esdhc_pltfm_clk_enable(struct sdhci_host *host)
{
	struct sdhci_pltfm_host *pltfm_host = sdhci_priv(host);
	struct pltfm_imx_data *imx_data = pltfm_host->priv;

	clk_prepare_enable(imx_data->clk_per);
	clk_prepare_enable(imx_data->clk_ipg);
	clk_prepare_enable(imx_data->clk_ahb);
}

static void esdhc_pltfm_clk_disable(struct sdhci_host *host)
{
	struct sdhci_pltfm_host *pltfm_host = sdhci_priv(host);
	struct pltfm_imx_data *imx_data = pltfm_host->priv;

	clk_disable_unprepare(imx_data->clk_per);
	clk_disable_unprepare(imx_data->clk_ipg);
	clk_disable_unprepare(imx_data->clk_ahb);
}

static unsigned int esdhc_pltfm_get_max_clock(struct sdhci_host *host)
{
	struct sdhci_pltfm_host *pltfm_host = sdhci_priv(host);
	struct pltfm_imx_data *imx_data = pltfm_host->priv;
	struct esdhc_platform_data *boarddata = &imx_data->boarddata;

	u32 f_host = clk_get_rate(pltfm_host->clk);

	if (boarddata->f_max && (boarddata->f_max < f_host))
		return boarddata->f_max;
	else
		return f_host;
}

static unsigned int esdhc_pltfm_get_min_clock(struct sdhci_host *host)
{
	struct sdhci_pltfm_host *pltfm_host = sdhci_priv(host);

	return clk_get_rate(pltfm_host->clk) / 256 / 16;
}

static unsigned int esdhc_pltfm_get_ro(struct sdhci_host *host)
{
	struct sdhci_pltfm_host *pltfm_host = sdhci_priv(host);
	struct pltfm_imx_data *imx_data = pltfm_host->priv;
	struct esdhc_platform_data *boarddata = &imx_data->boarddata;

	switch (boarddata->wp_type) {
	case ESDHC_WP_GPIO:
		if (gpio_is_valid(boarddata->wp_gpio))
			return gpio_get_value(boarddata->wp_gpio);
	case ESDHC_WP_CONTROLLER:
		return !(readl(host->ioaddr + SDHCI_PRESENT_STATE) &
			       SDHCI_WRITE_PROTECT);
	case ESDHC_WP_NONE:
		break;
	}

	return -ENOSYS;
}

static int esdhc_pltfm_bus_width(struct sdhci_host *host, int width)
{
	u32 ctrl;

	switch (width) {
	case MMC_BUS_WIDTH_8:
		ctrl = ESDHC_CTRL_8BITBUS;
		break;
	case MMC_BUS_WIDTH_4:
		ctrl = ESDHC_CTRL_4BITBUS;
		break;
	default:
		ctrl = 0;
		break;
	}

	esdhc_clrset_le(host, ESDHC_CTRL_BUSWIDTH_MASK, ctrl,
			SDHCI_HOST_CONTROL);

	return 0;
}

static void plt_clk_ctrl(struct sdhci_host *host, bool enable)
{
	if (enable) {
		esdhc_pltfm_clk_enable(host);
		host->clk_status = true;
	} else {
		esdhc_pltfm_clk_disable(host);
		host->clk_status = false;
	}
}

static unsigned int esdhc_get_max_timeout_count(struct sdhci_host *host)
{
	struct sdhci_pltfm_host *pltfm_host = sdhci_priv(host);
	struct pltfm_imx_data *imx_data = pltfm_host->priv;

	return is_imx6q_usdhc(imx_data) ? 1 << 28 : 1 << 27;
}

static void esdhc_set_timeout(struct sdhci_host *host, struct mmc_command *cmd)
{
	struct sdhci_pltfm_host *pltfm_host = sdhci_priv(host);
	struct pltfm_imx_data *imx_data = pltfm_host->priv;

	/* use maximum timeout counter */
	sdhci_writeb(host, is_imx6q_usdhc(imx_data) ? 0xF : 0xE,
			SDHCI_TIMEOUT_CONTROL);
}

static struct sdhci_ops sdhci_esdhc_ops = {
	.read_l = esdhc_readl_le,
	.read_w = esdhc_readw_le,
	.read_b = esdhc_readb_le,
	.write_l = esdhc_writel_le,
	.write_w = esdhc_writew_le,
	.write_b = esdhc_writeb_le,
	.set_clock = esdhc_set_clock,
	.get_max_clock = esdhc_pltfm_get_max_clock,
	.get_min_clock = esdhc_pltfm_get_min_clock,
	.get_max_timeout_count = esdhc_get_max_timeout_count,
	.get_ro = esdhc_pltfm_get_ro,
	.set_timeout = esdhc_set_timeout,
	.platform_bus_width = esdhc_pltfm_bus_width,
	.platform_clk_ctrl = plt_clk_ctrl,
	.set_uhs_signaling = esdhc_set_uhs_signaling,
};

static struct sdhci_pltfm_data sdhci_esdhc_imx_pdata = {
	.quirks = ESDHC_DEFAULT_QUIRKS | SDHCI_QUIRK_NO_HISPD_BIT
			| SDHCI_QUIRK_NO_ENDATTR_IN_NOPDESC
			| SDHCI_QUIRK_BROKEN_ADMA_ZEROLEN_DESC
			| SDHCI_QUIRK_BROKEN_CARD_DETECTION
			| SDHCI_QUIRK_BROKEN_ADMA,
	.ops = &sdhci_esdhc_ops,
};

static irqreturn_t cd_irq(int irq, void *data)
{
	struct sdhci_host *sdhost = (struct sdhci_host *)data;

	esdhc_reset(sdhost);

	tasklet_schedule(&sdhost->card_tasklet);
	return IRQ_HANDLED;
};

#ifdef CONFIG_OF
static int
sdhci_esdhc_imx_probe_dt(struct platform_device *pdev,
			 struct esdhc_platform_data *boarddata)
{
	struct device_node *np = pdev->dev.of_node;
	struct sdhci_host *host = platform_get_drvdata(pdev);
	int bus_width = 0;
	u32 val;

	if (!np)
		return -ENODEV;

	if (of_get_property(np, "non-removable", NULL))
		boarddata->cd_type = ESDHC_CD_PERMANENT;

	if (of_get_property(np, "fsl,cd-controller", NULL))
		boarddata->cd_type = ESDHC_CD_CONTROLLER;

	if (of_get_property(np, "fsl,wp-controller", NULL))
		boarddata->wp_type = ESDHC_WP_CONTROLLER;

	if (of_get_property(np, "mmc-shutdown", NULL))
		boarddata->mmc_shutdown = 1;

	if (!of_property_read_u32(np, "fsl,mmcblk_devidx", &val))
		boarddata->mmcblk_devidx = val;

	boarddata->cd_gpio = of_get_named_gpio(np, "cd-gpios", 0);
	if (gpio_is_valid(boarddata->cd_gpio))
		boarddata->cd_type = ESDHC_CD_GPIO;

	boarddata->wp_gpio = of_get_named_gpio(np, "wp-gpios", 0);
	if (gpio_is_valid(boarddata->wp_gpio))
		boarddata->wp_type = ESDHC_WP_GPIO;

	boarddata->power_gpio = of_get_named_gpio(np, "power-gpios", 0);
	if (gpio_is_valid(boarddata->power_gpio))
		boarddata->power_type = ESDHC_POWER_GPIO;

	if (of_get_property(np, "support-vdd-180", NULL))
		boarddata->vdd_180 = 1;

	if (of_get_property(np, "fsl,use-pio", NULL))
		boarddata->pio_mode = 1;

	of_property_read_u32(np, "bus-width", &bus_width);
	switch (bus_width) {
	case 1:
		boarddata->bus_width = 0;
		break;
	case 4:
		boarddata->bus_width = MMC_CAP_4_BIT_DATA;
		break;
	case 8:
		boarddata->bus_width = MMC_CAP_8_BIT_DATA | MMC_CAP_4_BIT_DATA;
		break;
	default:
		pr_warn("%s: Unsupported bus width\n", np->full_name);
	}

	of_property_read_u32(np, "bus-width", &boarddata->max_bus_width);

	of_property_read_u32(np, "max-frequency", &boarddata->f_max);

	if (of_find_property(np, "keep-power-in-suspend", NULL))
		host->mmc->pm_caps |= MMC_PM_KEEP_POWER;

	if (of_find_property(np, "enable-sdio-wakeup", NULL))
		host->mmc->pm_caps |= MMC_PM_WAKE_SDIO_IRQ;

	of_property_read_u32(np, "init-delay", &host->mmc->init_delay);

	return 0;
}
#else
static inline int
sdhci_esdhc_imx_probe_dt(struct platform_device *pdev,
			 struct esdhc_platform_data *boarddata)
{
	return -ENODEV;
}
#endif

static int sdhci_esdhc_imx_probe(struct platform_device *pdev)
{
	const struct of_device_id *of_id =
			of_match_device(imx_esdhc_dt_ids, &pdev->dev);
	struct sdhci_pltfm_host *pltfm_host;
	struct sdhci_host *host;
	struct esdhc_platform_data *boarddata;
	int err;
	struct pltfm_imx_data *imx_data;
	u32 reg;

	host = sdhci_pltfm_init(pdev, &sdhci_esdhc_imx_pdata);
	if (IS_ERR(host))
		return PTR_ERR(host);

	pltfm_host = sdhci_priv(host);

	imx_data = devm_kzalloc(&pdev->dev, sizeof(*imx_data), GFP_KERNEL);
	if (!imx_data) {
		err = -ENOMEM;
		goto free_sdhci;
	}

	if (of_id)
		pdev->id_entry = of_id->data;
	imx_data->devtype = pdev->id_entry->driver_data;
	pltfm_host->priv = imx_data;

	imx_data->clk_ipg = devm_clk_get(&pdev->dev, "ipg");
	if (IS_ERR(imx_data->clk_ipg)) {
		err = PTR_ERR(imx_data->clk_ipg);
		goto free_sdhci;
	}

	imx_data->clk_ahb = devm_clk_get(&pdev->dev, "ahb");
	if (IS_ERR(imx_data->clk_ahb)) {
		err = PTR_ERR(imx_data->clk_ahb);
		goto free_sdhci;
	}

	imx_data->clk_per = devm_clk_get(&pdev->dev, "per");
	if (IS_ERR(imx_data->clk_per)) {
		err = PTR_ERR(imx_data->clk_per);
		goto free_sdhci;
	}

	pltfm_host->clk = imx_data->clk_per;

	esdhc_pltfm_clk_enable(host);

	imx_data->pinctrl = devm_pinctrl_get(&pdev->dev);
	if (IS_ERR(imx_data->pinctrl)) {
		err = PTR_ERR(imx_data->pinctrl);
		goto disable_clk;
	}

	imx_data->pins_default = pinctrl_lookup_state(imx_data->pinctrl,
			PINCTRL_STATE_DEFAULT);
	if (IS_ERR(imx_data->pins_default)) {
		err = PTR_ERR(imx_data->pins_default);
		dev_err(mmc_dev(host->mmc),
				"could not get default state: %d\n", err);
	}

	reg = readl(host->ioaddr + ESDHC_MIX_CTRL);
	reg &= ~ESDHC_MIX_CTRL_DDREN;
	writel(reg, host->ioaddr + ESDHC_MIX_CTRL);

	if (!is_imx25_esdhc(imx_data))
		host->quirks |= SDHCI_QUIRK_BROKEN_TIMEOUT_VAL;

	if (is_imx25_esdhc(imx_data) || is_imx35_esdhc(imx_data))
		/* Fix errata ENGcm07207 present on i.MX25 and i.MX35 */
		host->quirks |= SDHCI_QUIRK_NO_MULTIBLOCK
			| SDHCI_QUIRK_BROKEN_ADMA;

	if (is_imx53_esdhc(imx_data))
		imx_data->flags |= ESDHC_FLAG_MULTIBLK_NO_INT;

	/*
	 * The imx6q ROM code will change the default watermark level setting
	 * to something insane.  Change it back here.
	 */
	if (is_imx6q_usdhc(imx_data))
		writel(0x08100810, host->ioaddr + ESDHC_WTMK_LVL);

	boarddata = &imx_data->boarddata;
	if (sdhci_esdhc_imx_probe_dt(pdev, boarddata) < 0) {
		if (!host->mmc->parent->platform_data) {
			dev_err(mmc_dev(host->mmc), "no board data!\n");
			err = -EINVAL;
			goto disable_clk;
		}
		imx_data->boarddata = *((struct esdhc_platform_data *)
					host->mmc->parent->platform_data);
	}

	/* copy mmc block device index */
	host->mmc->mmcblk_devidx = boarddata->mmcblk_devidx;
	host->mmc->mmc_shutdown = boarddata->mmc_shutdown;

	/* write_protect */
	if (boarddata->wp_type == ESDHC_WP_GPIO) {
		err = devm_gpio_request_one(&pdev->dev, boarddata->wp_gpio,
					    GPIOF_IN, "ESDHC_WP");
		if (err) {
			dev_warn(mmc_dev(host->mmc),
				 "no write-protect pin available!\n");
			boarddata->wp_gpio = -EINVAL;
		}
	} else {
		boarddata->wp_gpio = -EINVAL;
	}

	host->ocr_avail_sd = MMC_VDD_29_30 | MMC_VDD_30_31 | \
			MMC_VDD_32_33 | MMC_VDD_33_34;
	host->ocr_avail_mmc = MMC_VDD_29_30 | MMC_VDD_30_31 | \
			MMC_VDD_32_33 | MMC_VDD_33_34;
	host->ocr_avail_sdio = MMC_VDD_29_30 | MMC_VDD_30_31 | \
			MMC_VDD_32_33 | MMC_VDD_33_34;

	if (is_imx6q_usdhc(imx_data))
		sdhci_esdhc_ops.platform_execute_tuning = esdhc_execute_tuning;

	if (boarddata->vdd_180) {
		host->ocr_avail_sd |= MMC_VDD_165_195;
		host->ocr_avail_mmc |= MMC_VDD_165_195;
		host->ocr_avail_sdio |= MMC_VDD_165_195;

		imx_data->pins_100mhz = pinctrl_lookup_state(
				imx_data->pinctrl,
				ESDHC_PINCTRL_STATE_100MHZ);
		imx_data->pins_200mhz = pinctrl_lookup_state(
				imx_data->pinctrl,
				ESDHC_PINCTRL_STATE_200MHZ);
		if (IS_ERR(imx_data->pins_100mhz) ||
				IS_ERR(imx_data->pins_200mhz)) {
			err = -ENODEV;
			dev_err(mmc_dev(host->mmc),
					"could not get ultra high speed state: %d\n",
					err);
			/* fall back to not support sd30 */
			host->quirks2 |= SDHCI_QUIRK2_NO_1_8_V;
		}
	} else
		host->quirks2 |= SDHCI_QUIRK2_NO_1_8_V;

	/* power enable pin */
	if (boarddata->power_type == ESDHC_POWER_GPIO) {
		err = devm_gpio_request_one(&pdev->dev, boarddata->power_gpio,
					    GPIOF_OUT_INIT_HIGH, "ESDHC_POWER");
		if (err) {
			dev_warn(mmc_dev(host->mmc),
				 "power pin failed!\n");
			boarddata->power_gpio = -EINVAL;
		}
	} else {
		boarddata->power_gpio = -EINVAL;
	}

	/* The imx6q uSDHC capabilities will always claim to support 1.8V
	 * while this is board specific,  should be initialized properly
	 */
	if (is_imx6q_usdhc(imx_data)) {
		host->quirks |= SDHCI_QUIRK_MISSING_CAPS;
		host->caps = readl(host->ioaddr + SDHCI_CAPABILITIES);
		host->caps1 = SDHCI_SUPPORT_DDR50 |
				SDHCI_SUPPORT_SDR104 | SDHCI_SUPPORT_SDR50;
		if (boarddata->pio_mode)
			host->caps &= ~(SDHCI_CAN_DO_SDMA | SDHCI_CAN_DO_ADMA2);
		if (!boarddata->vdd_180)
			host->caps &= ~SDHCI_CAN_VDD_180;
		host->clk_mgr_en = true;
	}

	/* card_detect */
	if (boarddata->cd_type != ESDHC_CD_GPIO)
		boarddata->cd_gpio = -EINVAL;

	host->mmc->pm_caps = MMC_PM_KEEP_POWER;

	switch (boarddata->cd_type) {
	case ESDHC_CD_GPIO:
		err = devm_gpio_request_one(&pdev->dev, boarddata->cd_gpio,
					    GPIOF_IN, "ESDHC_CD");
		if (err) {
			dev_err(mmc_dev(host->mmc),
				"no card-detect pin available!\n");
			goto disable_clk;
		}

		err = devm_request_irq(&pdev->dev,
				 gpio_to_irq(boarddata->cd_gpio), cd_irq,
				 IRQF_TRIGGER_FALLING | IRQF_TRIGGER_RISING,
				 mmc_hostname(host->mmc), host);
		if (err) {
			dev_err(mmc_dev(host->mmc), "request irq error\n");
			goto disable_clk;
		}
		/* fall through */

	case ESDHC_CD_CONTROLLER:
		/* we have a working card_detect back */
		host->quirks &= ~SDHCI_QUIRK_BROKEN_CARD_DETECTION;
		break;

	case ESDHC_CD_PERMANENT:
		host->mmc->caps = MMC_CAP_NONREMOVABLE;
		break;

	case ESDHC_CD_NONE:
		break;
	}

	if (boarddata->bus_width == 0)
		host->quirks |= SDHCI_QUIRK_FORCE_1_BIT_DATA;
	else
		host->mmc->caps |= boarddata->bus_width;

	switch (boarddata->max_bus_width) {
	case 8:
		host->mmc->caps |= MMC_CAP_8_BIT_DATA | MMC_CAP_4_BIT_DATA;
		break;
	case 4:
		host->mmc->caps |= MMC_CAP_4_BIT_DATA;
		break;
	case 1:
	default:
		host->quirks |= SDHCI_QUIRK_FORCE_1_BIT_DATA;
		break;
	}

	if (is_imx6q_usdhc(imx_data)) {
		host->mmc->caps |= MMC_CAP_1_8V_DDR;
		host->tuning_min = ESDHC_TUNE_CTRL_MIN;
		host->tuning_max = ESDHC_TUNE_CTRL_MAX;
		host->tuning_step = ESDHC_TUNE_CTRL_STEP;
	}

#ifdef CONFIG_PM_RUNTIME
	host->mmc->caps |= MMC_CAP_POWER_OFF_CARD;
#endif

	err = sdhci_add_host(host);
	if (err)
		goto disable_clk;

	if (host->clk_mgr_en)
		esdhc_pltfm_clk_disable(host);
	return 0;

disable_clk:
	if (host->clk_mgr_en)
		esdhc_pltfm_clk_disable(host);
free_sdhci:
	sdhci_pltfm_free(pdev);
	return err;
}

static int sdhci_esdhc_imx_remove(struct platform_device *pdev)
{
	struct sdhci_host *host = platform_get_drvdata(pdev);
	int dead = (readl(host->ioaddr + SDHCI_INT_STATUS) == 0xffffffff);

	sdhci_remove_host(host, dead);

	if (!host->clk_mgr_en)
		esdhc_pltfm_clk_disable(host);

	sdhci_pltfm_free(pdev);

	return 0;
}

static void sdhci_esdhc_imx_probe_wrapper(void *d, async_cookie_t c)
{
	struct platform_device *pdev = (struct platform_device *)d;
	int ret = sdhci_esdhc_imx_probe(pdev);

	if (ret != 0)
		dev_err(&pdev->dev,
			"sdhci_esdhc_imx_probe failed with error %ds\n", ret);

	put_device(&pdev->dev);
}

static int sdhci_esdhc_imx_probe_async(struct platform_device *pdev)
{
	get_device(&pdev->dev); /* prevent release before async_schedule */
	async_schedule(sdhci_esdhc_imx_probe_wrapper, pdev);
	return 0;
}

static struct platform_driver sdhci_esdhc_imx_driver = {
	.driver		= {
		.name	= "sdhci-esdhc-imx",
		.owner	= THIS_MODULE,
		.of_match_table = imx_esdhc_dt_ids,
		.pm	= SDHCI_PLTFM_PMOPS,
	},
	.id_table	= imx_esdhc_devtype,
	.probe		= sdhci_esdhc_imx_probe_async,
	.remove		= sdhci_esdhc_imx_remove,
};

module_platform_driver(sdhci_esdhc_imx_driver);

MODULE_DESCRIPTION("SDHCI driver for Freescale i.MX eSDHC");
MODULE_AUTHOR("Wolfram Sang <w.sang@pengutronix.de>");
MODULE_LICENSE("GPL v2");
