/*
 * Copyright (C) 2013 Robert Bosch Car Multimedia GmbH
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/delay.h>
#include <linux/spi/spi.h>
#include <linux/gpio.h>
#include <linux/fb.h>
#include <linux/err.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/of_gpio.h>

#define ACCESS_TIMEOUT	0x80

#define REVISION_REG	0x00
#define REVISION_ID	0xA9

#define PLL_M_DIV_REG	0x20
#define PLL_LOCKED	0x80

#define PLL_0_REG	0x22
#define PLL_1_REG	0x24
#define PLL_2_REG	0x26
#define PLL_3_REG	0x28
#define PLL_4_REG	0x2A
#define CLK_SRC_REG	0x2C
#define CLK_SRC_TV_DDS_PLL	(1 << 2)
#define CLK_SRC_TV_TIMING_PLL	(1 << 1)
#define CLK_SRC_SYSCLK_PLL	(1 << 0)

#define POWER_SAVE_REG	0x2E
#define TVCONF_REG	0x40
#define TVC_OUTFMT_BITMASK	(1 << 4)
#define TVC_OUTFMT_S_Video	(1 << 4)
#define TVC_TVOUT_PAL		(0x00 << 1)
#define TVC_TVOUT_PAL_M		(0x01 << 1)
#define TVC_TVOUT_PAL_N		(0x02 << 1)
#define TVC_TVOUT_PAL_Nc	(0x03 << 1)
#define TVC_TVOUT_NTSC_M	(0x04 << 1)
#define TVC_TVOUT_NTSC_J	(0x05 << 1)

#define INPUT_FMT_REG	0x60
#define INPUT_FMT_DEFAULT	0x30
#define INPUT_FMT_RGB332	0x00
#define INPUT_FMT_RGB565	0x01
#define INPUT_FMT_RGB666	0x02
#define INPUT_FMT_RGB888	0x03

#define SPEC_EFFECT_REG		0x62
#define SPEC_EFFECT_DEFAULT	0x20

#define OUTPUT_CTRL_REG	0x80
#define OUTPUT_TV_EN	(1 << 2)

/*NTSC 720*480 and PAL 720*576*/
/*height*/
#define OUTPUT_HEIGHT_LOW_REG	0x8A
#define OUTPUT_HEIGHT_HIGH_REG	0x8C
#define OUTPUT_NTSC_HEIGHT_LOW  0xE0  /*0x1E0*/
#define OUTPUT_NTSC_HEIGHT_HIGH 0x01
#define OUTPUT_PAL_HEIGHT_LOW   0x40  /*0x240*/
#define OUTPUT_PAL_HEIGHT_HIGH  0x02

/*width, NTSC and PAL have the same width*/
#define OUTPUT_WIDTH_LOW_REG	0x8E
#define OUTPUT_WIDTH_HIGH_REG	0x90
#define OUTPUT_NTSC_WIDTH_LOW   0xD0  /*0x2D0*/
#define OUTPUT_NTSC_WIDTH_HIGH  0x02
#define OUTPUT_PAL_WIDTH_LOW    0xD0  /*0x2D0*/
#define OUTPUT_PAL_WIDTH_HIGH   0x02

#define DAC_REF_REG	0x9E
#define DAC_IREF_EN	(1 << 1)
#define DAC_VREF_EN	1

struct s1d13746_data {
	bool pll_tvclk;
	bool pll_sysclk;
	bool iref;
	bool vref;
	bool composite;
	unsigned int clk_rate;	/* 1K units */
	enum {
	TVT_PAL,
	TVT_PAL_M,
	TVT_PAL_N,
	TVT_PAL_Nc,
	TVT_NTSC_M,
	TVT_NTSC_J,
	TVT_MAX
	} tv_type;
	int reset_pin;
	int reset_active;
	int standby_pin;
	int standby_active;
};


struct imx_tvout_display {
	struct device *pdev;
	u32 interface_pix_fmt;
	int mode_valid;
	void *data;
};

struct clk_setting {
	unsigned int clk_rate;
	unsigned char div;
	unsigned char lcnt;
};

static struct clk_setting clk_table[] = {
	{.clk_rate = 27000, .div = 0x1A, .lcnt = 0x35},
	{.clk_rate = 26000, .div = 0x19, .lcnt = 0x35},
	{.clk_rate = 24000, .div = 0x15, .lcnt = 0x23},
};


static struct of_device_id s1d13746_dt_ids[] = {
	{.compatible = "RBCM,S1D13746", 0},
	{}
};

/* SPI driver functions */
#define S1D13746_CMD_ADDR	0x40
#define S1D13746_CMD_WRITE	0x80
#define S1D13746_CMD_READ	0xC0

/* s1d13746 spi access operations */
static inline int s1d13746_writeb(void *client, u8 reg, u8 val)
{
	int ret;
	u8 cmd[2] = { S1D13746_CMD_ADDR, reg };
	u8 data[2] = { S1D13746_CMD_WRITE, val };
	ret = spi_write(client, cmd, 2);
	return spi_write(client, data, 2);
}

static inline int s1d13746_readb(void *client, u8 reg)
{
	int ret;
	u8 cmd[2] = { S1D13746_CMD_ADDR, reg };
	u8 data[2] = { S1D13746_CMD_READ, };
	ret = spi_write(client, cmd, 2);
	ret = spi_write_then_read(client, data, 1, data + 1, 1);
	if (ret < 0)
		return ret;
	return data[1];
}

static inline int setup_s1d13746_clock(struct spi_device *spi)
{
	int i, tmp;
	struct imx_tvout_display *pd;
	struct s1d13746_data *pdata;
	struct clk_setting *pnode;

	pd = spi_get_drvdata(spi);
	pdata = (struct s1d13746_data *)pd->data;

	tmp = sizeof(clk_table) / sizeof(clk_table[0]);
	pnode = clk_table;
	for (i = 0; i < tmp; i++, pnode++) {
		if (pdata->clk_rate == pnode->clk_rate)
			break;
	}
	if (i >= tmp)
		return -EINVAL;

	s1d13746_writeb(spi, PLL_M_DIV_REG, pnode->div);
	s1d13746_writeb(spi, PLL_4_REG, pnode->lcnt);

	tmp = 0;
	if (pdata->pll_tvclk)
		tmp |= CLK_SRC_TV_TIMING_PLL;
	if (pdata->pll_sysclk)
		tmp |= CLK_SRC_SYSCLK_PLL;

	s1d13746_writeb(spi, CLK_SRC_REG, tmp);
	s1d13746_writeb(spi, POWER_SAVE_REG, 0x00);
	for (i = 0; i < ACCESS_TIMEOUT; i++) {
		tmp = s1d13746_readb(spi, PLL_M_DIV_REG);
		if (tmp & PLL_LOCKED)
			break;
		mdelay(1);
	}
	if (i >= ACCESS_TIMEOUT)
		pr_warn("Waiting clock stable Timeout\n");
	return 0;
}

static void enable_s1d13746_device(struct imx_tvout_display *pd, bool en)
{
	struct s1d13746_data *pdata = (struct s1d13746_data *)pd->data;
	if (en)
		gpio_direction_output(pdata->standby_pin,
					!pdata->standby_active);
	else
		gpio_direction_output(pdata->standby_pin,
					pdata->standby_active);
}

static void reset_s1d13746_device(struct imx_tvout_display *pd)
{
	struct s1d13746_data *pdata = (struct s1d13746_data *)pd->data;
	gpio_direction_output(pdata->reset_pin, pdata->reset_active);
	udelay(1);
	gpio_direction_output(pdata->reset_pin, !(pdata->reset_active));
}

static int setup_s1d13746_device(struct spi_device *spi)
{
	unsigned int val;
	struct imx_tvout_display *pd;
	struct s1d13746_data *pdata;

	pd = spi_get_drvdata(spi);
	pdata = (struct s1d13746_data *)pd->data;

	setup_s1d13746_clock(spi);

	/* TODO: Add more fmt */
	val = INPUT_FMT_DEFAULT;
	switch (pd->interface_pix_fmt) {
	case INPUT_FMT_RGB888:
		/*TODO:
		 * DP format is RGB24. Just use high 6 bits for each component.
		 * Make align in future
		 */
		val |= INPUT_FMT_RGB888;
		break;
	case INPUT_FMT_RGB666:
		/*TODO:
		 * DP format is RGB24. Just use high 6 bits for each component.
		 * Make align in future
		 */
		val |= INPUT_FMT_RGB666;
		break;
	case INPUT_FMT_RGB565:
		/*TODO:
		 * DP format is RGB24. Just use high 6 bits for each component.
		 * Make align in future
		 */
		val |= INPUT_FMT_RGB565;
		break;
	default:
		return -EFAULT;
	}

	s1d13746_writeb(spi, INPUT_FMT_REG, val);
	s1d13746_writeb(spi, SPEC_EFFECT_REG, SPEC_EFFECT_DEFAULT);

	val = 0;
	if (!pdata->composite)
		val |= TVC_OUTFMT_S_Video;

	switch (pdata->tv_type) {
	case TVT_PAL:
		break;
	case TVT_PAL_M:
		val |= TVC_TVOUT_PAL_M;
		break;
	case TVT_PAL_N:
		val |= TVC_TVOUT_PAL_N;
		break;
	case TVT_PAL_Nc:
		val |= TVC_TVOUT_PAL_Nc;
		break;
	case TVT_NTSC_M:
		val |= TVC_TVOUT_NTSC_M;
		break;
	case TVT_NTSC_J:
		val |= TVC_TVOUT_NTSC_J;
	default:
		return -EFAULT;
	}

	s1d13746_writeb(spi, TVCONF_REG, val);

	if (pdata->tv_type >= TVT_NTSC_M) {
		s1d13746_writeb(spi, OUTPUT_HEIGHT_LOW_REG,
					 OUTPUT_NTSC_HEIGHT_LOW);
		s1d13746_writeb(spi, OUTPUT_HEIGHT_HIGH_REG,
					 OUTPUT_NTSC_HEIGHT_HIGH);
		s1d13746_writeb(spi, OUTPUT_WIDTH_LOW_REG,
					OUTPUT_NTSC_WIDTH_LOW);
		s1d13746_writeb(spi, OUTPUT_WIDTH_HIGH_REG,
					 OUTPUT_NTSC_WIDTH_HIGH);
	} else {
		s1d13746_writeb(spi, OUTPUT_HEIGHT_LOW_REG,
					 OUTPUT_PAL_HEIGHT_LOW);
		s1d13746_writeb(spi, OUTPUT_HEIGHT_HIGH_REG,
					 OUTPUT_PAL_HEIGHT_HIGH);
		s1d13746_writeb(spi, OUTPUT_WIDTH_LOW_REG,
					 OUTPUT_PAL_WIDTH_LOW);
		s1d13746_writeb(spi, OUTPUT_WIDTH_HIGH_REG,
					 OUTPUT_PAL_WIDTH_HIGH);
	}

	val = 0;
	if (pdata->iref)
		val |= DAC_IREF_EN;
	if (pdata->vref)
		val |= DAC_VREF_EN;
	s1d13746_writeb(spi, DAC_REF_REG, val);
	s1d13746_writeb(spi, OUTPUT_CTRL_REG, OUTPUT_TV_EN);
	return 0;
}

static int s1d13746_of_probe(struct spi_device *spi)
{
	struct imx_tvout_display *pd;
	struct device_node *np;
	struct s1d13746_data *pdata;
	struct property *prop;
	const __be32 *val;
	enum of_gpio_flags gpio_flags;
	int ret;
	const char *fmt;

	np = spi->dev.of_node;

	pd = spi_get_drvdata(spi);
	if (IS_ERR(pd))
		return -EFAULT;

	pdata = (struct s1d13746_data *)pd->data;

	pdata->standby_pin = of_get_named_gpio_flags(np, "standby-pin", 0,
						     &gpio_flags);
	if (pdata->standby_pin == -EPROBE_DEFER)
		return -EPROBE_DEFER;

	if (gpio_is_valid(pdata->standby_pin)) {
		pdata->standby_active = (gpio_flags == OF_GPIO_ACTIVE_LOW) ?
		    0 : 1;
		ret = gpio_request_one(pdata->standby_pin, GPIOF_DIR_OUT |
				       ((pdata->standby_active) ? GPIOF_INIT_LOW
					: GPIOF_INIT_HIGH), "s1d13746-standby");
		if (ret < 0) {
			dev_err(pd->pdev, "request gpio:power pin failed!\n");
			return ret;
		}
	}

	pdata->reset_pin = of_get_named_gpio_flags(np, "reset-pin", 0,
						   &gpio_flags);
	if (pdata->reset_pin == -EPROBE_DEFER) {
		ret = -EPROBE_DEFER;
		goto gpio_err;
	}

	if (gpio_is_valid(pdata->reset_pin)) {
		pdata->reset_active = (gpio_flags == OF_GPIO_ACTIVE_LOW) ?
		    0 : 1;
		ret = gpio_request_one(pdata->reset_pin, GPIOF_DIR_OUT |
				       ((pdata->reset_active) ? GPIOF_INIT_LOW :
					GPIOF_INIT_HIGH), "s1d13746-reset");
		if (ret < 0) {
			dev_err(pd->pdev, "request gpio:reset pin failed!\n");
			goto gpio_err;
		}
	}

	pdata->tv_type = TVT_NTSC_M;
	prop = of_find_property(np, "tv-standard", NULL);
	if ((prop != NULL) && (prop->value)) {
		if (!strncmp(prop->value, "PAL_", 4)) {
			unsigned char *s = (unsigned char *)prop->value;
			switch (s[4]) {
			case 'M':
				pdata->tv_type = TVT_PAL_M;
				break;
			case 'N':
				if (s[5] == 'c')
					pdata->tv_type = TVT_PAL_Nc;
				else
					pdata->tv_type = TVT_PAL_N;
				break;
			}
		} else if (!strncmp(prop->value, "NTSC_", 5)) {
			unsigned char *s = (unsigned char *)prop->value;
			switch (s[5]) {
			case 'M':
				pdata->tv_type = TVT_NTSC_M;
				break;
			case 'J':
				pdata->tv_type = TVT_NTSC_J;
				break;
			};
		}
	}

	ret = of_property_read_string(np, "input-pix-fmt", &fmt);
	if (!ret) {
		if (!strcmp(fmt, "rgb24"))
			pd->interface_pix_fmt = INPUT_FMT_RGB888;
		else if (!strcmp(fmt, "rgb565"))
			pd->interface_pix_fmt = INPUT_FMT_RGB565;
		else if (!strcmp(fmt, "rgb18"))
			pd->interface_pix_fmt = INPUT_FMT_RGB666;
	}

	prop = of_find_property(np, "ext_tv-timing", NULL);
	if (prop != NULL)
		pdata->pll_tvclk = false;
	else
		pdata->pll_tvclk = true;

	prop = of_find_property(np, "ext_sysclk", NULL);
	if (prop != NULL)
		pdata->pll_sysclk = false;
	else
		pdata->pll_sysclk = true;

	prop = of_find_property(np, "iref", NULL);
	if (prop != NULL)
		pdata->iref = true;
	else
		pdata->iref = false;

	prop = of_find_property(np, "vref", NULL);
	if (prop != NULL)
		pdata->vref = true;
	else
		pdata->vref = false;

	prop = of_find_property(np, "S_Video", NULL);
	if (prop != NULL)
		pdata->composite = false;
	else
		pdata->composite = true;

	val = of_get_property(np, "clock-rate", NULL);
	if (val)
		pdata->clk_rate = be32_to_cpup(val);
	return 0;

gpio_err:
	enable_s1d13746_device(pd, 0);
	if (gpio_is_valid(pdata->standby_pin)) {
		gpio_free(pdata->standby_pin);
		pdata->standby_pin = -1;
	}
	return ret;
}

static int s1d13746_probe(struct spi_device *spi)
{
	struct imx_tvout_display *pd;
	const struct of_device_id *of_id;
	struct s1d13746_data *pdata;
	int ret, version;
	int timeout;

	of_id = of_match_device(s1d13746_dt_ids, &spi->dev);
	if (!of_id)
		return -ENODEV;

	pd = devm_kzalloc(&spi->dev, sizeof(*pd) + sizeof(*pdata), GFP_KERNEL);
	if (!pd) {
		dev_err(&spi->dev, "Allocate imx_tvout_display fail!\n");
		return -ENOMEM;
	}

	pd->pdev = &spi->dev;
	pd->data = pdata = (struct s1d13746_data *)(pd + 1);

	spi_set_drvdata(spi, pd);
	ret = s1d13746_of_probe(spi);
	if (ret != 0)
		goto free_mem;
	reset_s1d13746_device(pd);

	for (timeout = 0; timeout < ACCESS_TIMEOUT; timeout++) {
		version = s1d13746_readb(spi, REVISION_REG);
		if (version >= 0)
			break;
		udelay(1);
	}
	if (version != REVISION_ID) {
		dev_err(pd->pdev, "incorrect version:0x%x\n", version);
		ret = -EINVAL;
		goto chip_err;
	}
	setup_s1d13746_device(spi);
	return 0;

chip_err:
	if (gpio_is_valid(pdata->reset_pin)) {
		gpio_set_value(pdata->reset_pin, pdata->reset_active);
		gpio_free(pdata->reset_pin);
		pdata->reset_pin = -1;
	}
	enable_s1d13746_device(pd, 0);
	if (gpio_is_valid(pdata->standby_pin)) {
		gpio_free(pdata->standby_pin);
		pdata->standby_pin = -1;
	}
free_mem:
	spi_set_drvdata(spi, NULL);
	devm_kfree(&spi->dev, pd);
	return ret;
}

static int s1d13746_remove(struct spi_device *spi)
{
	struct imx_tvout_display *pd;
	struct s1d13746_data *pdata;
	pd = spi_get_drvdata(spi);
	if (IS_ERR(pd))
		return 0;
	pdata = (struct s1d13746_data *)pd->data;

	if (IS_ERR(pdata))
		return 0;

	if (gpio_is_valid(pdata->reset_pin)) {
		gpio_set_value(pdata->reset_pin, pdata->reset_active);
		gpio_free(pdata->reset_pin);
		pdata->reset_pin = -1;
	}

	enable_s1d13746_device(pd, 0);
	if (gpio_is_valid(pdata->standby_pin)) {
		gpio_free(pdata->standby_pin);
		pdata->standby_pin = -1;
	}

	spi_set_drvdata(spi, NULL);
	devm_kfree(&spi->dev, pd);
	return 0;
}

/* Encoder drivers */
static struct spi_driver s1d13746_driver = {
	.driver = {
		   .name = "s1d13746",
		   .owner = THIS_MODULE,
		   .of_match_table = s1d13746_dt_ids,
		   },
	.probe = s1d13746_probe,
	.remove = s1d13746_remove,
};

module_spi_driver(s1d13746_driver);

MODULE_AUTHOR("Fred Fan <Yefeng.fan2@de.bosch.com>");
MODULE_DESCRIPTION("Epson S1d13746 TVOUT driver");
MODULE_LICENSE("GPL v2");
MODULE_VERSION("0001");
