/***********************************************************************
 * \file: masca_util_rspc.h
 *
 * \version:
 *
 * \release:
 *
 * Interface description for the RSPC (Read Solomon Product like Correction)
 * library.
 *
 * \component: UTIL Library (UTIL LIB)
 *
 * \author          Kurt Gieske kgieske (at) de.adit-jv.com
 * \author          Simon Maleyka smaleyka (at) de.adit-jv.com
 * \author          Christoph Gellner cgellner (at) de.adit-jv.com
 * \author          sundhar.ashokan@in.bosch.com
 * \author          saurabh.arora@in.bosch.com
 * \copyright: (c) 2003 - 2008 ADIT Corporation
 *
 ***********************************************************************/
#ifndef UTIL_RSPC_H
#define UTIL_RSPC_H

#ifdef __cplusplus
extern "C" {
#endif

#include "masca_common_types.h"

enum cd_mode_te {
	CD_MODE1 = 1, CD_MODE2 = 2
};

enum cdedqoc_te {
	CDED_QOC_NONE = 0, /* don't use, it's not supported */
	CDED_QOC_LOW = 1, CDED_QOC_MID = 2, CDED_QOC_HIGH = 3
};

/**
 * \func util_rspc_descramble_data
 *  Description:   Descramble CDED data block
 *
 * \param unsigned int *      sector_data      poiner to the sync word of the
 *  sector (in out)
 *
 * \return          None
 */
void util_rspc_descramble_data(unsigned int *sector_data);

/**
 * \func util_rspc_correction
 *  Description:   Execute the Read Solomon error correction on CD block data
 *		   for Mode 1 and Mode2 XA Form 1. Other formats must not be
 *		   used with this function
 *  After the correction the CRC is recalulated to check data integrity.
 *
 * \param unsigned short *      sector_data        poiner to the sync word of
 *  the sector
 * \param bool       crc_check_enabled  enable CRC check after RSPC correction
 * \param cdedqoc_te iterations         Num of iterations that the RPSC decoder
					can do
 *
 * \return          \li E_OK,  sector OK
 *                  \li E_IO,  CRC check failed
 */
int util_rspc_correction(unsigned short *sector_data,
						bool crc_check_enabled,
						enum cdedqoc_te iterations);

/**
 * \func UTIL_rspc_calc_crc32
 * Description: Calculate the CRC over one block of CD block data
 *
 * \param unsigned short *        data_pt      pointer to the sector, starting
 *  at the
 *				    sync-word
 * \param cd_mode_te   cd_mode      1 for Mode 1, 2 for Mode 2 XA Form 1
 *
 * \return          \li 0 - CRC check passed
 *                  \li other - check failed
 */
unsigned int UTIL_rspc_calc_crc32(unsigned short *data_pt,
						enum cd_mode_te cd_mode);

#define UTIL_RSPC_CDED_SECTOR_SIZE	2352
#define UTIL_RSPC_CDED_SYNCWORD_SIZE	12
#define UTIL_RSPC_CDED_SIZEOF_EDC_FIELD	4

/*is the code executed on big or little endian machine*/
#undef UTIL_RSPC_CDED_BIG_ENDIAN

/*is the input data byte swapped*/
#undef UTIL_RSPC_CDED_BYTE_SWAP

#ifdef UTIL_RSPC_CDED_BYTE_SWAP
#define UTIL_RSPC_CDED_BYTE_SWAP_CORR_VAL	1
#else
#define UTIL_RSPC_CDED_BYTE_SWAP_CORR_VAL	0
#endif

/* for C-DED usage, the DMA-tranfersize must be set by define:
 * UTIL_RSPC_CDED_DATA_BLOCK_SIZE:
 * 2352B Sector (CD-ROM mode 01) - 12B sync + 4 Byte EDC addon from C-DED
 */
#define UTIL_RSPC_CDED_DATA_BLOCK_SIZE	((UTIL_RSPC_CDED_SECTOR_SIZE \
					- UTIL_RSPC_CDED_SYNCWORD_SIZE)\
					+ UTIL_RSPC_CDED_SIZEOF_EDC_FIELD)

/* position of the mode information inside the datastream, corrected by sync-
 * wordsize and with respect to the byteswap
 */
#define UTIL_RSPC_CDED_MODE_POSITION	(3 - UTIL_RSPC_CDED_BYTE_SWAP_CORR_VAL)

/*#define UTIL_RSPC_CDED_HEADER_MODE_POSITION_16 8*/
#define  UTIL_RSPC_CD_DATA_SECTOR_HEADER_SIZE		4u
#define  UTIL_RSPC_CRC32_MODE1_LEN			1028u
#define  UTIL_RSPC_CRC32_MODE2_LEN			1032u
#define  UTIL_RSPC_CRC32_MODE2_START_OFFSET		4u
#define  UTIL_RSPC_LOWER_BYTE_MASK			0x000000FF
#define  UTIL_RSPC_FULL_U32_MASK			0xFFFFFFFF
#define  UTIL_RSPC_SHIFT_BITS_FOR_BYTE			8u
#define  UTIL_RSPC_SIM_SYNC_WORD_RESULT_MODE1		0xDF1DFF92

#ifdef __cplusplus
}
#endif

#endif /* UTIL_RSPC */
